/* @(#)fileio.c 1.12   8/28/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, *
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <math.h>

#include "typedef.h"
#include "main.h"
#include "render.h"
#include "gifload.h"
#ifndef	WIN32
#include "jpgload.h"
#endif
#include "glib.h"

#ifndef	PATH_MAX
#define	PATH_MAX	256
#endif

#ifndef _POSIX_PATH_MAX
#define _POSIX_PATH_MAX PATH_MAX
#endif                            

/*************************
 * Variable definitions. *
 *************************/

/*
 * Static variables.
 */
static int	version, revision;
static FILE	*open_fp;
static int	line_no;
static char 	buffer[BUFSIZ], buffer_token[BUFSIZ], *buffer_ptr;

/****************************************************************************
 * Open a file or URL.  For files, if the path is relative we first look in *
 * the current working directory, then in the directories given by the file *
 * path list.  On a successful open we return the file pointer, otherwise   *
 * we return NULL.                                                          *
 ****************************************************************************/

static FILE *
open_file(char *name)
{
	FILE *fp;
	char new_name[_POSIX_PATH_MAX];
	int index;

	/*
	 * Otherwise attempt to open the file.
	 */
	if ((fp = fopen(name, "rb")) == NULL && *name != DIR_SEPARATOR_CHAR &&
	    file_path_list) {
		for (index = 0; index < paths; index++) {
			strcpy(new_name, file_path_list[index]);
			strcat(new_name, DIR_SEPARATOR_STR);
			strcat(new_name, name);
			if ((fp = fopen(new_name, "rb")) != NULL)
				return(fp);
		}
		return(NULL);
	}
	return(fp);
}

/*****************************************************************************
 * Display a formatted error message in a similiar fashion to printf(), then *
 * close the open file and return to the main function with an error.        *
 *****************************************************************************/

void
file_error(char *format, ...)
{
	va_list arg_ptr;
	char message[BUFSIZ];

	system_message("Error in file, line %d:\n\t", line_no);
	va_start(arg_ptr, format);
	vsprintf(message, format, arg_ptr);
	va_end(arg_ptr);
	system_message("%s\n", message);
	fclose(open_fp);
	longjmp(context, TRUE);
}

/***************************************************************************
 * Display a formatted memory error message, then close the open file (if  *
 * there is one), and return to the caller function with an error.	   *
 ***************************************************************************/

void
memory_error(char *object)
{
	system_message("Could not allocate memory for %s\n", object);
	if (open_fp)
		fclose(open_fp);
	longjmp(context, TRUE);
}

/************************************************************************
 * Verify that the given integer is within the specified range; if not, *
 * generate a fatal error.                                              *
 ************************************************************************/

void
check_int_range(char *message, int value, int min, int max)
{
	if (value < min || value > max) 
		file_error("%s must be between %d and %d", message, min, max);
}

/***********************************************************************
 * Verify that the given double is within the specified range; if not, *
 * generate a fatal error.                                             *
 ***********************************************************************/

void
check_double_range(char *message, double value, double min, double max)
{
	if (value < min || value > max)
		file_error("%s must be between %g and %g", message, min, max);
}

/*************************************************************************
 * Extract a token from the start of a string.  The token buffer must be *
 * provided and the string pointer will be adjusted to point to the end  *
 * of the token found.	    						 *
 *************************************************************************/

static void
extract_token(char **string, char *token)
{
	char *string_ptr, *token_ptr;
	int string_len;

	/*
	 * Skip over leading white space.
	 */
	string_ptr = *string;
	while (*string_ptr == ' ' || *string_ptr == '\t' ||
	       *string_ptr == '\n' || *string_ptr == '\r')
		string_ptr++;

	/*
	 * If we've come to the end of the string, return an empty token.
	 */
	if (!*string_ptr) {
		*token = '\0';
		*string = string_ptr;
		return;
	}

	/*
	 * Commas, equal signs, parentheses or hashs are considered tokens in
	 * their own right.
	 */
	if (*string_ptr == ',' || *string_ptr == '=' || *string_ptr == '#' ||
	    *string_ptr == '(' || *string_ptr == ')') {
		*token++ = *string_ptr;
		*token = '\0';
		*string = string_ptr + 1;
		return;
	}

	/*
	 * Search for the end of the token (a white space character, a comma,
	 * an equal sign, a hash, a parenthesis, or an end of line character)
	 * and return that token.
	 */
	token_ptr = string_ptr;
	do
		string_ptr++;
	while (*string_ptr != ' ' && *string_ptr != '\t' &&
	       *string_ptr != '\n' && *string_ptr != '\0' &&
	       *string_ptr != ',' && *string_ptr != '=' &&
	       *string_ptr != '(' && *string_ptr != ')' &&
	       *string_ptr != '#' && *string_ptr != '\r');
	string_len = string_ptr - token_ptr;
	strncpy(token, token_ptr, string_len);
	token[string_len] = '\0';
	*string = string_ptr;
	return;
}

/****************************************************************************
 * Read the next non-blank, non-command line and extract the first token in *
 * that line.                                                               *
 ****************************************************************************/

static void
read_line(void)
{
	do {
		line_no++;
		if (!fgets(buffer, BUFSIZ, open_fp))
			file_error("unexpected end of file");
		buffer_ptr = buffer;
		extract_token(&buffer_ptr, buffer_token);
	} while (*buffer_token == '\0' || *buffer_token == '#');
}

/**************************************************************************
 * Read one token from the currently open file.  A token is a sequence of *
 * characters seperated by white space.                                   *
 **************************************************************************/

static void
read_next_token(void)
{
	if (line_no > 0) {
		extract_token(&buffer_ptr, buffer_token);
		if (*buffer_token == '\0' || *buffer_token == '#')
			read_line();
	} else
                read_line();
}

/****************************************************************************
 * Parse the BSP file according a format string.  This function operates    *
 * in a similiar fashion to scanf() except that white space is ignored in   *
 * both the format string and the BSP file (including end of line) and      *
 * comments in the BSP file are skipped over.  The format string can have   *
 * %s, %d and %lf format specifies for parsing strings, integers and double *
 * floats respectively.                                                     *
 ****************************************************************************/

static void
read_file(char *format, ...)
{
	va_list arg_ptr;
	char string_token[BUFSIZ];
	char *string_ptr;
	int int_value, *int_ptr;
	double double_value, *double_ptr;

	va_start(arg_ptr, format);
	extract_token(&format, string_token);
	while (*string_token) {
		read_next_token();
		if (!strcmp(string_token, "%s")) {
			string_ptr = va_arg(arg_ptr, char *);
			strcpy(string_ptr, buffer_token);
		} else if (!strcmp(string_token, "%d")) {
			int_ptr = va_arg(arg_ptr, int *);	
			if (sscanf(buffer_token, "%d", &int_value) != 1)
				file_error("'%s' is not a valid integer",
					buffer_token);
			*int_ptr = int_value;
		} else if (!strcmp(string_token, "%lf")) {
			double_ptr = va_arg(arg_ptr, double *);			
			if (sscanf(buffer_token, "%lf", &double_value) != 1)
				file_error("'%s' is not a valid real number",
					buffer_token);
			*double_ptr = double_value;
		} else if (strcmp(string_token, buffer_token))
			file_error("I was expecting '%s' instead of '%s'",
				string_token, buffer_token);
		extract_token(&format, string_token);
	}
	va_end(arg_ptr);
}

/*********************************************
 * Read and parse the format version number. *
 *********************************************/

static void
read_version(int *version, int *revision)
{
	read_next_token();
	if (sscanf(buffer_token, "%d.%d", version, revision) != 2)
		file_error("'%s' is not a valid version number", buffer_token);
}

/*****************************************************************
 * Read the start line giving starting position and orientation. *
 *****************************************************************/

static void
read_start_line(void)
{
	double start_x, start_y, start_z;
	int start_angle_x, start_angle_y;

	read_file("start position (%lf,%lf,%lf) turn angle %d look angle %d",
		&start_x, &start_y, &start_z, &start_angle_y, &start_angle_x);
	check_double_range("starting x coordinate", start_x,
		MIN_FIXED, MAX_FIXED);
	check_double_range("starting y coordinate", start_y,
		MIN_FIXED, MAX_FIXED);
	check_double_range("starting z coordinate", start_z,
		MIN_FIXED, MAX_FIXED);
	check_int_range("starting turn angle", start_angle_y,
		0, 359);
	check_int_range("starting look angle", start_angle_x,
		-90, 90);
	view_x = FIXED(start_x);
	view_y = FIXED(start_y);
	view_z = FIXED(start_z);
	view_angle_y = start_angle_y;
	view_angle_x = start_angle_x;
	if (view_angle_x < 0)
		view_angle_x += angles;
}

/**************************************************************************
 * Load a texture map from a JPG or GIF file.  If a relative path name is *
 * given, we first search the current directory, then the directory given *
 * by the TEX_DIR environment variable (if it was set).                   *
 **************************************************************************/

static void
load_texture_map(char *tex_file_name,
		 pixel **image_ptr, int *image_width, int *image_height)
{
	char *extension;
	char file_class[10];
	FILE *fp;

	/*
	 * Attempt to open the file using the given absolute or relative path.
	 * If that fails and the path was relative, add the prefix given by
	 * tex_dir.
	 */
	if ((fp = open_file(tex_file_name)) == NULL)
		file_error("Unable to open texture map file '%s'",
			tex_file_name);

	/*
	 * Use the file extention to determine the file type, then call the
	 * approapiate load function.
	 */
	if (!strncmp(tex_file_name, "http://", 7))
		strcpy(file_class, "URL");
	else
		strcpy(file_class, "File");

	extension = strrchr(tex_file_name, '.') + 1;
#ifndef	WIN32
	if (!strcmp(extension, "jpg")) {
		load_JPG(fp, image_ptr, image_width, image_height);
		system_message("%s '%s' is a %dx%d JPG image\n", file_class,
			tex_file_name, *image_width, *image_height);
	} else
#endif
	if (!strcmp(extension, "gif")) {
		load_GIF(fp, image_ptr, image_width, image_height);
		system_message("%s '%s' is a %dx%d GIF image\n", file_class,
			tex_file_name, *image_width, *image_height);
	}

	/*
	 * Close the Texture map file.
	 */
	fclose(fp);
}

/****************************************************************************
 * Determine the best size for a texture map (dimensions must be a power of *
 * 2). A smaller size is prefered to a larger one to minimise pixelation.   *
 ****************************************************************************/

#define IMAGE_SIZES	11

static void
get_best_image_size(int image_width, int image_height,
		    int *best_image_width, int *best_image_height,
		    int *best_column_shift)
{ 
	int dimension[IMAGE_SIZES] = {
		1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024
	};
	int index;

	/*
	 * Step through each dimension, and choose the largest width that is
	 * less than the image width.
	 */
	for (index = 0; index < IMAGE_SIZES; index++)
		if (dimension[index] > image_width) 
			break;
	*best_image_width = dimension[index - 1];

	/*
	 * Step through each dimension, and choose the largest height that is
	 * less than the image height.  The index of the chosen height becomes
	 * the column shift constant (used to scale a column offset by the
	 * column height).
	 */
	for (index = 0; index < IMAGE_SIZES; index++)
		if (dimension[index] > image_height)
			break;
	*best_image_height = dimension[index - 1];
	*best_column_shift = index - 1;
}

/***********************************
 * Read and parse the vertex list. *
 ***********************************/

static void
read_vertex_list(void)
{
	int vertex_no, actual_no;
	double x, y, z;
	vertex *vertex_ptr;

	/*
	 * Read the number of vertices, and allocate the list of vertex
	 * pointers.
	 */
	read_file("vertices = %d", &vertices);
	if (vertices < 1)
		file_error("number of vertices must be > 0");
	if ((vertex_ptr_list = (vertex **)calloc(vertices, sizeof(vertex *)))
	    == NULL)
		memory_error("vertex list");

	/*
	 * Set all vertex pointers to NULL in case of error.
	 */
	for (vertex_no = 0; vertex_no < vertices; vertex_no++)
		vertex_ptr_list[vertex_no] = NULL;

	/*
	 * Read each vertex definition and create a new vertex.
	 */
	for (vertex_no = 0; vertex_no < vertices; vertex_no++) {
		read_file("vertex %d at (%lf,%lf,%lf)", &actual_no, &x, &y, &z);
		if (actual_no != vertex_no + 1)
			file_error("vertex number is out of sequence");
		check_double_range("vertex x coordinate", x,
			MIN_FIXED, MAX_FIXED);
		check_double_range("vertex y coordinate", y,
			MIN_FIXED, MAX_FIXED);
		check_double_range("vertex z coordinate", z,
			MIN_FIXED, MAX_FIXED);
		if ((vertex_ptr = (vertex *)malloc(sizeof(vertex))) == NULL)
			memory_error("vertex");
		vertex_ptr->x = FIXED(x);
		vertex_ptr->y = FIXED(y);
		vertex_ptr->z = FIXED(z);
		vertex_ptr_list[vertex_no] = vertex_ptr;
	}
}

/**************************************************************
 * Read the texture map list, and load the texture map files. *
 **************************************************************/

static void
read_texture_list(void)
{
	int texture_no, actual_no;
	texture *texture_ptr;
	char tex_file_name[_POSIX_PATH_MAX];
	pixel *image_ptr, *tex_ptr;
	int image_width, image_height;
	int best_width, best_height, best_column_shift;
	double delta_width, delta_height;
	double image_row, image_col;
	int tex_row, tex_col;

	/*
	 * Read the number of textures and create the texture pointer list.
	 */
	read_file("textures = %d", &textures);
	if (textures < 1)
		file_error("number of textures must be > 0");
	if ((texture_ptr_list = (texture **)calloc(textures,
						   sizeof(texture *))) == NULL)
		memory_error("texture list");

	/*
	 * Set all texture pointers to NULL in case of error.
	 */
	for (texture_no = 0; texture_no < textures; texture_no++)
		texture_ptr_list[texture_no] = NULL;

	/*
	 * Read each texture definition and create a new texture.
	 */
	for (texture_no = 0; texture_no < textures; texture_no++) {
		/*
		 * Read the texture definition and load the texture map.
		 */
		read_file("texture %d is %s\n", &actual_no, tex_file_name);
		if (actual_no != texture_no + 1)
			file_error("texture number is out of sequence");
		load_texture_map(tex_file_name, &image_ptr, &image_width,
			&image_height);

		/*
		 * Scale the loaded image to the best size (dimensions must be
		 * a power of 2).  Scaling down is prefered to scaling up.
		 */
		get_best_image_size(image_width, image_height,
			&best_width, &best_height, &best_column_shift);
		delta_width = (double)image_width / (double)best_width;
		delta_height = (double)image_height / (double)best_height;
		if ((tex_ptr = (pixel *)malloc(best_width * best_height *
					       sizeof(pixel))) == NULL)
			memory_error("texture map");
		for (tex_row = 0, image_row = 0.0; tex_row < best_height;
		     tex_row++, image_row += delta_height)
			for (tex_col = 0, image_col = 0.0; tex_col < best_width;
			     tex_col++, image_col += delta_width)
				*(tex_ptr + tex_row + tex_col * best_height) =
					*(image_ptr + (int)image_row +
					  (int)(image_col) * image_height);
		free(image_ptr);

		/*
		 * Allocate a new texture element, and copy the texture
		 * definition to it.
		 */
		if ((texture_ptr = (texture *)malloc(sizeof(texture))) == NULL)
			memory_error("texture");
		texture_ptr->image_ptr = tex_ptr;
		texture_ptr->width = INT_TO_FIXED(best_width);
		texture_ptr->height = INT_TO_FIXED(best_height);
		texture_ptr->width_mask = best_width - 1;
		texture_ptr->height_mask = best_height - 1;
		texture_ptr->column_shift = best_column_shift;
		texture_ptr_list[texture_no] = texture_ptr;
	}
}

/************************************
 * Read and parse the polygon list. *
 ************************************/

static void
read_polygon_list(void)
{
	int polygon_no, actual_no;
	polygon *polygon_ptr;
	int polygon_vertices;
	vertex **polygon_vertex_ptr_list, *vertex_ptr;
	int vertex_no, polygon_vertex_no;
	tpoint **tpoint_ptr_list, *tpoint_ptr;
	texture *texture_ptr;
	int texture_no;
	int tpoint_no;
	int u, v;
	int max_vertices;
	int light_level;

	/*
	 * Read the number of polygons and create the polygon pointer list.
	 */
	max_vertices = 0;
	read_file("polygons = %d", &polygons);
	if (polygons < 1)
		file_error("number of polygons must be > 0");
	if ((polygon_ptr_list = (polygon **)calloc(polygons,
						   sizeof(polygon *))) == NULL)
		memory_error("polygon list");

	/*
	 * Set all polygon pointers to NULL in case of error.
	 */
	for (polygon_no = 0; polygon_no < polygons; polygon_no++)
		polygon_ptr_list[polygon_no] = NULL;

	/*
	 * Read each polygon definition and create a new polygon.
	 */
	for (polygon_no = 0; polygon_no < polygons; polygon_no++) {
		/*
		 * Read the number of vertices in polygon.
		 */
		read_file("polygon %d has %d vertices", &actual_no,
			&polygon_vertices);
		if (actual_no != polygon_no + 1)
			file_error("polygon number is out of sequence");
		if (polygon_vertices < 3)
			file_error("a polygon must have at least 3 vertices");
		if (polygon_vertices > max_vertices)
			max_vertices = polygon_vertices;

		/*
		 * Create the vertex list and read in the vertex numbers.
		 */
		if ((polygon_vertex_ptr_list =
		     (vertex **)calloc(polygon_vertices, sizeof(vertex *)))
		     == NULL)
			memory_error("polygon vertex list");
		for (polygon_vertex_no = 0;
		     polygon_vertex_no < polygon_vertices;
		     polygon_vertex_no++) {
			if (polygon_vertex_no)
				read_file(",%d", &vertex_no);
			else
				read_file("%d", &vertex_no);
			check_int_range("polygon vertex number", vertex_no,
				1, vertices);
			vertex_ptr = vertex_ptr_list[vertex_no - 1];
			polygon_vertex_ptr_list[polygon_vertex_no] = vertex_ptr;
		}

		/*
		 * Create the texture point list and read in the texture
		 * points.
		 */
		read_file("with texture points");
		if ((tpoint_ptr_list =
		     (tpoint **)calloc(polygon_vertices, sizeof(tpoint *)))
		     == NULL)
			memory_error("polygon texture point list");
		for (tpoint_no = 0; tpoint_no < polygon_vertices; tpoint_no++) {
			if (tpoint_no)
				read_file(",(%d,%d)", &u, &v);
			else
				read_file("(%d,%d)", &u, &v);
			if ((tpoint_ptr = (tpoint *)malloc(sizeof(tpoint)))
			    == NULL)
				memory_error("polygon texture point");
			tpoint_ptr->u = INT_TO_FIXED(u);
			tpoint_ptr->v = INT_TO_FIXED(v);
			tpoint_ptr_list[tpoint_no] = tpoint_ptr;
		}

		/*
		 * Read the texture number.
		 */
		read_file("using texture %d", &texture_no);
		check_int_range("polygon texture number", texture_no, 1,
			textures);
		texture_ptr = texture_ptr_list[texture_no - 1];

		/*
		 * Read the light level.
		 */
		read_file("at light level %d", &light_level);
		check_int_range("polygon light level", light_level,
			0, LIGHT_MAX);

		/*
		 * Create the polygon and fill in the fields.
		 */
		if ((polygon_ptr = (polygon *)malloc(sizeof(polygon))) == NULL)
			memory_error("polygon");
		polygon_ptr->vertices = polygon_vertices;
		polygon_ptr->vertex_ptr_list = polygon_vertex_ptr_list;
		polygon_ptr->tpoint_ptr_list = tpoint_ptr_list;
		polygon_ptr->texture_ptr = texture_ptr;
		polygon_ptr->light_level = light_level;
		polygon_ptr_list[polygon_no] = polygon_ptr;

		/*
		 * Compute the surface normal for this polygon.
		 */
		{
			vertex *vertex1_ptr, *vertex2_ptr, *vertex3_ptr;
			fixed Ax, Ay, Az, Bx, By, Bz, Nx, Ny, Nz;
			fixed magnitude;

			/*
			 * Use the first three vertices of the polygon.
			 */
			vertex1_ptr = polygon_vertex_ptr_list[0];
			vertex2_ptr = polygon_vertex_ptr_list[1];
			vertex3_ptr = polygon_vertex_ptr_list[2];
				
			/*
			 * Compute vector A (from vertex 1 to vertex 2).
			 */
			Ax = vertex2_ptr->x - vertex1_ptr->x;
			Ay = vertex2_ptr->y - vertex1_ptr->y;	
			Az = vertex2_ptr->z - vertex1_ptr->z;

			/*
			 * Compute vector B (from vertex 3 to vertex 2).
			 */
			Bx = vertex2_ptr->x - vertex3_ptr->x;
			By = vertex2_ptr->y - vertex3_ptr->y;	
			Bz = vertex2_ptr->z - vertex3_ptr->z;

			/*
			 * Compute the normal vector N.  This is pointing
			 * away from the back face.
			 */
			Nx = fmul(Ay, Bz) - fmul(Az, By);
			Ny = fmul(Az, Bx) - fmul(Ax, Bz);
			Nz = fmul(Ax, By) - fmul(Ay, Bx);

			/*
			 * Compute the magnitude of the normal vector, then
			 * use it to convert the normal vector to unit length.
			 * Having done that, we subtract the normal vector from
			 * vertex 2 to generate a normal vertex that is
			 * pointing towards the viewer when we are looking at
			 * the front face of the polygon.
			 */
			magnitude = FIXED(sqrt(FLOAT(fmul(Nx, Nx) +
				fmul(Ny, Ny) + fmul(Nz, Nz))));
			polygon_ptr->normal_vertex.x = vertex2_ptr->x -
				fdiv(Nx, magnitude);
			polygon_ptr->normal_vertex.y = vertex2_ptr->y -
				fdiv(Ny, magnitude);
			polygon_ptr->normal_vertex.z = vertex2_ptr->z -
				fdiv(Nz, magnitude);
		}
	}

	/*
	 * Allocate the global screen point list.
	 */
	if ((spoint_list = (spoint *)calloc(max_vertices + 1, sizeof(spoint)))
	    == NULL)
		memory_error("global screen point list");
}

void
read_world_file(char *world_file_name)
{
	/*
	 * Attempt to open the world file.
	 */
	if ((open_fp = open_file(world_file_name)) == NULL) {
		system_message("Could not open file '%s'", world_file_name);
		longjmp(context, TRUE);
	}
	line_no = 0;

	/*
	 * Verify the file header.
	 */
	read_file("Threedom world file, version");
	read_version(&version, &revision);
	if (version < 1 || revision > 0)
		file_error("Only version 1.0 world files can be read by this"
			"version of Threedom");

	/*
	 * Read the starting player position and orientation.
	 */
	read_start_line();

	/*
	 * Read the vertex list.
	 */
	read_vertex_list();
	system_message("Loaded %d %s\n", vertices,
		vertices == 1 ? "vertex" : "vertices");
	read_texture_list();
	system_message("Loaded %d %s\n", textures,
		textures == 1 ? "texture" : "textures");
	read_polygon_list();
	system_message("Loaded %d %s\n", polygons,
		polygons == 1 ? "polygon" : "polygons");

	/*
	 * Close the world file.
	 */
	fclose(open_fp);
}
