/* @(#)main.c 1.16   8/28/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, *
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <setjmp.h>

#include "typedef.h"
#include "dither.h"
#include "fileio.h"
#include "render.h"
#include "glib.h"

/*
 * Some default settings.
 */
#define VERSION_STR		"Threedom version 1.0 beta 2"
#define	DEFAULT_WINDOW_WIDTH	320
#define DEFAULT_WINDOW_HEIGHT	200
#define DEFAULT_PIXEL_SPAN	32
#define DEFAULT_SPAN_SHIFT	5
#define DEFAULT_FIELD_OF_VIEW_X	90.0
#define DEFAULT_FIELD_OF_VIEW_Y	90.0

/*
 * Definition of the path seperator in the THREEDOM_PATH environment variable.
 */
#ifndef	WIN32
#define PATH_SEPARATOR_CHAR	':'
#define PATH_SEPARATOR_STR	":"
#else
#define PATH_SEPARATOR_CHAR	';'
#define PATH_SEPARATOR_STR	";"
#endif

/*
 * Turn off some warning messages under Borland C++.
 */
#ifdef	WIN32
#pragma warn -aus
#pragma warn -par
#endif

/*******************************************
 * Global and static variable definitions. *
 *******************************************/

/*
 * Command line arguments and environment variables.
 */
char		*program_name;
static char	*file_path_str;
char		**file_path_list = NULL;
int		paths;

/*
 * Tables allocated and/or initialised at run time.
 */
fbpixel		*frame_buffer = NULL;
fbpixel		**fb_row = NULL;
fixed		*z_buffer = NULL;
fixed		**zb_row = NULL;
fixed		z_factor = 0;
fixed		*sine = NULL;
fixed		*cosine = NULL;
int		angles = 360;

/*
 * Global variables and their default values (if any).
 */
int		window_width = DEFAULT_WINDOW_WIDTH;
int		window_height = DEFAULT_WINDOW_HEIGHT;
int		pixel_span = DEFAULT_PIXEL_SPAN;
int		span_shift = DEFAULT_SPAN_SHIFT;
fixed 		view_x = 0;
fixed		view_y = 0;
fixed		view_z = 0;
int 		view_angle_x = 0;
int		view_angle_y = 0;
fixed		min_px, max_px, delta_px;
fixed		min_py, max_py, delta_py;
fixed		half_window_width;
fixed		half_window_height;
jmp_buf		context;

/*
 * Static variables and their default values.
 */
static boolean	renderer_on;
#ifndef	WIN32
static int	display_width;
static int	display_height;
#endif
static double	field_of_view_x = DEFAULT_FIELD_OF_VIEW_X;
static double	field_of_view_y = DEFAULT_FIELD_OF_VIEW_Y;
static time_t	start_time;
static int	frames_rendered;
static int	ft = 0, fl = 0;
static fixed	fx = 0, fz = 0;


/*
 * Data structures for describing world layout.
 */
vertex	**vertex_ptr_list = NULL;
texture	**texture_ptr_list = NULL;
polygon	**polygon_ptr_list = NULL;
spoint *spoint_list;
int vertices = 0, textures = 0, polygons = 0, spoints = 0;

/***************************************************
 * Free all data related to the world description. *
 ***************************************************/

static void
free_world_data(void)
{
	int index, index2;

	if (vertex_ptr_list) {
		for (index = 0; index < vertices; index++) {
			vertex *vertex_ptr = vertex_ptr_list[index];
			if (vertex_ptr)
				free(vertex_ptr);
		}
		free(vertex_ptr_list);
		vertex_ptr_list = NULL;
		vertices = 0;
	}

	if (texture_ptr_list) {
		for (index = 0; index < textures; index++) {
			texture *texture_ptr = texture_ptr_list[index];
			if (texture_ptr) {
				if (texture_ptr->image_ptr)
					free(texture_ptr->image_ptr);
				free(texture_ptr);
			}
		}
		free(texture_ptr_list);
		texture_ptr_list = NULL;
		textures = 0;
	}

	if (polygon_ptr_list) {
	    for (index = 0; index < polygons; index++) {
		polygon *polygon_ptr = polygon_ptr_list[index];
		if (polygon_ptr) {
		    if (polygon_ptr->vertex_ptr_list)
			free(polygon_ptr->vertex_ptr_list);
		    if (polygon_ptr->tpoint_ptr_list) {
			for (index2 = 0; index2 < polygon_ptr->vertices;
			     index2++) {
			    tpoint *tpoint_ptr =
				polygon_ptr->tpoint_ptr_list[index2];
			    if (tpoint_ptr)
				free(tpoint_ptr);
			}
			free(polygon_ptr->tpoint_ptr_list);
		    }
		    free(polygon_ptr);
		}
	    }
	    free(polygon_ptr_list);
	    polygon_ptr_list = NULL;
	    polygons = 0;
	}

	if (spoint_list) {
		free(spoint_list);
		spoint_list = NULL;
		spoints = 0;
	}
}

/*************************************************************
 * Free all allocated rendering data related to window size. *
 *************************************************************/

void
reset_renderer(void)
{
	if (sine) {
		free(sine);
		sine = NULL;
	}
	if (cosine) {
		free(cosine);
		cosine = NULL;
	}
	if (fb_row) {
		free(fb_row);
		fb_row = NULL;
	}
	if (zb_row) {
		free(zb_row);
		zb_row = NULL;
	}
	if (frame_buffer) {
		destroy_frame_buffer();
		frame_buffer = NULL;
	}
	if (z_buffer) {
		free(z_buffer);
		z_buffer = NULL;
	}
}

/*****************************************************
 * Initialise renderer for a particular window size. *
 *****************************************************/

static void
initialise_renderer(void)
{
	int index;
	double angle;
	int row;
	fbpixel *fb_ptr;
	fixed *zb_ptr;

	system_message("Initialising renderer for window size of %d x %d "
		"pixels\n\n", window_width, window_height);

	/*
	 * Generate sine and cosine tables.
	 */
	sine = (fixed *)malloc((angles + 1) * sizeof(fixed));
	cosine = (fixed *)malloc((angles + 1) * sizeof(fixed));
	if (!sine || !cosine)
		memory_error("trig tables");
	for (index = 0, angle = 0.0; index <= angles; index++, angle += 1.0) {
		sine[index] = FIXED(sin(RAD(angle)));
		cosine[index] = FIXED(cos(RAD(angle)));
	}

	/*
	 * Determine min_px, max_px and delta_px; min_py, max_py, and delta_py;
	 * half_window_width and half_window_height.
	 */
	min_px = FIXED(-tan(RAD(field_of_view_x / 2.0)));
	max_px = FIXED(tan(RAD(field_of_view_x / 2.0)));
	delta_px = fdiv(max_px - min_px, FIXED(window_width));
	min_py = FIXED(-tan(RAD(field_of_view_y / 2.0)));
	max_py = FIXED(tan(RAD(field_of_view_y / 2.0)));
	delta_py = fdiv(max_py - min_py, FIXED(window_height));
	half_window_width = FIXED(window_width >> 1);
	half_window_height = FIXED(window_height >> 1);

	/*
	 * Allocate frame buffer.
	 */
	if ((frame_buffer = create_frame_buffer(window_width, window_height))
	    == NULL)
		memory_error("frame buffer");

	/*
	 * Construct a table of frame buffer row addresses.
	 */
	if ((fb_row = (fbpixel **)malloc(window_height * sizeof(fbpixel *)))
	    == NULL)
		memory_error("frame buffer row address table");
	for (row = 0, fb_ptr = frame_buffer; row < window_height;
	     row++, fb_ptr += window_width)
		fb_row[row] = fb_ptr;

	/*
	 * Allocate Z buffer.
	 */
	if ((z_buffer = (fixed *)malloc(window_width * window_height *
					sizeof(fixed))) == NULL)
		memory_error("Z buffer");

	/*
	 * Construct a table of Z buffer row addresses.
	 */
	if ((zb_row = (fixed **)malloc(window_height * sizeof(fixed *)))
	    == NULL)
		memory_error("Z buffer row address table");
	for (row = 0, zb_ptr = z_buffer; row < window_height;
	     row++, zb_ptr += window_width)
		zb_row[row] = zb_ptr;
}

/****************************
 * The input event handler. *
 ****************************/

void
handle_input_event(event_record *event)
{
	/*
	 * If a resize event is pending that changes the size of the
	 * window, process it.  If initialisation of the renderer fails
	 * because of low memory, we turn off the renderer.
	 */
	if (event->resize &&
	    (window_width != event->window_width ||
	     window_height != event->window_height)) {
		window_width = event->window_width;
		window_height = event->window_height;
		reset_renderer();
		if (!setjmp(context))
			initialise_renderer();
		else {
			renderer_on = FALSE;
			return;
		}
	}

	/*
	 * If a turn event is pending, process it.
	 */
	if (event->turn) {
		/*
		 * Determine the forces for the angular motion around the
		 * X and Y axes.
		 */
		switch(event->turn) {
		case TURNLEFT:
			ft = -TURN_RATE;
			fl = 0;
			break;
		case TURNRIGHT:
			ft = TURN_RATE;
			fl = 0;
			break;
		case LOOKUP:
			ft = 0;
			fl = -TURN_RATE;
			break;
		case LOOKDOWN:
			ft = 0;
			fl = TURN_RATE;
		}
		if (event->fast) {
			ft *= 2;
			fl *= 2;
		}
	} else {
		ft = 0;
		fl = 0;
	}

	/*
	 * If a move event is pending, process it.
	 */
	if (event->move) {
		/*
		 * Determine the forces for the X and Z directions.
		 */
		switch (event->move) {
		case FORWARD:
			fx = fmul(MOVE_RATE, sine[view_angle_y]);
			fz = fmul(MOVE_RATE, cosine[view_angle_y]);
			break;
		case BACKWARD:
			fx = -fmul(MOVE_RATE, sine[view_angle_y]);
			fz = -fmul(MOVE_RATE, cosine[view_angle_y]);
			break;
		case LEFT:
			fz = fmul(MOVE_RATE, sine[view_angle_y]);
			fx = -fmul(MOVE_RATE, cosine[view_angle_y]);
			break;
		case RIGHT:
			fz = -fmul(MOVE_RATE, sine[view_angle_y]);
			fx = fmul(MOVE_RATE, cosine[view_angle_y]);
		}
		if (event->fast) {
			fx *= 2;
			fz *= 2;
		}
	} else {
		fx = 0;
		fz = 0;
	}

	/*
	 * If a world load event is pending, process it.
	 */
	if (event->world_file_name) {
		free_world_data();
		if (!setjmp(context)) {
			read_world_file(event->world_file_name);
			renderer_on = TRUE;
		} else
			renderer_on = FALSE;
	}
}

/*************************
 * Render event handler. *
 *************************/

void
handle_render_event(void)
{
	/*
	 * If the renderer is off, do nothing.
	 */
	if (!renderer_on)
		return;

	/*
	 * Turning around can be 360 degrees, so make sure the turn
	 * angle doesn't get out of range.
	 */
	view_angle_y += ft;
	if (view_angle_y > angles)
		view_angle_y -= angles;
	if (view_angle_y < 0)
		view_angle_y += angles;

	/*
	 * Looking up and down is capped at -90 degrees and 90 degrees
	 * respectively i.e. you can't bend backwards and look upside
	 * down.
	 */
	view_angle_x += fl;
	if (view_angle_x > angles)
		view_angle_x -= angles;
	if (view_angle_x < 0)
		view_angle_x += angles;
	if (fl < 0) {
		if (view_angle_x > 90 && view_angle_x < 270)
			view_angle_x = 270;
	} else {
		if (view_angle_x > 90 && view_angle_x < 270)
			view_angle_x = 90;
	}


	/*
	 * Update the user's position.
	 */
	view_x += fx;
	view_z += fz;

	/*
	 * Render and display the frame.
	 */
	render_frame();
	display_frame_buffer();
	frames_rendered++;
}

/************************
 * Handle a quit event. *
 ************************/

void
handle_quit_event(void)
{
#ifndef	WIN32
	time_t end_time;
#endif

	reset_renderer();
	free_world_data();
	graphics_shutdown();

#ifndef	WIN32
	end_time = time(NULL) - start_time;

	printf("%s: Average frame rate was %f frames per second\n",
		program_name, (float)frames_rendered / (float)end_time);
#endif

	exit(0);
}

/************************
 * Main program section *
 ************************/

#ifndef	WIN32

static void
usage(void)
{
	fprintf(stderr, "Usage:\t%s [options]\n", program_name);
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "\t-x x-field-of-view (default %g)\n",
		DEFAULT_FIELD_OF_VIEW_X);
	fprintf(stderr, "\t-y y-field-of-view (default %g)\n",
		DEFAULT_FIELD_OF_VIEW_Y);
	fprintf(stderr, "\t-w window-width    (default %d)\n",
		DEFAULT_WINDOW_WIDTH);
	fprintf(stderr, "\t-h window-height   (default %d)\n",
		DEFAULT_WINDOW_HEIGHT);
	fprintf(stderr, "\t-s pixels-in-span  (default %d)\n",
		DEFAULT_PIXEL_SPAN);

	graphics_shutdown();
	exit(1);
}

int
main(int argc, char **argv)
{
	extern char *optarg;
	extern int optind;
	int c, temp_span;

#else

int WINAPI
WinMain(HINSTANCE instance, HINSTANCE previnst, LPSTR command_line,
	int window_state)
{

#endif

	char *env_str;

	/*
	 * Initialise the graphics library and dither arrays.
	 */
#ifndef	WIN32
	program_name = argv[0];
	graphics_startup(program_name, &display_width, &display_height,
		handle_input_event, handle_render_event, handle_quit_event);
#else
	graphics_startup(NULL, handle_input_event, handle_render_event,
			 handle_quit_event, instance);
#endif
	create_dither_arrays();

#ifndef	WIN32
	/*
	 * Parse command line arguments.
	 */
	while ((c = getopt(argc, argv, "x:y:w:h:s:")) != -1)
		switch (c) {
		case 'x':
			field_of_view_x = atof(optarg);
			if (field_of_view_x < 45.0 || field_of_view_x > 90.0) {
				fprintf(stderr,
					"%s: X field of view must be between "
					"45 and 90 degrees\n",
					program_name);
				exit(1);
			}
			break;
		case 'y':
			field_of_view_y = atof(optarg);
			if (field_of_view_y < 45.0 || field_of_view_y > 90.0) {
				fprintf(stderr,
					"%s: Y field of view must be between "
					"45 and 90 degrees\n",
					program_name);
				exit(1);
			}
			break;
		case 'w':
			window_width = atoi(optarg);
			if (window_width < 320 ||
			    window_width > display_width) {
				fprintf(stderr,
					"%s: window width must be between "
					"320 and %d pixels\n", program_name,
					display_width);
				exit(1);
			}
			break;
		case 'h':
			window_height = atoi(optarg);
			if (window_height < 200 || 
			    window_height > display_height) {
				fprintf(stderr,
					"%s: window height must be between "
					"200 and %d pixels\n", program_name,
					display_height);
				exit(1);
			}
			break;
		case 's':
			temp_span = atoi(optarg);
			if (temp_span <= 1) {
				fprintf(stderr, "%s: pixel span must be "
					"greater than 1\n", program_name);
				exit(1);
			}
			pixel_span = temp_span;
			span_shift = 0;
			while (!(temp_span & 1)) {
				span_shift++;
				temp_span = temp_span >> 1;
			}	
			if (temp_span > 1) {
				fprintf(stderr, "%s: pixel span must be "
					"a power of 2\n", program_name);
				exit(1);
			}
		}
	if (optind != argc)
		usage();

	/*
	 * Create the viewing window.
	 */
	create_window(window_width, window_height,
		display_width, display_height, VERSION_STR);
#else
	create_window(window_width, window_height, VERSION_STR, window_state);
#endif

	/*
	 * Print banner in text output window
	 */
	system_message(VERSION_STR);
	system_message("\nA 3D polygon rendering engine\n"
		"(C) Copyright 1996 by Philip J. Stephens\n"
		"All Rights Reserved\n\n");

	/*
	 * Get the file path list from the environment, if it exists, and
	 * construct a list of file paths from it.
	 */
	system_message("File path list:\n");
	if ((env_str = getenv("THREEDOM_PATH")) != NULL && (char)*env_str) {
		char *path_ptr;
		int index;

		/*
		 * First count the number of file paths present, and allocate
		 * an array of string pointers.
		 */
		paths = 0;
		for (path_ptr = env_str; *path_ptr; path_ptr++)
			if (*path_ptr == PATH_SEPARATOR_CHAR)
				paths++;
		paths++;
		if (!setjmp(context)) {
			if ((file_path_list = (char **)calloc(paths,
			    sizeof(char *))) == NULL)
				memory_error("file path list");
		}

		/*
		 * Next extract each file path.
		 */
		if (!setjmp(context)) {
			if ((file_path_str = (char *)malloc(strlen(env_str)+1))
			     == NULL)
				memory_error("file path string");
		}
		strcpy(file_path_str, env_str);
		file_path_list[0] = strtok(file_path_str, PATH_SEPARATOR_STR);
		system_message(".\n%s\n", file_path_list[0]);
		for (index = 1; index < paths; index++)	{
			file_path_list[index] = strtok(NULL, PATH_SEPARATOR_STR);
			system_message("%s\n", file_path_list[index]);
		}
		system_message("\n");
	} else
		system_message(".\n\n");

	/*
	 * Initialise the renderer, which is turned off by default.
	 */
	if (!setjmp(context))
		initialise_renderer();
	else
		reset_renderer();
	renderer_on = FALSE;

	/*
	 * Get the time of day, and save as the starting time; all
	 * time values used in this program will be computed relative
	 * to it.
	 */
	start_time = time(NULL);

	/*
	 * Enter the main event loop; this never returns (except under Windows
	 * 95).
	 */
	enter_event_loop();
	return(TRUE);
}
