/***********************************************************************/
/* COMM1.C - Commands A-D                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: comm1.c 2.0 1995/01/26 16:29:49 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*man-start*********************************************************************
COMMAND
     add - add blank line

SYNTAX
     Add [n]

DESCRIPTION
     The ADD command inserts the specified number of blank lines after
     the current_line (if issued from the command line) or after the
     focus_line (if issued from the FILEAREA or PREFIX areas).
     If SET NEWLINE is set to ALIGNED, the cursor is positioned in
     the column corresponding to the first column not containing a 
     space in the line above.
     If SET NEWLINE is set to LEFT, the cursor is positioned in the
     first column.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1

SEE ALSO
     SOS ADDLINE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Add(CHARTYPE *params)
#else
short Add(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#define ADD_PARAMS  1
 CHARTYPE *word[ADD_PARAMS+1];
 unsigned short num_params=0;
 LINETYPE num_lines=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Add");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be a positive integer greater than zero.           */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,ADD_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (CHARTYPE *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,word[1],FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (!valid_positive_integer(word[0]))
    {
     display_error(4,word[0],FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 num_lines = atol(word[0]);
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 insert_new_line((CHARTYPE *)"",0,num_lines,get_true_line(),FALSE,FALSE,CURRENT_VIEW->display_low);
 if (curses_started
 && CURRENT_VIEW->current_window == WINDOW_COMMAND)
    cursor_home(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     all - select and display restricted set of lines

SYNTAX
     ALL [rtarget]

DESCRIPTION
     The ALL command allows for the selective display, and editting
     (subject to SET SCOPE) of lines that match the specified target.
     This target consists of any number of individual targets
     seperated by '&' (logical and) or '|' (logical or). 
     For example, to display all lines in a file that contain the 
     strings 'ball' and 'cat' on the same line or the named lines 
     .fred or .bill, use the following command:

     ALL /ball/ & /cat/ | .fred | .bill

     Logical operators act left to right, with no precedence for &.

     ALL without any arguments, displays all lines in the file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET SCOPE, SET DISPLAY, SET SELECT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short All(CHARTYPE *params)
#else
short All(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINE *curr=NULL;
 bool target_found=FALSE,status=FALSE;
 short target_type=TARGET_NORMAL;
 TARGET target;
 LINETYPE line_number=0L;
 unsigned short x=0,y=0;
 bool save_scope=FALSE;
 LINETYPE num_lines=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   All");
#endif
 if (strlen(params) == 0)
   {
    curr = CURRENT_FILE->first_line->next;
    while(1)
      {
       curr->select = 0;
       curr = curr->next;
       if (curr->next == NULL)
          break;
      }
    CURRENT_VIEW->display_low = 0;
    CURRENT_VIEW->display_high = 0;
    build_current_screen(); 
    display_current_screen();
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Validate the parameters as valid targets...                         */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 rc = parse_target(params,get_true_line(),&target,target_type,TRUE,TRUE);
 if (rc != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Save the select levels for all lines in case no target is found.    */
/*---------------------------------------------------------------------*/
 curr = CURRENT_FILE->first_line->next;
 while(1)
   {
    curr->save_select = curr->select;
    curr = curr->next;
    if (curr->next == NULL)
       break;
   }
/*---------------------------------------------------------------------*/
/* Find all lines for the supplied target...                           */
/*---------------------------------------------------------------------*/
 curr = CURRENT_FILE->first_line;
 status = FALSE;
 save_scope = CURRENT_VIEW->scope_all;
 CURRENT_VIEW->scope_all = TRUE;
 for (line_number=0L;curr->next != NULL;line_number++)
   {
    status = find_rtarget_target(curr,&target,0L,line_number,&num_lines);
    if (status)
      {
       target_found = TRUE;
       curr->select = 1;
      }
    else
       curr->select = 0;
    curr = curr->next;
   }
/*---------------------------------------------------------------------*/
/* If at least one line matches the target, set DISPLAY to 1 1,        */
/* otherwise reset the select levels as they were before the command.  */
/*---------------------------------------------------------------------*/
 if (target_found)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    CURRENT_VIEW->display_low = 1;
    CURRENT_VIEW->display_high = 1;
    CURRENT_VIEW->scope_all = FALSE;
    CURRENT_VIEW->current_line = find_next_in_scope(CURRENT_FILE->first_line->next,1L,DIRECTION_FORWARD);
    build_current_screen(); 
    display_current_screen();
    CURRENT_VIEW->focus_line = calculate_focus_line(CURRENT_VIEW->focus_line,
                                                    CURRENT_VIEW->current_line);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
      {
       getyx(CURRENT_WINDOW,y,x);
       y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                                  CURRENT_VIEW->current_row);
       wmove(CURRENT_WINDOW,y,x);
      }
   }
  else
   {
    CURRENT_VIEW->scope_all = save_scope;
    curr = CURRENT_FILE->first_line->next;
    while(1)
      {
       curr->select = curr->save_select;
       curr = curr->next;
       if (curr->next == NULL)
          break;
      }
    display_error(17,params,FALSE);
    rc = RC_TARGET_NOT_FOUND;
   }
 free_target(&target);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     backward - scroll backwards [n] screens

SYNTAX
     BAckward [n|*]

DESCRIPTION
     The BACKWARD command scrolls the file contents backwards through
     the file [n|*] screens.

     If 0 is specified as the number of screens to scroll, the last
     line of the file becomes the current line. 
     If the BACKWARD command is issued while the current line is
     the "Top of File" line, the last line of the file becomes the 
     current line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Does not support HALF or Lines options.

DEFAULT
     1

SEE ALSO
     FORWARD, TOP

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Backward(CHARTYPE *params)
#else
short Backward(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#define BAC_PARAMS  1
 CHARTYPE *word[BAC_PARAMS+1];
 unsigned short num_params=0;
 LINETYPE num_pages=0L;
 unsigned short x=0,y=0;
 short rc=RC_OK;
 short direction=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Backward");
#endif
/*---------------------------------------------------------------------*/
/* Validate parameters...                                              */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,BAC_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (CHARTYPE *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,(CHARTYPE *)word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* If parameter is '*', set current line equal to "Top of File".       */
/*---------------------------------------------------------------------*/
 if (strcmp(word[0],"*") == 0)
   {
    rc = Top((CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If the parameter is not a valid integer, error.                     */
/*---------------------------------------------------------------------*/
 if (!valid_integer(word[0]))
   {
    display_error(1,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Number of screens to scroll is set here.                            */
/*---------------------------------------------------------------------*/
 num_pages = atol(word[0]);
/*---------------------------------------------------------------------*/
/* If the number specified is < 0, error...                            */
/*---------------------------------------------------------------------*/
 if (num_pages < 0L)
   {
    display_error(5,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the current line is already on "Top of File" or the parameter is */
/* 0, go to the bottom of the file.                                    */
/*---------------------------------------------------------------------*/
 if (num_pages == 0
 || CURRENT_TOF)
   {
    rc = Bottom((CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Scroll the screen num_pages...                                      */
/*---------------------------------------------------------------------*/
 rc = scroll_page(DIRECTION_BACKWARD,num_pages,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     bottom - move to the bottom of the file

SYNTAX
     BOTtom

DESCRIPTION
     The BOTTOM command moves to the very end of the current file.
     The last line of the file is set to the current line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     FORWARD, TOP

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Bottom(CHARTYPE *params)
#else
short Bottom(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
 extern bool in_macro;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Bottom");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (CURRENT_VIEW->scope_all)
    CURRENT_VIEW->current_line = CURRENT_FILE->number_lines;
 else
    CURRENT_VIEW->current_line = find_next_in_scope(CURRENT_FILE->last_line->prev,CURRENT_FILE->number_lines,DIRECTION_BACKWARD);
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen(); 
 if (!line_in_view(CURRENT_VIEW->focus_line))
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (!in_profile && !in_macro)
   {
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       getyx(PREVIOUS_WINDOW,y,x);
    else
       getyx(CURRENT_WINDOW,y,x);
    display_current_screen();
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       wmove(PREVIOUS_WINDOW,y,x);
    else
       wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     cancel - quit from all unaltered files in the ring

SYNTAX
     CANcel

DESCRIPTION
     The CANCEL command exits from THE quickly by executing a QQUIT
     command for every file in the ring that does not have any 
     outstanding alterations.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     CCANCEL

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Cancel(CHARTYPE *params)
#else
short Cancel(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_first;
 extern bool curses_started;
 extern CHARTYPE number_of_files;
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 VIEW_DETAILS *save_current_view=(VIEW_DETAILS *)NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cancel");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 CURRENT_VIEW = vd_first;
 while (CURRENT_VIEW != (VIEW_DETAILS *)NULL)
   {
    if (CURRENT_FILE->save_alt == 0)
       free_view_memory();
    else
      {
       save_current_view = CURRENT_VIEW;
       CURRENT_VIEW = CURRENT_VIEW->next;
      }
   }
 if (save_current_view != (VIEW_DETAILS *)NULL)
   {
    CURRENT_VIEW = save_current_view;
    CURRENT_SCREEN.screen_view = CURRENT_VIEW;
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    build_current_screen(); 
    display_current_screen();
    if (curses_started)
      {
       if (CURRENT_WINDOW_PREFIX != NULL)
          touchwin(CURRENT_WINDOW_PREFIX);
       if (CURRENT_WINDOW_COMMAND != NULL)
          touchwin(CURRENT_WINDOW_COMMAND);
       touchwin(CURRENT_WINDOW_MAIN);
       touchwin(CURRENT_WINDOW);
      }
   }
 if (number_of_files > 0)
   {
    sprintf(temp_cmd,"%d file(s) remain with outstanding changes",number_of_files);
    display_error(0,(CHARTYPE *)temp_cmd,TRUE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(QUIT);
}
/*man-start*********************************************************************
COMMAND
     ccancel - qquit from all files in the ring

SYNTAX
     CCancel

DESCRIPTION
     The CCANCEL command exits from THE quickly by executing the QQUIT
     command for every file in the ring. Any changes made to any of 
     the files will be lost.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     CANCEL

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Ccancel(CHARTYPE *params)
#else
short Ccancel(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_first;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Ccancel");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW = vd_first;
 while (CURRENT_VIEW != (VIEW_DETAILS *)NULL)
   {
    free_view_memory();
   }
#ifdef TRACE
 trace_return();
#endif
 return(QUIT);
}
/*man-start*********************************************************************
COMMAND
     change - change one string to another

SYNTAX
     Change /string1/string2/ [target] [n] [m]

DESCRIPTION
     The CHANGE command changes one string of text to another.

     The first parameter to the change command is the old and new
     string values, seperated by delimiters.
     The allowable delimiters are '/' '\' and '@'.

     The second parameter is the target; how many lines are to be
     searched for occurrences of the first string to be changed.

     The third parameter determines how many occurrences of 'string1'
     are to be changed on each line.

     The fourth parameter determines at which occurrences of 'string1'
     on the line are changes to commence.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1 1 1

SEE ALSO
     SCHANGE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Change(CHARTYPE *params)
#else
short Change(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Change");
#endif
 rc = execute_change_command(params,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     cmatch - find matching bracket character

SYNTAX
     cmatch 

DESCRIPTION
     The CMATCH command searches for the matching bracket character to
     the character under the cursor.

     It handles nested sets of matching pairs.
     The matching character pairs are '[]{}<>()'.

     This command can only be used by assigning it to a function key
     with the DEFINE command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Cmatch(CHARTYPE *params)
#else
short Cmatch(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
 static CHARTYPE *match = (CHARTYPE *)"[]{}<>()";
 unsigned short x=0,y=0;
 CHARTYPE ch=0,match_ch=0;
 register short i=0;
 short direction_backward=0;
 short matches=1,match_col=(-1),start_col=0;
 LINETYPE offset=0L;
 LINE *curr=NULL;
 WINDOW *w=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cmatch");
#endif
/*---------------------------------------------------------------------*/
/* This command only allowed to be issued from with the MAIN window.   */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window != WINDOW_MAIN
 || in_profile)
   {
    display_error(66,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* This command cannot be issued on TOF or BOF.                        */
/*---------------------------------------------------------------------*/
 if (FOCUS_TOF
 ||  FOCUS_BOF)
   {
    display_error(66,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_TOF_EOF_REACHED);
   }
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* This command cannot be entered on a shadow line.                    */
/*---------------------------------------------------------------------*/
 if (CURRENT_SCREEN.sl[y].line_type == LINE_SHADOW)
   {
    display_error(87,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_TARGET_NOT_FOUND);
   }
/*---------------------------------------------------------------------*/
/* Check if the character under the cursor is a valid match character. */
/*---------------------------------------------------------------------*/
 w = CURRENT_WINDOW;
 ch = (CHARTYPE)winch(w) & A_CHARTEXT;
 match_ch = 0;
 for (i=0;i<strlen(match);i++)
    if (ch == *(match+i))
      {
       direction_backward = (i % 2);
       match_ch = (direction_backward) ? *(match+i-1) : *(match+i+1);
       break;
      }
 if (match_ch == 0)
   {
    display_error(67,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Calculate the actual position of the character in the LINE.         */
/*---------------------------------------------------------------------*/
 start_col = CURRENT_VIEW->verify_col + x - 1;
 start_col += (direction_backward) ? (-1) : 1;
/*---------------------------------------------------------------------*/
/* Find the focus line linked list entry.                              */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_VIEW->focus_line);
 while (curr->next != NULL && curr->prev != NULL)
   {
    if (direction_backward)
      {
       for (i=start_col;i>(-1);i--)
          {
           if (*(curr->line+i) == ch)
             matches++;
           else
              if (*(curr->line+i) == match_ch)
                matches--;
           if (matches == 0)       /* found matching one */
             {
              match_col = i;
              break;
             }
          }
       if (match_col != (-1))
         break;
       curr = curr->prev;
       offset--;
       start_col = curr->length;
      }
    else
      {
       for (i=start_col;i<curr->length;i++)
          {
           if (*(curr->line+i) == ch)
             matches++;
           else
              if (*(curr->line+i) == match_ch)
                matches--;
           if (matches == 0)       /* found matching one */
             {
              match_col = i;
              break;
             }
          }
       if (match_col != (-1))
         break;
       curr = curr->next;
       offset++;
       start_col = 0;
      }
   }
/*---------------------------------------------------------------------*/
/* If no match found, return with error.                               */
/*---------------------------------------------------------------------*/
 if (match_col == (-1))  /* no match found */
   {
    display_error(68,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_TARGET_NOT_FOUND);
   }
/*---------------------------------------------------------------------*/
/* If we get here, we have found the matching character, so we have to */
/*  move the cursor to the new column and/or line.                     */
/*---------------------------------------------------------------------*/
 if (offset == 0L)
   {
    if (match_col >= CURRENT_VIEW->verify_col-1
    &&  match_col <= (CURRENT_SCREEN.cols[WINDOW_MAIN]+(CURRENT_VIEW->verify_col-1))-1)
/*---------------------------------------------------------------------*/
/* If the new cursor position is in the same panel and on the same line*/
/* just move the cursor there and get out.                             */
/*---------------------------------------------------------------------*/
      {
       wmove(CURRENT_WINDOW,y,match_col-(CURRENT_VIEW->verify_col-1));
#ifdef TRACE
       trace_return();
#endif
       return(RC_OK);
      }
   else
      {
       x = CURRENT_SCREEN.cols[WINDOW_MAIN] / 2;
       CURRENT_VIEW->verify_col = max(1,match_col-(short)x);
       build_current_screen(); 
       display_current_screen();
       wmove(CURRENT_WINDOW,y,(match_col-(CURRENT_VIEW->verify_col-1)));
#ifdef TRACE
       trace_return();
#endif
       return(RC_OK);
      }
   }
/*---------------------------------------------------------------------*/
/* If a match IS found on a different line, further checks are required*/
/* for SCOPE.                                                          */
/*---------------------------------------------------------------------*/
 if (in_scope(curr))
   {
/*---------------------------------------------------------------------*/
/* Set the cursor position for the matching character.                 */
/*---------------------------------------------------------------------*/
    CURRENT_VIEW->focus_line += offset;
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    if (line_in_view(CURRENT_VIEW->focus_line))
       y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                                  CURRENT_VIEW->current_row);
    else
      {
       CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
       y = CURRENT_VIEW->current_row;
      }
   }
 else
    if (CURRENT_VIEW->scope_all)
      {
       curr->select = CURRENT_VIEW->display_low;
       CURRENT_VIEW->focus_line += offset;
       CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
       pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
       y = CURRENT_VIEW->current_row;
      }
    else
      {
       display_error(68,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_TARGET_NOT_FOUND);
      }
#if 0
 if (offset + y < 0
 ||  offset + y >= CURRENT_SCREEN.rows)
   {
    CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
    y = CURRENT_VIEW->current_row;
   }
 else
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
#endif
 if (match_col >= CURRENT_VIEW->verify_col-1
 &&  match_col <= (CURRENT_SCREEN.cols[WINDOW_MAIN]+(CURRENT_VIEW->verify_col-1))-1)
    x = match_col-(CURRENT_VIEW->verify_col-1);
 else
   {
    x = CURRENT_SCREEN.cols[WINDOW_MAIN] / 2;
    CURRENT_VIEW->verify_col = max(1,match_col-(short)x);
    x = (match_col-(CURRENT_VIEW->verify_col-1));
   }

 build_current_screen(); 
 display_current_screen();
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     cmsg - display text on command line

SYNTAX
     CMSG [text]

DESCRIPTION
     The CMSG command, primarily used in macros, displays text on the
     command line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     EMSG, MSG

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Cmsg(CHARTYPE *params)
#else
short Cmsg(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *cmd_rec;
 extern LENGTHTYPE cmd_rec_len;
 extern bool clear_command;
 extern bool ETMODEx;
 register short i;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cmsg");
#endif
 memset(cmd_rec,' ',COLS);
 cmd_rec_len = strlen(params);
 memcpy(cmd_rec,params,cmd_rec_len);
 wmove(CURRENT_WINDOW_COMMAND,0,0);
 my_wclrtoeol(CURRENT_WINDOW_COMMAND);
/*---------------------------------------------------------------------*/
/* If the terminal is in ETMODE, display all characters as is, else    */
/* display message with translation of non-displaying characters.      */
/*---------------------------------------------------------------------*/
 if (ETMODEx)
   {
    for (i=0;i<cmd_rec_len;i++)
        mvwaddch(CURRENT_WINDOW_COMMAND,0,i,cmd_rec[i]);
   }
 else
    put_string(CURRENT_WINDOW_COMMAND,0,0,cmd_rec,cmd_rec_len);
 clear_command = FALSE;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     command - execute a command without translation

SYNTAX
     COMMAND command [options]

DESCRIPTION
     The COMMAND command executes the specified command without
     synonym or macro translation. THE does not attempt to execute 
     the command as a macro even if IMPMACRO is ON. The command will 
     be passed to the operating system if IMPOS is ON.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short THECommand(CHARTYPE *params)
#else
short THECommand(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   THECommand");
#endif
 rc = command_line(params,COMMAND_ONLY_TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     control_char - allow control characters to be entered

SYNTAX
     control_char

DESCRIPTION
     The CONTROL_CHAR command prompts the user to enter a control 
     character.

     This command can only be used by assigning it to a function key
     with the DEFINE command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Control_char(CHARTYPE *params)
#else
short Control_char(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 int key=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Control_char");
#endif
/*---------------------------------------------------------------------*/
/* If in the MAIN window, this command can only be issued on a real    */
/* line.                                                               */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    getyx(CURRENT_WINDOW,y,x);
    if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
      {
       display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
   }
 display_prompt("Press the character you require.");
 key = my_getch(CURRENT_WINDOW);
 clear_msgline();
 if (islower(key))
    key = toupper(key);
 if (key >= (int)'@'
 &&  key <= (int)'_')
   {
#ifdef TRACE
    trace_return();
#endif
    return((RAW_KEY*2)+(short)key-(short)'@');
   }
 display_error(69,(CHARTYPE *)"- must be between '@' and '_'",FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_INVALID_OPERAND);
}
/*man-start*********************************************************************
COMMAND
     copy - copies text from one position to another

SYNTAX
     COPY target1 target2
     COPY BLOCK [RESET]

DESCRIPTION
     With the first form of the COPY command, text is copied from the
     first target area to the line specified by target2. Text can
     only be copied within the same view of the file.

     The second form of the COPY command copies text within the
     currently marked block to the current cursor position.
     The text can be in the same file or a different file.

COMPATIBILITY
     XEDIT: COPY BLOCK not available.
     KEDIT: Adds extra functionality with [RESET] option.
            With the cursor in the marked block this command in KEDIT
            acts like DUPLICATE BLOCK.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Copy(CHARTYPE *params)
#else
short Copy(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 CHARTYPE reset_block=SOURCE_UNKNOWN;
 short rc=RC_OK;
 LINETYPE start_line=0L,end_line=0L,true_line=0L;
 VIEW_DETAILS *source_view=NULL,*dest_view=NULL;
 TARGET target1,target2;
 short target_type1=TARGET_NORMAL|TARGET_BLOCK_ANY|TARGET_ALL|TARGET_SPARE;
 short target_type2=TARGET_NORMAL;
 bool lines_based_on_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Copy");
#endif
 initialise_target(&target1);
 initialise_target(&target2);
 if ((rc = validate_target(params,&target1,target_type1,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target1);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If there is no second argument, the only valid target type for the  */
/* first argument then is BLOCK.                                       */
/*---------------------------------------------------------------------*/
 if (target1.spare == (-1))
   {
    if (target1.rt[0].target_type != TARGET_BLOCK_ANY
    &&  target1.rt[0].target_type != TARGET_BLOCK_CURRENT)
      {
       free_target(&target1);
       display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    else
       reset_block = SOURCE_BLOCK;
   }
 else
   {
    if (equal((CHARTYPE *)"reset",strtrunc(target1.rt[target1.spare].string),5))
       reset_block = SOURCE_BLOCK_RESET;
    else
       reset_block = SOURCE_COMMAND;
   }
/*---------------------------------------------------------------------*/
/* Validate the arguments following the target...                      */
/*---------------------------------------------------------------------*/
 switch(reset_block)
   {
    case SOURCE_BLOCK:
    case SOURCE_BLOCK_RESET:
/*---------------------------------------------------------------------*/
/* For box blocks, call the appropriate function...                    */
/*---------------------------------------------------------------------*/
         if (MARK_VIEW->mark_type != M_LINE)
           {
            free_target(&target1);
            box_operations(BOX_C,reset_block,FALSE,' ');
#ifdef TRACE
            trace_return();
#endif
            return(RC_OK);
           }
         source_view = MARK_VIEW;
         dest_view = CURRENT_VIEW;
         start_line = MARK_VIEW->mark_start_line;
         end_line = MARK_VIEW->mark_end_line;
         true_line = get_true_line();
         lines_based_on_scope = FALSE;
         break;
    default:
         if ((rc = validate_target(target1.rt[target1.spare].string,&target2,target_type2,get_true_line(),TRUE,TRUE)) != RC_OK)
           {
            free_target(&target2);
#ifdef TRACE
            trace_return();
#endif
            return(rc);
           }
         source_view = CURRENT_VIEW;
         dest_view = CURRENT_VIEW;
#if 0
         if (TOF(target1.true_line))
           {
            target1.true_line = 1L;
            target1.num_lines--;
           }
#endif
         start_line = target1.true_line;
         end_line = (target1.true_line + target1.num_lines) - 1L;
         true_line = target2.true_line + target2.num_lines;
         lines_based_on_scope = TRUE;
         break;
   }
 free_target(&target1);
 free_target(&target2);
/*---------------------------------------------------------------------*/
/* If the destination line for the copy is the *** Bottom of File ***  */
/* line, then subtract 1 to ensure lines don't get copied below the    */
/* *** Bottom of File *** line.                                        */
/*---------------------------------------------------------------------*/
 if (BOF(true_line))
    true_line--;
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 rc = rearrange_line_blocks(COMMAND_COPY,(CHARTYPE)reset_block,start_line,
                            end_line,true_line,1,source_view,dest_view,lines_based_on_scope);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     cursor - move cursor to specified position

SYNTAX
     CURsor Screen UP|DOWN|LEFT|RIGHT
     CURsor Screen row [col]
     CURsor [Escreen] UP|DOWN
     CURsor [Escreen|Kedit] LEFT|RIGHT
     CURsor [Escreen] row [col]
     CURsor CMdline [n]
     CURsor HOME [SAVE]

DESCRIPTION
     The CURSOR command allows the user to specify where the cursor
     is to be positioned.

     CURSOR SCREEN UP|DOWN|LEFT|RIGHT moves the cursor in the
     indicated direction one line or column. If the cursor is
     positioned on the first or last line of the screen, the cursor
     wraps to the first or last enterable lines. If the cursor is
     positioned on the left or right edges of the screen, the cursor
     moves to the left or right edge of the screen on the same line.

     CURSOR SCREEN row [col] moves the cursor to the specified
     row/col position within the file area. The top left corner of
     the file area is 1,1.
     row and col may be specified as '=', which will default to the
     current row and/or column position.
     If row or col are greater than the maximum number of rows or
     columns in the file area, the cursor will move to the last 
     row/column available.
     If the specified row is a reserved line, scale line or tab line
     an error will be displayed.
     If the row specified is above "Top of File" or below
     "Bottom of File" the cursor will be placed on the appropriate
     line.

     CURSOR [ESCREEN] UP|DOWN|LEFT|RIGHT is similar to CURSOR SCREEN
     UP|DOWN|LEFT|RIGHT, except that where scrolling of the window is
     possible, then scrolling will take place.

     CURSOR [ESCREEN] row [col] is similar to CURSOR SCREEN row [col],
     but all coordinates are relative the the top left corner of the
     logical window, not the top left corner of the file area. Hence,
     1,1 would be an invalid cursor position because it would result
     in the cursor being moved to the id line.
     Specification of row and/or column outside the boundaries of the
     logical window is regarded as an error.

     CURSOR KEDIT LEFT|RIGHT mimics the default behaviour of CURL and
     CURR in KEDIT.

     CURSOR CMDMLINE moves the cursor to the indicated column of the
     command line.

     CURsor HOME moves the cursor to the first column of the command
     line (if not on the command line), or to the last row/column of
     the file area if on the command line. With the [SAVE] option,
     the cursor will move to the last row/column of the file area or
     prefix area (which ever was the last position) if on the
     command line.


COMPATIBILITY
     XEDIT: Does not support COLUMN or FILE options.
     KEDIT: Does not support COLUMN or FILE options.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Cursor(CHARTYPE *params)
#else
short Cursor(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short idx=0;
#define CUR_PARAMS  3
 CHARTYPE *word[CUR_PARAMS+1];
 unsigned short num_params=0;
 bool time_to_leave=FALSE;
 bool escreen=FALSE;
 short error_number=1;
 CHARTYPE *error_message=(CHARTYPE *)"";
 short colno=1;
 short rc=RC_OK;
 short state=CURSOR_START;
 short row=0,col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cursor");
#endif
 num_params = param_split(params,word,CUR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params ==0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 error_message = word[0];
 state = CURSOR_START;
 idx = 0;
 while(1)
   {
    switch(state)
      {
       case CURSOR_START:
            if (equal("escreen",word[idx],1))
              {
               state = CURSOR_ESCREEN;
               idx++;
               break;
              }
            if (equal("screen",word[idx],1))
              {
               state = CURSOR_SCREEN;
               idx++;
               break;
              }
            if (equal("kedit",word[idx],1))
              {
               state = CURSOR_KEDIT;
               idx++;
               break;
              }
            if (equal("left",word[idx],4))
              {
               if (num_params > 1)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_left(TRUE,FALSE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("right",word[idx],5))
              {
               if (num_params > 1)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_right(TRUE,FALSE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("up",word[idx],2))
              {
               if (num_params > 1)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_up(TRUE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("down",word[idx],4))
              {
               if (num_params > 1)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_down(TRUE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("home",word[idx],4))
              {
               state = CURSOR_HOME;
               idx++;
               break;
              }
            if (equal("cmdline",word[idx],2))
              {
               state = CURSOR_CMDLINE;
               idx++;
               break;
              }
            if (equal("column",word[idx],1))
              {
               state = CURSOR_ERROR;
               error_message = (CHARTYPE *)"not yet implemented";
               error_number = 0;
               break;
              }
            if (equal("file",word[idx],1))
              {
               state = CURSOR_ERROR;
               error_message = (CHARTYPE *)"not yet implemented";
               error_number = 0;
               break;
              }
            state = CURSOR_ESCREEN;
            break;
       case CURSOR_HOME:
            if (num_params > 2)
              {
               state = CURSOR_ERROR;
               error_message = word[idx];
               error_number = 1;
               break;
              }
            if (num_params == 2
            &&  !equal("save",word[1],4))
              {
               state = CURSOR_ERROR;
               error_number = 1;
               error_message = word[idx];
               break;
              }
            if (num_params == 2)
               rc = cursor_home(TRUE);
            else
               rc = cursor_home(FALSE);
            time_to_leave = TRUE;
            break;
       case CURSOR_CMDLINE:
            if (num_params > 2)
              {
               state = CURSOR_ERROR;
               error_message = word[idx];
               error_number = 1;
               break;
              }
            if (num_params == 2)
              {
               colno = atoi(word[idx]);
               if (colno < 1)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
              }
            rc = cursor_cmdline(colno);
            time_to_leave = TRUE;
            break;
       case CURSOR_SCREEN:
       case CURSOR_ESCREEN:
            if (state == CURSOR_ESCREEN)
               escreen = TRUE;
            else
               escreen = FALSE;
            if (equal("left",word[idx],4))
              {
               if (num_params > 2)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_left(escreen,FALSE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("right",word[idx],5))
              {
               if (num_params > 2)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_right(escreen,FALSE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("up",word[idx],2))
              {
               if (num_params > 2)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_up(escreen);
               time_to_leave = TRUE;
               break;
              }
            if (equal("down",word[idx],4))
              {
               if (num_params > 2)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_down(escreen);
               time_to_leave = TRUE;
               break;
              }
             /* validate numbers */
            if (strcmp(word[idx],"=") == 0)
               row = 0;
            else
              {
               if (!valid_positive_integer(word[idx]))
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 4;
                  break;
                 }
               row = atoi(word[idx]);
              }
            idx++;
            if (strcmp(word[idx],"") == 0)
              {
               if (escreen)
                  col = 1;
               else
                  col = (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT ? PREFIX_WIDTH + 1 : 1;
              }
            else
              {
               if (strcmp(word[idx],"=") == 0)
                  col = 0;
               else
                 {
                  if (!valid_positive_integer(word[idx]))
                    {
                     state = CURSOR_ERROR;
                     error_message = word[idx];
                     error_number = 4;
                     break;
                    }
                  col = atoi(word[idx]);
                 }
              }
            rc = cursor_move(escreen,row,col);
            time_to_leave = TRUE;
            break;
       case CURSOR_KEDIT:
            if (equal("left",word[idx],4))
              {
               if (num_params > 2)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_left(TRUE,TRUE);
               time_to_leave = TRUE;
               break;
              }
            if (equal("right",word[idx],5))
              {
               if (num_params > 2)
                 {
                  state = CURSOR_ERROR;
                  error_message = word[idx];
                  error_number = 1;
                  break;
                 }
               rc = cursor_right(TRUE,TRUE);
               time_to_leave = TRUE;
               break;
              }
            state = CURSOR_ERROR;
            error_message = word[idx];
            error_number = 1;
            break;
       case CURSOR_ERROR:
            display_error(error_number,error_message,FALSE);
            rc = RC_INVALID_OPERAND;
            time_to_leave = TRUE;
            break;
      }
    if (time_to_leave)
       break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     define - assign one or many commands to a key

SYNTAX
     DEFine key-name [command [args] [[#command [args]...]]]

DESCRIPTION
     The DEFINE command allows the user to assign one or many 
     commands and optional parameter(s) to a key. 

     Commands may be abbreviated.

     If multiple commands are assigned, then the LINEND setting
     must be ON and the LINEND character must match the character
     that delimits the commands at the time that the DEFINE command
     is executed. LINEND can be OFF at the time the key is pressed.

     With no arguments, any existing definition for that key is
     removed and the key reverts back to its default assignation (if
     it had any).

     key-names correspond to the key name shown with the SHOW command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Minimal. No support for in-memory macro commands.
            KEDIT does not allow multiple commands except as KEXX
            macros.

SEE ALSO
     SHOW

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Define(CHARTYPE *params)
#else
short Define(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define DEF_PARAMS  2
 CHARTYPE *word[DEF_PARAMS+1];
 unsigned short num_params=0;
 int key_value=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Define");
#endif
 num_params = param_split(params,word,DEF_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params ==0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* The first parameter is the key name mnemonic , the next is one or   */
/* more commands and/or parameters.                                    */
/* First check the mnemonic for decimal string value. ie begins with \ */
/*---------------------------------------------------------------------*/
 if (word[0][0] == '\\')
   {
    if ((key_value = atoi(word[0]+1)) == 0)
      {
       display_error(13,word[0],FALSE);
       rc = RC_INVALID_OPERAND;
      }
   }
 else
   {
    if ((key_value = find_key_value(word[0])) == (-1))
      {
       display_error(13,word[0],FALSE);
       rc = RC_INVALID_OPERAND;
       }
   }
 if (rc == RC_OK)
    rc = add_define(key_value,word[1]);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     delete - delete lines from a file

SYNTAX
     DELete [target]

DESCRIPTION
     The DELETE command removes lines from the current file.
     The number of lines removed depends on the target specified.
     Lines are removed starting with the focus line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1

SEE ALSO
     SOS DELLINE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short DeleteLine(CHARTYPE *params)
#else
short DeleteLine(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 LINETYPE start_line=0L,end_line=0L,dest_line=0L;
 short rc=RC_OK;
 CHARTYPE *args=NULL;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_ALL|TARGET_BLOCK_CURRENT;
 bool lines_based_on_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   DeleteLine");
#endif
/*---------------------------------------------------------------------*/
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 if (blank_field(params))
    args = (CHARTYPE *)"1";
 else
    args = params;
 initialise_target(&target);
 if ((rc = validate_target(args,&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If the target is BLOCK and the marked block is a box block, call    */
/* box_operations(), otherwise delete specified lines.                 */
/*---------------------------------------------------------------------*/
 if (target.rt[0].target_type == TARGET_BLOCK_CURRENT)
   {
/*---------------------------------------------------------------------*/
/* For box blocks, call the appropriate function...                    */
/*---------------------------------------------------------------------*/
    if (MARK_VIEW->mark_type != M_LINE)
      {
       free_target(&target);
       box_operations(BOX_D,SOURCE_BLOCK_RESET,FALSE,' ');
#ifdef TRACE
       trace_return();
#endif
       return(RC_OK);
      }
    start_line = MARK_VIEW->mark_start_line;
    end_line = MARK_VIEW->mark_end_line;
    dest_line = MARK_VIEW->mark_start_line;
    lines_based_on_scope = FALSE;
   }
 else
   {
    start_line = target.true_line;
    if (target.num_lines < 0L)
      {
       end_line = (target.true_line + target.num_lines) + 1L;
       dest_line = end_line;
      }
    else
      {
       end_line = (target.true_line + target.num_lines) - 1L;
       dest_line = start_line;
      }
    lines_based_on_scope = TRUE;
   }
 free_target(&target);
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (target.num_lines != 0L)
    rc = rearrange_line_blocks(COMMAND_DELETE,SOURCE_COMMAND,start_line,
                            end_line,dest_line,1,CURRENT_VIEW,CURRENT_VIEW,lines_based_on_scope);
 CURRENT_VIEW->current_line = find_next_in_scope(NULL,CURRENT_VIEW->current_line,DIRECTION_FORWARD);
 CURRENT_VIEW->focus_line = find_next_in_scope(NULL,CURRENT_VIEW->focus_line,DIRECTION_FORWARD);
 if (rc == RC_OK)
   {
    if (CURRENT_BOF || CURRENT_TOF)
       rc = RC_TOF_EOF_REACHED;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     directory - list the specified directory as an editable file

SYNTAX
     DIRectory [filespec]

DESCRIPTION
     The DIRECTORY command displays all files matching the specified 
     file specification.
     When no parameter is supplied, all files in the current directory 
     are displayed subject to any SET DIRINCLUDE restrictions.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     LS, SET DIRINCLUDE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Directory(CHARTYPE *params)
#else
short Directory(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *temp_cmd;
 extern CHARTYPE dir_filename[10];
 extern CHARTYPE dir_pathname[MAX_FILE_NAME+1];
/*--------------------------- local data ------------------------------*/
#define DIR_PARAMS  1
 CHARTYPE *word[DIR_PARAMS+1];
 unsigned short num_params=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Directory");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be the directory to display.                       */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,DIR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params > 1)
   {
    display_error(1,(CHARTYPE *)word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate that the supplied directory is valid.                      */
/*---------------------------------------------------------------------*/
 if ((rc = splitpath(strtrans(word[0],OSLASH,ISLASH))) != RC_OK)
   {
    display_error(10,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if ((rc = read_directory()) != RC_OK)
   {
    display_error(10,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 strcpy(temp_cmd,dir_pathname);
 strcat(temp_cmd,dir_filename);
 Xedit(temp_cmd);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     dos - execute an operating system command

SYNTAX
     DOS [command]

DESCRIPTION
     The DOS command executes the supplied operating system command 
     or runs an interactive shell if no command is supplied.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     OS, !

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     dosnowait - execute an operating system command - no prompt

SYNTAX
     DOSNowait command

DESCRIPTION
     The DOSNOWAIT command executes the supplied operating system 
     command not waiting for the user to be prompted once the
     command has completed.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     OSNOWAIT

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     dosquiet - execute an operating system command quietly

SYNTAX
     DOSQuiet command

DESCRIPTION
     The OSQUIET command executes the supplied operating system command 
     as quietly as possible.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     OSQUIET

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     down - move forward in the file a number of lines

SYNTAX
     Down [relative_target]

DESCRIPTION
     The DOWN command moves the current line forwards the number of
     lines specified by the relative_target. This relative_target can 
     only be a positive integer or the character "*". 

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1

SEE ALSO
     NEXT, UP

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     duplicate - duplicate lines

SYNTAX
     DUPlicate [n [target|BLOCK]]

DESCRIPTION
     The DUPLICATE command copies the number of lines extrapolated from
     target, n times.

COMPATIBILITY
     XEDIT: Equivalent of DUPLICAT command.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Duplicate(CHARTYPE *params)
#else
short Duplicate(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
#define DUP_PARAMS  2
 CHARTYPE *word[DUP_PARAMS+1];
 unsigned short num_params=0;
 short rc=RC_OK,num_occ=0;
 LINETYPE start_line=0L,end_line=0L,dest_line=0L;
 CHARTYPE command_source=0;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 bool lines_based_on_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Duplicate");
#endif
 num_params = param_split(params,word,DUP_PARAMS,WORD_DELIMS,TEMP_PARAM);
/*---------------------------------------------------------------------*/
/* If no parameters, default to 1 1                                    */
/*---------------------------------------------------------------------*/
 if (num_params == 0)
   {
    word[0] = (CHARTYPE *)"1";
    word[1] = (CHARTYPE *)"1";
   }
/*---------------------------------------------------------------------*/
/* If 1 parameter, default 2nd parameter to 1                          */
/*---------------------------------------------------------------------*/
 if (num_params == 1)
    word[1] = (CHARTYPE *)"1";
/*---------------------------------------------------------------------*/
/* If first parameter is not an integer, error.                        */
/*---------------------------------------------------------------------*/
 if (!valid_integer(word[0]))
   {
    display_error(4,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 num_occ = atoi(word[0]);
/*---------------------------------------------------------------------*/
/* Validate second parameter is a valid target...                      */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 if ((rc = validate_target(word[1],&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Duplicate lines depending on target type...                         */
/*---------------------------------------------------------------------*/
 switch(target.rt[0].target_type)
   {
    case TARGET_BLOCK_CURRENT:
/*---------------------------------------------------------------------*/
/* This function not valid for box  blocks.                            */
/*---------------------------------------------------------------------*/
         if (MARK_VIEW->mark_type == M_BOX)
           {
            display_error(48,(CHARTYPE *)"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         command_source = SOURCE_BLOCK;
         start_line = MARK_VIEW->mark_start_line;
         end_line = dest_line = MARK_VIEW->mark_end_line;
         lines_based_on_scope = FALSE;
         break;
    default:
         command_source = SOURCE_COMMAND;
         start_line = target.true_line;
         end_line = dest_line = (target.true_line + target.num_lines) - 1L;
         lines_based_on_scope = TRUE;
         break;
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 rc = rearrange_line_blocks(COMMAND_DUPLICATE,command_source,
                            start_line,end_line,dest_line,num_occ,
                            CURRENT_VIEW,CURRENT_VIEW,lines_based_on_scope);
 free_target(&target);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
