/***********************************************************************/
/* COMM2.C - Commands E-J                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: comm2.c 2.0 1995/01/26 16:29:54 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*man-start*********************************************************************
COMMAND
     edit - edit another file or switch to next file

SYNTAX
     Edit [filename]

DESCRIPTION
     The EDIT command allows the user to edit another file. The new file
     is placed in the file ring. The previous file being edited remains
     in memory and can be returned to by issuing an EDIT command without
     any parameters. Several files can be edited at once, and all files
     are arranged in a ring, with subsequent EDIT commands moving through
     the ring, one file at a time.

COMPATIBILITY
     XEDIT: Does not provide options switches.
     KEDIT: Does not provide options switches.

SEE ALSO
     THE, XEDIT

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     emsg - display message

SYNTAX
     EMSG [message]

DESCRIPTION
     The EMSG command displays an error message on the error line.
     This command is usually issued from a macro file.

COMPATIBILITY
     XEDIT: Does not support [mmmnnns text] option
     KEDIT: Compatible

SEE ALSO
     CMSG, MSG

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Emsg(CHARTYPE *params)
#else
short Emsg(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Emsg");
#endif
 display_error(0,params,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     enter - execute a command

SYNTAX
     enter

DESCRIPTION
     The ENTER command executes the command currently displayed on the
     command line, if the cursor is currently displayed there.
     If the key associated with ENTER is pressed while in the FILEAREA,
     then the cursor will move to the first column of the
     next line. If the cursor is in the PREFIX area, any pending 
     prefix commands will be executed. If the mode is currently in 
     'insert', then a new line is added and the cursor placed on the
     next line depending on the value of SET NEWLINE.

     This command can only be used by assigning it to a function key
     with the DEFINE command.

     This command will be removed in a future version.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     SOS EXECUTE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Enter(CHARTYPE *params)
#else
short Enter(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool INSERTMODEx;
 extern bool readonly;
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Enter");
#endif
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         rc = Sos_execute("");
         break;
    case WINDOW_PREFIX:
         post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         if (CURRENT_FILE->first_ppc == NULL)/* no pending prefix cmds */
           {
            cursor_down(TRUE);
            rc = Sos_firstcol("");
           }
         else
            Sos_doprefix("");
         break;
    case WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* If in readonly mode, scroll down 1 line...                          */
/*---------------------------------------------------------------------*/
         if (readonly)
           {
            cursor_down(TRUE);
            getyx(CURRENT_WINDOW,y,x);
            wmove(CURRENT_WINDOW,y,0);
            break;
           }
         if (INSERTMODEx)
           {
            post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
            insert_new_line((CHARTYPE *)"",0,1,get_true_line(),FALSE,FALSE,CURRENT_VIEW->display_low);
            break;
           }
         cursor_down(TRUE);
         getyx(CURRENT_WINDOW,y,x);
         wmove(CURRENT_WINDOW,y,0);
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     expand - expand tab characters to spaces

SYNTAX
     EXPand [target]

DESCRIPTION
     The EXPAND command converts all tab characters to spaces depending
     on the size of a tab determined by the SET TABS command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET TABS

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Expand(CHARTYPE *params)
#else
short Expand(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Expand");
#endif
 rc = execute_expand(params,TRUE,TRUE,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     extract - obtain various internal information about THE

SYNTAX
     EXTract /item/[...]

DESCRIPTION
     The EXTRACT command is used to relay information about settings
     within THE from within a REXX macro. EXTRACT is only valid within
     a REXX macro.

COMPATIBILITY
     XEDIT: Only allows '/' as delimiter.
     KEDIT: Only allows '/' as delimiter.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Extract(CHARTYPE *params)
#else
short Extract(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern bool rexx_support;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short rc=RC_OK,itemno=0,num_items=0,len=0,num_values=0;
 short pos=0;
 CHARTYPE *args=NULL;
 bool invalid_item=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Extract");
#endif
 if (!in_macro
 ||  !rexx_support)
   {
    display_error(53,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Check that the first character of parameters is /.                  */
/*---------------------------------------------------------------------*/
 if (*(params) != '/')
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 params++;                               /* throw away first delimiter */
 len = strlen(params);
/*---------------------------------------------------------------------*/
/* Check that we have an item to extract...                            */
/*---------------------------------------------------------------------*/
 if (len == 0)
    invalid_item = TRUE;
 else
    if (len == 1 && (*(params) == '/'))
       invalid_item = TRUE;
 if (invalid_item)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Allow for no trailing '/'...                                        */
/*---------------------------------------------------------------------*/
 if ((*(params+len-1) == '/'))
    num_items = 0;
 else
    num_items = 1;
/*---------------------------------------------------------------------*/
/* Replace all / with nul character to give us seperate strings.       */
/*---------------------------------------------------------------------*/
 for (i=0;i<len;i++)
   {
    if (*(params+i) == '/')
      {
       *(params+i) = '\0';
       num_items++;
      }
   }
/*---------------------------------------------------------------------*/
/* For each item, extract its variables...                             */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_items;i++)
   {
/*---------------------------------------------------------------------*/
/* First check if the item has any arguments with it.                  */
/*---------------------------------------------------------------------*/
    pos = strzeq(params,' ');
    if (pos == (-1))
       args = (CHARTYPE *)"";
    else
      {
       *(params+pos) = '\0';
       args = strtrunc(params+pos+1);
      }
/*---------------------------------------------------------------------*/
/* Find the item in the list of valid extract options...               */
/*---------------------------------------------------------------------*/
    if ((itemno = find_item(params,QUERY_EXTRACT)) == (-1))
       {
        display_error(1,params,FALSE);
#ifdef TRACE
        trace_return();
#endif
        return(RC_INVALID_OPERAND);
       }
/*---------------------------------------------------------------------*/
/* Get the current settings for the valid item...                      */
/*---------------------------------------------------------------------*/
    num_values = get_item_values(itemno,args,QUERY_EXTRACT,0L,NULL,0L);
/*---------------------------------------------------------------------*/
/* If the arguments to the item are invalid, return with an error.     */
/*---------------------------------------------------------------------*/
    if (num_values == EXTRACT_ARG_ERROR)
       {
#ifdef TRACE
        trace_return();
#endif
        return(RC_INVALID_OPERAND);
       }
/*---------------------------------------------------------------------*/
/* If the REXX variables have already been set, don't try to set them. */
/*---------------------------------------------------------------------*/
    if (num_values != EXTRACT_VARIABLES_SET)
      {
       rc = set_extract_variables(itemno);
       if (rc == RC_SYSTEM_ERROR)
          break;
      }
    params += strlen(params)+1;
   }

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ffile - force a FILE of the current file to disk

SYNTAX
     FFile  [filename]

DESCRIPTION
     The FFILE command writes the current file to disk to the current
     file name or to the supplied filename.
     Unlike the FILE command, if the optional filename exists, this
     command will overwrite the file.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     With no parameters, the current file is written.

SEE ALSO
     FILE, SAVE, SSAVE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Ffile(CHARTYPE *params)
#else
short Ffile(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Ffile");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if ((rc = save_file(CURRENT_FILE,params,TRUE,CURRENT_FILE->number_lines,1L,FALSE,0,max_line_length,TRUE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of FFiling, remove the .aus file...   */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->autosave > 0)
    rc = remove_aus_file(CURRENT_FILE);
 free_view_memory();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     file - write the current file to disk and remove from ring

SYNTAX
     FILE  [filename]

DESCRIPTION
     The FILE command writes the current file to disk to the current
     file name or to the supplied filename.
     Unlike the FFILE command, if the optional filename exists, this
     command will not overwrite the file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     With no parameters, the current file is written.

SEE ALSO
     FFILE, SAVE, SSAVE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short File(CHARTYPE *params)
#else
short File(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   File");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* If we are filing the current file with the same name AND the number */
/* of alterations is zero, then quit the file.                         */
/* Removed to be consistant with XEDIT/KEDIT.                          */
/*---------------------------------------------------------------------*/
/*
 if (CURRENT_FILE->save_alt == 0 && strcmp(params,"") == 0)
    Quit((CHARTYPE *)"");
 else
*/
   {
    if ((rc = save_file(CURRENT_FILE,params,FALSE,CURRENT_FILE->number_lines,1L,FALSE,0,max_line_length,TRUE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of Filing, remove the .aus file...    */
/*---------------------------------------------------------------------*/
    if (CURRENT_FILE->autosave > 0)
       rc = remove_aus_file(CURRENT_FILE);
    free_view_memory();
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     fillbox - fill the marked box block with a character

SYNTAX
     FILLbox [c]

DESCRIPTION
     The FILLBOX command fills the marked block with the specified
     character. If no parameters are supplied and the command is run
     from the command line, then the box will be filled with spaces.
     If the command is not run from the command line, the user is
     prompted for a character to fill the box.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Fillbox(CHARTYPE *params)
#else
short Fillbox(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 CHARTYPE chr=0;
 short len_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Fillbox");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Validate the marked block.                                          */
/*---------------------------------------------------------------------*/
 if (marked_block(TRUE) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* This function not valid for line blocks.                            */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW->mark_type == M_LINE)
   {
    display_error(47,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Check if hex on in effect and translate hex char if required...     */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->hex)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen(params);
/*---------------------------------------------------------------------*/
/* Whew, now do something...                                           */
/*---------------------------------------------------------------------*/
 if (len_params > 1)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (len_params == 0)
    chr = ' ';
 else
    chr = *(params);
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND
 && len_params != 1) 
   {
    display_prompt("Enter fill character...");
    chr = (CHARTYPE)my_getch(stdscr);
    clear_msgline();
   }
 box_operations(BOX_F,SOURCE_BLOCK,TRUE,chr);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     forward - scroll forwards [n] screens

SYNTAX
     FOrward [n]

DESCRIPTION
     The FORWARD command scrolls the file contents forwards the number
     of screens specified.

     If 0 is specified as the number of screens to scroll, the "Top
     of File" line becomes the current line. 
     If the FORWARD command is issued while the current line is the 
     "Bottom of File" line, the "Top of File" line becomes the 
     current line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Does not support HALF or Lines options.

DEFAULT
     1

SEE ALSO
     BACKWARD, TOP

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Forward(CHARTYPE *params)
#else
short Forward(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
 extern bool in_macro;
/*--------------------------- local data ------------------------------*/
#define FOR_PARAMS  1
 CHARTYPE *word[FOR_PARAMS+1];
 unsigned short num_params=0;
 LINETYPE num_pages=0L;
 unsigned short x=0,y=0;
 short rc=RC_OK;
 short direction=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Forward");
#endif
/*---------------------------------------------------------------------*/
/* Validate parameters...                                              */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,FOR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (CHARTYPE *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,(CHARTYPE *)word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* If parameter is '*', set current line equal to last line in file... */
/*---------------------------------------------------------------------*/
 if (strcmp(word[0],"*") == 0)
   {
    rc = Bottom((CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If the parameter is not a valid integer, error.                     */
/*---------------------------------------------------------------------*/
 if (!valid_integer(word[0]))
   {
    display_error(1,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Number of screens to scroll is set here.                            */
/*---------------------------------------------------------------------*/
 num_pages = atol(word[0]);
/*---------------------------------------------------------------------*/
/* If the number specified is < 0, error...                            */
/*---------------------------------------------------------------------*/
 if (num_pages < 0L)
   {
    display_error(5,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the current line is already on "Bottom of File" or the parameter */
/* is 0, go to the top of the file.                                    */
/*---------------------------------------------------------------------*/
 if (num_pages == 0
 || CURRENT_BOF)
   {
    rc = Top((CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Scroll the screen num_pages...                                      */
/*---------------------------------------------------------------------*/
 rc = scroll_page(DIRECTION_FORWARD,num_pages,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     get - insert into file the contents of specified file

SYNTAX
     GET [fileid]

DESCRIPTION
     The GET command reads a file into the current file, inserting
     lines after the current line.
     When no fileid is supplied the temporary file generated by the
     PUT command is used.

COMPATIBILITY
     XEDIT: Does not support optional firstrec - numrecs
     KEDIT: Does not support optional fromlines - forlines.

SEE ALSO
     PUT, PUTD

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Get(CHARTYPE *params)
#else
short Get(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *tempfilename;
 extern CHARTYPE *temp_cmd;
 extern CHARTYPE sp_path[MAX_FILE_NAME+1] ;
 extern CHARTYPE sp_fname[MAX_FILE_NAME+1] ;
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
 CHARTYPE *filename=NULL;
 FILE *fp=NULL;
 LINE *curr=NULL;
 LINETYPE old_number_lines=0L,true_line=0L;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Get");
#endif
 if (strcmp(params,"") == 0)   /* no fileid supplied */
    filename = tempfilename;
 else
   {
    if ((rc = splitpath(strtrans(params,OSLASH,ISLASH))) != RC_OK)
      {
       display_error(10,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    strcpy(temp_cmd,sp_path);
    strcat(temp_cmd,sp_fname);
    filename = temp_cmd;
   }

 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);

 if ((fp = fopen(filename,"r")) == NULL)
   {
    display_error(9,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_ACCESS_DENIED);
   }
 true_line = get_true_line();
 curr = lll_find(CURRENT_FILE->first_line,true_line);
 if (curr->next == NULL)   /* on bottom of file */
    curr = curr->prev;
 old_number_lines = CURRENT_FILE->number_lines;
 if ((curr = read_file(fp,curr,filename)) == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_ACCESS_DENIED);
   }

 fclose(fp);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Fix the positioning of the marked block (if there is one and it is  */
/* in the current view).                                               */
/*---------------------------------------------------------------------*/
 if (!in_profile)
   {
    adjust_marked_lines(TRUE,true_line,CURRENT_FILE->number_lines - old_number_lines);
    adjust_pending_prefix(CURRENT_VIEW,TRUE,true_line,CURRENT_FILE->number_lines - old_number_lines);
   }
/*---------------------------------------------------------------------*/
/* Increment the number of lines counter for the current file and the  */
/* number of alterations.                                              */
/*---------------------------------------------------------------------*/
 if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 build_current_screen(); 
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     help - edit help file for THE

SYNTAX
     HELP

DESCRIPTION
     The HELP command displays help for the editor.
     Uses THE_HELP_FILE environment variable to point to the help file.

COMPATIBILITY
     XEDIT: Similar in concept.
     KEDIT: Similar in concept.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Help(CHARTYPE *params)
#else
short Help(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE the_help_file[MAX_FILE_NAME+1];
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Help");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!file_exists(the_help_file))
   {
    display_error(23,(CHARTYPE *)the_help_file,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_FILE_NOT_FOUND);
   }
 Xedit(the_help_file);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     input - insert the command line contents into the file

SYNTAX
     Input [line contents]

DESCRIPTION
     The INPUT command inserts the remainder of the command line into the
     file after the current line.

COMPATIBILITY
     XEDIT: Does not provide full input mode option.
     KEDIT: Does not provide full input mode option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Input(CHARTYPE *params)
#else
short Input(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short len_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Input");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (CURRENT_VIEW->hex)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen(params);
 insert_new_line(params,len_params,1L,get_true_line(),TRUE,TRUE,CURRENT_VIEW->display_low);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     join - join a line with the line following

SYNTAX
     Join [ALigned]

DESCRIPTION
     The JOIN command makes one line out of the focus line and the
     line following.
     If ALIGNED is specified, any leading spaces in the following line
     are ignored. If ALIGNED is not specified, all characters, including
     spaces are added.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SPLIT, SPLTJOIN

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Join(CHARTYPE *params)
#else
short Join(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short x=0,y=0,col=0;
 bool aligned=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Join");
#endif
/*---------------------------------------------------------------------*/
/* Check here for parameter value of 'Aligned'.                        */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"aligned",params,2))
    aligned = TRUE;
 else
    if (strcmp(params,"") == 0)
       aligned = FALSE;
    else
      {
       display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
 if (curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    col = (x+CURRENT_VIEW->verify_col-1);
   }
 else
    col = 0;
 rc = execute_split_join(FALSE,aligned,col);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
