/***********************************************************************/
/* COMM3.C - Commands K-O                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: comm3.c 2.0 1995/01/26 16:29:56 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*man-start*********************************************************************
COMMAND
     left - scroll the screen to the left

SYNTAX
     LEft [n|HALF]

DESCRIPTION
     The LEFT command scrolls the screen n columns to the left.
     If no parameter is supplied, the screen is scrolled by one
     column. If HALF is specified the screen is scrolled by half 
     the number of columns in the FILEAREA.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     RIGHT, RGTLEFT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Left(CHARTYPE *params)
#else
short Left(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINETYPE shift_val=1L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Left");
#endif
/*---------------------------------------------------------------------*/
/* Validate only parameter, HALF or positive integer. 1 if no argument.*/
/*---------------------------------------------------------------------*/
 if (equal(params,(CHARTYPE *)"half",4))
    shift_val = -(CURRENT_SCREEN.cols[WINDOW_MAIN]/2);
 if (blank_field(params))
    shift_val = (-1L);
 if (shift_val == 1)                 /* argument not HALF or empty ... */
   {
    if (valid_positive_integer(params))
      {
       shift_val = atol(params);
       if (shift_val != 0)
          shift_val = -shift_val;
      }
   }
 if (shift_val == 1)                               /* invalid argument */
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the argument is 0, restore the original verify columns display.  */
/*---------------------------------------------------------------------*/
 if (shift_val == 0L)
    CURRENT_VIEW->verify_col = CURRENT_VIEW->verify_start;
 else
    CURRENT_VIEW->verify_col = max(1,CURRENT_VIEW->verify_col+shift_val);
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     locate - search for a target

SYNTAX
     [Locate] target [command]

DESCRIPTION
     The LOCATE command searches for the next or previous occurrence 
     of the specified target.  If no parameter is supplied, LOCATE
     uses the the last target specified. If no prior target has been
     specified, an error message is displayed. With an optional operand, 
     it executes the command after finding the target.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Locate(CHARTYPE *params)
#else
short Locate(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *last_target;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Locate");
#endif
/*---------------------------------------------------------------------*/
/* If no parameter is specified, use the last_target. If that doesn't  */
/* exist, error.                                                       */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)
   {
    if (strcmp(last_target,"") == 0)
      {
       display_error(39,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    rc = command_line(last_target,COMMAND_ONLY_FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 rc = command_line(params,COMMAND_ONLY_FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     lowercase - change uppercase characters to lowercase

SYNTAX
     LOWercase [target]

DESCRIPTION
     The LOWERCASE command changes all uppercase characters in all 
     lines up to the target line to lowercase. All other characters 
     remain untouched.

COMPATIBILITY
     XEDIT: Equivalent of LOWERCAS command.
     KEDIT: Compatible.

SEE ALSO
     UPPERCASE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Lowercase(CHARTYPE *params)
#else
short Lowercase(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Lowercase");
#endif
 rc = execute_change_case(params,CASE_LOWER);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ls - list the specified directory as an editable file

SYNTAX
     LS [filespec]

DESCRIPTION
     The LS command displays all files matching the specified
     file specification.
     When no parameter is supplied, all files in the current directory 
     are displayed subject to any SET DIRINCLUDE restrictions.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     DIRECTORY, SET DIRINCLUDE

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     macro - execute a macro command file

SYNTAX
     MACRO filename [arguments]

DESCRIPTION
     The MACRO command executes the contents of the specified file
     as command line commands.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.
     Only REXX macros accept the passing of arguments to the macro file.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Macro(CHARTYPE *params)
#else
short Macro(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Macro");
#endif
 rc = execute_macro(params,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     mark - mark a portion of text

SYNTAX
     MARK Line|Box|Stream|Word|Column

DESCRIPTION
     The MARK command marks a portion of text for later processing
     by a COPY, MOVE or DELETE command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Does not implement Stream.
            Adds WORD option and COLUMN options.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Mark(CHARTYPE *params)
#else
short Mark(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
/*--------------------------- local data ------------------------------*/
 LINETYPE true_line=0L;
 unsigned short y=0,x=0;
 unsigned short real_col=0;
 bool type_param=FALSE;
 LENGTHTYPE first_col=0,last_col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Mark");
#endif
/*---------------------------------------------------------------------*/
/* Marking text sets the following variables:                          */
/* LINE:                                                               */
/*         CURRENT_VIEW->marked_line:       TRUE                       */
/*         CURRENT_VIEW->marked_start_line: line number of first line  */
/*         CURRENT_VIEW->marked_end_line:   line number of last line   */
/*         CURRENT_VIEW->marked_col:        FALSE                      */
/*         CURRENT_VIEW->marked_start_col:  (ignored)                  */
/*         CURRENT_VIEW->marked_end_col:    (ignored)                  */
/* BOX:                                                                */
/* STREAM:                                                             */
/* WORD:                                                               */
/*         CURRENT_VIEW->marked_line:       TRUE                       */
/*         CURRENT_VIEW->marked_start_line: line number of first line  */
/*         CURRENT_VIEW->marked_end_line:   line number of last line   */
/*         CURRENT_VIEW->marked_col:        TRUE                       */
/*         CURRENT_VIEW->marked_start_col:  first column               */
/*         CURRENT_VIEW->marked_end_col:    last column                */
/* COLUMN:                                                             */
/*         CURRENT_VIEW->marked_line:       FALSE                      */
/*         CURRENT_VIEW->marked_start_line: (ignored)                  */
/*         CURRENT_VIEW->marked_end_line:   (ignored)                  */
/*         CURRENT_VIEW->marked_col:        TRUE                       */
/*         CURRENT_VIEW->marked_start_col:  first column               */
/*         CURRENT_VIEW->marked_end_col:    last column                */
/*---------------------------------------------------------------------*/
/* Validate the argument...                                            */
/*---------------------------------------------------------------------*/
 if (equal(params,(CHARTYPE *)"line",1))
    type_param = M_LINE;
 if (equal(params,(CHARTYPE *)"box",1))
    type_param = M_BOX;
 if (equal(params,(CHARTYPE *)"word",1))
    type_param = M_WORD;
 if (equal(params,(CHARTYPE *)"column",1))
    type_param = M_COLUMN;
 if (type_param == 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 true_line = get_true_line();
/*---------------------------------------------------------------------*/
/* If we are on 'Top of File' or 'Bottom of File' lines, error.        */
/*---------------------------------------------------------------------*/
 if (TOF(true_line) || BOF(true_line))
   {
    display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* If we are in the file area or prefix area and the focus line is not */
/* a real line, error.                                                 */
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_MAIN
 ||  CURRENT_VIEW->current_window == WINDOW_PREFIX)
   {
    if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
      {
       display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
   }
/*---------------------------------------------------------------------*/
/* Reset the previous marked view...                                   */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW != (VIEW_DETAILS *)NULL
 &&  MARK_VIEW != CURRENT_VIEW)
    MARK_VIEW->marked_line = MARK_VIEW->marked_col = FALSE;
 MARK_VIEW = CURRENT_VIEW;
 CURRENT_VIEW->mark_type = type_param;
/*---------------------------------------------------------------------*/
/* Set the new values for top and bottom lines marked.                 */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->marked_line)
   {
    if (true_line > CURRENT_VIEW->mark_end_line)
       CURRENT_VIEW->mark_end_line = true_line;
    if (true_line < CURRENT_VIEW->mark_start_line)
       CURRENT_VIEW->mark_start_line = true_line;
    if (true_line < CURRENT_VIEW->mark_end_line
    &&  true_line > CURRENT_VIEW->mark_start_line)
      {
       if (true_line-CURRENT_VIEW->mark_end_line >
           CURRENT_VIEW->mark_start_line-true_line)
          CURRENT_VIEW->mark_end_line = true_line;
       else
          CURRENT_VIEW->mark_start_line = true_line;
      }
   }
 else
   {
    CURRENT_VIEW->mark_start_line = CURRENT_VIEW->mark_end_line = true_line;
   }
/*---------------------------------------------------------------------*/
/* Set the new values for first and last columns marked.               */
/*---------------------------------------------------------------------*/
 real_col = x + CURRENT_VIEW->verify_col;
 if (CURRENT_VIEW->marked_col)
   {
    if ((real_col) > CURRENT_VIEW->mark_end_col)
       CURRENT_VIEW->mark_end_col = (real_col);
    if ((real_col) < CURRENT_VIEW->mark_start_col)
       CURRENT_VIEW->mark_start_col = (real_col);
    if ((real_col) < CURRENT_VIEW->mark_end_col
    &&  (real_col) > CURRENT_VIEW->mark_start_col)
      {
       if ((real_col)-CURRENT_VIEW->mark_end_col > CURRENT_VIEW->mark_start_col-(real_col))
          CURRENT_VIEW->mark_end_col = (real_col);
       else
          CURRENT_VIEW->mark_start_col = (real_col);
      }
   }
 else
   {
    CURRENT_VIEW->mark_start_col = CURRENT_VIEW->mark_end_col = real_col;
   }
/*---------------------------------------------------------------------*/
/* Set flags for various marked text types...                          */
/*---------------------------------------------------------------------*/
 switch(type_param)
   {
    case M_LINE:
         CURRENT_VIEW->marked_col = FALSE;
         CURRENT_VIEW->marked_line = TRUE;
         break;
    case M_BOX:
         CURRENT_VIEW->marked_col = TRUE;
         CURRENT_VIEW->marked_line = TRUE;
         break;
    case M_WORD:
         if (get_word(rec,rec_len,real_col-1,&first_col,&last_col) == 0)
           {
            CURRENT_VIEW->marked_line = CURRENT_VIEW->marked_col = FALSE;
            MARK_VIEW = (VIEW_DETAILS *)NULL;
            break;
           }
         CURRENT_VIEW->marked_col = TRUE;
         CURRENT_VIEW->marked_line = TRUE;
         CURRENT_VIEW->mark_start_line = CURRENT_VIEW->mark_end_line = true_line;
         CURRENT_VIEW->mark_start_col = first_col+1;
         CURRENT_VIEW->mark_end_col = last_col+1;
         break;
    case M_COLUMN:
         CURRENT_VIEW->marked_line = FALSE;
         CURRENT_VIEW->marked_col = TRUE;
         CURRENT_VIEW->mark_start_line = 1L;
         CURRENT_VIEW->mark_end_line = MAX_LONG;
         break;
    case M_STREAM:
         CURRENT_VIEW->marked_col = TRUE;
         CURRENT_VIEW->marked_line = TRUE;
         break;
   }
 build_current_screen(); 
 display_current_screen();
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     modify - display current SET command for alteration

SYNTAX
     MODify set-command

DESCRIPTION
     The MODIFY command displays the current setting of a set command
     on the command line enabling the user to change that setting.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET, QUERY

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Modify(CHARTYPE *params)
#else
short Modify(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[18];
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short itemno=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Modify");
#endif
 if ((itemno = find_item(params,QUERY_MODIFY)) == (-1))
    {
     display_error(1,params,FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }

 itemno = get_item_values(itemno,(CHARTYPE *)"",QUERY_MODIFY,0L,NULL,0L);
 strcpy(temp_cmd,"set ");
 for (i=0;i<itemno+1;i++)
   {
    strcat(temp_cmd,item_values[i].value);
    strcat(temp_cmd," ");
   }
 Cmsg(temp_cmd);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     move - move a portion of text

SYNTAX
     MOVE BLOCK [RESET]

DESCRIPTION
     The MOVE command copies the contents of the marked block to the
     current cursor position and deletes the characters/lines from the
     original position of the marked block.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Adds extra functionality with [RESET] option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short THEMove(CHARTYPE *params)
#else
short THEMove(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
#define MOV_PARAMS 2
 CHARTYPE *word[MOV_PARAMS+1];
 unsigned short num_params=0;
 unsigned short y=0,x=0;
 LINETYPE true_line=0L;
 CHARTYPE reset_block=SOURCE_UNKNOWN;
 CHARTYPE copy_command=0,delete_command=0;
 short rc=RC_OK;
 LINETYPE start_line=0L,end_line=0L,num_lines=0L,dest_line=0L;
 VIEW_DETAILS *old_mark_view=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   THEMove");
#endif
 num_params = param_split(params,word,MOV_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
    {
     display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (num_params > 2)
    {
     display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Test for valid parameters...                                        */
/*---------------------------------------------------------------------*/
 if (num_params == 1
 &&  equal((CHARTYPE *)"block",word[0],5))
    reset_block = SOURCE_BLOCK;
 if (num_params == 2
 &&  equal((CHARTYPE *)"block",word[0],5)
 &&  equal((CHARTYPE *)"reset",word[1],5))
    reset_block = SOURCE_BLOCK_RESET;
 if (reset_block == SOURCE_UNKNOWN)
    {
     display_error(1,params,FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Validate marked block, can be in any view.                          */
/*---------------------------------------------------------------------*/
 if (marked_block(FALSE) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* If the cursor is in the marked block...error.                       */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW == CURRENT_VIEW)
   {
    getyx(CURRENT_WINDOW_MAIN,y,x);
    switch(MARK_VIEW->mark_type)
      {
       case M_LINE:
                   if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                   &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line))
                     {
                      display_error(44,(CHARTYPE *)"for line move",FALSE);
#ifdef TRACE
                      trace_return();
#endif
                      return(RC_INVALID_ENVIRON);
                     }
                   break;
       case M_BOX:
                   if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                   &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
                   &&  (x + CURRENT_VIEW->verify_col >= MARK_VIEW->mark_start_col)
                   &&  (x + CURRENT_VIEW->verify_col <= MARK_VIEW->mark_end_col))
                     {
                      display_error(50,(CHARTYPE *)"",FALSE);
#ifdef TRACE
                      trace_return();
#endif
                      return(RC_INVALID_ENVIRON);
                     }
                   break;
       default:
                   break;
      }
   }
/*---------------------------------------------------------------------*/
/* If block is a box, call its function.                               */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW->mark_type == M_BOX)
   {
    box_operations(BOX_M,reset_block,FALSE,' ');/* don't reset and don't overlay */
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Determine the target line. If on the command line, target is current*/
/* line, else target line is focus line.                               */
/*---------------------------------------------------------------------*/
 true_line = get_true_line();
/*---------------------------------------------------------------------*/
/* If the true  line is the bottom of file line, subtract 1 from it.   */
/*---------------------------------------------------------------------*/
 if (BOF(true_line))
    true_line--;

 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 start_line = MARK_VIEW->mark_start_line;
 end_line = MARK_VIEW->mark_end_line;
 num_lines = end_line - start_line + 1L;
 dest_line = true_line;
 old_mark_view = MARK_VIEW;
 if (MARK_VIEW == CURRENT_VIEW)
   {
    copy_command = COMMAND_MOVE_COPY_SAME;
    delete_command = COMMAND_MOVE_DELETE_SAME;
   }
 else
   {
    copy_command = COMMAND_MOVE_COPY_DIFF;
    delete_command = COMMAND_MOVE_DELETE_DIFF;
   }

 rc = rearrange_line_blocks(copy_command,reset_block,start_line,
                            end_line,dest_line,1,MARK_VIEW,CURRENT_VIEW,FALSE);
 if (rc == RC_OK)
   {
    if (old_mark_view == CURRENT_VIEW)
      {
       if (dest_line < start_line)
         {
          start_line += num_lines;
          end_line += num_lines;
          dest_line += num_lines;
         }
      }
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    rc = rearrange_line_blocks(delete_command,reset_block,start_line,
                            end_line,start_line,1,old_mark_view,old_mark_view,FALSE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     msg - display message on error line

SYNTAX
     MSG [message]

DESCRIPTION
     The MSG command displays an error message on the error line.
     This command is usually issued from a macro file.
     This is similar to EMSG, but MSG does not sound the bell if SET
     BELL is on.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     CMSG, EMSG

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Msg(CHARTYPE *params)
#else
short Msg(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
extern WINDOW *error_window;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Msg");
#endif
 display_error(0,params,TRUE);
 touchwin(error_window);
 wrefresh(error_window);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     next - move forward in the file a number of lines

SYNTAX
     Next [relative_target]

DESCRIPTION
     The NEXT command moves the current line forwards the number of
     lines specified by the relative_target. This relative_target can 
     only be a positive integer or the character "*". 

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1

SEE ALSO
     DOWN, UP

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Next(CHARTYPE *params)
#else
short Next(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 short rc=RC_OK;
 LINE *curr=NULL;
 LINETYPE num_lines=0L,true_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Next");
#endif
 if (strcmp("",params) == 0)
    params = (CHARTYPE *)"1";
 true_line = get_true_line();
 if (strcmp("*",params) == 0)
    num_lines = CURRENT_FILE->number_lines - true_line + 1L;
 else
   {
    if (!valid_integer(params))
      {
       display_error(4,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    num_lines = atol(params);
    if (num_lines < 0L)
      {
       display_error(5,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
    rc = advance_current_line(num_lines);
 else
    rc = advance_focus_line(num_lines);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     nextwindow - switch focus of editing session to other window

SYNTAX
     NEXTWindow

DESCRIPTION
     The NEXTWINDOW command moves the focus of the editing session to
     the other window (if more than one window is currently displayed)
     or to the next file in the ring.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     EDIT, SCREEN

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Nextwindow(CHARTYPE *params)
#else
short Nextwindow(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE display_screens;
 extern WINDOW *foot;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Nextwindow");
#endif
 if (strcmp(params,"") != 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (display_screens == 1)
   {
    rc = Xedit((CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 current_screen = (current_screen == 0) ? 1 : 0;
 CURRENT_VIEW = CURRENT_SCREEN.screen_view;
 if (curses_started)
   {
    if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
      {
       wattrset(CURRENT_WINDOW_COMMAND,set_colour(CURRENT_FILE->attr+ATTR_CMDLINE));
       touchwin(CURRENT_WINDOW_COMMAND);
       wnoutrefresh(CURRENT_WINDOW_COMMAND);
      }
    if (CURRENT_WINDOW_ARROW != (WINDOW *)NULL)
      {
       wattrset(CURRENT_WINDOW_ARROW,set_colour(CURRENT_FILE->attr+ATTR_ARROW));
       redraw_window(CURRENT_WINDOW_ARROW);
       wnoutrefresh(CURRENT_WINDOW_ARROW);
      }
    if (foot != (WINDOW *)NULL)
      {
       wattrset(foot,set_colour(CURRENT_FILE->attr+ATTR_STATAREA));
       redraw_window(foot);
      }
    if (CURRENT_WINDOW_IDLINE != (WINDOW *)NULL)
      {
       wattrset(CURRENT_WINDOW_IDLINE,set_colour(CURRENT_FILE->attr+ATTR_IDLINE));
       redraw_window(CURRENT_WINDOW_IDLINE);
      }
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     nomsg - execute a command suppressing any messages 

SYNTAX
     NOMSG command [parameters]

DESCRIPTION
     The NOMSG command executes the supplied command but suppresses
     messages that would normally be displayed as a result of the 
     command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Nomsg(CHARTYPE *params)
#else
short Nomsg(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 bool save_msgmode=CURRENT_VIEW->msgmode_status;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Nomsg");
#endif
 CURRENT_VIEW->msgmode_status = FALSE;
 rc = command_line(params,COMMAND_ONLY_FALSE);
 CURRENT_VIEW->msgmode_status = save_msgmode;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     nop - no operation command

SYNTAX
     NOP

DESCRIPTION
     The NOP command doesn't do anything. It is used as a means of
     setting a key to do nothing.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     DEFINE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Nop(CHARTYPE *params)
#else
short Nop(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Nop");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     os - execute an operating system command

SYNTAX
     OS [command]

DESCRIPTION
     The OS command executes the supplied operating system command 
     or runs an interactive shell if no command is supplied.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent to DOS command.

SEE ALSO
     DOS, !

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Os(CHARTYPE *params)
#else
short Os(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Os");
#endif
/*---------------------------------------------------------------------*/
/* Execute the supplied parameters as OS commands. Run with output     */
/* displayed and pause before redrawing the windows.                   */
/*---------------------------------------------------------------------*/
 rc = execute_os_command(params,FALSE,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     osnowait - execute an operating system command - no prompt

SYNTAX
     OSNowait command

DESCRIPTION
     The OSNOWAIT command executes the supplied operating system 
     command not waiting for the user to be prompted once the
     command has completed.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of DOSNOWAIT command.

SEE ALSO
     DOSNOWAIT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Osnowait(CHARTYPE *params)
#else
short Osnowait(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Osnowait");
#endif
/*---------------------------------------------------------------------*/
/* Execute the supplied parameters as OS commands. Run with output     */
/* displayed but no pause before redrawing the windows.                */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)                     /* no params....error */
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 rc = execute_os_command(params,FALSE,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     osquiet - execute an operating system command quietly

SYNTAX
     OSQuiet command

DESCRIPTION
     The OSQUIET command executes the supplied os command as quietly
     as possible.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of DOSQUIET command.

SEE ALSO
     DOSQUIET

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Osquiet(CHARTYPE *params)
#else
short Osquiet(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Osquiet");
#endif
/*---------------------------------------------------------------------*/
/* Execute the supplied parameters as OS commands. Run with no output  */
/* displayed and no pause before redrawing the windows.                */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)                     /* no params....error */
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 rc = execute_os_command(params,TRUE,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     overlaybox - overlay marked box block on current cursor position

SYNTAX
     OVERLAYBox

DESCRIPTION
     The OVERLAYBOX copies the contents of the marked box block over the
     characters at the current cursor position or to column 1 of the
     current line if issued from the command line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     MOVE, COPY

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Overlaybox(CHARTYPE *params)
#else
short Overlaybox(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Overlaybox");
#endif
/*---------------------------------------------------------------------*/
/* Ensure there are no parameters.                                     */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate marked block, can be in any view.                          */
/*---------------------------------------------------------------------*/
 if (marked_block(FALSE) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Only allow command for box, column and word blocks.                 */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW->mark_type != M_BOX
 &&  MARK_VIEW->mark_type != M_COLUMN
 &&  MARK_VIEW->mark_type != M_WORD)
   {
    display_error(47,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 box_operations(BOX_C,SOURCE_BLOCK,TRUE,' ');  /* no reset, overlay */

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
