/***********************************************************************/
/* COMM5.C - Commands T-Z                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: comm5.c 2.0 1995/01/26 16:30:03 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/
/*man-start*********************************************************************
COMMAND
     tabpre - switch between FILEAREA and PREFIX area

SYNTAX
     tabpre

DESCRIPTION
     The TABPRE command switches the focus of the editor from the
     FILEAREA to the PREFIX area and vice versa, depending
     on which window is currently active.

     This command can only be used by assigning it to a function key.

     This command will be removed in a future version.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of SOS LEFTEDGE and SOS PREFIX

SEE ALSO
     SOS LEFTEDGE, SOS PREFIX

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Tabpre(CHARTYPE *params)
#else
short Tabpre(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Tabpre");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the cursor is in the command line or there is no prefix on, exit.*/
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 ||  !CURRENT_VIEW->prefix)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (CURRENT_VIEW->current_window == WINDOW_MAIN)
    rc = Sos_prefix("");
 else
    rc = Sos_leftedge("");
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     text - simulate keyboard entry of characters

SYNTAX
     TEXT text

DESCRIPTION
     The TEXT command simulates the entry of characters from the
     keyboard. This command is actually called when you enter text
     from the keyboard.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.
            Does not allow trailing spaces in text.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Text(CHARTYPE *params)
#else
short Text(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern bool INSERTMODEx;
 extern bool prefix_changed;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
 extern bool readonly;
 extern short prefix_width;
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 CHARTYPE real_key=0;
 chtype chtype_key=0;
 unsigned short y=0,x=0;
 short len_params=0;
 short rc=RC_OK;
 short off=0;
#if defined(USE_EXTCURSES)
 ATTR attr=0;
#else
 chtype attr=0;
#endif
 bool need_to_build_screen=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Text");
#endif
/*---------------------------------------------------------------------*/
/* If running in read-only mode, do not allow any text to be entered   */
/* in the main window.                                                 */
/*---------------------------------------------------------------------*/
 if (readonly && CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    display_error(56,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* If no parameters, return without doing anything.                    */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->hex)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen(params);
 for (i=0;i<len_params;i++)
   {
    real_key = case_translate((CHARTYPE)*(params+i));
    chtype_key = (chtype)(real_key & A_CHARTEXT);
    getyx(CURRENT_WINDOW,y,x);
#if defined(USE_EXTCURSES)
    attr = CURRENT_WINDOW->_a[y][x];
    wattrset(CURRENT_WINDOW,attr);
    attr = 0;
#else
    attr = winch(CURRENT_WINDOW) & A_ATTRIBUTES;
#endif
    switch(CURRENT_VIEW->current_window)
      {
       case WINDOW_MAIN:
            if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
               break;
            if (x+CURRENT_VIEW->verify_start > CURRENT_VIEW->verify_end)
               break;
            if (INSERTMODEx)
              {
               rec = meminschr(rec,real_key,CURRENT_VIEW->verify_col-1+x,
                               max_line_length,rec_len);
               put_char(CURRENT_WINDOW,chtype_key|attr,INSCHAR);
              }
            else
              {
               rec[CURRENT_VIEW->verify_col-1+x] = real_key;
               if (x == CURRENT_SCREEN.cols[WINDOW_MAIN]-1)
                  put_char(CURRENT_WINDOW,chtype_key|attr,INSCHAR);
               else
                  put_char(CURRENT_WINDOW,chtype_key|attr,ADDCHAR);
              }
            rc = memrevne(rec,' ',max_line_length);
            if (rc == (-1))
               rec_len = 0;
            else
               rec_len = rc+1;
            /* check for the cursor moving past the right   */
            /* margin when WORDWRAP is ON. If true, then    */
            /* don't execute the cursor_right() function, as */
            /* this could cause a window scroll.            */
            if (CURRENT_VIEW->wordwrap
            &&  rec_len > CURRENT_VIEW->margin_right)
               execute_wrap_word(x+CURRENT_VIEW->verify_col);
            else    
               {
               /* this is done here so that the show_page() in */
               /* cursor_right() is executed AFTER we get the   */
               /* new length of rec_len.                       */
#if defined(USE_EXTCURSES)
               if (x == CURRENT_SCREEN.cols[WINDOW_MAIN]-1)
                 {
                  wmove(CURRENT_WINDOW,y,x);
     /*           wrefresh(CURRENT_WINDOW); */
                  cursor_right(TRUE,FALSE);
                 }
#else
               if (INSERTMODEx
               || x == CURRENT_SCREEN.cols[WINDOW_MAIN]-1)
                  cursor_right(TRUE,FALSE);
#endif
               }
/*---------------------------------------------------------------------*/
/* If HEXSHOW is on and we are on the current line, build screen...    */
/*---------------------------------------------------------------------*/
            if (CURRENT_VIEW->hexshow_on
            &&  CURRENT_VIEW->focus_line == CURRENT_VIEW->current_line)
               need_to_build_screen = TRUE;
            break;
       case WINDOW_COMMAND:
            if (INSERTMODEx)
              {
               cmd_rec = (CHARTYPE *)meminschr((CHARTYPE *)cmd_rec,
                               real_key,x,
                               COLS,cmd_rec_len);
               put_char(CURRENT_WINDOW,chtype_key,INSCHAR);
#if !defined(USE_EXTCURSES)
               cursor_right(TRUE,FALSE);
#endif
              }
            else
              {
               cmd_rec[x] = real_key;
               put_char(CURRENT_WINDOW,chtype_key,ADDCHAR);
              }
            rc = memrevne(cmd_rec,' ',COLS);
            if (rc == (-1))
               cmd_rec_len = 0;
            else
               cmd_rec_len = rc+1;
            break;
       case WINDOW_PREFIX:
            if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
            &&  prefix_width != PREFIX_WIDTH)
               off = 1;
            else
               off = 0;
            prefix_changed = TRUE;
            if (pre_rec_len == 0)
              {
               wmove(CURRENT_WINDOW,y,off);
               my_wclrtoeol(CURRENT_WINDOW);
               wrefresh(CURRENT_WINDOW);
               x = off;
              }
            if (INSERTMODEx)
              {
               pre_rec = (CHARTYPE *)meminschr((CHARTYPE *)pre_rec,
                               real_key,x,
                               PREFIX_WIDTH,pre_rec_len);
               put_char(CURRENT_WINDOW,chtype_key,INSCHAR);
              }
            else
              {
               pre_rec[x] = real_key;
               put_char(CURRENT_WINDOW,chtype_key,ADDCHAR);
              }
            wmove(CURRENT_WINDOW,y,min(x+1,(prefix_width-1)+off));
            rc = memrevne(pre_rec,' ',PREFIX_WIDTH);
            if (rc == (-1))
               pre_rec_len = 0;
            else
               pre_rec_len = rc+1;
            break;
      }
   }
/*---------------------------------------------------------------------*/
/* If text is being inserted on a line which is in the marked block,   */
/* build and redisplay the window.                                     */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW == MARK_VIEW
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN
 &&  INSERTMODEx
 &&  CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line
 &&  CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
     need_to_build_screen = TRUE;
 if (need_to_build_screen)
   {
    build_current_screen();
    display_current_screen();
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     the - edit another file or switch to next file

SYNTAX
     THE [filename]

DESCRIPTION
     The THE command allows the user to edit another file. The new file
     is placed in the file ring. The previous file being edited remains
     in memory and can be returned to by issuing a THE command without
     any parameters. Several files can be edited at once, and all files
     are arranged in a ring, with subsequent THE commands moving through
     the ring, one file at a time.

COMPATIBILITY
     XEDIT: Does not provide options switches.
     KEDIT: Does not provide options switches.

SEE ALSO
     XEDIT, EDIT

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     top - move to the top of the file

SYNTAX
     TOP

DESCRIPTION
     The TOP command moves to the very start of the current file.
     The "Top-of-file" line is set to the current_line.

     "TOP" is equivalent to "BACKWARD *".

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     BACKWARD, BOTTOM

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Top(CHARTYPE *params)
#else
short Top(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
 extern bool in_macro;
/*--------------------------- local data ------------------------------*/
 short rc=RC_TOF_EOF_REACHED;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Top");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->current_line = 0L;
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen(); 
 if (!line_in_view(CURRENT_VIEW->focus_line))
    CURRENT_VIEW->focus_line = 0L;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (!in_profile && !in_macro)
   {
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       getyx(PREVIOUS_WINDOW,y,x);
    else
       getyx(CURRENT_WINDOW,y,x);
    display_current_screen();
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       wmove(PREVIOUS_WINDOW,y,x);
    else
       wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     up - move backward in the file a number of lines

SYNTAX
     Up [relative_target]

DESCRIPTION
     The UP command moves the current line backwards the number of
     lines specified by the relative_target. This relative_target can 
     only be a positive integer or the character "*". 

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1

SEE ALSO
     NEXT, DOWN

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Up(CHARTYPE *params)
#else
short Up(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 short rc=RC_OK;
 LINE *curr=NULL;
 LINETYPE num_lines=0L,true_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Up");
#endif
 if (strcmp("",params) == 0)
    params = (CHARTYPE *)"1";
 true_line = get_true_line();
 if (strcmp("*",params) == 0)
    num_lines = true_line + 1L;
 else
   {
    if (!valid_integer(params))
      {
       display_error(4,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    num_lines = atol(params);
    if (num_lines < 0L)
      {
       display_error(5,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
    rc = advance_current_line(-num_lines);
 else
    rc = advance_focus_line(-num_lines);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     uppercase - change lowercase characters to uppercase

SYNTAX
     UPPercase [target]

DESCRIPTION
     The UPPERCASE command changes all lowercase characters in all
     lines up to the target line to uppercase. All other characters 
     remain untouched.

COMPATIBILITY
     XEDIT: Equivalent of UPPERCAS
     KEDIT: Compatible.

SEE ALSO
     LOWERCASE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Uppercase(CHARTYPE *params)
#else
short Uppercase(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Uppercase");
#endif
 rc = execute_change_case(params,CASE_UPPER);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     xedit - edit another file or switch to next file

SYNTAX
     Xedit [filename]

DESCRIPTION
     The XEDIT command allows the user to edit another file. The new file
     is placed in the file ring. The previous file being edited remains
     in memory and can be returned to by issuing an XEDIT command without
     any parameters. Several files can be edited at once, and all files
     are arranged in a ring, with subsequent XEDIT commands moving through
     the ring, one file at a time.

COMPATIBILITY
     XEDIT: Does not provide options switches.
     KEDIT: Does not provide options switches.

SEE ALSO
     EDIT, THE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Xedit(CHARTYPE *params)
#else
short Xedit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_first;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern bool REPROFILEx;
 extern CHARTYPE number_of_files;
 extern CHARTYPE *prf_arg;
 extern short file_start;
 extern CHARTYPE *dirfilename;
 extern bool curses_started;
 extern bool execute_profile;
 extern bool in_profile;
 extern CHARTYPE display_screens;
 extern WINDOW *foot;
 extern WINDOW *divider;
 extern bool horizontal;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 int y=0,x=0;
 VIEW_DETAILS *save_current_view=NULL;
 VIEW_DETAILS *previous_current_view=NULL;
 bool two_views_of_same_file=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Xedit");
#endif
/*---------------------------------------------------------------------*/
/* Before we do anything, determine if there are 2 views of the same   */
/* file being displayed...                                             */
/*---------------------------------------------------------------------*/
 if (number_of_files > 0
 &&  display_screens > 1
 &&  CURRENT_SCREEN.screen_view->file_for_view == OTHER_SCREEN.screen_view->file_for_view)
    two_views_of_same_file = TRUE;
/*---------------------------------------------------------------------*/
/* With no arguments, edit the next file in the ring...                */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)
   {
/*---------------------------------------------------------------------*/
/* If this is the only file, ignore the command...                     */
/*---------------------------------------------------------------------*/
    if (number_of_files < 2)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_OK);
      }
    memset(cmd_rec,' ',COLS);
    cmd_rec_len = 0;
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* If more than one screen is displayed and the file displayed in each */
/* screen is the same, remove the 'current' view from the linked list. */
/*---------------------------------------------------------------------*/
    if (two_views_of_same_file)
      {
       save_current_view = CURRENT_VIEW;
       free_a_view();
       CURRENT_FILE->file_views--;
       CURRENT_SCREEN.screen_view = CURRENT_VIEW;
      }
/*---------------------------------------------------------------------*/
/* Save the position of the cursor for the current view before getting */
/* the contents of the new file...                                     */
/*---------------------------------------------------------------------*/
    if (curses_started)
      {
       if (CURRENT_WINDOW_COMMAND != NULL)
         {
          wmove(CURRENT_WINDOW_COMMAND,0,0);
          my_wclrtoeol(CURRENT_WINDOW_COMMAND);
         }
       getyx(CURRENT_WINDOW_MAIN,CURRENT_VIEW->y[WINDOW_MAIN],CURRENT_VIEW->x[WINDOW_MAIN]);
       if (CURRENT_WINDOW_PREFIX != NULL)
          getyx(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
      }
/*---------------------------------------------------------------------*/
/* Make the next view the current view...                              */
/*---------------------------------------------------------------------*/
    save_current_view = CURRENT_VIEW;
    if (CURRENT_VIEW->next == (VIEW_DETAILS *)NULL)
       CURRENT_VIEW = vd_first;
    else
       CURRENT_VIEW = CURRENT_VIEW->next;
    CURRENT_SCREEN.screen_view = CURRENT_VIEW;
    if ((save_current_view->prefix&PREFIX_LOCATION_MASK)
         != (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK)
    ||  save_current_view->cmd_line != CURRENT_VIEW->cmd_line
    ||  save_current_view->id_line != CURRENT_VIEW->id_line)
      {
/*---------------------------------------------------------------------*/
/* If the position of the prefix or command line for the new view is   */
/* different from the previous view, rebuild the windows...            */
/*---------------------------------------------------------------------*/
       set_screen_defaults();
       if (curses_started)
         {
          if (set_up_windows(current_screen) != RC_OK)
            {
#ifdef TRACE
            trace_return();
#endif
            return(RC_OK);
           }
         }
      }
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    build_current_screen(); 
    display_current_screen();
    if (curses_started)
      {
       if (foot != NULL)
         {
          wattrset(foot,set_colour(CURRENT_FILE->attr+ATTR_STATAREA));
          redraw_window(foot);
          touchwin(foot);
         }
       if (divider != NULL)
         {
          if (display_screens > 1
          && !horizontal)
            {
#ifdef A_ALTCHARSET
             wattrset(divider,A_ALTCHARSET|set_colour(CURRENT_FILE->attr+ATTR_DIVIDER));
#else
             wattrset(divider,set_colour(CURRENT_FILE->attr+ATTR_DIVIDER));
#endif
             redraw_window(divider);
             touchwin(divider);
             wnoutrefresh(divider);
            }
         }
       wmove(CURRENT_WINDOW_MAIN,CURRENT_VIEW->y[WINDOW_MAIN],CURRENT_VIEW->x[WINDOW_MAIN]);
       if (CURRENT_WINDOW_PREFIX != NULL)
          wmove(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
       getyx(CURRENT_WINDOW,y,x);
       wmove(CURRENT_WINDOW,y,x);
      }
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If there are still file(s) in the ring, clear the command line and  */
/* save any changes to the focus line.                                 */
/*---------------------------------------------------------------------*/
 if (number_of_files > 0)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    memset(cmd_rec,' ',COLS);
    cmd_rec_len = 0;
    previous_current_view = CURRENT_VIEW;
   }
/*---------------------------------------------------------------------*/
/* Save the position of the cursor for the current view before getting */
/* the contents of the new file...                                     */
/*---------------------------------------------------------------------*/
 if (curses_started
 &&  number_of_files > 0)
   {
    if (CURRENT_WINDOW_COMMAND != NULL)
      {
       wmove(CURRENT_WINDOW_COMMAND,0,0);
       my_wclrtoeol(CURRENT_WINDOW_COMMAND);
      }
    getyx(CURRENT_WINDOW_MAIN,CURRENT_VIEW->y[WINDOW_MAIN],CURRENT_VIEW->x[WINDOW_MAIN]);
    if (CURRENT_WINDOW_PREFIX != NULL)
       getyx(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
   }
/*---------------------------------------------------------------------*/
/* Read the contents of the new file into memory...                    */
/*---------------------------------------------------------------------*/
 if ((rc = get_file(strtrans(params,OSLASH,ISLASH))) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If more than one screen is displayed and the file displayed in each */
/* screen is the same, remove the 'current' view from the linked list. */
/*---------------------------------------------------------------------*/
 if (two_views_of_same_file)
   {
    save_current_view = CURRENT_VIEW;
    CURRENT_VIEW = previous_current_view;
    CURRENT_FILE->file_views--;
    free_a_view();
    CURRENT_SCREEN.screen_view = CURRENT_VIEW = save_current_view;
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (REPROFILEx && !in_profile)
   {
    in_profile = TRUE;
    if (execute_profile)
       (void)get_profile(prf_arg);
    in_profile = FALSE;
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
/*---------------------------------------------------------------------*/
/* If curses hasn't started, don't try to use curses functions...      */
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    display_current_screen();
    if (strcmp(CURRENT_FILE->fname,dirfilename) == 0)
       wmove(CURRENT_WINDOW_MAIN,CURRENT_VIEW->current_row,file_start-1);
    else
       wmove(CURRENT_WINDOW_MAIN,CURRENT_VIEW->current_row,0);
    wmove(CURRENT_WINDOW_COMMAND,0,0);
    if (CURRENT_WINDOW_PREFIX != NULL)
       touchwin(CURRENT_WINDOW_PREFIX);
    if (CURRENT_WINDOW_COMMAND != NULL)
       touchwin(CURRENT_WINDOW_COMMAND);
    if (CURRENT_WINDOW_IDLINE != NULL)
       touchwin(CURRENT_WINDOW_IDLINE);
    touchwin(CURRENT_WINDOW_MAIN);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     retrieve - return the next/prior command on the command line

SYNTAX
     ?[+]

DESCRIPTION
     The ? command returns the next or prior command from the command
     line stack and displays it on the command line.
     With no parameters, the most recent command entered on the command
     line is retrieved.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: See below..
     Does not support multiple '?' as in ??? to retrieve the third last
     command.
     This command is bound to the up and down arrows when on the
     command line depending on the setting of CMDARROWS.

SEE ALSO
     SET CMDARROWS

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Retrieve(CHARTYPE *params)
#else
short Retrieve(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 CHARTYPE *current_command=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Retrieve");
#endif
 if (strcmp(params,"") == 0)
   {
    current_command = get_next_command(DIRECTION_FORWARD);
    current_command = get_next_command(DIRECTION_FORWARD);
   }
 else
    if (strcmp(params,"+") == 0)
      current_command = get_next_command(DIRECTION_BACKWARD);
    else
      {
       display_error(1,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }

 wmove(CURRENT_WINDOW_COMMAND,0,0);
 my_wclrtoeol(CURRENT_WINDOW_COMMAND);
 if (current_command != (CHARTYPE *)NULL)
    Cmsg(current_command);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     = - re-execute the last command issued on the command line

SYNTAX
     =

DESCRIPTION
     The = command retrieves the most recently issued command from
     the command line and re-executes it.

COMPATIBILITY
     XEDIT: Does not support optional [subcommand] option.
     KEDIT: Does not support optional [command] option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Reexecute(CHARTYPE *params)
#else
short Reexecute(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 CHARTYPE *current_command=NULL;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Reexecute");
#endif
 if (strcmp(params,""))
    {
     display_error(1,params,FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Retrieve the last command and execute it.                           */
/*---------------------------------------------------------------------*/
 current_command = get_next_command(DIRECTION_NONE);
 if (current_command != NULL)
    rc = command_line(current_command,COMMAND_ONLY_FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ! - execute an operating system command

SYNTAX
     ! [command]

DESCRIPTION
     The OS command executes the supplied operating system command 
     or runs an interactive shell if no command is supplied.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent to DOS command.

SEE ALSO
     DOS, OS

STATUS
     Complete.
**man-end**********************************************************************/
