/***********************************************************************/
/* COMMUTIL.C -                                                        */
/* This file contains all utility functions used when processing       */
/* commands.                                                           */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: commutil.c 2.0 1995/01/26 16:30:23 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "key.h"
#include "command.h"
#include "proto.h"

 static CHARTYPE cmd[MAX_SAVED_COMMANDS][MAX_COMMAND_LENGTH];
 static short last_cmd=(-1),current_cmd=0,number_cmds=0,offset_cmd=0;

/*---------------------------------------------------------------------*/
/* The following two static variables are for reserving space for the  */
/* parameters of a command. Space for temp_params is allocated and     */
/* freed in the.c. If the size of the string to be placed into         */
/* temp_params is > length_temp_params, reallocate a larger area and   */
/* set the value of length_temp_params to reflect the new size.        */
/*---------------------------------------------------------------------*/
 static CHARTYPE *temp_params=NULL;
 static unsigned short length_temp_params=0;
/*---------------------------------------------------------------------*/
/* The following two static variables are for reserving space for the  */
/* directories in a macro path. Space for temp_macros is allocated and */
/* freed in the.c. If the size of the string to be placed into         */
/* temp_macros is > length_temp_macros  reallocate a larger area and   */
/* set the value of length_temp_macros to reflect the new size.        */
/*---------------------------------------------------------------------*/
 static CHARTYPE *temp_macros=NULL;
 static unsigned short length_temp_macros=0;
/*---------------------------------------------------------------------*/
/* The following two static variables are for reserving space for the  */
/* contents of   a command. Space for tmp_cmd     is allocated and     */
/* freed in the.c. If the size of the string to be placed into         */
/* tmp_cmd     is > length_tmp_cmd    , reallocate a larger area and   */
/* set the value of length_tmp_cmd     to reflect the new size.        */
/*---------------------------------------------------------------------*/
 static CHARTYPE *tmp_cmd=NULL;
 static unsigned short length_tmp_cmd=0;
/*---------------------------------------------------------------------*/
/* The following two        variables are for reserving space for the  */
/* contents of   a command. Space for temp_cmd    is allocated and     */
/* freed in the.c. If the size of the string to be placed into         */
/* temp_cmd    is > length_temp_cmd   , reallocate a larger area and   */
/* set the value of length_temp_cmd    to reflect the new size.        */
/*---------------------------------------------------------------------*/
 CHARTYPE *temp_cmd=NULL;
 static unsigned short length_temp_cmd=0;
/*---------------------------------------------------------------------*/
/* The following two are to specify the first and last items in the    */
/* linked list for key definitions.                                    */
/*---------------------------------------------------------------------*/
DEFINE *first_define=NULL;
DEFINE *last_define=NULL;

bool clear_command=TRUE;

/***********************************************************************/
#ifdef PROTO
CHARTYPE *get_key_name(int key)
#else
CHARTYPE *get_key_name(key)
int key;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 CHARTYPE *keyname=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:get_key_name");
#endif
/*---------------------------------------------------------------------*/
/* Get name of key...                                                  */
/*---------------------------------------------------------------------*/
 for (i=0;key_table[i].mnemonic!=NULL;i++)
    {
     if (key == key_table[i].key_value)
       {
        keyname = key_table[i].mnemonic;
        break;
       }
    }
#ifdef TRACE
 trace_return();
#endif
 return(keyname);
}
/***********************************************************************/
#ifdef PROTO
CHARTYPE *get_key_definition(int key)
#else
CHARTYPE *get_key_definition(key)
int key;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 DEFINE *curr=NULL;
 CHARTYPE delim[2];
 bool key_defined=FALSE;
 bool valid_key=FALSE;
 bool first_time=TRUE;
 bool locate_command=FALSE;
 CHARTYPE *keyname=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:get_key_definition");
#endif
/*---------------------------------------------------------------------*/
/* First determine if the key is a named key.                          */
/*---------------------------------------------------------------------*/
 keyname = get_key_name(key);
 if (keyname != NULL)
   {
    strcpy(temp_cmd,"Key: ");
    valid_key = TRUE;
    strcat(temp_cmd,keyname);
   }
/*---------------------------------------------------------------------*/
/* If key is invalid,  show it as a character and decimal; provided it */
/* is an ASCII or extended character.                                  */
/*---------------------------------------------------------------------*/
 if (!valid_key && key < 256)
    sprintf(temp_cmd,"Key: %c \\%d",(CHARTYPE)key,key);
/*---------------------------------------------------------------------*/
/* Next check to see if the key has been "defined".                    */
/*---------------------------------------------------------------------*/
 delim[1] = '\0';
 delim[0] = CURRENT_VIEW->linend_value;
 curr = first_define;
 while(curr != NULL)
  {
   if (key == curr->def_funkey)
     {
      key_defined = TRUE;
      if (first_time)
         strcat(temp_cmd," - assigned to '");
      else
        {
         strcat(temp_cmd,delim);
        }
/*---------------------------------------------------------------------*/
/* Append the command to the string.                                   */
/*---------------------------------------------------------------------*/
      strcat(temp_cmd,command[curr->def_command].text);
/*---------------------------------------------------------------------*/
/* Append any parameters.                                              */
/*---------------------------------------------------------------------*/
      if (strcmp(curr->def_params,"") != 0)
        {
         strcat(temp_cmd," ");
         strcat(temp_cmd,curr->def_params);
        }
      first_time = FALSE;
     }
   curr = curr->next;
  }
 if (key_defined)
   {
    strcat(temp_cmd,"'");
#ifdef TRACE
    trace_return();
#endif
    return(temp_cmd);
   }
/*---------------------------------------------------------------------*/
/* If not, check for the default function key values.                  */
/*---------------------------------------------------------------------*/
 for (i=0;command[i].text != NULL;i++)
    {
      if (key == command[i].funkey)
        {
         strcat(temp_cmd," - assigned to '");
/*---------------------------------------------------------------------*/
/* If a SET command, prefix with 'set'                                 */
/*---------------------------------------------------------------------*/
         if (command[i].set_command)
            strcat(temp_cmd,"set ");
/*---------------------------------------------------------------------*/
/* If a SOS command, prefix with 'sos'                                 */
/*---------------------------------------------------------------------*/
         if (command[i].sos_command)
            strcat(temp_cmd,"sos ");
/*---------------------------------------------------------------------*/
/* Append the command name.                                            */
/*---------------------------------------------------------------------*/
         strcat(temp_cmd,command[i].text);
/*---------------------------------------------------------------------*/
/* Append any parameters.                                              */
/*---------------------------------------------------------------------*/
         if (strcmp(command[i].params,"") != 0)
           {
            strcat(temp_cmd," ");
            strcat(temp_cmd,command[i].params);
           }
         strcat(temp_cmd,"'");
#ifdef TRACE
         trace_return();
#endif
         return(temp_cmd);
        }
    }
/*---------------------------------------------------------------------*/
/* If none of the above, it is unassigned                              */
/*---------------------------------------------------------------------*/
 strcat(temp_cmd," - unassigned");
#ifdef TRACE
 trace_return();
#endif
 return(temp_cmd);
}
/***********************************************************************/
#ifdef PROTO
short function_key(int key)
#else
short function_key(key)
int key;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE number_of_files;
 extern bool readonly;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 DEFINE *curr=NULL;
 bool key_defined=FALSE;
 short rc=RC_OK;
 CHARTYPE *cmd=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:function_key");
#endif
/*---------------------------------------------------------------------*/
/* First check to see if the function key has been redefined.          */
/*---------------------------------------------------------------------*/
 curr = first_define;
 while(curr != NULL)
  {
   if (key == curr->def_funkey)
     {
      key_defined = TRUE;
/*---------------------------------------------------------------------*/
/* If running in read-only mode and the function selected is not valid */
/* display an error.                                                   */
/*---------------------------------------------------------------------*/
      if (readonly && !command[curr->def_command].valid_in_readonly)
        {
         display_error(56,"",FALSE);
         rc = RC_INVALID_ENVIRON;
         break;
        }
/*---------------------------------------------------------------------*/
/* If there are no more files in the ring, and the command is not a    */
/* command to edit a new file, then ignore the command.                */
/*---------------------------------------------------------------------*/
      if (number_of_files == 0
      &&  !command[curr->def_command].edit_command)
        {
         rc = RC_OK;
         break;
        }
      if ((cmd = (CHARTYPE *)my_strdup(curr->def_params)) == NULL)
        {
         display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(RC_OUT_OF_MEMORY);
        }
      rc = (*command[curr->def_command].function)((CHARTYPE *)cmd);
      (*the_free)(cmd);
      if (rc != RC_OK
      &&  rc != RC_TOF_EOF_REACHED
      &&  rc != RC_NO_LINES_CHANGED
      &&  rc != RC_TARGET_NOT_FOUND)
         break;
     }
   curr = curr->next;
  }
 if (key_defined)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If not, check for the default function key values.                  */
/*---------------------------------------------------------------------*/
 for (i=0;command[i].text != NULL;i++)
      if (key == command[i].funkey)
        {
/*---------------------------------------------------------------------*/
/* If running in read-only mode and the function selected is not valid */
/* display an error.                                                   */
/*---------------------------------------------------------------------*/
         if (readonly && !command[i].valid_in_readonly)
           {
            display_error(56,"",FALSE);
            rc = RC_INVALID_ENVIRON;
           }
         else
           {
            if ((cmd = (CHARTYPE *)my_strdup(command[i].params)) == NULL)
              {
               display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
               trace_return();
#endif
               return(RC_OUT_OF_MEMORY);
              }
            rc = (*command[i].function)((CHARTYPE *)cmd);
            (*the_free)(cmd);
           }
#ifdef TRACE
         trace_return();
#endif
         return(rc);
        }
#ifdef TRACE
 trace_return();
#endif
 return(RAW_KEY);
}
/***********************************************************************/
#ifdef PROTO
short command_line(CHARTYPE *cmd_line,bool command_only)
#else
short command_line(cmd_line,command_only)
CHARTYPE *cmd_line;
bool command_only;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool error_on_screen;
 extern CHARTYPE *last_target;
 extern bool LINEND_STATUSx;
 extern CHARTYPE LINEND_VALUEx;
 extern bool in_macro;
 extern CHARTYPE number_of_files;
 extern short lastrc;
 extern bool readonly;
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
 bool valid_command=FALSE;
 bool linend_status=(number_of_files) ? CURRENT_VIEW->linend_status : LINEND_STATUSx;
 CHARTYPE linend_value=0;
 register short i=0,j=0;
 short rc=RC_OK,pos=0;
 LINETYPE num_lines=0L;
 CHARTYPE *cmd[MAX_COMMANDS+1];
 unsigned short num_commands=0;
 CHARTYPE command_delim[2];
 CHARTYPE *command_entered=NULL;
 CHARTYPE *cl_cmd=NULL;
 CHARTYPE *cl_param=NULL;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_SPARE;
 bool display_parse_error=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:command_line");
#endif
/*---------------------------------------------------------------------*/
/* If the command line is blank, just return.                          */
/*---------------------------------------------------------------------*/
 if (strlen(cmd_line) == 0)
   {
    if (!in_profile)
       wmove(CURRENT_WINDOW_COMMAND,0,0);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Set up values for LINEND for later processing...                    */
/*---------------------------------------------------------------------*/
 if (number_of_files == 0)
   {
    linend_status = LINEND_STATUSx;
    linend_value = LINEND_VALUEx;
   }
 else
   {
    linend_status = CURRENT_VIEW->linend_status;
    linend_value = CURRENT_VIEW->linend_value;
   }
/*---------------------------------------------------------------------*/
/* If the command is to be kept displayed on the command line...       */
/*---------------------------------------------------------------------*/
 if (*(cmd_line) == '&')
   {
    cmd_line++;
    clear_command = FALSE;
   }
 else
    if (!(in_macro && !clear_command))
       clear_command = TRUE;
/*---------------------------------------------------------------------*/
/* Copy the incoming cmd_line, so we can play with it.                 */
/*---------------------------------------------------------------------*/
 if ((command_entered = (CHARTYPE *)my_strdup(cmd_line)) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
/*---------------------------------------------------------------------*/
/* Allocate some space to cl_cmd and cl_param for the a command when   */
/* it is split into a command and its parameters.                      */
/*---------------------------------------------------------------------*/
 if ((cl_cmd = (CHARTYPE *)(*the_malloc)((strlen(cmd_line)+1)*sizeof(CHARTYPE))) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 if ((cl_param = (CHARTYPE *)(*the_malloc)((strlen(cmd_line)+1)*sizeof(CHARTYPE))) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
/*---------------------------------------------------------------------*/
/* If [SET] LINENd is set to ON, split the line up into a number of    */
/* individual commands.                                                */
/*---------------------------------------------------------------------*/
 if (linend_status
 && !define_command(cmd_line))
   {
    command_delim[0] = linend_value;
    command_delim[1] = '\0';
    num_commands = command_split(cmd_line,cmd,MAX_COMMANDS,command_delim,command_entered);
   }
 else
   {
    cmd[0] = command_entered;
    num_commands = 1;
   }
/*---------------------------------------------------------------------*/
/* For each command entered, split it up into command and params, and  */
/* process it...                                                       */
/*---------------------------------------------------------------------*/
 for (j=0;j<num_commands;j++)
   {
    valid_command = FALSE;
    split_command(cmd[j],cl_cmd,cl_param);
/*---------------------------------------------------------------------*/
/* Here is where we could check for synonyms first.                    */
/*---------------------------------------------------------------------*/
    if (!command_only)
       ;       /* get synonym for entered command */
/*---------------------------------------------------------------------*/
/* Look up the command in the command array in command.h               */
/*---------------------------------------------------------------------*/
    for (i=0;command[i].text != NULL;i++)
      {
/*---------------------------------------------------------------------*/
/* If no command text, continue.                                       */
/*---------------------------------------------------------------------*/
       if (strcmp(command[i].text,"") == 0)
         continue;
       rc = RC_OK;
/*---------------------------------------------------------------------*/
/* Check that the supplied command matches the command for the length  */
/* of the command and that the length is at least as long as the       */
/* necessary significance.                                             */
/*---------------------------------------------------------------------*/
       if (equal(command[i].text,cl_cmd,command[i].min_len)
       && command[i].min_len != 0
       && !command[i].sos_command)
         {
          if (in_profile
          && !command[i].valid_profile_command)
            {
             display_error(24,command[i].text,FALSE);
             lastrc = rc = RC_INVALID_ENVIRON;
             break;
            }
          valid_command = TRUE;
/*---------------------------------------------------------------------*/
/* Here is a big kludge. Because only a few commands need leading      */
/* spaces to be present in temp_params and all other commands barf at  */
/* leading spaces, we need to left truncate temp_params for most       */
/* commands.                                                           */
/*---------------------------------------------------------------------*/
          if (command[i].strip_param)
            {
             pos = strzne(cl_param,' ');
             if (pos == (-1))
                pos = 0;
            }
          else
             pos = 0;
/*---------------------------------------------------------------------*/
/* If running in read-only mode and the function selected is not valid */
/* display an error.                                                   */
/*---------------------------------------------------------------------*/
         if (readonly && !command[i].valid_in_readonly)
           {
            display_error(56,"",FALSE);
            rc = RC_INVALID_ENVIRON;
            break;
           }
/*---------------------------------------------------------------------*/
/* If there are no more files in the ring, and the command is not a    */
/* command to edit a new file, then ignore the command.                */
/*---------------------------------------------------------------------*/
         if (number_of_files == 0
         &&  !command[i].edit_command)
           {
            rc = RC_OK;
            break;
           }
/*---------------------------------------------------------------------*/
/* Now call the function associated with the supplied command string   */
/* and the possibly stripped parameters.                               */
/*---------------------------------------------------------------------*/
          lastrc = rc = (*command[i].function)(cl_param+pos);
          break;
         }
      }
/*---------------------------------------------------------------------*/
/* If an error occurred while executing a command above, break.        */
/*---------------------------------------------------------------------*/
    if (rc != RC_OK
    &&  rc != RC_TOF_EOF_REACHED)
       break;
/*---------------------------------------------------------------------*/
/* If we found and successfully executed a command above, process the  */
/* next command.                                                       */
/*---------------------------------------------------------------------*/
    if (valid_command)
       continue;
/*---------------------------------------------------------------------*/
/* To get here the command was not a 'command'; check if a valid target*/
/*---------------------------------------------------------------------*/
    initialise_target(&target);
    if (!CURRENT_VIEW->imp_macro
    &&  !CURRENT_VIEW->imp_os)
       display_parse_error = TRUE;
    else
       display_parse_error = FALSE;
    rc = validate_target(cmd[j],&target,target_type,get_true_line(),display_parse_error,TRUE);
/*---------------------------------------------------------------------*/
/* If a valid target, but target not found, continue...                */
/*---------------------------------------------------------------------*/
    if (rc == RC_TARGET_NOT_FOUND)
      {
       strcpy(last_target,target.string);
       free_target(&target);
       lastrc = rc = RC_TARGET_NOT_FOUND;
       continue;
      }
/*---------------------------------------------------------------------*/
/* If a valid target and found, go there and execute any following     */
/* command.                                                            */
/*---------------------------------------------------------------------*/
    if (rc == RC_OK)
      {
       strcpy(last_target,target.string);
       if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
          lastrc = rc = advance_current_line(target.num_lines);
       else
          lastrc = rc = advance_focus_line(target.num_lines);
       if ((rc == RC_OK
           || rc == RC_TOF_EOF_REACHED)
       &&  target.spare != (-1))
          rc = lastrc = command_line(strtrunc(target.rt[target.spare].string),FALSE);
       free_target(&target);
       continue;
      }
    free_target(&target);
/*---------------------------------------------------------------------*/
/* If return is RC_INVALID_OPERAND, check if command is OS command...  */
/*---------------------------------------------------------------------*/
    if (cmd[j][0] == '!')
      {
       strcpy(command_entered,cmd[j]);
       lastrc = rc = Os(command_entered+1);
       continue;
      }
/*---------------------------------------------------------------------*/
/* ...or if command is a macro command (as long as IMPMACRO is ON) and */
/* command_only is FALSE...                                            */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->imp_macro
    && !command_only)
      {
       strcpy(command_entered,cmd[j]);
       if (CURRENT_VIEW->imp_os)
         {
          rc = execute_macro(command_entered,FALSE);
          if (rc != RC_FILE_NOT_FOUND)
            {
             lastrc = rc;
             continue;
            }
         }
       else
         {
          rc = execute_macro(command_entered,TRUE);
          if (rc == RC_FILE_NOT_FOUND)
            {
             lastrc = rc = RC_NOT_COMMAND;
             break;
            }
          else
            {
             lastrc = rc;
             continue;
            }
         }
      }
/*---------------------------------------------------------------------*/
/* ...or if command is an OS command (as long as IMPOS is ON).         */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->imp_os)
      {
       error_on_screen = FALSE;
       strcpy(command_entered,cmd[j]);
       rc = Os(command_entered);
      }
    else
      {
       display_error(21,cmd[j],FALSE);
       rc = RC_NOT_COMMAND;
      }
/*---------------------------------------------------------------------*/
/* If the 'command' is not a command then do not process any more.     */
/*---------------------------------------------------------------------*/
    lastrc = rc;
    if (rc == RC_NOT_COMMAND)
       break;
   }
 cleanup_command_line();
 (*the_free)(command_entered);
 (*the_free)(cl_cmd);
 (*the_free)(cl_param);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
void cleanup_command_line(void)
#else
void cleanup_command_line()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern bool in_macro;
 extern CHARTYPE number_of_views;
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:cleanup_command_line");
#endif
 if (in_profile || in_macro || number_of_views == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 if (clear_command)
   {
    if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
      {
       wmove(CURRENT_WINDOW_COMMAND,0,0);
       my_wclrtoeol(CURRENT_WINDOW_COMMAND);
      }
    memset(cmd_rec,' ',COLS);
    cmd_rec_len = 0;
   }
 if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
    wmove(CURRENT_WINDOW_COMMAND,0,0);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void split_command(CHARTYPE *cmd_line,CHARTYPE *cmd,CHARTYPE *param)
#else
void split_command(cmd_line,cmd,param)
CHARTYPE *cmd_line,*cmd,*param;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
{
/*--------------------------- local data ------------------------------*/
 short pos=0;
 CHARTYPE *param_ptr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:split_command");
#endif
 strcpy(cmd,cmd_line);
 strtrunc(cmd);
 if ((param_ptr = (CHARTYPE *)strpbrk(cmd," \\/-+@")) == NULL)
    {
     strcpy(param,"");
#ifdef TRACE
     trace_return();
#endif
     return;
    }
 pos = strzne(param_ptr,' ');
 if (param_ptr == cmd
 || pos == (-1))
    {
     strcpy(param,"");
#ifdef TRACE
     trace_return();
#endif
     return;
    }
 strcpy(param,param_ptr+(*(param_ptr) == ' ' ? 1 : 0));
 *(param_ptr) = '\0';
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
short param_split(CHARTYPE *params,CHARTYPE *word[],short words,
                CHARTYPE *delims,CHARTYPE param_type)
#else
short param_split(params,word,words,delims,param_type)
CHARTYPE *params;
CHARTYPE *word[];
short words;
CHARTYPE *delims;
CHARTYPE param_type;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0,k=0;
 unsigned short len=0;
 CHARTYPE j=0;
 bool end_of_string=FALSE,end_of_word=FALSE;
 CHARTYPE *param_ptr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:param_split");
#endif
/*---------------------------------------------------------------------*/
/* Allocate some memory to the temporary area.                         */
/*---------------------------------------------------------------------*/
 if (params != NULL)
   {
    if (allocate_temp_space(strlen(params),param_type) != RC_OK)
      {
#ifdef TRACE
      trace_return();
#endif
      return(-1);
      }
   }
/*---------------------------------------------------------------------*/
/* Based on param_type, point param_ptr to appropriate buffer.         */
/*---------------------------------------------------------------------*/
 switch(param_type)
   {
    case TEMP_PARAM:
         param_ptr = temp_params;
         break;
    case TEMP_MACRO:
         param_ptr = temp_macros;
         break;
    case TEMP_TEMP_CMD:
         param_ptr = temp_cmd;
         break;
    default:
         return(-1);
         break;
   }
/*---------------------------------------------------------------------*/
/* In case params is NULL, copy an empty string into param_ptr...      */
/*---------------------------------------------------------------------*/
 if (params == NULL)
    strcpy(param_ptr,"");
 else
    strcpy(param_ptr,params);

 for (i=0;i<words;i++)
     word[i] = (CHARTYPE *)"";
 j = 0;
 end_of_string = TRUE;
 len = strlen(param_ptr);
 for (i=0;i<len && j<words;i++)
   {
    end_of_word = FALSE;
    for (k=0;k<strlen(delims);k++)
      {
       if (*(param_ptr+i) == *(delims+k))
          end_of_word = TRUE;
      }
    if (end_of_word)
      {
       *(param_ptr+i) = '\0';
       end_of_string = TRUE;
      }
    else
       if (end_of_string)
         {
          word[j++] = param_ptr+i;
          end_of_string = FALSE;
         }
   }
#ifdef TRACE
 trace_return();
#endif
 return(j);
}
/***********************************************************************/
#ifdef PROTO
short command_split(CHARTYPE *params,CHARTYPE *word[],short words,
                CHARTYPE *delims,CHARTYPE *buffer)
#else
short command_split(params,word,words,delims,buffer)
CHARTYPE *params;
CHARTYPE *word[];
short words;
CHARTYPE *delims;
CHARTYPE *buffer;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0,k=0;
 unsigned short len=0;
 CHARTYPE j=0;
 bool end_of_string=FALSE,end_of_word=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:command_split");
#endif
/*---------------------------------------------------------------------*/
/* In case params is NULL, copy an empty string into buffer...         */
/*---------------------------------------------------------------------*/
 if (params == NULL)
    strcpy(buffer,"");
 else
    strcpy(buffer,params);

 for (i=0;i<words;i++)
     word[i] = (CHARTYPE *)"";
 j = 0;
 end_of_string = TRUE;
 len = strlen(buffer);
 for (i=0;i<len && j<words;i++)
   {
    end_of_word = FALSE;
    for (k=0;k<strlen(delims);k++)
      {
       if (*(buffer+i) == *(delims+k))
          end_of_word = TRUE;
      }
    if (end_of_word)
      {
       *(buffer+i) = '\0';
       end_of_string = TRUE;
      }
    else
       if (end_of_string)
         {
          word[j++] = buffer+i;
          end_of_string = FALSE;
         }
   }
#ifdef TRACE
 trace_return();
#endif
 return(j);
}
/***********************************************************************/
#ifdef PROTO
LINETYPE get_true_line(void)
#else
LINETYPE get_true_line()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
 LINETYPE true_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:get_true_line");
#endif
/*---------------------------------------------------------------------*/
/* Determine 'true_line'.                                              */
/*---------------------------------------------------------------------*/
 if (in_profile)
    true_line = CURRENT_VIEW->current_line;
 else
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       true_line = CURRENT_VIEW->current_line;
    else
       true_line = CURRENT_VIEW->focus_line;
#ifdef TRACE
 trace_return();
#endif
 return(true_line);
}
#ifndef MSWIN
/***********************************************************************/
#ifdef PROTO
void print_line(bool close_spooler,LINETYPE true_line,LINETYPE num_lines,
                short pagesize,CHARTYPE *text,CHARTYPE *line_term)
#else
void print_line(close_spooler,true_line,num_lines,pagesize,text,line_term)
bool close_spooler;
LINETYPE true_line,num_lines;
short pagesize;
CHARTYPE *text;
CHARTYPE *line_term;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
#if defined(UNIX) || defined(OS2)
 extern CHARTYPE *spooler_name;
#endif
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#if defined(UNIX) || defined(OS2)
 static bool spooler_open=FALSE;
#  if defined(OS2)
 HFILE Lpt;
#    if defined(__32BIT__)
 ULONG Action=0L;
 ULONG NoWritten=0L;
#    else
 USHORT Action=0;
 USHORT NoWritten=0;
#    endif
#  endif
#endif
 static FILE *pp;
 register short i=0;
 LINETYPE j=0L;
 LINE *curr=NULL;
 short line_number=0;
 LINETYPE num_excluded=0L;
 LINETYPE abs_num_lines=(num_lines < 0L ? -num_lines : num_lines);
 short direction=(num_lines < 0L ? DIRECTION_BACKWARD : DIRECTION_FORWARD);
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:print_line");
#endif
#if defined(DOS)
 pp = stdprn;
#endif
#if defined(UNIX) || defined(OS2)
 if (close_spooler)
   {
    if (spooler_open)
      {
       spooler_open = FALSE;
#if defined(OS2)
       DosClose(Lpt);
#else
       pclose(pp);
#endif
      }
#ifdef TRACE
    trace_return();
#endif
    return;
   }
#endif
#if defined(UNIX)
 if (!spooler_open)
   {
    pp = popen(spooler_name,"w");
    spooler_open = TRUE;
   }
#endif
#if defined(OS2)
 if (!spooler_open)
   {
#ifdef __32BIT__
   if (DosOpen(spooler_name, &Lpt, &Action, 0,FILE_NORMAL,FILE_OPEN,
               OPEN_ACCESS_READWRITE|OPEN_SHARE_DENYNONE,(PEAOP2)NULL) != 0)
#else
   if (DosOpen(spooler_name, &Lpt, &Action, 0,FILE_NORMAL,FILE_OPEN,
               OPEN_ACCESS_WRITEONLY|OPEN_SHARE_DENYWRITE,NULL) != 0)
#endif
     {
#ifdef TRACE
      trace_return();
#endif
      return;
     }
    spooler_open = TRUE;
   }
#endif

 if (num_lines == 0L)
   {
#if defined(OS2)
    DosWrite(Lpt,text,strlen(text),&NoWritten);
#else
    fprintf(pp,"%s%s",text,line_term);
#endif
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Once we get here, we are to print lines from the file.              */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (curses_started)
   {
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       getyx(CURRENT_WINDOW_MAIN,y,x);
    else
       getyx(CURRENT_WINDOW,y,x);
   }
 curr = lll_find(CURRENT_FILE->first_line,true_line);
 for (j=0L;j<abs_num_lines;j++)
   {
    if (in_scope(curr)
    || CURRENT_VIEW->scope_all)
      {
       if (TOF(true_line)
       ||  BOF(true_line))
          ;
       else
         {
          if (num_excluded != 0)
            {
#if defined(OS2)
             print_shadow_line(Lpt,line_term,num_excluded);
#else
             print_shadow_line(pp,line_term,num_excluded);
#endif
             num_excluded = 0L;
            }
#if defined(OS2)
          DosWrite(Lpt,curr->line,curr->length,&NoWritten);
          DosWrite(Lpt,line_term,strlen(line_term),&NoWritten);
#else
          for (i=0;i<curr->length;i++)
              fputc(*(curr->line+i) & A_CHARTEXT,pp);
          fprintf(pp,"%s",line_term);
#endif
          line_number++;
          if (line_number == pagesize
          && pagesize != 0)
            {
#if defined(OS2)
             DosWrite(Lpt,"\f",1,&NoWritten);
#else
             fputc('\f',pp);
#endif
             line_number = 0;
            }
         }
      }
    else
       num_excluded++;
/*---------------------------------------------------------------------*/
/* Proceed to the next record, even if the current record not in scope.*/
/*---------------------------------------------------------------------*/
    true_line += (LINETYPE)direction;
    if (num_lines < 0L)
       curr = curr->prev;
    else
       curr = curr->next;
    if (curr == NULL)
       break;
   }
/*---------------------------------------------------------------------*/
/* If we have a shadow line remaining, print it...                     */
/*---------------------------------------------------------------------*/
 if (num_excluded != 0)
   {
#if defined(OS2)
    print_shadow_line(Lpt,line_term,num_excluded);
#else
    print_shadow_line(pp,line_term,num_excluded);
#endif
    num_excluded = 0L;
   }
/*---------------------------------------------------------------------*/
/* If STAY is OFF, change the current and focus lines by the number    */
/* of lines calculated from the target.                                */
/*---------------------------------------------------------------------*/
 if (!CURRENT_VIEW->stay)                               /* stay is off */
   {
    CURRENT_VIEW->focus_line = min(CURRENT_VIEW->focus_line+num_lines-1L,CURRENT_FILE->number_lines+1L);
    CURRENT_VIEW->current_line = min(CURRENT_VIEW->current_line+num_lines-1L,CURRENT_FILE->number_lines+1L);
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();
 if (curses_started)
   {
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       wmove(CURRENT_WINDOW_MAIN,y,x);
    else
       wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
#if defined(OS2)
/***********************************************************************/
#ifdef PROTO
void print_shadow_line(HFILE Lpt,CHARTYPE *line_term,LINETYPE num_excluded)
#else
void print_shadow_line(Lpt,line_term,num_excluded)
HFILE Lpt;
CHARTYPE *line_term;
LINETYPE num_excluded;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short num_dashes=0;
 CHARTYPE temp_buf[7]="";
#if defined(__32BIT__)
 ULONG NoWritten=0L;
#else
 USHORT NoWritten=0;
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:print_shadow_line");
#endif
 if (CURRENT_VIEW->shadow)
   {
    num_dashes = (CURRENT_SCREEN.cols[WINDOW_MAIN] - 26) / 2;
    for (i=0;i<num_dashes;i++)
       DosWrite(Lpt,"-",1,&NoWritten);
    sprintf(temp_buf,"%6ld",num_excluded);
    DosWrite(Lpt,temp_buf,strlen(temp_buf),&NoWritten);
    DosWrite(Lpt," line(s) not displayed ",23,&NoWritten);
    for (i=0;i<num_dashes;i++)
       DosWrite(Lpt,"-",1,&NoWritten);
    DosWrite(Lpt,line_term,strlen(line_term),&NoWritten);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
#else
/***********************************************************************/
#ifdef PROTO
void print_shadow_line(FILE *pp,CHARTYPE *line_term,LINETYPE num_excluded)
#else
void print_shadow_line(pp,line_term,num_excluded)
FILE *pp;
CHARTYPE *line_term;
LINETYPE num_excluded;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short num_dashes=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:print_shadow_line");
#endif
 if (CURRENT_VIEW->shadow)
   {
    num_dashes = (CURRENT_SCREEN.cols[WINDOW_MAIN] - 26) / 2;
    for (i=0;i<num_dashes;i++)
       fputc('-',pp);
    fprintf(pp,"%6ld line(s) not displayed ",num_excluded);
    for (i=0;i<num_dashes;i++)
       fputc('-',pp);
    fprintf(pp,"%s",line_term);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
#endif
#endif
/***********************************************************************/
#ifdef PROTO
CHARTYPE next_char(LINE *curr,short *off,short end_col)
#else
CHARTYPE next_char(curr,off,end_col)
LINE *curr;
short *off;
short end_col;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:next_char");
#endif
 if (*(off) < min(curr->length,end_col))
   {
    (*(off))++;
#ifdef TRACE
    trace_return();
#endif
    return(*(curr->line+((*(off))-1)));
   }
 *(off) = (-1);
#ifdef TRACE
 trace_return();
#endif
 return(0);
}
/***********************************************************************/
#ifdef PROTO
short add_define(int key_value,CHARTYPE *commands)
#else
short add_define(key_value,commands)
int key_value;
CHARTYPE *commands;
#endif
/***********************************************************************/
/* Parameters:                                                         */
/*  key_value: numeric representation of function key                  */
/*   commands: commands and parameters                                 */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short j=0;
 short cmd_nr=0;
 CHARTYPE *word[MAX_COMMANDS+1];
 unsigned short num_commands=0;
 CHARTYPE command_delim[2];
 short rc=RC_OK;
 CHARTYPE *command_entered=NULL,*cl_cmd=NULL,*cl_param=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:add_define");
#endif
/*---------------------------------------------------------------------*/
/* If the commands argument is empty, delet the definition of the key  */
/* definitions for the key, so just return.                            */
/*---------------------------------------------------------------------*/
 if (strcmp(commands,"") == 0)
   {
    remove_define(key_value);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Copy the incoming commands, so we can play with it.                 */
/*---------------------------------------------------------------------*/
 if ((command_entered = (CHARTYPE *)my_strdup(commands)) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
/*---------------------------------------------------------------------*/
/* Allocate some space to cl_cmd and cl_param for the a command when   */
/* it is split into a command and its parameters.                      */
/*---------------------------------------------------------------------*/
 if ((cl_cmd = (CHARTYPE *)(*the_malloc)((strlen(commands)+1)*sizeof(CHARTYPE))) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 if ((cl_param = (CHARTYPE *)(*the_malloc)((strlen(commands)+1)*sizeof(CHARTYPE))) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
/*---------------------------------------------------------------------*/
/* If [SET] LINENd is set to ON, split the args up into a number of    */
/* individual commands.                                                */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->linend_status)
   {
    command_delim[0] = CURRENT_VIEW->linend_value;
    command_delim[1] = '\0';
    num_commands = command_split(commands,word,MAX_COMMANDS,command_delim,command_entered);
   }
 else
   {
    word[0] = command_entered;
    num_commands = 1;
   }
/*---------------------------------------------------------------------*/
/* For each command entered, split it up into command and params, and  */
/* validate that each command is valid...                              */
/*---------------------------------------------------------------------*/
 for (j=0;j<num_commands;j++)
   {
    split_command(word[j],cl_cmd,cl_param);
    if ((cmd_nr = find_command(cl_cmd,FALSE)) == (-1))
      {
       display_error(21,cl_cmd,FALSE);
       rc = RC_INVALID_OPERAND;
       break;
      }
   }
/*---------------------------------------------------------------------*/
/* Now we know each command is valid, we can remove any prior          */
/* definition and assign the new one.                                  */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    remove_define(key_value);
    for (j=0;j<num_commands;j++)
      {
       split_command(word[j],cl_cmd,cl_param);
       if ((cmd_nr = find_command(cl_cmd,FALSE)) == (-1))
         {
          display_error(21,cl_cmd,FALSE);      /* this should not be reached */
          rc = RC_INVALID_OPERAND;
          break;
         }
       rc = append_define(key_value,cmd_nr,cl_param);
       if (rc != RC_OK)
          break;
      }
   }
 (*the_free)(command_entered);
 (*the_free)(cl_cmd);
 (*the_free)(cl_param);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short remove_define(int key_value)
#else
short remove_define(key_value)
int key_value;
#endif
/***********************************************************************/
/* Parameters:                                                         */
/*  key_value: numeric representation of function key                  */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 DEFINE *curr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:remove_define");
#endif
/*---------------------------------------------------------------------*/
/* Find all items in the linked list for the key_value and remove them */
/* from the list.                                                      */
/*---------------------------------------------------------------------*/
 curr = first_define;
 while(curr != NULL)
   {
    if (curr->def_funkey == key_value)
      {
       if (curr->def_params != NULL)
          (*the_free)(curr->def_params);
       curr = dll_del(&first_define,&last_define,curr,DIRECTION_FORWARD);
      }
    else
       curr = curr->next;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short append_define(int key_value,short cmd,CHARTYPE *prm)
#else
short append_define(key_value,cmd,prm)
int key_value;
short cmd;
CHARTYPE *prm;
#endif
/***********************************************************************/
/* Parameters:                                                         */
/*  key_value: numeric representation of function key                  */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 DEFINE *curr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:append_define");
#endif
/*---------------------------------------------------------------------*/
/* Find all items in the linked list for the key_value and remove them */
/* from the list.                                                      */
/*---------------------------------------------------------------------*/
 curr = dll_add(first_define,last_define,sizeof(DEFINE));
 if (curr == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 curr->def_params = (CHARTYPE *)(*the_malloc)((strlen(prm)+1)*sizeof(CHARTYPE));
 if (curr->def_params == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 strcpy(curr->def_params,prm);
 curr->def_funkey = key_value;
 curr->def_command = cmd;
 last_define = curr;
 if (first_define == NULL)
    first_define = last_define;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
int find_key_value(CHARTYPE *mnemonic)
#else
int find_key_value(mnemonic)
CHARTYPE *mnemonic;
#endif
/***********************************************************************/
/*   Function: find the matching key value for the supplied key name   */
/* Parameters:                                                         */
/*   mnemonic: the key name to be matched                              */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:find_key_value");
#endif
 for (i=0;key_table[i].mnemonic!=NULL;i++)
    if (equal(key_table[i].mnemonic,mnemonic,strlen(key_table[i].mnemonic)))
      {
#ifdef TRACE
       trace_return();
#endif
       return(key_table[i].key_value);
      }
#ifdef TRACE
 trace_return();
#endif
 return(-1);
}
/***********************************************************************/
#ifdef PROTO
short find_command(CHARTYPE *cmd,bool search_for_target)
#else
short find_command(cmd,search_for_target)
CHARTYPE *cmd;
bool search_for_target;
#endif
/***********************************************************************/
/*   Function: determine if the string supplied is a valid abbrev for  */
/*             a command.                                              */
/* Parameters:                                                         */
/*        cmd:               the string to be checked                  */
/*        search_for_target: determine if command is a valid target    */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short rc=RC_OK;
 LINETYPE num_lines=0L;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_BLOCK|TARGET_ALL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:find_command");
#endif
 for (i=0;command[i].text != NULL;i++)
    if (equal(command[i].text,cmd,(command[i].min_len == 0) ? strlen(command[i].text) : command[i].min_len)
    &&  !command[i].sos_command)
      {
#ifdef TRACE
       trace_return();
#endif
       return(i);
      }
/*---------------------------------------------------------------------*/
/* To get here the command was not a 'command'. If we don't want to    */
/* search for targets, exit with (-1).                                 */
/*---------------------------------------------------------------------*/
 if (!search_for_target)
   {
#ifdef TRACE
    trace_return();
#endif
    return(-1);
   }
/*---------------------------------------------------------------------*/
/* Find if it is a valid target...                                     */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 rc = validate_target(cmd,&target,target_type,get_true_line(),TRUE,TRUE);
 if (rc != RC_OK
 &&  rc != RC_TARGET_NOT_FOUND)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(-1);
   }
 free_target(&target);
/*---------------------------------------------------------------------*/
/* If a valid target, find 'LOCATE' command and return the index.      */
/*---------------------------------------------------------------------*/
 strcpy(temp_params,cmd);
 for (i=0;command[i].text != NULL;i++)
   {
    if (strcmp(command[i].text,"locate") == 0)
       break;
   }
#ifdef TRACE
 trace_return();
#endif
  return(i);
}
/***********************************************************************/
#ifdef PROTO
void init_command(void)
#else
void init_command()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:init_command");
#endif
 for (i=0;i<MAX_SAVED_COMMANDS;i++)
     strcpy(cmd[i],"");
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void add_command(CHARTYPE *new_cmd)
#else
void add_command(new_cmd)
CHARTYPE *new_cmd;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:add_command");
#endif
 offset_cmd = 0;
/*---------------------------------------------------------------------*/
/* If the command to be added is the same as the current command or if */
/* the command line is empty or if the command is "=" or "?", return   */
/* without adding command to array.                                    */
/*---------------------------------------------------------------------*/
 if (strcmp(new_cmd,cmd[current_cmd]) == 0
 || strcmp(new_cmd,"") == 0
 || strcmp(new_cmd,"=") == 0
 || new_cmd[0] == '?')
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 if (number_cmds == MAX_SAVED_COMMANDS)
    current_cmd = last_cmd = (last_cmd == MAX_SAVED_COMMANDS-1) ? 0 : ++last_cmd;
 else
    current_cmd = ++last_cmd;
 strcpy(cmd[current_cmd],new_cmd);
 number_cmds++;
 if (number_cmds > MAX_SAVED_COMMANDS)
    number_cmds = MAX_SAVED_COMMANDS;
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
CHARTYPE *get_next_command( short direction)
#else
CHARTYPE *get_next_command(direction)
short direction;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 CHARTYPE *cmd_to_return=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:get_next_command");
#endif
 if (number_cmds == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return((CHARTYPE *)NULL);
   }
 switch(direction)
   {
    case DIRECTION_BACKWARD:
         if (current_cmd+1 == number_cmds)
           {
            current_cmd = 0;
            cmd_to_return = cmd[current_cmd];
           }
         else
            cmd_to_return = cmd[++current_cmd];
         break;
    case DIRECTION_FORWARD:
         if (current_cmd+offset_cmd < 0)
           {
            current_cmd = number_cmds-1;
            cmd_to_return = cmd[current_cmd];
           }
         else
           {
            current_cmd = current_cmd+offset_cmd;
            cmd_to_return = cmd[current_cmd];
           }
         offset_cmd = (-1);
         break;
    case DIRECTION_NONE:
         cmd_to_return = cmd[current_cmd];
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(cmd_to_return);
}
/***********************************************************************/
#ifdef PROTO
bool is_tab_col(LENGTHTYPE x)
#else
bool is_tab_col(x)
LENGTHTYPE x;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 bool rc=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:is_tab_col");
#endif
 for (i=0;i<CURRENT_VIEW->numtabs;i++)
   {
    if (CURRENT_VIEW->tabs[i] == x)
      {
       rc = TRUE;
       break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
LENGTHTYPE find_next_tab_col(LENGTHTYPE x)
#else
LENGTHTYPE find_next_tab_col(x)
LENGTHTYPE x;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 LENGTHTYPE next_tab_col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:find_next_tab_col");
#endif
 for (i=0;i<CURRENT_VIEW->numtabs;i++)
   {
    if (CURRENT_VIEW->tabs[i] > x)
      {
       next_tab_col = CURRENT_VIEW->tabs[i];
       break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(next_tab_col);
}
/***********************************************************************/
#ifdef PROTO
LENGTHTYPE find_prev_tab_col(LENGTHTYPE x)
#else
LENGTHTYPE find_prev_tab_col(x)
LENGTHTYPE x;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 LENGTHTYPE next_tab_col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:find_prev_tab_col");
#endif
 for (i=CURRENT_VIEW->numtabs-1;i>-1;i--)
   {
    if (CURRENT_VIEW->tabs[i] < x)
      {
       next_tab_col = CURRENT_VIEW->tabs[i];
       break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(next_tab_col);
}
/***********************************************************************/
#ifdef PROTO
short tabs_convert(LINE *curr,bool expand_tabs,bool inc_alt,bool use_tabs,
                   bool add_to_recovery)
#else
short tabs_convert(curr,expand_tabs,inc_alt,use_tabs,add_to_recovery)
LINE *curr;
bool expand_tabs,inc_alt,use_tabs,add_to_recovery;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern CHARTYPE TABI_Nx;
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 bool expanded=FALSE;
 bool tabs_exhausted=FALSE;
 LENGTHTYPE tabcol=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:tabs_convert");
#endif
/*---------------------------------------------------------------------*/
/* If we are expanding tabs to spaces, do the following...             */
/*---------------------------------------------------------------------*/
 if (expand_tabs)
   {
    for (i=0,j=0;i<curr->length;i++)
     {
      if (curr->line[i] == '\t')
        {
         if (use_tabs)
           {
            if (tabs_exhausted)
              {
               rec[j++] = ' ';
               if (j >= max_line_length)
                  break;
              }
            else
              {
               tabcol = find_next_tab_col(j+1);
               if (tabcol == 0)
                  tabs_exhausted = TRUE;
               else
                 {
                  tabcol--;
                  do
                    {
                     rec[j++] = ' ';
                     if (j >= max_line_length)
                        break;
                    }
                  while (j<tabcol);
                 }
              }
           }
         else
           {
            do
              {
               rec[j++] = ' ';
               if (j >= max_line_length)
                  break;
              }
            while ((j % TABI_Nx) != 0);
           }
         expanded = TRUE;
        }
      else
        {
         rec[j++] = curr->line[i];
         if (j >= max_line_length)
           break;
        }
     }
/*---------------------------------------------------------------------*/
/* If we expanded tabs, we need to reallocate memory for the line.     */
/*---------------------------------------------------------------------*/
  if (expanded)
    {
     if (add_to_recovery)
        add_to_recovery_list(curr->line,curr->length);
     curr->line = (CHARTYPE *)(*the_realloc)((void *)curr->line,(j+1)*sizeof(CHARTYPE));
     if (curr->line == (CHARTYPE *)NULL)
       {
        display_error(30,"",FALSE);
#ifdef TRACE
        trace_return();
#endif
        return(RC_OUT_OF_MEMORY);
       }
/*---------------------------------------------------------------------*/
/* Copy the contents of rec into the line.                             */
/*---------------------------------------------------------------------*/
    memcpy(curr->line,rec,j);
    curr->length = j;
    *(curr->line+j) = '\0';
/*---------------------------------------------------------------------*/
/* Increment the number of alterations count.                          */
/*---------------------------------------------------------------------*/
    if (inc_alt)
      {
       if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
         {
#ifdef TRACE
         trace_return();
#endif
         return(rc);
        }
      }
    }
 }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short convert_hex_strings(CHARTYPE *str)
#else
short convert_hex_strings(str)
CHARTYPE *str;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 CHARTYPE *p=NULL;
 bool dec_char=FALSE;
 CHARTYPE temp_str[MAX_COMMAND_LENGTH];
 short num=0;
 CHARTYPE ch1=0,ch2=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:convert_hex_strings");
#endif
/*---------------------------------------------------------------------*/
/* Check if the string begins with d',D',x' or X'. If not return the   */
/* string unchanged.                                                   */
/*---------------------------------------------------------------------*/
 if ((*(str) == 'd'
 ||   *(str) == 'D'
 ||   *(str) == 'x'
 ||   *(str) == 'X')
 &&   *(str+1) == '\'')
    ;
 else
   {
#ifdef TRACE
    trace_return();
#endif
    return(strlen(str));
   }
/*---------------------------------------------------------------------*/
/* Check if the last character is a single quote. If not return (-1)   */
/* to indicate an error.                                               */
/*---------------------------------------------------------------------*/
 if (*(str+strlen(str)-1) != '\'')
   {
#ifdef TRACE
    trace_return();
#endif
    return((-1));
   }
/*---------------------------------------------------------------------*/
/* If we got here we can validate the contents of the string.          */
/*---------------------------------------------------------------------*/
 *(str+strlen(str)-1) = '\0';
 if (*(str) == 'd'
 ||  *(str) == 'D')
    dec_char = TRUE;
 else
    dec_char = FALSE;
 p = (CHARTYPE *)strtok(str+2," ");
 while(p != NULL)
   {
    switch(dec_char)
      {
       case TRUE: /* parse decimal number */
                  if (equal((CHARTYPE *)"000000",p,1))
                     temp_str[i++] = (CHARTYPE)0;
                  else
                    {
                     num = atoi(p);
                     if (num < 1 || num > 255)
                       {
#ifdef TRACE
                        trace_return();
#endif
                        return((-1));
                       }
                     temp_str[i++] = (CHARTYPE)num;
                    }
                  break;
       case FALSE: /* parse hexidecimal number */
                  ch1 = *(p);
                  ch2 = *(p+1);
                  if (strlen(p) != 2
                  || !isxdigit(ch1)
                  || !isxdigit(ch2))
                    {
#ifdef TRACE
                     trace_return();
#endif
                     return((-1));
                    }
                  if (isupper(ch1)!=0)
                     ch1 = tolower(ch1);
                  if (isupper(ch2)!=0)
                     ch2 = tolower(ch2);
                  num =  ((isdigit(ch1)!=0) ? ch1-48 : ch1-87) * 16;
                  num += ((isdigit(ch2)!=0) ? ch2-48 : ch2-87);
                  temp_str[i++] = (CHARTYPE)num;
                  break;
      }
    p = (CHARTYPE *)strtok(NULL," ");
   }
/*
 temp_str[i] = '\0';
 memcpy(str,temp_str,i+1);
*/
 memcpy(str,temp_str,i);
#ifdef TRACE
 trace_return();
#endif
 return(i);
}
/***********************************************************************/
#ifdef PROTO
short marked_block(bool in_current_view)
#else
short marked_block(in_current_view)
bool in_current_view;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:marked_block");
#endif
 if (in_profile)                  /* block commands invalid in profile */
   {
    display_error(24,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 if (MARK_VIEW == (VIEW_DETAILS *)NULL)             /* no marked block */
   {
    display_error(44,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 if (MARK_VIEW != CURRENT_VIEW     /* marked block not in current view */
 && in_current_view)
   {
    display_error(45,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short suspend_curses(void)
#else
short suspend_curses()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:suspend_curses");
#endif
#ifdef UNIX
# if defined(USE_EXTCURSES)
 csavetty(FALSE);
 reset_shell_mode();
# else
 reset_shell_mode();
#  ifdef BSD
 noraw();
 nl();
 echo();
 nocbreak();
#  endif
# endif
#endif
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short resume_curses(void)
#else
short resume_curses()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:resume_curses");
#endif
#ifdef UNIX
# if defined(USE_EXTCURSES)
 cresetty(FALSE);
# else
 reset_prog_mode();
#  ifdef BSD
 raw();
 nonl();
 noecho();
 cbreak();
#  endif
# endif
#endif
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short restore_THE(void)
#else
short restore_THE()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern WINDOW *divider;
 extern WINDOW *foot;
 extern CHARTYPE display_screens;
 extern bool horizontal;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 VIEW_DETAILS *save_current_view=NULL;
 CHARTYPE save_current_screen=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:restore_THE");
#endif
/*---------------------------------------------------------------------*/
/* If curses hasn't started, no point in doing anything...             */
/*---------------------------------------------------------------------*/
 if (!curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 getyx(CURRENT_WINDOW,y,x);
 wclear(stdscr);
 refresh();
 if (display_screens > 1)
   {
    save_current_view = CURRENT_VIEW;
    save_current_screen = current_screen;
    CURRENT_VIEW = OTHER_SCREEN.screen_view; /* make other view current */
    current_screen = (current_screen==0)?1:0; /* make other screen current */
    touch_current_screen();
    refresh_current_screen();
    CURRENT_VIEW = save_current_view;
    current_screen = save_current_screen;
    if (!horizontal)
      {
       touchwin(divider);
       wnoutrefresh(divider);
      }
   }
 touch_current_screen();
 if (foot != (WINDOW *)NULL)
    touchwin(foot);
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_set_sos_command(bool set_command,CHARTYPE *params)
#else
short execute_set_sos_command(set_command,params)
bool set_command;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define SETSOS_PARAMS  2
 CHARTYPE *word[SETSOS_PARAMS+1];
 unsigned short num_params=0;
 short rc=RC_OK,command_index=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_set_sos_command");
#endif
 num_params = param_split(params,word,SETSOS_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 1)
   {
    display_error(1,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if ((command_index = valid_command_type(set_command,word[0])) == RC_NOT_COMMAND)
   {
    display_error(set_command ? 42 : 41,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 rc = (*command[command_index].function)(word[1]);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short valid_command_type(bool set_command,CHARTYPE *cmd_line)
#else
short valid_command_type(set_command,cmd_line)
bool set_command;
CHARTYPE *cmd_line;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:valid_command_type");
#endif
 for (i=0;command[i].text != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* If no command text, continue.                                       */
/*---------------------------------------------------------------------*/
     if (strcmp(command[i].text,"") == 0)
        continue;
/*---------------------------------------------------------------------*/
/* Check that the supplied command matches the command for the length  */
/* of the command and that the length is at least as long as the       */
/* necessary significance.                                             */
/*---------------------------------------------------------------------*/
     if (equal(command[i].text,cmd_line,command[i].min_len)
     && command[i].min_len != 0)
       {
#ifdef TRACE
        trace_return();
#endif
        if (set_command && command[i].set_command)
           return(i);
        if (!set_command && command[i].sos_command)
           return(i);
       }
    }
#ifdef TRACE
 trace_return();
#endif
 return(RC_NOT_COMMAND);
}
/***********************************************************************/
#ifdef PROTO
short allocate_temp_space(unsigned short length,CHARTYPE param_type)
#else
short allocate_temp_space(length,param_type)
unsigned short length;
CHARTYPE param_type;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 CHARTYPE *temp_ptr=NULL;
 unsigned short *temp_length=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:allocate_temp_space");
#endif
/*---------------------------------------------------------------------*/
/* Based on param_type, point param_ptr to appropriate buffer.         */
/*---------------------------------------------------------------------*/
 switch(param_type)
   {
    case TEMP_PARAM:
         temp_ptr = temp_params;
         temp_length = &length_temp_params;
         break;
    case TEMP_MACRO:
         temp_ptr = temp_macros;
         temp_length = &length_temp_macros;
         break;
    case TEMP_TMP_CMD:
         temp_ptr = tmp_cmd;
         temp_length = &length_tmp_cmd;
         break;
    case TEMP_TEMP_CMD:
         temp_ptr = temp_cmd;
         temp_length = &length_temp_cmd;
         break;
    default:
         return(-1);
         break;
   }
 if (*temp_length >= length)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (temp_ptr == NULL)
    temp_ptr = (CHARTYPE *)(*the_malloc)(sizeof(CHARTYPE)*(length+1));
 else
    temp_ptr = (CHARTYPE *)(*the_realloc)(temp_ptr,sizeof(CHARTYPE)*(length+1));
 if (temp_ptr == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
/*---------------------------------------------------------------------*/
/* Based on param_type, point param_ptr to appropriate buffer.         */
/*---------------------------------------------------------------------*/
 switch(param_type)
   {
    case TEMP_PARAM:
         temp_params = temp_ptr;
         break;
    case TEMP_MACRO:
         temp_macros = temp_ptr;
         break;
    case TEMP_TMP_CMD:
         tmp_cmd = temp_ptr;
         break;
    case TEMP_TEMP_CMD:
         temp_cmd = temp_ptr;
         break;
    default:
         return(-1);
         break;
   }
 *temp_length = length;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
void free_temp_space(CHARTYPE param_type)
#else
void free_temp_space(param_type)
CHARTYPE param_type;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 CHARTYPE *temp_ptr=NULL;
 unsigned short *temp_length=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:free_temp_space");
#endif
/*---------------------------------------------------------------------*/
/* Based on param_type, point param_ptr to appropriate buffer.         */
/*---------------------------------------------------------------------*/
 switch(param_type)
   {
    case TEMP_PARAM:
         temp_ptr    = temp_params;
         temp_params = NULL;
         temp_length = &length_temp_params;
         break;
    case TEMP_MACRO:
         temp_ptr    = temp_macros;
         temp_macros = NULL;
         temp_length = &length_temp_macros;
         break;
    case TEMP_TMP_CMD:
         temp_ptr = tmp_cmd;
         tmp_cmd  = NULL;
         temp_length = &length_tmp_cmd;
         break;
    case TEMP_TEMP_CMD:
         temp_ptr    = temp_cmd;
         temp_cmd    = NULL;
         temp_length = &length_temp_cmd;
         break;
    default:
         return;
         break;
   }
 (*the_free)(temp_ptr);
 *temp_length = 0;
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
CHARTYPE calculate_actual_row(CHARTYPE base,short off,CHARTYPE rows)
#else
CHARTYPE calculate_actual_row(base,off,rows)
CHARTYPE base,rows;
short off;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short row=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:calculate_actual_row");
#endif
 switch(base)
   {
    case POSITION_TOP:
                     row = off;
                     break;
    case POSITION_MIDDLE:
                     row = (rows /2 ) + off;
                     break;
    case POSITION_BOTTOM:
                     row = rows+off+1;
                     break;
   }
/*---------------------------------------------------------------------*/
/* If the calculated row is outside the screen size, default to middle.*/
/*---------------------------------------------------------------------*/
 if (row < 0 || row > rows)
    row = rows / 2;
#ifdef TRACE
 trace_return();
#endif
 return((CHARTYPE)row-1);
}
/*man***************************************************************************
NAME
     get_valid_macro_file_name

SYNOPSIS
     short get_valid_macro_file_name(macroname,filename,errnum)
     CHARTYPE *macroname;
     CHARTYPE *filename;
     short *errnum;

DESCRIPTION
     The get_valid_macro_file_name function determines the fully qualified
     file name for the supplied macroname.

     If the macroname contains any path specifiers, then the macro name
     is used as the filename and a check is made to ensure that the file
     exists and is readable.

     If the macroname does not contain any path specifiers, each
     directory in the MACROPATH variable is searched for a file that
     consists of the macroname appended with the current value for
     MACROEXT. If a file is found, it is checked to ensure it is
     readable.

RETURN VALUE
     If a file is found based on the above matching process, the fully
     qualified file name is copied into filename, errnum is set to 0
     and the function returns with RC_OK.

     If a file is not found, the macroname is copied into filename, the
     error number of the error message is copied into errnum and the
     function returns with RC_FILE_NOT_FOUND.

     If a file is found but the file is not readable, the macroname is
     copied into filename, the error number of the error message is
     copied into errnum and the function returns with RC_ACCESS_DENIED.
*******************************************************************************/
#ifdef PROTO
short get_valid_macro_file_name(CHARTYPE *macroname,CHARTYPE *filename,short *errnum)
#else
short get_valid_macro_file_name(macroname,filename,errnum)
CHARTYPE *macroname,*filename;
short *errnum;
#endif
/***********************************************************************/
{
#if defined(UNIX)
#   define PATH_DELIM ":"
#else
#   define PATH_DELIM ";"
#endif
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE the_macro_path[MAX_FILE_NAME+1] ;
 extern CHARTYPE sp_path[MAX_FILE_NAME+1] ;
 extern CHARTYPE sp_fname[MAX_FILE_NAME+1] ;
 extern CHARTYPE macro_suffix[12];
/*--------------------------- local data ------------------------------*/
 register short i=0;
 CHARTYPE delims[3];
 bool file_found=FALSE;
 CHARTYPE *path[MAX_MACRO_DIRS+1];    /* max number of macro dirs */
 unsigned short num_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:get_valid_macro_file_name");
#endif
/*---------------------------------------------------------------------*/
/* Create the full name of the macro file by prepending the default    */
/* macropath provided the filename does not already contain a path.    */
/*---------------------------------------------------------------------*/
 (void *)strtrans(macroname,OSLASH,ISLASH);
#ifdef UNIX
 strcpy(delims,ISTR_SLASH);
 if (strpbrk(macroname,delims) == NULL
 && *(macroname) != '~')
#endif
#if defined(DOS) || defined(OS2)
 strcpy(delims,ISTR_SLASH);
 strcat(delims,":");
 if (strpbrk(macroname,delims) == NULL)
#endif
/*---------------------------------------------------------------------*/
/* The supplied macro file name does not contain a path...so for each  */
/* directory in the_macro_path, try to find the supplied file in that  */
/* directory.                                                          */
/*---------------------------------------------------------------------*/
   {
    strcpy(filename,the_macro_path);
    num_params = param_split(filename,path,MAX_MACRO_DIRS,PATH_DELIM,TEMP_MACRO);
    file_found = FALSE;
    for (i=0;i<num_params;i++)
      {
       strcpy(filename,path[i]);
       if (strlen(filename) == 0)
          continue;
       if (*(filename+strlen(filename)-1) != ISLASH)
          strcat(filename,ISTR_SLASH);
       strcat(filename,macroname);               /* append the file name */
       strcat(filename,macro_suffix);         /* append default suffix */
       if (file_exists(filename))           /* check if file exists... */
         {
          file_found = TRUE;
          break;
         }
      }
    if (!file_found)
      {
       strcpy(filename,macroname);
       strcat(filename,macro_suffix);
       *errnum = 11;
#ifdef TRACE
       trace_return();
#endif
       return(RC_FILE_NOT_FOUND);
      }
   }
 else                                /* file contains a path specifier */
/*---------------------------------------------------------------------*/
/* The supplied macro file name does contain a path...so just check to */
/* ensure that the file exists.                                        */
/*---------------------------------------------------------------------*/
   {
    if (splitpath(macroname) != RC_OK)
      {
       *errnum = 9;
#ifdef TRACE
       trace_return();
#endif
       return(RC_FILE_NOT_FOUND);
      }
    strcpy(filename,sp_path);
    strcat(filename,sp_fname);
    if (!file_exists(filename)
    ||  strcmp(sp_fname,"") == 0)
      {
       *errnum = 9;
#ifdef TRACE
       trace_return();
#endif
       return(RC_FILE_NOT_FOUND);
      }
   }
/*---------------------------------------------------------------------*/
/* If the file is not readable, error.                                 */
/*---------------------------------------------------------------------*/
 if (!file_readable(filename))
   {
    *errnum = 8;
#ifdef TRACE
    trace_return();
#endif
    return(RC_ACCESS_DENIED);
   }
#ifdef TRACE
 trace_return();
#endif
 *errnum = 0;
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
bool define_command(CHARTYPE *cmd_line)
#else
bool define_command(cmd_line)
CHARTYPE *cmd_line;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 CHARTYPE buf[7];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:define_command");
#endif
/*---------------------------------------------------------------------*/
/* First check if the command is a synonym, and use the real name to   */
/* search the command array.                                           */
/*---------------------------------------------------------------------*/

 memset(buf,'\0',7);
 memcpy(buf,cmd_line,min(6,strlen(cmd_line)));
 for (i=0;i<7;i++)
   {
    if (buf[i] == ' ')
       buf[i] = '\0';
   }
 if ((i = find_command(buf,FALSE)) == (-1))
   {
#ifdef TRACE
    trace_return();
#endif
    return(FALSE);
   }
 if (strcmp("define",command[i].text) == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return(TRUE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(FALSE);
}
/***********************************************************************/
#ifdef PROTO
void touch_current_screen(void)
#else
void touch_current_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:touch_current_screen");
#endif
 if (CURRENT_WINDOW_PREFIX != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_PREFIX);
 if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_COMMAND);
 touchwin(CURRENT_WINDOW_MAIN);
 touchwin(CURRENT_WINDOW_IDLINE);
 if (CURRENT_WINDOW_ARROW != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_ARROW);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void refresh_current_screen(void)
#else
void refresh_current_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:refresh_current_screen");
#endif
 show_heading();
 wnoutrefresh(CURRENT_WINDOW_MAIN);
 if (CURRENT_WINDOW_PREFIX != (WINDOW *)NULL)
    wnoutrefresh(CURRENT_WINDOW_PREFIX);
 if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
    wnoutrefresh(CURRENT_WINDOW_COMMAND);
 if (CURRENT_WINDOW_ARROW != (WINDOW *)NULL)
   {
    touchwin(CURRENT_WINDOW_ARROW);
    wnoutrefresh(CURRENT_WINDOW_ARROW);
   }
 wnoutrefresh(CURRENT_WINDOW);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void redraw_screen(CHARTYPE scrn)
#else
void redraw_screen(scrn)
CHARTYPE scrn;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:redraw_screen");
#endif
 if (curses_started)
   {
    if (screen[scrn].win[WINDOW_COMMAND] != NULL)
      {
       wattrset(screen[scrn].win[WINDOW_COMMAND],set_colour(screen[scrn].screen_view->file_for_view->attr+ATTR_CMDLINE));
       touchwin(screen[scrn].win[WINDOW_COMMAND]);
       wnoutrefresh(screen[scrn].win[WINDOW_COMMAND]);
      }
    if (screen[scrn].win[WINDOW_ARROW] != NULL)
      {
       wattrset(screen[scrn].win[WINDOW_ARROW],set_colour(screen[scrn].screen_view->file_for_view->attr+ATTR_ARROW));
       redraw_window(screen[scrn].win[WINDOW_ARROW]);
       wnoutrefresh(screen[scrn].win[WINDOW_ARROW]);
      }
    if (screen[scrn].win[WINDOW_IDLINE] != NULL)
      {
       wattrset(screen[scrn].win[WINDOW_IDLINE],set_colour(screen[scrn].screen_view->file_for_view->attr+ATTR_IDLINE));
       redraw_window(screen[scrn].win[WINDOW_IDLINE]);
      }
    if (screen[scrn].win[WINDOW_PREFIX] != NULL)
       touchwin(screen[scrn].win[WINDOW_PREFIX]);
    touchwin(screen[scrn].win[WINDOW_MAIN]);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
