/***********************************************************************/
/* EXECUTE.C -                                                         */
/* This file contains all functions that actually execute one or other */
/* commands.                                                           */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: execute.c 2.0 1995/01/26 16:30:58 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/***********************************************************************/
#ifdef PROTO
short execute_change_command(CHARTYPE *params,bool selective)
#else
short execute_change_command(params,selective)
CHARTYPE *params;
bool selective;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern unsigned short rec_len;
/*--------------------------- local data ------------------------------*/
 LINETYPE num_lines=0L,long_n=0L,long_m=0L;
 LINE *curr=NULL;
 CHARTYPE *old_str=NULL,*new_str=NULL;
 short rc=0,selective_rc=RC_OK;
 short  direction=DIRECTION_FORWARD;
 short number_lines=0,number_changes=0,number_of_changes=0,number_of_occ=0;
 short start_col=0,real_start=0,real_end=0,loc=0;
 LINETYPE true_line=0L,last_true_line=0L,final_target=0L;
 LINETYPE num_actual_lines=0L,abs_num_lines=0L,i=0L;
 short len_old_str=0,len_new_str=0;
 TARGET target;
 CHARTYPE message[40];
 bool lines_based_on_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_change_command");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. Up to 4 parameters */
/* may be supplied. The first is the string to change and its new      */
/* value, the second is the target, the third is the number of times   */
/* to change the value on one line and lastly is which occurrence to   */
/* change first.                                                       */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 rc = split_change_params(params,&old_str,&new_str,&target,&long_n,&long_m);
 if (rc != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 num_lines = target.num_lines;
 true_line = target.true_line;
 if (target.rt == NULL)
    lines_based_on_scope = TRUE;
 else
    lines_based_on_scope = (target.rt[0].target_type == TARGET_BLOCK_CURRENT) ? FALSE : TRUE;
 free_target(&target);
/*---------------------------------------------------------------------*/
/* Check for any hex strings in both old_str and new_str.              */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->hex)
   {
    if ((len_old_str = convert_hex_strings(old_str)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if ((len_new_str = convert_hex_strings(new_str)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   {
    len_old_str = strlen(old_str);
    len_new_str = strlen(new_str);
   }
/*---------------------------------------------------------------------*/
/* If the number of lines is zero, don't make any  changes. Exit with  */
/* no rows changed.                                                    */
/*---------------------------------------------------------------------*/
 if (num_lines == 0L)
   {
    display_error(36,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_NO_LINES_CHANGED);
   }
 if (num_lines < 0)
   {
    direction = DIRECTION_BACKWARD;
    abs_num_lines = -num_lines;
   }
 else
   {
    direction = DIRECTION_FORWARD;
    abs_num_lines = num_lines;
   }

 final_target = true_line+num_lines;

 if (true_line != CURRENT_VIEW->focus_line)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    pre_process_line(CURRENT_VIEW,true_line);
   }
 number_lines = 0;
 number_changes = 0;
 number_of_changes = 0;
 number_of_occ = 0;
 start_col = 0;
 last_true_line = true_line;
 curr = lll_find(CURRENT_FILE->first_line,true_line);
 for (i=0L,num_actual_lines=0L;;i++)
   {
    if (lines_based_on_scope)
      {
       if (num_actual_lines == abs_num_lines)
          break;
      }
    else
      {
       if (abs_num_lines == i)
          break;
      }
    rc = processable_line(CURRENT_VIEW,true_line,curr);
    switch(rc)
      {
       case LINE_SHADOW:
            break;
       case LINE_TOF_EOF:
            num_actual_lines++;
            break;
       default:
            loc = 0;
            number_of_changes = number_of_occ = 0;
            while(loc != (-1))
              {
               real_end = min(rec_len+len_old_str,CURRENT_VIEW->zone_end-1);
               real_start = max(start_col,CURRENT_VIEW->zone_start-1);

               if (rec_len < real_start && blank_field(old_str))
                 {
                  loc = 0;
                  rec_len = real_start+1;
                 }
               else
                 {
                  loc = memfind(rec+real_start,old_str,(real_end-real_start+1),
                                len_old_str,
                                (CURRENT_VIEW->case_change == CASE_IGNORE) ? TRUE : FALSE,
                                CURRENT_VIEW->arbchar_status,
                                CURRENT_VIEW->arbchar_single,
                                CURRENT_VIEW->arbchar_multiple);
                 }
               if (loc != (-1))
                 {
                  start_col = loc+real_start;
                  if (number_of_changes <= long_n-1 && number_of_occ >= long_m-1)
                    {
                    /* the following block is done for change or confirm of sch */
                     if (!selective)
                       {
                        memdelchr(rec,start_col,rec_len,len_old_str);
                        rec_len = max(start_col,rec_len - len_old_str);
                        meminsmem(rec,new_str,len_new_str,start_col,max_line_length,rec_len);
                        rec_len += len_new_str;
                        if (rec_len > max_line_length)
                          {
                           rec_len = max_line_length;
                           loc = (-1);
                          }
                        start_col += len_new_str;
                        number_changes++;
                        number_of_changes++;
                       }
                     else
                       {
                       /* selective */
                        selective_rc = selective_change(old_str,len_old_str,new_str,len_new_str,
                                                        true_line,last_true_line,start_col);
                        last_true_line = true_line;
                        switch(selective_rc)
                          {
                           case QUITOK:
                           case RC_OK:
                                start_col += len_new_str;
                                number_changes++;
                                number_of_changes++;
                                if (rec_len > max_line_length)
                                  {
                                   rec_len = max_line_length;
                                   loc = (-1);
                                  }
                                break;
                           case SKIP:
                                start_col += len_old_str;
                                break;
                           case QUIT:
                                break;
                          }
                        if (selective_rc == QUIT || selective_rc == QUITOK)
                           break;
                       }
                     number_of_occ++;
                    }
                  else
                    {
                     start_col += len_old_str;
                     number_of_occ++;
                    }
                  if (number_of_changes > long_n-1)
        /*          ||  number_of_occ > long_n-1)*/
                     loc = (-1);
                 }
              } /* end while */
            if (number_of_changes != 0)       /* changes made */
              {
               post_process_line(CURRENT_VIEW,true_line);
               number_lines++;
              }
            num_actual_lines++;
            break;
      }
    if (selective_rc == QUIT || selective_rc == QUITOK)
       break;
    start_col = 0;
    if (direction == DIRECTION_FORWARD)
       curr = curr->next;
    else
       curr = curr->prev;
    true_line += (LINETYPE)(direction);
    pre_process_line(CURRENT_VIEW,true_line);
   }
/*---------------------------------------------------------------------*/
/* If no changes were made, display error message and return.          */
/*---------------------------------------------------------------------*/
 if (number_changes == 0)
   {
    display_error(36,(CHARTYPE *)"",FALSE);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
#ifdef TRACE
    trace_return();
#endif
    return(RC_NO_LINES_CHANGED);
   }
/*---------------------------------------------------------------------*/
/* If STAY is OFF, change the current and focus lines by the number    */
/* of lines calculated from the target.                                */
/*---------------------------------------------------------------------*/
 if (!CURRENT_VIEW->stay)                                /* stay is off */
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line = final_target;

 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();

 sprintf(message,"%d occurrence(s) changed on %d line(s)",number_changes,number_lines);
 display_error(0,message,TRUE);
#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short selective_change(CHARTYPE *old_str,short len_old_str,CHARTYPE *new_str,
                       short len_new_str,LINETYPE true_line,LINETYPE last_true_line,short start_col)
#else
short selective_change(old_str,len_old_str,new_str,len_new_str,true_line,last_true_line,start_col)
CHARTYPE *old_str;
short len_old_str;
CHARTYPE *new_str;
short len_new_str;
LINETYPE true_line;
LINETYPE last_true_line;
short start_col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern unsigned short rec_len;
 extern CHARTYPE *rec;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short y=0,x=0,rc=RC_OK;
 int key=0;
 bool changed=FALSE;
 bool line_displayed=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:selective_change");
#endif

 getyx(CURRENT_WINDOW_MAIN,y,x);
                /* move cursor to old string a la cmatch */
                /* display message */
                /* accept key - C next, - N change, - Q to quit */

 CURRENT_VIEW->focus_line = true_line;
/*---------------------------------------------------------------------*/
/* Check if the true_line is in the currently displayed window.        */
/* If not, then change the current_line to the true_line.              */
/*---------------------------------------------------------------------*/
 line_displayed = FALSE;
 for (i=0;i<CURRENT_SCREEN.rows[WINDOW_MAIN];i++)
   {
    if (CURRENT_SCREEN.sl[i].line_number == true_line
    &&  CURRENT_SCREEN.sl[i].line_type == LINE_LINE)
      {
       line_displayed = TRUE;
       y = i;
       break;
      }
   }
 if (!line_displayed)
   {
    CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
    y = CURRENT_VIEW->current_row;
   }

 if (start_col >= CURRENT_VIEW->verify_col-1
 &&  start_col <= (CURRENT_SCREEN.cols[WINDOW_MAIN]+(CURRENT_VIEW->verify_col-1))-1)
    x = start_col-(CURRENT_VIEW->verify_col-1);
 else
   {
    x = CURRENT_SCREEN.cols[WINDOW_MAIN] / 2;
    CURRENT_VIEW->verify_col = max(1,start_col-(short)x);
    x = (start_col-(CURRENT_VIEW->verify_col-1));
   }

 key = 0;
 changed = FALSE;
 while(key == 0)
   {
    build_current_screen();
    display_current_screen();
    if (changed)
       display_prompt("Press 'N' for next,'C' to undo 'Q' to quit");
    else
       display_prompt("Press 'N' for next,'C' to change 'Q' to quit");
    wmove(CURRENT_WINDOW_MAIN,y,x);
    wrefresh(CURRENT_WINDOW_MAIN);

    key = my_getch(stdscr);
    clear_msgline();
    switch(key)
      {
       case 'N':
       case 'n':
            if (changed)
               rc = RC_OK;
            else
               rc = SKIP;
            break;
       case 'C':
       case 'c':
            if (changed)
              {
               memdelchr(rec,start_col,rec_len,len_new_str);
               rec_len -= len_new_str;
               meminsmem(rec,old_str,len_old_str,start_col,max_line_length,rec_len);
               rec_len += len_old_str;
              }
            else
              {
               memdelchr(rec,start_col,rec_len,len_old_str);
               rec_len -= len_old_str;
               meminsmem(rec,new_str,len_new_str,start_col,max_line_length,rec_len);
               rec_len += len_new_str;
              }
            changed = (changed) ? FALSE : TRUE;
            key = 0;
            break;
       case 'Q':
       case 'q':
            if (changed)
               rc = QUITOK;
            else
               rc = QUIT;
            break;
       default:
            key = 0;
            break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short insert_new_line(CHARTYPE *line,short len,LINETYPE num_lines,LINETYPE true_line,bool start_left_col,bool make_current,CHARTYPE select)
#else
short insert_new_line(line,len,num_lines,true_line,start_left_col,make_current,select)
CHARTYPE *line;
short len;
LINETYPE num_lines;
LINETYPE true_line;
bool start_left_col;
bool make_current;
CHARTYPE select;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 LINE *curr=NULL,*save_curr=NULL;
 unsigned short x=0,y=0;
 short rc=RC_OK,new_col=0;
 bool on_bottom_of_file=FALSE,on_bottom_of_screen=FALSE;
 short number_focus_rows=0;
 bool leave_cursor=FALSE;
 LINETYPE new_focus_line=0L,new_current_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:insert_new_line");
#endif
 if (!CURRENT_VIEW->scope_all)
    true_line = find_last_not_in_scope(NULL,true_line,DIRECTION_FORWARD);
/*---------------------------------------------------------------------*/
/* If we are on the 'Bottom of File' line reduce the true_line by 1    */
/* so that the new line is added before the bottom line.               */
/*---------------------------------------------------------------------*/
 if (true_line == CURRENT_FILE->number_lines+1L)
    true_line--;
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the true_line.                    */
/* This is the line after which the line(s) are to be added.           */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,true_line);
/*---------------------------------------------------------------------*/
/* Insert into the linked list the number of lines specified. All lines*/
/* will contain a blank line and a length of zero.                     */
/*---------------------------------------------------------------------*/
 save_curr = curr;
 for (i=0;i<num_lines;i++)
    {
     if ((curr = add_line(CURRENT_FILE->first_line,curr,line,len,select)) == NULL)
       {
        display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
        trace_return();
#endif
        return(RC_OUT_OF_MEMORY);
       }
    }
/*---------------------------------------------------------------------*/
/* Fix the positioning of the marked block (if there is one and it is  */
/* in the current view) and any pending prefix commands.               */
/*---------------------------------------------------------------------*/
 adjust_marked_lines(TRUE,true_line,num_lines);
 adjust_pending_prefix(CURRENT_VIEW,TRUE,true_line,num_lines);
/*---------------------------------------------------------------------*/
/* Increment the number of lines counter for the current file and the  */
/* number of alterations.                                              */
/*---------------------------------------------------------------------*/
 if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 CURRENT_FILE->number_lines += num_lines;
/*---------------------------------------------------------------------*/
/* Sort out focus and current line.                                    */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         CURRENT_VIEW->focus_line = true_line + 1L;
         if (make_current)
            CURRENT_VIEW->current_line = true_line + 1L;
         pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         break;
    case WINDOW_MAIN:
    case WINDOW_PREFIX:
         build_current_screen();
         getyx(CURRENT_WINDOW,y,x);
         calculate_scroll_values(&number_focus_rows,&new_focus_line,
                                 &new_current_line,
                                 &on_bottom_of_screen,&on_bottom_of_file,
                                 &leave_cursor,DIRECTION_FORWARD);
         new_col = x;
         if (CURRENT_VIEW->current_window == WINDOW_MAIN)
           {
            if (!start_left_col)
              {
               if (CURRENT_VIEW->newline_aligned)
                 {
                  new_col = memne(save_curr->line,' ',save_curr->length);
                  if (new_col == (-1))
                     new_col = 0;
/*---------------------------------------------------------------------*/
/* Special case when right margin is > than screen width...            */
/*---------------------------------------------------------------------*/
                  if (CURRENT_VIEW->verify_start != CURRENT_VIEW->verify_col)
                    {
/*---------------------------------------------------------------------*/
/* If the new column position will be on the same page...              */
/*---------------------------------------------------------------------*/
                     if (CURRENT_VIEW->verify_col < new_col
                     &&  CURRENT_VIEW->verify_col + CURRENT_SCREEN.screen_cols > new_col)
                        new_col = (new_col - CURRENT_VIEW->verify_col) + 1;
                     else
                       {
                        x = CURRENT_SCREEN.cols[WINDOW_MAIN] / 2;
                        CURRENT_VIEW->verify_col = max(1,new_col - (short)x + 2);
                        new_col = (CURRENT_VIEW->verify_col == 1) ? new_col : x - 1;
                       }
                    }
                 }
               else
                 {
                  new_col = 0;
                  CURRENT_VIEW->verify_col = 1;
                 }
              }
           }
/*---------------------------------------------------------------------*/
/* Move the cursor to where it should be and display the page.         */
/*---------------------------------------------------------------------*/
         if (on_bottom_of_screen)
           {
            CURRENT_VIEW->current_line = new_current_line;
            CURRENT_VIEW->focus_line = new_focus_line;
            wmove(CURRENT_WINDOW,y-((leave_cursor) ? 0 : 1),new_col);
           }
         else
           {
/*---------------------------------------------------------------------*/
/* We are in the middle of the window, so just move the cursor down    */
/* 1 line.                                                             */
/*---------------------------------------------------------------------*/
           wmove(CURRENT_WINDOW,y+number_focus_rows,new_col);
           CURRENT_VIEW->focus_line = new_focus_line;
          }
         break;
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_os_command(CHARTYPE *cmd,bool quiet,bool pause)
#else
short execute_os_command(cmd,quiet,pause)
CHARTYPE *cmd;
bool quiet;
bool pause;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *temp_cmd;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#if defined(DOS) || defined(OS2)
#define SHELL "COMSPEC"
#else
#define SHELL "SHELL"
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_os_command");
#endif

#ifdef MSWIN
 quiet = 1;
 pause = 0;
#endif

 if (!quiet && curses_started)
   {
    attrset(A_NORMAL);
    touchwin(stdscr);
    wmove(stdscr,0,0);
    addch(' ');
    wmove(stdscr,1,0);
    wrefresh(stdscr);   /* clear screen */
    suspend_curses();
   }
 if (allocate_temp_space(strlen(cmd),TEMP_TEMP_CMD) != RC_OK)
    {
     display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_OUT_OF_MEMORY);
    }
 if (strcmp(cmd,"") == 0)
    strcpy(temp_cmd,getenv(SHELL));
 else
    strcpy(temp_cmd,cmd);
#ifdef UNIX
 if (strcmp(temp_cmd,"") == 0)           /* no SHELL env variable set */
   {
    printf("No SHELL environment variable set - using /bin/sh\n");
    fflush(stdout);
    strcpy(temp_cmd,"/bin/sh");
   }
#endif
 if (quiet)
   {
#ifdef UNIX
    strcat(temp_cmd," > /dev/null");
#endif
#if defined(DOS) || defined(OS2)
    strcat(temp_cmd," > nul:");
#endif
   }
 system(temp_cmd);
 if (pause)
   {
    printf("\n\n%s",HIT_ANY_KEY);
    fflush(stdout);
   }
 if (!quiet && curses_started)
   {
    resume_curses();
    if (pause)
       (void)my_getch(stdscr);
    restore_THE();
   }
 if (curses_started)
    draw_cursor(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_makecurr(LINETYPE line)
#else
short execute_makecurr(line)
LINETYPE line;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_makecurr");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);

 CURRENT_VIEW->current_line = line;
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
    getyx(CURRENT_WINDOW,y,x);
 else
    getyx(CURRENT_WINDOW_MAIN,y,x);
 build_current_screen();
 display_current_screen();
 y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                            CURRENT_VIEW->current_row);
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
    wmove(CURRENT_WINDOW,y,x);
 else
    wmove(CURRENT_WINDOW_MAIN,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_shift_command(short shift_left,short num_cols,LINETYPE true_line,LINETYPE num_lines,bool lines_based_on_scope)
#else
short execute_shift_command(shift_left,num_cols,true_line,num_lines,lines_based_on_scope)
short shift_left,num_cols;
LINETYPE true_line,num_lines;
bool lines_based_on_scope;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*-------------------------- external data ----------------------------*/
 extern unsigned short rec_len;
 extern CHARTYPE *rec;
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 LINE *curr=NULL;
 LINETYPE abs_num_lines=(num_lines < 0L ? -num_lines : num_lines);
 LINETYPE i=0L;
 LINETYPE num_actual_lines=0L;
 register short j=0;
 short actual_cols=0;
 short rc=RC_OK;
 short direction=(num_lines < 0L ? DIRECTION_BACKWARD : DIRECTION_FORWARD);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_shift_command");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
    getyx(CURRENT_WINDOW_MAIN,y,x);
 else
    getyx(CURRENT_WINDOW,y,x);
 curr = lll_find(CURRENT_FILE->first_line,true_line);
/* for (i=0L;i<abs_num_lines;i++)*/
/*   {*/
 for (i=0L,num_actual_lines=0L;;i++)
   {
    if (lines_based_on_scope)
      {
       if (num_actual_lines == abs_num_lines)
          break;
      }
    else
      {
       if (abs_num_lines == i)
          break;
      }
    rc = processable_line(CURRENT_VIEW,true_line+(LINETYPE)(i*direction),curr);
    switch(rc)
      {
       case LINE_SHADOW:
            break;
       case LINE_TOF_EOF:
            num_actual_lines++;
            break;
       default:
            memset(rec,' ',max_line_length);
            memcpy(rec,curr->line,curr->length);
            rec_len = curr->length;
            if (shift_left)
              {
               actual_cols = min(num_cols,max(0,rec_len-(CURRENT_VIEW->zone_start-1)));
               memdelchr(rec,CURRENT_VIEW->zone_start-1,rec_len,actual_cols);
               rec_len -= actual_cols;
              }
            else
              {
               for (j=0;j<num_cols;j++)
                  meminschr(rec,' ',CURRENT_VIEW->zone_start-1,max_line_length,rec_len++);
               rec_len = min(rec_len,max_line_length);
               actual_cols = num_cols;
              }
/*---------------------------------------------------------------------*/
/* Increment the alteration counters only if line has changed.         */
/*---------------------------------------------------------------------*/
            if (actual_cols != 0)
              {
               if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
                 {
#ifdef TRACE
                  trace_return();
#endif
                  return(rc);
                 }
/*---------------------------------------------------------------------*/
/* Add the old line contents to the line recovery list.                */
/*---------------------------------------------------------------------*/
               add_to_recovery_list(curr->line,curr->length);
/*---------------------------------------------------------------------*/
/* Realloc the dynamic memory for the line if the line is now longer.  */
/*---------------------------------------------------------------------*/
               if (rec_len > curr->length)
                              /* what if realloc fails ?? */
                  curr->line = (CHARTYPE *)(*the_realloc)((void *)curr->line,(rec_len+1)*sizeof(CHARTYPE));
/*---------------------------------------------------------------------*/
/* Copy the contents of rec into the line.                             */
/*---------------------------------------------------------------------*/
               memcpy(curr->line,rec,rec_len);
               curr->length = rec_len;
               *(curr->line+rec_len) = '\0';
              }
            num_actual_lines++;
            break;
      }
/*---------------------------------------------------------------------*/
/* Proceed to the next record, even if the current record not in scope.*/
/*---------------------------------------------------------------------*/
    if (direction == DIRECTION_BACKWARD)
       curr = curr->prev;
    else
       curr = curr->next;
   }
/*---------------------------------------------------------------------*/
/* If STAY is OFF, change the current and focus lines by the number    */
/* of lines calculated from the target.                                */
/*---------------------------------------------------------------------*/
 if (!CURRENT_VIEW->stay                                 /* stay is off */
 &&  num_lines != 0L)
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line = true_line+num_lines-(LINETYPE)direction;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND
 &&  curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_change_case(CHARTYPE *params,CHARTYPE which_case)
#else
short execute_change_case(params,which_case)
CHARTYPE *params;
CHARTYPE which_case;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 LINETYPE num_lines=0L,true_line=0L,num_actual_lines=0L;
 unsigned short x=0,y=0;
 short  direction=0;
 LINE *curr=NULL;
 short start_col=0,end_col=0;
 short rc=RC_OK;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 bool lines_based_on_scope=TRUE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_change_case");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/* Valid values are: a target or "block".                              */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 if (strcmp("",params) == 0)
    params = (CHARTYPE *)"1";
 initialise_target(&target);
 if ((rc = validate_target(params,&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Determine in which direction we are working.                        */
/*---------------------------------------------------------------------*/
 if (target.num_lines < 0L)
   {
    direction = DIRECTION_BACKWARD;
    num_lines = target.num_lines * (-1L);
   }
 else
   {
    direction = DIRECTION_FORWARD;
    num_lines = target.num_lines;
   }
 true_line = target.true_line;
/*---------------------------------------------------------------------*/
/* If the target is BLOCK set the left and right margins to be the     */
/* margins of the BOX BLOCK, otherwise use ZONE settings.              */
/*---------------------------------------------------------------------*/
 start_col = CURRENT_VIEW->zone_start-1;
 end_col = CURRENT_VIEW->zone_end-1;
 if (target.rt[0].target_type == TARGET_BLOCK_CURRENT)
   {
    num_lines = MARK_VIEW->mark_end_line-MARK_VIEW->mark_start_line+1L;
    true_line = MARK_VIEW->mark_start_line;
    direction = DIRECTION_FORWARD;
    lines_based_on_scope = FALSE;
    if (MARK_VIEW->mark_type != M_LINE)
      {
       start_col = MARK_VIEW->mark_start_col-1;
       end_col   = MARK_VIEW->mark_end_col-1;
      }
   }
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the true_line.                    */
/* This is the first line to change.                                   */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,true_line);
/*---------------------------------------------------------------------*/
/* Change the case for the target lines and columns...                 */
/*---------------------------------------------------------------------*/
 for (i=0L,num_actual_lines=0L;;i++)
   {
    if (lines_based_on_scope)
      {
       if (num_actual_lines == num_lines)
          break;
      }
    else
      {
       if (num_lines == i)
          break;
      }
    rc = processable_line(CURRENT_VIEW,true_line+(LINETYPE)(i*direction),curr);
    switch(rc)
      {
       case LINE_SHADOW:
            break;
       case LINE_TOF_EOF:
            num_actual_lines++;
            break;
       default:
            add_to_recovery_list(curr->line,curr->length);
            change_case(curr->line,max(0,start_col),min(curr->length-1,end_col),which_case);
            num_actual_lines++;
            break;
       }
     if (direction == DIRECTION_FORWARD)
        curr = curr->next;
     else
        curr = curr->prev;
    }
 if (!CURRENT_VIEW->stay                                 /* stay is off */
 &&  target.num_lines != 0L)
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line = target.true_line+target.num_lines-(LINETYPE)direction;
 free_target(&target);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND
 &&  curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short change_case(CHARTYPE *str,short start,short end,CHARTYPE which_case)
#else
short change_case(str,start,end,which_case)
CHARTYPE *str;
short start,end;
CHARTYPE which_case;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 bool altered=FALSE;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:change_case");
#endif
 for (i=start;i<end+1;i++)
   {
    switch(which_case)
      {
       case CASE_UPPER:
                       if (islower(*(str+i)))
                         {
                          *(str+i) = toupper(*(str+i));
                          altered = TRUE;
                         }
                       break;
       case CASE_LOWER:
                       if (isupper(*(str+i)))
                         {
                          *(str+i) = tolower(*(str+i));
                          altered = TRUE;
                         }
                       break;
      }
   }
/*---------------------------------------------------------------------*/
/* Increment the number of alterations count if anything cahnged.      */
/*---------------------------------------------------------------------*/
 if (altered)
    if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short rearrange_line_blocks(CHARTYPE command,CHARTYPE source,
                          LINETYPE start_line,LINETYPE end_line,LINETYPE dest_line,short num_occ,
                          VIEW_DETAILS *src_view,VIEW_DETAILS *dst_view,bool lines_based_on_scope)
#else
short rearrange_line_blocks(command,source,start_line,end_line,dest_line,num_occ,src_view,dst_view,lines_based_on_scope)
CHARTYPE command,source;
LINETYPE start_line,end_line,dest_line;
short num_occ;
VIEW_DETAILS *src_view,*dst_view;
bool lines_based_on_scope;
#endif
/***********************************************************************/
/* Parameters:                                                         */
/*    command: the command being executed; COPY,DELETE,DUPLICATE,MOVE  */
/*     source: where the command is executed; COMMAND, PREFIX, BLOCK   */
/* start_line: the first line (or only line) number to be acted on     */
/*   end_line: the last line number to be acted on                     */
/*  dest_line: the destination line for copy,move and duplicate. For   */
/*             delete this is not applicable.                          */
/*    num_occ: the number of times to execute the command; only for DUP*/
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 register short j=0,k=0;
 short rc=RC_OK;
 static unsigned short y=0,x=0;
 bool dst_inside_src=FALSE,lines_added=FALSE,reset_block=FALSE;
 bool dest_in_block=FALSE;
 short  direction=0;
 LINETYPE num_lines=0L,off=0L,adjust_line=dest_line,num_actual_lines=0L;
 LINETYPE i=0L,num_pseudo_lines=0L;
 LINE *curr_src=NULL,*curr_dst=NULL;
 LINE *save_curr_src=NULL,*save_curr_dst=NULL;
 FILE_DETAILS *src_file=NULL,*dst_file=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:rearrange_line_blocks");
#endif
 src_file = src_view->file_for_view;
 dst_file = dst_view->file_for_view;
 if (source == SOURCE_BLOCK)
    reset_block = FALSE;
 else
    reset_block = TRUE;
/*---------------------------------------------------------------------*/
/* This block of commands is for copying lines...                      */
/*---------------------------------------------------------------------*/
 switch(command)
   {
    case COMMAND_COPY:
    case COMMAND_MOVE_COPY_SAME:
    case COMMAND_MOVE_COPY_DIFF:
    case COMMAND_DUPLICATE:
         lines_added = TRUE;
         switch(source)
           {
            case SOURCE_BLOCK:
            case SOURCE_BLOCK_RESET:
                 if (src_view == dst_view
                 &&  dest_line >= start_line
                 &&  dest_line <  end_line)
                     dest_in_block = TRUE;
                 break;
            case SOURCE_PREFIX:
                 if (dest_line >= start_line
                 &&  dest_line <  end_line)
                     dest_in_block = TRUE;
                 break;
            default:
                 break;
           }
/*---------------------------------------------------------------------*/
/* If the destination line is within the marked block then we have to  */
/* handle the processing of the src_curr pointer differently.          */
/*---------------------------------------------------------------------*/
         if (dest_in_block)
           {
            dst_inside_src = TRUE;
            off = dest_line - start_line;
           }
         else
            dst_inside_src = FALSE;
         num_lines = end_line - start_line + 1L;
         save_curr_src = lll_find(src_file->first_line,start_line);
         save_curr_dst = lll_find(dst_file->first_line,dest_line);
         for (k=0;k<num_occ;k++)
           {
            curr_src = save_curr_src;
            curr_dst = save_curr_dst;
            for (i=0L,num_actual_lines=0L;;i++)
               {
                if (lines_based_on_scope)
                  {
                   if (num_actual_lines == num_lines)
                      break;
                  }
                else
                  {
                   if (num_lines == i)
                      break;
                  }
                rc = processable_line(src_view,start_line+i,curr_src);
                switch(rc)
                  {
                   case LINE_SHADOW:
                        break;
                   case LINE_TOF_EOF:
                        num_actual_lines++;
                        num_pseudo_lines++;
                        break;
                   default:
                        if ((curr_dst = add_line(dst_file->first_line,curr_dst,
                                        curr_src->line,curr_src->length,
                                        curr_src->select)) == NULL)
                          {
                           display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
                           trace_return();
#endif
                           return(RC_OUT_OF_MEMORY);
                          }
/*---------------------------------------------------------------------*/
/* If moving lines within the same file, move any line name with the   */
/* line also.                                                          */
/*---------------------------------------------------------------------*/
                          if (command == COMMAND_MOVE_COPY_SAME)
                            {
                             if (curr_src->name != (CHARTYPE *)NULL)
                               {
                                curr_dst->name = curr_src->name;
                                curr_src->name = (CHARTYPE *)NULL;
                               }
                            }
                          num_actual_lines++;
                          break;
                  }
                   if (dst_inside_src && i == off)
                      for (j=0;j<off+1;j++)
                         curr_src = curr_src->next;
                   curr_src = curr_src->next;
              }
            }
         dst_file->number_lines += (num_actual_lines-num_pseudo_lines)*num_occ;
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* This block of commands is for deleting lines...                     */
/*---------------------------------------------------------------------*/
 switch(command)
   {
    case COMMAND_DELETE:
    case COMMAND_MOVE_DELETE_SAME:
    case COMMAND_MOVE_DELETE_DIFF:
         lines_added = FALSE;
         if (start_line > end_line)
           {
            direction = DIRECTION_BACKWARD;
            num_lines = start_line - end_line + 1L;
           }
         else
           {
            direction = DIRECTION_FORWARD;
            num_lines = end_line - start_line + 1L;
           }
         curr_dst = lll_find(dst_file->first_line,start_line);
/*         for (i=0,num_actual_lines=0;num_actual_lines<num_lines,curr_dst!=NULL;i++)*/
/*         for (i=0,num_actual_lines=0;i<num_lines;i++)*/
         for (i=0L,num_actual_lines=0L;;i++)
           {
            if (lines_based_on_scope)
              {
               if (num_actual_lines == num_lines)
                  break;
              }
            else
              {
               if (num_lines == i)
                  break;
              }
            rc = processable_line(dst_view,start_line+i,curr_dst);
            switch(rc)
              {
               case LINE_TOF_EOF:
                    num_actual_lines++; /* this is meant to fall through */
                    num_pseudo_lines++;
               case LINE_SHADOW:
                    if (direction == DIRECTION_FORWARD)
                       curr_dst = curr_dst->next;
                    else
                       curr_dst = curr_dst->prev;
                    break;
               default:
                    if (command != COMMAND_MOVE_DELETE_SAME)
                       add_to_recovery_list(curr_dst->line,curr_dst->length);
                    curr_dst = delete_line(dst_file->first_line,curr_dst,direction);
                    num_actual_lines++;
              }
            if (curr_dst == NULL)
               break;
           }
         dst_file->number_lines -= (num_actual_lines-num_pseudo_lines)*num_occ;
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* Increment alteration count for all but COMMAND_MOVE_COPY_SAME...    */
/*---------------------------------------------------------------------*/
 if (command != COMMAND_MOVE_COPY_SAME
 &&  (num_actual_lines-num_pseudo_lines) != 0)
   {
    if ((rc = increment_alt(dst_file)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
/*---------------------------------------------------------------------*/
/* This block of commands is for sorting out cursor position...        */
/*---------------------------------------------------------------------*/
 switch(command)
   {
    case COMMAND_COPY:
    case COMMAND_MOVE_COPY_SAME:
    case COMMAND_MOVE_COPY_DIFF:
    case COMMAND_DUPLICATE:
         if (source == SOURCE_COMMAND
         &&  CURRENT_VIEW->stay)
             break;
         if (IN_VIEW(dst_view,dest_line))
           {
            dst_view->focus_line = dest_line+1L;
           }
         if (dst_view->current_window != WINDOW_COMMAND
         &&  dst_view == CURRENT_SCREEN.screen_view)
           {
            getyx(CURRENT_WINDOW,y,x);
            if (y == CURRENT_SCREEN.rows[WINDOW_MAIN]-1)/* on bottom line of window */
              {
               dst_view->current_line = dst_view->focus_line;
               y = dst_view->current_row;
              }
            else
               y = get_row_for_focus_line(dst_view->focus_line,
                                          dst_view->current_row);
           }
         break;
    case COMMAND_DELETE:
    case COMMAND_MOVE_DELETE_SAME:
    case COMMAND_MOVE_DELETE_DIFF:
         if (dst_view->focus_line >= start_line
         &&  dst_view->focus_line <= end_line)
           {
            if (IN_VIEW(dst_view,dest_line))
              {
               if (dst_view->current_line > dst_file->number_lines+1L)
                  dst_view->current_line -= (num_actual_lines-num_pseudo_lines);
               dst_view->focus_line = dest_line;
              }
            else
              {
               if (dest_line > dst_file->number_lines)
                  dst_view->focus_line = dst_view->current_line = dst_file->number_lines;
               else
                  dst_view->focus_line = dst_view->current_line = dest_line;
              }
           }
         else
           {
            dest_line = (dst_view->focus_line < start_line ? dst_view->focus_line : dst_view->focus_line - num_lines);
            if (IN_VIEW(dst_view,dest_line))
              {
               if (dst_view->current_line > dst_file->number_lines+1L)
                  dst_view->current_line -= (num_actual_lines-num_pseudo_lines);
               dst_view->focus_line = dest_line;
              }
            else
              {
               if (dest_line > dst_file->number_lines)
                  dst_view->focus_line = dst_view->current_line = dst_file->number_lines;
               else
                  dst_view->focus_line = dst_view->current_line = dest_line;
              }
           }
         if (dst_file->number_lines == 0L)
               dst_view->focus_line = dst_view->current_line = 0L;
         if (dst_view->current_window != WINDOW_COMMAND
         &&  dst_view == CURRENT_SCREEN.screen_view)
           {
            getyx(CURRENT_WINDOW,y,x);
            y = get_row_for_focus_line(dst_view->focus_line,
                                       dst_view->current_row);
           }
/*---------------------------------------------------------------------*/
/* This is set here so that the adjust_pending_prefix command will work*/
/*---------------------------------------------------------------------*/
         if (direction == DIRECTION_BACKWARD)
            adjust_line = end_line;
         else
            adjust_line = start_line;
         dst_view->current_line = find_next_in_scope(NULL,dst_view->current_line,DIRECTION_FORWARD);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* This block of commands is for adjusting prefix and block lines...   */
/*---------------------------------------------------------------------*/
 switch(source)
   {
    case SOURCE_BLOCK:
    case SOURCE_BLOCK_RESET:
         adjust_pending_prefix(dst_view,lines_added,adjust_line,(num_actual_lines-num_pseudo_lines)*num_occ);
         if (command == COMMAND_MOVE_DELETE_SAME)
            adjust_marked_lines(lines_added,adjust_line,(num_actual_lines-num_pseudo_lines)*num_occ);
         else
            if (command != COMMAND_MOVE_DELETE_DIFF)
              {
               MARK_VIEW->marked_line = MARK_VIEW->marked_col = FALSE;
/*---------------------------------------------------------------------*/
/* The following does a 'reset block' in the current view.             */
/*---------------------------------------------------------------------*/
               if (reset_block)
                 {
                  dst_view->marked_line = dst_view->marked_col = FALSE;
                  MARK_VIEW = (VIEW_DETAILS *)NULL;
                 }
               else
                 {
                  dst_view->mark_start_line = dest_line + 1L;
                  dst_view->mark_end_line = dest_line + (num_actual_lines-num_pseudo_lines);
                  dst_view->marked_col = FALSE;
                  dst_view->mark_type = M_LINE;
                  dst_view->focus_line = dst_view->mark_start_line;
                  MARK_VIEW = dst_view;
                 }
              }
         break;
    case SOURCE_PREFIX:
    case SOURCE_COMMAND:
         adjust_marked_lines(lines_added,adjust_line,(num_actual_lines-num_pseudo_lines)*num_occ);
         adjust_pending_prefix(dst_view,lines_added,adjust_line,(num_actual_lines-num_pseudo_lines)*num_occ);
         break;
   }
 if (command != COMMAND_MOVE_DELETE_DIFF)
    pre_process_line(CURRENT_VIEW,dst_view->focus_line);
/*---------------------------------------------------------------------*/
/* If run from the profile, exit here.                                 */
/*---------------------------------------------------------------------*/
 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (command != COMMAND_MOVE_DELETE_DIFF
 &&  command != COMMAND_MOVE_COPY_SAME)
   {
    build_current_screen();
    display_current_screen();
   }
 if (dst_view->current_window != WINDOW_COMMAND
 &&  dst_view == CURRENT_SCREEN.screen_view)
    wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_set_point(CHARTYPE *name,LINETYPE true_line,bool point_on)
#else
short execute_set_point(name,true_line,point_on)
CHARTYPE *name;
LINETYPE true_line;
bool point_on;
#endif
/***********************************************************************/
/* Parameters:                                                         */
/*       name: the name of the line to be processed                    */
/*  true_line: the line number of the line                             */
/*   point_on: indicates if the line name is to be turned on or off    */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINE *curr=NULL;
 LINETYPE dummy=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_set_point");
#endif
 if (point_on)
   {
/*---------------------------------------------------------------------*/
/* Find a line that already has the same name. If one exists, remove   */
/* the name.                                                           */
/*---------------------------------------------------------------------*/
    if ((curr = find_named_line(name,&dummy,FALSE)) != (LINE *)NULL)
      {
       (*the_free)(curr->name);
       curr->name = (CHARTYPE *)NULL;
      }
/*---------------------------------------------------------------------*/
/* Allocate space for the name and attach it to the true_line.         */
/*---------------------------------------------------------------------*/
    curr = lll_find(CURRENT_FILE->first_line,true_line);
    if ((curr->name=(CHARTYPE *)(*the_malloc)(strlen(name)+1)) == NULL)
      {
       display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY);
      }
    strcpy(curr->name,name);
   }
 else
   {
/*---------------------------------------------------------------------*/
/* Find a line that already has the same name. If one exists, remove   */
/* the name otherwise display an error.                                */
/*---------------------------------------------------------------------*/
    if ((curr = find_named_line(name,&dummy,FALSE)) != (LINE *)NULL)
      {
       (*the_free)(curr->name);
       curr->name = (CHARTYPE *)NULL;
      }
    else
      {
       display_error(60,name,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_wrap_word(unsigned short col)
#else
short execute_wrap_word(col)
unsigned short col;
#endif
/***********************************************************************/
/* Parameters: col   - current column position within rec              */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool INSERTMODEx;
 extern unsigned short rec_len;
 extern CHARTYPE *rec;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short col_break=0,cursor_offset=0;
 LINE *curr=NULL,*next_line=NULL;
 bool newline=FALSE,cursor_wrap=FALSE;
 CHARTYPE *buf=NULL,*word_to_wrap=NULL;
 short next_line_start=0,length_word=0,last_col=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:execute_wrap_word");
#endif
/*---------------------------------------------------------------------*/
/* This function is called when the length of the focus line exceeds   */
/* the right margin. If the cursor is positioned in the last word of   */
/* the line, the cursor moves with that word to the next line.         */
/* If the combined length of the word to be wrapped and a space and the*/
/* line following the focus line exceeds the right margin, a new line  */
/* is inserted with the word being wrapped, otherwise the word to be   */
/* wrapped is prepended to the following line.                         */
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the focus_line.                   */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Determine where to start splitting the line in relation to end of   */
/* line...                                                             */
/*---------------------------------------------------------------------*/
 col_break = memreveq(rec,' ',rec_len);
/*---------------------------------------------------------------------*/
/* If there is no word break, don't attempt any wrap.                  */
/*---------------------------------------------------------------------*/
 if (col_break == (-1))
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Actual column to break on is 1 character to right of last space.    */
/*---------------------------------------------------------------------*/
 col_break++;
/*---------------------------------------------------------------------*/
/* Make a null terminated string out of current line so we can grab the*/
/* word to be wrapped.                                                 */
/*---------------------------------------------------------------------*/
 rec[rec_len] = '\0';
/*---------------------------------------------------------------------*/
/* Point to word to wrap and determine its length.                     */
/*---------------------------------------------------------------------*/
 length_word = rec_len - col_break;
 word_to_wrap = (CHARTYPE *)rec+col_break;
/*---------------------------------------------------------------------*/
/* If the position of the cursor is before the word to wrap leave the  */
/* cursor where it is.                                                 */
/*---------------------------------------------------------------------*/
 if (col >= col_break)
   {
    cursor_wrap = TRUE;
    cursor_offset = col - col_break - 1;
   }
 else
    cursor_wrap = FALSE;
/*---------------------------------------------------------------------*/
/* Now we have to work out if a new line is to added or we prepend to  */
/* the following line...                                               */
/*---------------------------------------------------------------------*/
 if (curr->next->next == NULL)             /* next line bottom of file */
    newline = TRUE;
 else
   {
    next_line = curr->next;
    if (!in_scope(next_line))
       newline = TRUE;
    else
      {
       next_line_start = memne(next_line->line,' ',next_line->length);
       if (next_line_start != CURRENT_VIEW->margin_left-1) /* next line doesn't start in left margin */
          newline = TRUE;
       else
         {
          if (next_line->length + length_word + 1 > CURRENT_VIEW->margin_right)
             newline = TRUE;
         }
      }
   }
/*---------------------------------------------------------------------*/
/* Save the word to be wrapped...                                      */
/*---------------------------------------------------------------------*/
 buf = (CHARTYPE *)(*the_malloc)(length_word+CURRENT_VIEW->margin_left);
 if (buf == NULL)
   {
    display_error(30,"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 memcpy(buf,word_to_wrap,length_word);
/*---------------------------------------------------------------------*/
/* Remove the word to be wrapped from the focus line buffer; rec...    */
/*---------------------------------------------------------------------*/
 for (i=col_break;i<rec_len+1;i++)
    rec[i] = ' ';
 last_col = memrevne(rec,' ',max_line_length);
 rec_len = (last_col == (-1)) ? 0 : last_col + 1;
/*---------------------------------------------------------------------*/
/* We now should know if a new line is to added or not.                */
/*---------------------------------------------------------------------*/
 if (newline)
   {
    for (i=0;i<CURRENT_VIEW->margin_left-1;i++)
       (void)meminschr(buf,' ',0,MAX_LENGTH_OF_LINE,i+length_word);
    curr = add_line(CURRENT_FILE->first_line,curr,buf,length_word+CURRENT_VIEW->margin_left-1,curr->select);
    CURRENT_FILE->number_lines++;
   }
 else
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line+1L);
    (void)meminschr(rec,' ',CURRENT_VIEW->margin_left-1,MAX_LENGTH_OF_LINE,rec_len++);
    (void)meminsmem(rec,buf,length_word,CURRENT_VIEW->margin_left-1,
                    MAX_LENGTH_OF_LINE,rec_len);
    rec_len += length_word;
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line+1L);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
   }
 (*the_free)(buf);
/*---------------------------------------------------------------------*/
/* We now should know if the cursor is to wrap or stay where it is.    */
/*---------------------------------------------------------------------*/
 if (cursor_wrap)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    build_current_screen();
    cursor_down(TRUE);
    rc = Sos_firstchar("");
    for (i=0;i<cursor_offset+1;i++)
       rc = cursor_right(TRUE,FALSE);
   }
 else
   {
    if (INSERTMODEx)
       rc = cursor_right(TRUE,FALSE);
   }
 build_current_screen();
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short execute_split_join(bool action,bool aligned,short col)
#else
short execute_split_join(action,aligned,col)
bool action,aligned;
short col;
#endif
/***********************************************************************/
/* Parameters: action  - split or join line                            */
/*             aligned - whether to align text or not                  */
/*             col     - column to split on                            */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern unsigned short rec_len;
 extern CHARTYPE *rec;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short num_cols=0,num_blanks_focus=0,num_blanks_next=0;
 unsigned short x=0,y=0;
 LINE *curr=NULL;
 LINETYPE true_line=CURRENT_VIEW->focus_line;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: execute_split_join");
#endif
 if (CURRENT_VIEW->current_window != WINDOW_MAIN)
   {
    display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Reject the command if on the top or bottom line.                    */
/*---------------------------------------------------------------------*/
 if (FOCUS_TOF || FOCUS_BOF)
   {
    display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Copy any changes in the focus line to the linked list.              */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the focus_line.                   */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,true_line);

 getyx(CURRENT_WINDOW,y,x);
 switch(action)
   {
    case TRUE:
         memset(rec,' ',max_line_length);
         memcpy(rec,curr->line+col,curr->length);
         rec_len = curr->length-col;
/*---------------------------------------------------------------------*/
/* Calculate the number of leading blanks on the current line so that  */
/* the new line can have this many blanks prepended to align properly. */
/*---------------------------------------------------------------------*/
         if (aligned)
           {
            num_cols = memne(curr->line,' ',curr->length);
            if (num_cols == (-1))
               num_cols = 0;
            for (i=0;i<num_cols;i++)
               meminschr(rec,' ',0,max_line_length,rec_len++);
            rec_len = min(rec_len,max_line_length);
           }
         add_line(CURRENT_FILE->first_line,curr,(rec),rec_len,curr->select);
         CURRENT_FILE->number_lines++;
         pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         Sos_delend("");
         post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         break;
    case FALSE:
         if (curr->next->next == NULL)
           {
/*---------------------------------------------------------------------*/
/* Trying to join with the bottom of file line.                        */
/*---------------------------------------------------------------------*/
#ifdef TRACE
           trace_return();
#endif
           return(RC_INVALID_ENVIRON);
          }
/*---------------------------------------------------------------------*/
/* Calculate the number of leading blanks for the focus line and also  */
/* for the line to be joined. To align the join properly, we have to   */
/* remove up the number of leading blanks in the focus line from the   */
/* beginning of the line to be joined.                                 */
/*---------------------------------------------------------------------*/
         if (aligned)
           {
            num_blanks_focus = memne(curr->line,' ',curr->length);
            if (num_blanks_focus == (-1))
               num_blanks_focus = 0;
            num_blanks_next = memne(curr->next->line,' ',curr->length);
            if (num_blanks_next == (-1))
               num_blanks_next = 0;
            num_cols = min(num_blanks_focus,num_blanks_next);
           }
         else
            num_cols = 0;
         meminsmem(rec,curr->next->line+num_cols,curr->next->length-num_cols,
                      col,max_line_length,col);
         rec_len = min(max_line_length,col+curr->next->length-num_cols);
         post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         curr = delete_line(CURRENT_FILE->first_line,curr->next,DIRECTION_BACKWARD);
/*---------------------------------------------------------------------*/
/* If on the bottom line, use the previous line.                       */
/*---------------------------------------------------------------------*/
         if (CURRENT_BOF)
           {
            CURRENT_VIEW->current_line--;
            y++;
           }
/*---------------------------------------------------------------------*/
/* Decrement the number of lines counter for the current file and move */
/* the cursor to the appropriate line.                                 */
/*---------------------------------------------------------------------*/
         CURRENT_FILE->number_lines--;
         wmove(CURRENT_WINDOW,y,x);
         break;
   }
/*---------------------------------------------------------------------*/
/* Fix the positioning of the marked block (if there is one and it is  */
/* in the current view) and any pending prefix commands.               */
/*---------------------------------------------------------------------*/
 adjust_marked_lines(action,true_line,1L);
 adjust_pending_prefix(CURRENT_VIEW,action,true_line,1L);
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_put(CHARTYPE *params,bool putdel)
#else
short execute_put(params,putdel)
CHARTYPE *params;
bool putdel;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *tempfilename;
 extern CHARTYPE sp_path[MAX_FILE_NAME+1] ;
 extern CHARTYPE sp_fname[MAX_FILE_NAME+1] ;
 extern VIEW_DETAILS *vd_mark;
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 LINETYPE num_lines=0L,true_line=0L;
 bool append=FALSE;
 CHARTYPE *filename=NULL;
 short rc=RC_OK,start_col=0,end_col=max_line_length;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL|TARGET_SPARE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: execute_put");
#endif
/*---------------------------------------------------------------------*/
/* If there are no arguments, default to "1"...                        */
/*---------------------------------------------------------------------*/
 if (strcmp("",params) == 0)
    params = (CHARTYPE *)"1";
/*---------------------------------------------------------------------*/
/* Validate first argument as a target...                              */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 if ((rc = validate_target(params,&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If there is no second argument, no filename supplied...             */
/*---------------------------------------------------------------------*/
 if (target.spare == (-1))
   {
    append = FALSE;
    filename = tempfilename;
   }
 else
   {
    if ((rc = splitpath(strtrunc(target.rt[target.spare].string))) != RC_OK)
      {
       display_error(10,strtrunc(target.rt[target.spare].string),FALSE);
       free_target(&target);
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    strcpy(temp_cmd,sp_path);
    strcat(temp_cmd,sp_fname);
    filename = temp_cmd;
    append = TRUE;
   }
 true_line = target.true_line;
 num_lines = target.num_lines;
/*---------------------------------------------------------------------*/
/* If the marked block is a BOX block, set up the left and right column*/
/* values.                                                             */
/*---------------------------------------------------------------------*/
 if (target.rt[0].target_type == TARGET_BLOCK_CURRENT
 && MARK_VIEW->mark_type == M_BOX)
   {
    start_col = MARK_VIEW->mark_start_col-1;
    end_col = MARK_VIEW->mark_end_col-1;
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if ((rc = save_file(CURRENT_FILE,filename,TRUE,num_lines,true_line,append,start_col,end_col,FALSE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If we are executing a putd command, delete the target...            */
/*---------------------------------------------------------------------*/
 if (putdel)
    rc = DeleteLine(target.string);
 free_target(&target);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short execute_macro(CHARTYPE *params,bool error_on_not_found)
#else
short execute_macro(params,error_on_not_found)
CHARTYPE *params;
bool error_on_not_found;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern bool rexx_support;
 extern CHARTYPE number_of_files;
 extern LINETYPE original_screen_line;
 extern LINETYPE original_screen_column;
 extern LINETYPE original_file_line;
 extern LINETYPE original_file_column;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 short errnum=0;
 FILE *fp=NULL;
#define MAC_PARAMS  2
 CHARTYPE *word[MAC_PARAMS+1];
 unsigned short num_params=0;
 CHARTYPE *macroname=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: execute_macro");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters. At least 1 must be present, the filename.  */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,MAC_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Allocate some space for macroname...                                */
/*---------------------------------------------------------------------*/
 if ((macroname = (CHARTYPE *)(*the_malloc)((MAX_FILE_NAME+1)*sizeof(CHARTYPE))) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
/*---------------------------------------------------------------------*/
/* Find the fully qualified file name for the supplied macro name.     */
/*---------------------------------------------------------------------*/
 rc = get_valid_macro_file_name(word[0],macroname,&errnum);
/*---------------------------------------------------------------------*/
/* Validate the return code...                                         */
/*---------------------------------------------------------------------*/
 switch(rc)
   {
/*---------------------------------------------------------------------*/
/* If RC_OK, continue to process the macro...                          */
/*---------------------------------------------------------------------*/
    case RC_OK:
         break;
/*---------------------------------------------------------------------*/
/* If RC_FILE_NOT_FOUND and IMPOS is not on, display an error and exit.*/
/* If IMPOS is on, just return without displaying an error.            */
/*---------------------------------------------------------------------*/
    case RC_FILE_NOT_FOUND:
         if (error_on_not_found)
            display_error(errnum,macroname,FALSE);
         (*the_free)(macroname);
#ifdef TRACE
         trace_return();
#endif
         return(rc);
         break;
/*---------------------------------------------------------------------*/
/* All other cases, display error and return.                          */
/*---------------------------------------------------------------------*/
    default:
         (*the_free)(macroname);
         display_error(errnum,macroname,FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(rc);
   }
/*---------------------------------------------------------------------*/
/* Set in_macro = TRUE to stop multiple show_page()s being performed.  */
/*---------------------------------------------------------------------*/
 in_macro = TRUE;
/*---------------------------------------------------------------------*/
/* Save the values of the cursor position...                           */
/*---------------------------------------------------------------------*/
 get_cursor_position(&original_screen_line,&original_screen_column,
                     &original_file_line,&original_file_column);
/*---------------------------------------------------------------------*/
/* If REXX is supported, process the macro as a REXX macro...          */
/*---------------------------------------------------------------------*/
 if (rexx_support)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    rc = execute_macro_file(macroname,word[1]);
    if (rc != RC_OK)
      {
       display_error(54,(CHARTYPE *)"",FALSE);
       rc = RC_SYSTEM_ERROR;
      }
   }
 else
   {
/*---------------------------------------------------------------------*/
/* ...otherwise, process the file as a non-REXX macro file...          */
/*---------------------------------------------------------------------*/
    if ((fp = fopen(macroname,"r")) == NULL)
      {
       rc = RC_ACCESS_DENIED;
       display_error(8,macroname,FALSE);
      }
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    if (rc == RC_OK)
      {
       rc = execute_command_file(fp);
       fclose(fp);
      }
    (*the_free)(macroname);
    if (rc == RC_SYSTEM_ERROR)
       display_error(53,"",FALSE);
   }
/*---------------------------------------------------------------------*/
/* Set in_macro = FALSE to indicate we are out of the macro and do a   */
/* show_page() now as long as there are still file(s) in the ring.     */
/*---------------------------------------------------------------------*/
 in_macro = FALSE;
 if (number_of_files > 0)
   {
    build_current_screen();
    display_current_screen();
   }

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short execute_set_on_off(CHARTYPE *params,bool *flag)
#else
short execute_set_on_off(params,flag)
CHARTYPE *params;
bool *flag;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: execute_set_on_off");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameter. It must be ON or OFF.                       */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"off",params,3))
    *flag = FALSE;
 else
    if (equal((CHARTYPE *)"on",params,2))
       *flag = TRUE;
    else
      {
       display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       rc = RC_INVALID_OPERAND;
      }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short execute_set_row_position(CHARTYPE *params,short *base,short *off)
#else
short execute_set_row_position(params,base,off)
CHARTYPE *params;
short *base,*off;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: execute_set_row_position");
#endif
 if (*params == 'M'
 ||  *params == 'm')
   {
    *base = POSITION_MIDDLE;
    params++;
    if (blank_field(params))
       *off = 0;
    else
      {
       if ((*params != '-'
       &&  *params != '+')
       || ((*off = atoi(params)) == 0))
         {
          display_error(1,params,FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
   }
 else
   {
    if ((*off = atoi(params)) == 0)
      {
       display_error(1,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    *base = (*off > 0) ? POSITION_TOP : POSITION_BOTTOM;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short processable_line(VIEW_DETAILS *view,LINETYPE true_line,LINE *curr)
#else
short processable_line(view,true_line,curr)
VIEW_DETAILS *view;
LINETYPE true_line;
LINE *curr;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: processable_line");
#endif
 if (VIEW_TOF(view,true_line)
 ||  VIEW_BOF(view,true_line))
   {
#ifdef TRACE
    trace_return();
#endif
    return(LINE_TOF_EOF);
   }
 if (in_scope(curr))
   {
#ifdef TRACE
    trace_return();
#endif
    return(LINE_LINE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(LINE_SHADOW);
}
/***********************************************************************/
#ifdef PROTO
short execute_expand(CHARTYPE *params,bool inc_alt,bool use_tabs,bool add_to_recovery)
#else
short execute_expand(params,inc_alt,use_tabs,add_to_recovery)
CHARTYPE *params;
bool inc_alt,use_tabs,add_to_recovery;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 LINETYPE i=0L,num_actual_lines=0L;
 LINETYPE num_lines=0L,true_line=0L;
 unsigned short x=0,y=0;
 short direction=0,rc=RC_OK;
 LINE *curr=NULL;
 TARGET target;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 bool lines_based_on_scope=FALSE;
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 true_line = get_true_line();
 if (strcmp("",params) == 0)
    params = (CHARTYPE *)"1";
 initialise_target(&target);
 if ((rc = validate_target(params,&target,target_type,true_line,TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Determine in which direction we are working.                        */
/*---------------------------------------------------------------------*/
 if (target.num_lines < 0L)
   {
    direction = DIRECTION_BACKWARD;
    num_lines = -target.num_lines;
   }
 else
   {
    direction = DIRECTION_FORWARD;
    num_lines = target.num_lines;
   }
 true_line = target.true_line;
 lines_based_on_scope = (target.rt[0].target_type == TARGET_BLOCK_CURRENT) ? FALSE : TRUE;
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the true_line.                    */
/* This is the first line to change.                                   */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,true_line);
/*---------------------------------------------------------------------*/
/* Convert all tabs in the current line to spaces.                     */
/*---------------------------------------------------------------------*/
 for (i=0L,num_actual_lines=0L;;i++)
   {
    if (lines_based_on_scope)
      {
       if (num_actual_lines == num_lines)
          break;
      }
    else
      {
       if (num_lines == i)
          break;
      }
    rc = processable_line(CURRENT_VIEW,true_line+(LINETYPE)(i*direction),curr);
    switch(rc)
      {
       case LINE_SHADOW:
            break;
       case LINE_TOF_EOF:
            num_actual_lines++;
            break;
       default:
            tabs_convert(curr,TRUE,inc_alt,use_tabs,add_to_recovery);
            num_actual_lines++;
      }
    if (direction == DIRECTION_FORWARD)
       curr = curr->next;
    else
       curr = curr->prev;
   }
/*---------------------------------------------------------------------*/
/* If STAY is OFF, change the current and focus lines by the number    */
/* of lines calculated from the target.                                */
/*---------------------------------------------------------------------*/
 if (!CURRENT_VIEW->stay                                 /* stay is off */
 &&  target.num_lines != 0L)
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line = target.true_line+target.num_lines-(LINETYPE)direction;
 free_target(&target);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen(); 
 display_current_screen();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND
 &&  curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short execute_move_cursor(LENGTHTYPE col)
#else
short execute_move_cursor(col)
LENGTHTYPE col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short y=0,x=0,rc=RC_OK;
 COLTYPE new_screen_col=0;
 LENGTHTYPE new_verify_col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("execute.c: execute_move_cursor");
#endif
/*---------------------------------------------------------------------*/
/* Don't do anything for PREFIX window...                              */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }

 getyx(CURRENT_WINDOW,y,x);
 calculate_new_column(x,CURRENT_VIEW->verify_col,col,&new_screen_col,&new_verify_col);
 if (CURRENT_VIEW->verify_col != new_verify_col
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    CURRENT_VIEW->verify_col = new_verify_col;
    build_current_screen(); 
    display_current_screen();
   }
 wmove(CURRENT_WINDOW,y,new_screen_col);

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
