/* LZSMALL.C ---------------------------------------------------*
 * Small Archiv-Viewer based on LZAPI.
 *
 * Does about the same as LZVIEW without it's size (and Features...):
 *
 * - Allows to Select an Archiv to Display. (File Open)
 * - Shows the Archives Contents in an Owner-Drawn Listbox
 * - On a Double-Klick, extracts the File and Shows it with an
 *   external Viewer (or NOTPAD, if none is known)
 * - Allows to Delete Files from an Archiv with DEL-Key
 * - Multiple Files may be Dropped from File-Manager
 *
 * (C) 1995 Dipl. Ing. Bernd Herd Dipl. Ing Harald Nuding
 *--------------------------------------------------------------
 * This file is prepared to be Compiled with:
 * BORLAND C++     LZSMALL.IDE 16/32-Bit Version
 * MICOROSFT C 8.5 MAKEFILE.   32-Bit Version	
 *--------------------------------------------------------------*/

#include <windows.h>
#include <lzapi.h>
#include <commdlg.h>
#include <shellapi.h>
#include "lzsmall.rh"
#ifdef WIN32
# include <memory.h>	// Borland would just accept "MEM.H", but MS C...
#endif

// --------- Column-Definition Information ---------------
#define COLUMNS 6
int ColumnPos[COLUMNS+1] = { 10, 50, 90, 120, 150, 170, 190 };

// --------- Structure for Window-Private Data -----------
typedef struct {
	HWND  hwList;
	HFONT hFont;	// Font-Handle for Painting...
	int   ItemHeight;
	int   ColumnPos[COLUMNS+1];
	char  szArchivFile[200];
	BOOL  Inhibit;
	} LZDATA, NEAR *NPLZDATA;

// --------- Global Variables ----------------------------

static NPSTR AppName = "Small archiv viewer",
	     HeadingText = "Name\tSize\tCompressed\tRatio\tDate\tTime";
HINSTANCE hInst;
HWND	  hwMain;
HACCEL    hAccel;
LPCSTR	  CmdLine;

#define   FILETYPES 4
HBITMAP	  hft;


// --------- Dirty Tricks --------------------------------
#if defined(__BORLANDC__) && !defined(__WIN32__)
 void _cdecl _setupio(void) {}; // This Line saves a Bit of .EXE-File Size with Borland C++
#endif

#if defined(__BORLANDC__) && defined(__WIN32__)

// --------- Omit General Protection Handler Stuff--------
void   _RTLENTRY _EXPFUNC free(void _FAR *__block) { LocalFree( (HLOCAL) __block ); }
void  *_RTLENTRY _EXPFUNC malloc(size_t __size)    { return (void _FAR *) LocalAlloc(LMEM_FIXED, __size); }
void  *_RTLENTRY _EXPFUNC calloc(size_t __nitems, size_t __size) { return (void _FAR *) LocalAlloc(LPTR, __size * __nitems); }

void _ExceptInit(void) {};
void _GetExceptDLLinfoInternal(void) {};
long _DestructorCountPtr=0;

void _InitDefaultHander(void) {};
void _fpreset(void) {};
void _initmatherr() {};
void _initfmode(){};
void _ErrorMessage(void){};
int  _sys_nerr=0;
void _free_heaps(void) {};
#endif


// --------- Function Portotypes -------------------------
LRESULT FAR PASCAL LzSmallWndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
void OpenArchivFile(HWND hWnd, LPCSTR FileName);
void FileOpen(HWND hWnd);
void FilePrint(HWND hWnd);
void DrawItem(HWND hWnd, LPDRAWITEMSTRUCT dis);
void DrawLine(HWND hWnd, HDC hDC, LPRECT rcwhere, NPSTR p, int itemState, int near *Pos);
void PaintHeading(HWND hWnd);
void ShowSelectedFile(HWND hWnd);
void DeleteSelectedFile(HWND hWnd);
void OpenViewer(HWND hWnd, NPSTR FileName);
BOOL GetSelectedFileName(HWND hWnd, NPSTR FileName);
void DragDropFiles(HWND hWnd, HANDLE hDrop);
BOOL InitInstance(HINSTANCE hInst);
void ExitInstance(void);


// --------- Makros --------------------------------------
#ifdef WIN32
# define GetData(hWnd) ((NPLZDATA) GetWindowLong(hWnd, 0))
#else
# define GetData(hWnd) ((NPLZDATA) GetWindowWord(hWnd, 0))
#endif

#define ID_LISTBOX 70

#ifdef WIN32
# define _export
#endif


/* InitApplication ---------------------------------------------*
 * Registers the Window-Class
 *--------------------------------------------------------------*/
void InitApplication(HINSTANCE hinstCurrent)
{ static WNDCLASS cl = {
	0,		// style
	NULL,		// Window Procedure
	0,sizeof(HLOCAL),// Class Extrabytes, Window Extrabytes
	NULL,		// hInstance
	NULL,		// hIcon
	NULL,		// hCursor
	(HBRUSH) (COLOR_BTNFACE+1),	// Background-Brush
	MAKEINTRESOURCE(RID_MENU),	// Menu-Name / Id
	"LZSMALLWND"
  };

  cl.hInstance  = hinstCurrent;
  cl.hCursor    = LoadCursor(NULL, IDC_ARROW);
  cl.hIcon      = LoadIcon(hInst, MAKEINTRESOURCE(RID_ICON));
  cl.lpfnWndProc= LzSmallWndProc;

  RegisterClass(&cl);
}



/* InitInstance ------------------------------------------------*
 * Create Applications Main Window.
 *--------------------------------------------------------------*/
BOOL InitInstance(HINSTANCE hInst)
{       hAccel = LoadAccelerators(hInst, MAKEINTRESOURCE(RID_MENU));

	hwMain = CreateWindow(
		"LZSMALLWND", AppName,
		WS_OVERLAPPEDWINDOW | WS_VISIBLE | WS_CAPTION | WS_THICKFRAME,
		CW_USEDEFAULT,CW_USEDEFAULT,CW_USEDEFAULT,CW_USEDEFAULT,
		NULL, NULL, hInst, NULL
		);

	hft = LoadBitmap(hInst, MAKEINTRESOURCE(BID_FILETYPE));

	return hwMain!=NULL;
}



/* ExitInstance ------------------------------------------------*
 * Clean Up Applications Data
 *--------------------------------------------------------------*/
void ExitInstance(void)
{ DeleteObject(hft);
}




/* WinMain -----------------------------------------------------*
 * Windows Startap Procedure
 *--------------------------------------------------------------*/
#pragma argsused
int PASCAL WinMain(hinstCurrent, hinstPrevious, lpszCmdLine, nCmdShow)
HINSTANCE hinstCurrent;	/* handle of current instance	*/
HINSTANCE hinstPrevious;	/* handle of previous instance	*/
LPSTR lpszCmdLine;	/* address of command line	*/
int nCmdShow;	/* show-window type (open/icon)	*/
{  MSG msg;

   hInst   = hinstCurrent;
   CmdLine = lpszCmdLine;

   // -------- Register Window Class ------------------------
   if (!hinstPrevious)
       InitApplication(hinstCurrent);

   // -------- Create Main Window ---------------------------
   if (!InitInstance(hinstCurrent))
	return 1;

   // -------- Main Message-Loop ----------------------------
   while (GetMessage(&msg, NULL, 0,0))
   {
      TranslateAccelerator(hwMain, hAccel, &msg);
      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }

   // -------- Close Down application------------------------
   ExitInstance();

   return 0;
}



/* LzSmallWndProc ----------------------------------------------*
 * Window-procedure of Main Window
 *--------------------------------------------------------------*/
LRESULT FAR PASCAL _export LzSmallWndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ NPLZDATA LzData = (NPLZDATA) GetData(hWnd);

  switch (msg) {
     case WM_CREATE: LzData = (NPLZDATA) LocalAlloc(LPTR, sizeof(LZDATA));
     		     LzData->hFont = (HFONT) GetStockObject(ANSI_VAR_FONT);
#		     ifdef WIN32
		     SetWindowLong(hWnd, 0, (DWORD) LzData);
#		     else
		     SetWindowWord(hWnd, 0,  (WORD) LzData);
#		     endif

		     { HDC hDC = GetDC(hWnd);
		       HFONT hfo = SelectObject(hDC, LzData->hFont);
		       TEXTMETRIC tm;
		       int i;
		       GetTextMetrics(hDC, &tm);
		       LzData->ItemHeight = tm.tmHeight + tm.tmInternalLeading;
		       for (i=0; i<=COLUMNS; ++i)
			  LzData->ColumnPos[i] = MulDiv(ColumnPos[i], tm.tmAveCharWidth, 2);
		       SelectObject(hDC, hfo);
		       ReleaseDC(hWnd, hDC);
		     }
		     LzData->hwList = CreateWindow(
			"LISTBOX", "",
			WS_CHILD | WS_VISIBLE | LBS_OWNERDRAWFIXED | LBS_HASSTRINGS | LBS_NOTIFY | WS_BORDER,
			0,LzData->ItemHeight,1,1,
			hWnd, (HMENU) ID_LISTBOX, hInst, NULL
			);

		     if (CmdLine && *CmdLine)
			OpenArchivFile(hWnd, CmdLine);

		     DragAcceptFiles(hWnd, TRUE);

		     break;

     case WM_SIZE:    SetWindowPos(LzData->hwList, NULL, 0, 0, LOWORD(lParam), HIWORD(lParam)-LzData->ItemHeight, SWP_NOMOVE);
		      break;

     case WM_DESTROY: LocalFree((HLOCAL) LzData );
		      DragAcceptFiles(hWnd, FALSE);
		      break;
     case WM_DROPFILES: DragDropFiles(hWnd, (HANDLE) wParam); break;
     case WM_SETFOCUS:  SetFocus(LzData->hwList);
                        break;

     case WM_MEASUREITEM: ((LPMEASUREITEMSTRUCT) (lParam))->itemHeight = LzData->ItemHeight; return 0;
     case WM_DRAWITEM   : DrawItem(hWnd, (LPDRAWITEMSTRUCT) lParam); return 0;

     case WM_CLOSE:  if (LzData->Inhibit) {
			MessageBeep(0);
			return 0;
		     }
		     else
		        PostQuitMessage(0);
		     break;
     case WM_PAINT: PaintHeading(hWnd); return 0;
     case WM_COMMAND:
	if (!LzData->Inhibit)
	switch (LOWORD(wParam)) {
	    case IDM_FILEOPEN : FileOpen(hWnd); break;
	    case IDM_FILEPRINT: FilePrint(hWnd); break;
	    case IDM_SELECT   : ShowSelectedFile(hWnd); break;
	    case IDM_EXIT     : PostMessage(hWnd, WM_CLOSE, 0, 0L); break;
	    case IDM_DELETE   : DeleteSelectedFile(hWnd); break;
            case IDM_WINFILE  : WinExec("WINFILE", SW_SHOW); break;
	    case ID_LISTBOX:
#		 ifdef WIN32
		 if (HIWORD(wParam)==LBN_DBLCLK)
#		 else
		 if (HIWORD(lParam)==LBN_DBLCLK)
#		 endif
		 	    ShowSelectedFile(hWnd);
	}
	return 0;
  }
  return DefWindowProc(hWnd, msg, wParam, lParam);
}



/* FileOpen ----------------------------------------------------*
 * Select an Archiv to Display
 *--------------------------------------------------------------*/
#ifdef OBSOLETE
void FileOpen(HWND hWnd)
{ NPLZDATA LzData = (NPLZDATA) GetData(hWnd);
  OPENFILENAME near *of = (OPENFILENAME near *) LocalAlloc(LPTR, sizeof(OPENFILENAME));
  char         szFile[120],
	       szFilter[200];
  LPCSTR       ExtensionString = LAGetExtensionString(FALSE);
  NPSTR	       p;

  // ------ Prepare GetOpenFileName - Call -----------------
  wsprintf(szFilter, "Archives (%s)|%s|All Files (*.*)|*.*|", ExtensionString, ExtensionString);

  for (p=szFilter; *p; ++p) if (*p=='|') *p=0;

  of->lStructSize = sizeof(OPENFILENAME);
  of->hwndOwner   = hWnd;
  of->lpstrFilter = szFilter;
  of->nFilterIndex= 1;
  of->lpstrFile   = szFile;
  of->nMaxFile    = sizeof(szFile);
  of->Flags	  = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST;

  lstrcpy(szFile, ExtensionString);

  // -------- Ask for Filename from User -------------------
  if (GetOpenFileName(of)) {
       OpenArchivFile(hWnd, szFile);
  }

  LocalFree((HLOCAL) of);
  SetFocus(LzData->hwList);
}
#endif

void FileOpen(HWND hWnd)
{ NPLZDATA LzData = (NPLZDATA) GetData(hWnd);
  OPENFILENAME near *of = (OPENFILENAME near *) LocalAlloc(LPTR, sizeof(OPENFILENAME));
  char         szFile[120],
	       szFilter[200];
  LPCSTR       ExtensionString = LAGetExtensionString(FALSE);
  NPSTR	       p;
  OFSTRUCT     dummy; 

  // ------ Prepare GetOpenFileName - Call -----------------
  wsprintf(szFilter, "Archives (%s)|%s|All Files (*.*)|*.*|", ExtensionString, ExtensionString);

  for (p=szFilter; *p; ++p) if (*p=='|') *p=0;

  of->lStructSize = sizeof(OPENFILENAME);
  of->hwndOwner   = hWnd;
  of->lpstrFilter = szFilter;
  of->nFilterIndex= 1;
  of->lpstrFile   = szFile;
  of->nMaxFile    = sizeof(szFile);
  of->Flags	  = OFN_PATHMUSTEXIST;

  lstrcpy(szFile, ExtensionString);

  // -------- Ask for Filename from User -------------------
  if (GetSaveFileName(of))
  {
       if (HFILE_ERROR==OpenFile(szFile, &dummy, OF_EXIST))
       {
           if (IDYES==MessageBox(hWnd, "Archiv does not exist. Create it?", "Warning", MB_YESNO))
           {
                lstrcpy(LzData->szArchivFile, szFile);

                // ---- Copy Directory-Listing into the ListBox -------
                SendMessage(LzData->hwList, LB_RESETCONTENT, 0, 0L);
           }
       }
       else
           OpenArchivFile(hWnd, szFile);
  }

  LocalFree((HLOCAL) of);
  SetFocus(LzData->hwList);
}



/* OpenArchivFile ----------------------------------------------*
 * Read in File-Listing of Archiv-File
 *--------------------------------------------------------------*/
void OpenArchivFile(HWND hWnd, LPCSTR FileName)
{ NPLZDATA LzData = (NPLZDATA) GetData(hWnd);
  LACTL near *Ctl     = (LACTL near *) LocalAlloc(LPTR, sizeof(LACTL));
  HGLOBAL      hg;
  LPSTR	       pg;
  NPSTR	       p;

  LAERR	       err;

  int	       Count=0;
  RECT	       rc;		

  char szBuf[120];

       // --- Prepare Memory for Listing -------------------
       hg = GlobalAlloc(GHND, 0xffff);
       pg = (LPSTR) GlobalLock(hg);

       // --- Get Archiv Listing ---------------------------
       Ctl->lStructSize    = sizeof(LACTL);
       Ctl->hwndOwner      = hWnd;
       Ctl->lpstrListing   = pg;
       Ctl->lpstrArchivFile= FileName;
       if ((err=LAList(Ctl))!=LAE_OK)
	   LAErrMsg(err, Ctl);
       else {
	  // ---- Set Window Title ------------------------------
	  wsprintf(szBuf, "%s [%s]", (LPSTR) AppName, (LPSTR) FileName);
	  SetWindowText(hWnd, szBuf);
	  lstrcpy(LzData->szArchivFile, FileName);

	  // ---- Copy Directory-Listing into the ListBox -------
	  SendMessage(LzData->hwList, LB_RESETCONTENT, 0, 0L);
	  while (*pg) {

	    // -- Copy a Line... --------------------------------
	    for (p=szBuf; *pg && *pg!='\r' && *pg!='\n'; )
		*p++ = *pg++;
	    *p=0;
	    SendMessage(LzData->hwList, LB_ADDSTRING, 0, (LPARAM) (LPSTR) szBuf);
            Count++;

	    // -- ignore CR+LF ---------------------------------------
	    while (*pg=='\r' || *pg=='\n') ++pg;
	  }

	  // ---- Adopt Window Size --------------------------------
	  SetRect(&rc, 0, 0, LzData->ColumnPos[COLUMNS]+ 2*GetSystemMetrics(SM_CXBORDER),
//			     (Count+1) * LzData->ItemHeight + 2*GetSystemMetrics(SM_CYBORDER));
			     (Count+1) * (int) SendMessage(LzData->hwList, LB_GETITEMHEIGHT, 0, 0L) + 2*GetSystemMetrics(SM_CYBORDER));
	  AdjustWindowRect(&rc, GetWindowLong(hWnd, GWL_STYLE), TRUE);
	  SetWindowPos(hWnd, NULL, 0,0,rc.right-rc.left, rc.bottom-rc.top, SWP_NOMOVE);
       }

       LocalFree((HLOCAL) Ctl);
       GlobalUnlock(hg);
       GlobalFree(hg);
}

/* DrawLine ----------------------------------------------------*
 * Draw a Line with Tabs
 *--------------------------------------------------------------*/
void DrawLine(HWND hWnd, HDC hDC, LPRECT rcwhere, NPSTR p, int itemState, int near *Pos)
{ NPLZDATA LzData = (NPLZDATA) GetData(hWnd);
  NPSTR ptab;
  HFONT hfo = SelectObject(hDC, LzData->hFont);
  RECT  rc;
  int   sp;


  for (sp=0, ptab=p;
       sp<COLUMNS &&	     *p;
       sp++, p=ptab, ++Pos) {
     // -------- Scan for next tab ----------------------------
     while (*ptab && *ptab!='\t') ++ptab;

     // -------- Get Column Rectangle -------------------------
     SetRect(&rc, rcwhere->left+Pos[0], rcwhere->top+1, rcwhere->left+Pos[1], rcwhere->bottom-1);

     SetTextAlign(hDC, sp ? TA_RIGHT : TA_LEFT);

     SetTextColor(hDC, GetSysColor(itemState>=0 && itemState & ODS_SELECTED ? COLOR_CAPTIONTEXT : COLOR_WINDOWTEXT));
     SetBkColor  (hDC, GetSysColor(itemState<0  ? COLOR_BTNFACE : itemState & ODS_SELECTED ? COLOR_ACTIVECAPTION : COLOR_WINDOW));

     ExtTextOut(hDC, sp?rc.right:rc.left, rc.top, ETO_CLIPPED | ETO_OPAQUE, &rc, p, ptab-p, NULL);

     // -------- Step over tab --------------------------------
     if (*ptab) ++ptab;
  }

  SelectObject(hDC, hfo);
}


/* DrawItem ----------------------------------------------------*
 * Reaction in WM_DRAWITEM
 *--------------------------------------------------------------*/
void DrawItem(HWND hWnd, LPDRAWITEMSTRUCT dis)
{ NPLZDATA LzData = (NPLZDATA) GetData(hWnd);
  char 	   szText[120];
  NPSTR	   p, pp;
  HDC	   hMemDC;
  HBITMAP  hbo;

# define BM_BMP 0
# define BM_TXT 1
# define BM_LZH 2
# define BM_REG 3

  struct { NPSTR Ext; char Id;} TypeList[] = {
	{ "LZH", BM_LZH },
	{ "ZIP", BM_LZH },
	{ "ARJ", BM_LZH },
	{ "RAR", BM_LZH },
	{ "TAR", BM_LZH },
	{ "ARC", BM_LZH },
	{ "ZOO", BM_LZH },
	{ "BMP", BM_BMP },
	{ "PCX", BM_BMP },
	{ "TXT", BM_TXT },
	{ "C"  , BM_TXT },
	{ "H"  , BM_TXT },
	{ "PAS", BM_TXT },
	{ NULL , BM_REG }
	}, near *ps;

  szText[0]=0;
  SendMessage(LzData->hwList, LB_GETTEXT, dis->itemID, (LPARAM) (LPSTR) szText);
  DrawLine(hWnd, dis->hDC, &dis->rcItem, szText, dis->itemState, LzData->ColumnPos);

  // ---------- Scan for File-Extension --------------------
  for (p=szText, pp=NULL; *p && *p!='\t'; ++p)
     switch (*p) {
	case '.' : pp=p+1; break;
	case '\\': pp=NULL; break;
     }

  // ---------- Distinguish File-Type ----------------------
  if (*p=='\t' &&
      pp!=NULL	
     ) {
     *p=0;

     for (ps=TypeList; ps->Ext && lstrcmpi(pp, ps->Ext); ++ps)
	;

     hMemDC = CreateCompatibleDC(dis->hDC);
     hbo    = SelectObject(hMemDC, hft);
 
     BitBlt(dis->hDC, dis->rcItem.left, dis->rcItem.top, 12, 13, hMemDC, ps->Id*12,0, SRCCOPY);

     SelectObject(hMemDC, hbo);
     DeleteDC(hMemDC);
  }
}


/* PaintHEading ------------------------------------------------*
 * Draw Heading abve List.
 *--------------------------------------------------------------*/
void PaintHeading(HWND hWnd)
{ NPLZDATA    LzData = (NPLZDATA) GetData(hWnd);
  PAINTSTRUCT ps;
  HDC 	      hDC = BeginPaint(hWnd, &ps);
  RECT        rc;

  GetClientRect(hWnd, &rc);
  rc.bottom=LzData->ItemHeight;
  DrawLine(hWnd, hDC, &rc, HeadingText, -1, LzData->ColumnPos);
  EndPaint(hWnd, &ps);
}



/* GetSelectedFileName -----------------------------------------*
 * Get Name of File selected in Listbox
 *--------------------------------------------------------------*/
BOOL GetSelectedFileName(HWND hWnd, NPSTR FileName)
{ NPLZDATA    LzData = (NPLZDATA) GetData(hWnd);
  int 	      Count  = (int) SendMessage(LzData->hwList, LB_GETCOUNT, 0, 0L),
	      Current= (int) SendMessage(LzData->hwList, LB_GETCURSEL,0, 0L);
  NPSTR	      p;

  if (Current < Count) {
    SendMessage(LzData->hwList, LB_GETTEXT, Current, (LPARAM) (LPSTR) FileName);

    // -------------------------------------------------------
    for (p=FileName; *p!='\t'; ++p) ;
    *p=0;
  }

  return Current<Count;
}

/* ShowSelectedFile --------------------------------------------*
 * Open the File selected in the List with a Viewer
 *--------------------------------------------------------------*/
void ShowSelectedFile(HWND hWnd)
{ NPLZDATA    LzData = (NPLZDATA) GetData(hWnd);
  char 	   FileName[120],
	   TempName[120];
  OFSTRUCT dummy;
  LACTL    near *Ctl = (LACTL near *) LocalAlloc(LPTR, sizeof(LACTL));
  NPSTR    pl;
# ifndef WIN32
  NPSTR    p;
# endif
  LAERR	   e;
  HCURSOR  hco = SetCursor( LoadCursor(NULL, IDC_WAIT));

  if (GetSelectedFileName(hWnd, FileName)) {
#  ifdef WIN32
   GetTempPath(sizeof(TempName), TempName);
   pl = TempName+lstrlen(TempName)-1;
   *pl=0;
#  else
   GetTempFileName(0, "LZH", 0, TempName);		// Get at Temporary File Name from Windows
   OpenFile(TempName, &dummy, OF_DELETE);

   for (p=TempName, pl=NULL; *p; ++p) 
	if (*p=='\\') pl=p;

   if (pl) *pl=0;
      else  pl=p;
#  endif
   Ctl->lStructSize     = sizeof(Ctl);
   Ctl->hwndOwner       = hWnd;
   Ctl->lpstrArchivFile = LzData->szArchivFile;
   Ctl->lpstrWildcards  = FileName;
   Ctl->lpstrPath       = TempName;
   Ctl->Flags	        = LAF_SHORTNAMES;
//   Ctl->lpfnCallback    = ExpandCallback;

   LzData->Inhibit=TRUE;

   e = LAExtract(Ctl);

   wsprintf(pl, "\\%s", (LPSTR) FileName);

   if (e)
      LAErrMsg(e, Ctl);
   else {
	OpenViewer(hWnd, TempName);
   }

   LzData->Inhibit=FALSE;
  }

  LocalFree((HLOCAL) Ctl);
  SetCursor(hco);
}





/* OpenViewer---------------------------------------------------*
 * Execute external Viewer-Application (Paintbrush, notepad...)
 *--------------------------------------------------------------*/
void OpenViewer(HWND hWnd, NPSTR FileName)
{ NPSTR 		p;
  char 			ModuleFileName[120];
  HINSTANCE 		Instance;
  MSG 			msg;
# ifdef WIN32
  STARTUPINFO		si;
  PROCESS_INFORMATION 	pi;
# endif

#ifdef WIN32
  for (p=FileName; *p && *p!='.'; ++p) ;
  if (!lstrcmpi(p, ".LZH") ||
      !lstrcmpi(p, ".ZIP") ||
      !lstrcmpi(p, ".ARC") ||
      !lstrcmpi(p, ".ARJ") ||
      !lstrcmpi(p, ".RAR") ||
      !lstrcmpi(p, ".TAR")) {

       /* Start a Second Instance of yourself.... */
       GetModuleFileName(hInst, ModuleFileName, sizeof(ModuleFileName));
       lstrcat(ModuleFileName, " ");
       lstrcat(ModuleFileName, FileName);
     } else {
       /* Find out which EXE cutable Program shall handle our File. */
       if (FindExecutable(FileName, ".", ModuleFileName)<(HINSTANCE) 32)
	  lstrcpy(ModuleFileName, "NOTEPAD");

       lstrcat(ModuleFileName, " ");
       lstrcat(ModuleFileName, FileName);
     }

     memset(&si, 0, sizeof(si)); si.cb = sizeof(si);
     memset(&pi, 0, sizeof(pi));
     if (CreateProcess(
		NULL, 			/* ImageName */
		ModuleFileName,		/* Command Line */
		NULL,			/* Process Security Attributes */
		NULL,			/* Thread Security Attributes */
		FALSE,			/* Intherit Handles */
		NORMAL_PRIORITY_CLASS,	/* fdwCreate */
		NULL,			/* Environment String */
		NULL,			/* Current Directory */
		&si,			/* Startupinfo-Structure */
		&pi                      /* PROCESS_INFO-Structure */
	 )) {
	       while (WaitForSingleObject(pi.hProcess, 200)==WAIT_TIMEOUT)
		   if (PeekMessage(&msg, NULL, 0,0,PM_REMOVE))
		   {
//		      TranslateAccelerator(hwMain, hAccel, &msg);
		      TranslateMessage(&msg);
		      DispatchMessage(&msg);
		   }
	       CloseHandle(pi.hProcess);
	       CloseHandle(pi.hThread);
	    }
	    else { Instance = NULL; MessageBox(hWnd, "CreateProcess Failed", "error", MB_OK | MB_ICONSTOP); }
#else
  for (p=FileName; *p && *p!='.'; ++p) ;
  if (!lstrcmpi(p, ".LZH") ||
      !lstrcmpi(p, ".ZIP") ||
      !lstrcmpi(p, ".ARC") ||
      !lstrcmpi(p, ".ARJ") ||
      !lstrcmpi(p, ".RAR") ||
      !lstrcmpi(p, ".TAR")) {

       /* Start a Second Instance of yourself.... */
       GetModuleFileName(hInst, ModuleFileName, sizeof(ModuleFileName));
       lstrcat(ModuleFileName, " ");
       lstrcat(ModuleFileName, FileName);
       Instance = WinExec(ModuleFileName, SW_SHOW);
     } else {
       /* Start a Program to show us the File */
       Instance =
       ShellExecute(hWnd, "open",
		    FileName, ".", "",
		    SW_SHOW
		   );

       if (Instance < HINSTANCE_ERROR) {
	  lstrcpy(ModuleFileName, "NOTEPAD ");
	  lstrcat(ModuleFileName, FileName);
	  Instance = WinExec(ModuleFileName, SW_SHOW);
       }
    }

    /*   Wait until the User closes down the Viewer  */
    if (Instance > HINSTANCE_ERROR) {
	while (GetModuleUsage(Instance)!=NULL) {
	   if (PeekMessage(&msg, NULL, 0,0,PM_REMOVE))
	   {
//	      TranslateAccelerator(hwMain, hAccel, &msg);
	      TranslateMessage(&msg);
	      DispatchMessage(&msg);
	   }
	}
    }
#endif
}




/* DeleteSelectedFile ------------------------------------------*
 * Delete the selected File from the Archiv.
 *--------------------------------------------------------------*/
void DeleteSelectedFile(HWND hWnd)
{ NPLZDATA    LzData = (NPLZDATA) GetData(hWnd);
  char 	   FileName[120],
	   TempName[120],
	   StringBuf[120];
  LACTL    near *Ctl = (LACTL near *) LocalAlloc(LPTR, sizeof(LACTL));
  LAERR	   e;
  HCURSOR  hco = SetCursor( LoadCursor(NULL, IDC_WAIT));
  int	   Current;

  if (GetSelectedFileName(hWnd, FileName) &&
     LoadString(hInst, IDS_DELETEQUESTION, StringBuf, sizeof(StringBuf)) &&
     IDYES == MessageBox(hWnd, StringBuf, "Security", MB_YESNOCANCEL | MB_ICONQUESTION)
     ) {

   Ctl->lStructSize     = sizeof(Ctl);
   Ctl->hwndOwner       = hWnd;
   Ctl->lpstrArchivFile = LzData->szArchivFile;
   Ctl->lpstrWildcards  = FileName;
   Ctl->lpstrPath       = TempName;
   Ctl->Flags	        = LAF_SHORTNAMES;

   LzData->Inhibit=TRUE;

   e = LADelete(Ctl);

   if (e)
      LAErrMsg(e, Ctl);
   else {
	Current= (int) SendMessage(LzData->hwList, LB_GETCURSEL,0, 0L);
	SendMessage(LzData->hwList, LB_DELETESTRING, Current, 0L);
   }

   LzData->Inhibit=FALSE;
  }

  LocalFree((HLOCAL) Ctl);
  SetCursor(hco);
}


/* DragDropFiles -----------------------------------------------*
 * WM_DROP-Message: Fiels have been dropped from File-Manager
 *--------------------------------------------------------------*/
void DragDropFiles(HWND hWnd, HANDLE hDrop)
{ NPLZDATA    LzData = (NPLZDATA) GetData(hWnd);
  UINT	      Count  = DragQueryFile(hDrop, -1, NULL, 0),
	      i;
  HGLOBAL     hg = GlobalAlloc(GHND, 0xffffL);
  LPSTR	      pg = (LPSTR) GlobalLock(hg),
	      pn = pg;
  LACTL    near *Ctl = (LACTL near *) LocalAlloc(LPTR, sizeof(LACTL));
  HCURSOR  	hco = SetCursor( LoadCursor(NULL, IDC_WAIT));
  LAERR 	e;

  SetFocus(LzData->hwList);

  for (i=0; i<Count; ++i) {

    // ------- Get next Drop-Filenname -----------------------
    DragQueryFile(hDrop, i, pn, 120);

    if (i!=Count-1) {
	pn+=lstrlen(pn);
	*pn++=' ';
    }
  }

  DragFinish(hDrop);

  if (Count) {

   Ctl->lStructSize     = sizeof(Ctl);
   Ctl->hwndOwner       = hWnd;
   Ctl->lpstrArchivFile = LzData->szArchivFile;
   Ctl->lpstrWildcards  = pg;
//   Ctl->lpstrPath       = NULL;
   Ctl->Flags	        = LAF_SHORTNAMES;

   LzData->Inhibit=TRUE;

   e = LAAppend(Ctl);

   LzData->Inhibit=FALSE;
   if (e)
	LAErrMsg(e, Ctl);
   else OpenArchivFile(hWnd, LzData->szArchivFile);
  }

  GlobalUnlock(hg);
  GlobalFree(hg);
  LocalFree((HLOCAL) Ctl);
  SetCursor(hco);
}



/* FilePrint ---------------------------------------------------*
 * Print Archiv-Listing
 *--------------------------------------------------------------*/
void FilePrint(HWND hWnd)
{ NPLZDATA    LzData = (NPLZDATA) GetData(hWnd);
  int 	      Count  = (int) SendMessage(LzData->hwList, LB_GETCOUNT, 0, 0L),
	      Current,
	      Y, Page, i;

  PRINTDLG near *pd=(PRINTDLG near *) LocalAlloc(LPTR, sizeof(PRINTDLG));
  HDC	      hDC;
  DOCINFO     di;

  char	      szBuf[200];

  TEXTMETRIC  tm;
  int	      Height, Width,
	      Border,
	      VertRes,
	      PrintColumnPos[COLUMNS+1];

  RECT 	      rc;

  BOOL        PrintThisPage,
	      FirstPage=TRUE;

  HFONT	      hfo;

  /* Initialize the necessary PRINTDLG structure members. */

  pd->lStructSize = sizeof(PRINTDLG);
  pd->hwndOwner = hWnd;
  pd->Flags = PD_RETURNDC | PD_NOSELECTION | PD_USEDEVMODECOPIES;
  pd->nMaxPage = 0xfffe;

  /* Print a test page if successful */

  if (Count!=0 &&
      PrintDlg(pd) != 0) {

    // ----------- Device-Context ----------------------------
    hDC = pd->hDC;
    di.cbSize = sizeof(di);
    di.lpszDocName = LzData->szArchivFile;
    di.lpszOutput  = NULL;

    StartDoc(hDC, &di);

    // ----------- Font Height -------------------------------
    hfo = SelectObject(hDC, LzData->hFont);

    GetTextMetrics(hDC, &tm);
    Height = tm.tmHeight + tm.tmInternalLeading;
    Width  = tm.tmAveCharWidth;
    Border = 6 * Width;
    VertRes= GetDeviceCaps(hDC, VERTRES);
    for (i=0; i<=COLUMNS; ++i)
       PrintColumnPos[i] = MulDiv(ColumnPos[i], tm.tmAveCharWidth, 2);

    for (Current=Page=0; Current<Count; ++Current) {
	// ----------- Page Change... ------------------------
	if (!Current || Y>VertRes-4*Height) {
	   Y = 5*Height;
	   Page++;

	   PrintThisPage = Page>=pd->nFromPage &&
			   (!pd->nToPage || pd->nToPage >= Page);

	   if (PrintThisPage) {

	      // -------- Chaneg Page, reselect Font ------------
	      if (!FirstPage)
		 EndPage(hDC);
	      StartPage(hDC);
	      FirstPage=FALSE;
	      SelectObject(hDC, LzData->hFont);

	      // -------- Print Heading -------------------------
	      GetWindowText(hWnd, szBuf, sizeof(szBuf));
	      wsprintf(szBuf+lstrlen(szBuf), " Page %d", Page);

	      TextOut(hDC,  Border+PrintColumnPos[0], 2*Height, szBuf, lstrlen(szBuf));

	      SetRect(&rc, Border, 4*Height, 80 * Width, 5*Height);
	      DrawLine(hWnd, hDC, &rc, HeadingText, -1, PrintColumnPos);

	      MoveToEx(hDC, Border+PrintColumnPos[0]      , 5*Height-1, NULL);
	      LineTo(  hDC, Border+PrintColumnPos[COLUMNS], 5*Height-1);
	   }
	}

	// ----------- Print next Line ---------------------------
	SendMessage(LzData->hwList, LB_GETTEXT, Current, (LPARAM) (LPSTR) szBuf);

	SetRect(&rc, Border, Y, 80 * Width, Y+Height);

	if (PrintThisPage)
	   DrawLine(hWnd, hDC, &rc, szBuf, 0, PrintColumnPos);

	Y+=Height;
    }

    SelectObject(hDC, hfo);
    EndPage(hDC);
    EndDoc(hDC);

    DeleteDC(hDC);
    if (pd->hDevMode != NULL)
       GlobalFree(pd->hDevMode);
    if (pd->hDevNames != NULL)
       GlobalFree(pd->hDevNames);
  }

  LocalFree((HLOCAL) pd);
}

