//      BRECORD.CPP  ...  Boot record routines

//      Copyright 1992 by George H. Mealy

#include    "brecord.h"
#include	<stdlib.h>
#include    <string.h>
#include    <alloc.h>
#include	<dos.h>
#include	<stdio.h>

brecord::brecord(char drv)          // Constructor.  Reads boot record.
{
    rootdict = NULL;
    FAT = NULL;
    FAT12 = NULL;
    drive = drv&0xf - 1;
	if (absread(drive, 1, 0, bjunk1))
        {puts("Cannot read boot sector\n\a"); exit(-1);}
    sectorsize = bbytes;
    clustersize = bsect;
    dictsize = bndir;
    dictsects = dictsize*32/sectorsize;
    nFATs = bfat;
	FATsize = bfatsect;
    data_origin = brsvd+nFATs*FATsize+dictsects;
    if (btotal) b32total = btotal;
    is16bit = b32total >= 20740;
    endtest = is16bit? 0xfff8: 0xff8;
    endcluster = (b32total - data_origin) / clustersize + 2;
	struct dfree dfree;
	getdfree(drive + 1, &dfree);
	freecount = dfree.df_avail;
}

brecord::~brecord()
{
    if (rootdict) delete rootdict;
    if (FAT) delete FAT;
}

/*	Print the boot record data	*/

void brecord::listboot()      /* Print boot record data */
{
    *btail = '\0';
	printf(
"\33[2J\n\n\n	\33[1;4mDrive %c: boot record information\33[0m\n\n\
    Vendor ID           %8s\n    Bytes per sector    %u\n\
    Sectors per cluster	%u\n",
    'A' + drive, bvendor, bbytes, bsect);
	printf(
"    # reserved sectors  %u\n    # FATs              %u\n\
    # directory entries %u\n    # sectors           %lu\n",
    brsvd, bfat, bndir, b32total);
	printf(
"    Media type          %2X\n    Sectors per FAT     %u\n\
    Sectors per track   %u\n    # heads             %u\n\
    # hidden sectors    %lu\n    # free clusters     %u\n\
    Bits per FAT entry  %u\n\
    Volume serial       %u.%u\n\
    Volume label        %11s\n\n",
    btype, bfatsect, btracks,
    bheads, bhidden, freecount, is16bit? 16: 12,
    FP_SEG(bserial), FP_OFF(bserial), bvolume);
}

/*	Get the root directory	*/

void brecord::getrootdict()
{
    unsigned start = brsvd+nFATs*FATsize, err;
    if (!rootdict) rootdict = (DIR *)new char[dictsects*sectorsize];
    #pragma option -w-pia
    if (absread(drive, dictsects, start, rootdict))
        {perror("\aBad dictionary read"); exit(-1);}
}

/*	Get the file allocation table	*/

void brecord::getFAT()
{
    FAT12 = FAT = (unsigned *)malloc((FATsize/2)*sectorsize);
    absread(drive, FATsize, brsvd, FAT);    /* Read a FAT */
}

int brecord::readdir(unsigned cluster)
{
	int i = 0;
    while (cluster < endtest)
	{	// Reads a subdirectory into the root directory buffer
		absread(drive, clustersize, sector(cluster), &rootdict[i]);
        cluster = FATentry(cluster);
        i += (sectorsize/sizeof(DIR)) * clustersize;
	}
    rootdict[i].stem[0] = '\0';

    return i;
}

int brecord::writedir(unsigned cluster)
{
	int i = 0;
	if (! cluster)
	      return abswrite(drive, dictsects,
			brsvd+nFATs*FATsize, rootdict);
    while (1)
	{
		abswrite(drive, clustersize, sector(cluster), &rootdict[i]);
		if (cluster >= endtest) break;
		cluster = FATentry(cluster);
        i += (sectorsize/sizeof(DIR))*clustersize;
		}
    return i;
}

int brecord::clength(unsigned x) 			/* Get number of clusters in file */
{
	unsigned n = 0;
	for (; x < 0xfff8; x = FATentry(x)) n++;
	return n;
}


unsigned brecord::sector(unsigned cluster) 	/* Convert cluster # to sector # */
{
	return (cluster-2)*clustersize+data_origin;
}

unsigned brecord::FATentry(unsigned c) 			/* Get FAT entry for cluster */
{
	if (is16bit) return FAT[c];	/* 16 bit FAT entry */
	else {				/* 12 bit FAT entry */
		unsigned index = (3*c)>>1;
		unsigned word = FAT12[index] | (FAT12[index+1]<<8);
		unsigned result = (c&1? word>>4: word&0xfff);
		return result;
		}
}

void brecord::derase()              // Quick unformat
{
    if (drive > 1) {puts("Cannot recycle a hard disk"); return;}
    bdos(0xd, 0, 0);                            // Flush I/O
    FAT = (unsigned *)new char[sectorsize];     // Allocate only one FAT sector
    absread(drive, 1, brsvd, FAT);              // Read first sector of FAT
    memset((char *)FAT+3, 0, sectorsize-3);     // Set most of it to zeroes
    abswrite(drive, 1, brsvd, FAT);             // Write to both FATs
    abswrite(drive, 1, brsvd+FATsize, FAT);
    memset(FAT, 0, 3);                          // Clear the rest to zeroes
    for (int i = 1; i < FATsize; i++) {
        abswrite(drive, 1, brsvd+i, FAT);       // Erase both FATs
        abswrite(drive, 1, brsvd+FATsize+i, FAT);
    }
    for (i = 0; i < dictsects; i++)
        abswrite(drive, 1, brsvd+2*FATsize+ i, FAT);   // and the dictionary
    bdos(0xd, 0, 0);                            // Flush I/O
    delete FAT;
}

