// GENDISP.CPP - Generates the earthquake displacements to be used.
// Created by Misha Koshelev.

#include <stdio.h>       // Standard functions
#include <stdlib.h>      // Standard functions
#include <math.h>                // Math stuff
#include <time.h>                // For random numbers
#include <alloc.h>               // For allocating memory
#include <graphics.h>            // For error messages
#include <dos.h>
#include "gendisp.h"

				 // Global variables
unsigned int waveletn;           // Number of wavelets
double maxt;                     // Maximum parameter t in call to disp(t)
unsigned int fract_dim;          // The fractal dimension
double intensity;                // The Earthquake Intensity
double far *offset = NULL;       // Array of random phases (or offsets)
double far *steps = NULL;        // Steps
double far *spectrand = NULL;    // Spectrum random numbers

// Inits global variables
// Input: Intensity, Wavelet Number, Maximum time, the
// step, and the fractal dimension.
//
// Maximum time is the biggest parameter time with which disp() is called.
// Needed for a non-repetative wave.
//
// Output: None
//
void init_globals(double intens, unsigned int wavelet_n, double max_t,
     double stp, double fd)
{
   if (stp >= 2 * M_PI / max_t)
   {
      closegraph();
      printf("                          2 * PI\n");
      printf("Step must be smaller than ------\n");
      printf("                          %f    \n", max_t);
      exit(1);
   }
   waveletn = wavelet_n;            // Set values
   maxt = max_t;
   intensity = intens;
   fract_dim = fd;
   if (offset && steps && spectrand)       // If arrays not empty destroy
      done_globals();
   offset = (double far *)farcalloc(waveletn, sizeof(double));
   steps = (double far *)farcalloc(waveletn, sizeof(double));
   spectrand = (double far *)farcalloc(waveletn, sizeof(double));
   if (!offset || !steps || !spectrand)
   {
      closegraph();
      printf("Not enough memory.\n");
      exit(1);
   }
   for (int n=0; n<waveletn; n++)
   {
      offset[n] = dblrand(2 * M_PI, 4);
      spectrand[n] = dblrand(1, 4);
      steps[n] = stp * (n + 1);
   }
};

// Done globals
void done_globals(void)
{
   farfree(offset);
   farfree(steps);
   farfree(spectrand);
   offset = NULL; steps = NULL; spectrand = NULL;
};

// A doubleing point random number generator
// Input: Limit, Number of digits
//
// Output: Random number from 0 to limit
//
double dblrand(double limit, unsigned char digitn)
{
    int n;
    time_t t;

    srand((unsigned) time(&t));                // Start random number generator
    n = random((int)(limit * pow10(digitn)));  // Get a random integer
    if (n < 0)                           // If that number is negative
       n *= -1;                          // make it positive
    return (n / pow10(digitn));   // Make integer a decimal number
};

// Spectrum function.
// Input: Omega, Earthquake Intensity
//
// Output:   I
//         ----- * random
//           fd
//          w
//
double spectrum(double omega, double intensity)
{
    int idx = (int)(omega / steps[0]);

    return ((intensity / pow(omega, fract_dim)) * spectrand[idx]);
}

// Generates the displacement for a certain time.
// Input: Time
//
// Output: A displacement
// Each element is:
//        N
//        --
// d(t) = \   A(w ) sin (w t + l )
//        /      n        n     n
//        --
//        n=0
//
// where d is the displacement, w  is n * step, the function A is the
//                               n
// spectrum function, and l is an array of random offsets.
//
double disp(double time)
{
    double cur_disp = 0.0;

    if (time > maxt)
    {
       closegraph();
       printf("The parameter time as passed to function disp(time) exceeds\n");
       printf("the maximum specified.\n");
       exit(1);
    }
    for (int n=0; n<waveletn; n++)       // Loop wavelets
	 cur_disp += spectrum(steps[n], intensity) * sin((steps[n] * time) +
		    offset[n]);
    return (double)cur_disp;
}
