// TUTOR.CPP - The tutorial for the Earthquake
// Damage Prevention program. Displays the tutorial.
// Created by Misha Koshelev.

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <graphics.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#include <math.h>
#include <time.h>
#include "tutor.h"

// How many pixels we skip in between lines.
#define LINESKIP 5

// Copies one string to another starting at charachter n.
//
void strcpyn(char *dest, char *src, int n)
{
   int i;

   for (i=n; i < strlen(src); i++)
   {
      dest[i-n] = src[i];
   }
   dest[i+1-n] = NULL;
}

// Sets n charachters in a string to ch.
//
void strsetchr(char *dest, char ch, int n)
{
   int i;

   for (i=0; i<n; i++)
   {
	 dest[i] = ch;
   }
}

/* Not used in this program.  *
 *                            *
// Writes something to the screen. Wraps it when it gets to RIGHT_WRAP pixels
// or less. Writes it starting from LEFT_WRAP.
//
void gprintf(char *str)
{
   int done = 0, ci = 0;
   char cs[80], os[80];
   int estimatedsize = (getmaxx()/textwidth("a"));

   while (!done)
   {
	 strsetchr(cs, ' ', 81);
	 strsetchr(os, ' ', 81);
	 strcpyn(cs, str, ci);
	 if (strlen(cs) < estimatedsize - (getx() / textwidth("a")))
	 {
	    done = 1;
	    outtext(cs);
	 }
	 else
	 {
	    strncpy(os, cs, estimatedsize);
	    os[estimatedsize] = NULL;
	    outtext(os);
	    moverel(-textwidth(os), textheight(os) + LINESKIP);
	    ci += estimatedsize;
      }
   }
}
*/

// Moves the current position to the next line.
//
void gnewline(void)
{
   moveto(0, gety() + textheight("some text") + LINESKIP);
}

// Prints the message "Press any key to continue..." and waits for a key.
// Then deletes the message.
//
int waitforkey(void)
{
   int savec, ch, savex, savey, ret = 1;

   savec = getcolor();
   savex = getx(); savey = gety();
   setviewport(1,1,639,479,1);
   setcolor(BLUE);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtextxy(216,450,"Press any key to continue,");
   outtextxy(196,460,"or press the <ESC> key to quit.");
   while (!kbhit());
   ch = getch();
   if (ch == 27)
	 ret = 0;
   if (ch == 0)
	 getch();
   setfillstyle(SOLID_FILL, 24);
   bar(0,448,640,480);
   // Sets the viewport to inside the square
   setviewport(25,36,615,440,1);
   moveto(savex, savey);
   setcolor(savec);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   return ret;
}

// Prints a message and waits for a key to be struck. Returns 0 if that
// key is not struck, or 1 if it is.
//
int waitforkeymsg(char *msg1, char *msg2, char c)
{
   int savec, ch, savex, savey, ret = 1;

   savec = getcolor();
   savex = getx(); savey = gety();
   setviewport(1,1,639,479,1);
   setcolor(BLUE);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtextxy(320 - (textwidth(msg1)/2),450,msg1);
   outtextxy(320 - (textwidth(msg2)/2),460,msg2);
   while (!kbhit());
   ch = getch();
   if (toupper(ch) == c)
	 ret = 0;
   if (ch == 0)
	 getch();
   setfillstyle(SOLID_FILL, 24);
   bar(0,448,640,480);
   // Sets the viewport to inside the square
   setviewport(25,36,615,440,1);
   moveto(savex, savey);
   setcolor(savec);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   return ret;
}


// Displays the tutorial window with the defined topic.
//
void tutorwindow(char *topic)
{
   // Fills the borders of the windows with a grayish color
   setfillstyle(SOLID_FILL, 24);
   bar(0,0,640,32);
   bar(0,448,640,480);
   bar(0,0,20,480);
   bar(620,0,640,480);
   // Draws a black border between the interior and exterior windows
   setcolor(BLACK);
   rectangle(21,33,619,447);
   // Fills the interior of the window with a lighter grayish color
   setcolor(20);
   setfillstyle(SOLID_FILL, 20);
   bar(22,34,618,446);
   // Writes the title and the topic
   setcolor(BLUE);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtextxy(320-64, 5, "Tutorial");
   setcolor(GREEN);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtextxy(320 - (textwidth(topic)/2), 23, topic);
   // Sets the viewport to inside the square
   setviewport(25,36,615,440,1);
}

// Show the tutorial introduction.
//
void tutorintro(void)
{
   cleardevice();
   tutorwindow("Introduction");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("The goal: ");
   setcolor(BLACK);
   outtext("To compare several methods");
   gnewline();
   outtext("of earthquake damage prevention.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 goto Done;
   setcolor(GREEN);
   outtext("Method: ");
   setcolor(BLACK);
   outtext("I simulated the effect of an");
   gnewline();
   outtext("earthquake on a building with");
   gnewline();
   outtext("different earthquake damage");
   gnewline();
   outtext("prevention methods.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 goto Done;
   setcolor(GREEN);
   outtext("Programs: ");
   setcolor(BLACK);
   outtext("This project consists of");
   gnewline();
   outtext("two programs:");
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("A program that simulates an");
   gnewline();
   moverel(40,0);
   outtext("  earthquake.");
   gnewline();
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("A program that simulates how a");
   gnewline();
   moverel(40,0);
   outtext("  building moves during an");
   gnewline();
   moverel(40,0);
   outtext("  earthquake.");
   gnewline();
   if (!waitforkey())
	 goto Done;
Done:
   setviewport(1,1,639,479,1);
   cleardevice();
}

int tutoreqsimgenidea()
{
   int ret;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: General Idea");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("The main problem: ");
   setcolor(BLACK);
   outtext("earthquakes are");
   gnewline();
   setcolor(RED);
   outtext("unpredictable");
   setcolor(BLACK);
   outtext(". There is no known");
   gnewline();
   outtext("formula to describe an earthquake.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;
   setcolor(GREEN);
   outtext("Solution: ");
   setcolor(BLACK);
   outtext("To describe unpredictable");
   gnewline();
   outtext("phenomena, Professor Mandlebrot");
   gnewline();
   outtext("of Yale University invented");
   gnewline();
   setcolor(RED);
   outtext("fractals");
   setcolor(BLACK);
   outtext(".");
   gnewline();
   gnewline();
   ret = waitforkeymsg("Press any key to continue,",
		       "or press the <ESC> key to skip to waves", 27);
   if (ret == 0)
	 return 3;
   if (ret == 1)
	 return 2;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimfractals()
{
   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Fractals");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("Definiton: ");
   setcolor(BLACK);
   outtext("A fractal is a set of");
   gnewline();
   outtext("points whose dimension is a fraction");
   gnewline();
   outtext("(not an integer).");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;
   setcolor(GREEN);
   outtext("What is dimension? ");
   setcolor(BLACK);
   outtext("Dimension sort of");
   gnewline();
   outtext("describes how \`\`big\'\' a set is.");
   gnewline();
   gnewline();
   if (!waitforkey())
	  return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimpasta()
{
   int fettucini[8];

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Dimension example");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("Example: ");
   setcolor(BLACK);
   outtext("Pasta");
   gnewline();

   setcolor(YELLOW);
   line(20,300,100,50);
   moveto(20,320);
   setcolor(BLACK);
   outtext("Spaghetti");
   moveto(20,340);
   outtext("   1-D   ");
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, YELLOW);
   fettucini[0] = 220;
   fettucini[1] = 300;

   fettucini[2] = 230;
   fettucini[3] = 310;

   fettucini[4] = 310;
   fettucini[5] = 70;

   fettucini[6] = 300;
   fettucini[7] = 50;

   fillpoly(4, fettucini);
   moveto(220,320);
   setcolor(BLACK);
   outtext("Fettucini");
   moveto(220,340);
   outtext("   2-D   ");
   if (!waitforkey())
	 return 0;

   setcolor(BLACK);
   ellipse(460, 150, 0, 360, 51, 71);
   setfillstyle(SOLID_FILL, YELLOW);
   fillellipse(460, 150, 50, 70);
   setcolor(BLACK);
   line(460,80,480,220);
   moveto(405,320);
   outtext("Pasta shell");
   moveto(405,340);
   outtext("   3-D   ");
   if (!waitforkey())
	 return 0;

   moveto(320 - (textwidth("Where is more dough?")/2),370);
   setcolor(BLACK);
   outtext("Where is more dough?");
   gnewline();
   if (!waitforkey())
	 return 0;

   settextstyle(DEFAULT_FONT, HORIZ_DIR, 4);
   setcolor(GREEN);
   moveto(170, 330);
   outtext("<");
   sound(440);
   delay(100);
   nosound();
   delay(700);
   moveto(370, 330);
   outtext("<");
   sound(440);
   delay(100);
   nosound();
   delay(700);
   setcolor(BLACK);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);

   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimdimension()
{
   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: General idea of dimension");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("General idea: ");
   setcolor(BLACK);
   outtext("A set is bigger if it");
   gnewline();
   outtext("contains more elements.");
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Question: ");
   setcolor(BLACK);
   outtext("How many elements are in");
   gnewline();
   outtext("a set?");
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Solution: ");
   setcolor(BLACK);
   outtext("Ask a fast food Co. (e.g. ");
   gnewline();
   outtext("Subway) to build its restaurants so");
   gnewline();
   outtext("that wherever you are in this set,");
   gnewline();
   outtext("the nearest Subway is less than or");
   gnewline();
   outtext("equal to D yards from you.");
   gnewline();

   // Draw a stick figure
   // The head
   circle(20,250,10);
   // The body
   line(20,261,20,311);
   // The legs
   line(20,311,5,331);
   line(20,311,35,331);
   // The arms
   line(20,266,60,256);
   line(20,266,60,276);

   // Draw the Subway building
   rectangle(200,331,320,200);
   setcolor(YELLOW);
   outtextxy(210,210,"SUBWAY");
   setcolor(BLACK);

   // Draw the road between the subway building and the man
   line(35,331,200,331);
   outtextxy(120,336, "< D");
   line(120,350,136,350);

   if (!waitforkey())
	 return 0;

   moveto(0,355);
   outtext("The more restaurants needed, the");
   gnewline();
   outtext("bigger the set.");

   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimdim1d()
{
   int i;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: 1-D Case");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   outtext("Road of length L.");
   line(20,100,540,100);
   setcolor(RED);
   outtextxy(262,45,"L");
   if (!waitforkey())
	 return 0;

   // Draw the first subway
   setcolor(BLUE);
   line(20,100,60,100);
   setfillstyle(SOLID_FILL, RED);
   fillellipse(40,100,3,3);
   setcolor(GREEN);
   line(20,75,20,125);
   line(60,75,60,125);
   outtextxy(22,75,"D");
   outtextxy(42,75,"D");
   sound(440);
   delay(100);
   nosound();
   delay(200);

   setcolor(BLACK);
   outtextxy(25,105,"/\\");
   outtextxy(32,120,"|");
   outtextxy(32,135,"|");
   outtextxy(0,155, "1st Subway");
   if (!waitforkey())
	 return 0;

   // Point to the zone served by the first subway.
   setfillstyle(SOLID_FILL, 20);
   bar(0,105,640,200);
   setcolor(GREEN);
   line(20,75,20,125);
   line(60,75,60,125);

   setcolor(BLACK);
   outtextxy(10,105,"/\\");
   outtextxy(17,120,"|");
   outtextxy(17,135,"|");
   outtextxy(0,155, "Zone served by");
   outtextxy(0,175, "1st Subway");
   setcolor(GREEN);
   if (!waitforkey())
	 return 0;

   // Draw the second subway
   setfillstyle(SOLID_FILL, 20);
   bar(0,105,640,200);
   line(20,75,20,125);
   line(60,75,60,125);

   setcolor(BLUE);
   line(60,100,100,100);
   setfillstyle(SOLID_FILL, RED);
   fillellipse(80,100,3,3);
   setcolor(GREEN);
   line(100,75,100,125);
   outtextxy(62,75,"D");
   outtextxy(82,75,"D");
   sound(440);
   delay(100);
   nosound();
   delay(200);

   setcolor(BLACK);
   outtextxy(65,105,"/\\");
   outtextxy(72,120,"|");
   outtextxy(72,135,"|");
   outtextxy(0,155, "2nd Subway");
   if (!waitforkey())
	 return 0;

   // Point to the zone served by the second subway.
   setfillstyle(SOLID_FILL, 20);
   bar(0,105,640,200);
   setcolor(GREEN);
   line(20,75,20,125);
   line(60,75,60,125);
   line(100,75,100,125);

   setcolor(BLACK);
   outtextxy(50,105,"/\\");
   outtextxy(57,120,"|");
   outtextxy(57,135,"|");
   outtextxy(0,155, "Zone served by");
   outtextxy(0,175, "2nd Subway");
   setcolor(GREEN);
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,105,640,200);
   setcolor(GREEN);
   line(20,75,20,125);
   line(60,75,60,125);
   line(100,75,100,125);

   // Draw the rest of the subways
   for (i=120;i<540;i+=40)
   {
	 setcolor(BLUE);
	 line(i-20, 100, i+20,100);
	 setfillstyle(SOLID_FILL, RED);
	 fillellipse(i,100,3,3);
	 setcolor(GREEN);
	 line(i+20,75,i+20,125);
	 outtextxy(i-18,75,"D");
	 outtextxy(i+2,75,"D");
	 sound(440);
	 delay(100);
	 nosound();
	 delay(200);
   }

   if (!waitforkey())
	 return 0;

   moveto(0,150);
   setcolor(BLACK);
   outtext("Each Subway serves the area of 2D");
   gnewline();
   outtext("yards. The total length is L. So");
   gnewline();
   outtext("the total number N of Subways is:");
   gnewline();

   moveto(0,230);
   outtext("N =");
   moveto(64, 212);
   outtext("L");
   line(64,236,96,236);
   moveto(64, 254);
   outtext("2D");
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimdim2d()
{
   int i, j;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: 2-D Case");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   outtext("City with a total area of A.");
   rectangle(20,50,540,250);
   if (!waitforkey())
	 return 0;

   // Draw the Subways

   for (i=70;i<250;i+=40)
   {
	 for (j=40;j<540;j+=40)
	 {
	    setfillstyle(SOLID_FILL, RED);
	    fillellipse(j, i, 3, 3);
	    setcolor(GREEN);
	    circle(j,i,27);
	    outtextxy(j-18,i,"D");
	    sound(440);
	    delay(100);
	    nosound();
	 }
   }

   if (!waitforkey())
	 return 0;

   moveto(0,260);
   setcolor(BLACK);
   outtext("Each Subway serves the area of");
   gnewline();
   outtext("PI * D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("2");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext(" yards. The total area is A.");
   gnewline();
   outtext("So the total number N of Subways is:");
   gnewline();

   moveto(0,340);
   outtext("N =");
   moveto(64, 322);
   outtext("A");
   line(64,346,200,346);
   moveto(64, 364);
   outtext("PI * D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("2");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimdim3d()
{
   int i, j, l;
   int toppoly[10] =
		{100,100,
		300,100,
		250,150,
		50,150,
		100,100};
   int frontpoly[10] =
		{50,150,
		 250,150,
		 250,300,
		 50,300,
		 50,150};
   int sidepoly[10] =
		{250,150,
		 300,100,
		 300,250,
		 250,300,
		 250,150};
   int backline1[4] =
		{100,100,
		 100,250};
   int backline2[4] =
		{300,250,
		 100,250};
   int sideline[4] =
		{50,300,
		 100,250};

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: 3-D Case");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   outtext("Year 2015: Subways in space.");
   gnewline();
   outtext("Total volume of V.");
   drawpoly(5, toppoly);
   drawpoly(5, frontpoly);
   drawpoly(5, sidepoly);
   setlinestyle(DASHED_LINE, 0, 1);
   line(backline1[0], backline1[1], backline1[2], backline1[3]);
   line(backline2[0], backline2[1], backline2[2], backline2[3]);
   line(sideline[0], sideline[1], sideline[2], sideline[3]);
   setlinestyle(SOLID_LINE, 0, 1);
   if (!waitforkey())
	 return 0;

   // Draw the Subways

   for (l=0; l<4; l++)
   {
	 for (i=170;i<300;i+=40)
	 {
	    for (j=70;j<250;j+=40)
	    {
		  setfillstyle(SOLID_FILL, RED);
		  fillellipse(j + (l * 20), i - (l * 20), 3, 3);
		  sound(440);
		  delay(100);
		  nosound();
	    }
	 }
   }
   if (!waitforkey())
	 return 0;

   moveto(320,100);
   setcolor(BLACK);
   outtext("Each Subway");
   gnewline();
   moverel(320,0);
   outtext("serves the area of");
   gnewline();
   moverel(320,0);
   outtext("4/3 * PI * D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("3");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   gnewline();
   moverel(320,0);
   outtext("yards. The total");
   gnewline();
   moverel(320,0);
   outtext("volume is V. So");
   gnewline();
   moverel(320,0);
   outtext("the total number");
   gnewline();
   moverel(320,0);
   outtext("N of Subways is:");
   gnewline();

   moveto(0,340);
   outtext("N =");
   moveto(64, 322);
   outtext("V");
   line(64,346,300,346);
   moveto(64, 364);
   outtext("4/3 * PI * D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("3");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimpattern()
{
   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: A pattern");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);

   setcolor(BLACK);
   moverel(0,20);
   outtext("1-D Case:");
   moverel(-getx(), 0);
   moverel(300,0);
   outtext("N = ");
   moverel(0, -18);
   outtext("c");
   moverel(-16, 20);
   outtext("D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("1");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moverel(-24, -2);
   line(getx(),gety(),getx()+20,gety());

   setcolor(BLACK);
   moverel(-getx(),50);
   outtext("2-D Case:");
   moverel(-getx(), 0);
   moverel(300,0);
   outtext("N = ");
   moverel(0, -18);
   outtext("c");
   moverel(-16, 20);
   outtext("D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("2");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moverel(-24, -2);
   line(getx(),gety(),getx()+20,gety());

   setcolor(BLACK);
   moverel(-getx(),50);
   outtext("3-D Case:");
   moverel(-getx(), 0);
   moverel(300,0);
   outtext("N = ");
   moverel(0, -18);
   outtext("c");
   moverel(-16, 20);
   outtext("D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("3");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moverel(-24, -2);
   line(getx(),gety(),getx()+20,gety());
   if (!waitforkey())
	 return 0;

   // Highlight the matching numbers
   setcolor(GREEN);
   moveto(0,20);
   outtext("1");
   moverel(-getx(), 0);
   moverel(364, 0);
   moverel(16,2);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("1");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moveto(0,70);
   outtext("2");
   moverel(-getx(), 0);
   moverel(364, 0);
   moverel(16,2);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("2");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moveto(0,120);
   outtext("3");
   moverel(-getx(), 0);
   moverel(364, 0);
   moverel(16,2);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("3");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   if (!waitforkey())
	 return 0;

   moverel(-getx(), 50);
   setcolor(BLACK);
   moverel(320 - (textwidth("GENERAL DEFINITION")/2), 0);
   outtext("GENERAL DEFINITON");
   gnewline();
   gnewline();
   outtext("If N =    , a is called dimension.");
   moveto(128,gety());
   moverel(0,-18);
   outtext("c");
   moverel(-16,+20);
   line(getx(), gety()-2, getx() + 20, gety()-2);
   outtext("D");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("a");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moverel(-32,-2);
   gnewline();
   gnewline();
   setcolor(BLACK);
   outtext("A ");
   setcolor(RED);
   outtext("fractal");
   setcolor(BLACK);
   outtext(" is a set of");
   gnewline();
   outtext("points whose dimension is a fraction");
   gnewline();
   outtext("(not an integer).");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimwgenidea()
{
   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: General Idea (Continued)");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("Summary: ");
   setcolor(BLACK);
   outtext("An earthquake can be");
   gnewline();
   outtext("described by a fractal.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Problem: ");
   setcolor(BLACK);
   outtext("How to simulate a fractal?");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Method: ");
   setcolor(BLACK);
   outtext("We represent a fractal as a");
   gnewline();
   outtext("sum of monochromatic waves.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Question: ");
   setcolor(BLACK);
   outtext("What is a wave?");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimw1(void)
{
   float step;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Waves");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moveto(0,30);
   setcolor(BLACK);
   outtext("A ");
   setcolor(RED);
   outtext("wave");
   setcolor(BLACK);
   outtext(" is a real-life pattern that");
   gnewline();
   outtext("repeats itself again and again.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   step = M_PI / 90;
   line(20,200,20+(4 * M_PI) / step,200);
   for (float a=0.0; a<4 * M_PI; a+=step)
   {
	  putpixel(20+a/step,200+(sin(a)*100),BLUE);
   }

   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,0,640,100);
   setcolor(BLACK);
   outtextxy(50,10, "How to characterize a wave?");
   moveto(0, 330);
   setcolor(BLACK);
   outtext("1) ");
   setcolor(GREEN);
   outtext("Amplitude A: ");
   setcolor(BLACK);
   outtext("distance from the");
   gnewline();
   outtext("middle to the crest.");
   setcolor(RED);
   outtextxy(0,142,"A");
   line(18,100,18,200);
   line(18,100,22,100);
   line(18,200,22,200);
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,310,640,480);
   moveto(0, 330);
   setcolor(BLACK);
   outtext("2) ");
   setcolor(GREEN);
   outtext("Frequency F: ");
   setcolor(BLACK);
   outtext("number of");
   gnewline();
   outtext("repetative segments in a second.");
   setcolor(RED);
   line(20,300,20,100);
   line(20+(2 * M_PI)/step,300,20+(2 * M_PI)/step,100);
   line(20,300,20+(2 * M_PI)/step,300);
   outtextxy(20,310,"1 sec., F = 1");
   line(18,100,18,200);
   line(18,100,22,100);
   line(18,200,22,200);
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,310,640,480);
   moveto(0, 330);
   setcolor(BLACK);
   outtext("3) ");
   setcolor(GREEN);
   outtext("Phase P: ");
   setcolor(BLACK);
   outtext("how far the initial");
   gnewline();
   outtext("position of the wave is from the");
   gnewline();
   outtext("middle");
   setcolor(RED);
   line(20,300,20,100);
   outtextxy(0,192,"P");
   outtextxy(0,210,"=");
   outtextxy(0,228,"0");
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,0,640,100);
   setcolor(BLACK);
   outtextxy(50,10, "Monochromatic wave:");
   moveto(0, 330);
   setfillstyle(SOLID_FILL, 20);
   bar(0,310,640,480);
   moveto(0, 330);
   setcolor(BLACK);
   moveto(320-(textwidth("x(t) = A sin(wt + p)")/2), gety());
   outtext("x(t) = A sin(wt + p)");
   gnewline();
   moveto(320-(textwidth("w = f * 2PI")/2), gety());
   outtext("w = f * 2PI");
   gnewline();
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimwlight(void)
{
   int prism[8] =
	    {320,100,
		400,300,
		240,300,
		320,100};
   int i;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Light Waves");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   outtext("Light from any source can be");
   gnewline();
   outtext("separated into several basic colors,");
   gnewline();
   outtext("as in a prism.");
   gnewline();

   drawpoly(4, prism);
   for (i=50;i<281;i++)
   {
	 putpixel(i,200,YELLOW);
	 delay(20);
   }
   for (i=280;i<500;i++)
   {
	 putpixel(i,200,RED);
	 putpixel(i,200+((i-280)/5),57 /*ORANGE*/);
	 putpixel(i,200+((i-280)/4),GREEN);
	 putpixel(i,200+((i-280)/3),BLUE);
	 putpixel(i,200+((i-280)/2),173 /*VIOLET*/);
	 delay(40);
   }

   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimwsound(void)
{
   int piano[30] =
	    {100,100,
		120,100,
		120,300,
		140,300,
		140,100,
		160,100,
		160,300,
		180,300,
		180,100,
		200,100,
		200,300,
		220,300,
		220,100,
		240,100,
		240,300};
   int bk1[8] =
	    {115,100,
		125,100,
		125,200,
		115,200};
   int bk2[8] =
	    {135,100,
		145,100,
		145,200,
		135,200};
   int bk3[8] =
	    {155,100,
		165,100,
		165,200,
		155,200};
   int bk4[8] =
	    {195,100,
		205,100,
		205,200,
		195,200};
   int bk5[8] =
	    {215,100,
		225,100,
		225,200,
		215,200};
   int i;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Sound Waves");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   outtext("Any sound can be simulated by a");
   gnewline();
   outtext("piano.");
   gnewline();
   gnewline();

   drawpoly(15, piano);
   rectangle(100,100,240,300);
   setfillstyle(SOLID_FILL, BLACK);
   fillpoly(4, bk1);
   fillpoly(4, bk2);
   fillpoly(4, bk3);
   fillpoly(4, bk4);
   fillpoly(4, bk5);
   setcolor(RED);
   outtextxy(120,320,"YAMAHA");
   if (!waitforkey())
	 return 0;

   setcolor(BLACK);
   moveto(0,340);
   outtext("This means that a sound can be");
   gnewline();
   outtext("played as a chord.");
   if (!waitforkey())
	 return 0;

   for (i=0;i<1000;i++)
   {
	  sound(440);
	  delay(1);
	  nosound();
	  sound(880);
	  delay(1);
	  nosound();
   }
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutoreqsimwwave(void)
{
   int x,y;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Waves");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   outtext("Every wave can be represented as");
   gnewline();
   outtext("a sum of monochromatic waves:");
   gnewline();
   gnewline();
   gnewline();
   gnewline();
   outtext("x(t) = ");
   // Draw sigma
   x=getx(); y = gety();
   line(x-20,y-20,x,y);
   line(x-20,y+20,x,y);
   line(x,y-20,x-20,y-20);
   line(x,y+20,x-20,y+20);
   // Write text on it
   outtextxy(x-18,y-38,"N");
   outtextxy(x-36,y+22,"n=0");
   // Continue with the formula
   outtext("A");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   moverel(0,10);
   outtext("n");
   moverel(0,-10);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext("sin(w");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   moverel(0,10);
   outtext("n");
   moverel(0,-10);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext("t");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   moverel(0,10);
   outtext("n");
   moverel(0,-10);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext("+p");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   moverel(0,10);
   outtext("n");
   moverel(0,-10);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext(")");
   if (!waitforkey())
	 return 0;

   gnewline();
   gnewline();
   gnewline();
   outtext("To get a ");
   setcolor(RED);
   outtext("fractal");
   setcolor(BLACK);
   outtext(" you must take:");
   gnewline();
   gnewline();
   outtext("A");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   moverel(0,10);
   outtext("n");
   moverel(0,-10);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext(" = I / (w");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   moverel(0,10);
   outtext("n");
   moverel(0,-10);
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   outtext(")");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
   outtext("a");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   if (!waitforkey())
	 return 0;

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

// Global variables for the wave example.
float *eqsimoffset = NULL;            // Array of random phases (or eqsimoffsets)
float *eqsimsteps = NULL;             // Steps
float *eqsimspectrand = NULL;            // Spectrum random numbers

// Destruction function for the wave example.
void tutoreqsimwdoneglobals(void)
{
   if (eqsimoffset)
	 free(eqsimoffset);
   if (eqsimsteps)
	 free(eqsimsteps);
   if (eqsimspectrand)
	 free(eqsimspectrand);
}

// Double random function.
float tutoreqsimwdblrand(float limit, unsigned char digitn)
{
    int n;
    time_t t;

    srand((unsigned) time(&t));                // Start random number generator
    n = random((int)(limit * pow10(digitn)));  // Get a random integer
    if (n < 0)                           // If that number is negative
       n *= -1;                          // make it positive
    return (n / pow10(digitn));   // Make integer a decimal number
};

// Initialization function for the wave example.
void tutoreqsimwinitglobals(int waveletn)
{
   if (eqsimoffset && eqsimsteps && eqsimspectrand)        // If arrays not empty destroy
	 tutoreqsimwdoneglobals();
   eqsimoffset = (float *)malloc(sizeof(float) * waveletn);
   eqsimsteps = (float *)malloc(sizeof(float) * waveletn);
   eqsimspectrand = (float *)malloc(sizeof(float) * waveletn);
   if (!eqsimoffset || !eqsimsteps || !eqsimspectrand)
   {
      closegraph();
      printf("Not enough memory.\n");
      exit(1);
   }
   for (int n=0; n<waveletn; n++)
   {
	 eqsimoffset[n] = tutoreqsimwdblrand(2 * M_PI, 4);
	 eqsimspectrand[n] = tutoreqsimwdblrand(1, 4);
	 eqsimsteps[n] = 0.2 * (n + 1);
   }
};

// Spectrum function. Modified slightly.
float tutoreqsimwspectr(float omega, float intensity)
{
    int idx = (int)(omega / eqsimsteps[0]);

    return ((intensity / pow(omega, 0.35)) * eqsimspectrand[idx]);
}

// Earthquake displacement function. Modified slightly.
// Generates the displacement for a certain time.
float tutoreqsimwdisp(float time, int waveletn)
{
    float cur_disp = 0.0;

    if (time > 30.0)
    {
       closegraph();
       printf("The parameter time as passed to function disp(time) exceeds\n");
       printf("the maximum specified.\n");
       exit(1);
    }
    for (int n=0; n<waveletn; n++)       // Loop wavelets
	 cur_disp += tutoreqsimwspectr(eqsimsteps[n], 25) * sin((eqsimsteps[n] * time) +
		    eqsimoffset[n]);
    return (float)cur_disp;
}

int tutoreqsimwexample(void)
{
   int y = 310, i;
   float j, ret;
   char *tmp;

   cleardevice();
   tutorwindow("EARTHQUAKE SIMULATION: Fractal wave example");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLACK);
   tutoreqsimwinitglobals(5);
   setfillstyle(SOLID_FILL, BLACK);
   bar(0,0,300,300);

   moveto(320,10);
   setcolor(RED);
   outtext("Example: ");
   setcolor(BLACK);
   outtext("how to");
   gnewline();
   moverel(320,0);
   outtext("generate a");
   gnewline();
   moverel(320,0);
   outtext("fractal.");

   for (i=1;i<6;i++)
   {
	  setcolor(BLACK);
	  sprintf(tmp, "%d monochromatic wave(s): ", i);
	  moveto(0,y);
	  outtext(tmp);
	  setcolor(i);
	  moverel(10,0);
	  linerel(10,0);
	  y+=18;
	  for (j=0.0; j<30.0; j+=0.1)
	  {
		ret = tutoreqsimwdisp(j, i);
		if (ret * 10 <= 150 && ret * 10 >= -150)
		   putpixel(j*10, 150 + (ret * 10), i);
	  }
	  if (!waitforkey())
		return 0;
   }

   tutoreqsimwdoneglobals();

   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

void tutoreqsim(void)
{
   int ret;

   ret = tutoreqsimgenidea();
   setviewport(1,1,639,479,1);
   cleardevice();
   if (ret == 0)
	 goto Done;
   if (ret == 3)
	 goto Waves;

   if (!tutoreqsimfractals())
	 goto Done;
   if (!tutoreqsimpasta())
	 goto Done;
   if (!tutoreqsimdimension())
	 goto Done;
   if (!tutoreqsimdim1d())
	 goto Done;
   if (!tutoreqsimdim2d())
	 goto Done;
   if (!tutoreqsimdim3d())
	 goto Done;
   if (!tutoreqsimpattern())
	 goto Done;
Waves:
   if (!tutoreqsimwgenidea())
	 goto Done;
   if (!tutoreqsimw1())
	 goto Done;
   if (!tutoreqsimwlight())
	 goto Done;
   if (!tutoreqsimwsound())
	 goto Done;
   if (!tutoreqsimwwave())
	 goto Done;
   if (!tutoreqsimwexample())
	 goto Done;
Done:
   setviewport(1,1,639,479,1);
   cleardevice();
}

int tutorbmsimgenidea(void)
{
   cleardevice();
   tutorwindow("BUILDING MOVEMENT SIMULATION: General idea");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("General Idea: ");
   setcolor(BLACK);
   outtext("Netwon's law f(t) = Ma(T)");
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("f(t) is a force.");
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("M is a mass.");
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("a(t) is acceleration.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Problems: ");
   setcolor(BLACK);
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("How force depends on time?");
   gnewline();
   moverel(40,0);
   setcolor(RED);
   outtext("- ");
   setcolor(BLACK);
   outtext("How to simulate the motion");
   gnewline();
   moverel(40,0);
   outtext("  if we know the acceleration?");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

Done:
   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutorbmsimforce(void)
{
   int i;

   cleardevice();
   tutorwindow("BUILDING MOVEMENT SIMULATION: Components of the force");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(GREEN);
   outtext("Elastic force: ");
   setcolor(BLACK);
   outtext("f_elastic(t) = -kx(t)");
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(YELLOW);
   for (i=100;i<400;i++)
   {
	 line(10,40,i,40);
	 delay(10);
   }

   delay(100);

   for (i=400;i>100;i--)
   {
	 setcolor(20);
	 line(10,40,i+1,40);
	 setcolor(YELLOW);
	 line(10,40,i,40);
	 delay(10);
   }
   if (!waitforkey())
	 return 0;

   moveto(0,50);
   setcolor(GREEN);
   outtext("Friction force: ");
   setcolor(BLACK);
   gnewline();
   outtext("f_friction(t) = -nv(t)");
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   line(22,141,320,141);

   setcolor(BLACK);
   rectangle(22,100,122,140);

   for (i=23;i<=101;i++)
   {
	  setcolor(20);
	  rectangle(i-1,100,i+99,140);
	  setcolor(BLACK);
	  rectangle(i,100,i+100,140);
	  delay(20);
   }
   for (i=102;i<=150;i++)
   {
	  setcolor(20);
	  rectangle(i-1,100,i+99,140);
	  setcolor(BLACK);
	  rectangle(i,100,i+100,140);
	  delay(50);
   }
   for (i=151;i<=200;i++)
   {
	  setcolor(20);
	  rectangle(i-1,100,i+99,140);
	  setcolor(BLACK);
	  rectangle(i,100,i+100,140);
	  delay(100);
   }
   for (i=201;i<=220;i++)
   {
	  setcolor(20);
	  rectangle(i-1,100,i+99,140);
	  setcolor(BLACK);
	  rectangle(i,100,i+100,140);
	  delay(200);
   }

   moveto(0,155);
   setcolor(GREEN);
   outtext("Earthquake force.");
   if (!waitforkey())
	 return 0;

Done:
   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutorbmsimcntrlforce(void)
{
   int i; int bpoly[10] =
			 {70,220,
			  120,220,
			  120,180,
			  70,180,
			  70,220};
   int beampoly[10] =
			 {70,220,
			  80,220,
			  115,185,
			  110,185,
			  70,220};

   cleardevice();
   tutorwindow("BUILDING MOVEMENT SIMULATION: Components of the force");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   moverel(150,0);
   setcolor(BLUE);
   outtext("CONTROL FORCE");
   gnewline();
   gnewline();
   gnewline();
   setcolor(GREEN);
   outtext("No control: ");
   setcolor(BLACK);
   outtext("f_control(t) = 0");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("Active control: ");
   setcolor(BLACK);
   outtext("pulls the building");
   gnewline();
   outtext("in the direction opposite to");
   gnewline();
   outtext("the earthquake.");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   drawpoly(5, bpoly);

   setcolor(RED);
   moveto(2,180);
   outtext("-");
   delay(200);
   outtext("-");
   delay(200);
   outtext("-");
   moverel(0,-14);
   outtext("\\");
   moverel(-16,24);
   outtext("/");
   moverel(0, -2);

   for (i=0;i<20;i++)
   {
	  setcolor(20);
	  drawpoly(5,bpoly);
	  setcolor(GREEN);
	  bpoly[4]++;
	  bpoly[6]++;
	  drawpoly(5,bpoly);
	  delay(50);
   }

   setcolor(GREEN);
   moveto(130,196);
   moverel(0,-24);
   outtext("/");
   moverel(-16,14);
   outtext("\\");
   moverel(0, -2);
   outtext("-");
   delay(200);
   outtext("-");
   delay(200);
   outtext("-");

   for (i=0;i<20;i++)
   {
	  setcolor(20);
	  drawpoly(5,bpoly);
	  setcolor(GREEN);
	  bpoly[4]--;
	  bpoly[6]--;
	  drawpoly(5,bpoly);
	  delay(50);
   }
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   moveto(0,230);
   gnewline();
   gnewline();
   outtext("Problem: ");
   setcolor(BLACK);
   outtext("We cannot react");
   gnewline();
   outtext("immediately so there is a delay T.");
   gnewline();
   outtext("f_control(t) = -hx(t-T)");
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,18,640,479);

   moveto(0,30);
   setcolor(GREEN);
   outtext("Hybrid control: ");
   setcolor(BLACK);
   outtext("Use an extra beam");
   gnewline();
   outtext("that is normally unattached. If the");
   gnewline();
   outtext("building moves too far, attach the");
   gnewline();
   outtext("beam and thus, get an extra elastic");
   gnewline();
   outtext("force.");
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   drawpoly(5, bpoly);
   setcolor(BLACK);
   drawpoly(5, beampoly);

   setcolor(RED);
   moveto(2,180);
   outtext("-");
   delay(200);
   outtext("-");
   delay(200);
   outtext("-");
   moverel(0,-14);
   outtext("\\");
   moverel(-16,24);
   outtext("/");
   moverel(0, -2);

   for (i=0;i<20;i++)
   {
	  setcolor(20);
	  drawpoly(5,bpoly);
	  drawpoly(5,beampoly);
	  setcolor(GREEN);
	  bpoly[4]++;
	  bpoly[6]++;
	  beampoly[4]++;
	  beampoly[6]++;
	  drawpoly(5,bpoly);
	  setcolor(BLACK);
	  drawpoly(5,beampoly);
	  delay(70);
   }

   setcolor(20);
   drawpoly(5,bpoly);
   drawpoly(5,beampoly);
   beampoly[4] += 5;
   beampoly[6] += 5;
   beampoly[5] -= 5;
   beampoly[7] -= 5;

   for (i=0;i<20;i++)
   {
	  setcolor(20);
	  drawpoly(5,bpoly);
	  drawpoly(5,beampoly);
	  setcolor(GREEN);
	  bpoly[4]--;
	  bpoly[6]--;
	  beampoly[4]--;
	  beampoly[6]--;
	  drawpoly(5,bpoly);
	  setcolor(BLACK);
	  drawpoly(5,beampoly);
	  delay(70);
   }
   if (!waitforkey())
	 return 0;

Done:
   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutorbmsimmotionsim(void)
{
   int i;

   cleardevice();
   tutorwindow("BUILDING MOVEMENT SIMULATION: How to simulate motion?");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(RED);
   outtext("Velocity ");
   setcolor(BLACK);
   delay(1000);
   line(1,300,400,300);
   line(150,295,150,305);
   line(350,295,350,305);
   outtextxy(138,310,"x(t)");
   sound(440);
   delay(200);
   nosound();
   outtextxy(320,310,"x(t+dt)");
   sound(440);
   delay(200);
   nosound();

   for (i=150; i<350; i++)
   {
	  if (i<155)
	  {
		setcolor(BLACK);
		line(150,295,150,305);
	  }
	  setcolor(20);
	  setfillstyle(SOLID_FILL, 20);
	  fillellipse(i-1,300,5,5);
	  setcolor(BLACK);
	  line(i-6,300,i+4,300);
	  setcolor(RED);
	  setfillstyle(SOLID_FILL, RED);
	  fillellipse(i,300,5,5);
	  delay(5);
   }

   delay(100);

   moveto(146,0);
   setcolor(BLACK);
   outtext("is distance over time:");
   if (!waitforkey())
	 return 0;

   moveto(194, 20);
   outtext("x(t+dt)-x(t)");
   sound(440);
   delay(200);
   nosound();
   moveto((17 * 16) + 146,20);
   outtext("dt");
   sound(440);
   delay(200);
   nosound();

   delay(1000);

   gnewline();
   gnewline();
   gnewline();
   outtext("v(t) = (x(t+dt) - x(t))/dt");
   if (!waitforkey())
	 return 0;

   setfillstyle(SOLID_FILL, 20);
   bar(0,0,600,400);

   moveto(1,1);
   setcolor(RED);
   outtext("Acceleration ");
   setcolor(BLACK);
   outtext("is velocity over time.");
   gnewline();
   gnewline();
   gnewline();
   outtext("a(t) = (v(t+dt) - v(t))/dt");
   if (!waitforkey())
	 return 0;

   gnewline();
   gnewline();
   outtext("So: a(t+dt) = a(t) * dt + v(t)");
   if (!waitforkey())
	 return 0;

Done:
   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutorbmsimgetcoordvel(void)
{
   int i;

   cleardevice();
   tutorwindow("BUILDING MOVEMENT SIMULATION");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLUE);
   outtext("How to get the coordinates and the");
   gnewline();
   outtext("velocity at the next moment of time?");
   gnewline();
   gnewline();
   gnewline();
   setcolor(GREEN);
   outtext("Known: ");
   setcolor(BLACK);
   outtext("x(t), v(t).");
   gnewline();
   gnewline();
   outtext("x(t+dt) = ");
   setcolor(GREEN);
   outtext("?");
   gnewline();
   setcolor(BLACK);
   outtext("v(t+dt) = ");
   setcolor(GREEN);
   outtext("?");
   setcolor(BLACK);
   if (!waitforkey())
	 return 0;

   moveto(0,gety()-16-LINESKIP);
   setfillstyle(SOLID_FILL, 20);
   bar(getx(),gety(),640,400);
   outtext("x(t+dt) = v(t) * dt + x(t)");
   gnewline();
   outtext("v(t+dt) = a(t) * dt + v(t)");
   gnewline();
   gnewline();
   outtext("where: a(t) = f(t) / M");
   if (!waitforkey())
	 return 0;

Done:
   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}

int tutorbmsimenergy(void)
{
   int i;

   cleardevice();
   tutorwindow("BUILDING MOVEMENT SIMULATION");
   settextstyle(DEFAULT_FONT, HORIZ_DIR, 2);
   setcolor(BLUE);
   outtext("How to computer the energy used by");
   gnewline();
   outtext("active control?");
   gnewline();
   gnewline();
   gnewline();
   setcolor(BLACK);
   outtext("The bigger the force we apply, the");
   gnewline();
   outtext("more energy we use:");
   gnewline();
   outtext("e(t) is proportional to f(t)");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   outtext("The more we move the building, the");
   gnewline();
   outtext("more energy we use:");
   gnewline();
   outtext("e(t) is proportional to x(t)");
   gnewline();
   gnewline();
   if (!waitforkey())
	 return 0;

   setcolor(GREEN);
   outtext("So: ");
   setcolor(BLACK);
   outtext("e(t) = f(t) * x(t),");
   gnewline();
   outtext("   E = e(0) + e(1) + ... + e(T)");
   if (!waitforkey())
	 return 0;

Done:
   setviewport(1,1,639,479,1);
   cleardevice();
   return 1;
}


void tutorbmsim(void)
{
   if (!tutorbmsimgenidea())
	 goto Done;
   if (!tutorbmsimforce())
	 goto Done;
   if (!tutorbmsimcntrlforce())
	 goto Done;
   if (!tutorbmsimmotionsim())
	 goto Done;
   if (!tutorbmsimgetcoordvel())
	 goto Done;
   if (!tutorbmsimenergy())
	 goto Done;
Done:
   setviewport(1,1,639,479,1);
   cleardevice();
}
