// envmap.cpp
// Environment mapping

//// last modified March 20, 1996

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>

#include "fixed32.h"


int ledge_x[200];  // destination x coordinate for left edge
int ledge_sx[200]; // source x coordinate for left edge
int ledge_sy[200]; // source y coordinate for left edge

int redge_x[200]; // destination x coordinate for right edge
int redge_sx[200]; // source x coordinate for right edge
int redge_sy[200]; // source y coordinate for right edge

int ytable[200]; // so u can do ytable[y]+x (fast) instead of y*320+x (slow)

// structure for a 4 sided texture mapped polygon
typedef struct
{
	int sx1,sy1,sx2,sy2,sx3,sy3,sx4,sy4; // source coordinates (c-clockwise)
	int dx1,dy1,dx2,dy2,dx3,dy3,dx4,dy4; // distination coordinates (c-clockwise)
	unsigned char *bitmap; // points to source bitmap
	int height; // height of source bitmap
	int width; // width of source bitmap
} TPolygonTYPE;

TPolygonTYPE poly;

char *screen=(char *)0x0a0000; // mode 13h screen
char screen_buff[64000]; // 320x200 system ram screen buffer
char screen_bg[64000]; // 320x200 system ram background buffer

unsigned char bitmap[64000]; // this is texure mapped
unsigned char palette[768];
unsigned char temp_palette[768];

unsigned char dummy_buffer[64000]; // load files into here


//------------- 3d stuff -----------------------


#define MAXDEGREES 1024
#define EYE_DISTANCE Int2Fixed(256)
#define MAXVERTICES 700
#define MAXPOLYGONS 700
#define MAXPOLYVERT 20

Fixedpoint cosine[MAXDEGREES];  // cosine lookup table
Fixedpoint sine[MAXDEGREES];    // sine lookup table

int xangle,yangle,zangle; // angles of rotation
int xpos,ypos,zpos; // position of object in 3d

int eboxtop; // erase box top coordinate (current frame)
int eboxbottom; // erase box bottom coordinate (current frame)
int eboxleft; // erase box left coordinate (current frame)
int eboxright; // erase box right coordinate (current frame)
int eboxtop1; // erase box top coordinate (1 frame ago)
int eboxbottom1; // erase box bottom coordinate (1 frame ago)
int eboxleft1; // erase box left coordinate (1 frame ago)
int eboxright1; // erase box right coordinate (1 frame ago)

int ClipTop=1;
int ClipBottom=198;
int ClipLeft=1;
int ClipRight=318;


// structure of one vertex
typedef struct
{
	Fixedpoint ox,oy,oz; // vertex original coordinates
	Fixedpoint wx,wy,wz; // vertex working  coordinates
	Fixedpoint Nox,Noy,Noz; // vertex normal original coordinates
	Fixedpoint Nwx,Nwy,Nwz; // vertex normal working coordinates
	int sx,sy,sz; // vertex screen coordinates
} VertexTYPE;

// structure of one polygon
typedef struct
{
	int NumOfVertices; // number of vertices that make up the polygon
	int Vertex[MAXPOLYVERT]; // vertex indices in counter clockwise order
	int zcenter; // for sorting
	Fixedpoint Nox,Noy,Noz;
} PolygonTYPE;

// object structure
typedef struct
{
	Fixedpoint Ox,Oy,Oz; // coordinates of object's origin
	int Ax,Ay,Az; // object's rotation angles
	int NumOfVertices; // number of vertices in object
	VertexTYPE Vertex[MAXVERTICES]; // all vertices in object
	int NumOfPolygons; // number of polygons in object
	PolygonTYPE Polygon[MAXPOLYGONS]; // all polygons in object
} ObjectTYPE;

ObjectTYPE Object; // one object

// structure of light source
typedef struct
{
	Fixedpoint x,y,z; // coodinates of light source
	Fixedpoint wx,wy,wz; // working (intermediate) coordinates
	int ax,ay,az; // rotation angles
} LightSourceTYPE;

LightSourceTYPE LightSource; // one light source

int NumOfSortedPolygons; // number of sorted visible polygons
int PolygonOrderList[MAXPOLYGONS]; // list of polygon indices for qsorting

//---------------------- function prototypes ------------------------
void TextureMap(TPolygonTYPE *poly);
void TScanEdge(int xd1,int yd1,int xd2,int yd2,
							 int xs1,int ys1,int xs2,int ys2);
int main(void);
void WaitVRT(void);
void SetUpScreen(void);
void InitYTable(void);
void InitTrigTables(void);
void LoadObject(void);
void InitNormals(void);
void RotatePoints(void);
void RotateNormals(void);
void DrawObject(void);
void GetPalette(unsigned char *palettebuffer);
void SetPalette(unsigned char *palettebuffer);
void FadeInPalette(unsigned char *palettebuffer,int speed);
void FadeOutPalette(int speed);
void MakePolygonList(void);
void SortPolygonList(void);
int ComparePolygons(const void *a, const void *b);
void Demo(void);
void LoadGraphics();

//-------------------------------------------------------------------

void SetMode13h(void);
#pragma aux SetMode13h = \
		"mov eax,013h " \
		"int 10h" \
		modify [eax];


void SetMode03h(void);
#pragma aux SetMode03h = \
		"mov eax,003h " \
		"int 10h" \
		modify [eax];



void WaitVRT(void)
{
	while(inp(0x3da) & 0x08);
	while(!(inp(0x3da) & 0x08));
}


void InitYTable(void)
{
	int i;
	for(i=0;i<200;i++)
	{
		ytable[i]=i*320;
	}
}


void InitTrigTables(void)
{
	int i;
	for(i=0; i<MAXDEGREES; i++)
	{
		cosine[i]=Float2Fixed(cos((float)i*360/MAXDEGREES * 3.14159265 / 180.0));
		sine[i]  =Float2Fixed(sin((float)i*360/MAXDEGREES * 3.14159265 / 180.0));
	}
}


void SetUpScreen(void)
{
	// set mode 13h screen
	SetMode13h();

	int i;

	// clear screen memory
	for(i=0;i<64000;i++)
	{
		screen[i]=0;
	}
}

//----------------------------------------------------------------------


// initialize normals 


void InitNormals(void)
{
	double x1,x2,x3,y1,y2,y3,z1,z2,z3,xlen,ylen,zlen,length;
	int i,j,k;
	int count;

	// first calculate the polygon normals
	for(i=0;i < Object.NumOfPolygons; i++)
	{
		x1=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[0]].ox);
		x2=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[1]].ox);
		x3=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[2]].ox);
		y1=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[0]].oy);
		y2=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[1]].oy);
		y3=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[2]].oy);
		z1=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[0]].oz);
		z2=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[1]].oz);
		z3=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[2]].oz);

		// calculate perpendicular via the cross product of 1st vertex & normal
		xlen = (y1-y2)*(z1-z3) - (z1-z2)*(y1-y3);
		ylen = (z1-z2)*(x1-x3) - (x1-x2)*(z1-z3);
		zlen = (x1-x2)*(y1-y3) - (y1-y2)*(x1-x3);

		// calculate the length of the normal
		length = sqrt(xlen*xlen + ylen*ylen + zlen*zlen);

		// scale it to a unit normal
		Object.Polygon[i].Nox = Float2Fixed(xlen / length);
		Object.Polygon[i].Noy = Float2Fixed(ylen / length);
		Object.Polygon[i].Noz = Float2Fixed(zlen / length);
	}

	// now figure out the vertex normals

	//This will calculate the normals of the vertices for environment mapping
	for(i=0;i < Object.NumOfVertices; i++)
	{
		count=0;
		xlen=0;
		ylen=0;
		zlen=0;

		// find all the polygons that this vertex belongs to
		for(j=0;j<Object.NumOfPolygons;j++)
		{
			for(k=0;k<Object.Polygon[j].NumOfVertices;k++)
			{
				if(Object.Polygon[j].Vertex[k]==i)
				{
					// vertex belongs to this polygon so add it on
					xlen+=Fixed2Float(Object.Polygon[j].Nox);
					ylen+=Fixed2Float(Object.Polygon[j].Noy);
					zlen+=Fixed2Float(Object.Polygon[j].Noz);
					count++;
				}
			}
		}

		// calculate the average normal
		if(count>0)
		{
			xlen=xlen/count;
			ylen=ylen/count;
			zlen=zlen/count;
		}

		// calculate the length of the normal
		length = sqrt(xlen*xlen + ylen*ylen + zlen*zlen);

		// scale it to a unit normal
		Object.Vertex[i].Nox = Float2Fixed(xlen / length);
		Object.Vertex[i].Noy = Float2Fixed(ylen / length);
		Object.Vertex[i].Noz = Float2Fixed(zlen / length);
	}
}


// rotate normals 


void RotateNormals(void)
{
	int i;
	Fixedpoint nx,ny,nz,cosxangle,sinxangle,cosyangle,sinyangle,coszangle,sinzangle;
	VertexTYPE *vert; // pointer to a vertex structure

	// get sine and cosine angles to save time from table lookup in inner loop
	sinxangle=sine[xangle];
	cosxangle=cosine[xangle];
	sinyangle=sine[yangle];
	cosyangle=cosine[yangle];
	sinzangle=sine[zangle];
	coszangle=cosine[zangle];

	for(i=0;i<Object.NumOfVertices;i++)
	{
		vert=&Object.Vertex[i];

		// rotate around the x-axis
		vert->Nwz=FixedMul(vert->Noy , cosxangle) - FixedMul(vert->Noz , sinxangle);
		vert->Nwy=FixedMul(vert->Noy , sinxangle) + FixedMul(vert->Noz , cosxangle);
		vert->Nwx=vert->Nox;

		// rotate around the y-axis
		nx=FixedMul(vert->Nwx , cosyangle) - FixedMul(vert->Nwz , sinyangle);
		nz=FixedMul(vert->Nwx , sinyangle) + FixedMul(vert->Nwz , cosyangle);
		vert->Nwx=nx;
		vert->Nwz=nz;

		// rotate around the z-axis
		nx=FixedMul(vert->Nwx , coszangle) - FixedMul(vert->Nwy , sinzangle);
		ny=FixedMul(vert->Nwx , sinzangle) + FixedMul(vert->Nwy , coszangle);

		// reverse the direction of the normals for lightsource shading
		vert->Nwx=nx;
		vert->Nwy=ny;
		vert->Nwz=nz;
	}
}


//----------------------------------------------------------------------

void RotatePoints(void)
{
	int i;
	Fixedpoint nx,ny,nz,cosxangle,sinxangle,cosyangle,sinyangle,coszangle,sinzangle;

	// get the sine and cosine angles to save time from table lookup
	sinxangle=sine[xangle];
	cosxangle=cosine[xangle];
	sinyangle=sine[yangle];
	cosyangle=cosine[yangle];
	sinzangle=sine[zangle];
	coszangle=cosine[zangle];


	// initialize the erase-box info to the extreme values
	eboxtop=200;
	eboxleft=320;
	eboxright=0;
	eboxbottom=0;


	// loop through all vertices in object
	for(i=0;i<Object.NumOfVertices;i++)
	{
		// make a pointer to the current vertex
		VertexTYPE *vert=&Object.Vertex[i];

		// rotate around the x-axis
		vert->wz=FixedMul(vert->oy,cosxangle) - FixedMul(vert->oz,sinxangle);
		vert->wy=FixedMul(vert->oy,sinxangle) + FixedMul(vert->oz,cosxangle);
		vert->wx=vert->ox;

		// rotate around the y-axis
		nx=FixedMul(vert->wx,cosyangle) - FixedMul(vert->wz,sinyangle);
		nz=FixedMul(vert->wx,sinyangle) + FixedMul(vert->wz,cosyangle);
		vert->wx=nx;
		vert->wz=nz;

		// rotate around the z-axis
		nx=FixedMul(vert->wx,coszangle) - FixedMul(vert->wy,sinzangle);
		ny=FixedMul(vert->wx,sinzangle) + FixedMul(vert->wy,coszangle);
		vert->wx=nx;
		vert->wy=ny;

		// project the 3-D coordinates to screen coordinates
		vert->sx=Fixed2Int(FixedMul(FixedDiv(vert->wx+Int2Fixed(xpos),vert->wz+Int2Fixed(zpos)),Int2Fixed(256))) + 160;
		vert->sy=Fixed2Int(FixedMul(FixedDiv(vert->wy+Int2Fixed(ypos),vert->wz+Int2Fixed(zpos)),Int2Fixed(256))) + 100;
		vert->sz=Fixed2Int(vert->wz+Int2Fixed(zpos));

		// while we are at it, update the erase-box information
		if(vert->sx < eboxleft) eboxleft = vert->sx;
		if(vert->sx > eboxright) eboxright = vert->sx;
		if(vert->sy < eboxtop) eboxtop = vert->sy;
		if(vert->sy > eboxbottom) eboxbottom = vert->sy;
	}

	// increase the size of the dirty rectangle if necessary
	if(eboxleft1>eboxleft) eboxleft1=eboxleft;
	if(eboxright1<eboxright) eboxright1=eboxright;
	if(eboxtop1>eboxtop) eboxtop1=eboxtop;
	if(eboxbottom1<eboxbottom) eboxbottom1=eboxbottom;

	// clip the erase box
	if(eboxleft<ClipLeft) eboxleft=ClipLeft;
	if(eboxtop<ClipTop) eboxtop=ClipTop;
	if(eboxright>ClipRight) eboxright=ClipRight;
	if(eboxbottom>ClipBottom) eboxbottom=ClipBottom;
}

//--------------------------------------------------------------------------

void DrawObject(void)
{
	int i,j;
	char *temp_ptr;
	char *temp_ptr2;


	// wait for vertical retrace
//  WaitVRT();

	// copy the background into screen buffer
	for(i=eboxtop1;i<=eboxbottom1;i++)
	{
		temp_ptr=&screen_bg[ytable[i]+eboxleft1];
		temp_ptr2=&screen_buff[ytable[i]+eboxleft1];
		for(j=eboxleft1;j<=eboxright1;j++)
		{
			*temp_ptr2 = *temp_ptr;
			temp_ptr++;
			temp_ptr2++;
		}
	}


	MakePolygonList();
	SortPolygonList();

	VertexTYPE *v0,*v1,*v2,*v3;

	for(i=0;i<NumOfSortedPolygons;i++)
	{
		for(j=0;j<Object.Polygon[PolygonOrderList[i]].NumOfVertices-2;j++)
		{
			v0=&Object.Vertex[Object.Polygon[PolygonOrderList[i]].Vertex[0]];
			v1=&Object.Vertex[Object.Polygon[PolygonOrderList[i]].Vertex[j+1]];
			v2=&Object.Vertex[Object.Polygon[PolygonOrderList[i]].Vertex[j+2]];

			// if expression results in a negative then polygon is visible
			if( ((v1->sx - v0->sx) * (v2->sy - v0->sy) - (v1->sy - v0->sy) * (v2->sx - v0->sx)) < 0 )
			{
				// initilize the source texture coordinates to do environment mapping
				poly.sx1=160+Fixed2Int(FixedMul(v0->Nwx,Int2Fixed(-160)));
				poly.sy1=100+Fixed2Int(FixedMul(v0->Nwy,Int2Fixed(-100)));
				poly.sx2=160+Fixed2Int(FixedMul(v1->Nwx,Int2Fixed(-160)));
				poly.sy2=100+Fixed2Int(FixedMul(v1->Nwy,Int2Fixed(-100)));
				poly.sx3=160+Fixed2Int(FixedMul(v2->Nwx,Int2Fixed(-160)));
				poly.sy3=100+Fixed2Int(FixedMul(v2->Nwy,Int2Fixed(-100)));

				// initialize the screen texture coordinates
				poly.dx1=v0->sx;
				poly.dy1=v0->sy;
				poly.dx2=v1->sx;
				poly.dy2=v1->sy;
				poly.dx3=v2->sx;
				poly.dy3=v2->sy;

				TextureMap(&poly);
			}
		}
	}

	// fire the dirty rectangle to the screen
	for(i=eboxtop1;i<=eboxbottom1;i++)
	{
		temp_ptr=&screen_buff[ytable[i]+eboxleft1];
		temp_ptr2=&screen[ytable[i]+eboxleft1];
		for(j=eboxleft1;j<=eboxright1;j++)
		{
			*temp_ptr2 = *temp_ptr;
			temp_ptr++;
			temp_ptr2++;
		}
	}

	// pass down the erase box information
	eboxtop1=eboxtop;
	eboxbottom1=eboxbottom;
	eboxleft1=eboxleft;
	eboxright1=eboxright;
}

// load object

void LoadObject(void)
{
	FILE *file;
	int i,j;
	short temp;
	int result;
	VertexTYPE *vert; // pointer to a vertex structure
	char filename[80];

	system("dir *.v10/w");
	printf("\nEnter the filename (filename.v10): ");
	gets(filename);


	// open the file to read from it
	if ((file = fopen(filename,"rb")) == NULL)
	{
		printf("\n\nCannot open input file.\n");
	}
	else
	{
		// okay file is ready to read data from it

		// read number of dots in file
		fread(&temp,sizeof(short),1,file);
		Object.NumOfVertices=(int)temp;
		// read in all of the object's dots
		for(i=0;i < Object.NumOfVertices; i++)
		{
			fread(&temp,sizeof(short),1,file);
			Object.Vertex[i].ox=Int2Fixed(temp);
			fread(&temp,sizeof(short),1,file);
			Object.Vertex[i].oy=Int2Fixed(temp);
			fread(&temp,sizeof(short),1,file);
			Object.Vertex[i].oz=Int2Fixed(temp);
		}

		// read in the number of polygons
		fread(&temp,sizeof(short),1,file);
		Object.NumOfPolygons=(int)temp;
		// read in the information for each polygon
		for(i=0; i < Object.NumOfPolygons; i++)
		{
			// read in the number of vertices in polygon
			fread(&temp,sizeof(short),1,file);
			Object.Polygon[i].NumOfVertices=(int)temp;
			// read in the polygon's vertices (already in counter clockwise order)
			for(j=0; j< Object.Polygon[i].NumOfVertices; j++)
			{
				fread(&temp,sizeof(short),1,file);
				Object.Polygon[i].Vertex[j]=(int)temp;
			}
			// read the color of the polygon
			fread(&temp,sizeof(short),1,file);
		}

		// we're finished, close the file
		fclose(file);
	}
}

//---------------------- texture mapping stuff -----------------------

/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////

void TextureMap(TPolygonTYPE *poly)
{
	int i; // for looping
	char *temp_ptr;
	int temp;

	int sx,sy; // source texture coordinates
	int x,y; // screen coordinates
	int msx,msy; // slopes of source edge
	int ymin = ClipBottom;
	int ymax = ClipTop; // optimization

	// initialize the edge buffers to their extremes
	for(i=ClipTop;i<=ClipBottom;i++)
	{
		ledge_x[i]=ClipRight;
		redge_x[i]=ClipLeft;
	}

	// scan each edge of triangle
	TScanEdge(poly->dx1,poly->dy1,poly->dx2,poly->dy2,
						poly->sx1,poly->sy1,poly->sx2,poly->sy2);
	TScanEdge(poly->dx2,poly->dy2,poly->dx3,poly->dy3,
						poly->sx2,poly->sy2,poly->sx3,poly->sy3);
	TScanEdge(poly->dx3,poly->dy3,poly->dx1,poly->dy1,
						poly->sx3,poly->sy3,poly->sx1,poly->sy1);

	// find min and max y coordinates
	if(poly->dy1 < ymin) ymin = poly->dy1;
	if(poly->dy1 > ymax) ymax = poly->dy1;
	if(poly->dy2 < ymin) ymin = poly->dy2;
	if(poly->dy2 > ymax) ymax = poly->dy2;
	if(poly->dy3 < ymin) ymin = poly->dy3;
	if(poly->dy3 > ymax) ymax = poly->dy3;

	// make sure ymin and ymax are between ClipTop and ClipBottom
	if(ymin<ClipTop) ymin=ClipTop;
	if(ymax>ClipBottom) ymax=ClipBottom;

	// texture fill each horizontal scanline
	for(y=ymin;y<=ymax;y++)
	{
		// if the scanline belongs to the polygon... (scanline has pos width)
		if(ledge_x[y] <= redge_x[y])
		{
			// find the slopes of the source edges
			if( (redge_x[y] - ledge_x[y]) != 0)
			{
				msx = ((redge_sx[y] - ledge_sx[y]) << 16) /(redge_x[y] - ledge_x[y]);
				msy = ((redge_sy[y] - ledge_sy[y]) << 16) /(redge_x[y] - ledge_x[y]);
			}
			else
			{
				msx = (redge_sx[y] - ledge_sx[y])<<16;
				msy = (redge_sy[y] - ledge_sy[y])<<16;
			}

			// initialize source texture coordinates
			sx=ledge_sx[y]<<16;
			sy=ledge_sy[y]<<16;

			// clip the left edge if neccessary
			if(ledge_x[y] < ClipLeft)
			{
				temp = ClipLeft - ledge_x[y]; // find the amount to skip
				if(temp<0) temp=-temp; // make sure its positive

				ledge_x[y]=ClipLeft; // set it to left edge of clipping area
				sx+=FixedMul(msx,temp<<16);
				sy+=FixedMul(msy,temp<<16);
			}

			// clip right edge if necessary
			if(redge_x[y] > ClipRight)
			{
				redge_x[y] = ClipRight; // hehehe this is too easy!
			}

			// draw the horizontal scanline
			temp_ptr=&screen_buff[ytable[y]+ledge_x[y]];
			for(x=ledge_x[y];x<redge_x[y];x++)
			{
				*temp_ptr = poly->bitmap[ ytable[sy>>16] + (sx>>16) ];
				// increment source texture coordinates to trace edge
				sx+=msx;
				sy+=msy;
				temp_ptr++;
			}
		}
	}
}


void TScanEdge(int xd1,int yd1,int xd2,int yd2,
							 int xs1,int ys1,int xs2,int ys2)
{
	int msx,msy; // slopes of source x and y
	int mdx,mdy; // slopes of destination x and y
	int temp; // for swapping
	int ssx,ssy; // source x and y screen coordinates
	int dsx,dsy; // destination x and y screen coordinates

	// make sure that source edge goes from top to bottom
	if(yd1 > yd2)
	{
		// we need to swap the coordinates around
		temp=xd1;
		xd1=xd2;
		xd2=temp;

		temp=yd1;
		yd1=yd2;
		yd2=temp;

		temp=xs1;
		xs1=xs2;
		xs2=temp;

		temp=ys1;
		ys1=ys2;
		ys2=temp;
	}

	// initialize the slopes for stepping the edges
	if((yd2-yd1) != 0)
	{
		mdx = ((xd2-xd1) << 16) / (yd2-yd1); // dx_dest/dy_dest
		msx = ((xs2-xs1) << 16) / (yd2-yd1); // dx_src/dy_dest
		msy = ((ys2-ys1) << 16) / (yd2-yd1); // dy_src/dy_dest
	}
	else
	{
		mdx = ((xd2-xd1) << 16); // dx_dest
		msx = ((xs2-xs1) << 16); // dx_src
		msy = ((ys2-ys1) << 16); // dy_src
	}

	// initialize first coordinates
	ssx=xs1<<16;
	ssy=ys1<<16;
	dsx=xd1<<16;

	// step through edge and record coordinates along the way
	for(dsy=yd1;dsy<yd2;dsy++)
	{
		// as long as we are between the vertical clipping area... trace it
		if( (dsy>=ClipTop) && (dsy<=ClipBottom) )
		{
			if( (dsx>>16) < ledge_x[dsy])
			{
				// update left edge information
				ledge_x[dsy]=(dsx>>16);
				ledge_sx[dsy]=(ssx>>16);
				ledge_sy[dsy]=(ssy>>16);
			}

			if( (dsx>>16) > redge_x[dsy])
			{
				// update right edge information
				redge_x[dsy]=(dsx>>16);
				redge_sx[dsy]=(ssx>>16);
				redge_sy[dsy]=(ssy>>16);
			}
		}

		// increment the coordinates by their respective slopes
		dsx+=mdx;
		ssx+=msx;
		ssy+=msy;
	}
}


/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////



//------------------------------- palette functions ---------------

void GetPalette(unsigned char *palettebuffer)
{
	int i;

	for(i=0;i<256;i++)
	{
		outp(0x3c7,i);  // color number to get data from
		palettebuffer[i*3]   = inp(0x3c9);  // red
		palettebuffer[i*3+1] = inp(0x3c9);  // green
		palettebuffer[i*3+2] = inp(0x3c9);  // blue
	}
}

void SetPalette(unsigned char *palettebuffer)
{
	int i;

	for(i=0;i<256;i++)
	{
		outp(0x3c8,i);  // color number to set
		outp(0x3c9,palettebuffer[i*3]);   // red
		outp(0x3c9,palettebuffer[i*3+1]); // green
		outp(0x3c9,palettebuffer[i*3+2]); // blue
	}
}

void FadeInPalette(unsigned char *palettebuffer,int speed)
{
	// palettebuffer is the destination palette you want to reach
	// uses interpolation

	int i,j,k;
	short temppalette[768]={0};
	unsigned char temppalette2[768]={0};
	short delta[768];

	// find out the delta value of each color component for interpolation
	for(i=0;i<768;i++)
	{
		delta[i] = ( ( (short)(palettebuffer[i]) ) << 8 ) /64;
	}


	for(i=0;i<64;i++)
	{
		for(j=0;j<768;j++)
		{
			temppalette[j] += delta[j];
			temppalette2[j] = (unsigned char)(temppalette[j]>>8);
		}
		for(k=0;k<speed;k++)
		{
			WaitVRT();
		}
		SetPalette(temppalette2);
	}
}

void FadeOutPalette(int speed)
{
	int i,j,k;
	short temppalette[768]={0};
	unsigned char temppalette2[768]={0};
	short delta[768];

	GetPalette(temppalette2);

	for(i=0;i<768;i++)
	{
		temppalette[i] = ((short)(temppalette2[i]))<<8;
		delta[i] = (((short)temppalette2[i]) <<8 ) / (-64);
	}

	for(i=0;i<64;i++)
	{
		for(j=0;j<768;j++)
		{
			temppalette[j] += delta[j];
			temppalette2[j] = (unsigned char)(temppalette[j] >> 8);
		}
		for(k=0;k<speed;k++)
		{
			WaitVRT();
		}
		SetPalette(temppalette2);
	}
}




//---------------------------------------------------------------------

int main(void)
{
	SetMode03h();

	printf("Environment Mapping\n");
	printf("coded by Tumblin / BIM\n");

	Demo();

	SetMode03h();
	return(0);
}



//------------------------- PART ONE OF DEMO ---------------------------

void Demo(void)
{
	char key;
	int i; // for looping
	int deltaz; // add this to zpos for forward and backward object movement
	int deltay; // add this to ypos for up and down object movement
	int deltax; // add this to xpos for left and right object movement

	LoadObject();
	LoadGraphics();
	InitNormals();

	SetUpScreen();
	for(i=0;i<768;i++)
	{
		temp_palette[i]=0;
	}
	SetPalette(temp_palette);


	InitYTable();
	InitTrigTables();

	poly.bitmap=&bitmap[0];
	poly.height=64; // actual height of bitmap
	poly.width=64; // actual width of bitmap


	// display original bitmap on screen
	for(i=0;i<64000;i++)
	{
		screen[i]=screen_bg[i];
	}


	xangle=0;
	yangle=0;
	zangle=0;
	xpos=0;
	ypos=0;
	zpos=-1024;


	// draw first frame so that globe won't "pop out of nowhere" (:
	RotatePoints();
	RotateNormals();
	DrawObject();

	FadeInPalette(palette,2);

	deltax = 0;
	deltay = 0;
	deltaz = 10;

	do
	{
		zpos += deltaz;
		if( (zpos<-2048) || (zpos>-1024) ) deltaz = -deltaz;

		xangle+=4; if(xangle >= MAXDEGREES) xangle=0;
		yangle+=6; if(yangle >= MAXDEGREES) yangle=0;
		zangle+=8; if(zangle >= MAXDEGREES) zangle=0;

		RotatePoints();
		RotateNormals();
		DrawObject();
	} while(!kbhit());
	getch();
}

//-----------------------------------------------------------------

void LoadGraphics(void)
{
	int i;
	FILE *fp;


	if( (fp=fopen("texture.pal","rb")) == NULL)
	{
		printf("ERROR: Couldn't load palette data\n");
		exit(1);
	}
	fread(dummy_buffer,sizeof(char),768,fp);
	fclose(fp);

	// copy loaded palette to palette buffer
	for(i=0;i<768;i++)
	{
		palette[i]=dummy_buffer[i];
	}


	// load bitmap file used in environment mapping
	if( (fp=fopen("texture.raw","rb")) == NULL)
	{
		printf("ERROR: Couldn't load bitmap data.\n");
		exit(1);
	}
	fread(dummy_buffer,sizeof(char),64000,fp);
	fclose(fp);

	// copy dummy buffer to bitmap buffer used in texture mapping routine
	for(i=0;i<64000;i++)
	{
		bitmap[i]=dummy_buffer[i];
		screen_bg[i]=dummy_buffer[i];
	}
}


// make visible polygon list 

void MakePolygonList(void)
{
	int i,j,k;

	VertexTYPE *v0,*v1,*v2;

	j=0;
	for(i=0;i<Object.NumOfPolygons;i++)
	{
		v0=&Object.Vertex[Object.Polygon[i].Vertex[0]];
		v1=&Object.Vertex[Object.Polygon[i].Vertex[1]];
		v2=&Object.Vertex[Object.Polygon[i].Vertex[2]];

		// if expression results in a negative then polygon is visible
		if( ((v1->sx - v0->sx) * (v2->sy - v0->sy) - (v1->sy - v0->sy) * (v2->sx - v0->sx)) < 0 )
		{
			PolygonOrderList[j++]=i;
		}
	}
	NumOfSortedPolygons=j;
}

// sort polygons 


void SortPolygonList(void)
{
	int i,j;
	int maxz,minz;
	int temp;
	PolygonTYPE *poly;

	// first find the distance of each polygon (from midpoint of max & min z's)
	for(i=0;i<Object.NumOfPolygons;i++)
	{

		poly=&Object.Polygon[i];
		minz=65535;
		maxz=-65536;
		for(j=0;j<poly->NumOfVertices;j++)
		{
			if(Object.Vertex[poly->Vertex[j]].sz < minz)
			{
				minz=Object.Vertex[poly->Vertex[j]].sz;
			}
			if(Object.Vertex[poly->Vertex[j]].sz > maxz)
			{
				maxz=Object.Vertex[poly->Vertex[j]].sz;
			}
		}
		// now calculate the distance
		poly->zcenter=(maxz+minz)<<1;
	}

	// qsort the polygons
	qsort(PolygonOrderList,NumOfSortedPolygons,sizeof(int),ComparePolygons);

	// all done the sorting process
}

// compare balls (for qsort) 

int ComparePolygons(const void *a, const void *b)
{
	if( Object.Polygon[*(int *)a].zcenter < Object.Polygon[*(int *)b].zcenter )
	{
		return -1;
	}
	else if( Object.Polygon[*(int *)a].zcenter > Object.Polygon[*(int *)b].zcenter )
	{
		return +1;
	}
	else
	{
		return 0;
	}
}
