#ifndef _fi_full_h_included
#define _fi_full_h_included		// Prevent multiple includes
//******************************************************************************
//
//							   Full File Iterator
//								  (FI_FULL.H)
//
//	PURPOSE:	Used to find all instances of a file specification, within
//				a subdirectory tree. Unlike File Iterator, the search can
//				be fine-tuned, based on the file attributes, size, date,
//				& time.
//
//
//	  Date		Revision History
//	--------	------------------------------------------------------------
//	 7- 1-95	Initial draft of FullFileIterator.
//
//******************************************************************************

//------------------------------- HEADER FILES ---------------------------------

#include	<stdlib.h>		//  NULL
#include	<windows.h>		//  BOOL, BYTE, TRUE
#include	<_defs.h>		//  _CLASSTYPE()

#include	<file_itr.h>	//	FileIterator

//---------------------------- FORWARD DECLARATIONS ----------------------------

//--------------------------------- CONSTANTS ----------------------------------

//---------------------------- TYPE DECLARATIONS -------------------------------

typedef enum
{
	FI_Ignore, FI_EQ, FI_NE, FI_GT, FI_GE, FI_LT, FI_LE
} FI_Comparison;

//-------------------------- FUNCTION DECLARATIONS -----------------------------

//---------------------------- CLASS DECLARATIONS ------------------------------
//
//

class _CLASSTYPE FullFileIterator : public FileIterator
{
public:

    //----------------------------------------------------------
    //      Constructors.
    //

	FullFileIterator( void );
	FullFileIterator( const char *ApszFileSpec, BOOL AbRecurseSubdirs = TRUE );


    //----------------------------------------------------------
    //      Destructor.
    //

	~FullFileIterator( void );


    //----------------------------------------------------------
    //      Set the file attribute search criteria:
    //
    //          a		- Archive attribute
	//			h		- Hidden attribute
	//			r		- Read-only attribute
    //          s		- System File attribute
    //
    //      If an attribute is followed by a '+', or another
	//		attribute, the attribute must be set. If an
	//		attribute is followed by a '-', the attribute
	//		must be clear. If the attribute is not present,
	//		the attribute is ignored. For example, using the
	//		following call,
	//
	//				SetAttributes( "s-h-r" );
	//
	//		Finds all files that are read-only, but are not
    //      system, or hidden files.
	//			If successful, TRUE is returned; else, FALSE.
    //

	BOOL	SetAttributes( const char *ApszAttributes = NULL );

	BOOL	SetSize( FI_Comparison AeComparison = FI_Ignore, long AlSize = 0L );


    //----------------------------------------------------------
    //      Set the date & time search criteria.
	//			If an attribute is specified incorrectly,
	//		TRUE is returned; else, FALSE.
    //

	BOOL	SetDate( FI_Comparison AeComparison = FI_Ignore,
					 int AiDay = CiCurrent,
					 int AiMonth = CiCurrent,
					 int AiYear = CiCurrent );
	BOOL	SetTime( FI_Comparison AeComparison = FI_Ignore,
					 int AiHour = CiCurrent,
					 int AiMinute = 0 );


    //----------------------------------------------------------
    //      Typical for() usuage:
    //
    //          char            *pszDataFile;
    //          FileIterator    FileSearch( "*.dat" );
    //
    //          for (FileSearch.FirstFile();
    //               FileSearch.MoreFiles();
    //               FileSearch.NextFile())
    //          {
    //              pszDataFile = FileSearch;
    //
    //              //  process the data file
    //          }
    //

    void    FirstFile( void );
    void    NextFile( void );

	static int const CiCurrent;


private:

	BYTE			FActiveAttributes;
    BYTE			FAttributes;
	FI_Comparison	FeDateComparison;
	unsigned		FuDate;
	FI_Comparison	FeTimeComparison;
	unsigned		FuTime;
	FI_Comparison	FeSizeComparison;
	long			FlSize;


    //----------------------------------------------------------
    //      Given all the special search criteria, confirm
    //		that the current file does actual match.
    //

	BOOL	FileMatch( void ) const;


    //----------------------------------------------------------
    //      Helper functions for FileMatch(), for comparing
    //		the individual criteria.
    //

	BOOL	AttributesMatch( BYTE AAttribute ) const
	{
		return (((FActiveAttributes & AAttribute) == 0)? TRUE :
				((FfblkFileInfo.ff_attrib & AAttribute) ==
			  	(FAttributes & AAttribute)));
	}

	BOOL	DatesAreEqual( unsigned AuDate1, unsigned AuDate2 ) const;
	BOOL	DatesAreLessThan( unsigned AuDate1, unsigned AuDate2 ) const;

	BOOL	TimesAreEqual( unsigned AuTime1, unsigned AuTime2 ) const;
	BOOL	TimesAreLessThan( unsigned AuTime1, unsigned AuTime2 ) const;
};

#endif

