/*
    LogicDemo.java

    Greg Cusick
    http://www.cis.ufl.edu
    gjc@cis.ufl.edu

    OVERVIEW:

    This applet is an interactive demonstration of digital logic gates.
    It allows the construction and simulation of simple logic gate circuits.
    It has two basic modes:  edit mode and simulate mode.

    In edit mode, the circuit is constructed by selecting a gate from the
    toolbar with a mouse click and then moving the mouse to the gate's
    position and setting it with a second mouse click.  In the edit mode the
    inputs and outputs can be clicked on to make connections.

    In simulate mode, the values of each gate's inputs and output are shown.
    By clicking the mouse on a free input (an input that is not connected
    to another gate) the input value can be changed.  The circuit will be
    updated immediately.

    CONCLUSION:

    For future refinement and expansion, I have detailed some areas for
    improvement:

    There is no facility for deletion of gates or clearing the circuit.
    The wire routing could be done in a prettier fashion.
    There really should be a class for the pins (input and output).
    The handling of the coordinate systems (grid coordinates and pixel
    coordinates) could be cleaned up to look nicer.
    The gate classes could be smoothed over with better functionality.
    The painting functions could be examined to reduce flickering.
*/

import java.applet.Applet;                // applet class
import java.awt.*;                        // graphics
import java.util.Vector;                  // vector class

public class LogicDemo extends Applet {
    BoxedArea boxedArea;                  // panel with box and canvas
    Button modebutton;                    // the mode button
    Label modelabel;                      // the text label stating mode
    boolean edit = true;                  // mode control variable

    public void init() {                  // this runs first
        GridBagLayout gridBag = new GridBagLayout();  // using grid bag
        GridBagConstraints constraints = new GridBagConstraints(); // layout

        setLayout(gridBag);

        boxedArea = new BoxedArea(this);  // make boxed area
        constraints.fill = GridBagConstraints.BOTH;
        constraints.weighty = 1.0;        // vertical change when resizing
        constraints.gridwidth = GridBagConstraints.REMAINDER; // extend width
        gridBag.setConstraints(boxedArea, constraints); // set constraints 
        add(boxedArea);                   // add boxed area

        modebutton = new Button("Mode");  // make mode button
        constraints.fill = GridBagConstraints.HORIZONTAL;
        constraints.weighty = 0.0;        // no vertical change when resizing
        constraints.gridwidth = 1;        // width is 1 x required width
        gridBag.setConstraints(modebutton, constraints); // set constraints
        add(modebutton);                  // add mode button

        modelabel = new Label("Mode: " + (edit ? "Edit" : "Simulate"));
        constraints.weightx = 1.0;        // horizontal change when resizing
        constraints.gridwidth = GridBagConstraints.REMAINDER; // extend width
        gridBag.setConstraints(modelabel, constraints); // set constraints
        add(modelabel);                   // add mode label

        validate();                       // draw for first time
    }

    public boolean action(Event e, Object arg) { // handles events
        Object target = e.target;

        if (target == modebutton) {       // if mode button was pressed
            edit = !edit;                 // change between edit and simulate
            modelabel.setText("Mode: " + (edit ? "Edit" : "Simulate"));
            boxedArea.repaint();          // repaint the area
            return true;                  // true means we handled it
        }
        return false;                     // pass handling to parent/superclass
    }
}

class BoxedArea extends Panel {           // making a box with a border
    private CoordinateArea coordinateArea; // for the drawing area

    public BoxedArea(LogicDemo controller) { // pass in controller
        super();                          // call parent

        setLayout(new GridLayout(1,0));   // layout as big as possible

        add(coordinateArea = new CoordinateArea(controller)); // make the
                                          // canvas area for drawing
        validate();                       // draw for the first time
    }

    public void paint(Graphics g) {       // how to paint
        coordinateArea.repaint();         // repain coordinate area
    }
}

class CoordinateArea extends Canvas {     // the canvas for drawing
    Point point = null;                   // mouse and pixel locations
    LogicDemo controller;                 // record the controller
    Gate gate = null;                     // used to hold new gates
    Vector gates = new Vector();          // vector of all gates in circuit
    private final int POINTING = 0;       // pointing mode
    private final int GATEPLACE = 1;      // gate placement mode
    private final int GATECONNECT = 2;    // gate connection mode
    private final int GATEDELETE = 3;
    private int editmode = POINTING;      // initialize to pointing mode
    private Gate inputgate = null;        // input selected for a connection
    private Gate outputgate = null;       // output selected for a connection
    private int inputpinnum;              // the input pin number

    private int x, y;                     // canvas coordinates

    public CoordinateArea(LogicDemo controller) { // pass in controller
        super();                          // call superclass 
        this.controller = controller;     // record the controller
    }

    public boolean mouseDown(Event event, int x, int y) { // mouse clicked
        if (point == null) {              // if point is not initialized
            point = new Point(x, y);      // allocate and set point
        }
        else {
            point.x = x;                  // set x coordinate
            point.y = y;                  // set y coordinate
        }
        if (controller.edit) {            // if edit mode
            switch (editmode) {           // then check which state we are in
                case POINTING:            // if pointing or
                case GATECONNECT:         // if gate connecting then
                    if (y < 25) {         // if in the tool bar area
                        int gatenum = x / 50; // see which gate was clicked
                        switch (gatenum) {
                            case 0:       // this is where the toolbar text is
                                break;
                            case 1:       // this is where the and gate is
                                gate = new AndGate(point);
                                inputgate = null;
                                outputgate = null;
                                editmode = GATEPLACE;  // change state
                                break;
                            case 2:       // this is where the or gate is
                                gate = new OrGate(point);
                                inputgate = null;
                                outputgate = null;
                                editmode = GATEPLACE;  // change state
                                break;
                            case 3:       // this is where the not gate is
                                gate = new NotGate(point);
                                inputgate = null;
                                outputgate = null;
                                editmode = GATEPLACE;  // change state
                                break;
                            default:      // this handles clicking nothing
                                break;
                        }
                    }
                    else                  // clicked inside the work area
                        if (gates.size() > 0) { // find input or output selected
                            Gate temp = null;
                            for (int index = 0; // for each gate in circuit
                                (index < gates.size()) && (temp == null);
                                index++)
                                temp = ((Gate) gates.elementAt(index)).selected(point);
                            if (temp != null) { // see if gate was selected
                                switch (temp.selectedLocation(point)) {
                                    case 1: // selected input pin 1
                                        if ((inputgate == null) && // not an input pin-to-input pin connection
                                            (temp != outputgate) && // not feeding back from own output
                                            (temp.input1gate == null)) { // input not already connected
                                            inputgate = temp; // set input pin of connection
                                            inputpinnum = 1; // set input pin number
                                            if (outputgate == null) // if the output gate is not selected
                                                editmode = GATECONNECT; // then wait for selection of second gate
                                            else // otherwise it is already selected so complete the connection
                                                connect();
                                        }
                                        break;
                                    case 2: // selected input pin 2
                                        if ((inputgate == null) && // the input gate cannot be selected already
                                            (temp != outputgate) && // not feeding back from own output
                                            (temp.input2gate == null)) { // input not already connected
                                            inputgate = temp; // set input pin of connection
                                            inputpinnum = 2; // set input pin number
                                            if (outputgate == null) // if the output gate is not selected
                                                editmode = GATECONNECT; // then wait for selection of second gate                         
                                            else // otherwise it is already selected so complete the connection
                                                connect();
                                        }
                                        break;
                                    case 3: // selected output pin 1
                                        if ((outputgate == null) && // the output gate cannot be selected already
                                            (temp != inputgate)) { // not feeding back from own input
                                            outputgate = temp; // set output pin of connection
                                            if (inputgate == null) // if the input gate is not selected
                                                editmode = GATECONNECT; // then wait for selection of second gate
                                            else // otherwise it is already selected so complete the connection
                                                connect();
                                        }
                                        break;
                                    default: // selected something unknown
                                        break;
                                }
                            }
                        }
                        break;
                case GATEPLACE:           // placing a gate somewhere
                    int index;            // index all gates in circuit
                    for (index = 0;       // check to see if we are on top of another gate
                        (index < gates.size()) &&
                        !((((Gate) gates.elementAt(index)).x == gate.x) &&
                         (((Gate) gates.elementAt(index)).y == gate.y));
                        index++);
                    if (index == gates.size()) { // if we aren't on top of another gate
                        gates.addElement(gate); // add the gate to the circuit
                        gate = null;      // no gate selected now
                        editmode = POINTING; // back to pointing mode
                    }
                    break;
                case GATEDELETE:          // NOT IMPLEMENTED YET !!!
                    break;
            }
        }
        else {                            // simulating the circuit
            int input = 0;                // input number selected
            int index;                    // index all of the gates
            for (index = 0;               // check to see if an input was selected (>0)
                (index < gates.size()) && (input == 0);
                index++) {
                input = ((Gate) gates.elementAt(index)).inputSelected(point); // which input pin?
                if (input > 0) {
                    if (input == 1) {     // if pin number 1
                        if (((Gate) gates.elementAt(index)).input1gate == null) { // can't be connected to another gate
                            ((Gate) gates.elementAt(index)).changeInputState(input); // change the value
                            repaint();    // redraw
                        }
                    }
                    else                  // if pin number 2
                        if (((Gate) gates.elementAt(index)).input2gate == null) { // can't be connected to another gate
                            ((Gate) gates.elementAt(index)).changeInputState(input); // change the value
                            repaint();    // redraw
                        }
                }
            }
        }
        return false;                     // pass event to parent
    }

    public boolean mouseMove(Event event, int x, int y) { // the mouse moved
        if (controller.edit) {            // if edit mode
            updateGridCoords(x, y);       // move a gate around the grid
            if (editmode == GATECONNECT) { // if connecting up gates
                this.x = x;               // set the x for the wire
                this.y = y;               // set the y for the wire
                repaint();                // repaint
            }
        }
        return false;                     // pass event to parent
    }

    private void connect() {              // makes a connection
        if (inputpinnum == 1)             // if input pin 1 was selected
            inputgate.input1gate = outputgate; // set the input1 variable
        else                              // else if input pin 2 was selected
            inputgate.input2gate = outputgate; // set the input2 variable
        outputgate.outset.addElement(inputgate); // add the connection to the gate's outset
        if (inputgate.inputValue(inputpinnum) != outputgate.outputValue()) { // if output value needs adjusting
            inputgate.changeInputState(inputpinnum); // propogate the value
        }
        inputgate = null;                 // reset to null (nothing selected)
        outputgate = null;                // reset to null (nothing selected)
        editmode = POINTING;              // change the mode back to pointing
        repaint();                        // update the screen
    }

    private void updateGridCoords(int x, int y) { // snap x, y to grid coordinates
        boolean change = false;           // did we make a change?
        if (gate != null) {               // if there is a gate being dragged
            if ((x / gate.width +         // find x grid coordinate
                ((((x / gate.width) % 2) == 0) ? 1 : 0)) != gate.x) {
                gate.x = x / gate.width +
                    ((((x / gate.width) % 2) == 0) ? 1 : 0);
                change = true;            // a change was made
            }
            if ((y / gate.height +        // find y grid coordinant
                ((((y / gate.height) % 2) == 0) ? 1 : 0)) != gate.y) {
                gate.y = y / gate.height +
                    ((((y / gate.height) % 2) == 0) ? 1 : 0);
                change = true;            // a change was made
            }
        }
        if (change)                       // if a change was made
            repaint();                    // then redraw the screen
    }

    public void paint(Graphics g) {       // draw all the gates
        int index;                        // index through all gates
        Dimension dimension = size();     // get the dimensions

        g.drawRect(0, 0, dimension.width - 1, dimension.height - 1); // draw bounding box
        for(index = 0; index < gates.size(); index++) // for each gate
            ((Gate) gates.elementAt(index)).paintGate(g); // draw the gate
        if (controller.edit) {            // if in edit mode
            paintEditArea(g);             // draw the tool bar
            if (gate != null) {           // if we are dragging a gate
                gate.paintGate(g);        // draw the gate
            }
            if (editmode == GATECONNECT) {// if connecting a gate
                if (inputgate != null) {  // and if we selected the input pin already
                    if (inputpinnum == 1) {  // and if it is pin number 1
                        g.drawLine(inputgate.x * inputgate.width, // draw a connection line
                                   inputgate.y * inputgate.height + 6,
                                   x, y);
                    }
                    else {                // else draw a connection line from pin number 2
                        g.drawLine(inputgate.x * inputgate.width, // draw a connection line
                                   inputgate.y * inputgate.height + 18,
                                   x, y);
                    }
                }
                else {                    // else we selected an output pin already
                    g.drawLine(outputgate.x * outputgate.width + 45, // draw a connection line
                               outputgate.y * outputgate.height + 12,
                               x, y);
                }
            }
        }
        else                              // else we are in simulate mode
            for(index = 0; index < gates.size(); index++) // so display all the values
                ((Gate) gates.elementAt(index)).paintValues(g); // for each gate
    }

    public void paintEditArea(Graphics g) { // draw the tool bar
        Dimension dimension = size();     // get the dimensions
        g.drawLine(0, 25, dimension.width - 1, 25); // draw a horizontal line
        g.setFont(new Font("Helvetica", Font.PLAIN, 12)); // set a font
        g.drawString("Add", 3, 11);       // display:  Add
        g.drawString("Gate:", 3, 23);     // display:  Gate
        AndGate andGate = new AndGate(1, 0); // make the and gate
        andGate.paintGate(g);             // draw the and gate
        OrGate orGate = new OrGate(2, 0); // make the or gate
        orGate.paintGate(g);              // draw the or gate
        NotGate notGate = new NotGate(3, 0); // make the not gate
        notGate.paintGate(g);             // draw the not gate
    }
}

abstract class Gate {                     // superclass for gates
    public int width = 50;                // all have width of 50
    public int height = 25;               // all have height of 25
    public boolean not = false;           // set true for not gates
    public Gate input1gate = null;        // no input pin connections yet
    public Gate input2gate = null;        // no output pin connections yet
    public Vector outset = new Vector();  // set of output connections
    public int x;                         // x grid location
    public int y;                         // y grid location

    abstract public void changeInputState(int input); // toggle input values
    abstract public boolean inputValue(int inputnum); // check input values
    abstract public boolean outputValue();// check output value
    abstract public void compute();       // compute the gate function
    abstract public Gate selected(Point point); // does point select this gate
    abstract public int selectedLocation(Point point); // which part of gate is selected?
    abstract public boolean outputSelected(Point point); // is an output being selected?
    abstract public int inputSelected(Point point); // which input pin is selected?
    abstract public void paintGate(Graphics g); // draw the gate and its connections
    abstract public void paintValues(Graphics g); // draw the pin values next to the pins
}

class AndGate extends Gate {              // class for and gates
    private boolean input1 = false;       // input1's value
    private boolean input2 = false;       // input2's value
    private boolean output1;              // output1's value

    public AndGate(int x, int y) {        // constructor when giving grid coordinates
        this.x = x;                       // set x
        this.y = y;                       // set y
        compute();                        // compute the output value
    }

    public AndGate(Point point) {         // constructor when giving pixel point
        this.x = point.x / this.width +   // find x grid coordinate
            ((((point.x / this.width) % 2) == 0) ? 1 : 0);
        this.y = point.y / this.height +  // find y grid coordinate
            ((((point.y / this.height) % 2) == 0) ? 1 : 0);
        compute();                        // compute the output value
    }

    public void changeInputState(int inputpinnum) { // toggles input values
        if (inputpinnum == 1)             // if pin 1
            input1 = !input1;             // change input1's value
        else                              // otherwise assuming it must be pin 2
            input2 = !input2;             // change input2's value
        compute();                        // compute the output value
    }

    public Point scaleLocation(int x, int y) { // given grid coordinates
        return new Point(x * width, y * height); // return pixel location
    }

    public boolean inputValue(int inputpinnum) { // given input pin number
        if (inputpinnum == 1)             // if pin 1
            return input1;                // return input1's value
        return input2;                    // otherwise return input2's value
    }

    public boolean outputValue() {        // checking the output value
        return output1;                   // return output1's value
    }

    public void compute() {               // compute the and function
        if (output1 != (input1 & input2)) { // if output1 is incorrect
            output1 = (input1 & input2);  // change its value
            for (int index = 0; index < outset.size(); index++) { // for each gate in outset
                if (((Gate) outset.elementAt(index)).input1gate == this) { // if gate's input is connected to this output
                    if (((Gate) outset.elementAt(index)).inputValue(1) != // and the value is incorrect
                        outputValue()) {
                        ((Gate) outset.elementAt(index)).changeInputState(1); // change the value
                    }
                }
                else {                    // else must be connected to input pin 2
                    if (((Gate) outset.elementAt(index)).inputValue(2) != // if value is incorrect
                        outputValue()) {
                        ((Gate) outset.elementAt(index)).changeInputState(2); // change the value
                    }
                }
            }
        }
    }

    public Gate selected(Point point) {   // given pixel location 
        if (((point.x / width) == x) && ((point.y / height) == y)) // if gate is selected
            return this;                  // return itself
        return null;                      // otherwise return null
    }

    public int selectedLocation(Point point) { // given pixel location
        int pinnum = inputSelected(point);// get input pin number selected
        if (pinnum > 0)                   // if an input pin was selected
            return pinnum;                // return the input pin number
        if (outputSelected(point))        // if the output pin was selected
            return 3;                     // return constant 3
        return 0;                         // otherwise return constant 0
    }

    public boolean outputSelected(Point point) { // given pixel location
        Point scaled = scaleLocation(x, y); // get pixel location of gate
        return ((point.x > scaled.x + 35) && (point.y > scaled.y + 5) && // return true if near output
            (point.x < scaled.x + width) && (point.y < scaled.y + height - 5));
    }

    public int inputSelected(Point point) { // given pixel location
        Point scaled = scaleLocation(x, y); // get pixel location of gate
        if ((point.x > scaled.x) && (point.y > scaled.y - 4) && // if near input pins
            (point.x < scaled.x + 10) && (point.y < scaled.y + 20))
           if (point.y < scaled.y + 8)    // if near upper pin
                return 1;                 // return constant 1
            else                          // otherwise in lower region
                return 2;                 // return constant 2
        else                              // otherwise not near input pins
            return 0;                     // return constant 0
    }

    public void paintGate(Graphics g) {   // draws the and gate
        Point scaled = scaleLocation(x, y); // get pixel location
        g.drawLine(scaled.x + 24, scaled.y + 22,
                   scaled.x + 11, scaled.y + 22); // bottom edge
        g.drawLine(scaled.x + 10, scaled.y + 22,
                   scaled.x + 10, scaled.y +  3); // left edge
        g.drawLine(scaled.x + 10, scaled.y +  2,
                   scaled.x + 24, scaled.y +  2); // top edge
        g.drawArc (scaled.x + 15, scaled.y + 2,
                   20, 20, 90, -180);     // right edge (arc)
        g.drawLine(scaled.x,      scaled.y +  6,
                   scaled.x +  9, scaled.y +  6); // input1 line
        g.drawLine(scaled.x,      scaled.y + 18,
                   scaled.x +  9, scaled.y + 18); // input2 line
        g.drawLine(scaled.x + 35, scaled.y + 12,
                   scaled.x + 44, scaled.y + 12); // output1 line
        for (int index = 0; index < outset.size(); index++) { // for each connection
            Gate gate = (Gate) outset.elementAt(index);
            Point inputPoint = scaleLocation(gate.x, gate.y); // get pixel location
            if (gate.input1gate == this) {// if input pin 1 is connected to us
                if (gate.not)             // if it is a not gate (ie. only one input)
                    g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                               inputPoint.x, inputPoint.y + 12);
                else                      // otherwise it is an and gate or an or gate
                    g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                               inputPoint.x, inputPoint.y +  6);
            }
            else                          // otherwise if must be input pin 2
                g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                           inputPoint.x, inputPoint.y + 18);
        }
    }

    public void paintValues(Graphics g) { // draw logic values around gate
        Point scaled = scaleLocation(x, y); // get pixel location
        g.setFont(new Font("Helvetica", Font.PLAIN, 8)); // set font
        g.drawString(input1  ? "1" : "0", scaled.x +  3,
                                          scaled.y +  5); // input1 value
        g.drawString(input2  ? "1" : "0", scaled.x +  3,
                                          scaled.y + 17); // input2 value
        g.drawString(output1 ? "1" : "0", scaled.x + 38,
                                          scaled.y + 11); // output1 value
    }
}

class OrGate extends Gate {               // class for or gate
    private boolean input1 = false;       // input1's value
    private boolean input2 = false;       // input2's value
    private boolean output1;              // output1's value

    public OrGate(int x, int y) {         // constructor for grid coordinates
        this.x = x;                       // set x grid coordinate
        this.y = y;                       // set y grid coordinate
        compute();                        // compute the gate's function
    }

    public OrGate(Point point) {          // constructor for pixel location
        this.x = point.x / this.width +   // get x grid coordinate
            ((((point.x / this.width) % 2) == 0) ? 1 : 0);
        this.y = point.y / this.height +  // get y grid coordinate
            ((((point.y / this.height) % 2) == 0) ? 1 : 0);
        compute();                        // compute the gate's function
    }

    public void changeInputState(int inputpinnum) { // given an input pin number
        if (inputpinnum == 1)             // if input pin 1
            input1 = !input1;             // change input1's value
        else                              // otherwise must be input pin 2
            input2 = !input2;             // change input2's value
        compute();                        // compute the gate's function
    }

    public Point scaleLocation(int x, int y) { // given grid coordinates
        return new Point(x * width, y * height); // return pixel location
    }

    public boolean inputValue(int inputpinnum) { // given an input pin number
        if (inputpinnum == 1)             // if input pin 1
            return input1;                // return input1's value
        return input2;                    // otherwise return input2's value
    }

    public boolean outputValue() {        // get a gate's output value
        return output1;                   // return output1's value
    }

    public void compute() {               // compute the or function
        if (output1 != (input1 | input2)) { // if output1 is incorrect
            output1 = (input1 | input2);  // update output1
            for (int index = 0; index < outset.size(); index++) { // for each gate
                if (((Gate) outset.elementAt(index)).input1gate == this) { // if connected to this gate
                    if (((Gate) outset.elementAt(index)).inputValue(1) != // and the value is incorrect
                        outputValue()) {
                        ((Gate) outset.elementAt(index)).changeInputState(1); // change the input's value
                    }
                }
                else {                    // must be connected to input pin 2
                    if (((Gate) outset.elementAt(index)).inputValue(2) != // if the value is incorrect
                        outputValue()) {
                        ((Gate) outset.elementAt(index)).changeInputState(2); // change the input's value
                    }
                }
            }
        }
    }

    public Gate selected(Point point) {   // given pixel location
        if (((point.x / width) == x) && ((point.y / height) == y)) // if gate is selected
            return this;                  // return this gate
        return  null;                     // otherwise return null
    }

    public int selectedLocation(Point point) { // given pixel location
        int pinnum = inputSelected(point);// get number of pin selected
        if (pinnum > 0)                   // if an input pin was selected
            return pinnum;                // return the input pin number
        if (outputSelected(point))        // if an output pin was selected
            return 3;                     // return the output pin constant 3
        return 0;                         // otherwise return constant 0
    }

    public boolean outputSelected(Point point) { // given pixel lcoation
        Point scaled = scaleLocation(x, y); // get pixel location of gate
        return ((point.x > scaled.x + 35) && (point.y > scaled.y + 5) && // return true if output is selected
            (point.x < scaled.x + width) && (point.y < scaled.y + height - 5));
    }

    public int inputSelected(Point point) { // given pixel location
        Point scaled = scaleLocation(x, y); // get pixel location of gate
        if ((point.x > scaled.x) && (point.y > scaled.y - 4) && // if near input pins
            (point.x < scaled.x + 10) && (point.y < scaled.y + 20))
           if (point.y < scaled.y + 8)    // if in upper region
                return 1;                 // return 1 for input pin 1
            else                          // otherwise must be in the lower region
                return 2;                 // return 2 for input pin 2
        else                              // otherwise
            return 0;                     // return 0 for no input pin selected
    }

    public void paintGate(Graphics g) {   // draw the or gate
        Point scaled = scaleLocation(x, y); // get pixel location
        g.drawArc (scaled.x +  4, scaled.y + 2,
                   12, 20, 90, -180);             // left edge (arc)
        g.drawArc (scaled.x - 15, scaled.y + 2,
                   50, 20, 90, -180);             // right edge (arc)
        g.drawLine(scaled.x,      scaled.y +  6,
                   scaled.x + 13, scaled.y +  6); // input1 line
        g.drawLine(scaled.x,      scaled.y + 18,
                   scaled.x + 13, scaled.y + 18); // input2 line
        g.drawLine(scaled.x + 35, scaled.y + 12,
                   scaled.x + 44, scaled.y + 12); // output1 line
        for (int index = 0; index < outset.size(); index++) {  // for each gate in outset
            Gate gate = (Gate) outset.elementAt(index);
            Point inputPoint = scaleLocation(gate.x, gate.y); // get pixel location
            if (gate.input1gate == this) {// if connected to input pin 1
                if (gate.not)             // if connected to a not gate
                    g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                               inputPoint.x, inputPoint.y + 12);
                else                      // otherwise must be an and or an or gate
                    g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                               inputPoint.x, inputPoint.y +  6);
            }
            else                          // otherwise must be connected to input pin 2
                g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                           inputPoint.x, inputPoint.y + 18);
        }
    }

    public void paintValues(Graphics g) { // draw the values around the gate
        Point scaled = scaleLocation(x, y); // get pixel location
        g.setFont(new Font("Helvetica", Font.PLAIN, 8)); // set the font
        g.drawString(input1  ? "1" : "0", scaled.x +  3,
                                          scaled.y +  5); // input1 value
        g.drawString(input2  ? "1" : "0", scaled.x +  3,
                                          scaled.y + 17); // input2 value
        g.drawString(output1 ? "1" : "0", scaled.x + 38,
                                          scaled.y + 11); // output1 value
    }
}

class NotGate extends Gate {              // class for not (inverter) gate
    private boolean input1 = false;       // input1's value
    private boolean input2 = false;       // input2's value
    private boolean output1;              // output1's value

    public NotGate(int x, int y) {        // constructor for grid coordinates
        not = true;                       // this IS a not gate
        this.x = x;                       // set x grid coordinate
        this.y = y;                       // set y grid coordinate
        compute();                        // compute the not function
    }

    public NotGate(Point point) {         // constructor for pixel location
        not = true;                       // this IS a not gate
        this.x = point.x / this.width +   // get x grid coordinate
            ((((point.x / this.width) % 2) == 0) ? 1 : 0);
        this.y = point.y / this.height +  // get y grid coordinate
            ((((point.y / this.height) % 2) == 0) ? 1 : 0);
        compute();                        // compute the not function
    }

    public void changeInputState(int inputpinnum) { // given input pin number
        input1 = !input1;                 // change input pin 1's value
        compute();                        // compute the not function
    }

    public Point scaleLocation(int x, int y) { // given grid coordinates
        return new Point(x * width, y * height); // return pixel location
    }

    public boolean inputValue(int inputpinnum) { // given input pin number
        return input1;                    // return input1's value
    }

    public boolean outputValue() {        // check output value
        return output1;                   // return output1's value
    }

    public void compute() {               // compute the not function
        if (output1 == input1) {          // if values are the same
            output1 = !input1;            // change the output value
            for (int index = 0; index < outset.size(); index++) { // for all gates in outset
                if (((Gate) outset.elementAt(index)).input1gate == this) { // if connected to input pin 1
                    if (((Gate) outset.elementAt(index)).inputValue(1) != // and the value is incorrect
                        outputValue()) {
                        ((Gate) outset.elementAt(index)).changeInputState(1); // change the input pin's value
                    }
                }
                else {                    // otherwise must be connected to input pin 2
                    if (((Gate) outset.elementAt(index)).inputValue(2) != // if value is incorrect
                        outputValue()) {
                        ((Gate) outset.elementAt(index)).changeInputState(2); // change the input pin's value
                    }
                }
            }
        }
    }

    public Gate selected(Point point) {   // given pixel location
        if (((point.x / width) == x) && ((point.y / height) == y)) // if selecting gate
            return this;                  // return this gate
        return null;                      // otherwise return null
    }

    public int selectedLocation(Point point) { // given pixel location
        int pinnum = inputSelected(point);// get selected input pin number
        if (pinnum > 0)                   // if an input pin is selected
            return pinnum;                // return the number of the pin selected
        if (outputSelected(point))        // if an output pin is selected
            return 3;                     // return output pin constant 3
        return 0;                         // otherwise return constant 0
    }

    public boolean outputSelected(Point point) { // given pixel location
        Point scaled = scaleLocation(x, y);// get pixel location of gate
        return ((point.x > scaled.x + 35) && (point.y > scaled.y + 5) && // return true if output selected
            (point.x < scaled.x + width) && (point.y < scaled.y + height - 5));
    }

    public int inputSelected(Point point) { // given pixel location
        Point scaled = scaleLocation(x, y); // get pixel location of gate
        if ((point.x > scaled.x) && (point.y > scaled.y - 4) && // if near input pin
            (point.x < scaled.x + 10) && (point.y < scaled.y + 20))
            return 1;                     // return constant 1 for input pin 1
        else                              // otherwise
            return 0;                     // return constant 0 for no input pin
    }

    public void paintGate(Graphics g) {   // draw the not gate
        Point scaled = scaleLocation(x, y); // get pixel location
        g.drawLine (scaled.x + 10, scaled.y + 21,
                    scaled.x + 10, scaled.y +  3); // left edge
        g.drawLine (scaled.x + 10, scaled.y +  2,
                    scaled.x + 34, scaled.y + 12); // top edge
        g.drawLine (scaled.x + 34, scaled.y + 12,
                    scaled.x + 10, scaled.y + 22); // bottom edge
        g.drawLine (scaled.x,      scaled.y + 12,
                    scaled.x +  9, scaled.y + 12); // input1 line
        g.drawLine (scaled.x + 35, scaled.y + 12,
                    scaled.x + 44, scaled.y + 12); // output1 line
        for (int index = 0; index < outset.size(); index++) { // for each gate in outset
            Gate gate = (Gate) outset.elementAt(index);
            Point inputPoint = scaleLocation(gate.x, gate.y); // get pixel location of gate
            if (gate.input1gate == this) {// if connected to input pin 1
                if (gate.not)             // if it is a not gate
                    g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                               inputPoint.x, inputPoint.y + 12);
                else                      // otherwise it must be an and or an or gate
                    g.drawLine(scaled.x + 45, scaled.y + 12, // draw connection
                               inputPoint.x, inputPoint.y +  6);
            }
            else                          // otherwise must be connected to input pin 2
                g.drawLine(scaled.x + 45, scaled.y + 12, // draw the connection
                           inputPoint.x, inputPoint.y + 18);
        }
    }

    public void paintValues(Graphics g) { // draw values around gate
        Point scaled = scaleLocation(x, y); // get pixel location
        g.setFont(new Font("Helvetica", Font.PLAIN, 8)); // set font
        g.drawString(input1  ? "1" : "0", scaled.x +  3,
                                          scaled.y + 11); // input1 value
        g.drawString(output1 ? "1" : "0", scaled.x + 38,
                                          scaled.y + 11); // output1 value
    }
}
