-- This program is based on the genetic algorithm from the July 1992 issue of
-- Scientific American.
-- It represents values of x for the function f(x)=x + |sin(32x)| as strings
-- of bits. The population initially has random bits set.
-- Each generation, the program evaluates and sorts the population, then plays
-- them off against each other in a tournament, where the fittest of 2 random
-- individuals is usually copied to the next generation. This mimics natural
-- selection in a population of real creatures.
-- It then mates some of the copied strings with each other by exchanging some
-- of their bits at a randomlly selected crossover point - this is akin to
-- offspring having some DNA from each parent.
-- Finally, it mutates a number of bits at random. This and the crossover help
-- avoid the population getting stuck at a local maximum.
-- The variable best contains the bit string for the fittest individual the
-- program has yet found.
-- I have found that for a population of 200, less than 100 generations are
-- often enough for it to arrive at the optimum value of around 4.09299, even
-- starting from a population of all 0's (ie pcent in generate() set to 0).

-- This is a conversion from Turbo Pascal for Windows 1.5, and it runs just
-- over than 3 times as fast in Euphoria.

-- Gavin Doig, 24.3.97
-- marrcps@aol.com

include machine.e
include sort.e

global constant len = 16,      --# of bits in gene string
		size = 200,    --Size of pop
		mutate = 15,    --%age bits to mutate 5=0.5%
		ngen = 100,    --# gens to run for
		cross = 0.75,  --rate of crossover
		select = 75,   --selection: 50 => none, 100 => always
		copyfirst = 0, --1 autocopy fittest, 0 no autocopy
		pi = 3.141592653585, 
		scalefactor = pi/65536,     --pi/(2^16))
		f = 1,         --fitness part of population sequence
		g = 2          --gene bits of population sequence

global atom avfit
avfit=0                 --Average fitness of population

global sequence population --Sequence holding fitnesses and gene strings
population = repeat({0,repeat(len,0)},size)

global sequence best

-- Function to fill a sequence with random bits
function rnd(integer x, object l)
object r
if l=1 then
    return rand(2)-1
end if
r=repeat(0,l)
for rndloop = 1 to l do
    if rand(100)<x then
	r[rndloop] =  1
    else
	r[rndloop] = 0
    end if
end for
return r
end function

-- Returns absolute value of x
function abs(atom x)
	if x < 0 then
		return -x
	else
		return x
	end if
end function


--Generate population, pcent% of bits true
function generate(atom pcent,sequence pop)
    for loop = 1 to size do
	pop[loop][g]=rnd(pcent,len)
    end for
    return pop
end function



-- Find fitness for gene string ind, f(x)=x+|sin 32x|
function findfitness(sequence ind)
    atom x
    x=bits_to_int(ind)*scalefactor
    return x+abs(sin(32*x))
end function


--Display the entire population, mostly used for debugging
procedure display(sequence pop)
sequence disp
disp=repeat({repeat({},len)},2*len+1)
    for dl=1 to size do
	disp=pop[dl]
	? disp
    end for
end procedure


--Evaluate entire population
function evaluate(sequence pop)
atom total
total=0
    for evloop = 1 to size do
	pop[evloop][f]=findfitness(pop[evloop][g])
	total=total+pop[evloop][f]
    end for
    avfit=total/size
    return pop
end function


--Mate a,b at rand point, offspring into c,d. By using different values of
--c, d, less fit individuals could be overwritten by offspring of more fit. 
--parents={a,b,c,d}
function crossover(sequence parents,sequence pop)
atom a,b,c,d,point
sequence ab,ba,gc,gd
a=parents[1]
b=parents[2]
c=parents[3]
d=parents[4]
    point=rand(len-1)+1
    ab=pop[a][g]
    ba=pop[b][g]
    gc=ba
    gd=ab
    gc[1..point]=ab[1..point]
    gd[1..point]=ba[1..point]
    pop[c][g]=gc
    pop[d][g]=gd
    return pop
end function


--Mutate random bits
function domutate(sequence pop)
atom r
sequence i
for l1=1 to size do
    for l2=1 to len do
	if rand(10000)<=mutate then
	    i=pop[l1][g]
	    r=rnd(50,1)
	    i[l2]=r
	    pop[l1][g]=i
	end if
    end for
end for
    return pop
end function


--Main tournament and genetic algorithm
procedure genetic()
sequence newpop
atom a,b,best,nbest
a=0
b=0
newpop=repeat({0,repeat(len,0)},size)

    --Copy fittest?
    if copyfirst=1 then
	newpop[size][g]=population[size][g]
    end if

    -- Do tournament until newpop is filled
    for loop =1 to (size-copyfirst) do
	a=rand(size)
	b=rand(size)
	--sorted, so a>b => fitness(a)>fitness(b)
	best=b
	nbest=a
	if a>b then
	    best=a
	    nbest=b
	end if
	if rand(100)<select then
	    newpop[loop][g]=population[best][g]
	else
	    newpop[loop][g]=population[nbest][g]
	end if
    end for

    --Try turning this part off. It will run much faster, and if you set
    --mutate to around 15, and select to 75 the program will arrive at the
    --optimum value in around the same number of generation.
    --I suspect this is because offspring overwrite their parents, ie if
    --c, d were set to individuals at lower fitnesses then mating might be
    --more effective.
    --Mate strings into selves
    for loop = 1 to floor(0.5*size*cross) do
	a=rand(size)
	b=rand(size)
	newpop=crossover({a,b,(loop+size*0.5),(loop+size*0.5-1)},newpop)
    end for

    --Mutations
    newpop=domutate(newpop)

    --update population
    population=newpop
end procedure


-- Setup variables
atom t
t=time()
population=generate(10,population)
population=evaluate(population)
population=sort(population)
printf(1,"Av. fitness at start: %9.8f\n",avfit)
best={}
-- And do main loop
for gen = 1 to ngen do
    if findfitness(best) < population[size][f] then
	best=population[size][g]
    end if
    genetic()
    population=evaluate(population)
    population=sort(population)
    printf(1,"Best so far: %9.8f\n",findfitness(best))
end for
puts(1,"Best individual is ")
? best
t = time()-t
printf(1,"In %5.1f seconds",t)
