%TITLE  "Asynchronous Communications Module for Turbo Pascal V2.03"
%OUT	ASYNC203 Assembly drivers              (C) 1989-1993, Rising Edge Data Services
%PAGESIZE 62,255
;********************************************************
;*							*
;*   Low-level serial communications service routines	*
;*	   For the ASYNC200 Turbo Pascal UNIT		*
;*							*
;*------------------------------------------------------*
;*							*
;*  Copyright (C) 1988-1990, Rising Edge Data Services	*
;*		Author: Mark L. Schultz			*
;*							*
;*------------------------------------------------------*
;*							*
;*	Routines compatable with all 80xxx CPU's	*
;*    Designed to operate on PC-compatable hardware	*
;*							*
;* Developed using the Borland Turbo Assembler (V1.01)	*
;*		Syntax: MASM compatable			*
;*							*
;*------------------------------------------------------*
;* Version 2.03				   Mon 08/09/93	*
;********************************************************
%NEWPAGE
%SUBTTL	"Bit-mapped variable definitions"
;********************************************************
;*							*
;*	    Bit-mapped variable definitions		*
;*							*
;********************************************************

;	Status byte definition (C_Status[Port#])

;	7   6   5   4   3   2   1   0
;	|   |   |   |   |   |   |   |___ Receive buffer empty
;	|   |   |   |   |   |   |_______ Receive buffer full
;	|   |   |   |   |   |___________ Transmit buffer empty
;	|   |   |   |   |_______________ Transmit buffer full
;	|   |   |   |___________________
;	|   |   |_______________________ Transmit interrupt mask
;	|   |___________________________ Soft handshake holdoff
;	|_______________________________ Hard handshake holdoff

;******************************************************************************

;	Control byte definition (C_Ctrl[Port#])

;	7   6   5   4   3   2   1   0
;	|   |   |   |   |   |   |   |___ Enable transmit hard handshake
;	|   |   |   |   |   |   |_______ Enable transmit soft handshake
;	|   |   |   |   |   |___________ Enable receive hard handshake
;	|   |   |   |   |_______________ Enable receive soft handshake
;	|   |   |   |___________________
;	|   |   |_______________________ Soft handshake pseudo-signal
;	|   |___________________________ Force-transmit XOFF
;	|_______________________________ Force-transmit XON
%NEWPAGE
;******************************************************************************

;	HWCheck[Port#] and HWMatch[Port#] register declarations

;	HWCheck[] determines which RS-232 signal lines will be checked to
;	determine the transmit enable state (0=Ignore, 1=Check).
;	HWMatch[] designates the state (0=Negative, 1=Positive) that the
;	checked lines must be in if transmission is to be enabled.
;	Typically, positive logic is used by RS-232 devices to signal that
;	transmission is OK, so HWCheck[] and HWModify[] are usually programmed
;	to identical values.

;	7   6   5   4   3   2   1   0
;	|   |   |   |   |   |   |   |___ Clear to send (CTS)
;	|   |   |   |   |   |   |_______ Data set ready (DSR)
;	|   |   |   |   |   |___________ Ring indicator (RI)
;	|   |   |   |   |_______________ Data carrier detect (DCD)
;	|   |   |   |___________________ Unused, must be 0
;	|   |   |_______________________ Unused, must be 0
;	|   |___________________________ Unused, must be 0
;	|_______________________________ Unused, must be 0

;******************************************************************************

;	HWModify[Port#] and HWAssert[Port#] register declarations

;	HWModify determines which RS-232 signal lines will be modified when
;	a change of state in handshaking occurs (0=Ignore, 1=Modify)
;	HWAssert determines the state in which the RS-232 output lines should
;	be placed in when handshaking is asserted (0=Negative, 1=Positive).
;	This register is internally inverted when handshaking is unasserted.
;	Typically, positive logic is used by RS-232 devices to signal that
;	transmission is OK, so HWModify[] and HWAssert[] are usually programmed
;	to identical values.

;	7   6   5   4   3   2   1   0
;	|   |   |   |   |   |   |   |___ Data terminal ready (DTR)
;	|   |   |   |   |   |   |_______ Request to send (RTS)
;	|   |   |   |   |   |___________ OUT1 signal line
;	|   |   |   |   |_______________ Unused, must be 0
;	|   |   |   |___________________ Unused, must be 0
;	|   |   |_______________________ Unused, must be 0
;	|   |___________________________ Unused, must be 0
;	|_______________________________ Unused, must be 0
%NEWPAGE
%SUBTTL	"Data segment (variable) declarations"
;********************************************************
;*							*
;*	   DATA segment (variable) declarations		*
;*							*
;********************************************************

DATA	SEGMENT	WORD PUBLIC

;	Externally accessed variables (declared in ASYNC.PAS)
;	Note: Most of these variables are declared as ARRAYs in ASYNC.PAS;
;	      i.e. C_InSize : Array[1..MaxPort] of Word

	EXTRN	C_InBufPtr	:DWORD	;Pointer to input (receive) buffers
	EXTRN	C_OutBufPtr	:DWORD	;Pointer to output (transmit) buffers
	EXTRN	C_InSize	:WORD	;Size (bytes) of input buffers
	EXTRN	C_OutSize	:WORD	;Size (bytes) of output buffers
	EXTRN	C_InHead	:WORD	;Input head pointers
	EXTRN	C_InTail	:WORD	;Input tail pointers
	EXTRN	C_OutHead	:WORD	;Output head pointers
	EXTRN	C_OutTail	:WORD	;Output tail pointers
	EXTRN	C_PortAddr	:WORD	;Addresses of 8250 ports
	EXTRN	C_HSOn		:WORD	;Handshake assert point
	EXTRN	C_HSOff		:WORD	;Handshake unassert point
	EXTRN	C_PortOpen	:BYTE	;Port-open flags
	EXTRN	C_PortExist	:BYTE	;Port-exist flags
	EXTRN	C_Status	:BYTE	;Status byte (see above declaration)
	EXTRN	C_Ctrl		:BYTE	;Control byte (see above declaration)
	EXTRN	C_XON		:BYTE	;XON software handshake character
	EXTRN	C_XOFF		:BYTE	;XOFF software handshake character
	EXTRN	C_HWCheck	:BYTE	;Hard Handshake input lines to check
	EXTRN	C_HWMatch	:BYTE	;Hard Handshake input lines match byte
	EXTRN	C_HWModify	:BYTE	;Hard Handshake output lines to modify
	EXTRN	C_HWAssert	:BYTE	;Hard Handshake output lines to assert
	EXTRN	C_RcvError	:BYTE	;Count of receive errors
	EXTRN	C_RcvBreak	:BYTE	;Count of BREAK signals received
	EXTRN	C_RcvWait	:BYTE	;Flag: Wait for character in ComReadCh
        EXTRN   C_XmitWait      :BYTE   ;Flag: Wait for buf space in ComWriteCh

	EXTRN	C_Error		:BYTE	;Error return (used by procedures)
	EXTRN	C_MaxCom	:BYTE	;Highest port # available
	EXTRN	C_Temp		:BYTE	;Temporary used for debugging

;	Variables used internally (not accessable from UNIT)

PortNbr	DB	?			;Port #/byte array index (ISR use only)

;	8250/16450 Register offsets

RXB	EQU	0			;Receive holding register
TXB	EQU	0			;Transmit holding register
DLL	EQU	0			;Divisor latch low byte
DLH	EQU	1			;Divisor latch high byte
IER	EQU	1			;Interrupt enable register
IIR	EQU	2			;Interrupt identification register
LCR	EQU	3			;Line control register
MCR	EQU	4			;Modem control register
LSR	EQU	5			;Line status register
MSR	EQU	6			;Modem status register
SCR	EQU	7			;Scratch register

;	Error return #'s (C_Error) :

NoError		EQU	0		;No error
InvalidPort	EQU	1		;Invalid port # passed
PortNotFound	EQU	2		;Port hardware not present
PortNotOpen	EQU	3		;Port was not opened
ReceiveEmpty	EQU	10		;No character to recieve (buffer empty)
TransmitFull	EQU	11		;Cannot send character (buffer full)

DATA	ENDS
%NEWPAGE
%SUBTTL	"Interrupt service routines"
;********************************************************
;*							*
;*   ASYNC Interrupt Service Routine (ISR) Main Entry	*
;*							*
;********************************************************

CODE	SEGMENT	WORD PUBLIC
	ASSUME	CS:CODE,DS:DATA

;	Declare global procedures & functions

	PUBLIC	ComISR			;Communications/UART management ISR
	PUBLIC	ComWriteCh		;PROCEDURE: Send a single character
	PUBLIC	ComBlockWrite		;PROCEDURE: Send multiple characters
	PUBLIC	ComReadCh		;FUNCTION:  Read a single character
	PUBLIC	ComBlockRead		;PROCEDURE: Read multiple characters

;********************************************************
;*							*
;* ISR:		ComISR					*
;* Process communications interrupt			*
;*							*
;* Checks each active port for activity & sets up	*
;* registers for the various port service routines.	*
;*							*
;********************************************************

ComISR	PROC	FAR

	PUSH	AX			;Save environment
	PUSH	BX			;
	PUSH	CX			;
	PUSH	DX			;
	PUSH	SI			;
	PUSH	DI			;
	PUSH	DS			;
	PUSH	ES			;

	MOV	AX,SEG DATA		;AX <- Current data segment
	MOV	DS,AX			;Set data segment to local area

;	Scan ports for activity

ChkP:	XOR	BX,BX			;Start with first port

ChkP1:	CMP	C_PortExist[BX],0	;Is port hardware present ?
	JZ	ChkP2			; No, ignore

	SHL	BL,1			;BX <- Word array index
	MOV	DX,C_PortAddr[BX]	;DX <- Base address of port
	SHR	BL,1			;Restore byte array index
	ADD	DL,IIR			;DX <- Address of IIR
	IN	AL,DX			;AL <- Interrupt ID register
	TEST	AL,00000001b		;Active interrupt on this port ?
	JNZ	ChkP2			; No, check next port

	CMP	C_PortOpen[BX],0	;Port open ?
	JZ	ChkP2			; No, check next port
	SUB	DL,IIR			;Point DX back to port base

;	Active interrupt found
;	Compute address to ISR & execute
;	On entry to ISR, registers are defined as follows:
;	AX......ISR address (free for transient operations)
;	BX......Byte array index
;	CL......Status register for port
;	CH......Control register for port
;	DX......Port base address

	MOV	[PortNbr],BL		;Save port #/byte array index
	MOV	CL,C_Status[BX]		;CL <- Status register
	MOV	CH,C_Ctrl[BX]		;CH <- Control register
	XOR	AH,AH			;AX <- UART interrupt type * 2
	MOV	SI,AX			;To SI for indexing
	MOV	AX,CS:WORD PTR ISRAdr[SI] ;AX <- ISR address
	CALL	AX			;Execute ISR
	JMP	SHORT ChkP		;Resume scan

ChkP2:	INC	BX			;Bump port #
	CMP	BL,[C_MaxCom]		;Last port ?
	JB	ChkP1			; No, continue scan

;	Interrupt processing complete, exit

Exit:	POP	ES			;Restore environment
	POP	DS			;
	POP	DI			;
	POP	SI			;
	POP	DX			;
	POP	CX			;
	POP	BX			;
	STI				;Enable interrupts
	MOV	AL,20h			;Acknowledge this interrupt
	OUT	20h,AL			;
	POP	AX			;Recover last used register
	IRET				;Exit

;	ISR address table

ISRAdr:	DW	ComMSC,ComXmt,ComRcv,ComLSC

;	Copyright notice - DO NOT REMOVE

Copyrt:	DB	'Copyright (C) 1989-1993, Rising Edge Data Services',0

ComISR	ENDP

;********************************************************
;*							*
;* ISR:		ComMSC					*
;* Modem Status Change ISR				*
;*							*
;* Determine if RS-232 handshake lines have changed	*
;* such that transmit interrupt enable status should	*
;* be changed.						*
;*							*
;********************************************************

ComMSC	PROC	NEAR

	ADD	DL,MSR			;Point to modem status register
	IN	AL,DX			;AL <- Modem status register
	TEST	CH,00000001b		;Transmit hardware handshake enabled ?
	JNZ	MSC1			; Yes, continue
	RET				; No, exit now

MSC1:	SUB	DL,MSR			;Point back to port base
	MOV	AH,CL			;Save status register
	MOV	CL,4			;CL <- # of shifts
	SHR	AL,CL			;Move input bit status to low nibble
	MOV	CL,AH			;Restore status register
	AND	AL,C_HWCheck[BX]	;Mask off "don't care" bits
	MOV	AH,C_HWMatch[BX]	;AH <- Match byte
	AND	AH,00001111b		;Only low four bits valid
	AND	AH,C_HWCheck[BX]	;Mask any illegal bits from check byte
	CMP	AL,AH			;Status match (positive assert) ?
	JE	MSC2			; Yes, enable transmitter

;	Status mismatch - disable transmitter

	OR	CL,10000000b		;Set hard handshake holdoff
	TEST	CH,11000000b		;Force-transmit XON/XOFF in progress ?
	JNZ	MSC4			; Yes, don't disable transmit
	ADD	DL,IER			;Point to interrupt enable register
	IN	AL,DX			;AL <- Interrupt enable register
	AND	AL,11111101b		;Disable transmitter
	JMP	SHORT MSC3		;Save IER and exit

;	Status match - conditionally enable transmitter

MSC2:	AND	CL,01111111b		;Reset hard handshake holdoff
	TEST	CL,11100100b		;Holdoff or transmit buffer empty?
	JNZ	MSC4			; Yes, don't enable transmit
	ADD	DL,IER			;Point to interrupt enable register
	IN	AL,DX			;AL <- Interrupt enable register
	OR	AL,00000010b		;Enable transmitter
MSC3:	OUT	DX,AL			;Save
MSC4:	MOV	C_Status[BX],CL		;Save status register
	RET				;Exit

ComMSC	ENDP

;********************************************************
;*							*
;* ISR:		ComXmt					*
;* Character Transmit ISR				*
;*							*
;* Remove a character from the transmit buffer and	*
;* place it in the appropriate UART's transmit holding	*
;* register.						*
;*							*
;********************************************************

ComXmt	PROC	NEAR

	ADD	DL,LSR			;Point to line status register
	IN	AL,DX			;AL <- Line status register
	TEST	AL,01100000b		;Transmit holding register empty ?
	JNZ	Xmt0			; Yes, continue
	RET				; No, exit (must be false request)

;	Transmit XON/XOFF if requested

Xmt0:	SUB	DL,LSR			;Point back to base
	TEST	CH,00001000b		;Receive software handshake enabled?
	JZ	Xmt3			; No, continue
	TEST	CH,01000000b		;Transmit XOFF ?
	JZ	Xmt1			; No, check XON
	AND	CH,10011111b		;Reset XOFF transmit and SW HS signal
	MOV	AL,C_XOFF[BX]		;AL <- XOFF character
	JMP	SHORT Xmt2		;Transmit & exit

Xmt1:	TEST	CH,10000000b		;Transmit XON ?
	JZ	Xmt3			; No, continue
	AND	CH,01111111b		;Reset XON transmit flag
	OR	CH,00100000b		;Assert SW handshake pseudo-signal
	MOV	AL,C_XON[BX]		;AL <- XON character

Xmt2:	OUT	DX,AL			;Send character
	MOV	C_Ctrl[BX],CH		;Save control register
XmtX:	RET				;Exit

;	Normal character transmission

Xmt3:	TEST	CL,11100100b		;Holdoff or transmit buffer empty ?
	JNZ	Xmt5			; Yes, disable transmit & exit

;	Bump transmit tail pointer

	SHL	BL,1			;BX <- Word array pointer
	MOV	SI,C_OutTail[BX]	;SI <- Transmit tail pointer
	MOV	DI,C_OutHead[BX]	;DI <- Transmit head pointer
	INC	SI			;Bump tail pointer
	CMP	SI,C_OutSize[BX]	;Tail >= Buffer size ?
	JB	Xmt4			; No, continue
	XOR	SI,SI			; Yes, reset to 0
Xmt4:	MOV	C_OutTail[BX],SI	;Save tail pointer

;	Send character from buffer

	SHL	BL,1			;BX <- Longword array pointer
	LES	BX,[C_OutBufPtr+BX]	;ES:BX <- Pointer to buffer base
	MOV	AL,ES:[BX+SI]		;AL <- Character from buffer
	OUT	DX,AL			;Send character
	MOV	BL,[PortNbr]		;Recover byte index

;	Check for empty buffer

	AND	CL,11110111b		;Reset buffer full flag
	MOV	C_Status[BX],CL		;Save status register
	CMP	SI,DI			;Buffer empty (Tail = Head) ?
	JNE	XmtX			; No, exit
	OR	CL,00000100b		;Set transmit buffer empty flag
	MOV	C_Status[BX],CL		;Save status register

;	Disable transmitter & exit

Xmt5:	ADD	DL,IER			;Point to interrupt enable register
	IN	AL,DX			;AL <- Interrupt enable register
	AND	AL,11111101b		;Disable transmit interrupt
	OUT	DX,AL			;Save IER
	RET				;Exit

ComXmt	ENDP

;********************************************************
;*							*
;* ISR:		ComRcv					*
;* Character Reception ISR				*
;*							*
;* Remove character from the UART's receive holding	*
;* register and place it in the receive buffer.		*
;*							*
;********************************************************

ComRcv	PROC	NEAR

	IN	AL,DX			;AL <- Character from UART

;	Check for software handshake characters

	TEST	CH,00000010b		;Transmit software handshake enabled?
	JZ	Rcv3			; No, process character normally
	CMP	AL,C_XON[BX]		;Character = XON ?
	JE	Rcv2			; Yes, enable transmitter
	CMP	AL,C_XOFF[BX]		;Character = XOFF ?
	JNE	Rcv3			; No, process character normally

;	XOFF received
;	Disable transmitter

	OR	CL,01000000b		;Set software handshake holdoff flag
	TEST	CH,00001000b		;Receive soft handshake enabled ?
	JZ	Rcv1			; No, disable transmitter
	TEST	CH,11000000b		;XON/XOFF transmission request ?
	JNZ	RcvX			; Yes, don't disable transmitter
Rcv1:	ADD	DL,IER			;Point to interrupt enable register
	IN	AL,DX			;AL <- Interrupt enable register
	AND	AL,11111101b		;Disable transmitter
	OUT	DX,AL			;Save IER
RcvX:	MOV	C_Status[BX],CL		;Save status register
	RET				;Exit

;	XON Received
;	Conditionally enable transmitter

Rcv2:	AND	CL,10111111b		;Reset soft handshake holdoff flag
	TEST	CL,11100100b		;Holdoff or transmit buffer empty?
	JNZ	RcvX			; Yes, keep transmitter off
	ADD	DL,IER			;Point to interrupt enable register
	IN	AL,DX			;AL <- Interrupt enable register
	OR	AL,00000010b		;Enable transmitter
	OUT	DX,AL			;Save IER
	JMP	SHORT RcvX		;Save status register & exit

;	Normal character reception
;	Check for buffer overflow

Rcv3:	TEST	CL,00000010b		;Buffer full ?
	JZ	Rcv4			; No, continue
	RET				; Yes, exit (toss away character)

;	Bump buffer pointer

Rcv4:	SHL	BL,1			;BX <- Word array pointer
	MOV	SI,C_InTail[BX]		;SI <- Tail pointer
	MOV	DI,C_InHead[BX]		;DI <- Head pointer
	INC	DI			;Bump head pointer
	CMP	DI,C_InSize[BX]		;Head >= Buffer size ?
	JB	Rcv5			; No, continue
	XOR	DI,DI			;Reset head to 0
Rcv5:	MOV	C_InHead[BX],DI		;Save head pointer

;	Place received character in buffer

	SHL	BL,1			;BX <- Pointer array pointer
	LES	BX,C_InBufPtr[BX]	;ES:BX <- Pointer to receive buffer
	MOV	ES:[BX+DI],AL		;Save character
	MOV	BL,PortNbr		;Restore byte array pointer

;	Check for full buffer

	AND	CL,11111110b		;Reset buffer empty flag
	CMP	SI,DI			;Buffer full (Tail = Head) ?
	JNE	Rcv6			; No, continue
	OR	CL,00000010b		;Set buffer-full flag

;	Handshake unassert test
;	Compute # bytes remaining in buffer

Rcv6:	SHL	BL,1			;BX <- Word array index
	SUB	SI,DI			;SI <- Tail - Head
	JAE	Rcv7			;Continue if no underflow
	ADD	SI,C_InSize[BX]		;SI <- # Bytes remaining
Rcv7:	CMP	SI,C_HSOff[BX]		;Buffer left <= Turnoff point ?
	JA	Rcv9			; No, save status & exit

;	Buffer is nearly full, unassert controls/send XOFF
;	Change RS-232 signals to "unassert" state if hard handshaking enabled

	SHR	BL,1			;BX <- Byte array index
	TEST	CH,00000100b		;Hardware handshake enabled ?
	JZ	Rcv8			; No, check software handshake

	ADD	DL,MCR			;Point to modem control register
	IN	AL,DX			;AL <- Modem control register
	MOV	AH,C_HWModify[BX]	;AH <- Bits to modify in MCR
	AND	AH,00000111b		;Allow mods on bits 0-2 only
	AND	AH,C_HWAssert[BX]	;Mask with assertion mode
	NOT	AH			;Flip bits to form mask
	AND	AL,AH			;Mask bits in MCR that should be off
	MOV	AH,C_HWAssert[BX]	;AH <- Bits to assert when HS true
	NOT	AH			;Flip bits for handshake false
	AND	AH,C_HWModify[BX]	;Mask with modification enable byte
	AND	AH,00000111b		;Allow mods on bits 0-2 only
	OR	AL,AH			;Turn on bits that should be on
	OUT	DX,AL			;Save modified MCR
	SUB	DL,MCR			;Point back to base

;	Send XOFF if software handshake enabled

Rcv8:	TEST	CH,00001000b		;Software handshaking enabled ?
	JZ	Rcv9			; No, save status & exit
	TEST	CH,00100000b		;Soft handshake XOFF already sent ?
	JZ	Rcv9			; Yes, save status & exit
	AND	CH,11011111b		;Reset soft handshake pseudo signal
	OR	CH,01000000b		;Flag transmitter to send XOFF
	MOV	C_Ctrl[BX],CH		;Save control register

	ADD	DL,IER			;Point to interrupt enable register
	IN	AL,DX			;AL <- Interrupt enable register
	OR	AL,00000010b		;Enable transmit interrupt
	OUT	DX,AL			;Save IER

;	Save status register & exit

Rcv9:	MOV	BL,[PortNbr]		;BX <- Byte array index
	MOV	C_Status[BX],CL		;Save status register
	RET				;Exit

ComRcv	ENDP

;********************************************************
;*							*
;* ISR:		ComLSC					*
;* Line Status Change ISR (receive error or BREAK)	*
;*							*
;* Counts recieve errors & received BREAK signals and	*
;* tallies these events in seperate counters.		*
;*							*
;********************************************************

ComLSC	PROC	NEAR

	ADD	DL,LSR			;Point to line status register
	IN	AL,DX			;Read LSR

	TEST	AL,00010000b		;BREAK received ?
	JZ	LSC1			; No, check for errors
	INC	C_RcvBreak[BX]		;Bump # of breaks received

LSC1:	TEST	AL,00001110b		;Framing, parity or overrun ?
	JZ	LSC2			; No, exit
	INC	C_RcvError[BX]		;Bump # of receive errors

LSC2:	RET				;Exit

ComLSC	ENDP
%NEWPAGE
%SUBTTL	"Turbo Pascal Procedures/Functions"
;********************************************************
;*							*
;*	   Turbo Pascal Procedures & Functions		*
;*							*
;********************************************************

;************************************************************************
;*									*
;* Procedure:	ComWriteCh(ComPort:Byte; Chr:Char)			*
;* Place a character in the transmit buffer (send a character)		*
;*									*
;* ComWriteCh will place a character (Chr) in the transmit buffer of	*
;* the port designated by (ComPort).  Actual transmission of characters *
;* to the UART is managed by the ComXmit ISR.				*
;*									*
;* Handling of buffer-full conditions is arbitrated by the C_XmitWait	*
;* flag (independently controllable for each port).  When C_XmitWait is	*
;* TRUE, the procedure will delay transmission until space in the	*
;* transmit buffer is available (determined by the transmit buffer full	*
;* flag in C_Status).  If C_XmitWait is FALSE, no delay takes place; if	*
;* there is no space in the buffer, the character is "tossed away" and	*
;* C_Error is set to a nonzero value.					*
;*									*
;* Possible error returns (in C_Error):					*
;* 0	No error							*
;* 1	Invalid port #							*
;* 2	Port hardware does not exist					*
;* 3	Port not initialzed (opened)					*
;* 11	Transmit buffer full (only possible if C_XmitWait is FALSE)	*
;*									*
;************************************************************************

ComWriteCh	PROC	FAR

ComPort	=	[BP+8]			;Location of port # variable
Chr	=	[BP+6]			;Location of character variable

;	Set up Pascal entry environment

	PUSH	BP			;Save base pointer
	MOV	BP,SP			;BP <- Pointer to parameters

;	Make sure port # is valid

	DEC	BYTE PTR ComPort	;Decrement port # (port #'s start w/1)
	MOV	BL,ComPort		;BX <- Port #
	XOR	BH,BH			;
	CMP	BL,[C_MaxCom]		;Port # < Highest port # ?
	JB	CWrt1			; Yes, continue
	MOV	AH,InvalidPort		;Return error
	JMP	CWrtX			;Exit

CWrt1:	CMP	C_PortExist[BX],0	;Does port hardware exist ?
	JNZ	CWrt2			; Yes, continue
	MOV	AH,PortNotFound		;Return error
	JMP	CWrtX			;Exit

CWrt2:	CMP	C_PortOpen[BX],0	;Has port been opened ?
	JNZ	CWrt3			; Yes, continue
	MOV	AH,PortNotOpen		;Return error
	JMP	CWrtX			;Exit

;	Normal character placement
;	Wait for buffer opening if C_XmitWait is TRUE
;	Return error if C_XmitWait is FALSE and buffer is full

CWrt3:	MOV	AL,C_XmitWait[BX]	;AL <- Wait flag
CWrt4:	TEST	C_Status[BX],00001000b	;Transmit buffer full ?
	JZ	CWrt5			; No, send character
	OR	AL,AL			;Wait flag set ?
	JNZ	CWrt4			; Yes, wait for opening
	MOV	AH,TransmitFull		; No, signal transmit buffer full error
	JMP	SHORT CWrtX		;Exit

;	Disable transmit interrupt during buffer update

CWrt5:	MOV	SI,BX			;SI <- Byte array index
	SHL	BL,1			;BX <- Word array pointer
	MOV	DX,C_PortAddr[BX]	;DX <- Base of port
	ADD	DL,IER			;Point to interrupt enable register
	CLI				;Disable interrupts
	IN	AL,DX			;Get IER
	AND	AL,11111101b		;Mask transmit interrupt
	OUT	DX,AL			;Save IER
	OR	C_Status[SI],00100000b	;Set transmit interrupt mask
	STI				;Enable interrupts
	SUB	DL,IER			;Point back to base

;	Bump transmit head pointer

	MOV	SI,C_OutTail[BX]	;SI <- Tail pointer
	MOV	DI,C_OutHead[BX]	;DI <- Head pointer
	INC	DI			;Bump head pointer
	CMP	DI,C_OutSize[BX]	;Head >= Buffer size ?
	JB	CWrt6			; No, continue
	XOR	DI,DI			; Yes, reset head to 0
CWrt6:	MOV	C_OutHead[BX],DI	;Save head pointer

;	Place character in buffer

	SHL	BL,1			;BX <- Longword array pointer
	LES	BX,C_OutBufPtr[BX]	;ES:BX <- Base of transmit buffer
	MOV	AL,Chr			;AL <- Character passed
	MOV	ES:[BX+DI],AL		;Save character in buffer

;	Check for full buffer

	MOV	BL,ComPort		;Restore index
	XOR	BH,BH			;
	MOV	AH,NoError		;Return null error code
	AND	C_Status[BX],11111011b	;Reset transmit buffer empty flag
	CMP	SI,DI			;Tail = Head (buffer full) ?
	JNE	CWrt7			; No, continue
	OR	C_Status[BX],00001000b	;Set transmit buffer full flag

;	Enable interrupts

CWrt7:	ADD	DL,IER			;Point to interrupt enable register
	CLI				;Disable interrupts
	IN	AL,DX			;AL <- Interrupt enable register
	OR	AL,00000010b		;Enable transmit interrupt
	TEST	C_Ctrl[BX],11000000b	;Force transmit XON/XOFF ?
	JNZ	CWrt8			; Yes, exit w/transmit enabled
	TEST	C_Status[BX],11000100b	;Handshake holdoff or tx buffer empty?
	JZ	CWrt8			; No, exit w/transmit enabled
	AND	AL,11111101b		;Disable transmit interrupt

CWrt8:	OUT	DX,AL			;Save revised IER
	XOR	AH,AH			;The following code clears the IER
	XCHG	AL,AH			; and then resets it to it's correct
	OUT	DX,AL			; value.  Required by some 8250 clones
	XCHG	AL,AH			; to ensure that a transmit interrupt
	OUT	DX,AL			; is generated (also sets C_Error=0).
	AND	C_Status[BX],11011111b	;Unmask transmit interrupt holdoff
	STI				;Enable interrupts

;	Exit

CWrtX:	MOV	[C_Error],AH		;Save error code
	POP	BP			;Restore base pointer
	RETF	4			;Exit (removes two word parameters)

ComWriteCh	ENDP

;************************************************************************
;*									*
;* Procedure:	ComBlockWrite(ComPort:Byte; Buffer:Pointer;		*
;*		Var Count:Word);					*
;* Transfers multiple characters to the receive buffer (write a block)	*
;*									*
;* ComBlockWrite transfers up to (Count) characters from the block of	*
;* memory pointed to by (Buffer) to the designated (ComPort) transmit	*
;* buffer.								*
;*									*
;* The global variable C_XmitWait is interrogated and determines how	*
;* ComBlockWrite operates when the transmit buffer is full.  If		*
;* C_XmitWait is TRUE, ComBlockWrite will wait for buffer space to	*
;* become available, transferring exactly (Count) characters before	*
;* exiting.  If C_XmitWait is FALSE,  ComBlockWrite will return		*
;* immediately upon encoutering a full buffer, passing back the number	*
;* of characters transferred to the transmit buffer in (Count).		*
;*									*
;* Possible error returns (in C_Error):					*
;* 0	No error							*
;* 1	Invalid port #							*
;* 2	Port hardware does not exist					*
;* 3	Port not initialzed (opened)					*
;* 11	Transmit buffer empty (only possible if C_XmitWait is FALSE)	*
;*									*
;************************************************************************

ComBlockWrite	PROC	FAR

ComPort	=	BYTE PTR [BP+14]	;Port #
Buffer	=	DWORD PTR [BP+10]	;Pointer to memory buffer (xmit src)
MaxCnt	=	DWORD PTR [BP+6]	;Pointer to max chars to transmit
Count	=	WORD PTR [BP-2]		;# of characters transmitted

;	Set up Pascal entry environment

	PUSH	BP			;Save base pointer
	MOV	BP,SP			;BP <- Pointer to parameters
	SUB	SP,2			;Allocate space for local vars

	MOV	Count,0			;Reset byte count

;	Make sure port # is valid & opened

	DEC	BYTE PTR ComPort	;Decrement port # (port #'s start w/1)
	MOV	BL,ComPort		;BX <- Port #
	XOR	BH,BH			;
	CMP	BL,[C_MaxCom]		;Port # < Highest port # ?
	JB	CBWr1			; Yes, continue
	MOV	AH,InvalidPort		;Return error
	JMP	SHORT CBWrX		;Exit

CBWr1:	CMP	C_PortExist[BX],0	;Does port hardware exist ?
	JNZ	CBWr2			; Yes, continue
	MOV	AH,PortNotFound		;Return error
	JMP	SHORT CBWrX		;Exit

CBWr2:	CMP	C_PortOpen[BX],0	;Has port been opened ?
	JNZ	CBWr3			; Yes, continue
	MOV	AH,PortNotOpen		;Return error
	JMP	SHORT CBWrX		;Exit

;	Compute IER port address

CBWr3:	SHL	BL,1			;BX <- Word array pointer
	MOV	DX,C_PortAddr[BX]	;DX <- Base of port
	ADD	DL,IER			;Point to interrupt enable reg.
	SHR	BL,1			;BX <- Byte array pointer

;	Main loop - place characters in transmit buffer
;	Check for termination (all chars transmitted)

CBWr4:	MOV	AH,NoError		;Assume no error
	LES	SI,MaxCnt		;ES:SI <- Pointer to # chars to write
	MOV	DI,ES:[SI]		;DI <- # Chars to write
	CMP	Count,DI		;# Chars written >= maximum ?
	JAE	CBWrX			; Yes, exit

;	Check for full buffer

CBWr5:	TEST	C_Status[BX],00001000b	;Transmit buffer full ?
	JZ	CBWr6			; No, transmit another character
	CMP	C_XmitWait[BX],0	;OK to wait ?
	JNZ	CBWr5			; Yes, wait for buffer space
	MOV	AH,TransmitFull		; No, return error

;	Exit

CBWrX:	LES	SI,MaxCnt		;ES:SI <- Pointer to max byte count
	MOV	DI,Count		;DI <- Present byte count
	MOV	ES:[SI],DI		;Save byte count
	MOV	C_Error,AH		;Save error code
	MOV	SP,BP			;Deallocate local space
	POP	BP			;Restore base pointer
	RETF	10			;Exit, removing parameters

;	Turn transmit interrupt off during buffer update

CBWr6:	CLI				;Disable interrupts for a bit
	IN	AL,DX			;AL <- Interrupt enable register
	AND	AL,11111101b		;Mask transmit interrupt
	OUT	DX,AL			;Save IER
	OR	C_Status[BX],00100000b	;Make sure transmit stays disabled
	STI				;Enable interrupts

;	Get character from memory buffer

	MOV	DI,BX			;Save array index
	LES	BX,Buffer		;ES:BX <- Pointer to memory buffer
	MOV	SI,Count		;SI <- Buffer index
	MOV	AL,ES:[BX+SI]		;AL <- Character from buffer
	MOV	BX,DI			;Restore array index

;	Bump transmit buffer head pointer

	SHL	BL,1			;BX <- Word array pointer
	MOV	DI,C_OutHead[BX]	;DI <- Head pointer
	INC	DI			;Bump head pointer
	CMP	DI,C_OutSize[BX]	;Head >= Buffer size ?
	JB	CBWr7			;No, continue
	XOR	DI,DI			;Reset head pointer
CBWr7:	MOV	C_OutHead[BX],DI	;Save head pointer

;	Place character in transmit buffer

	MOV	SI,BX			;Save array index
	SHL	BX,1			;BX <- Longword array pointer
	LES	BX,C_OutBufPtr[BX]	;ES:BX <- Pointer to transmit buffer
	MOV	ES:[BX+DI],AL		;Save character
	MOV	BX,SI			;Restore array index

;	Increment byte count
;	Check for full transmit buffer

	INC	Count			;Bump byte count
	MOV	SI,C_OutTail[BX]	;SI <- Tail pointer
	SHR	BL,1			;BX <- Byte array pointer
	AND	C_Status[BX],11111011b	;Reset buffer empty flag
	CMP	SI,DI			;Tail = Head (buffer full) ?
	JNE	CBWr8			; No, continue
	OR	C_Status[BX],00001000b	;Set buffer full flag

;	Enable transmit interrupt

CBWr8:	CLI				;Disable interrupts
	IN	AL,DX			;Get current IER
	OR	AL,00000010b		;Enable transmit interrupt
	TEST	C_Ctrl[BX],11000000b	;Force transmit XON/XOFF ?
	JNZ	CBWr9			; Yes, keep transmit enabled
	TEST	C_Status[BX],11000100b	;Handshake holdoff or Tx buffer empty ?
	JZ	CBWr9			; No, leave transmit enabled
	AND	AL,11111101b		;Mask (disable) transmit interrupt

CBWr9:	OUT	DX,AL			;Save IER
	XOR	AH,AH			;The following code clears the IER
	XCHG	AL,AH			; and then resets it to it's correct
	OUT	DX,AL			; value.  Required by some 8250 clones
	XCHG	AL,AH			; to ensure that a transmit interrupt
	OUT	DX,AL			; is generated (also sets C_Error=0).
	AND	C_Status[BX],11011111b	;Unmask transmit interrupt holdoff
	STI				;Enable interrupts
	JMP	CBWr4			;Send another character

ComBlockWrite	ENDP

;************************************************************************
;*									*
;* Function:	ComReadCh(ComPort:Byte) : Char				*
;* Get a character from the receive buffer (receive a character)	*
;*									*
;* ComReadCh removes a character (if there are any) from the receive	*
;* buffer and passes the character back to the caller.  It is also	*
;* partially responsible for the management of receive hard and soft	*
;* handshaking (determines assertion mode; unassertion managed by ISR).	*
;* Management of the UART's receiver is handled by the ComRcv ISR.	*
;*									*
;* Handling of buffer-empty conditions is arbitrated by the C_RcvWait	*
;* flag (independently controllable for each port).  When C_RcvWait is	*
;* TRUE, the function will delay execution until at least one character	*
;* is available in the receive buffer.  If C_RcvWait is FALSE, a "null"	*
;* value (ASCII 00h) is returned and the C_Error flag is set to a non-	*
;* zero value.								*
;*									*
;* Possible error returns (in C_Error):					*
;* 0	No error							*
;* 1	Invalid port #							*
;* 2	Port hardware does not exist					*
;* 3	Port not initialzed (opened)					*
;* 10	Receive buffer empty (only possible if C_RcvWait is FALSE)	*
;*									*
;************************************************************************

ComReadCh	PROC	FAR

ComPort	=	[BP+6]			;Location of port #

;	Set up Pascal entry environment

	PUSH	BP			;Save base pointer
	MOV	BP,SP			;BP <- Pointer to parameters

;	Make sure port # is valid & opened

	DEC	BYTE PTR ComPort	;Decrement port # (port #'s start w/1)
	MOV	BL,ComPort		;BX <- Port #
	XOR	BH,BH			;
	CMP	BL,[C_MaxCom]		;Port # < Highest port # ?
	JB	CRd1			; Yes, continue
	MOV	AH,InvalidPort		;Return error
	XOR	AL,AL			;
	JMP	CRdX			;Exit

CRd1:	CMP	C_PortExist[BX],0	;Does port hardware exist ?
	JNZ	CRd2			; Yes, continue
	MOV	AH,PortNotFound		;Return error
	XOR	AL,AL			;
	JMP	CRdX			;Exit

CRd2:	CMP	C_PortOpen[BX],0	;Has port been opened ?
	JNZ	CRd3			; Yes, continue
	MOV	AH,PortNotOpen		;Return error
	XOR	AL,AL			;
	JMP	CRdX			;Exit

;	Normal character read
;	Wait for character if C_RcvWait is TRUE
;	Return error & null char if C_RcvWait is FALSE and buffer is empty

CRd3:	TEST	C_Status[BX],00000001b	;Buffer empty ?
	JZ	CRd4			; No, get character
	CMP	C_RcvWait[BX],0		;Ok to wait ?
	JNZ	CRd3			; Yes, wait for character
	MOV	AH,ReceiveEmpty		; No, signal receive buffer empty error
	XOR	AL,AL			;Return null result
	JMP	CRdX			;Exit

;	Disable receive interrupt during buffer update

CRd4:	SHL	BL,1			;BX <- Word array pointer
	MOV	DX,C_PortAddr[BX]	;DX <- Base of port
	ADD	DL,IER			;Point to interrupt enable register
	CLI				;Disable interrupts during modify
	IN	AL,DX			;AL <- Interrupt enable register
	AND	AL,11111110b		;Mask receive interrupt
	OUT	DX,AL			;Save IER
	STI				;Enable interrupts
	SUB	DL,IER			;Point back to base

;	Bump receive tail pointer

	MOV	SI,C_InTail[BX]		;SI <- Tail pointer
	MOV	DI,C_InHead[BX]		;DI <- Head pointer
	INC	SI			;Bump tail pointer
	CMP	SI,C_InSize[BX]		;Tail >= Buffer size ?
	JB	CRd5			; No, continue
	XOR	SI,SI			;Reset tail to 0
CRd5:	MOV	C_InTail[BX],SI		;Save tail pointer

;	Get character from buffer

	SHL	BL,1			;BX <- Longword array pointer
	LES	BX,C_InBufPtr[BX]	;ES:BX <- Base of receive buffer
	MOV	AL,ES:[BX+SI]		;AL <- Character from buffer
	MOV	AH,NoError		;Zero error code

;	Determine if buffer is empty

	MOV	BL,ComPort		;Restore array index
	XOR	BH,BH			;
	AND	C_Status[BX],11111101b	;Reset buffer full flag
	CMP	SI,DI			;Tail = Head (buffer empty) ?
	JNE	CRd6			; No, continue
	OR	C_Status[BX],00000001b	;Set buffer empty flag

;	Soft/hard handshake assertion test
;	Compute remaining space in buffer

CRd6:	MOV	CX,AX			;Save character
	SHL	BL,1			;BX <- Word array index
	SUB	SI,DI			;SI <- Tail - Head
	JAE	CRd7			;Continue if no underflow
	ADD	SI,C_InSize[BX]		;Add buffer size to result (now +)
CRd7:	MOV	DI,C_HSOn[BX]		;DI <- Assertion/turnon point
	SHR	BL,1			;BX <- Byte array index
	CMP	SI,DI			;Size >= turnon point ?
	JB	CRd9			; No, exit

;	Check & process hardware handshake

	TEST	C_Ctrl[BX],00000100b	;Receive hardware handshake enabled ?
	JZ	CRd8			; No, check software handshake

	ADD	DL,MCR			;Point to modem control register
	IN	AL,DX			;AL <- Modem control register
	MOV	AH,C_HWModify[BX]	;AH <- MCR bits to modify
	AND	AH,00000111b		;Allow setting on bits 2-0 only
	AND	AH,C_HWAssert[BX]	;Mask with assertion mode
	OR	AL,AH			;OR-in bits that should be asserted
	MOV	AH,C_HWAssert[BX]	;AH <- Assertion mode
	NOT	AH			;
	AND	AH,C_HWModify[BX]	;Mask /Assert with modify bits
	NOT	AH			;AH <- 1's on bits not to reset
	OR	AH,11111000b		;Allow resets on bits 2-0 only
	AND	AL,AH			;Reset bits that should be unasserted
	OUT	DX,AL			;Save revised MCR
	SUB	DL,MCR			;Point back to port base

;	Check & process software handshake

CRd8:	TEST	C_Ctrl[BX],00001000b	;Receive software handshake enabled ?
	JZ	CRd9			; No, exit
	TEST	C_Ctrl[BX],00100000b	;XON already sent ?
	JNZ	CRd9			; Yes, exit
	OR	C_Ctrl[BX],10100000b	;Set force-xmit XON, SW pseudo-signal

;	Enable UART interrupts

CRd9:	ADD	DL,IER			;Point to interrupt enable register
	CLI				;Disable interrupts
	IN	AL,DX			;Get current IER
	OR	AL,00000011b		;Enable receive, transmit interrupts
	TEST	C_Ctrl[BX],11000000b	;Force transmit XON/XOFF ?
	JNZ	CRd10			; Yes, exit w/transmit enabled
	TEST	C_Status[BX],11100100b	;Holdoff or transmit buffer empty ?
	JZ	CRd10			; No, exit w/transmit enabled
	AND	AL,11111101b		;Mask transmit interrupt
CRd10:	OUT	DX,AL			;Restore IER
	STI				;Enable interrupts
	MOV	AX,CX			;Recover character & error flag

;	Exit

CRdX:	MOV	[C_Error],AH		;Save error code
	XOR	AH,AH			;Reset high byte of function result
	POP	BP			;Restore base pointer
	RETF	2			;Exit (removes a single word parameter)

ComReadCh	ENDP

;************************************************************************
;*									*
;* Procedure:	ComBlockRead(ComPort:Byte; Buffer:Pointer;		*
;*		Var Count:Word);					*
;* Retrieve multiple characters from the receive buffer (read a block)	*
;*									*
;* ComBlockRead removes up to (Count) characters from the designated	*
;* buffer (ComPort) into the memory area pointed to by (Buffer).  As	*
;* with ComReadCh, handshaking is managed if enabled.			*
;*									*
;* The global variable C_RcvWait is interrogated and determines how	*
;* ComBlockRead operates when the receive buffer is empty.  If		*
;* C_RcvWait is TRUE, ComBlockRead will wait until exactly (Count)	*
;* characters have been received before exiting.  If C_RcvWait is	*
;* FALSE,  ComBlockRead will return immediately upon encoutering an	*
;* empty buffer, passing back the number of characters transferred to	*
;* the destination buffer in (Count).					*
;*									*
;* Possible error returns (in C_Error):					*
;* 0	No error							*
;* 1	Invalid port #							*
;* 2	Port hardware does not exist					*
;* 3	Port not initialzed (opened)					*
;* 10	Receive buffer empty (only possible if C_RcvWait is FALSE)	*
;*									*
;************************************************************************

ComBlockRead	PROC	FAR

ComPort	=	BYTE PTR [BP+14]	;Port #
Buffer	=	DWORD PTR [BP+10]	;Pointer to memory buffer to store to
MaxCnt	=	DWORD PTR [BP+6]	;Pointer to max chars to read/count
Count	=	WORD PTR [BP-2]		;# of characters read

;	Set up Pascal entry environment

	PUSH	BP			;Save base pointer
	MOV	BP,SP			;BP <- Pointer to parameters
	SUB	SP,2			;Allocate space for local vars

	MOV	Count,0			;Reset byte count

;	Make sure port # is valid & opened

	DEC	BYTE PTR ComPort	;Decrement port # (port #'s start w/1)
	MOV	BL,ComPort		;BX <- Port #
	XOR	BH,BH			;
	CMP	BL,[C_MaxCom]		;Port # < Highest port # ?
	JB	CBRd1			; Yes, continue
	MOV	AH,InvalidPort		;Return error
	JMP	SHORT CBRdX		;Exit

CBRd1:	CMP	C_PortExist[BX],0	;Does port hardware exist ?
	JNZ	CBRd2			; Yes, continue
	MOV	AH,PortNotFound		;Return error
	JMP	SHORT CBRdX		;Exit

CBRd2:	CMP	C_PortOpen[BX],0	;Has port been opened ?
	JNZ	CBRd3			; Yes, continue
	MOV	AH,PortNotOpen		;Return error
	JMP	SHORT CBRdX		;Exit

;	Compute IER port address

CBRd3:	SHL	BL,1			;BX <- Word array pointer
	MOV	DX,C_PortAddr[BX]	;DX <- Base of port
	ADD	DL,IER			;Point to interrupt enable reg.
	SHR	BL,1			;BX <- Byte array pointer

;	Main loop - read characters from buffer
;	Check for termination (all chars read)

CBRd4:	MOV	AH,NoError		;Assume no error
	LES	SI,MaxCnt		;ES:SI <- Pointer to # chars to read
	MOV	DI,ES:[SI]		;DI <- # Chars to read
	CMP	Count,DI		;# chars read >= maximum ?
	JAE	CBRdX			; Yes, exit

;	Check for empty buffer

CBRd5:	TEST	C_Status[BX],00000001b	;Receive buffer empty ?
	JZ	CBRd6			; No, read character
	CMP	C_RcvWait[BX],0		;OK to wait ?
	JNZ	CBRd5			; Yes, wait for character
	MOV	AH,ReceiveEmpty		; No, return error

;	Exit

CBRdX:	LES	SI,MaxCnt		;ES:SI <- Pointer to byte count
	MOV	DI,Count		;DI <- Present byte count
	MOV	ES:[SI],DI		;Save byte count
	MOV	C_Error,AH		;Save error code
	MOV	SP,BP			;Deallocate local space
	POP	BP			;Restore base pointer
	RETF	10			;Exit, removing parameters

;	Disable receive interrupt during buffer update

CBRd6:	CLI				;Disable interrupts for a bit
	IN	AL,DX			;AL <- Interrupt enable register
	AND	AL,11111110b		;Mask receive interrupt
	OUT	DX,AL			;Save IER
	STI				;Enable interrupts

;	Bump buffer pointer

	SHL	BL,1			;BX <- Word array pointer
	MOV	SI,C_InTail[BX]		;SI <- Tail pointer
	INC	SI			;Bump tail pointer
	CMP	SI,C_InSize[BX]		;Tail >= Buffer size ?
	JB	CBRd7			;No, continue
	XOR	SI,SI			;Reset tail pointer
CBRd7:	MOV	C_InTail[BX],SI		;Save tail pointer

;	Get character from recieve buffer
;	Transfer to memory @ (Buffer)

	SHL	BX,1			;BX <- Longword array pointer
	LES	BX,C_InBufPtr[BX]	;ES:BX <- Pointer to receive buffer
	MOV	AL,ES:[BX+SI]		;AL <- Character from buffer
	LES	BX,Buffer		;ES:BX <- Pointer to memory buffer
	MOV	DI,Count		;DI <- Memory buffer index
	MOV	ES:[BX+DI],AL		;Save character

;	Increment byte count
;	Check for empty receive buffer

	INC	Count			;Bump byte count
	MOV	BL,ComPort		;Restore array index
	XOR	BH,BH			;
	SHL	BL,1			;BX <- Word array pointer
	MOV	DI,C_InHead[BX]		;DI <- Head pointer
	SHR	BL,1			;BX <- Byte array pointer
	AND	C_Status[BX],11111101b	;Reset buffer full flag
	CMP	SI,DI			;Tail = Head (buffer empty) ?
	JNE	CBRd8			; No, continue
	OR	C_Status[BX],00000001b	;Set buffer empty flag

;	Compute remaining space in buffer

CBRd8:	SHL	BL,1			;BX <- Word array index
	SUB	SI,DI			;SI <- Tail - Head
	JAE	CBRd9			;Continue if no underflow
	ADD	SI,C_InSize[BX]		;Add buffer size to result (now +)
CBRd9:	MOV	DI,C_HSOn[BX]		;DI <- Assertion/turnon point
	SHR	BL,1			;BX <- Byte array index
	CMP	SI,DI			;Size >= turnon point ?
	JB	CBRd11			; No, skip handshake control

;	Assert hardware control lines if enabled

	TEST	C_Ctrl[BX],00000100b	;Receive hardware handshake enabled ?
	JZ	CBRd10			; No, check software handshake

	ADD	DL,MCR-IER		;Point to modem control register
	IN	AL,DX			;AL <- Modem control register
	MOV	AH,C_HWModify[BX]	;AH <- MCR bits to modify
	AND	AH,00000111b		;Allow setting on bits 2-0 only
	AND	AH,C_HWAssert[BX]	;Mask with assertion mode
	OR	AL,AH			;OR-in bits that should be asserted
	MOV	AH,C_HWAssert[BX]	;AH <- Assertion mode
	NOT	AH			;
	AND	AH,C_HWModify[BX]	;Mask /Assert with modify bits
	NOT	AH			;AH <- 1's on bits not to reset
	OR	AH,11111000b		;Allow resets on bits 2-0 only
	AND	AL,AH			;Reset bits that should be unasserted
	OUT	DX,AL			;Save revised MCR
	SUB	DL,MCR-IER		;Point back to IER

;	Send XON if software handshaking enabled

CBRd10:	TEST	C_Ctrl[BX],00001000b	;Receive software handshake enabled ?
	JZ	CBRd11			; No, continue
	TEST	C_Ctrl[BX],00100000b	;XON already sent ?
	JNZ	CBRd11			; Yes, don't send again
	OR	C_Ctrl[BX],10100000b	;Set force-xmit XON, SW pseudo-signal

;	Enable UART interrupts (allows anything pending to be serviced)

CBRd11:	CLI				;Disable interrupts
	IN	AL,DX			;Get current IER
	OR	AL,00000011b		;Enable receive, transmit interrupts
	TEST	C_Ctrl[BX],11000000b	;Force transmit XON/XOFF ?
	JNZ	CBRd12			; Yes, leave transmit enabled
	TEST	C_Status[BX],11100100b	;Holdoff or transmit buffer empty ?
	JZ	CBRd12			; No, leave transmit enabled
	AND	AL,11111101b		;Mask (disable) transmit interrupt
CBRd12:	OUT	DX,AL			;Save IER
	STI				;Enable interrupts

	JMP	CBRd4			;Get another character

ComBlockRead	ENDP

CODE	ENDS
	END
