Attribute VB_Name = "Module1"
Option Explicit
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' MODULE1.BAS
'
' What it's for:
'   1. Declarations of general-purpose constants,
'      global variables, and calls to the Windows API.
'      In alphabetical order, with all API calls at bottom.
'   2. Utility, helper, and form-independent routines
'   3. Localization--all string constants are defined here
'      symbolic values for ease of translation. Since VB3
'      doesn't support resources simple constant
'      declarations are used.
'
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' When the user clicks Add to add a file to the archive, the Add File
' dialog comes up with the filename to be added. By default the
' pathname of the file is not included, just the file and extension.
' However, there's a checkbox that lets the user toggle which is used.
' The full pathname of the file is stored here.
Global gAddFileFullPathname As String

' The filename & extension of the file to be added.
' See comments for gAddFileFullPathname above for details.
Global gAddFileOnlyFilename As String

' For VSFlex
Global Const ALIGN_RIGHT = 7

' Center-align a grid column.
Global Const ALIGN_CENTER = 4

' Buttons on a toolbar are treated as indices into the toolbar's
' Buttons collection. Use these constants to refer to them for
' maintenance. These are listed in order of their appearance on the
' toolbar.
Global Const BTN_OPEN = 1
Global Const BTN_NEW = 2
Global Const BTN_QUIT = 3
Global Const BTN_SEP1 = 4
Global Const BTN_ADD = 5
Global Const BTN_UNZIP = 6
Global Const BTN_DELETE = 7
Global Const BTN_SEP2 = 8
Global Const BTN_HELP = 9


' Global cancel-this-operation indicator.
' Set to nonzero when operation should be canceled
' (primarily when user presses Esc during unzip or zip).
Global gCancel As Integer

' Grid column filename appears in.
Global Const COL_FILENAME = 0

' Grid column date & time appear in.
Global Const COL_DATE_AND_TIME = 1

' Grid column uncompressed file size appears in.
Global Const COL_UNCOMPRESSED_SIZE = 2

' Grid column compressed file size appears in.
Global Const COL_COMPRESSED_SIZE = 3

' Grid column ratio (percent reduction) appears in.
Global Const COL_PERCENT_REDUCED = 4

' Grid column file type (attribute) appears in.
Global Const COL_ATTRIBUTE = 5

' Column titles.
' In order of appearance on grid.
Global Const STR_COL_TITLE_FILENAME = "Filename"
Global Const STR_COL_TITLE_DATE_AND_TIME = "Date & Time"
Global Const STR_COL_TITLE_UNCOMPRESSED_SIZE = "Uncompressed"
Global Const STR_COL_TITLE_COMPRESSED_SIZE = "Compressed"
Global Const STR_COL_TITLE_PERCENT_REDUCED = "Reduced"

' Default column width for the "uncompressed size" column.
' Should be made sticky through an Options dialog setting.
' Currently not in use.
'Global gCOL_UNCOMPRESSED_SIZE As Long

' Default column width for the "compressed size" column.
' Should be made sticky through an Options dialog setting.
' Currently not in use.
'Global gCOL_COMPRESSED_SIZE As Long



Global vbCRLF As String

' Wait cursor.
Global Const CURSOR_HOURGLASS = 11

' Put up help dialog (a CMDIALOG.VBX Action constant)
' Taken from VB's CONSTANT.TXT.
Global Const DLG_HELP = 6

' True when a .Zip archive has just been created but contains no files.
Global gEmptyArchive As Integer

' TRUE when a .Zip archive has been opened.
Global gFileOpen As Integer

' TRUE when the help file can be found, FALSE when not.
Global gHelpFilePresent As Integer

' Help dialog constant: Put up help index (a CMDIALOG.VBX constant).
' Taken from VB's CONSTANT.TXT.
Global Const HELP_INDEX = &H3

' Help dialog constant: Search based on a partial text string (a CMDIALOG.VBX constant).
' Taken from VB's CONSTANT.TXT.
Global Const HELP_PARTIALKEY = &H105

' Virtual key codes from VB's CONSTANT.TXT file.
Global Const KEY_DOWN = &H28
Global Const KEY_ESCAPE = &H1B
Global Const KEY_RETURN = &HD
Global Const KEY_UP = &H26

' Major and minor version identifiers.
Global Const MAJOR_VERSION = "1"
Global Const MINOR_VERSION = "0"

' Maximum length of a fully qualified filename, with path & extension.
Global Const MAX_FILENAME_LEN = 255

' Maximum length of a string stored in a profile.
Global Const MAX_PROFILE_STRING_LEN = 255

' Maximum # of digits used to represent the size of a file.
' Figured like this: 99,999,999,999
Global Const MAX_SIZE_CHARS = 14

' Means the Zip or Unzip control Action property's action is to do nothing.
Global Const NO_ACTION = 0

' Title of program, as opposed to executable name.
Global Const PROGRAM_NAME = "FreeZip32"

' Flags that right mouse button has been clicked.
Global Const RIGHT_BUTTON = 2

' VSFlex Sort property can be set to the following values:
' Guesses whether text is string or number.
Global Const SORT_GENERIC_ASCENDING = 1
Global Const SORT_GENERIC_DESCENDING = 2
' Converts strings to numbers
Global Const SORT_NUMERIC_ASCENDING = 3
Global Const SORT_NUMERIC_DESCENDING = 4
' Case-insensitive string comparison
Global Const SORT_STRING_ASCENDING_CASE_INSENSITIVE = 5
Global Const SORT_STRING_DESCENDING_CASE_INSENSITIVE = 6
' Case-sensitive string comparison
Global Const SORT_STRING_ASCENDING_CASE_SENSITIVE = 7
Global Const SORT_STRING_DESCENDING_CASE_SENSITIVE = 8

' # of seconds splash screen stays visible.
Global gSplashMax As Integer

' File attributes.
' See AttributeToText() routine.
Global Const STR_FILEATTR_NORMAL = "NORMAL"
Global Const STR_FILEATTR_READ_ONLY = "Read-only"
Global Const STR_FILEATTR_HIDDEN = "Hidden"
Global Const STR_FILEATTR_SYSTEM = "System"
Global Const STR_FILEATTR_VOLUME_NAME = "Volume name"
Global Const STR_FILEATTR_DIRECTORY = "Directory"
Global Const STR_FILEATTR_ZIP_FILE = ".ZIP file"
Global Const STR_FILEATTR_ENCRYPTED = "Encrypted"
Global Const STR_FILEATTR_UNKNOWN = "Unknown"

' General-purpose strings here for localization
Global Const STR_FILES = "files"
Global Const STR_FILE = "file"
Global Const STR_IN_ARCHIVE = " in archive"
Global Const STR_FILE_SELECTED = "file selected"
Global Const STR_FILES_SELECTED = "files selected"

' Profile information: "Directories" heading, which stores
' default directories for location of files to unzip to, etc.
Global Const STR_PROFILE_DIRECTORIES = "Directories"
' "Options" heading, which stores miscellaneous options, such
' as whether to show the splash screen.
Global Const STR_PROFILE_OPTIONS = "Options"
' Determines whether splash screen appears on startup.
' Default: Yes.
Global Const STR_PROFILE_SHOW_SPLASH = "Show Splash Screen"
' Default location to which files should be unzipped.
Global Const STR_PROFILE_UNZIP_TO = "Unzip Directory"
' Default location to look for .ZIP files.
Global Const STR_PROFILE_ZIP_FILES = "Zip Directory"

' Special indicator to unzip all files.
Global Const UNZIP_ALL_FILES = -1

' Default extension for .ZIP files.
Global Const ZIP_EXTENSION = "ZIP"

' Required to update DynaStat status control.
Declare Sub UpdateWindow Lib "USER" (ByVal h1 As Integer)

' For reading and writing .INI files
Declare Function GetPrivateProfileString Lib "Kernel" (ByVal lpAppName As String, ByVal lpKeyName As String, ByVal lpDefault As String, ByVal lpReturnedString$, ByVal nSize As Integer, ByVal lpFileName As String) As Integer
Declare Function WritePrivateProfileString Lib "Kernel" (ByVal lpAppName As String, ByVal lpKeyName As String, ByVal lpString As String, ByVal lpFileName As String) As Integer
' VB3 version:
' Declare Sub ShellAbout Lib "shell.dll" (ByVal hWndOwner As Integer, ByVal lpszAppName As String, ByVal lpszMoreInfo As String, ByVal hIcon As Integer)
' VB4 version:
Declare Function ShellAbout Lib "shell32.dll" Alias "ShellAboutA" (ByVal hwnd As Long, ByVal szApp As String, ByVal szOtherStuff As String, ByVal hIcon As Long) As Long

Sub SelectRow()
' What it does:
'   Ensures that focus is on grid, and that selection mode
'   is always a full row at a time.
'
   
    With Form1!Grid1
        ' Make sure grid's highlight is visible.
        .HighLight = 1
        ' Select by rows.
        .SelectionMode = 1
        ' OCX version doesn't seem to perform the
        ' SelectionMode line above when moving highlight
        ' using keyboard.
        .col = 0: .ColSel = .Cols - 1
    End With

End Sub


Function AttributeToText(FileAttribute As Long) As String
' What it does:
'   Converts the supplied archive file attribute to a
'   textual description, such as "Normal", "Hidden",
'   or ".ZIP file"
'
' Parameters:
'   FileAttribute: Indicator defined by .ZIP file format that
'   describes attributes for a file stored inside an archive.
'
' Returns:
'   On success:
'   The attribute as a string, such as "Normal" or "Hidden".
'
'   On failure:
'   If the attribute is unknown, the string "Unknown"
'
' Notes:
'   These are internal PKZIP-defined designations.
'
    Select Case FileAttribute
        Case &H0
            AttributeToText = STR_FILEATTR_NORMAL
        Case &H1
            AttributeToText = STR_FILEATTR_READ_ONLY
        Case &H2
            AttributeToText = STR_FILEATTR_HIDDEN
        Case &H4
            AttributeToText = STR_FILEATTR_SYSTEM
        Case &H8
            AttributeToText = STR_FILEATTR_VOLUME_NAME
        Case &H10
            AttributeToText = STR_FILEATTR_DIRECTORY
        Case &H20
            AttributeToText = STR_FILEATTR_ZIP_FILE
        Case &H8000
            AttributeToText = STR_FILEATTR_ENCRYPTED
        Case Else
            AttributeToText = STR_FILEATTR_UNKNOWN
    End Select
End Function

Sub CenterForm(Form As Form)
' What it does:
'   Lightweight utility routine to center a form
'   relative to screen coordinates.
'
' Parameters:
'   Form: No duh. The form to center.
'
'
    ' Works only on normally visible forms, not minimized
    ' or maximized.
    If Form.WindowState <> 0 Then
        Exit Sub
    End If

    ' Center the form on the screen, using faster integer divides.
    Form.Move (Screen.Width \ 2 - Form.Width \ 2), (Screen.Height \ 2 - Form.Height \ 2)
End Sub

Sub ClearGrid(Grid As Control, ClearFixed As Integer)
' What it does:
'   Empties a grid, chops it down either to a the fixed
'   portion (if specified) or to no rows and columns at
'   all (if grid has no fixed portion), and selects
'   the first row (if the grid has a fixed portion).
'
' Parameters:
'   Grid: The FlexArray object to clear.
'   ClearFixed: If zero, any fixed rows and columns
'   are left untouched. If nonzero, clears any fixed portion.
'
' Notes:
'   For VideoSoft's VSFlex control.
'   Haven't tested with fixed columns, only fixed rows.
'   Some of the old MS Grid code is left, but commented out.

    ' Preserves the grid's fill style while it's modified in this routine.
    Dim CurrFillStyle As Integer

    ' Clear the fixed portion if indicated.
    If (ClearFixed) Then
        Grid.col = 0
        Grid.row = 0
    Else
        Grid.col = Grid.FixedCols
        Grid.row = Grid.FixedRows
    End If

    ' Select the entire grid.
    Grid.ColSel = Grid.Cols - 1
    Grid.RowSel = Grid.Rows - 1
    CurrFillStyle = Grid.FillStyle

    ' FillStyle of 1 means that any text operation will now affect
    ' all selected cells.
    Grid.FillStyle = 1

    ' Set all grid text to empty.
    Grid.Text = ""

    ' Reselect the first nonfixed cell.
    Grid.ColSel = Grid.FixedCols
    Grid.RowSel = Grid.FixedRows

    ' Restore saved fill style.
    Grid.FillStyle = CurrFillStyle
End Sub

Sub ClearStatus(StatusBar As Control)
' What it does:
'   Clears text and zeroes out the status bar used for
'   overall file operations.
'
' Parameters:
'   StatusBar: A DynaStat object.
'
' Notes:
'   Not a general-purpose routine in that it's written for
'   the DynaStat control from Inner Media

    ' Unfill the status bar.
    StatusBar.StatusPercent = 0

    ' Clear any filename from the status bar.
    StatusBar.StatusText = ""

    ' Force an immediate update using Windows API.
    ' This code isn't necessary in Win32
    ' Call UpdateWindow(StatusBar.hWnd)

End Sub

Function DirExists(Pathname As String) As Integer
' What it does:
'   Determines whether the given pathname is a directory
'   on the disk.
'
' Parameters:
'   Pathname: String containing the directory name.
'
' Returns:
'   On success:
'   Nonzero if Pathname is a real directory on the disk.
'
'   On failure:
'   0 if Pathname is an invalid pathname or not a directory.
'
' See also:
'   FileExists()
'
' Notes:
'   Actually haven't tested properly for invalid directory names.
'
    ' The pathname is altered by this routine, so make a copy.
    Dim Path As String
    Path = Pathname

    ' Check for invalid pathnames (pathetic test so far).
    If Path = "" Then
        ' Exit with failure flagged.
        DirExists = 0
    End If

    ' Make sure the last char of the path specifier isn't a backslash.
    ' Dir$() doesn't like it.
    If (Path <> "") And (Right(Path, 1) = "\") Then
        Path = Left(Path, Len(Path) - 1)
    End If
    
    If (Dir$(Path, 16) = "") Or (Path = "") Then
        DirExists = False  ' Returns FALSE if file does not
    Else                    ' exist, TRUE if file does
        DirExists = True   ' exist.
    End If
    
End Function

Sub ShowNumberOfFilesInArchive(NumFiles As Integer)
' What it does:
'   Displays a custom message in the status bar showing how many
'   files are in the archive. Gets the 1 case right.
    Dim MsgText
    MsgText = IIf(NumFiles <> 1, STR_FILES, STR_FILE)
    MsgText = NumFiles & " " & MsgText & STR_IN_ARCHIVE
    MsgStatus MsgText

End Sub
Sub DisplayFilesInArchive(Filename As String)
' What it does:
'   Obtains information on each file in ZIP archive
'   and displays this info, one file per line, in the main grid.
'
' Parameters:
'   Filename: Name of Zip archive whose contents are to be displayed.
'
' See also:
'   DisplayNextFile()

    ' Preserves cursor while hourglass is displayed.
    Dim Cursor As Integer

    ' Loop counter
    Dim EachItem As Integer

    ' File size, formatted with commas.
    Dim FileSize As String

    ' Maximum length of a filename (in the archive) in characters.
    Dim MaxFilenameLen As Integer

    ' # of files in archive.
    Dim NumFiles As Integer

    ' Reset the status indicators.
    ClearStatus Form1!MajorStatus
    ClearStatus Form1!MinorStatus

    ' Zip file from which to obtain file info.
    Form1!Unzip.ZIPFile = Filename

    ' Create grid titles.
    SetupUnzipGrid Form1, Form1!Grid1

    ' Preserve cursor.
    Cursor = Screen.MousePointer

    ' Display a wait cursor during what may be a lengthy operation.
    Screen.MousePointer = CURSOR_HOURGLASS

    ' Make the zip/unzip progress status bars visible.
    ' ShowStatusIndicators

    ' Obtain # of items in zip file.
    Form1!Unzip.ActionDZ = UNZIP_COUNTALLZIPMEMBERS

    ' If (Unzip.ErrorCode <> ZE_OK) Or (Unzip.ReturnCount < 1) Then
    If Form1!Unzip.ErrorCode <> ZE_OK Then
        ' Restore cursor.
        Screen.MousePointer = Cursor
        Exit Sub
    End If

    ' Display filename above grid.
    Form1!Label1 = Filename
    Form1!Label1.Visible = True

    ' Note that a file has been opened.
    gFileOpen = True

    ' Make only fixed row portion and one normal row visible.
    Form1!Grid1.Rows = 2
    Form1!Grid1.row = 1

    ' Empty the grid. 0 means don't clear the fixed row portion.
    ClearGrid Form1!Grid1, 0

    ' Show grid.
    Form1!Grid1.Visible = True

    ' Stop updates so grid doesn't flicker.
    Form1!Grid1.Redraw = False

    ' Preserve # of files.
    NumFiles = Form1!Unzip.ReturnCount

    If NumFiles <= 0 Then
        ' Note that the archive is empty.
        gEmptyArchive = True
    Else
        ' Note that the archive has at least 1 file.
        gEmptyArchive = False
    End If

    For EachItem = 1 To NumFiles

        ' Retrieve next file from archive.
        Form1!Unzip.ActionDZ = UNZIP_GETNEXTZIPINFO
        If Len(Form1!Unzip.zi_FileName) > MaxFilenameLen Then
            MaxFilenameLen = Len(Form1!Unzip.zi_FileName)
        End If


        ' Think of the info for each file as a row (record) in a database table.
        ' Fill each column in this row.
        If EachItem < NumFiles Then
            Form1!Grid1.Rows = Form1!Grid1.Rows + 1
            Form1!Grid1.row = Form1!Grid1.Rows - 2
        Else
            Form1!Grid1.row = Form1!Grid1.Rows - 1
        End If

        DisplayNextFile Form1!Grid1, Form1!Unzip.zi_FileName, Form1!Unzip.zi_DateTime, Form1!Unzip.zi_oSize, Form1!Unzip.zi_cSize
        
    Next EachItem

    ' Resume updates.
    Form1!Grid1.Redraw = True

    HideStatusIndicators

    ' Restore cursor.
    Screen.MousePointer = Cursor

    ' Now that file is open, make appropriate controls available.
    UpdateUserInterface

    ' Inform user how many files are in archive.
    ShowNumberOfFilesInArchive NumFiles

    ' Give the grid the focus so the keyboard is usable immediately.
    ' Form1!Grid1.SetFocus
    Form1!Grid1.row = 1
    
    ' Select by rows.
    Form1!Grid1.SelectionMode = 1

    ' Make sure the grid is selected so the keyboard works.
    Form1!Grid1.SetFocus
End Sub

Sub DisplayNextFile(Grid As Control, ByVal Filename As String, ByVal DateTime As String, ByVal Size As Long, ByVal cSize As Long)
' What it does:
'   The main view of the Zip file is a grid (VideoSoft's vsFlex grid).
'   When a Zip file is opened, info on each file is displayed, one
'   file per row. This displays 1 row of information, for 1 file.
'
' Parameters:
'   Grid: The vsFlex grid used to display filenames.
'   Filename: Name of the file within the archive.
'   DateTime: File's "last-changed" system date and time as a string.
'   Size: The file's size when uncompressed.
'   cSize: The file's size as compressed within the archive.
'
' See also:
'   DisplayFilesInArchive()
'
    ' Used to hold massaged filename (swapped forward slashes with backward slashes).
    Dim NewFilename As String

    ' Grid column filename appears in.
    Grid.col = COL_FILENAME
    ' Filenames come in with directory separators UNIX style, as
    ' "/" instead of "/". So swap 'em using ReplaceChars.
    NewFilename = Filename
    ReplaceChars NewFilename, "/", "\"
    Grid.Text = NewFilename
    
    ' Date & time in 24-hour format: mm/dd/yy hh:mm
    Grid.col = COL_DATE_AND_TIME
    Grid.Text = DateTime
    
    ' Grid column uncompressed file size appears in.
    Grid.col = COL_UNCOMPRESSED_SIZE
    If Size > 0 Then
        Grid.Text = Format(Size, "##,###,###")
    Else
        Grid.Text = "0"
    End If
    
    ' Grid column compressed file size appears in.
    Grid.col = COL_COMPRESSED_SIZE
    If cSize > 0 Then
        Grid.Text = Format(cSize, "##,###,###")
    Else
        Grid.Text = "0"
    End If
    
    ' Grid column ratio (percent reduction) appears in.
    Grid.col = COL_PERCENT_REDUCED
    If (Size = 0) Or (cSize = Size) Then
        Grid.Text = "0%"
    Else
        Grid.Text = Format(1 - cSize / Size, "###%")
    End If
    
End Sub

Sub DoAddFileToArchive()
' What it does:
'   Puts up a common Open dialog. Retrieves the name of a single file.
'   Then puts up the Add File to Archive form, which handles the rest.
'
' See also:
'   AddFile form.
'
    ' Filename as retrieved from common dialog.
    Dim Filename As String

    ' Put up a common dialog to obtain a filename.
    Filename = GetFilename("*", "All Files", "Add file to archive", "")

    ' If user canceled out of dialog, we're finished.
    If Filename = "" Then
        Exit Sub
    End If

    ' Stuff the full pathname into the dialog.
    AddFile.txtFilename = Filename

    ' This is what I really want, but VB3 can't do it--to have
    ' a property on the AddFile form I can access through dot
    ' notation.
    ' AddFile.FullPathname = Filename
    gAddFileFullPathname = Filename
    gAddFileOnlyFilename = FilenameAndExtFromPath(Filename)

    ' Put up the dialog and take it from there.
    AddFile.Show

End Sub

Sub DoCmdHelpContents()
' What it does:
'   Brings up the Contents section of the help file (the first topic).
'
' See also:
'   DoCmdHelpSearch()
'
    ' Display the help index.
    Form1.CMDialog1.HelpCommand = HELP_INDEX

    ' This is really a method, not a property--Action 6 means bring up help.
    Form1.CMDialog1.Action = DLG_HELP

End Sub

Sub DoCmdHelpSearch()
' What it does:
'   Searches the help engine based on a text string.
'
' See also:
'   DoCmdHelpContents()
'
    ' Do a partial-text search on a string.
    Form1.CMDialog1.HelpCommand = HELP_PARTIALKEY

    ' This is really a method, not a property--Action 6 means bring up help.
    Form1.CMDialog1.Action = DLG_HELP
    
End Sub

Sub DoCreateZipFile()
' What it does:
'   Obtains the name of a file from the user.
'   Empties grid, sets up the UI, etc.
'
'
' See also:
'   DoAddFileToArchive()
'
    ' Filename entered by user.
    Dim Filename As String

    ' Ask for a .ZIP filename.
    ' Obtain the name of a .ZIP file.
    Filename = GetFilename(ZIP_EXTENSION, PROGRAM_NAME, "File to create", "")

    ' Quit if user canceled out of dialog.
    If Filename = "" Then
        Exit Sub
    End If

    ' Reset the status indicators.
    ClearStatus Form1!MajorStatus
    ClearStatus Form1!MinorStatus

    ' See if there's already a file by this name.
    If FileExists(Filename) Then
        ' If so, do the same as File|Open.
        DoFileOpen Filename
        Exit Sub
    Else
        ' No existing file. Initialize grid.
        ' Make only fixed row visible. Remove other rows.
        Form1.Grid1.Rows = 2
        
        ' Empty the grid. 0 means don't clear the fixed row portion.
        ClearGrid Form1.Grid1, 0

        SetupUnzipGrid Form1, Form1.Grid1
        
        ' Make the grid visible.
        Form1.Grid1.Visible = True

        ' Make appropriate controls (e.g. menu items and toolbar buttons) available.
        UpdateUserInterface

        ' Make sure no rows appear selected.
        ' When highlight is off, it means the archive has just
        Form1!Grid1.HighLight = False

    End If

    ' Initialize the ZIP component.
    initZipCmdStruct Form1!Zip

    ' User has now established a name for the .ZIP file to create.
    Form1.Label1 = Filename
    Form1.Label1.Visible = True
    Form1.Zip.ZIPFile = Filename

    ' This is required so that the Add button will work.
    gFileOpen = True

    ' Note that the .ZIP archive is empty.
    gEmptyArchive = True

    ' Make appropriate controls (e.g. menu items and toolbar buttons) available.
    UpdateUserInterface

End Sub

Sub DoDeleteSelectedFiles(DeleteAll As Integer)
' What it does:
'   When 1 or more files are selected and user choosed Delete,
'   this code is called. The DeleteAll parameter can be used as
'   a shortcut to wipe out the entire archive.
'
' Parameters:
'   DeleteAll: If True, user wants to delete all files in archive.
'   If False, user only wants to delete selected files.
'
    ' Loop counter.
    Dim EachFile As Integer
    Dim Files As String

    ' Customized message text.
    Dim Msg As String

    ' # of files to extract.
    Dim NumFiles As Integer
    
    ' Set the Zip control to a known state.
    initZipCmdStruct Form1!Zip

    ' See if only 1 file is in the archive.
    ' That makes it the same as deleting *.*
    If Form1!Grid1.Rows = 2 Then
        DeleteAll = True
    End If

    ' See if the filename cell is empty.
    ' Form1!Grid1.Col = COL_FILENAME
    If Form1!Grid1.Text = "" Then
        ' Select by rows.
        Form1!Grid1.SelectionMode = 1
        Exit Sub
    End If

    ' Figure out how many files have been selected.
    NumFiles = Form1!Grid1.RowSel - Form1!Grid1.row + 1
    
    ' Collect the files into a single string with the names separated
    ' by spaces, which is how the Zip control takes the file list.
   For EachFile = 1 To NumFiles
        Files = Files & Form1!Grid1.Text
        If EachFile < NumFiles Then
            Files = Files & " "
            Form1!Grid1.row = Form1!Grid1.row + 1
        End If
    Next EachFile
    
    ' Handle special case in which the user wishes to delete all files.
    ' That's either because DeleteAll was chosen, or because no files
    ' are selected on the grid, which presumably means the entire grid.
    If (Files = "") Or (DeleteAll) Then
        ' Apparently, user wants to delete all files.
        ' Confirm this.
        If GetYesNo("This will delete EVERYTHING in the .ZIP file." & vbCRLF & "Sure you want to do this?", "Delete all files?") <> "Yes" Then
            Exit Sub
        End If
        ' Yes. User wants to delete everything.
        ' It would be faster to delete the file and create an empty archive, but this gives the user
        ' the ability to cancel during the operation and at least save some files.
        Files = "\*.*"
    Else
        ' Construct a message explaining how many files are chosen for
        ' deletion. Handle the special case of 1 file selected.
        If NumFiles = 1 Then
            Msg = "You have selected 1 file for deletion." & vbCRLF & "Sure you want to delete it?"
        Else
            Msg = "There are" & Str(NumFiles) & " files selected for deletion." & vbCRLF & "Sure you want to delete these files?"
        End If

        ' User wants to delete selected files.
        If GetYesNo(Msg, "Delete selected files?") <> "Yes" Then
            Exit Sub
        End If
    End If
    
    ' Assign the list of files to be deleted to the Zip control.
    Form1!Zip.ItemList = Files

    ' Make the zip/unzip progress status bars visible.
    ShowStatusIndicators

    Form1!Zip.ZIPFile = Form1!Label1
    Form1!Zip.ActionDZ = ZIP_DELETE

    ' Reset the status indicators.
    ClearStatus Form1!MajorStatus
    ClearStatus Form1!MinorStatus

    ' Make them disappear.
    HideStatusIndicators

    ' Now that file set has changed, redisplay grid full of filenames.
    DisplayFilesInArchive (Form1!Label1)

    ' Ensure grid has highlight & is in row-select mode.
    SelectRow

End Sub

Sub DoFileExit()
' What it does:
'   Provides a central routine to call when exiting FreeZip.
'
    Unload Form1
    End
End Sub

Sub DoFileOpen(OpenFile As String)
' What it does:
'   Displays the standard Open dialog if OpenFile is empty.
'   Allows user to select a .ZIP archive.
'   Displays the contents of the .ZIP archive.
'   If OpenFile is nonempty, skips the open dialog and attempts to open the file by that name.
'
' Parameters:
'   OpenFile: If nonempty, open that file and skip the open dialog.
'

    ' Full filename as pulled from the Open dialog.
    Dim Filename As String

    If OpenFile = "" Then
        ' Obtain the name of a .ZIP file.
        ' Look in the last directory searched for Zip files.
        ' If this is the first time the program has run, look in the application's own directory.
        Filename = GetFilename(ZIP_EXTENSION, PROGRAM_NAME, "", GetSetting(STR_PROFILE_DIRECTORIES, STR_PROFILE_ZIP_FILES, (App.Path)))
    Else
        ' Usually, this means the filename came off the command line.
        Filename = OpenFile
        ' If there's no extension, add it.
        If FileExtension(Filename) = "" Then
            Filename = Filename & "." & ZIP_EXTENSION
        End If
    End If

    If Filename = "" Then
        Exit Sub
    End If

    If FileExists(Filename) Then

        ' Make sure this is actually a .Zip archive file.
        If Not ValidArchive(Filename) Then
            ErrorMsg "Either this is a bad .ZIP file, or it's not a .ZIP file at all."
            Exit Sub
        End If

        ' Update profile, making this the default directory to look for zip files.
        SaveSetting App.EXEName, STR_PROFILE_DIRECTORIES, STR_PROFILE_ZIP_FILES, Dir$
        
        ' Display filename on form.
        Form1!Label1.Caption = Filename
        
        ' Make sure the grid is visible.
        Form1!Grid1.Visible = True


        ' Show information about each file in archive.
        DisplayFilesInArchive Filename

        ' Make sure no rows appear selected.
        Form1!Grid1.HighLight = False

        ' When a file is first opened it's in a special state.
        ' No files are selected, so that if the user chooses
        ' Unzip, it will work on all files.
        ' (Could just select the whole grid automatically, but that looks tacky.)
        ' So erase the "1 file selected" message, which would ordinarily appear
        ' at this time because DisplayFilesInArchive() puts it there.
        MsgStatus " "
        
        
    Else
        ' File doesn't exist.
        ErrorMsg "Can't find the file " & Filename
    End If
    
End Sub


Sub DoUnzipSelectedFiles()
' What it does:
'   Central routine to call when 1 or more files are to
'   be extracted from the .ZIP archive.
'
' See also:
'   UnzipTo form
'
    ' Bring up Unzip To as a modeless dialog.
    UnzipTo.Show 1
    ' Select by rows.
    Form1!Grid1.SelectionMode = 1
End Sub

Sub ErrorMsg(Msg As String)
' What it does:
'   General-purpose routine to display an error message.
'   Using this instead of a plain message box allows later
'   expansion to include other features, such as logging.
    ' MsgBox type 0 is just the OK button.
    ' MsgBox type 48 adds the ! icon.
    MsgBox Msg, 48, "Error"
End Sub

Function FileExists(Filename As String) As Integer
' What it does:
'   Determines whether the named file is already on disk.
'
' Parameters:
'   Filename: Name of file whose existence is in question.
'
' Returns:
'   On success:
'   Nonzero if a file by the name Filename exists.
'
'   On failure:
'   0 if no file by the name Filename exists.
'
'
' See also:
'   DirExists
'
'
    ' Hack added to make life easier for FreeZip.
    ' Handle case of file being a directory here.
    If Right(Filename, 1) = "\" Then
        FileExists = DirExists(Filename)
        Exit Function
    End If

    If (Dir$(Filename) = "") Or (Filename = "") Then
        FileExists = False  ' Returns FALSE if file does not
    Else                    ' exist, TRUE if file does
        FileExists = True   ' exist.
    End If
End Function

Function FileExtension(Filespec As String) As String
' What it does:
'   Obtains the extension of a file, given its full pathname.
'
' Parameters:
'   Filename: Name of file whose extension is to be determined.
'
' Returns:
'   File extension
'
' See also:
'   FilenameAndExtFromPath()
'   SplitFilename()
'
    ' Dim tmpFilespec As String
    Dim Drive As String
    Dim Path As String
    Dim Filename As String
    Dim Ext As String
    ' Special case: filename ends with ".", which is considered
    ' an extension.
    If Right(Filespec, 1) = "." Then
        Ext = "."
    Else
        SplitFilename Filespec, Drive, Path, Filename, Ext
    End If
    FileExtension = Ext
End Function

Function FilenameAndExtFromPath(FullPathname As String)
' What it does:
'   Given the full file, path, and extension Fullpathname, returns
'   returns just the filename and pathname. For example, if the
'   full pathname is "D:\PL\VB\FREEZIP\ADA.ICO", returns just "ADA.ICO"
'
' Parameters:
'   FullPathname: A complete filename containing drive, path, filename, and extension.
'
' Returns:
'   On success: The filename, period, and extension if successful.
'
'   On failure: ""
'
' See also:
'   FileExtension()
'
' Notes:
'   Not adequately tested.
'
    ' The final result of this function.
    Dim Filename As String

    ' Parts of the file specification.
    Dim FileDrive As String
    Dim FilePath As String
    Dim FilenameOnly As String
    Dim FileExt As String

    ' Break the filename down into its component pieces.
    ' This is all to find out the extension.
    SplitFilename FullPathname, FileDrive, FilePath, FilenameOnly, FileExt

    If FileExt <> "" Then
        FilenameAndExtFromPath = FilenameOnly & "." & FileExt
    Else
        FilenameAndExtFromPath = FilenameOnly
    End If

End Function

Function GetFilename(DefaultExtension As String, ProgramName As String, DialogTitle As String, DefaultDirectory As String) As String
' What it does:
'   Displays the common file open dialog.
'   Obtains a filename from the user.
'
' Parameters:
'   DefaultExtension: File extension to use when common dialog
'   searches the directory for files of that type.
'
'   ProgramName: Name of application that appears in the "List files of type"
'   combo box in the common Open dialog.
'
'   DialogTitle: Window caption for common Open dialog.
'
' Returns:
'   Name of file user selected, or "" if user canceled out of dialog.
'
' See also:
'   Form1, Form1!CMDialog1
'
    ' The final result of this function.
    Dim Filename As String

    ' Alias for common dialog control.
    Dim FileOpenDialog As Control

    ' Parts of the file specification.
    Dim FileDrive As String
    Dim FilePath As String
    Dim FilenameOnly As String
    Dim FileExt As String

    ' Oh, for a WITH statement...
    Set FileOpenDialog = Form1!CMDialog1

    ' Clear any previous filenames.
    FileOpenDialog.Filename = ""

    If DefaultDirectory <> "" Then
        FileOpenDialog.InitDir = DefaultDirectory
    End If

    ' Default to .Zip file when dialog appears.
    FileOpenDialog.Filter = ProgramName & " (" & DefaultExtension & ")|*." & DefaultExtension

    ' If user specified a dialog title, use it instead of the default.
    If DialogTitle <> "" Then
        FileOpenDialog.DialogTitle = DialogTitle
    End If

    ' Bring up dialog showing all files using standard default
    FileOpenDialog.Action = 1

    ' With the grid invisible, must force repaint.
    ' Otherwise, file open dialog munges the screen display.
    Form1.Refresh

    ' Quit immediately if user cancels.
    If FileOpenDialog.Filename = "" Then
        GetFilename = ""
        Exit Function
    End If

    Filename = UCase(FileOpenDialog.Filename)

    ' Break the filename down into its component pieces.
    ' This is all to find out the extension.
    SplitFilename Filename, FileDrive, FilePath, FilenameOnly, FileExt
    ' SplitFilename (FileOpenDialog.FileName), FileDrive, FilePath, FilenameOnly, FileExt

    ' Make sure there was an extension.
    ' If user typed in filename, it may be missing. If so, add it.
    If FileExt = "" Then
        Filename = Filename & "." & DefaultExtension
    End If

    GetFilename = Filename

End Function

Function GetYesNo(Msg As String, Caption As String) As String
' What it does:
'   General-purpose utility routine to display a message and
'   wait for a "Yes", "No", or "Cancel" button response.
'
' Parameters:
'   Msg: The text of the message to display.
'
'   Caption: Window title.
'
' Returns:
'   A string, depending on the button clicked:
'     "Yes", "No", "Cancel"
'
' See also:
'   GetYesNoAll()
'
    Select Case MsgBox(Msg, 16 + 3, Caption)
        Case 6
            GetYesNo = "Yes"    ' User clicked Yes button.
        Case 7
            GetYesNo = "No"     ' User clicked No butotn.
        Case Else
            GetYesNo = "Cancel" ' user clicked Cancel button.
    End Select
End Function

Function GetYesNoAll(Msg As String, Caption As String) As String
' What it does:
'   General-purpose utility routine to display a message and
'   wait for a "Yes", "No", or "All" button response.
'   Uses the YesNoAll form.
'
' Parameters:
'   Msg: The text of the message to display.
'
'   Caption: Window title.
'
' Returns:
'   A string, depending on the button clicked:
'     "Yes", "No", "All"
'
' See also:
'   GetYesNo()
'
    YesNoAll!lblMsg = Msg
    YesNoAll.Caption = Caption
    ' Make the form modal, forcing an answer.
    YesNoAll.Show 1
    GetYesNoAll = YesNoAll.Tag
End Function

Sub HideStatusIndicators()
' What it does:
'   Makes the zip/unzip progress status bars invisible.
'
' See also:
'   ShowStatusIndicators
'
    Form1!lblCurrentFile.Visible = False
    Form1!lblZipFile.Visible = False

    Form1!MinorStatus.Visible = False
    Form1!MajorStatus.Visible = False
End Sub

Sub initUnzipCmdStruct(Unzip As Control)
' What it does:
'   Sets the Unzip control to a known state.
'   Certain properties are left untouched because they are
'   expected to be initialized outside this routine.
'
' Parameters:
'   Unzip: Dynazip's Unzip control.
'
' See also:
'   initZipCmdStruct()
'
    Unzip.ActionDZ = 0
    ' Set in form load.
    ' Unzip.BackgroundProcessFlag = False
    Unzip.ConvertLFtoCRLFFlag = False
    Unzip.DecryptCode = ""
    Unzip.DecryptFlag = False
    Unzip.Destination = ""
    Unzip.DiagnosticFlag = False
    Unzip.ErrorCode = 0
    Unzip.Filespec = ""
    Unzip.FreshenFlag = False
    Unzip.MajorStatusFlag = True
    Unzip.MessageCallbackFlag = False
    Unzip.MinorStatusFlag = True
    Unzip.NoDirectoryItemsFlag = True
    Unzip.NoDirectoryNamesFlag = True
    Unzip.OverwriteFlag = False
    Unzip.QuietFlag = False
    Unzip.RecurseFlag = True
    Unzip.ReturnCount = 0
    Unzip.ReturnString = ""
    Unzip.TestFlag = False
    Unzip.UnZIPIndex = -1
    Unzip.UnZipSubOptions = 0
    Unzip.UpdateFlag = False

    Unzip.zi_attr = 0
    Unzip.zi_cMethod = 0
    Unzip.zi_cPathType = 0
    Unzip.zi_crc_32 = 0
    Unzip.zi_cSize = 0
    Unzip.zi_DateTime = ""
    Unzip.zi_index = 0
    Unzip.zi_oSize = 0
    Unzip.zi_FileName = ""

    'Unzip.ZIPFile = ""

End Sub

Sub initZipCmdStruct(Zip As Control)
' What it does:
'   Sets the Zip control to a known state.
'   Certain properties are left untouched because they are
'   expected to be initialized outside this routine.
'
' Parameters:
'   Zip: Dynazip's Zip control.
'
' See also:
'   initUnZipCmdStruct()
'
    Zip.ActionDZ = NO_ACTION
    Zip.AddCommentFlag = False
    Zip.AfterDateFlag = False

    ' Allow Windows events to occur during zip process.
    ' Set in form load.
    ' Zip.BackgroundProcessFlag = False
    Zip.Comment = ""
    Zip.CompressionFactor = 5
    Zip.ConvertLFtoCRLFFlag = False
    Zip.Date = ""
    Zip.DeleteOriginalFlag = False
    Zip.DiagnosticFlag = False
    Zip.DontCompressTheseSuffixesFlag = False

    ' Don't force filenames to uppercase in archive.
    Zip.DosifyFlag = False
    Zip.EncryptCode = ""
    Zip.EncryptFlag = False
    Zip.ErrorCode = 0
    Zip.ExcludeFollowing = ""
    Zip.ExcludeFollowingFlag = False
    Zip.FixFlag = False
    Zip.FixHarderFlag = False
    Zip.GrowExistingFlag = False
    Zip.IncludeFollowing = ""
    Zip.IncludeOnlyFollowingFlag = False
    Zip.IncludeSysandHiddenFlag = False
    Zip.IncludeVolumeFlag = False
    Zip.ItemList = ""
    Zip.MajorStatusFlag = False
    Zip.MessageCallbackFlag = False
    Zip.MinorStatusFlag = False
    Zip.MultiVolumeControl = 0
    Zip.NoDirectoryEntriesFlag = False
    Zip.NoDirectoryNamesFlag = False
    Zip.OldAsLatestFlag = False

    ' Create temporary files in the same directory as the .ZIP file.
    Zip.PathForTempFlag = False
    Zip.QuietFlag = False
    Zip.RecurseFlag = False
    Zip.StoreSuffixes = ""
    Zip.TempPath = ""
    Zip.ZipSubOptions = 0

    'Zip.ZIPFile = ""


End Sub

Function IsAlpha(Value As String) As Integer
  ' What it does:
  '   Determines whether Value$ consists only of characters from the
  '   American alphabet ("A" to "Z").
  '
  ' Returns:
  '   -1 if Value$ consists only of letters from the American alphabet.
  '   0  if not.
  '
  ' Each character extracted from string.
  Dim Ch As String

  ' Length is the length of Value$, the input string.
  ' Figure out how long the string is.
  Dim Length As Integer

  ' Loop counter
  Dim EachChar As Integer

  Length = Len(Value)

  ' Check each character in the string.
  For EachChar = 1 To Length

    ' Get the next character.
    Ch = Mid(Value, EachChar, 1)

    ' If it's not a letter, return an error flag.
    If ((Ch < "A") Or (Ch > "Z")) And ((Ch < "a") Or (Ch > "z")) Then
      IsAlpha = 0
      ' Leave the function immediately; no need to waste time.
      Exit Function
    End If
  Next EachChar

  ' If here, all digits were valid.
  IsAlpha = -1

End Function

Sub MsgStatus(Msg As Variant)
' What it does:
'   Displays text in the status line.
'
' Parameters:
'   Msg: Text of message to display in status line.
'
' See also:
'   Form1
'
    Form1!lblStatus = Msg
End Sub

Sub ReplaceChars(Before As String, OldChar As String, NewChar As String)
' What it does:
'   Replaces (in place) all occurrences of OldChar with NewChar.
'
' Parameters:
'   Before: String containing characters to replace.
'   This routine is destructive--Before is written to, so
'   be sure to save a copy of the string if you want it preserved.
'
'   OldChar: Character to search for.
'
'   NewChar: Character to replace OldChar with.
'
' See also:
'   Form1
'
' Notes:
'   OldChar and NewChar may not be longer than 1 char.
'
'   Old, much less efficient code is preserved in comments.
'   It'd be interesting to see how much slower it is.

    ' Replacement string for ease of use.
    Dim After As String

    ' Placeholder in string
    Dim CurrPos As Integer

    ' Loop counter.
    Dim EachChar As Integer

    ' Current character in string.
    Dim NextChar As String * 1

    ' Make sure string isn't empty.
    If Len(Before) < 1 Then
        Exit Sub
    End If

    After = Before

    ' Simple optimization: Use Basic to check quickly whether any
    ' occurrences of OldChar are in the string.
    If InStr(Before, OldChar) = 0 Then
        Exit Sub
    End If

    ' Okay. Do a replacement.
    For CurrPos = 1 To Len(Before)
        NextChar = Mid(Before, CurrPos, 1)
        ' If Mid(Before, CurrPos, 1) = OldChar Then
        If NextChar = OldChar Then
            ' After = After + NewChar
            Mid(After, CurrPos, 1) = NewChar
        Else
            ' After = After + Mid(Before, CurrPos, 1)
            Mid(After, CurrPos, 1) = NextChar
        End If
    Next CurrPos

    Before = After

End Sub

Sub SetupUnzipGrid(Form As Form, Grid As Control)
' What it does:
'   Draws the grid's column titles, sets their justification
'   (for example, numbers are right-justified), and sets the
'   column widths.
'
' Parameters:
'   Form: Form containing the grid control.
'
'   Grid: The FlexArray grid control.
'
        ' FlexArray has a neat FormatString property that figures out column widths.
        ' It takes a string parameter, built up from this.
        Dim ColumnTitles As String
        ' # of columns for the unzip grid.
        Grid.Cols = 5

        Grid.row = 0

        ' Use special FormatString property for FlexArray's grid column titles.
        ' Title: Add some slop to the filename column.
        ColumnTitles = "<" & STR_COL_TITLE_FILENAME & String$(19, " ")
        ColumnTitles = ColumnTitles & "|"
        ' "^" means align center.
        ColumnTitles = ColumnTitles & "^" & STR_COL_TITLE_DATE_AND_TIME & "      "
        ColumnTitles = ColumnTitles & "|"
        ' ">" means align right.
        ColumnTitles = ColumnTitles & ">" & STR_COL_TITLE_UNCOMPRESSED_SIZE
        ColumnTitles = ColumnTitles & "|"
        ColumnTitles = ColumnTitles & ">" & STR_COL_TITLE_COMPRESSED_SIZE
        ColumnTitles = ColumnTitles & "|"
        ' "<" means align left, although it's not used here.
        ColumnTitles = ColumnTitles & ">" & STR_COL_TITLE_PERCENT_REDUCED
        Grid.FormatString = ColumnTitles
End Sub

Sub ShowNumberOfFilesSelected()
' What it does:
'   Displays a custom message in the status bar showing how many
'   files the user has selected. Gets the 1 case right.
'
    Dim MsgText
    Dim NumFilesSelected
    NumFilesSelected = GetNumberOfFilesSelected
    If NumFilesSelected <> 1 Then
        MsgText = "files selected"
    Else
        ' Handle case of a single file correctly.
        MsgText = "file selected"
    End If
    MsgStatus NumFilesSelected & " " & MsgText
End Sub

Sub ShowStatusIndicators()
' What it does:
'   Makes the zip/unzip progress status bars visible.
'   Normally they're hidden.
'
' See also:
'   HideStatusIndicators
'
'
    Form1!lblCurrentFile.Visible = True
    Form1!lblZipFile.Visible = True

    Form1!MinorStatus.Visible = True
    Form1!MajorStatus.Visible = True
End Sub

Sub SortByFilename()
' What it does:
'   Sorts contents of grid by filename.
'
    ' FlexArray takes this to mean sort all rows
    Form1!Grid1.row = 1
    ' Form1!Grid1.RowSel = Form1!Grid1.Row

    Form1!Grid1.col = COL_FILENAME
    Form1!Grid1.ColSel = COL_FILENAME
    Form1!Grid1.Sort = SORT_STRING_ASCENDING_CASE_INSENSITIVE
End Sub

Sub SortByUncompressedSize()
' What it does:
'   Sorts contents of grid by uncompressed size.
'
'  Notes:
'   Doesn't work. Think it may be due to commas in values.
'
    ' FlexArray takes this to mean sort all rows
    Form1!Grid1.row = 1
    ' Form1!Grid1.RowSel = Form1!Grid1.Row

    Form1!Grid1.col = COL_UNCOMPRESSED_SIZE
    Form1!Grid1.ColSel = COL_UNCOMPRESSED_SIZE
    ' Form1!Grid1.Sort = SORT_NUMERIC_ASCENDING
    Form1!Grid1.Sort = SORT_GENERIC_ASCENDING

End Sub

Sub SplitFilename(Filespec As String, Drive As String, Path As String, Filename As String, Ext As String)
' What it does:
'   Given the filename or file specification FileSpec, separates out
'   its component drive letter, directory (path), filename without
'   extension, and the file extension.
'
' Parameters:
'   FileSpec: The file specification to split. For example, "C:\PL\FOO.BAS"
'
'   Drive: If there's a drive specification in FileSpec, it will be
'   copied here; "C:" in the example above.
'
'   Path: If there's a directory in FileSpec, it is copied to this
'   variable.  "\PL\" in the example above.
'
'   Filename: If there's a filename in FileSpec, it is copied to this
'   variable.  "FOO" in the example above.
'
'   Ext: If there's a file extension in FileSpec, it is copied to this
'   variable.  "BAS" in the example above.
'
' Notes:
'   Assumes Filespec is a valid filename.
  ' Each character as extracted from filename.
  Dim Ch As String * 1
  ' Loop counter
  Dim EachChar As Integer

  ' F stands in for the filespec as it's slowly pried apart.
  Dim F As String

  ' Position of last backslash found.
  Dim LastBackSlash As Integer

  ' Position of last dot found.
  Dim LastPeriod As Integer

  ' Length is the length of the filespec.
  Dim Length As Integer

  Drive = ""
  Path = ""
  Filename = ""
  Ext = ""

  F = Filespec

  ' Copy the length to an integer to avoid numerous calls to LEN.
  Length = Len(F)

  ' Extract the directory drive letter and colon, if any.
  If IsAlpha(Left(F, 1)) And Mid(F, 2, 1) = ":" Then
    ' Extract the drive letter.
    Drive = Left(F, 2)
    ' And remove it from the filespec.
    F = Right(F, Length - 2)
  End If

  ' Assume there's neither a backslash or a period in the input.
  LastBackSlash = 0
  LastPeriod = 0

  ' Find the last backslash and/or period in the file specification.
  For EachChar = 1 To Length
    ' Get a copy of the next character.
    Ch = Mid(F, EachChar, 1)
    If Ch = "\" Then LastBackSlash = EachChar
    If Ch = "." Then LastPeriod = EachChar
  Next EachChar

  ' If the period comes after the backslash, we know:
  '   1. It signifies a file extension.
  '   2. The backslash comes right before the first letter of
  '      the filename.
  '   3. Everything from the first char of the string to the backslash
  '      (if any) is the directory.
  If LastPeriod > LastBackSlash Then
    Filename = Mid(F, LastBackSlash + 1, LastPeriod - 1 - LastBackSlash)
    Path = Mid(F, 1, LastBackSlash)
    Ext = Mid(F, LastPeriod + 1, Length)
  Else
    ' The period doesn't come after the last backslash. This could mean:
    ' 1. There's no period or no backslash; for example, "C:FOO".
    ' 2. The period comes before the backslash, as in the legal but
    '    unusual case of a directory name with a period in it:
    '      A.B\C
    If LastBackSlash > 0 Then
      ' Handle case 2.
      Path = Mid(F, 1, LastBackSlash)
      Filename = Mid(F, LastBackSlash + 1, Length)
    Else
      ' Handle case 1.
      Filename = F
    End If

  End If

  ' Make sure the last char of the path specifier is a backslash.
  If (Path <> "") And (Right(Path, 1) <> "\") Then
    Path = Path + "\"
  End If


End Sub ' SplitFilename

Sub UpdateUserInterface()
' What it does:
'   Selectively enables or disables controls according to the
'   state of the application.
'
'   For example, if there's no help file available, disables the menu entries
'   that would bring up the help file but leaves the Help|About... dialog intact.
'
' See also:
'   Form1
'   gHelpFilePresent global variable
'
    ' Make Help button and menu items available only if there's a help file.
    ' The About item should always be available.
    If gHelpFilePresent = True Then
        ' Command button:
        ' Form1!cmdHelp.Enabled = True
        Form1!Toolbar.Buttons(BTN_HELP).Enabled = True
        ' Menu items:
        Form1!mnuHelpContents.Enabled = True
        Form1!mnuHelpSearchForHelpOn.Enabled = True
    Else
        ' Command button:
        ' Form1!cmdHelp.Enabled = False
        Form1!Toolbar.Buttons(BTN_HELP).Enabled = False
        ' Menu items:
        Form1!mnuHelpContents.Enabled = False
        Form1!mnuHelpSearchForHelpOn.Enabled = False
    End If

    If gFileOpen Then
        ' Special case: File is open but archive is empty.
        If gEmptyArchive = True Then
            Form1!mnuZipAddFileToArchive.Enabled = True

            Form1!mnuZipUnzipSelectedFiles.Enabled = False
            Form1!mnuZipDeleteSelectedFiles.Enabled = False
            Form1!mnuZipDeleteAllFiles.Enabled = False

            Form1!Toolbar.Buttons(BTN_ADD).Enabled = True
            Form1!Toolbar.Buttons(BTN_UNZIP).Enabled = False
            Form1!Toolbar.Buttons(BTN_DELETE).Enabled = False
            
            Exit Sub
        End If

        ' File is open & archive is nonempty.

        ' Handle menu items.
        Form1!mnuZipUnzipSelectedFiles.Enabled = True
        Form1!mnuZipUnzipAllFiles.Enabled = True
        Form1!mnuZipAddFileToArchive.Enabled = True
        Form1!mnuZipDeleteSelectedFiles.Enabled = True
        Form1!mnuZipDeleteAllFiles.Enabled = True

        ' Update buttons on toolbar.
        Form1!Toolbar.Buttons(BTN_ADD).Enabled = True
        Form1!Toolbar.Buttons(BTN_UNZIP).Enabled = True
        Form1!Toolbar.Buttons(BTN_DELETE).Enabled = True
    Else
        ' No archive is open.

        ' Handle menu items.
        Form1!mnuZipUnzipSelectedFiles.Enabled = False
        Form1!mnuZipUnzipAllFiles.Enabled = False
        Form1!mnuZipAddFileToArchive.Enabled = False
        Form1!mnuZipDeleteSelectedFiles.Enabled = False
        Form1!mnuZipDeleteAllFiles.Enabled = False

        ' Update buttons on toolbar.
        Form1!Toolbar.Buttons(BTN_ADD).Enabled = False
        Form1!Toolbar.Buttons(BTN_UNZIP).Enabled = False
        Form1!Toolbar.Buttons(BTN_DELETE).Enabled = False
    End If

End Sub

Function ValidArchive(Filename As String) As Integer
' What it does:
'   Determines whether Filename is a valid Zip archive file.
'
' Parameters:
'   Filename : Full pathname of file in question.
'
' Returns:
'   On success: Nonzero if Filename is a valid Zip archive.
'   On failure: 0 if Filename is not a valid Zip archive.
'
' Notes:
'    Munges the Form1!Unzip properties.
'    Sets its QuietFlag to True.
'
  ' Bug:
  ' Should save all props here. Non reentrant.
  initUnzipCmdStruct Form1!Unzip
  Form1!Unzip.QuietFlag = True
  Form1!Unzip.ZIPFile = Filename
  Form1!Unzip.ActionDZ = UNZIP_COUNTALLZIPMEMBERS
  If Form1!Unzip.ErrorCode = UE_OK Then
    ValidArchive = -1
  Else
    ValidArchive = 0
  End If
    
End Function


Public Function GetNumberOfFilesSelected()
' What it does:
'   Obtains the number of files selected.
'
' Returns:
'   Well, the number of files selected.

    GetNumberOfFilesSelected = Abs(Form1!Grid1.RowSel - Form1!Grid1.row) + 1
End Function
