/*	EXPR_LEX.H
	Copyright (C) 1992	Keith L. Robertson	All Rights Reserved

	Lexical analyzer for the EXPR expression evaluator.
	Declares token and other constant values, external data, and
	external functions for lexical analysys.
*/
#ifndef	EXPR_LEX_H
#define	EXPR_LEX_H

#ifndef TYPES_H
 #include <types.h>
#endif


/************************/
/*	  TOKENS	*/
/************************/

enum	{	EOS,		/* End of input stream */

	REAL,	IDENTIFIER,
	COMMA,	SEMICOLON,
	EQUAL,
	OPEN_P,	CLOSE_P,
	PLUS,	MINUS,
	TIMES,	DIVIDE,
	POWER,	QUESTION,
	LIST,
	KILL,
	LISP,
	HELP,
};


/****************************************/
/*	     Helper Functions		*/

/* Write a string to Standard Output. */
extern VOID	put_string (CSTRING str);

/* Convert unsigned short to a string. */
extern STRING	ustoa (USHORT num);


/****************************************/
/*	     Lexical Functions		*/

/* Print an error message, desc.  'etype' is the type of error. */
/* 'desc' is err msg. */
extern VOID	log_error   (USHORT etype,  CSTRING desc);
extern VOID	log_warning (USHORT etype,  CSTRING desc);
enum	{ LEXICAL, SYNTAX, SEMANTIC, FATAL };	/* Types of errors */

extern SHORT	ERROR_THRESHOLD;	/* Max # errors before it quits */
extern SHORT	TAB_STOPS;		/* Spaces between tabs */


/* Initialize lexical analyzer.  Called before any other call. */
extern VOID	init_lexer ();		/* Reads from stdin */

/* Get data from input stream. */
extern USHORT	get_token ();
extern UCHAR	get_character ();

extern USHORT	last_token;		/* Last token returned by get_token () */


/****************************************/
/*		Lexical Info		*/
/* Used to transfer other lexical information from lexer to semantic actions
   and also to represent symbol attribute information on the parse stack. */

typedef struct	{
    USHORT	line;		/* Line number where lexeme began. (NONZERO) */
    USHORT	posn;		/* Position on line. */

    USHORT	token;		/* Token of this lexeme. */
    STRING	name;		/* Lexeme string for some tokens. */

    /* Which is used and its meaning is between lexer & sem actions. */
    union  {
	SHORT	s;
	LONG	l;
	FLOAT	f;
	VOID*	p;
    }	val;
}	LEX_INFO;

extern LEX_INFO		lexeme;


/*	Operations needed by the parser and semantic actions.	*/

#define		set_jumped(lex_info,val)	((lex_info).line = ! val)
#define		was_jumped(lex_info)		((lex_info).line == 0)


#endif	/* EXPR_LEX_H */
