#----------------------------------------------------------------------

# Script: IPAGE.S

# Usage: CIEX IPAGE.S (filename)

# Freely Distributed.
# Refer to accompanying README.TXT file for information on how to
# install, use, and customize this script.

# Conceived of by: Kent Widdle, 11-1-94
# Version 2.1, by New Line Software, Inc. 11-14-94

#-----------------------------------------------------------------------

# Setup the program as follows.

set -clear      # Clear any existing variables.
source -calls   # Speed up access to calls in this file.

# These parameters can be modified by the user.

set $max_lines_per_page = 55    # Max lines per page, excluding header.

set $start_page  = 1            # The start page number.
set $background  = blue         # Screen background color.
set $foreground  = yellow       # Screen foreground color.

set $tmpfile       = IPAGE.T-   # A temporary file.
set $toc_file_name = TOC.TXT    # File containing Table of Contents.

# This is the header at the start of each page. Users can customize this
# to include the filename, company name, or other information. The
# keyword %FILENAME% contains the name of the file; %DATE contains the
# current date; %PAGE% contains the current page number.

set $header = ~
" %FILENAME%                     %DATE%                     Page: %PAGE% "

# Get input file name. First check the command line. If not there,
# prompt for the file to paginate.

set $input_file_name   = $arg:1^value  # Filename

# Was a valid file specified on the command line?

if $input_file_name is not file and $arg:1^value is not null
   echo "Note - Invalid file name specified."
   return
endif

# If no file was specified, prompt for a name. Continue looping until a
# valid file is specified.

loop while -test $input_file_name is not file

   # Query for a file name.

   echo
   echo "Enter File Name, DIR, or Return To Cancel."
   echo -fmt 'ENTER - FILE NAME / DIR / RETURN > '
   read $input_file_name

   # Convert file name to upper case.

   set -inline $input_file_name = echo -upper $input_file_name

   # Allow the user to list the directory, or escape.

   if -test '^ +DIR +$' in $input_file_name

      # User typed, "DIR". List the directory.

      exec -force $input_file_name
      echo
      break -next

   else if -test $input_file_name is null

      # User entered null file name. Exit the script.

      echo "Okay - no action taken."
      return -entry

   else if -test $input_file_name is not file

      # Specified file is not valid.

      echo "Note - Invalid file name specified."

   endif

endloop

# A file name has been acquired. convert filename to upper case letters.

set -inline $input_file_name = echo -upper $input_file_name

# The base file (i.e. the file name with any leading path specification
# removed) is derived using the following commands.

set $base_file_name = $input_file_name
set -replace $base_file_name = all '*\'

#-----------------------------------------------------------------

callpoint: display_main_menu

# Display the main menu. The script jumps to this location at the end of
# each operation.

#-----------------------------------------------------------------

# Setup the screen as follows:

screen -default
box -style french
screen -buffer on
screen -back $background
screen -clear

# Display the file name at the top of the screen.

move 1 1
screen -fore black ; screen -back white
line -clear 80
echo -center " Input File: $input_file_name "

# Display the script and version number at the bottom of the screen.

move -home
line -clear 80
set -inline $arg:0 = echo -upper $arg:0
echo -center " $arg:0  V2.0 "

# Main box is as follows:

screen -fore $foreground
screen -back $background
box 1 2

# Menu box is as follows:

box 8 4

# Label the menu box.

move 8 4 ; echo -center " ENTER - Function Key. "

# Add the function keys.

screen -fore black
screen -back white
move 17 8   ; echo -scr '[ F1  ]'
move -down 2; echo -scr '[ F2  ]'
move -down 2; echo -scr '[ F3  ]'
move -down 2; echo -scr '[ F4  ]'
move -down 2; echo -scr '[ ESC ]'

# Add the function key labels.

screen -fore $foreground
screen -back $background
move 27 8   ; echo -scr "BEGIN Interactive File Pagination."
move -down 2; echo -scr "CHECK / ANALYZE Current File Pagination."
move -down 2; echo -scr "REMOVE Existing Page Breaks and Headers."
move -down 2; echo -scr "CREATE Table Of Contents For File."
move -down 2; echo -scr "RETURN To DOS."

# Flush the screen to the display here.

screen -flush

# Begin main service loop for script.

loop

   # Wait for a key. Program stops until a key is entered.

   read -key $keyname

   # Decode the key entered above.

   if $keyname eq f1

      # User selected F1 function key. Paginate this file.

      call paginate
      call -goto display_main_menu

   else if $keyname eq f2

      # User selected F2 key. Check the file pagination

      call check
      call -goto display_main_menu

   else if $keyname eq f3

      # User selected F3 key. Unpaginate this file.

      call unpaginate
      call -goto display_main_menu

   else if $keyname eq f4

      # User selected F4 key. Create a table of contents for file

      call create_toc
      call -goto display_main_menu

   else if $keyname eq esc

      # User selected ESC key. Return with no other action.

      return

   endif

endloop

#--------------------------------

callpoint: paginate

# This callpoint performs the file pagination, and is called when the
# user presses the F1 function key.

#--------------------------------

# Setup the screen.

screen -default
box -style french
screen -back $background
screen -clear

# Clear the top line.

screen -fore black
screen -back white
move 1 1
line -clear 80

# Display the version number at the bottom of the screen.

move -home
line -clear 80
echo -center " IPAGE.S  V2.0 "

# Initialize the following two values.

set $prev_break_line = 0
set $next_break_line = 0
set $page_number = $start_page

# Remove any existing temporary file.

if $tmpfile is file
   ERASE $tmpfile
endif

# Get the date, used in the header.

set -inline $date = call get_date

# Loop until an escape key is entered.

loop

   # Clear the text region of the screen.

   screen -back $background
   screen -fore $foreground
   screen -buffer on
   box -clear 1 2 80 22

   # Draw lines on the screen.

   move 1 2  ; line 80
   move 5 2  ; line -vertical 22
   move 1 23 ; line 80

   # Banner, at top of the display, indicating the page number.

   move 1 1
   screen -fore black
   screen -back white
   line -clear 80
   echo -scr-35 " Input File: $input_file_name "
   move -right 70
   echo -scr-10 "PAGE: $page_number"

   screen -fore $foreground
   screen -back $background

   # Get a block of text.
   # The displayed text block is $max_lines_per_page lines after last
   # page break and extends to six lines past the end of the page.

   set -file $text_block = $input_file_name ~
      [ $prev_break_line + $max_lines_per_page - 10 ] to ~
      [ $prev_break_line + $max_lines_per_page + 6 ]

   if $text_block^range lt 11

      # This is the last page. Perform terminal processing for this
      # callpoint, and then exit.

      # First output the last page.

      set -eval $next_break_line = ~
         $prev_break_line + $max_lines_per_page
      call output_page  >> $tmpfile

      # Display the summary screen.

      call display_summary "$page_number Page(s)."

      # Clear the screen.

      screen -clear

      # Copy the paginated file into the original file, and return.
      # This occurs only if at least one page break as been added to
      # the file. Otherwise, the input file is unchanged. Note that this
      # will prevent any changes to a binary input file which was
      # mistakenly specified.

      if $tmpfile is file

         if $page_number gt 1 # Replace input file
            ERASE $input_file_name
            RENAME $tmpfile $input_file_name
         else
            ERASE $tmpfile    # Do not modify input file.
         endif

      endif

      return

   endif

   # This is not the last of the text. Display the text as follows

   screen -flush
   screen -buffer off

   # The location of the next page break is as follows:

   set -eval $next_break_line = $prev_break_line + $max_lines_per_page

   # The next page header is as follows:

   set $real_header = $header
   set -replace $real_header = %FILENAME% by $base_file_name
   set -replace $real_header = %DATE% by $date
   set -replace $real_header = %PAGE% by [ $page_number + 1 ]

   # Derive the page line number.

   set -eval $page_line_number = $max_lines_per_page - 10

   # Display the text block. Loop for each value in the text block
   # line as follows:

   move 1 3
   loop for $line = $text_block

      if $page_line_number eq $max_lines_per_page

         # This is the location of the page break header. Insert the 3
         # line page header onto the display.

         # First line of the header is blank.

         echo -scr4 "$page_line_number"
         move -right 5
         line -clear 80
         move -left 5
         move -down
         set $page_line_number + 1

         # Second line of the header contains the text.

         echo -scr4 "$page_line_number"
         screen -fore black
         screen -back white
         move -right 5
         line -clear 80
         move -right
         echo -scr $real_header
         move -left
         screen -fore $foreground
         screen -back $background
         move -left 5
         move -down
         set $page_line_number + 1

         # Third line of the header is blank.

         echo -scr4 "$page_line_number"
         move -right 5
         line -clear 80
         move -left 5
         move -down
         set $page_line_number + 1

      endif

      # Add the next line of text from the file to the screen.

      echo -scr4 "$page_line_number"
      move -right 5
      echo -scr $line
      move -left 5
      move -down
      set $page_line_number + 1

   endloop

   # The text file with the header inserted is being displayed. Display
   # the prompt at the bottom of the display.

   move -home
   screen -fore black ; screen -back white
   line -clear 80
   echo -center ' [ PAGE BREAK? ( Enter / Up / Down / Esc=Cancel ) ] '
   screen -fore white ; screen -back $background

   # Get and process user keystrokes in the following loop.

   loop

      # Get a key. The program stops until a key is entered.

      read -key $keyname

      # Decode the keystroke.

      if $keyname eq up or $keyname eq pgup

         # User selected either the "up arrow" , or "page up" key. Move
         # the header up one line on the display. Begin by decrementing
         # the next break line.

         set $next_break_line - 1

         if [ $next_break_line - $prev_break_line]  ~
            lt [ $max_lines_per_page - 10 ]

            # Clip the next break line at this minimum.

            set $next_break_line = ~
            [ $prev_break_line + $max_lines_per_page - 10]

            break -next # Don't change the screen.

         endif

         # The location of the header on the display is as follows: This
         # is the location of the first blank line preceding the actual
         # header text.

         set $ycoord = ~
         [ 13 + $next_break_line - $prev_break_line - $max_lines_per_page ]

         # Redraw the header one line up. First line of the header is
         # blank.

         move 6 $ycoord; line -clear 80

         # Second line of the header contains the header text.

         move -down
         screen -fore black
         screen -back white
         line -clear 80
         move -right
         echo -scr $real_header
         move -left
         screen -fore $foreground
         screen -back $background

         # Third line of the header is blank.

         move -down; line -clear 80

         # Fourth line is text that previously preceded the header.

         move -down
         line -clear 80
         echo -scr $text_block:[$ycoord - 2]

      else if $keyname eq down or $keyname eq pgdn

         # User selected either the "down arrow" or "page down" key.
         # Move the header down one line on the display. Begin by
         # incrementing the next break line.

         set $next_break_line + 1

         if [  $next_break_line - $prev_break_line ] ~
            gt $max_lines_per_page

            # Clip the next break line at this maximum.

            set $next_break_line = ~
              [ $prev_break_line + $max_lines_per_page ]

            break -next # Don't change the screen.

         endif

         # The location of the header on the display is as follows:
         # This is the location of the first blank line preceding the
         # actual header text.

         set $ycoord = ~
         [ 12 + $next_break_line - $prev_break_line - $max_lines_per_page ]

         # Redraw the header one line up. First line of header is what
         # was previously the line immediately following the header.

         move 6 [$ycoord]
         line -clear 80
         echo -scr $text_block:[$ycoord - 2]

         # Second line of the header is blank.

         move -down; line -clear 80

         # Third line of the header contains the header text.

         move -down
         screen -fore black
         screen -back white
         line -clear 80
         move -right
         echo -scr $real_header
         move -left
         screen -fore $foreground
         screen -back $background

         # Fourth line of the header is blank.

         move -down; line -clear 80

      else if $keyname eq enter

         # User typed the "enter" key. Output this page as follows.
         # Both the "output_page" and "add_formfeed" calls are performed
         # in a block, so that their output is appended to the file
         # given by "$tmpfile".

         startblock >> $tmpfile
            call output_page
            call add_formfeed
         endblock

         # Save the line number of the page break, and increment the
         # page number.

         set $prev_break_line = $next_break_line
         set $page_number + 1
         break

      else if $keyname eq esc

         # User entered the "esc" key. Leave the program, deleting any
         # partially paginated file.

         if $tmpfile is file
            ERASE $tmpfile
         endif
         return

      else

         # No match for the key. Do nothing.

      endif

   endloop

endloop

return

#-----------------------------------------------------------------

callpoint: unpaginate

# Remove all headers from the file.
# This callpoint will work reliably only if the file was paginated
# with IPAGE. Otherwise, intended blank lines may be deleted from
# the target file. (Lines containing text are never modified.)

#-----------------------------------------------------------------

# Setup the screen.

screen -default
box -style french
screen -back $background
screen -clear
screen -fore black
screen -back white

move 1 1
line -clear 80
echo -center " Input File: $input_file_name "

move -home
line -clear 80
echo -center " IPAGE.S  V2.0 "
screen -fore $foreground
screen -back $background
box 1 2

box 8 4
move 8 4 ; echo -center " REMOVING Page Breaks - ANY KEY To Cancel. "

move 26 10  ; echo -scr '  Line Number : '
move -down 2; echo -scr '  Total Lines : '
move -down 2; echo -scr '  Page Number : '

screen -fore black
screen -back white
move 43 10  ; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'

# Initialize things.

set $skip_next = false
set -inline $formfeed = echo -ascii 12
unset $prev_line
unset $abort

# Erase any temporary file that may exist from a previous operation.

if $tmpfile is file
   ERASE $tmpfile
endif

screen -fore $foreground
screen -back $background

# Setup the following values.

set $line_number = 0
set $total_lines = 0
set $page_number = 1

# Output goes to the temporary file.

startblock > $tmpfile

   # Iterate, for each line in the file.

   loop for -file $line = $input_file_name

      set $line_number + 1
      set $total_lines + 1

      # Update the status box for this iteration.

      screen -fore black
      screen -back white
      move 45 10  ; echo -scr6 $line_number
      move -down 2; echo -scr6 $total_lines
      move -down 2; echo -scr6 $page_number

      # Check the following:

      if -key

         # User entered a key into the running program. Set the abort
         # flag and break out of this loop.

         read -key $keyname
         set $abort = true
         break

      else if "^ +$formfeed" in $line

         # This line contains a form feed. Increment the page number.

         set $page_number + 1
         set $line_number = 0

         # Determine whether to delete the previous line. The previous
         # line is deleted only if it is blank.

         if prev_line is not set

            # The previous line was a page break. Do nothing.

         else if '^ +$' in $prev_line

            # The previous line is blank. (If paginated using IPAGE, the
            # line preceding the form feed always is blank.) Discard
            # the previous line.

            unset $prev_line

         endif

         # Skip this line, and the next line (if it is blank.)

         set $skip_next = true
         break -next

      else if $skip_next eq true and '^ +$' in $line

         # The previous line contained a form feed, and this line is
         # blank. Skip this line.

         set $skip_next = false
         break -next

      else if prev_line is set

         # The previous line has a value. Output the line.

         echo $prev_line
         set $skip_next = false

      endif

      # Save this line. It will be printed the next iteration (providing
      # that conditions permit.)

      set $prev_line = $line

   endloop

endblock

# Display the last line (unless it contains a form feed.)

if abort is set

   # User aborted the unpaginate operation. Remove the temporary file
   # and return.

   if $tmpfile is file
      ERASE $tmpfile
   endif

   return

else if prev_line is not set

   # No previous line exists. Do nothing more.

else if $formfeed not in $prev_line

   # Add the last line to the temporary file.

   echo $prev_line >> $tmpfile

endif

# All page breaks and headers have been removed. Display the summary
# screen as follows:

call display_summary "[ $page_number - 1 ] Page Break(s) Removed."

# Clear the screen,

screen -fore $foreground
screen -back $background
screen -clear

# Replace the upaginated file with the original file, and return. This
# occurs only if at least one page was found in the file. Otherwise,
# the input file is unchanged. Note that this will prevent any changes
# to a binary input file which was mistakenly specified.

if $tmpfile is file

   if $page_number gt 1 # Replace input file
      ERASE $input_file_name
      RENAME $tmpfile $input_file_name
   else
      ERASE $tmpfile    # Do not modify input file.
   endif

endif

return

#-----------------------------------------------------------------

callpoint: check

# Check the file for excessively long lines, and then display a bar
# chart of the page sizes.

#-----------------------------------------------------------------

# Setup the screen.

screen -default
box -style french
screen -back $background
screen -clear
screen -fore black ; screen -back white

# Display the file name at the top of the screen.

move 1 1
line -clear 80
echo -center " Input File: $input_file_name "

# Display the version number at the bottom of the screen.

move -home
line -clear 80
echo -center " IPAGE.S  V2.0 "

# Main window is as follows:

screen -fore $foreground
screen -back $background
box 1 2

# Status box is as follows:

box 8 4
move 8 4 ; echo -center " SCANNING File - ANY KEY To Cancel. "

# The following values are tabulated as the script executes.

move 26 9   ; echo -scr '  Line Number : '
move -down 2; echo -scr '  Page Number : '
move -down 2; echo -scr '  Max Page Sz : '
move -down 2; echo -scr '  Error Count : '

screen -fore black
screen -back white
move 43 9   ; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'

# Initialize things.

set -inline $formfeed = echo -ascii 12
screen -fore $foreground
screen -back $background

# Setup the following values.

set $page_number     = 1
set $line_number     = 0
set $max_line_number = 0
set $error_count     = 0

# Iterate, for each line in the file.

loop for -file $line = $input_file_name

   # Increment the line number.

   set $line_number + 1

   # Update the status box for this iteration.

   screen -fore black
   screen -back white
   move 45 9   ; echo -scr6 $line_number
   move -down 2; echo -scr6 $page_number
   move -down 2; echo -scr6 $max_line_number
   move -down 2; echo -scr6 $error_count

   # Check for the following:

   if -key

      # User entered a key. Return to the calling program.

      read -key $keyname
      return

   else if "^ +$formfeed" in $line

      # This line is a form feed. Record the number of lines in this
      # page.

      set $linecount:$page_number = $line_number
      if $line_number gt $max_line_number

         # This is the longest line encountered so far. Record it.

         set $max_line_number = $line_number

      endif

      if $linecount:$page_number gt 63

         # This line is too long. Increment the error count.

         set $error_count + 1

      endif

      # Reset these values for the next page.

      set $line_number = 0
      set $page_number + 1

   endif

endloop

# Record the number of lines in the last page.

set $linecount:$page_number = $line_number

# Statistics have been derived. Display the statistics.

call display_statistics

# Clear the screen and return.

screen -fore $foreground
screen -back $background
screen -clear
return

#-----------------------------------------------------------------

callpoint: display_statistics

# Create a scrollable bar chart showing the number of lines in each
# page of the document.

#-----------------------------------------------------------------

# Setup the screen.

screen -default
box -style french
screen -back $background
screen -clear

# Clear the top line.

screen -fore black
screen -back white
move 1 1
line -clear 80

# Display the version number at the bottom of the screen.

move -home
line -clear 80
echo -center " IPAGE.S  V2.0 "

# This is the symbol which is displayed in each bar chart.

set -inline $rbullet = echo -ascii 16

# Initialize the following parameters.

set $next_page = 1
set $max_lines_to_display = 8
set $range = $linecount^range

if $range is null
   return # No lines were tabulated. Return.
endif

# Loop until an escape key is entered.

loop

   # Clear the text region of the screen.

   screen -buffer on
   screen -back $background
   screen -fore $foreground
   box -clear 1 2 80 22

   # Draw lines on the screen.

   move 1 2  ; line 80

   # Hash marks.

   move 15 2 ; line -vertical 22; echo -scr "0"
   move 16 2 ; line -vertical 22; echo -scr "0"
   move 36 2 ; line -vertical 22; echo -scr "20"
   move 56 2 ; line -vertical 22; echo -scr "40"
   move 76 2 ; line -vertical 22; echo -scr "60"

   move 1 23  ; line 80

   # Banner, at top of the display.

   move 1 1
   screen -fore black ; screen -back white
   line -clear 80
   echo -center " Input File: $input_file_name "

   # Chart the statistics as follows.

   screen -fore $foreground
   screen -back $background

   set $lines_displayed = 0
   move 2 3;      echo -scr "  PG:    SZ:"
   move -down 2

   # Loop for each page in the linecount array.

   loop for $pagenum = $next_page to $range

      # Display a line number.

      screen -fore black
      screen -back white
      echo -scr6 "$pagenum  "
      move -right 6

      if $linecount:$pagenum ge 63

         # More than the maximum number of lines for this page.

         screen -fore $foreground
         screen -back $background
         echo -scr6 "** "
         move -right 9
         screen -fore black
         screen -back white
         echo -scr63 $rbullet

      else

         # Chart the number of lines on this page.

         screen -fore $foreground
         screen -back $background
         echo -scr6 "$linecount:$pagenum "
         move -right 9
         screen -fore black
         screen -back white
         echo ["-scr" "$linecount:$pagenum"] $rbullet

      endif

      move -left 15
      move -down 2

      set $lines_displayed + 1
      if $lines_displayed eq $max_lines_to_display

         # No more lines can be displayed.

         break

      endif

   endloop

   # Display a prompt at the bottom of the screen.

   move -home
   screen -fore black ; screen -back white
   line -clear 80
   echo -center ~
   ' [ PgUp / PgDn / Up / Down / Home / End / Esc=Return ] '

   screen -flush
   screen -fore $foreground
   screen -back $background

   # Statistics are being displayed. Wait for and process a function
   # key as follows:

   read -key $keyname

   # Decode the key, entered above.

   if $keyname eq up

      # User entered the up arrow key. Scroll the display up by one.

      if $next_page gt 1
         set $next_page - 1
      endif

   else if $keyname eq down

      # User entered the down arrow key. Scroll the display down by one.

      if [ $next_page + $max_lines_to_display ] le $range
         set $next_page + 1
      endif

   else if $keyname eq pgup

      # User entered the page up key. Scroll the display up by one page.

      if [ $next_page - $max_lines_to_display ] gt 1
         set $next_page - $max_lines_to_display
      else
         set $next_page = 1
      endif

   else if $keyname eq pgdn

      # User entered the page down key. Scroll the display down by one
      # page.

      if [ $next_page + $max_lines_to_display ] le $range
         set $next_page + $max_lines_to_display
      endif

   else if $keyname eq home

      # User entered the home key. Display the home position at the top
      # of the statistics list.

      set $next_page = 1

   else if $keyname eq end

      # User entered the end key. Display the end position at the bottom
      # of the statistics list.

      if -eval ($range - $max_lines_to_display + 2) gt 0
         set $next_page = [ $range - $max_lines_to_display + 2]
      endif

   else if $keyname eq esc

      # User entered the escape key. Return to the calling program.

      return

   else

      # No match for the key.

   endif

   # Flash the prompt.

   screen -buffer off
   move -home
   screen -fore black ; screen -back white
   line -clear 80

endloop

return

#-----------------------------------------------------------------

callpoint: create_toc

# Create a table of contents (TOC) for the paginated file.

#-----------------------------------------------------------------

# Setup the screen.

screen -default
box -style french
screen -back $background
screen -clear
screen -fore black
screen -back white

# Display the file name at the top of the screen.

move 1 1
line -clear 80
echo -center " Input File: $input_file_name "

# Display the version number at the bottom of the screen.

move -home
line -clear 80
echo -center " IPAGE.S  V2.0 "

# Main box is as follows:

screen -fore $foreground
screen -back $background
box 1 2

# Status box is as follows:

box 8 4
move 8 4; echo -center " SCANNING Section Titles  - ANY KEY To Cancel. "

# The following values are tabulated as the script executes.

move 26 10  ; echo -scr '  Line Number : '
move -down 2; echo -scr '  Title Count : '
move -down 2; echo -scr '  Page Number : '

screen -fore black
screen -back white
move 43 10  ; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'
move -down 2; echo -scr '[         ]'

# Initialize things.

unset $possible_title
unset $check_next
unset $abort
set -inline $formfeed = echo -ascii 12
screen -fore $foreground
screen -back $background

# Erase any TOC file that may exist from a previous operation.

if $toc_file_name is file
   ERASE $toc_file_name
endif

# Setup the following values:

set $line_number = 0
set $title_count = 0
set $page_number = $start_page

# Output of this block goes to the table of contents file:

startblock > $toc_file_name

   # Iterate, for each line in the file.

   loop for -file $line = $input_file_name

      set -replace $line = '^ +' # Strip leading blanks.
      set -replace $line = '+ $' # Strip trailing blanks.

      set $line_number + 1

      # Update the status box for this iteration.

      screen -fore black
      screen -back white
      move 45 10  ; echo -scr6 $line_number
      move -down 2; echo -scr6 $title_count
      move -down 2; echo -scr6 $page_number

      # Check for the following.

      if -key

         # User entered a key into the running program. Set the abort
         # flag and break out of this loop.

         read -key $keyname
         set $abort = true
         break

      else if "^ +$formfeed" in $line

         # This line contains a form feed. Increment the page number.

         set $page_number + 1

      else if '^ +$' in $line

         # This is a blank line. The text line is potentially a title.

         set $check_next = true
         break -next # Break to top of this loop.

      else if check_next is set and $line^size lt 65

         # The preceding line was a blank line, and this line is less
         # than 65 characters. This is potentially a title.

         unset $check_next
         set $possible_title = $line
         break -next # Break to top of this loop.

      else if '^ +--+ +$' in $line and possible_title is set

         # This line is a row of dashes. The value of $possible_title
         # qualifies as a section header. Output this table of contents
         # entry.

         echo -fmt-65 $possible_title
         echo -fmt " ...... "
         echo $page_number
         set $title_count + 1

      endif

      # This line, and the next line, cannot qualify as a section title.

      unset $possible_title
      unset $check_next

   endloop

endblock

# Display a summary message.

if abort is set

   # User aborted the operation. Delete the temporary file and return.

   if $toc_file_name is file
      ERASE $toc_file_name
   endif

   return

endif

# Display a summary message.

call display_summary "Table Of Contents File - TOC.TXT - Created."

# Clear the screen, and return.

screen -fore $foreground
screen -back $background
screen -clear

return

#-----------------------------------------------------------------

callpoint: output_page

# Output one paginated page to standard output.

#-----------------------------------------------------------------

# Clear the bottom bar:

screen -fore black ; screen -back white
move 1 1
line -clear 80
move -home
line -clear 80

if $page_number eq 1

   # Never display a line number for page one.

   loop for -file $line = $input_file_name 1 to [ $next_break_line - 1]
      echo -value $line
   endloop
   return

else if $page_number eq $start_page

   # This is the first page, but not page one. The value of $start_page
   # must be other than 1. Handle this special case by prefacing the
   # header with a form feed, and continuing. (This allows a document
   # residing in separate files to be paginated.)

   call add_formfeed

endif

# Pages have the following header.

set $real_header = $header
set -replace $real_header = %FILENAME% by $base_file_name
set -replace $real_header = %DATE% by $date
set -replace $real_header = %PAGE% by $page_number

# Add the header.

echo $real_header
echo

if $prev_break_line le 0
   set $prev_break_line = 1 # Prevent errors.
endif

# Output the page as follows.

loop for -file $line = ~
   $input_file_name [ $prev_break_line ] to [ $next_break_line - 1]
   echo -value $line
endloop

# This page and header have been written to standard output. Return to
# the calling program.

return

#-----------------------------------------------------------------

callpoint: display_summary

# Display the summary screen after an operation

# Usage: call display_summary (message)

#-----------------------------------------------------------------

screen -buffer on

# Setup the summary screen.

screen -fore $foreground
screen -back $background
screen -clear
box 8 4

# Title of summary screen.

screen -fore black
screen -back white
move 15 8
line -clear
echo -center "SUMMARY"

# Summary message.

screen -fore $foreground
screen -back $background
move -down 3 ; echo -center "File : $input_file_name"
move -down 2 ; echo -center $arg:1

# Footer of summary screen.

screen -fore black
screen -back white
move 15 16
line -clear
echo -center "ANY KEY To Continue."
screen -fore $foreground
screen -back $background
screen -flush

# Summary screen is being displayed. Wait for a key before continuing.

read -key $keyname
return

#-----------------------------------------------------------------

callpoint: get_date

# Get the BIOS day of the week.

#-----------------------------------------------------------------


set -view $arg
set -head $T:date = startblock
   echo ""|date
endblock
set -list $T:date = $T:date
echo $T:date:($T:date^last)

unset $T; return

#-----------------------------------------------------------------

callpoint: add_formfeed

# Output a carriage return, and then a form feed character.

#-----------------------------------------------------------------

# Add the form feed.

echo
echo -ascii 12
return

#-----------------------------------------------------------------

callpoint: interrupt
call -goto exception
return

#-----------------------------------------------------------------

callpoint: exception

# Jump here on errors.

#-----------------------------------------------------------------

screen -default
move -home
line -clear
echo -center "!? - ERROR: $sys:errtxt $sys:errloc"
if $tmpfile^value is file
   ERASE $tmpfile
endif
return -entry
