/* TCPP.CPP:

   Simple example Program for Help Access Library
   explaining the use of Help Access Library for
   C++ programming.

   This example program Displays a list of all
   Topic-Entry and Hotspot-Macros used in a
   Help File.

   It is coded in a C++ manner with classes and
   shows how to Transfer class informations to Callbacks.
*/

#include <windows.h>
#pragma argsused
#include <time.h>
#include <stdio.h>
#include <malloc.h>
#include "hlpacces.h"

#ifdef __WIN32__
# define _export		// Microsoft C can't stand that
#endif

/* cHelpFile
 *
 * Let's define the Help File class as a Base class.
 *
 * This class is able to open a Help file and do the
 * neccessary callback translation stuff. To something
 * useful we have to derive it and overwrite the
 * "NewTopicData" member.
 */
class cHelpFile 
{
  public:
     LPHELPFILETITLE lpHelpFileTitle;       // Title-Informations retrieved
     HIFS            ifs;                   // Handle of internal File System
     HTOPIC          htopic;                // Handle of topic internal buffers
     LPCSTR          lpszHelpFileName;      // Filename of the Help File retrieved

     cHelpFile(LPCSTR AHelpFileName, DWORD hto_flags);// Constructor
     ~cHelpFile();                          // Destructor

     void TopicSeek(DWORD TopicOffset);     // Set topic-Offset for next enumeration.
     BOOL TopicEnumerate(void);             // Starts enumeration of Help Files.
     BOOL IsHelpFile(void);                 // Returns TRUE if all Help File informations are present
     LPCSTR FileTitle(void);                // Returns a Pointer to the Help files Title.
     LPCSTR TopicTitle(DWORD dwTopicOffset);// Returns a memory block containing the Title of the Topic addresses by dwTopicOffset 

     virtual BOOL NewTopicData(LPCSTR lpszTopicText, DWORD dwTopicOffset, TEXTTYPE eTextType)
                {return FALSE;};
                // Handles new strings retrieved from this File.
                // needs to be redefined in derived classes
};


        /* TopicEnumerationProcedure
         *
         * This is the callback-Function this application defined for the call to
         * HlpTopicEnumerate to Enumerate the TOPIC-Files contents
         *
         * For this Procedure need to be exportet as a callback, it cannot
         * be a C++ member function of the cHelpFile class.
         *
         * For this reason, it's just a regular function assuming that a
         * pointer to a cHelpFile object is passed in lParam
         */
	BOOL FAR PASCAL _export TopicEnumerationProcedure(
                LPCSTR    lpszTopicText,   // Text to be transfered to the Application program
                DWORD     dwTopicOffset,   // TopicOffset near where this Data has been retrieved from
                TEXTTYPE  eTextType,       // type of Text-String transfered to the Application
                LPARAM    lParam)          // Application-Defined user Data
        {
          cHelpFile FAR *pHelpFile = (cHelpFile FAR *) lParam;

          return pHelpFile->NewTopicData(lpszTopicText, dwTopicOffset, eTextType);
        }



/* ========== Define the Class members ================= */

/* Constructor for cHelpFile Class */
cHelpFile::cHelpFile(LPCSTR AHelpFileName, DWORD hto_flags)
{
  /* Initialize class members */
  lpHelpFileTitle = NULL;
  ifs = NULL;
  htopic = NULL;
  lpszHelpFileName = strdup(AHelpFileName);

  /* Try opening the IFS */
  ifs = IFSOpen(AHelpFileName);
  if (ifs!=NULL)
  {
     lpHelpFileTitle = HlpGetHelpTitle(ifs);
     htopic          = HlpTopicOpen(ifs, hto_flags);
  }
}


/* Destructor for cHelpFileClass
 *
 * Free allocated objects.
 */
cHelpFile::~cHelpFile()
{ if (htopic)           HlpTopicClose(htopic);
  if (lpHelpFileTitle)  HlpFreeHelpTitle(lpHelpFileTitle);
  if (lpszHelpFileName) free((void *) lpszHelpFileName);
  if (ifs)              IFSClose(ifs);
}


/* IsHelpFile
 *
 * returns TRUE if opening of Help File was Successfull
 */
BOOL cHelpFile::IsHelpFile(void)
{
  return ifs!=NULL &&
         lpszHelpFileName!= NULL &&
         htopic!=NULL &&
         lpHelpFileTitle!=NULL;
}


/* FileTitle
 *
 * Returns a Pointer to the Help files's title
 */
LPCSTR cHelpFile::FileTitle(void)
{
  return IsHelpFile()
           ? *lpHelpFileTitle->HelpTitle ? lpHelpFileTitle->HelpTitle : lpszHelpFileName
           : "Errornous File";

}

/* TopicTitle
 *
 * Returns a memory block containing the Title of the Topic addresses by dwTopicOffset
 */
LPCSTR cHelpFile::TopicTitle(DWORD dwTopicOffset)
{
  char   buffer[512];
  LPCSTR Result;

  if (IsHelpFile())
  {
       HlpGetTopicTitle(ifs, dwTopicOffset, buffer, sizeof(buffer), NULL);
       Result = strdup(buffer);
  } else
       Result = NULL;

  return Result;
}



/* TopicEnumerate
 *
 * Starts enumeration of a Help-Files Topic Data
 */
BOOL cHelpFile::TopicEnumerate(void)
{
  return IsHelpFile()
           ? HlpTopicEnumerate(htopic, TopicEnumerationProcedure, (LPARAM)this)
           : FALSE;
}


/* TopicSeek
 *
 * Set TopicOffset for next enumeration
 */
void cHelpFile::TopicSeek(DWORD TopicOffset)
{
  if (IsHelpFile())
      HlpTopicSeek(htopic, TopicOffset);
}



/* ============== cHelpMacros class ======================*/

/* cHelpMacros
 *
 * A class derived from cHelpFile to do the Help Macro enumeration
 */
class cHelpMacros : public cHelpFile
{
  public:
     cHelpMacros(LPCSTR AHelpFileName) : cHelpFile(AHelpFileName, 0) {};// Constructor
     virtual BOOL NewTopicData(LPCSTR lpszTopicText, DWORD dwTopicOffset, TEXTTYPE eTextType);
                // Handles new macro-String retrieved from this File.
};



/* NewTopicData
 *
 * This member function is called whenever new topic data has been
 * retrieved by the Callback. In this example we just want to list
 * the macro-Strings, so let's go
 */
BOOL cHelpMacros::NewTopicData(LPCSTR lpszTopicString, DWORD dwTopicOffset, TEXTTYPE eTextType)
{
    LPCSTR pTopicTitle;

    if (eTextType == TOPIC_MACRO_ENTRY ||
        eTextType == TOPIC_MACRO_HOTSPOT)
    {
        pTopicTitle = TopicTitle(dwTopicOffset);

        printf("-------------------------------------------\n");
        printf("Topic Offset: 0x%08lx\n", dwTopicOffset);
        printf("Topic Title : %s\n", pTopicTitle);
        printf("Macro       : %s\n", lpszTopicString);
        printf("Macro Type  : %s\n", eTextType == TOPIC_MACRO_ENTRY ? "TOPIC_MACRO_ENTRY" : "TOPIC_MACRO_HOTSPOT"); 

        free((void*)pTopicTitle);
    }

    return TRUE;
}


/* =================== Program MAIN part ====================== */
main(int argc, char **argv)
{ cHelpMacros *pHelpMacros;
  DWORD        TopicOffset=0;

  if (argc>=3)
    sscanf(argv[2], "%lx", &TopicOffset);

  if (argc>=2)
    { pHelpMacros = new cHelpMacros(argv[1]);

      if (pHelpMacros)
      {

         if (pHelpMacros->IsHelpFile())
         {
           pHelpMacros->TopicSeek(TopicOffset);
           pHelpMacros->TopicEnumerate();
         }
         else
           printf("File does not exist or is not a help file\n");

         delete pHelpMacros;
      }
    }
  else
    printf("TCPP    : Simple Example program for Help Access Library\n"
           "          Shows usage of Help Access Library with C++\n"
           "          Enumerates the Macros used in a Help File\n"
           "USAGE   : TCPP Filename [TopicOffset]\n"
           "EXAMPLE : TCPP c:\\windows\\winfile.hlp\n"
           "INFO    : Use example-Program TENUMTTL to retrieve valid TopicOffsets\n"
           "(C)1996 Herd Software Development,Rudolf-Virchow-Str. 8/68642 Buerstadt/Germany");

  return 0;
}
