/* HLPACCES.H ---------------------------------------------------------------------
 *
 * @doc
 *
 * @module HLPACCES.H |
 *
 * Include-File for programs that use the Herd Software help file access
 * library HLPACC16.DLL or HLPACC32.DLL.
 *
 *
 * (C) 1996 Herd Software Development
 * Rudolf-Virchow-Str. 8/68642 Brstadt / Germany
 *
 *---------------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#endif


#ifndef __HLPACCES_H
#define __HLPACCES_H


/* =================== Internal Declarations ================================= */

/* The following Header-File is only included during compilation of the DLL
 * source code by Herd Software Development. It is not part of the retail package and
 * not reqired to use the DLL
 */

#ifdef CREATE_HLPACCES_DLL
# include "whstruct.h"
# include "ifs.h"
#else
/* Define us a Handle for opened Files... */
DECLARE_HANDLE(HIFS);                       // @type HIFS     | Handle to an open File-System (.HLP, .MVB or .ANN-File)
DECLARE_HANDLE(HIFSFILE);                   // @type HIFSFILE | Handle to a file inside of a File-System, like a Baggage-File or a included SHG-File etc.
DECLARE_HANDLE(HTOPIC);                     // @type HTOPIC   | Handle to an opened TOPIC-File retrieved vom <f HlpTopicOpen> and closed by <f HlpTopicClose>    
#endif






/* =========== Exported File-Accessing functions ============================ */

/* IFSOpen
 *
 * Opens an existing .HLP or .MVB and reads in Header-Level Data structures
 */
HIFS WINAPI IFSOpen(LPCSTR HelpFileName);


/* IFSClose
 *
 * Closes Down .HLP-File and frees assigned Structures
 */
void WINAPI IFSClose(HIFS ifs);


/* @cb BOOL CALLBACK | IFSENUMPROC |
 * Application-defined callback-Function to be used for calls to <f IFSEnumerateFiles>.
 * IFSEnumerateFiles calls this function for every Filename retrieved from a file System
 *
 * @parm LPCSTR | lpszFileName | File name of the internal File
 * @parm DWORD  | dwFileOffset | Offset of the file in the physical Help-File
 * @parm LPARAM | lParam       | Application-Defined parameter transfered with the call to <f IFSEnumerateFiles>  
 */
  typedef BOOL (CALLBACK * IFSENUMPROC)(LPCSTR lpszFileName, DWORD dwFileOffset, LPARAM lParam);

/* IFSEnumerateFiles
 *
 * Enumerates the files contained in the File System.
 * Files starting with a "|"-Character are usually automatically added by the
 * Compiler, while other files are usually added by the [Baggage]
 */

void WINAPI IFSEnumerateFiles(HIFS ifs, IFSENUMPROC fnCallback, LPARAM lParam);


/* IFSOpenFile
 *
 * Opens an internal File
 *
 */
HIFSFILE WINAPI IFSOpenFile(HIFS ifs, LPCSTR InternalFileName);


/* IFSCloseFile
 *
 * Closes an internal File
 *
 */
void WINAPI IFSCloseFile(HIFSFILE ifsfile);


/* IFSSeekFile
 *
 * Positions an Internal File
 *
 */
long WINAPI IFSSeekFile(HIFSFILE ifsfile, DWORD pos, int whence);


/* IFSReadFile
 *
 * Reads data from an internal File
 *
 */
long WINAPI IFSReadFile(HIFSFILE ifsfile, LPVOID dest, DWORD count);


/* HlpGenerateHashValue
 *
 * Computes a Hash-Code that sinifies the given ContextSstring
 */
DWORD WINAPI HlpGenerateHashValue(LPCSTR p);


/* HlpIsTopic
 *
 * Tests if the given Hash-Code is in the list
 * of hash-Codes in the given File.
 *
 * Returns: TopicOffset Value, or 0xffffffff if not found.
 */
DWORD WINAPI HlpIsTopic(HIFS ifs, DWORD HashCode);



/* IsTopic
 *
 * The main Topic-Test Function
 *
 */
BOOL WINAPI IsTopic(LPCSTR FileName, LPCSTR ContextString);


/* IsSecondaryWindow
 *
 * Tests if the given Help-File contains a Definition of the diven Secondary Window-Name
 *
 */
BOOL WINAPI IsSecondaryWindow(LPCSTR FileName, LPCSTR WindowName);


/* ==== Enumeration of Keywords ===================== */

/* @cb BOOL CALLBACK | KWENUMPROC |
 * Application-defined callback-Function to be used for calls to <f HlpEnumKeyWords>.
 * HlpEnumKeyWords calls this function for every Keyword retrieved from a Help or Media Viewer File
 *
 * @rdesc The Callback-Function cancels the Enumeration by return 0, every other
 *        value continues enumeration
 *
 * @parm LPCSTR | lpszKeyWord  | Keyword retrieved
 * @parm DWORD  | dwDataOffset | Offset of the Keyword into the Keyword-Datafile (KWDATA, 0KWDATA or AKWDATA...)
 *                               where the TopicOffsets for use with <f HlpTopicSeek> can be retrieved
 * @parm UINT   | uDataCount   | Number of uses of this Keyword in the File. 
 * @parm LPARAM | lParam       | Application-Defined parameter transfered with the call to <f IFSEnumerateFiles>
 *
 * @comm The Informations retrieved from the KeyWord Binary Tree contain the Keyword strings in an
 *       alphabetically sorted list and a Reference to the Keyword Data File (KWDATA, 0KWDATA, AKWDATA...)
 *
 *       The KWDATA file contains a list of DWORD-Values for the TopicOffsets linkes to the Topics
 *       that used these Keyword. To jump to a Topic using <f HlpTopicSeek> or <f HlpTopicJumpAPI>
 *       You first have to open the KWDATA-File, read the Topic-Offset(s) and use these for the Jump.
 */
  typedef BOOL (CALLBACK * KWENUMPROC)(LPCSTR lpszKeyWord, DWORD dwDataOffset, UINT uDataCount, LPARAM lParam);



/* HlpEnumKeyWords
 *
 * This Function enumerates the List of Keywords from a Help or Media Viewer
 * Tile. This Function is relatively fast.
 *
 * Note that one Help or Media Viewer file can Have several Keyword lists.
 * Usually in Help Files the Keyword list is named "KWBTREE" in Media Viewe it's
 * Named "0KWBTREE", the ALink-List in Windows 95 Files is named "AWBTREE", others are
 * also possible.
 *
 * Return Value:
 * TRUE if the Function was Successfull
 * FALSE else
 */
BOOL WINAPI HlpEnumKeyWords(HIFS ifs, KWENUMPROC fnKWEnumProc, LPCSTR StartKeyWord, LPCSTR KWListName, LPARAM lParam);






/* ==== Enumeration of Topics and Topic-Titles ====== */

/* @cb BOOL CALLBACK | TITLEENUMPROC |
 *
 * Type-Definition for an application-supplied callback-Function
 * that will be called each time a Title has been retrieved from the
 * Help-File by the <f HlpEnumTitles> function.
 *
 * @rdesc The Callback-Function cancels the Enumeration by return 0, every other
 *        value continues enumeration
 *
 * @comm The number of enumerated Names can be huge, depending on the
 *       .MVB - or .HLP File.
 *
 * @comm The enumeration is sorted by the value of TopicOffset
 *
 * @comm As the Group-Definition Contained in a Media Viewer 2.0-File is
 *       not itself stored in the TTLBTREE internal File, this Function will
 *       not receive it. The <f HlpTopicEnumerate> Function can be used to
 *       retrieve these strings for one or all topics.
 *
 * @parm LPCSTR | lpszTitle      | Title-String retrieved from Help-File TTLBTREE
 * @parm LPCSTR | lpszDummy      | Dummy-String variable might eventually be used
 *                                 for Group names in the future and is currently always NULL
 * @parm DWORD  | dwTopicOffset  | Topic-Offset into the TOPIC-File.
 * @parm LPARAM | lParam         | Application-Defined parameter transfered to <f HlpEnumTitles>
 */
typedef BOOL (CALLBACK * TITLEENUMPROC)(LPCSTR lpszTitle, DWORD dwTopicOffset, LPARAM lParam);

     /* A typical definition of this application-defined function in C will look
	the following:

	BOOL FAR PASCAL _export TitleEnumerationProcedure(LPCSTR lpszTitle, LPCSTR lpszDummy, DWORD dwTopicOffset, LPARAM lParam)
	{
	  printf("Title:%ls\tTopicOffset:%08lx", lpszTitle, dwTopicOffset);
	  return TRUE;
	}
     */


/* HlpEnumTitles
 *
 * This Function enumerates the List of titles to the application-Given
 * callback Function.
 *
 * Return Value:
 * TRUE if the Function was Successfull
 * FALSE else
 */
BOOL WINAPI HlpEnumTitles(HIFS ifs, TITLEENUMPROC fnTitleEnumProc, LPARAM lParam);


/* HlpGetTopicTitle
 *
 * Retrieves the Topic Title for a given TopicOffset Value.
 * The TopicOffset does not neccessarily need to point directly
 * to the start of the Topic.
 *
 * @rdesc
 * TRUE if the Function was Successfull
 * FALSE else
 */
BOOL FAR PASCAL HlpGetTopicTitle(
                HIFS ifs,                       // Handle to a opened File-System opened by <f IFSOpen>
                DWORD dwTopicOffset,            // TopicOffset to search for
                LPSTR lpszBuffer,               // Pointer to a Buffer to copy the retrieved Title String or NULL if not needed
                UINT uBufferSize,               // Size of the Bufffer in Bytes
                LPDWORD lpdwTopicOffsetStart);  // Pointer to a DWORD that will contain the TopicOffset value for the begin of the Topic or NULL if this is not needed


/* =================== Get Help-File title informations ============================= */


// ---------- Get Title and Copyright ... ----------------
/* @struct HELPFILETITLE |
   a Memory-Object allocated and filled by <f HlpGetHelpTitle> and Freed by the
   <f HlpFreeHelpTitle>.
*/
typedef struct
        {
           LPSTR        HelpTitle;      /* @field Title of Help-File */
           LPSTR        Citation;       /* @field Citation of Help-File */
           LPSTR        Copyright;      /* @field Copyright of Help-File */

           BYTE         Version;        /* @field Version # of Help Compiler  */
           BYTE         Revision;       /* @field Revision code of Help Compiler  */
           WORD         Flags;          /* @field Compression-related Flag-Values */

           UINT         GroupCount;     /* @field Number of Groups defined in an Media Viewer 2.0-File */
           LPSTR FAR   *GroupNames;     /* @field Names and descriptions of the Groups, or NULL of no groups defined or no Media Viewer 2.0 file. */

        } HELPFILETITLE, FAR *LPHELPFILETITLE;

/* HlpGetHelpTitle
 *
 * Allocates a memory-Object with Informations read from a Help-Files SYSTEM-File
 * The memory object must be freed with HlpFreeHelpTitle afterwards.
 */
LPHELPFILETITLE WINAPI HlpGetHelpTitle(HIFS ifs);
void            WINAPI HlpFreeHelpTitle(LPHELPFILETITLE tit);


 
/* =================== Get Help topic contents ====================================== */

/* @enum TEXTTYPE |
 *
 * Topic text is transfered to the Application based on a Application-Defined Callback
 * procdedure. The callback-Function receives a Block of text Data from the Topic.
 *
 * The eTextType enumeration code tells the Application about the Type of Informations
 * retrieved.
 */
typedef enum {
        TOPIC_TEXT,                     /* @emem Transfered Data is normal Topic Text */
        TOPIC_TEXT_CONTROL,             /* @emem Transfered Data is Additional control-code, like Carriage Return, Tab etc. */ 
        TOPIC_TITLE,                    /* @emem Transfered Data is a Topic Title ('$'-Footnote) */
        TOPIC_TITLE_GROUP,              /* @emem Transfered Data is the group-name in case of a Media Viewer 2.0 MVB-File,
                                                                    This does not currently apply to Media View files which are renamed to ".MVB",
                                                                    like Development Network files. */
        TOPIC_MACRO_ENTRY,              /* @emem Transfered Data is a Macro-String defined in a "!"-Footnote */
        TOPIC_MACRO_HOTSPOT,            /* @emem Transfered Data is a Macro-String defined in a Hotspot (!-Hidden Text) */
        TOPIC_RTF_INIT,                 /* @emem Transfered Data is additional Formatting Information in RTF-Coding.
                                                                    The Data is Initialization Table Data such as colortbl etc. and should be
                                                                    contained in a Help File only once. */
        TOPIC_RTF_TEXT,                 /* @emem Transfered Data is additional Formatting Information in RTF-Coding. The Data is part of the Text formatting of Paragraphs, Tabels etc. */
        TOPIC_RTF_EXIT,                 /* @emem Transfered Data is additional Formatting Information in RTF-Coding.
                                                                    The Data is Deinitialization Data, usually no more than just "}" and should be
                                                                    contained in a Help File only once. */
        TOPIC_EWX,                      /* @emem Transfered Dats is { ewl ...}, { ewc ...}, { ewr ...} embedded Window String */
        TOPIC_BMX,                      /* @emem Transfered Data is { bml ...}, { bmr ...}, { bmc ...} Bitmap String. Filename: Internal File System Filename */
        TOPIC_MCI,                      /* @emem Transfered Data is MCI-Control string { mci ...} */
        TOPIC_BUTTON,                   /* @emem Transfered Data is Windows 95 { button...} string command */
        TOPIC_CONTEXTID,                /* @emem Transfered Data is a Context-Id (#-Footnote) */
        TOPIC_KEYWORD,                  /* @emem Transfered Data is a Keyword-Entry (K-Footnote) Format: Table-Name tab List of Keywords */
        TOPIC_HOTSPOT_START,            /* @emem Transfered Data is Hotspot at the starting-Point of the Jump-Hotspot. */
        TOPIC_HOTSPOT_END               /* @emem Transfered Data is Hotspot at the ending-Point of the Jump or Macro-Hotspot. */
        } TEXTTYPE;



/* @cb BOOL CALLBACK | TOPICENUMPROC |
 *
 * Application-Defined callback-routine called by <f HlpEnumerateTopic> whenever
 * Topic-Text is to be transfered to the Application.
 *
 * @rdesc The Callback-Function cancels the Enumeration by return 0, every other
 *        value continues enumeration
 */

typedef BOOL (CALLBACK * TOPICENUMPROC)(
                LPCSTR    lpszTopicText,   // @parm Text to be transfered to the Application program
                DWORD     dwTopicOffset,   // @parm TopicOffset where these Informations have been retrieved (not exact)
                TEXTTYPE  eTextType,       // @parm <t TEXTTYPE> type of Text-String transfered to the Application
                LPARAM    lParam);         // @parm Application-Defined user Data



/* HlpTopicOpen
 *
 * Opens the internal Help TOPIC-File and reads in and decompresses secondary
 * Tables, like Phrases, Keywords etc.
 *
 * The returned Handle need to be freed by calling <f HlpTopicClose>.
 * HlpTopicOpen is a rather slow operation, so when enumerating multiple Topics it's
 * a good idea to do them all using the same HTOPIC-Handle
 *
 * The return Value will be NULL if the File cannot be opened, for example due to
 * the fact that the IFS is not a Help-File.
 *
 * The HlpTopicEnumerate-Function does not stop enumeration until the
 * Callback-Functions tells it to stop by returning Zero.
 *
 * Flag-Values currently need to be Zero.
 */

#define HTO_FONTS       0x00000001L             // Load Fonts-file
#define HTO_KEYWORDS    0x00000002L             // Load Keywords file
#define HTO_CONTEXT     0x00000004L             // Load Context-File (#-footnotes)

HTOPIC WINAPI HlpTopicOpen(HIFS ifs, DWORD flags);


/* HlpTopicSeek
 *
 * Positions the TOPIC-Pointer to the given TopicOffset
 */
void WINAPI HlpTopicSeek(HTOPIC htopic, DWORD TopicOffset);


/* HlpTopicEnumerate
 *
 * Enumerates a topics contents to the Application-Defined Callback function
 *
 */
BOOL WINAPI HlpTopicEnumerate(HTOPIC htopic, TOPICENUMPROC fnenum, LPARAM lParam);



/* HlpTopicClose
 *
 * Closes a Handle opened by HlpTopicOpen
 */
void WINAPI HlpTopicClose(HTOPIC topic);



/* ================== Jump to a Help or Media Viewer Topic ================= */

/* HlpTopicJumpAPI
 *
 * Tries to launch Windows Help or Media Viewer 2.0 to display the
 * topic accessed with HlpTopicSeek.
 *
 * This is not always possible.
 *
 * The return Value is TRUE when the Function successfully started a Viewer,
 * it is FALSE if either the Media Viewer was not installed or the Windows
 * Help topic has no context Id to jump to.
 *
 */
BOOL WINAPI HlpTopicJumpAPI(LPCSTR lpszHelpFileName, HTOPIC topic);




/* @cb BOOL CALLBACK | CONTEXTNUMPROC |
 *
 * Type-Definition for an application-supplied callback-Function
 * that will be called each time a Hash-Value has been retrieved from the
 * Help-File by the <f HlpEnumContex> function.
 *
 * @rdesc The Callback-Function cancels the Enumeration by return 0, every other
 *        value continues enumeration
 *
 * @comm The enumeration is sorted by the HashValues
 *
 * @parm DWORD  | dwHashValue    | Hashcode for the Context-ID String (#-Footnote) a valid context-String for this value may be created by the <f HlpCreateStringForHashValue> function.
 * @parm DWORD  | dwTopicOffset  | Topic-Offset into the TOPIC-File.
 * @parm LPARAM | lParam         | Application-Defined parameter transfered to <f HlpEnumTitles>
 */
typedef BOOL (CALLBACK * CONTEXTENUMPROC)(DWORD dwHashValue, DWORD dwTopicOffset, LPARAM lParam);

/* HlpEnumContext
 *
 * Enumerates all the Context-Id-Hash-Values defined in a Help-File
 * By reading thru all of the CONTEXT internal File.
 *
 * TRUE if enumeration has been performed successfully.
 */
BOOL WINAPI HlpEnumContext(
            HIFS ifs,                        // Handle to a opened File-System opened by <f IFSOpen>
            CONTEXTENUMPROC fnTitleEnumProc, // CONTEXTENUMPROC Application-Defined Callback-Procedure to be called for every context-Id found in the help file
            LPARAM lParam);                  // Aplication-defined parameter to be transfered to the Callback-function


/* HlpCreateStringForHashValue
 *
 * Creates a String that has the same Hash-Code and can be used
 * to Transfer hash-code data to Winhelp as a context-String
 * for example in JumpId()-Macros and "-i"-command line Parameter.
 */
void WINAPI HlpCreateStringForHashValue(
        DWORD HashValue,                        // Hash-Code that shall be decoded
        LPSTR Buffer);                          // Buffer for the decoded hash-Value. Minimum Size is 20 Bytes.


#endif // __HLPACCES_H


#ifdef __cplusplus
}
#endif


