// main source for "meminfo"
// usage: "meminfo {msec}"
// where "msec" is an optional parameter specifying the update-frequency in milliseconds
// (default is 500 msec)
//
// Displays system status in 4 parts:
// at top RAM (field is "installed RAM"; black part "used RAM"; orange part "last 4 MB", which NT is trying to keep free by swapping as necessary)
// 	Numbers at right are "used RAM" and "installed RAM" in MB
// below pagefile (space is best guess for "maximum pagefile size"; field is "current pagefile size"; black part is "used in pagefile")
//	Numbers at right are "used pagefile" and "current pagefile size" in MB
// below ist "pagefile activity" indicator (normally active when "last 4 MB" of RAM being freed)
// below "% CPU usage" (red part "kernel mode"; blue part "user mode"; black part "unknown what used for, but used"
//	Number at right are: "% kernel mode", "% user mode", "% unknown activity", "% overall activity"
//	("unknown activity" is simply the comuted difference between the sum of user and kernel compared to overall)

// programm consists of a dialog; on WM_TIMER values are gained via GlobalMemoryStatus() and
// performance data in registry; displayed on "BlakenCustom"s (see balken.c)


#include <windows.h>
#include <windowsx.h>
#include <winperf.h>
#include <regstr.h>
#include <stdio.h>
#include <stdlib.h>

#include "balken.h"
#include "meminfo.rc"

#define HANDLE_DLG_MSG(hwnd, message, fn)    \
    case (message): return SetDlgMsgResult((hwnd), (message), HANDLE_##message((hwnd), (wParam), (lParam), (fn)))
	// missing in windowsx.h


static HINSTANCE thisProg;


static void main_OnCommand(const HWND dlg, const int id, const HWND hwndCtl, const UINT codeNotify)
{
	if(id==IDCANCEL) EndDialog(dlg,0);
}


static double fmin(const double a, const double b)
	// "min" normally is a macro which evaluates 1 parameter twice
	// whereas this function has a fine chance of getting optimized away
{
	return a<b ? a : b;
}


static void SetDlgItemDouble(const HWND item, const double value)
{
	char buffer[100];
	sprintf(buffer,"%#.4g",value);
	FORWARD_WM_SETTEXT(item,buffer,SendMessage);
}


// dialog position stored in registry
#define regstr_path "SOFTWARE\\ameise\\meminfo"
#define regstr_x_pos "X-Pos"
#define regstr_y_pos "Y-Pos"

static void get_old_pos(POINT *p)
{
	HKEY hk;
	p->x = p->y = -1;
	if(!RegOpenKeyEx(HKEY_CURRENT_USER,regstr_path,0,KEY_QUERY_VALUE,&hk))
	     { // Konfiguration gefunden, auslesen
	       DWORD len;
	       RegQueryValueEx(hk,regstr_x_pos,0,0,(LPBYTE)&p->x,(len=sizeof(p->x),&len));
	       RegQueryValueEx(hk,regstr_y_pos,0,0,(LPBYTE)&p->y,(len=sizeof(p->x),&len));
	       RegCloseKey(hk);
	     }
}

static void set_akt_pos(const POINT *p)
{
	HKEY hk;
	if (!RegCreateKey(HKEY_CURRENT_USER,regstr_path,&hk))
		{ RegSetValueEx(hk, regstr_x_pos, 0, REG_DWORD, (LPBYTE)&(p->x), sizeof(p->x));
		  RegSetValueEx(hk, regstr_y_pos, 0, REG_DWORD, (LPBYTE)&(p->y), sizeof(p->y));
		  RegCloseKey(hk);
		}
}

#undef regstr_path
#undef regstr_x_pos
#undef regstr_y_pos



// these 64-bit-integers actually make me angry...

static DWORD diff64(const LARGE_INTEGER * const minuend, const LARGE_INTEGER * const subtrahend)
{
	DWORD minor = (DWORD)minuend->LowPart-(DWORD)subtrahend->LowPart,
	      major = (DWORD)minuend->HighPart-(DWORD)subtrahend->HighPart;
	if(major) minor = (minor+(~(DWORD)0))+1;
	return minor;
}



static BOOL CALLBACK MainDlgProc(HWND dlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static HCURSOR hand_cursor = 0;
	static HWND children[num_children];	// for performance avoid GetDlgItem()
	static HBRUSH blue_brush=0, dark_red_brush = 0, light_red_brush = 0;
	static LARGE_INTEGER freq;
	
	switch(msg) { case WM_INITDIALOG: // init stuff - mostly providing brushes to the 4 "BalkenCustom"
		
					  hand_cursor = LoadCursor(thisProg,MAKEINTRESOURCE(idres_hand));
					  { POINT p; get_old_pos(&p);
					    if(p.x>=0 && p.y>=0) SetWindowPos(dlg,0,p.x,p.y,0,0,SWP_NOSIZE|SWP_NOZORDER|SWP_NOACTIVATE);
					  }
					  dark_red_brush = CreateSolidBrush(RGB(0xff,0,0));
					  light_red_brush = CreateSolidBrush(RGB(0xff,0x80,0));
					  blue_brush = CreateSolidBrush(RGB(0,0x80,0xff));
					  { int i; for(i=0;i<num_children;i++) children[i] = GetDlgItem(dlg,i); }
 					  SetWindowText(dlg,"MemInfo");	// zeigt Titel nicht, aber ohne tauchen wir nicht im Taskmanager auf
					  { HBRUSH brushes[6];
					    brushes[0] = dark_red_brush;
					    brushes[1] = blue_brush;
					    brushes[2] = GetStockBrush(BLACK_BRUSH);
					    brushes[3] = GetStockBrush(WHITE_BRUSH);
					    brushes[4] = 0;

    		      		            Balken_SetVal(children[id_cpu_balken], 0, 0);
    		      		            Balken_SetVal(children[id_cpu_balken], 1, 0);
    		      		            Balken_SetVal(children[id_cpu_balken], 2, 0);
    		      		            Balken_SetVal(children[id_cpu_balken], 3, 0);
    		      		            Balken_SetVal(children[id_cpu_balken], 4, 100);
					    SetzeBalkenFarben(children[id_cpu_balken],brushes);

					    brushes[0] = GetStockBrush(WHITE_BRUSH);
					    brushes[1] = GetStockBrush(BLACK_BRUSH);
					    brushes[2] = brushes[0];
					    brushes[3] = dark_red_brush;
					    brushes[4] = light_red_brush;
					    brushes[5] = 0;

    		      		            Balken_SetVal(children[id_ram_balken], 0, 0);
    		      		            Balken_SetVal(children[id_ram_balken], 1, 0);
    		      		            Balken_SetVal(children[id_ram_balken], 2, 1);
    		      		            Balken_SetVal(children[id_ram_balken], 3, 1);
    		      		            Balken_SetVal(children[id_ram_balken], 4, 1);
					    SetzeBalkenFarben(children[id_ram_balken],brushes+1);

					    Balken_SetVal(children[id_pf_balken], 0, 0);
    		      		            Balken_SetVal(children[id_pf_balken], 1, 0);
    		      		            Balken_SetVal(children[id_pf_balken], 2, 1);
    		      		            
					    {	// max. Pagefilesize - no secure method known, this is a "best guess"
					      HKEY hk;
					      char pf_info[1000];
					      DWORD len=sizeof(pf_info);
					      unsigned pf_size_max = 0;
					      if(!RegOpenKeyEx(HKEY_LOCAL_MACHINE,REGSTR_PATH_CURRENT_CONTROL_SET"\\Session Manager\\Memory Management",0,KEY_QUERY_VALUE,&hk))
					    	{ if(!RegQueryValueEx(hk,"PagingFiles",0,0,pf_info,&len))
					    		{ char *info = pf_info;
					    		  while(*info)
					    		  	{ char *valstr = strchr(info,' ');
					    		  	  if(valstr) { unsigned min_val = atoi(++valstr);
					    		  	  	       valstr = strchr(valstr,' ');
					    		  	  	       if(valstr) pf_size_max += atoi(++valstr);
					    		  	  	       else pf_size_max += min_val+50;
					    		  	  	     }
					    		  	  info = strchr(info,0)+1;
					    		  	}
					    		}
		    			          RegCloseKey(hk);
					    	}
    		      		              Balken_SetVal(children[id_pf_balken], 3, pf_size_max*1024*1024);
					    }
					    
    					    brushes[3] = GetStockBrush(LTGRAY_BRUSH);
					    brushes[4] = 0;
					    SetzeBalkenFarben(children[id_pf_balken],brushes+1);

					    Balken_SetVal(children[id_load_balken], 0, -100);
		      		            Balken_SetVal(children[id_load_balken], 1, 0);
		      		            Balken_SetVal(children[id_load_balken], 2, 0);
    		      		            Balken_SetVal(children[id_load_balken], 3, 100);
					    brushes[3] = 0;
					    SetzeBalkenFarben(children[id_load_balken],brushes);
					  }
					  
		      		          QueryPerformanceFrequency(&freq);
		      		          { extern int _argc;
		      		            extern char **_argv;
					    SetTimer(dlg,1,_argc>1?atoi(_argv[1]):500,0);
		      		          }
					  //return TRUE;		// get first result immediately
				          
		      case WM_TIMER: // checking memory and %CPU and massaging into Balken

		      			// memory
		      		     { MEMORYSTATUS ms;
		      		       LONG usedphys, usedpf;
		      		       ms.dwLength = sizeof(ms);
		      		       GlobalMemoryStatus(&ms);
		      		       if((LONG)ms.dwAvailPageFile<0) ms.dwAvailPageFile = 0;
		      		       if((LONG)ms.dwAvailPhys<0) ms.dwAvailPhys = 0;
		      		       usedphys = (LONG)ms.dwTotalPhys - (LONG)ms.dwAvailPhys;
		      		       usedpf = (LONG)ms.dwTotalPageFile - (LONG)ms.dwAvailPageFile;

		      		       Balken_SetVal(children[id_ram_balken], 1, fmin(usedphys,(LONG)ms.dwTotalPhys-4*1024*1024));
		      		       Balken_SetVal(children[id_ram_balken], 2, (LONG)ms.dwTotalPhys-4*1024*1024);
		      		       Balken_SetVal(children[id_ram_balken], 3, usedphys);
		      		       Balken_SetVal(children[id_ram_balken], 4, (LONG)ms.dwTotalPhys);
		      		       SetDlgItemDouble(children[id_ram_total],(double)(LONG)ms.dwTotalPhys/1024.0/1024.0);
		      		       SetDlgItemDouble(children[id_ram_avail],(double)usedphys/1024.0/1024.0);

		      		       Balken_SetVal(children[id_pf_balken], 1, usedpf);
		      		       Balken_SetVal(children[id_pf_balken], 2, (LONG)ms.dwTotalPageFile);
		      		       SetDlgItemDouble(children[id_pf_total],(double)(LONG)ms.dwTotalPageFile/1024.0/1024.0);
		      		       SetDlgItemDouble(children[id_pf_avail],(double)usedpf/1024.0/1024.0);

       		      		       Balken_SetVal(children[id_load_balken], 1, -(double)(LONG)ms.dwMemoryLoad);
       		      		       Balken_SetVal(children[id_load_balken], 2, (LONG)ms.dwMemoryLoad);

		      		     }

					// % CPU
		      		     { BYTE buffer[3000];
		      		       DWORD size = sizeof(buffer);
		      		       PERF_OBJECT_TYPE *pot;
		      		       PERF_COUNTER_DEFINITION *pcd;
		      		       LARGE_INTEGER *priv,*user,*entire, time;
		      		       static LARGE_INTEGER old_user,old_priv,old_entire;
		      		       static LARGE_INTEGER oldtime;
		      		       RegQueryValueEx(HKEY_PERFORMANCE_DATA, "2", 0, 0, buffer, &size);	// 2 is "system"
		      		       QueryPerformanceCounter(&time);
		      		       // skip header
		      		       pot = (PERF_OBJECT_TYPE*) (buffer + ((PERF_DATA_BLOCK*)buffer)->HeaderLength);
		      		       // Counter Definitions
		      		       pcd = (PERF_COUNTER_DEFINITION*)((PBYTE)pot + pot->HeaderLength);
		      		       // find % CPU (240), % user (242) and % priviliged (244)
		      		       priv=user=entire=0;
		      		       while(pot->NumCounters--)
		      		       	      { if(pcd->CounterNameTitleIndex==240)		// % CPU
		      		       	      		entire = (LARGE_INTEGER*)((PBYTE)pot + pot->DefinitionLength+pcd->CounterOffset);
		      		       		else if(pcd->CounterNameTitleIndex==242)		// % user
		      		       	      		user = (LARGE_INTEGER*)((PBYTE)pot + pot->DefinitionLength+pcd->CounterOffset);
		      		       		else if(pcd->CounterNameTitleIndex==244)		// % privileged (kernel)
		      		       	      		priv = (LARGE_INTEGER*)((PBYTE)pot + pot->DefinitionLength+pcd->CounterOffset);
		      		       	      	if(priv && user && entire) break;
		      		       		else pcd = (PERF_COUNTER_DEFINITION*)((PBYTE)pcd + pcd->ByteLength);
		      		       	      }
		      		       if(priv && user && entire)	// should always be there, but better watch out
			      		       { double priv_diff = diff64(priv,&old_priv);
			      		       	 double user_diff = diff64(user,&old_user);
			      		       	 double ent_diff = diff64(entire,&old_entire);
			      		       	 double time_diff = (double)diff64(&time,&oldtime)/freq.LowPart*1e7;	// in 100ns
			      		       	 double priv_usage = 100*priv_diff/time_diff;
			      		       	 double user_usage = 100*user_diff/time_diff;
			      		       	 double ent_usage = 100-100*ent_diff/time_diff;		// % CPU is actually % idle, so compute 1-value to get it right
			      		       	 { char buff[30]; 
						   FORWARD_WM_SETTEXT(children[id_priv_usage],itoa(priv_usage+.5,buff,10),SendMessage);
						   FORWARD_WM_SETTEXT(children[id_user_usage],itoa(user_usage+.5,buff,10),SendMessage);
						   FORWARD_WM_SETTEXT(children[id_lost_usage],itoa(ent_usage-user_usage-priv_usage+.5,buff,10),SendMessage);
						   FORWARD_WM_SETTEXT(children[id_ent_usage],itoa(ent_usage+.5,buff,10),SendMessage);
			      		       	 }
					         Balken_SetVal(children[id_cpu_balken],1,fmin(priv_usage,100));
					         Balken_SetVal(children[id_cpu_balken],2,fmin(priv_usage+user_usage,100));
					         Balken_SetVal(children[id_cpu_balken],3,fmin(ent_usage,100));
					         oldtime=time; old_priv=*priv; old_user=*user; old_entire=*entire;
			      		       }
		      		     }

		      		     RedrawWindow(dlg,0,0,RDW_UPDATENOW|RDW_ALLCHILDREN);
		      		     return TRUE;

		      HANDLE_DLG_MSG(dlg,WM_COMMAND,main_OnCommand);
		      case WM_CLOSE: EndDialog(dlg,0); return TRUE;
		      case WM_DESTROY: KillTimer(dlg,1);
		      		       DeleteBrush(blue_brush);DeleteBrush(light_red_brush);DeleteBrush(dark_red_brush);
		      		       RegCloseKey(HKEY_PERFORMANCE_DATA);
		      		       { RECT r; GetWindowRect(dlg,&r); set_akt_pos((POINT*)&r); }
		      		       return TRUE;
		      case WM_NCHITTEST: return SetDlgMsgResult(dlg, msg, HTCAPTION);
		      case WM_SETCURSOR: SetCursor(hand_cursor); return TRUE;
		      case WM_CTLCOLORSTATIC: SetBkMode((HDC)wParam,TRANSPARENT);
		      case WM_CTLCOLORDLG: return SetDlgMsgResult(dlg, msg, GetStockObject(LTGRAY_BRUSH));
		      default: return FALSE;
		    }
}


int WINAPI WinMain(HINSTANCE inst, HINSTANCE prev, LPSTR cmdline, int cmdshow)
{
	thisProg = inst;
	SetPriorityClass(GetCurrentProcess(),HIGH_PRIORITY_CLASS);
	return DialogBox(inst,MAKEINTRESOURCE(idres_main_dlg),0,MainDlgProc);
}

