/******************************************************************************
*                                                                             *
* TREEMENU.C: 9/28/96 by Mark Gamber                                          *
*                                                                             *
* This example demonstrates how to subclass a TreeView control and trap right *
*    mouse button messages to display a menu relative to the context of the   *
*    treeview item selected.                                                  *
*                                                                             *
* By using this program or any part of the program, the user assumes full     *
*    responsibility for it's use and may not hold the author liable for any   *
*    loss or damage. If unable to accept this restriction, the program and    *
*    any part of the program must be destroyed immediately and may not be     *
*    used in any part for any purpose. The program and any part of the        *
*    program may be freely distributed.                                       * 
*                                                                             *
******************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "commctrl.h"

// === Function Prototypes ====================================================

BOOL WINAPI TVMenuDlgProc( HWND, UINT, WPARAM, LPARAM );
LRESULT WINAPI NewTVWndProc( HWND, UINT, WPARAM, LPARAM );
BOOL AddTreeBitmaps( HWND );
BOOL AddTreeItems( HWND );

// === Global Variables =======================================================

HINSTANCE hInst;                                //  Application instance handle
int iImage[ 2 ];                             //  Offsets of treeview image list

// === Program Entry Point ====================================================

int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd,
                      int nShow )
{
   MSG msg;
   HWND hDlg;


   hInst = hInstance;                        //  Save instance handle in global
   InitCommonControls();                     //  and initialize common controls
                                             //  Start the main dialog modeless
   hDlg = CreateDialog( hInst, MAKEINTRESOURCE( 10000 ), NULL, TVMenuDlgProc );
   if( ! hDlg )
      return( FALSE );                    //  Quit program if dialog box failed

   while( GetMessage( &msg, NULL, 0, 0 ) )            //  Do message loop thang
   {
      if( IsDialogMessage( hDlg, &msg ) )     //  Process Tab messages and such
         continue;

      TranslateMessage( &msg );                      //  Process other messages
      DispatchMessage( &msg );
   }
   return( FALSE );
}                                                          //  Exit application

// === Main Window Procedure ==================================================

BOOL WINAPI TVMenuDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:                    //  When the dialog box is created
      {
         if( ! AddTreeBitmaps( GetDlgItem( hDlg, 100 ) ) )     //  Create image
         {                                //  list for treeview control or exit
            DestroyWindow( hDlg );
            PostQuitMessage( 0 );
         }
         if( ! AddTreeItems( GetDlgItem( hDlg, 100 ) ) )       //  Add items to
         {                                     //  the treeview control or exit
            DestroyWindow( hDlg );
            PostQuitMessage( 0 );
         }                      
                                //  Next, set the treeview control proc address
         SetProp( GetDlgItem( hDlg, 100 ), "oldproc",  //  as a property of the
                  (HANDLE)GetWindowLong( GetDlgItem( hDlg, 100 ),  //  treeview
                                         GWL_WNDPROC ) );  //  so it can get it
         SetWindowLong( GetDlgItem( hDlg, 100 ), GWL_WNDPROC,
                        (DWORD)NewTVWndProc );    //  Since it's now subclassed
         return( TRUE );
      }


      case WM_COMMAND:
         if( wParam == IDCANCEL )              //  Exit if dialog box is closed
         {                          
            DestroyWindow( hDlg );
            PostQuitMessage( 0 );
            break;
         }
         if( wParam == 1000 )                        //  TreeView menu messages
         {
            SetDlgItemText( hDlg, 101, "TreeView message received" );
            break;
         }
         break;
   }
   return( FALSE );
}                                                    //  End of TVMenuDlgProc()

// === Create ImageList for TreeView Control ==================================

BOOL AddTreeBitmaps( HWND hTree )
{
   HIMAGELIST hImage;
   HBITMAP hBmp;
                                                  
   hImage = ImageList_Create( 15, 15, ILC_COLOR, 2, 0 );  //  Create image list
   if( hImage == NULL )                             //  Do bad exit if it fails
      return( FALSE );
                                                //  Load 2 bitmaps and add them
   hBmp = LoadBitmap( hInst, MAKEINTRESOURCE( 1000 ) );   //  to the image list
   iImage[ 0 ] = ImageList_Add( hImage, hBmp, (HBITMAP)NULL ); 
   DeleteObject( hBmp );                                //  Delete after adding

   hBmp = LoadBitmap( hInst, MAKEINTRESOURCE( 1001 ) ); 
   iImage[ 1 ] = ImageList_Add( hImage, hBmp, (HBITMAP)NULL ); 
   DeleteObject( hBmp ); 
                                   //  Apply image list to the treeview control
   TreeView_SetImageList( hTree, hImage, TVSIL_NORMAL );
   return( TRUE );
}

// === Add some stuff to the TreeView control =================================

BOOL AddTreeItems( HWND hTree )
{
   TV_ITEM tvi;
   TV_INSERTSTRUCT tvins;
   HTREEITEM hPrev;
   char szStr[ 64 ];
   int i;


   tvi.mask = TVIF_TEXT | TVIF_IMAGE | TVIF_SELECTEDIMAGE;
   tvi.pszText = szStr;
   tvi.iSelectedImage = iImage[ 0 ];
   tvi.iImage = iImage[ 1 ];
   tvins.item = tvi;
   tvins.hInsertAfter = NULL;
   tvins.hParent = TVI_ROOT;

   hPrev = NULL;

   for( i = 0; i < 20; i++ )
   {
      wsprintf( szStr, "Item #%d", i + 1 );   
      tvi.cchTextMax = strlen( szStr );

      hPrev = TreeView_InsertItem( hTree, &tvins );
   }

   return( TRUE );
}

// === TreeView Subclass Procedure ============================================

LRESULT WINAPI NewTVWndProc( HWND hWnd, UINT msg, WPARAM wParam,
                             LPARAM lParam )
{
   WNDPROC lpOldProc;

   lpOldProc = (WNDPROC)GetProp( hWnd, "oldproc" );
   
   switch( msg )
   {
      case WM_DESTROY:                         //  Put back old window proc and
         SetWindowLong( hWnd, GWL_WNDPROC, (DWORD)lpOldProc );
         RemoveProp( hWnd, "oldproc" );              //  remove window property
         break;

      case WM_RBUTTONDOWN:              //  If right mouse button is pressed...
      {
         TV_ITEM tvi;
         HMENU hMenu;
         char szStr[ 64 ];
         POINT pt;

                           //  Make it select the item like a left button click
         SendMessage( hWnd, WM_LBUTTONDOWN, wParam, lParam );
                                                  //  Now get the item selected
         tvi.hItem = TreeView_GetSelection( hWnd );
         tvi.mask = TVIF_TEXT;                        //  We want the item text
         tvi.pszText = szStr;
         tvi.cchTextMax = 64;

         if( TreeView_GetItem( hWnd, &tvi ) )
         {                             //  If we were able to get the item text
            hMenu = CreatePopupMenu();              //  Create a new popup menu
            if( hMenu )
            {                    //  If the menu was created, add the item text
               AppendMenu( hMenu, MF_STRING | MF_ENABLED, 1000, szStr );
               AppendMenu( hMenu, MF_SEPARATOR, (UINT)-1, NULL ); // and Cancel
               AppendMenu( hMenu, MF_STRING | MF_ENABLED, 1001, "Cancel" );

               GetCursorPos( &pt );             //  Get the cursor position and 
               TrackPopupMenu( hMenu, TPM_LEFTALIGN | TPM_RIGHTBUTTON, 
                               pt.x, pt.y, 0, hWnd, NULL );  //  start the menu
               DestroyMenu( hMenu );                //  Kill the menu when done
            }
         }
         return( FALSE );     //  Don't let WM_RBUTTONDOWN messages to treeview
      }

      case WM_COMMAND:
      {
         if( wParam == 1000 )                //  If item menu item was selected
         {                               //  Relay message to parent dialog box
            SendMessage( GetParent( hWnd ), WM_COMMAND, 1000, 0 );
            return( FALSE );                         //  Don't pass to treeview
         }
      }
   }                        //  Pass all non-custom messages to old window proc
   return( CallWindowProc( lpOldProc, hWnd, msg, wParam, lParam ) );
}

// ============================================================================






