//***************************************************************
// From the book "Win32 System Services: The Heart of Windows NT"
// by Marshall Brain
// Published by Prentice Hall
//
// Copyright 1994, by Prentice Hall.
//
// This code implements a named pipe server that allows 
// communication among connected clients.
//***************************************************************

// mtnpserv.cpp

#include <windows.h>
#include <iostream.h>
#include <string.h>
#include <stddef.h>
#include <process.h>

#define BUFSIZE 128
#define MAX_INSTANCES 3

// Information needed for each connection
// to a client
typedef struct _CLIENT_INFO
{
  HANDLE h;
  OVERLAPPED overlappedRead;
  OVERLAPPED overlappedWrite;
} CLIENT_INFO;
  
// The array holds one entry for 
// each client
CLIENT_INFO clients[MAX_INSTANCES];
volatile int numClients=0;

// There is one thread for each client 
// attached to the server
void mtnpClientThread(HANDLE id)
{
  CHAR textBuffer[BUFSIZE];
  DWORD numBytesRead;
  DWORD numBytesWritten;
  int x;
  int ID = (int) id;

  while (1)
  {
    // Get input from the client
    if (!ReadFile(clients[ID].h,
      textBuffer, BUFSIZE,
      &numBytesRead,
      &(clients[ID].overlappedRead)))
    {
      if (GetLastError() != ERROR_IO_PENDING)
      {
        cerr << 
          "ERROR: Unable to read from named pipe "
          << GetLastError()
          << endl;
        break;
      }
    }
    GetOverlappedResult(clients[ID].h,
      &(clients[ID].overlappedRead), &numBytesRead,
      TRUE); 
    clients[ID].overlappedRead.Offset +=
      numBytesRead;

    // Convert the client's string to
    // upper case to show server is
    // doing something
    _strupr(textBuffer);

    // Send the message to ALL clients
    // in the client array
    for (x=0; x<numClients; x++)
    {
      if (!WriteFile(clients[x].h,
        textBuffer, strlen(textBuffer) + 1,
        &numBytesWritten, 
        &(clients[ID].overlappedWrite)))
      {
        if (GetLastError() != ERROR_IO_PENDING)
        {
          cerr 
            << "ERROR: Unable to write to pipe "
            << GetLastError()
            << endl;
          break;
        }
      }
      GetOverlappedResult(clients[ID].h,
        &(clients[ID].overlappedWrite), 
        &numBytesWritten,
        TRUE); 
      clients[ID].overlappedWrite.Offset += 
        numBytesWritten;
    }

    // Echo the string in the server's window
    cout << textBuffer << endl;
  } // while 

  // Clean up the pipe on termination
  // of the thread
  FlushFileBuffers(clients[ID].h);
  DisconnectNamedPipe(clients[ID].h);
  CloseHandle(clients[ID].h);
}

INT main(VOID)
{
  HANDLE mtnpPipe;
  DWORD mtnpThread;

  while (1)
  {
    // Create a named pipe for receiving messages
    mtnpPipe=CreateNamedPipe("\\\\.\\pipe\\mtnp",
      PIPE_ACCESS_DUPLEX | FILE_FLAG_OVERLAPPED,
      PIPE_TYPE_MESSAGE |PIPE_READMODE_MESSAGE|
      PIPE_WAIT,
      MAX_INSTANCES, 0, 0, 150,
      (LPSECURITY_ATTRIBUTES) NULL);

    // Check and see if the named pipe was created
    if (mtnpPipe == INVALID_HANDLE_VALUE)
    {
      cerr << "ERROR: Unable to create pipe"
        << endl;
      return (1);
    }

    // Allow a client to connect to the name pipe,
    // terminate if unsuccessful
    if (!ConnectNamedPipe(mtnpPipe, 
      (LPOVERLAPPED) NULL))
    {
      cerr << "ERROR: Unable to connect a pipe."
        << endl;
      CloseHandle(mtnpPipe);
      return (1);
    }

    // Init a client data structure. It contains 
    // a handle to the pipe and overlapped structs
    clients[numClients].h = mtnpPipe;
    clients[numClients].overlappedRead.Offset = 
      clients[numClients].overlappedWrite.Offset =
        0;
    clients[numClients].overlappedRead.OffsetHigh =
      clients[numClients].overlappedWrite.
        OffsetHigh = 0;
    clients[numClients].overlappedWrite.hEvent =
      CreateEvent(0, TRUE, FALSE, 0);
    clients[numClients].overlappedRead.hEvent =
      CreateEvent(0, TRUE, FALSE, 0);

    // Create a new thread to accept messages 
    // from the new client
    mtnpThread=_beginthread(mtnpClientThread,
      0, (HANDLE) numClients);
    if (mtnpThread == -1)
    {
      cerr << "ERROR: Unable to create thread"
        << endl;
      CloseHandle(mtnpPipe);
    }

    // increment the number of clients
    numClients++;

  } /* while */
}
