//
//  File.......: dbg.c
//
//  Description: Simple Win32 Debugger for displaying debug strings
//               to the command console.
//
//  Started On.: Tue  07-16-1995
//  Author.....: Chuck Price - price@dsr.com
//  Version....: 1.0
//
//  Modification History:
//
//  When       Who            What
//  --------   ------------   -----------------------------------------------
//   7/16/95   Chuck Price    Created
//

#define  STRICT
#include <windows.h>
#include <stdio.h>		// for printf()

// Internal Data
int ActiveProcesses = 0;

// Internal Function Prototypes
VOID  Logo( VOID );

int
main( int __argc, char** __argv )
{
    PROCESS_INFORMATION pi;        // process information
    STARTUPINFO         si;        // startup information
    DEBUG_EVENT         DebugEvent;
    BOOL                bNotDone = TRUE;

	if( __argc != 2 )
        Logo( );

	si.cb          = sizeof( STARTUPINFO );
	si.lpReserved  = NULL;
	si.lpDesktop   = NULL;
	si.lpTitle     = "Debugee";
	si.dwX         = 0;
	si.dwY         = 0;
	si.dwXSize     = 0;
	si.dwYSize     = 0;
	si.dwFlags     = (DWORD) NULL;
	si.wShowWindow = SW_SHOWDEFAULT;
	si.cbReserved2 = 0;
	si.lpReserved2 = NULL;

	pi.hProcess = NULL;

	// now create the debugee process
	if( !CreateProcess(
	     (LPCTSTR) NULL,
         __argv[1],
	     (LPSECURITY_ATTRIBUTES) NULL,
	     (LPSECURITY_ATTRIBUTES) NULL,
	     TRUE,
	     DEBUG_PROCESS,
	     (LPVOID) NULL,
	     (LPCTSTR) NULL,
	     &si, &pi ) )
	{
		return( (DWORD) -1L );
	}

    printf( "=> Start debugging %s\n", __argv[1] );

    while (bNotDone)
	{
		if( !WaitForDebugEvent( &DebugEvent, (DWORD) -1L ) )
		{
			printf( "=> Waiting...\n" );
			continue;
		}

		// handle minimal exception handling needs for particular events
		switch( DebugEvent.dwDebugEventCode )
		{
			case EXCEPTION_DEBUG_EVENT:
				switch( DebugEvent.u.Exception.ExceptionRecord.ExceptionCode )
				{
					case EXCEPTION_ACCESS_VIOLATION:
						printf( "=> EXCEPTION_ACCESS_VIOLATION\n" );
						ContinueDebugEvent( DebugEvent.dwProcessId, DebugEvent.dwThreadId,
							DBG_EXCEPTION_NOT_HANDLED );
				}
				break;

			case CREATE_PROCESS_DEBUG_EVENT:
				printf( "=> Process Created.\n" );
				ActiveProcesses++;
				break;

			// OutputDebugString() occured
			case OUTPUT_DEBUG_STRING_EVENT:
			{
				DWORD dwNumberOfBytesRead;
				TCHAR szOutputDebugString[256];

				ReadProcessMemory(
					pi.hProcess, DebugEvent.u.DebugString.lpDebugStringData,
					szOutputDebugString, DebugEvent.u.DebugString.nDebugStringLength,
					&dwNumberOfBytesRead );
				printf( TEXT( "%s" ), szOutputDebugString );
				break;
			}

			case EXIT_PROCESS_DEBUG_EVENT:
				printf( "=> Process Exited.\n" );
				CloseHandle( pi.hThread );
				CloseHandle( pi.hProcess );
				if( --ActiveProcesses == 0 )
                    bNotDone = FALSE;
				break;
		}

		// default action - just continue
		ContinueDebugEvent( DebugEvent.dwProcessId, DebugEvent.dwThreadId,
			DBG_CONTINUE );
    }

	printf( "=> Stop debugging %s\n", __argv[1] );

    return 0;
}


VOID
Logo( VOID )
{
    printf("\nWin32 Debugger, v1.0 - 7/25/95");
    printf("\nCharles D. Price - price@dsr.com");
	printf("\n\nSyntax: DBG debugee\n");

	ExitProcess( 0 );
}
