/*------------------------------------------------------------------------------
 *  DEMO.EXE	Demo Program for CKTBL Tabel Control using Windows SDK Api from C
 *	
 *  Copyright(C) 1994 by Christian Kratzer Software development and Consulting
 *
 *  File: Demo.c
 *
 *----------------------------------------------------------------------------*/

#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "demo.h"
#include "cktbl.h"

HANDLE hInst;
HANDLE hAccTable;          /*handle to accelerator table */
HWND hTblWnd;              /* handle to table control */
HWND hMainWnd;             /* handle to main window */


CHOOSECOLOR	CC_Foreground;
CHOOSECOLOR	CC_Background;
COLORREF CC_Custom[16];

#define APPLICATION_NAME			"CKTBL16.DLL - Demo"
#define APPLICATION_MENU			"MainMenu"
#define APPLICATION_ACCELERATORS	"MainMenuAcc"
#define APPLICATION_CLASS			"CKDemoWindow"

#define SIZEX 20
#define SIZEY 200

#define LINESTYLE_MASK (CKTBL_SINGLELINE|CKTBL_SINGLELINE_BOTTOM|CKTBL_MULTILINE|CKTBL_MULTILINEBREAK)

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void InitChooseColor( CHOOSECOLOR * cc )
{
	COLORREF clr;
	int i;

	for (i = 0; i < 16; i++) CC_Custom[i] = RGB(255, 255, 255);

	/* Initialize clr to black. */

	clr = RGB(0, 0, 0);

	/* Set all structure fields to zero. */

	memset(cc, 0, sizeof(CHOOSECOLOR));

	/* Initialize the necessary CHOOSECOLOR members. */

	cc->lStructSize = sizeof(CHOOSECOLOR);
	cc->hwndOwner = hMainWnd;
	cc->rgbResult = clr;

	cc->lpCustColors = CC_Custom;
	cc->Flags = 0;
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void DoInsertError()
{
	MessageBox( hMainWnd, "Please select exactly one row or column", APPLICATION_NAME, MB_OK );
}


void DoInsertRowColBeforeOrAfter( HWND hWnd, BOOL before )
{
	RECT	rect;
	int		nSel;

	nSel = CKTBLGetSelectionSize( hWnd );
	if(nSel!=1) {
		DoInsertError();
		return;
	}
	CKTBLGetSelection( hWnd, &rect, 1 );

	if(rect.left==0 && rect.top>0 ) {
		if(rect.top==rect.bottom) {
			if(before) rect.top--;
			CKTBLInsertRowsAfter( hWnd, rect.top , 1);
			return;
		}
	}

	if(rect.top==0 && rect.left>0 ) {
		if(rect.left==rect.right) {
			if(before) rect.left--;
			CKTBLInsertColumnsAfter( hWnd, rect.left , 1);
			return;
		}
	}

	DoInsertError();
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

void DoDeleteError()
{
	MessageBox( hMainWnd, "Please select either rows or columns", APPLICATION_NAME, MB_OK );
}


void DoDeleteRowCol( HWND hWnd )
{
	BOOL	rows,columns;
	RECT	rect;
	int		nSel;

	nSel = CKTBLGetSelectionSize( hWnd );
	if(nSel!=1) {
		DoDeleteError();
		return;
	}

	/* check that we have either rows or columns */

	rows=columns=FALSE;
	CKTBLGetSelection( hWnd, &rect, 1 );
	if((rect.left==0 && rect.top==0)||(rect.left!=0 && rect.top!=0)) {
		DoDeleteError();
		return;
	}
	if(rect.left==0) rows=TRUE;
	if(rect.top==0) columns=TRUE;
	if(rows&&columns) {
		DoDeleteError();
		return;
	}

	/* delete selected rows or columns */

	if(rows) 	CKTBLRemoveRows( hWnd, rect.top , rect.bottom );
	if(columns) CKTBLRemoveColumns( hWnd, rect.left ,rect.right );
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void DoSetSelectionForeColor( HWND hWnd )
{
	CKTBL_ATTRIB   attrib;

	memset( &attrib, 0, sizeof(CKTBL_ATTRIB));

	if(!ChooseColor(&CC_Foreground)) return;
	attrib.foreColor = CC_Foreground.rgbResult;

	CKTBLEndEdit( hWnd, TRUE );
	CKTBLModifySelAttr( hWnd, &attrib, CKTBL_ATTR_FORECOLOR, 0 );
}


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void DoSetSelectionBackColor( HWND hWnd )
{
	CKTBL_ATTRIB   attrib;

	memset( &attrib, 0, sizeof(CKTBL_ATTRIB));

	if(!ChooseColor(&CC_Background)) return;
	attrib.backColor = CC_Background.rgbResult;

	CKTBLEndEdit( hWnd, TRUE );
	CKTBLModifySelAttr( hWnd, &attrib, CKTBL_ATTR_BACKCOLOR, 0 );
}

/*------------------------------------------------------------------------------
 * the most part of this function is spent filling out the CHOOSEFONT and
 * associated LOGFONT structures so that we have a default font in the
 * CommonDialog Box
 *----------------------------------------------------------------------------*/
void DoSetSelectionFont( HWND hWnd )
{
	CKTBL_ATTRIB	attrib;
	RECT			rect;
	LOGFONT 		lf;
	CHOOSEFONT 		cf;
	int				logpixelSY;
	HDC				hDC;

	/* Set all structure fields to zero. */

	memset(&attrib, 0, sizeof(CKTBL_ATTRIB));
	memset(&cf, 0, sizeof(CHOOSEFONT));
	memset(&lf, 0, sizeof(LOGFONT));

	/* get default from left top cell in first rectangle of selection */

	CKTBLGetSelection( hWnd, &rect, 1 );
	CKTBLGetAttr( hTblWnd, max(rect.top,1) , max(rect.left,1), &attrib );


	/* set defaults */

	cf.lStructSize = sizeof(CHOOSEFONT);
	cf.hwndOwner = hMainWnd;
	cf.lpLogFont = &lf;
	cf.Flags = CF_SCREENFONTS | CF_EFFECTS | CF_INITTOLOGFONTSTRUCT;
	cf.rgbColors = attrib.foreColor;
	cf.nFontType = SCREEN_FONTTYPE;
	// cf.iPointSize = attrib.fontSize; // this is not used by ChooseFont()

	// default size is cumbersome beacause LOGFONT does not have a field for
	// the size in points

	hDC = GetDC( hTblWnd );			// quick hack to get logpixelSY
	logpixelSY = GetDeviceCaps( hDC, LOGPIXELSY );
	ReleaseDC( hTblWnd, hDC );

	lf.lfHeight = - (int)((long)attrib.fontSize * logpixelSY / 720 );
	lf.lfWidth  = 0;

	// rest is easy

	lf.lfWeight = (attrib.fontBold)?FW_BOLD:FW_NORMAL; 		// why not a boolean ? :-)
	lf.lfItalic = attrib.fontItalic;						// why not an angle ? :-)
	lf.lfUnderline = attrib.fontUnderline;
	lf.lfStrikeOut = attrib.fontStrikeOut;
	lf.lfOutPrecision	= OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision 	= CLIP_DEFAULT_PRECIS;
	lf.lfQuality 		= PROOF_QUALITY;
	lf.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

	strncpy( lf.lfFaceName, attrib.fontName, LF_FACESIZE );

	/* choose font */

	if(!ChooseFont(&cf)) return;
	
	/* create attrib from LOGFONT */

	attrib.fontName = lf.lfFaceName;
	attrib.fontSize = cf.iPointSize;  			// logfont does not have the size in points
	attrib.fontBold = lf.lfWeight==FW_BOLD;
	attrib.fontItalic = lf.lfItalic;
	attrib.fontUnderline = lf.lfUnderline;
	attrib.fontStrikeOut = lf.lfStrikeOut;
	attrib.foreColor = cf.rgbColors;

	CKTBLEndEdit( hWnd, TRUE );
	CKTBLModifySelAttr( hWnd, &attrib, CKTBL_ATTR_FONT|CKTBL_ATTR_FORECOLOR, 0 );
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

BOOL InitApplication(hInstance)
HANDLE hInstance;
{
	WNDCLASS  wc;

	wc.style = NULL;
	wc.lpfnWndProc = MainWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(APPLICATION_ICON));
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName =  APPLICATION_MENU;
	wc.lpszClassName = APPLICATION_CLASS;

	return (RegisterClass(&wc));
}


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

BOOL InitInstance(hInstance, nCmdShow )
HANDLE  hInstance;
int     nCmdShow;
{
	CKTBL_CREATE 	cktbl;
	RECT 			rect;


	hInst = hInstance;

	hAccTable = LoadAccelerators(hInst, APPLICATION_ACCELERATORS);

	hMainWnd = CreateWindow(
		  APPLICATION_CLASS,
		  APPLICATION_NAME,
		  WS_OVERLAPPEDWINDOW,
		  32,32,600,300,
		  NULL,
		  NULL,
		  hInstance,
		  NULL
	 );

	 if (!hMainWnd) return (FALSE);

	 GetClientRect(hMainWnd, (LPRECT) &rect);

	 /* Create CKTBL window */

	 memset( &cktbl, 0, sizeof(cktbl));
	 cktbl.flags = CKTBL_DEFAULT_FLAGS
		|CKTBL_RESIZE_ROWS|CKTBL_RESIZE_COLUMNS
		|CKTBL_EDIT_ROW_LABELS|CKTBL_EDIT_COLUMN_LABELS;

	 cktbl.rows = 32;
	 cktbl.columns = 8;

	 hTblWnd = CreateWindow(CKTBL_CLASS,
		  NULL,
		  WS_CHILD | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL,
		  0,
		  0,
		  (rect.right-rect.left),
		  (rect.bottom-rect.top),
		  hMainWnd,
		  0,
		  hInst,
		  &cktbl);

	if (!hTblWnd) {
		DestroyWindow(hMainWnd);
		return (NULL);
	}

	CKTBLSetColumnWidth( hTblWnd, 4, 180 );

	ShowWindow(hMainWnd, nCmdShow);
	UpdateWindow(hMainWnd);

	return (TRUE);

}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

CKTBL_ATTRIB FontAttribute;

void InitMenuFonts( HMENU hMenu )
{
	CKTBL_ATTRIB 	attrib;
	RECT 			rect;
	int				nSel;

	memset(&attrib, 0, sizeof(CKTBL_ATTRIB));

	nSel = CKTBLGetSelectionSize( hTblWnd );
	if(nSel==0) {
		rect.left = rect.right = CKTBLGetCurrentColumn( hTblWnd );
		rect.top = rect.bottom = CKTBLGetCurrentRow( hTblWnd );
	} else {
		CKTBLGetSelection( hTblWnd, &rect, 1 );
	}

	CKTBLGetAttr( hTblWnd, max(rect.top,1) , max(rect.left,1), &attrib );

	FontAttribute = attrib;

	CheckMenuItem( hMenu, IDM_FONTBOLD, 	MF_BYCOMMAND | (attrib.fontBold ? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FONTITALIC, 	MF_BYCOMMAND | (attrib.fontItalic ? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FONTUNDERLINE,MF_BYCOMMAND | (attrib.fontUnderline ? MF_CHECKED : MF_UNCHECKED));
}

/*------------------------------------------------------------------------------
 *----------------------------------------------------------------------------*/

void InitMenuFormats( HMENU hMenu )
{
	CKTBL_ATTRIB 	attrib;
	int				row, col;
	BOOL			b1,b2,b3,b4;

	memset(&attrib, 0, sizeof(CKTBL_ATTRIB));

	// get attribute of current cell

	col = CKTBLGetCurrentColumn( hTblWnd );
	row = CKTBLGetCurrentRow( hTblWnd );
	CKTBLGetAttr( hTblWnd, max(row,1) , max(col,1), &attrib );

	// check menu items according to attribute

	b1 = (attrib.format & CKTBL_MASK_HALIGN) == CKTBL_FMT_LEFT;
	b2 = (attrib.format & CKTBL_MASK_HALIGN) == CKTBL_FMT_RIGHT;
	b3 = (attrib.format & CKTBL_MASK_HALIGN) == CKTBL_FMT_CENTER;
	
	CheckMenuItem( hMenu, IDM_FORMATLEFT,  MF_BYCOMMAND | (b1 ? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FORMATRIGHT, MF_BYCOMMAND | (b2 ? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FORMATCENTER,MF_BYCOMMAND | (b3 ? MF_CHECKED : MF_UNCHECKED));

	b1 = (attrib.format & LINESTYLE_MASK)==CKTBL_SINGLELINE;
	b2 = (attrib.format & LINESTYLE_MASK)==CKTBL_SINGLELINE_BOTTOM;
	b3 = (attrib.format & LINESTYLE_MASK)==CKTBL_MULTILINE;
	b4 = (attrib.format & LINESTYLE_MASK)==CKTBL_MULTILINEBREAK;

	CheckMenuItem( hMenu, IDM_FORMATSINGLELINE,			MF_BYCOMMAND | (b1? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FORMATSINGLELINE_BOTTOM,	MF_BYCOMMAND | (b2? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FORMATMULTILINE, 			MF_BYCOMMAND | (b3? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem( hMenu, IDM_FORMATMULTILINE_BREAK, 	MF_BYCOMMAND | (b4? MF_CHECKED : MF_UNCHECKED));

}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void DoFileNew( HWND hWnd )
{
	CKTBLClear( hWnd );
	CKTBLSetSize( hWnd, SIZEY, SIZEX );
	InvalidateRect( hWnd, NULL, FALSE );
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

#define MAX_SIZEY	16000
#define LINESIZE	512

void InitTable( HWND hWnd, char * fileName )
{
	FILE 	* f;
	int   	row, col, maxCol = 8;
	char	line[LINESIZE+1];
	char    *s1, *s2;


	f = fopen( fileName, "r" );
	if(!f) return;

	row = 0;

	CKTBLClear( hWnd );

	while( row<=MAX_SIZEY && !feof(f) ) {
		line[0]=0;
		fgets( line, LINESIZE, f );
		line[LINESIZE] = 0;

		s1 = line;
		col = 1;
		while( *s1 ) {
			s2 = s1;
			while( *s2 && *s2!='\t' && *s2!='\n' ) s2++;
			// if(*s2=='\n') *s2=0;
			if(*s2) *s2++=0;

			CKTBLSetText( hWnd, row, col, s1 );
			s1 = s2;
			col++;
		}
		maxCol = max( maxCol, col-1 );

		row++;
	}
	row = max( row-1, 8 );
	CKTBLSetSize( hWnd, row, maxCol );
	fclose(f);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

void DoFileOpen( HWND hWnd )
{
	OPENFILENAME 	ofn;
	char 			szFile[256], szFileTitle[256], szFilter[256];
	UINT  			i;
	char  			chReplace;    /* string separator for szFilter */

	/* Get the system directory name, and store in szDirName */

	szFile[0] = '\0';  
	szFileTitle[0] = '\0';  
	
	strcpy( szFilter, "Text Files(*.TXT)!*.txt!All files(*.*)!*.*!" );
	chReplace = '!';
   	for (i = 0; szFilter[i] != '\0'; i++) {
		if (szFilter[i] == chReplace)
		   szFilter[i] = '\0';
	}

	/* Set all structure members to zero. */

	memset(&ofn, 0, sizeof(OPENFILENAME));

	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hMainWnd;
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile= szFile;
	ofn.nMaxFile = sizeof(szFile);
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = NULL;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

	if (GetOpenFileName(&ofn)) {
		InitTable( hWnd, ofn.lpstrFile );
		InvalidateRect( hWnd, NULL, FALSE );
	}
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

void DoFileSaveAs( HWND hWnd )
{
	OPENFILENAME 	ofn;
	char 			szFile[256], szFileTitle[256];
	UINT  			i;
	char  			chReplace;    /* string separator for szFilter */
	char  			* szFilter;

	MessageBox( hMainWnd, "SaveAs not yet implemented !", "", MB_OK );
	return;

	/* Get the system directory name, and store in szDirName */

	szFile[0] = '\0';
	szFilter = "Text Files(*.TXT)!*.txt!All files(*.*)!*.*!";
	chReplace = '!';

	for (i = 0; szFilter[i] != '\0'; i++) {
		if (szFilter[i] == chReplace)
			szFilter[i] = '\0';
	}

	/* Set all structure members to zero. */

	memset(&ofn, 0, sizeof(OPENFILENAME));

	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hMainWnd;
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile= szFile;
	ofn.nMaxFile = sizeof(szFile);
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = NULL;
	ofn.Flags = OFN_PATHMUSTEXIST;

	if (GetSaveFileName(&ofn)) {
		/*
		InitTable( hWnd, ofn.lpstrFile );
		InvalidateRect( hWnd, NULL, FALSE );
		*/
	}
}



/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

BOOL FAR PASCAL DoAbout(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
	HWND hWnd;

	switch (message) {
		case WM_INITDIALOG:
			hWnd = GetDlgItem( hDlg, IDC_1 );
			CKTBLSetSize( hWnd, 16, 16 );
			CKTBLSetFlags( hWnd, CKTBL_DEFAULT_FLAGS );
			CKTBLSetRowHeight( hWnd, 0, 20 );
			CKTBLSetColumnWidth( hWnd, 0, 40 );
			return (TRUE);

		case WM_COMMAND:
			if (wParam == IDOK || wParam == IDCANCEL) {
				 EndDialog(hDlg, TRUE);
				 return (TRUE);
			}
			break;
	}
	return (FALSE);
}


/*------------------------------------------------------------------------------
	DoCommand() handles the menu commands
------------------------------------------------------------------------------*/

LRESULT DoCommand( HWND hWnd, WPARAM wParam, LPARAM lParam )
{
	FARPROC 		lpProcAbout;
	LRESULT 		result = 0;
	CKTBL_ATTRIB	attrib;

	memset(&attrib, 0, sizeof(CKTBL_ATTRIB));

	switch (wParam) {

		/* file menu */

		case IDM_FILENEW:
			DoFileNew( hTblWnd );
			break;

		case IDM_FILEOPEN:
			DoFileOpen( hTblWnd );
			break;

		case IDM_FILESAVEAS:
			DoFileSaveAs( hTblWnd );
			break;

		case IDM_FILEABOUT:
			lpProcAbout = MakeProcInstance(DoAbout, hInst);
			DialogBox(hInst, "AboutBox", hWnd, lpProcAbout);
			FreeProcInstance(lpProcAbout);
			break;

		case IDM_FILEEXIT:
			DestroyWindow(hWnd);
			break;

		/* edit menu */

		case IDM_EDITINSERTBEFORE:
			DoInsertRowColBeforeOrAfter( hTblWnd, TRUE );
			break;

		case IDM_EDITINSERTAFTER:
			DoInsertRowColBeforeOrAfter( hTblWnd, FALSE );
			break;

		case IDM_EDITDELETE:
			DoDeleteRowCol( hTblWnd );
			break;

		/* color menu */

		case IDM_COLORFG:
			DoSetSelectionForeColor( hTblWnd );
			break;

		case IDM_COLORBG:
			DoSetSelectionBackColor( hTblWnd );
			break;

		/* font menu */

		case IDM_FONTCHOOSE:
			DoSetSelectionFont( hTblWnd );
			break;

		case IDM_FONTBOLD:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.fontBold = ! FontAttribute.fontBold;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FONTBOLD, 0  );
			break;

		case IDM_FONTITALIC:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.fontItalic = ! FontAttribute.fontItalic;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FONTITALIC, 0 );
			break;

		case IDM_FONTUNDERLINE:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.fontUnderline = ! FontAttribute.fontUnderline;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FONTUNDERLINE, 0 );
			break;

		/* format menu */

		case IDM_FORMATLEFT:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_FMT_LEFT;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, CKTBL_MASK_HALIGN );
			break;

		case IDM_FORMATRIGHT:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_FMT_RIGHT;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, CKTBL_MASK_HALIGN );
			break;

		case IDM_FORMATCENTER:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_FMT_CENTER;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, CKTBL_MASK_HALIGN );
			break;

		case IDM_FORMATSINGLELINE:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_SINGLELINE;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, LINESTYLE_MASK);
			break;

		case IDM_FORMATSINGLELINE_BOTTOM:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_SINGLELINE_BOTTOM;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, LINESTYLE_MASK);
			break;

		case IDM_FORMATMULTILINE:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_MULTILINE;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, LINESTYLE_MASK);
			break;

		case IDM_FORMATMULTILINE_BREAK:
			CKTBLEndEdit( hTblWnd, TRUE );
			attrib.format = CKTBL_MULTILINEBREAK;
			CKTBLModifySelAttr( hTblWnd, &attrib, CKTBL_ATTR_FORMAT, LINESTYLE_MASK);
			break;

	}
	return result;
}


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/


long FAR PASCAL MainWndProc(HWND hWnd, unsigned message, WPARAM wParam, LPARAM lParam)
{
	LRESULT 		result;

	switch (message) {
			case WM_COMMAND:
				result = DoCommand(hWnd, wParam, lParam );
				break;

			case WM_SIZE:
				MoveWindow(hTblWnd, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);				break;

			case WM_SETFOCUS:
				PostMessage( hTblWnd, message, wParam, lParam );
				break;

			case WM_DESTROY:
				PostQuitMessage(0);
				break;

			case WM_INITMENUPOPUP:
				switch( LOWORD(lParam) ) {
					case 0 : break;						/* File menu */
					case 1 : break;			 			/* Edit menu */
					case 2 : break; 					/* Table menu */
					case 3 : InitMenuFonts( wParam ); break;
					case 4 : InitMenuFormats( wParam ); break;
				}
				break;

		  default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return result;
}
/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
HANDLE hPrevInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
	MSG msg;

	if (!hPrevInstance)
		if (!InitApplication(hInstance))
			return (FALSE);

#ifdef CKTBL_LIBRARY
	if(!CKTBLInitLib(hInstance)) {
#if 0
		MessageBox( NULL, "Error initializing CKTBL", "error", MB_OK );
		return FALSE;
#endif
	}
#endif

	if(*lpCmdLine==0) lpCmdLine=NULL;

	if (!InitInstance(hInstance, nCmdShow))
		return (FALSE);

	InitChooseColor( &CC_Foreground );
	InitChooseColor( &CC_Background );
	InvalidateRect( hTblWnd, NULL, FALSE );
	InitTable( hTblWnd, lpCmdLine ? lpCmdLine : "a.txt"  );

	while (GetMessage(&msg, NULL, NULL, NULL)) {
		/* Only translate message if it is not an accelerator message */

		if (!TranslateAccelerator(hMainWnd, hAccTable, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	 }

#ifdef CKTBL_LIBRARY
	CKTBLExitLib();
#endif

	return (msg.wParam);
}

/*--end--*/
