/* clmail.cpp -- Command Line Mailer: NT console app to send SMTP mail
   Copyright 1994 by Dean Troyer (Dean.Troyer@AZ.Honeywell.COM)
   
   This program is derived from Blat, a public domain Windows NT
   SMTP mailer, which was in turn derived from the mail code in 
   WinVN.  There's not much of the Blat changes left.

   From the Blat README:
     The authors of Blat have placed it in the public domain. This 
     means you can use it free of charge for any purpose you like, 
     with no conditions being placed on its use by us. The source 
     code is also available free of charge and under the same 
     conditions as the executables.

     You have permission to modify, redistribute, hoard, or even 
     sell Blat in its executable or source form. If you do sell 
     Blat, though, we'd appreciate it if you'd provide your own 
     support (and send us a free copy).  We cannot take any support 
     load for Blat (we've got better things to do). 

     Various bits of the source code are copyright by other 
     people/organizations.  Look in the source code for copyright 
     ownership.

     The authors of the package are not responsible for any damage 
     or losses that the usage of Blat may cause. We are especially 
     not responsible for the misuse of the SMTP (or other) mail system.

     AUTHORS

     Mark Neal    (mjn@aber.ac.uk)
     Pedro Mendes (prm@aber.ac.uk)

   TODO:
   - verify what From header should default to (include machine?)
*/

#define PROGNAME "CLMail"
#define VERSION  "0.4"

char progname[] = PROGNAME;

#include "clmail.h"
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include "smtp.h"

#ifdef WIN32         
  #define __far far
  #define huge far
  #define __near near
#endif               

// Command line data
char *filename  = "";
char *recipient = "";
char *subject   = "";

char Sender[SENDER_SIZE];
char *Recipients;
char *cc_list="";
char *loginname="";
char *senderid="";
int impersonating = 0;
int Verbose = 0;
int debug = 0;                  // Debug level

char *usage[]= {
PROGNAME " ver " VERSION ": Send mail via SMTP",
"",
PROGNAME " <recipient> [-c <recipient>] [-f <filename>] [-s <subject>]",
"     [-i <address>] [-l <sender>]",
PROGNAME " -R <server address> <sender>",
PROGNAME " -h",
"",
"<recipient>    the recipient's address",
"-c <recipient> the carbon copy recipient's address",
"-f <filename>  a file to be sent as the message ",
"-s <subject>   the subject line",
"-i <address>   a 'From:' address, not necessarily known to the SMTP server.",
"-l <sender>    the sender's login name (must be known to the SMTP server)",
"",
"-R [<server address> <sender>]\tset's the address of the SMTP server to be used",
"",
"-h\tdisplays this help.",
//"",
//"Note that if the '-i' option is used, <sender> is included in 'Reply-to:'",
//"and 'Sender:' fields in the header of the message.",
NULL
};

int putline_internal (socktag sock, char * line, unsigned int nchars)
{
  int retval;

  if ((retval =
       (*pgensock_put_data) (sock,
                            (char FAR *) line,
                            (unsigned long) nchars)))
  {
    switch (retval)
    {
     case ERR_NOT_CONNECTED:
      gensock_error( "SMTP server has closed the connection", retval );
      break;

     default:
      gensock_error ("gensock_put_data", retval);
    }
    return -1;
  }
  return (0);
}  // putline_internal

void smtp_error (char * message)
{
  cout << message << "\n";
  put_smtp_line(SMTPSock, "QUIT\r\n", 6);
  close_smtp_socket();
}  // smtp_error

// 'destination' is the address the message is to be sent to
// 'message' is a pointer to a null-terminated 'string' containing the 
// entire text of the message. 

int prepare_smtp_message(char * MailAddress, char * destination)
{
  char out_data[MAXOUTLINE];
  char str[1024];
  char *ptr;
  int len, startLen;

  if (open_smtp_socket()) 
    return -1;

  if (get_smtp_line() != 220)
  {
    smtp_error("SMTP server error");
    return(-1);
  }

  sprintf(out_data, "HELO %s\r\n", my_hostname);
  put_smtp_line(SMTPSock, out_data, strlen (out_data));
  if (get_smtp_line() != 250)
  {
    smtp_error("SMTP server error");
    return -1;
  }

  sprintf(out_data, "MAIL From:<%s>\r\n", loginname);
  put_smtp_line(SMTPSock, out_data, strlen(out_data));
  if (get_smtp_line() != 250)
  {
    smtp_error("The mail server doesn't like the sender name,\nhave you set your mail address correctly?");
    return -1;
  }

  // do a series of RCPT lines for each name in address line
  for (ptr = destination; *ptr; ptr += len + 1)
  {
    // if there's only one token left, then len will = startLen,
    // and we'll iterate once only
    startLen = strlen(ptr);
    if ((len = strcspn(ptr, " ,\n\t\r")) != startLen)
    {
      ptr[len] = '\0';                  // replace delim with NULL char
      while (strchr(" ,\n\t\r", ptr[len+1]))   // eat white space
        ptr[len++] = '\0';
    }

    sprintf(out_data, "RCPT To: <%s>\r\n", ptr);
    putline_internal(SMTPSock, out_data, strlen (out_data));
    if (get_smtp_line() != 250)
    {
      sprintf(str, "The mail server doesn't like the name %s.\nHave you set the 'To: ' field correctly?", ptr);
      smtp_error(str);
      return -1;
    }

    if (len == startLen)        // last token, we're done
      break;
  }
  sprintf(out_data, "DATA\r\n");
  put_smtp_line(SMTPSock, out_data, strlen (out_data));
  if (get_smtp_line() != 354)
  {
    smtp_error("Mail server error accepting message data");
    return -1;
  }
  return(0);
}  // prepare_smtp_message

int transform_and_send_edit_data( socktag sock, char * editptr )
{
  char *index;
  char *header_end;
  char previous_char = 'x';
  unsigned int send_len;
  int retval;
  BOOL done = 0;

  send_len = lstrlen(editptr);
  index = editptr;

  header_end = strstr (editptr, "\r\n\r\n");

  while (!done)
  {
    // room for extra char for double dot on end case
    while ((unsigned int) (index - editptr) < send_len)
    {
      switch (*index)
      {
        case '.':
          if (previous_char == '\n')
            /* send _two_ dots... */
            if ((retval = (*pgensock_put_data_buffered) (sock, index, 1))) 
              return (retval);
            if ((retval = (*pgensock_put_data_buffered) (sock, index, 1))) 
              return (retval);
          break;
        case '\r':
          // watch for soft-breaks in the header, and ignore them
          if (index < header_end && (strncmp (index, "\r\r\n", 3) == 0))
            index += 2;
          else
            if (previous_char != '\r')
              if ((retval = (*pgensock_put_data_buffered) (sock, index, 1)))
                return (retval);
          // soft line-break (see EM_FMTLINES), skip extra CR */
          break;
        default:
          if ((retval = (*pgensock_put_data_buffered) (sock, index, 1)))
            return (retval);
      }
      previous_char = *index;
      index++;
    }
    if((unsigned int)(index - editptr) == send_len) 
      done = 1;
  }

  // this handles the case where the user doesn't end the last
  // line with a <return>
  if (editptr[send_len-1] != '\n')
  {
    if ((retval = (*pgensock_put_data_buffered) (sock, "\r\n.\r\n", 5)))
      return (retval);
  } else
    if ((retval = (*pgensock_put_data_buffered) (sock, ".\r\n", 3)))
      return (retval);

  /* now make sure it's all sent... */
  if ((retval = (*pgensock_put_data_flush)(sock))) 
    return (retval);
  return (TRUE);
}  // transform_and_send_edit_data

int send_smtp_edit_data(char * message)
{
  transform_and_send_edit_data(SMTPSock, message);
  if (get_smtp_line() != 250)
  {
    smtp_error("Message not accepted by server");
    return -1;
  }
  return(0);
}  // send_smtp_edit_data

int finish_smtp_message(void)
{
  return put_smtp_line(SMTPSock, "QUIT\r\n", 6);
}  //  finish_smtp_message

// Create CURRENT_USER/SOFTWARE registry entries
int CreateRegEntry( void )
{
  HKEY  hKey1;
  DWORD  dwDisposition;
  LONG   lRetCode;

  // Create the program key in the current user's profile
  lRetCode = RegCreateKeyEx(HKEY_CURRENT_USER,
                            REG_PROG_KEY,
                            0, 
                            NULL, 
                            REG_OPTION_NON_VOLATILE, 
                            KEY_WRITE,
                            NULL, 
                            &hKey1, 
                            &dwDisposition);
  if (lRetCode != ERROR_SUCCESS)
  {
    cout << "Error creating registry key " REG_PROG_KEY "\n";
    return 10;
  }

  // Set the SMTP server value
  lRetCode = RegSetValueEx(hKey1, 
                           REG_SMTP_SERVER, 
                           0, 
                           REG_SZ, 
                           (BYTE *) &SMTPHost[0], 
                           (strlen(SMTPHost)+1));
  if (lRetCode != ERROR_SUCCESS)
  {
    cout << "Error setting registry value " REG_SMTP_SERVER "\n";
    return 11;
  }
  if (debug)
    cout << "Set registry SMTPHost: " << SMTPHost << "\n";
  
  // Set the sender value
  lRetCode = RegSetValueEx(hKey1, 
                           REG_SENDER, 
                           0, 
                           REG_SZ, 
                           (BYTE *) &Sender[0], 
                           (strlen(Sender)+1));
  if (lRetCode != ERROR_SUCCESS)
  {
    cout << "Error setting registry value " REG_SENDER "\n";
    return 11;
  }
  if (debug)
    cout << "Set registry Sender: " << Sender << "\n";
  
  return 0;
}  // CreateRegEntry

// Retrieve CURRENT_USER/SOFTWARE registry entries
int GetRegEntry( void )
{
  HKEY  hKey1;
  DWORD  dwType;
  DWORD  dwBytesRead;
  LONG   lRetCode;

  // Open the registry key in read mode
  lRetCode = RegOpenKeyEx(HKEY_CURRENT_USER,
                          REG_PROG_KEY,
                          0, 
                          KEY_READ, 
                          &hKey1);
  // set the size of the buffer to contain the data returned from the registry
  // thanks to Beverly Brown "beverly@datacube.com" and "chick@cyberspace.com" for spotting it...
  dwBytesRead=SERVER_SIZE;

  // Read the SMTP server value
  lRetCode = RegQueryValueEx(hKey1, 
                             REG_SMTP_SERVER, 
                             NULL , 
                             &dwType, 
                             (BYTE *) &SMTPHost, 
                             &dwBytesRead); 
  if( lRetCode != ERROR_SUCCESS )
  {
    cout << "Error reading registry value " REG_SMTP_SERVER "\n";
    return 12;
  }
  if (debug)
    cout << "Read registry SMTPHost: " << SMTPHost << "\n";
    
  // Read the Sender value
  dwBytesRead=SENDER_SIZE;
  lRetCode = RegQueryValueEx(hKey1, 
                             REG_SENDER, 
                             NULL, 
                             &dwType, 
                             (BYTE *) &Sender, 
                             &dwBytesRead); 
  if( lRetCode != ERROR_SUCCESS )
  {
    cout << "Error reading registry value " REG_SENDER "\n";
    return 12;
  }
  if (debug)
    cout << "Read registry Sender: " << Sender << "\n";

 return 0;
}  // GetRegEntry

// Build the message header
// Call with a pointer to a buffer to receive the header text
// Returns the length of the generated header
// This routine reads the global message data vars
int BuildHeader(LPSTR header)
{
  char tmpstr[256];
  SYSTEMTIME curtime;
  TIME_ZONE_INFORMATION tzinfo;
  
  char * days[] = {"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
  char * months[] = {"Jan","Feb","Mar","Apr","May","Jun",
                     "Jul","Aug","Sep","Oct","Nov","Dec"};
  DWORD retval;

  // Add Date:
  GetLocalTime(&curtime);
  retval = GetTimeZoneInformation(&tzinfo);
  // rfc1036&rfc822 format
  // Mon, 29 Jun 94 02:15:23 GMT
  sprintf(tmpstr, "Date: %s, %.2d %s %.2d %.2d:%.2d:%.2d ",
          days[curtime.wDayOfWeek],
          curtime.wDay,
          months[curtime.wMonth - 1],
          curtime.wYear,
          curtime.wHour,
          curtime.wMinute,
          curtime.wSecond);
  strcpy(header, tmpstr);
  for(int i=0;i<32;i++)
  {
    if(retval == TIME_ZONE_ID_STANDARD) 
      tmpstr[i] = (char)tzinfo.StandardName[i];
    else 
      tmpstr[i] = (char)tzinfo.DaylightName[i];
  }
  strcat(header, tmpstr);
  strcat(header, "\r\n");

  // Add From:
  sprintf(tmpstr, "From: %s\r\n", senderid);
  strcat(header, tmpstr);

  // Forge Sender: & Reply-To:
  if (impersonating)
  {
    sprintf(tmpstr, "Sender: %s\r\n", loginname);
    strcat(header, tmpstr);
    sprintf(tmpstr, "Reply-to: %s\r\n", loginname);
    strcat(header, tmpstr);
  }
  
  // Add Subject:
  if (*subject)
  {
    sprintf(tmpstr, "Subject: %s\r\n", subject);
    strcat(header, tmpstr);
  } else {
    sprintf(tmpstr, "Subject: Contents of file: %s\r\n", filename);
    strcat(header, tmpstr);
  }
        
  // Add To:
  sprintf(tmpstr, "To: %s\r\n", recipient);
  strcat(header, tmpstr);

  // Add Cc:
  if (*cc_list)
  {
    sprintf(tmpstr, "Cc: %s\r\n", cc_list);
    strcat(header, tmpstr);
  }
  
  // Add X-Mailer:
  strcat(header, HED_X_MAILER);

  // Add blank line
  strcat(header, "\r\n");
  return(strlen(header));
}  // BuildHeader

// Build the complete message buffer
// Call with a pointer to the filename to read (stdin if empty),
// a pointer to the header text, and the header length
// Returns a pointer to an allocated buffer containing the complete 
// message.
LPSTR BuildMsgBuffer(LPSTR FileName, LPSTR Header, int HeaderLen)
{
  HANDLE fileh,maph;
  DWORD filesize;
  char *msgptr;
  char *buffer;
  char *tmpptr;
  char TempPath[80];
  char TempFile[MAX_PATH];
  
  if (strlen(FileName) <= 0)
  {     // Read input from stdin into a temp file
    if (GetTempPath(sizeof(TempPath), TempPath) == 0)
      strcpy(TempFile, "$mail$.tmp");
    else
      GetTempFileName(TempPath, "$mail", 0, TempFile);
    // CreateFile
    if ((fileh = CreateFile(TempFile, 
                            GENERIC_READ | GENERIC_WRITE,
                            FILE_SHARE_READ,
                            NULL,
                            CREATE_ALWAYS,
                            FILE_FLAG_SEQUENTIAL_SCAN | FILE_FLAG_DELETE_ON_CLOSE,
                            NULL)) == INVALID_HANDLE_VALUE)
    {
      cout << "error creating temporary file " << TempFile << "\n";              
      return NULL;
    }

    // CreateFileMapping
    if ((maph = CreateFileMapping(fileh,
                                  NULL,
                                  PAGE_READWRITE,
                                  0,
                                  MAX_MAP_FILE_SIZE,
                                  NULL)) == NULL)
    {
      cout << "error creating temporary file mapping\n";
      CloseHandle(fileh);
      return NULL;
    }
                                 
    // MapViewOfFile
    msgptr = (char *)MapViewOfFile(maph,FILE_MAP_WRITE,0,0,0);
    tmpptr = msgptr;
    if (Verbose)
      printf("Reading from stdin:\n");
    int ch = 0;
    while ((ch = getchar()) != EOF)
    {
      *tmpptr = (char)ch;
      tmpptr++;
    }
    tmpptr[0] = '\0';   // terminate message buffer
    filesize = GetFileSize(fileh, NULL);
  } else {      // Open the input file
    // open input file
    if ((fileh = CreateFile(FileName,
                            GENERIC_READ,
                            FILE_SHARE_READ,
                            NULL,
                            OPEN_EXISTING,
                            FILE_FLAG_SEQUENTIAL_SCAN,
                            NULL))==INVALID_HANDLE_VALUE)
    {
      cout << "error reading " << FileName << "\n";              
      return NULL;
    }
    if (GetFileType(fileh)!= FILE_TYPE_DISK)
    {
      cout << "Sorry, I can only mail messages from disk files...\n";
      CloseHandle(fileh);
      return NULL;
    }
    filesize = GetFileSize(fileh, NULL);

    // CreateFileMapping
    if ((maph = CreateFileMapping(fileh,
                             NULL,
                             PAGE_READONLY,
                             0,
                             0,
                             NULL)) == NULL)
    {
      cout << "error creating mapping\n";
      return NULL;
    }
                                 
    // MapViewOfFile
    msgptr = (char *)MapViewOfFile(maph,FILE_MAP_READ,0,0,0);
  }

  // Allocate the complete message buffer
  buffer = new char[filesize+HeaderLen+1];

  // put the header at the top...
  strcpy(buffer, Header);
  // point to the end of the header
  tmpptr = buffer + HeaderLen;
  // and put the whole file there
  DWORD dummy;
  if (!ReadFile(fileh, tmpptr, filesize, &dummy, NULL))
  {
    cout << "error reading " << filename << ", aborting\n";
    CloseHandle(fileh);
    delete [] buffer;              
    return NULL;
  }
  CloseHandle(fileh);
  return buffer;
}  // BuildMsgBuffer

// Dump usage text
void Usage()
{
  int i;
  
  for(i=0;usage[i] != NULL;i++) 
    cout << usage[i] << '\n';
}  // Usage

// Top 'O the thread
int main( int argc,        /* Number of strings in array argv          */
          char *argv[],    /* Array of command-line argument strings   */
          char **envp )    /* Array of environment variable strings    */
{
  char *s;
  OFSTRUCT of;
  char Header[1024];
  int  HeaderLen;
  char *buffer;
  
  gensock_lib = 0;
  senderid  = Sender;
  loginname = Sender;

  for (argc--,argv++; argc > 0; argc--,argv++) 
  {
    if (argv[0][0] != '-' || !argv[0][1])
    {    
      recipient = argv[0];
    } else {
      s = argv[0]+1;
      switch (*s) 
      {
        case 'c':
          cc_list = argv[1];
          argc--,argv++;
          break;
        
        case 'f':
          // get file name from argv[1]
          filename = argv[1];
          if (lstrlen(filename) <=0 || 
              OpenFile(filename, &of, OF_EXIST) == HFILE_ERROR)
          {
            cout << filename << " does not exist\n";            
            filename = "";
          }
          argc--,argv++;
          break;

        case 'h':
        case '?':
          Usage();
          return(1);
          break;

        case 'i':
          senderid = argv[1];
          argc--,argv++;
          impersonating = 1;
          break;

        case 'l':
          loginname = argv[1];
          argc--,argv++;
          if (!impersonating)
            senderid = loginname;
          break;
          
        case 'R':
          if (argc == 1)
          {
            cout << "Reading registry: HKEY_LOCAL_USER\\" REG_PROG_KEY ":\n";
            if (!GetRegEntry())
            {
              cout << "SMTP server=" << SMTPHost << "\n";
              cout << "Sender=" << Sender << "\n";
            }
            return 0;
          }
          strcpy(SMTPHost, argv[1]);      // Get SMTP hostname
          argc--,argv++;
          if (argv[1][0] != '-')
          {               
            strcpy(Sender, argv[1]);
            argc--,argv++;
          }
          if (CreateRegEntry() == 0)
          {
            cout << "SMTP server set to " << SMTPHost << "\n";
            cout << "Sender set to " << Sender << "\n";
          }
          break;
        
        case 's':
          subject = argv[1];
          argc--,argv++;
          break;

        case 'x':
          ++debug;
          break;
                
        case 'v':
          ++Verbose;
          break;
                
        default:
          cout << "Unrecognized switch: -" << s;
      }  // case
    }  // else
  }  // for

  if (debug)
  {
    // dump switch results
    cout << "recipient:    " << recipient << "\n";
    cout << "f: filename:  " << filename << "\n";
    cout << "c: cc_list:   " << cc_list << "\n";
    cout << "f: loginname: " << loginname << "\n";
    cout << "i: senderid:  " << senderid << "\n";
    cout << "s: subject:   " << subject << "\n";
    if (debug > 1)
      return(1);
  }
  
  // Check validity of arguments
  if (strlen(recipient) == 0)        // Anyone to sent to?
  {
    cout << "no destination\n";
    return(0);
  }

  if (GetRegEntry())
  {
    cout << "aborting, nothing sent\n";
    return 12;
  }

  // build the recipients list
  Recipients = new char [strlen(recipient) + strlen(cc_list) + 2];
  strcpy(Recipients, recipient);
  if (strlen(cc_list) > 0)
  {
    strcat(Recipients, ",");
    strcat(Recipients, cc_list);
  }

  // Build the message header
  HeaderLen = BuildHeader((LPSTR)&Header);

  // Append the message & return the unified buffer
  buffer = BuildMsgBuffer(filename, (LPSTR)&Header, HeaderLen);
  if (buffer == NULL)
  {
      delete [] Recipients;
      return 3;
  }  
  
  if (Verbose)
    cout << "Sending " << filename << " to " << (lstrlen(Recipients)?
         Recipients:"<unspecified>") << '\n';
  if(lstrlen(subject) && Verbose)
    cout << "Subject:" << subject << '\n';
  if(lstrlen(loginname) && Verbose)
    cout << "Login name is " << loginname << '\n';

  if (!prepare_smtp_message(loginname, Recipients))
  {
    if (!send_smtp_edit_data(buffer))
      finish_smtp_message();
    close_smtp_socket();
  }

  delete [] buffer;
  delete [] Recipients;
  return 0;
}  // main

