/*
 *  This Tcl routine squeezes Tcl scripts by running them through the C
 *  preprocessor, then stripping all blank lines.  Normal Tcl command lines
 *  are not stripped, and Tcl comments are hidden from the preprocessor.
 *
 *  In addition, a comment of the form:
 *
 *      #!\bin\tclsh
 *      # Created by sqzTcl on <date> from source <file>
 *
 *  may be prepended to the squeezed file.
 *
 *  This file assumes that it has been run through itself via the following
 *  command:
 *
 *      sqzTcl -o sqzTcl sqzTcl.tpp
 *
 *  It supports the following command line switches:
 *
 *      -h - Don't add the "# Created by..." header.  These headers confuse the
 *          C preprocessor when they appear in included files, so use this switch
 *          when the output file may be included into other files via the
 *          "#include" preprocessor directive.
 *
 *      -o <file> - Place the output into file <file>.  By default, the output
 *          file is the input file, minus the `.tpp' extension if it exists,
 *          plus a `.tcl' extension.
 *
 *      Anything else is passed on to the C preprocessor.
 *
 *  The input file must be the last element on the command line.
 *
 *  Known bugs and gotcha's:
 *
 *      o   cpp gleefully replaces all occurances of macro names in the Tcl
 *          script.  This can be a surprise when you've used some of the
 *          implicitly defined names.  For example, you may find all
 *          occurances of the string `mips' replaced by the digit `1'!
 *
 *      o   File names containing wild cards can sometimes be confused as
 *          comment delimiters.  Slash-asterisk and asterisk-slash appear
 *          frequently in glob commands, but cpp will treat them as comments!
 */

global argc argv
#ifdef _DEBUG
puts "Loading [loaddll filesys]"
#else
loaddll filesys
#endif
set lend [expr $argc-1]
set commandLine $argv


/*
 *  The input file must be the last element on the command line.  Extract it
 *  first.
 */

set in_file [lindex $commandLine $lend]
if {    [string match -* $in_file]
        || (! [string compare {} $in_file])
        || (! [file readable $in_file])} {
    error {Invalid input file.}
}
set commandLine [lreplace $commandLine $lend end]


/*
 *  Look for the -h switch, and if found, disable header output.
 */

set opos [lsearch $commandLine -h]
if {-1 != $opos} {

    /*
     *  We found the -h switch, so Disable headers.
     */

    set noheader 1
    set commandLine [lreplace $commandLine $opos $opos]
} else {

    /*
     * No -h switch, the headers stay in.
     */

    set noheader 0
}


/*
 *  Look for the -o switch, and set the output file name.
 */

set opos [lsearch $commandLine -o]
if {-1 != $opos} {

    /*
     *  We found a -o switch, so the next argument is the output file.
     */

    set fpos [expr {$opos + 1}]
    set out_file [lindex $commandLine $fpos]
    try {
        set commandLine [lreplace $commandLine $opos $fpos]
    } catch {
        error {Invalid output file.}
    }
} {

    /*
     *  No -o switch, so build the output file from the name of the input
     *  file.
     */

    if {0 == [string compare .tpp [file extension $in_file]]} {
        set out_file [file rootname [file tail $in_file]].tcl
    } {
        set out_file [file tail $in_file].tcl
    }
}


/*
 *  Try and find the Tcl application.
 */

#ifdef _DEBUG
puts ---------------------------------------
puts {Looking for tclsh executable:}
#endif
set Tcl tclsh
foreach loc [split $env(Path) ^;] {
    set tTcl [filesys path $loc $Tcl]
#ifdef _DEBUG
    puts "  Looking for $tTcl..."
#endif
    if {[file executable $tTcl.exe]} {
        set Tcl $tTcl
#ifdef _DEBUG
        puts "  Found it at $Tcl."
#endif
        break
    }
}


/*
 *  Build up the command to squeeze the file.
 */

set sed1tmp [filesys tempname s1 .tmp]
set sed [open $sed1tmp w]
foreach cppcmd {define elif else endif if ifdef ifndef include pragma undef} {
    puts $sed [format {/^#[     ]*%s/b} $cppcmd]
}
if {$noheader} {
    puts $sed [format {%s%s} {s"^#"/} {/"}]
} else {
    puts $sed {s"^#" _#"}
}
close $sed
set s1cmd "sed -f $sed1tmp"

set ccmd ^
    [format ^
        {cl -nologo -EP -u -I%s -I%s -D__LANGUAGE_TCL -D_WIN32 -Tp} ^
        [file dirname $in_file] ^
        [filesys path [info library] include]]

set sed2tmp [filesys tempname s2 .tmp]
set sed [open $sed2tmp w]
puts $sed {/^[     ]*$/d;s/^[  ][  ]*_#/#/;s|[     ][  ]*$||}
close $sed
set s2cmd "sed -f $sed2tmp"

set header ^
    [format "#!%s %s^n# This file created by %s^n#    source: %s^n#    date:   %s^n" ^
        $Tcl ^
        $out_file ^
        [file tail [info script]] ^
        $in_file ^
        [filesys date]]

set pass1 [filesys tempname p1 .tmp]
set pass2 [filesys tempname p2 .tmp]
set pass3 [filesys tempname p3 .tmp]
set errs  [filesys tempname ce .tmp]
#ifdef _DEBUG
if {$noheader} {
    puts [concat filesys create $pass1]
} else {
    puts [concat filesys write $pass1 ^$header]
}
puts [concat filesys append $pass1 ^[filesys contents $in_file^]]
#endif
if {$noheader} {
    filesys create $pass1
} else {
    filesys write $pass1 $header
}
filesys append $pass1 [filesys contents $in_file]

#ifdef _DEBUG
puts [concat $s1cmd $pass1 > $pass2]
#endif
exec $s1cmd $pass1 > $pass2
#ifndef _DEBUG
filesys delete $sed1tmp
filesys delete $pass1
#endif

#ifdef _DEBUG
puts [concat $ccmd $pass2 $commandLine > $pass3]
#endif
catch {exec $ccmd $pass2 $commandLine > $pass3 2> $errs}
puts [filesys contents $errs]
#ifndef _DEBUG
filesys delete $pass2
filesys delete $errs
#endif

#ifdef _DEBUG
puts [concat $s2cmd $pass3 > $out_file]
#endif
exec $s2cmd $pass3 > $out_file
#ifndef _DEBUG
filesys delete $sed2tmp
filesys delete $pass3
#endif

#ifdef _DEBUG
puts "File $out_file created from source $in_file on [filesys date]^nfor $Tcl"
#endif
return

