/*-----------------------------------------------------------------------------
    file:   boyer32.cpp
    desc:   Boyer-Moore text search algorithm (Windows NT version)
    by:     Patrick Ko
    date:   6 Mar 91 - born
    revi:   4 Apr 94 - port to Windows 3.1
            21 Aug 94 - support Windows DLL
            13 Jan 95 v0.4 - bug fix FindIC() and FindBackwardIC()
            05 Sep 96 v1.0 - supports Win32
    note:   this version works for Windows 95 and Windows NT only
-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
    Program Specification

    in:     search space s, pattern p
    out:    a pointer where p is exactly matched at s[i], NULL indicates fail
    why:    Boyer-Moore algorithm is best for general text search. On
            "average" it takes length(s)/length(p) steps to match p in s.

    ref:    I recommend the following references:

            "Algorithms". Robert Sedgewick. Addison Wesley Publishing Company.
            1988. 2nd addition. p286. QA76.6.S435 1983

            "Faster String Searches". Doctor Dobb's Journal. Volume 14
            Issue 7 July 1989. Costas Menico. p74.
-----------------------------------------------------------------------------*/

#include <windows.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "boyer32.h"


void
BOYER::Setup(BYTE * p, UINT plen)
{
	UINT j;
	BYTE c;

	m_plen = plen;
	m_p = (unsigned char *)new char[plen];
	memcpy(m_p, p, plen);

	m_pIC = (unsigned char *)new char[plen];
	memcpy(m_pIC, p, plen);

    for (j=0; j<CHARSETSIZE; j++)
    {
        m_skip[j] = m_skipIC[j] = plen;
        m_skipb[j] = m_skipbIC[j] = plen;
    }

    for (j=0; j<plen; j++)
    {
        m_pIC[j] = toupper(m_pIC[j]);
        c = p[j];
        m_skip[c] = m_skipIC[toupper(c)] = m_plen - j - 1;
    }

	for (j=plen; j>0; j--)
	{
        c = p[j-1];
        m_skipb[c] = m_skipbIC[toupper(c)] = j - 1;
	}
}

BOYER::BOYER(BYTE * p, UINT plen)
{
	Setup(p, plen);
}

BOYER::BOYER(BYTE * p)
{
	UINT plen = strlen((char *)p);

	Setup(p, plen);
}

BOYER::~BOYER()
{
	delete m_p;
	delete m_pIC;
}

BYTE * 
BOYER::Find(BYTE * s, UINT slen )
{
    UINT i, j, n;
    register BYTE c;

    i = j = m_plen;

    do
    {
        c = *(s + (i - 1));

        if (c == m_p[j - 1])
        {
			i--;
			j--;
        }
		else
        {
            n = m_plen - j + 1;
            if (n > m_skip[c] )
            {
                i += n;
            }
			else
            {
                i += m_skip[c];
            }
            j = m_plen;
        }
    }
    while (j >= 1 && i <= slen);

    /* match fails */
    if (i >= slen)
    {
        return NULL;
    }
    /* match successful */
    else
    {
        return (s + i);
    }
}

BYTE * 
BOYER::FindBackward(BYTE * s, UINT slen )
{
    UINT i, j, n;
    register BYTE c;

    i = j = m_plen;

	do
    {
        c = s[1 - i];
        if (c == m_p[m_plen - j])
        {
			i--;
			j--;
        }
		else
        {
            n = m_plen - j + 1;
            if (n > m_skipb[c])
            {
                i += n;
            }
            else
            {
                i += m_skipb[c];
            }
            j = m_plen;
        }
    }
    while (j >= 1 && i <= slen);

    /* match fails */
    if (i >= slen)
    {
        return NULL;
    }
    /* match successful */
	else
    {
        return (s - i - m_plen + 1);
    }

}

BYTE *
BOYER::FindIC( BYTE * s, UINT slen )
{
    UINT i, j, n;
    register BYTE c;

    i = j = m_plen;

	do
    {
        c = toupper(s[i - 1]);
        if (c == m_pIC[j - 1])
        {
			i--;
			j--;
        }
		else
        {
            n = m_plen - j + 1;
            if (n > m_skipIC[c])
            {
                i += n;
            }
			else
            {
                i += m_skipIC[c];
            }
            j = m_plen;
        }
    }
    while (j >= 1 && i <= slen);

    /* match fails */
    if (i >= slen)
    {
        return NULL;
    }
    /* match successful */
    else
    {
        return (s + i);
    }
}

BYTE *
BOYER::FindBackwardIC( BYTE * s, UINT slen )
{
    UINT i, j, n;
    register BYTE c;

    i = j = m_plen;

	do
    {
        c = toupper(s[1 - i]);
        if (c == m_pIC[m_plen - j])
        {
			i--;
			j--;
        }
		else
        {
            n = m_plen - j + 1;
            if (n > m_skipbIC[c] )
            {
                i += n;
            }
            else
            {
                i += m_skipbIC[c];
            }
            j = m_plen;
        }
    }
    while (j >= 1 && i <= slen);

    /* match fails */
    if (i >= slen)
    {
        return NULL;
    }
    /* match successful */
	else
    {
        return (s + 1 - i - m_plen);
    }
}

//---------------------------------------------------------------------------
// boyer32.dll API definitions
//---------------------------------------------------------------------------

#ifdef BOYERDLL

BOOL WINAPI DLLEntry(
    HANDLE hDll,
    DWORD  dwReason,
    LPVOID lpReserved)
    {
    switch(dwReason)
        {
        case DLL_PROCESS_ATTACH:
        case DLL_PROCESS_DETACH:
        case DLL_THREAD_ATTACH:
        case DLL_THREAD_DETACH:
        default:
            break;

        } // end switch()

    return TRUE;

    } // end DllEntryPoint()


#endif

HFIND WINAPI CreateFindHandle( BYTE * p_Pattern, UINT p_Size )
{
	BOYER ** boyer;
	HFIND hFind;

	if ((hFind = GlobalAlloc(GHND, sizeof(BOYER *))) != NULL)
	{
		boyer = (BOYER **)GlobalLock(hFind);
		*boyer = new BOYER(p_Pattern, p_Size);
		if (*boyer != NULL)
		{
			GlobalUnlock(hFind);
			return (hFind);
		}

		GlobalUnlock(hFind);
		GlobalFree(hFind);
	}

	return NULL;
}

void WINAPI FreeFindHandle( HFIND p_hFind )
{
	BOYER ** boyer;

	if (p_hFind != NULL)
	{
		boyer = (BOYER **)GlobalLock(p_hFind);
		delete *boyer;
		GlobalUnlock(p_hFind);
		GlobalFree(p_hFind);
	}
}

BYTE * WINAPI Find(HFIND p_hFind, BYTE * p_s, UINT p_slength)
{
	BOYER ** boyer;
	BYTE * result = NULL;

	if (p_hFind != NULL)
	{
		boyer = (BOYER **)GlobalLock(p_hFind);
		result = (*boyer)->Find(p_s, p_slength);
		GlobalUnlock(p_hFind);
		GlobalFree(p_hFind);
	}
	
	return (result);
}

BYTE * WINAPI FindIC(HFIND p_hFind, BYTE * p_s, UINT p_slength)
{
	BOYER ** boyer;
	BYTE * result = NULL;

	if (p_hFind != NULL)
	{
		boyer = (BOYER **)GlobalLock(p_hFind);
		result = (*boyer)->FindIC(p_s, p_slength);
		GlobalUnlock(p_hFind);
		GlobalFree(p_hFind);
	}
	
	return (result);
}

BYTE * WINAPI FindBackward(HFIND p_hFind, BYTE * p_s, UINT p_slength)
{
	BOYER ** boyer;
	BYTE * result = NULL;

	if (p_hFind != NULL)
	{
		boyer = (BOYER **)GlobalLock(p_hFind);
		result = (*boyer)->FindBackward(p_s, p_slength);
		GlobalUnlock(p_hFind);
		GlobalFree(p_hFind);
	}
	
	return (result);
}

BYTE * WINAPI FindBackwardIC(HFIND p_hFind, BYTE * p_s, UINT p_slength)
{
	BOYER ** boyer;
	BYTE * result = NULL;

	if (p_hFind != NULL)
	{
		boyer = (BOYER **)GlobalLock(p_hFind);
		result = (*boyer)->FindBackwardIC(p_s, p_slength);
		GlobalUnlock(p_hFind);
		GlobalFree(p_hFind);
	}
	
	return (result);
}

BYTE * WINAPI Find2(BYTE * p_p, UINT p_plength, BYTE * p_s, UINT p_slength)
{
	BYTE * result = NULL;

	BOYER * boyer = new BOYER(p_p, p_plength);
	if (boyer != NULL)
	{
		result = boyer->Find(p_s, p_slength);
		delete boyer;
	}

	return result;
}

BYTE * WINAPI Find3(BYTE * p_p, BYTE * p_s, UINT p_slength)
{
	UINT plength = strlen((char *)p_p);

	return Find2(p_p, plength, p_s, p_slength);
}

BYTE * WINAPI FindIC2(BYTE * p_p, UINT p_plength, BYTE * p_s, UINT p_slength)
{
	BYTE * result = NULL;

	BOYER * boyer = new BOYER(p_p, p_plength);
	if (boyer != NULL)
	{
		result = boyer->FindIC(p_s, p_slength);
		delete boyer;
	}

	return result;
}

BYTE * WINAPI FindIC3(BYTE * p_p, BYTE * p_s, UINT p_slength)
{
	UINT plength = strlen((char *)p_p);

	return FindIC2(p_p, plength, p_s, p_slength);
}

BYTE * WINAPI FindBackward2(BYTE * p_p, UINT p_plength, BYTE * p_s, UINT p_slength)
{
	BYTE * result = NULL;

	BOYER * boyer = new BOYER(p_p, p_plength);
	if (boyer != NULL)
	{
		result = boyer->FindBackward(p_s, p_slength);
		delete boyer;
	}

	return result;
}

BYTE * WINAPI FindBackward3(BYTE * p_p, BYTE * p_s, UINT p_slength)
{
	UINT plength = strlen((char *)p_p);

	return FindBackward2(p_p, plength, p_s, p_slength);
}

BYTE * WINAPI FindBackwardIC2(BYTE * p_p, UINT p_plength, BYTE * p_s, UINT p_slength)
{
	BYTE * result = NULL;

	BOYER * boyer = new BOYER(p_p, p_plength);
	if (boyer != NULL)
	{
		result = boyer->FindBackwardIC(p_s, p_slength);
		delete boyer;
	}

	return result;
}

BYTE * WINAPI FindBackwardIC3(BYTE * p_p, BYTE * p_s, UINT p_slength)
{
	UINT plength = strlen((char *)p_p);

	return FindBackwardIC2(p_p, plength, p_s, p_slength);
}

