/*  Project filter
					DCS, UP Olomouc
					Copyright  1995. All Rights Reserved.

					SUBSYSTEM:    filter.exe Application
					FILE:         filter.cpp
					AUTHOR:       Robert Batusek

		implementace tridy TFilter - rodicovske tridy vsech filtrovacich trid
		implementace nekterych trid pro filtracni metody

		implementation of TFilter class - parent of all filter classes
		implementation of some filter methods
*/

#include <except.h>
#pragma hdrstop

#include "filter.h"

/////////////////////////////////////////////////
// class TFilter
// -------------
// Popis: rodicovska trida vsech filtrovacich trid
//
TFilter::TFilter(TMatrix &m,BOOL CreateCopy)
{
	matrix = NULL;
	SetMatrix(m,CreateCopy);
}

TFilter::~TFilter()
{
	delete matrix;
}

void TFilter::FilterImage(int start,int count)
{ int j,i;
	long startrow,endrow;

	matrix->GetSize(xsize,ysize);
	startrow = (long)ysize*start/100;
	endrow = (long)ysize*(start + count)/100;
	for (i=(int)startrow;i<endrow;i++)
		for (j=0;j<xsize;j++)
			matrix->SetElement(j,i,FilterElement(j,i));
}

BOOL TFilter::SetMatrix(TMatrix &m,BOOL CreateCopy)
{
	if (matrix) delete matrix;
	if (CreateCopy)
		matrix = new TMatrix(m);
	else
		matrix = &m;
	return TRUE;
}

TMatrix *TFilter::GetMatrix()
{
	return matrix;
}


/////////////////////////////////////////////////
// class TDigitalize
// -------------
// Popis: testovaci trida, digitalizuje obrazek
//
TDigitalize::TDigitalize(TMatrix &m,BOOL CreateCopy,int newsize): TFilter(m,CreateCopy)
{
	size = newsize;
}

Element TDigitalize::FilterElement(int x,int y)
{
	return matrix->GetElement(x/size*size,y/size*size);
}


/////////////////////////////////////////////////
// class TNoise
// -------------
// Popis: testovaci trida, vytvori na obrazku umele sum
//
TNoise::TNoise(TMatrix &m,BOOL CreateCopy,BOOL newtype,int newpercent,int newsize): TFilter(m,CreateCopy)
{
	type = newtype;
	percent = newpercent;
	deviation = newsize;
}

void TNoise::FilterImage(int start,int count)
{

	if (type) //type  is Gausss
		TFilter::FilterImage(start,count);
	else {
		int xsize,ysize,x,y;
		long i,PixelsToFilter,startrow,endrow;

		matrix->GetSize(xsize,ysize);
		PixelsToFilter = ((long)percent*xsize*ysize)/100 * count /100;
		startrow = (long)ysize*start/100;
		endrow = startrow + (long)ysize*count/100;

		for (i=0;i<PixelsToFilter;i++) {
			x = random(xsize);
			y = (int)(startrow + random((int)(endrow-startrow)));
			matrix->SetElement(x,y,FilterElement(x,y));
		}
	}
}

Element TNoise::FilterElement(int x,int y)
{ long randnum,result;
	if (type) { //type is Gausss
		int sum = 0;
		for (int i=1;i<=12;i++)
			sum += random(256);
		randnum = ((long)(sum - 6) * deviation) / (12*256);
		result = matrix->GetElement(x,y) + randnum;
		if (result>255) result = 255;
		return (int) result;
		// the algorithm above is taken from book
		//Jaroslavskij,Bajla: Metody a systemy cislicoveho spracovania obrazov
		//Alfa Bratislava, 1989, page 201
	} else
		return random(256);
}

/////////////////////////////////////////////////
// class TNegative
// -------------
// Popis: udela z obrazku jeho negativ
// Description: makes a negative from the picture
//
Element TNegative::FilterElement(int x,int y)
{
	return (255 - matrix->GetElement(x,y));
}

/////////////////////////////////////////////////////////////////////////
// class TEnvironsFilter
// -------------
// Popis: zakladni trida pro metody filtrace, ktere vyuzivaji k filtraci
//                  okoli zpracovavneho pixelu
// Description: base class for those methods, which use environs to filtering -
//
TEnvironsFilter::TEnvironsFilter(TMatrix &m,BOOL CreateCopy,int newxsize,int newysize)
	:TFilter(m,CreateCopy)
{
	xenvsize = newxsize;
	if (newysize) yenvsize = newysize;
	else yenvsize = xenvsize;
	newmatrix = NULL;
}

TEnvironsFilter::~TEnvironsFilter()
{
	if (newmatrix) delete newmatrix;
}

TEnvironsFilter::StartFiltering()
{
	try {
		matrix->GetSize(xsize,ysize);
		newmatrix = new TMatrix(xsize,ysize);
	}
	catch(xalloc) {
		newmatrix = NULL;
		return FALSE;
	}
	return TRUE;
}

BOOL TEnvironsFilter::FinishFiltering()
{
	delete matrix;
	matrix = newmatrix;
	newmatrix = NULL;
	return TRUE;
}

void TEnvironsFilter::FilterImage(int start,int count)
{ int j,i;
	long startrow,endrow;

	matrix->GetSize(xsize,ysize);
	startrow = (long)ysize*start/100;
	endrow = (long)ysize*(start + count)/100;
	for (i=(int)startrow;i<endrow;i++)
		for (j=0;j<xsize;j++)
			newmatrix->SetElement(j,i,FilterElement(j,i));
}

void TEnvironsFilter::GetLimits(int x,int y,int &lxlimit,int &lylimit,int &rxlimit,int &rylimit)
{
	if (x<(xenvsize/2)) { //left side of picture
		lxlimit = -x;
		rxlimit = xenvsize/2;
	} else if (x>(xsize-xenvsize/2-1)) { //right side of picture
		lxlimit = -xenvsize/2;
		rxlimit = xsize-x-1;
	} else {
		lxlimit = -xenvsize/2;
		rxlimit = xenvsize/2;
	}

	if (y<(yenvsize/2)) { // top side of picture
		lylimit = -y;
		rylimit = yenvsize/2;
	} else if (y>(ysize-yenvsize/2-1)) { //bottom side of picture
		lylimit = -yenvsize/2;;
		rylimit = ysize-y-1;
	} else {
		lylimit = -yenvsize/2;;
		rylimit = yenvsize/2;
	}
}

/////////////////////////////////////////////////
// class TMaskedFilter
// -------------
// Popis: zakladni trida pro metody filtrace, ktere vyuzivaji k filtraci masku
//
TMaskedFilter::TMaskedFilter(TMatrix &m,BOOL CreateCopy): TEnvironsFilter(m,CreateCopy,0)
{
	mask = NULL;
}

TMaskedFilter::~TMaskedFilter()
{
	if (mask) delete mask;
}

BOOL TMaskedFilter::StartFiltering()
{
	mask->GetSize(xenvsize,yenvsize);
	return TEnvironsFilter::StartFiltering();
}

Element TMaskedFilter::FilterElement(int x,int y)
{	unsigned int sum = 0;
	int i,j,divisor = 0;
	int lxlimit,lylimit,rxlimit,rylimit;

	GetLimits(x,y,lxlimit,lylimit,rxlimit,rylimit);

	for (i=lylimit;i<=rylimit;i++)
		for (j=lxlimit;j<=rxlimit;j++) {
			sum += matrix->GetElement(x+j,y+i) * mask->GetElement(j+xenvsize/2,i+yenvsize/2);
			divisor += (signed char)(mask->GetElement(j+xenvsize/2,i+yenvsize/2));
		}
	return sum/divisor;
}

/////////////////////////////////////////////////
// class TAverage
// -------------
// Popis: prvni skutecne filtracni trida - filtruje tzv. prumerovanim
//
TAverage::TAverage(TMatrix &m,BOOL CreateCopy,int masksize,int agauss)
	:TMaskedFilter(m,CreateCopy)
{
	mask = new TMask(masksize,masksize,1,masksize*masksize);
	gauss = agauss;
}

BOOL TAverage::StartFiltering()
{ int masksize,i;
	Element val;
	BOOL result;

	result = TMaskedFilter::StartFiltering();

	mask->GetSize(masksize,masksize);
	if (gauss) {
		for (i=0,val=1;i<masksize/2;i++,val*=2)
			mask->FillRect(i,i,masksize-i,masksize-i,val);
		mask->CalculateDivisor();
	} else {
		mask->FillRect(0,0,masksize,masksize,1);
		mask->CalculateDivisor();
	}

	return result;
}

/////////////////////////////////////////////////
// class TModifiedAverage
// -------------
// Popis: modifikovane prumerovani - odchylka od puvodniho jasu nesmi presahnout urcitou mez
// Description: modified average - deviation of origin and new brightness must be less than limit
//
BOOL TModifiedAverage::StartFiltering()
{	BOOL result;
	int i,j,row;
	Element average;
	unsigned long sum=0,dev=0;

	result = TAverage::StartFiltering();

	// calculating limit
	randomize();
	matrix->GetSize(xsize,ysize);

	//calculating average of picture
	for (i=0;i<ysize;i++) {
		for (j=0;j<xsize;j++)
			sum+=matrix->GetElement(j,i);
	}
	average =(int)( sum / ((long)xsize*ysize));

	//calculating deviation from three random rows
	for (i=1;i<=3;i++) {
		row=random(ysize);
		for (j=0;j<xsize;j++)
			dev += abs(matrix->GetElement(j,row) - average);
	}

	//limit = 4*devitation
	limit =(int)( 4 * (dev/(3*xsize)));

	return result;
}


Element TModifiedAverage::FilterElement(int x, int y)
{ int newvalue;

	newvalue = TAverage::FilterElement(x,y);
	if (abs(newvalue - matrix->GetElement(x,y))>limit)
		return matrix->GetElement(x,y);
	else
		return newvalue;
}

/////////////////////////////////////////////////
// class TMedian
// -------------
// Popis: nova hodnota je rovna medianu pixelu v jeho okoli
// Description: new value is median of surrounding pixels
//
BOOL TMedian::StartFiltering()
{ int N = xenvsize*yenvsize; //size is always odd

	succesion = new Element[N];
	return TEnvironsFilter::StartFiltering();
}

BOOL TMedian::FinishFiltering()
{
	delete[] succesion;
	return TEnvironsFilter::FinishFiltering();
}

Element TMedian::FilterElement(int x,int y)
{	int N = xenvsize*yenvsize; //size is always odd
	Element xxx;
	int lxlimit,lylimit,rxlimit,rylimit;
	int i,j,counter;

	GetLimits(x,y,lxlimit,lylimit,rxlimit,rylimit);
	//filling succesion by values of surrounding pixels
	for (i=lylimit,counter=0;i<=rylimit;i++)
		for (j=lxlimit;j<=rxlimit;j++,counter++)
			succesion[counter]=matrix->GetElement(x+j,y+i);

	//sorting succesion (only half needed)
	for (i=0;i<=N/2+1;i++)
		for (j=0;j<N-i-1;j++)
			if (succesion[j]>succesion[j+1]) {
				xxx=succesion[j];
				succesion[j] = succesion[j+1];
				succesion[j+1] = xxx;
			}

	//new value is median (middle element) of succesion
	return succesion[N/2+1];
}

/////////////////////////////////////////////////
// class TRotatingMask
// -------------
// Popis: pomoci rotujici masky se urci ta cast okoli, ke ktere pixel patri
//                  a z te se pocita prumer
// Description: rotating mask determines part of environs which pixel belongs to
//                  and then the average is calculated from this part
//
TRotatingMaskFilter::TRotatingMaskFilter(TMatrix &m,BOOL CreateCopy)
	:TMaskedFilter(m,CreateCopy)
{
	average = NULL;
	deviation = NULL;
}

BOOL TRotatingMaskFilter::StartFiltering()
{
	mask = new TRotatingMask();
	average = new Element[9];
	deviation = new int[9];

	return TMaskedFilter::StartFiltering();
}

BOOL TRotatingMaskFilter::FinishFiltering()
{
	delete[] average;
	delete[] deviation;

	return TMaskedFilter::FinishFiltering();
}

Element TRotatingMaskFilter::FilterElement(int x,int y)
{ int i,j,m,minindex,divisor=0;
	int d=0;
	int lxlimit,lylimit,rxlimit,rylimit;
	unsigned sum;
	Element min;

	for (m=0;m<9;m++,average[m]=0,deviation[m]=-1);

	GetLimits(x,y,lxlimit,lylimit,rxlimit,rylimit);

	for (m=0;m<9;m++) {
		((TRotatingMask *)mask)->SetDirection((TDirection) d);
			divisor = 0;
			sum = 0;
			//calculating average from the environs specified by mask
			for (i=lylimit;i<=rylimit;i++)
				for (j=lxlimit;j<=rxlimit;j++)
					if (mask->GetElement(j+xenvsize/2,i+yenvsize/2)) {
						sum += matrix->GetElement(x+j,y+i);
						divisor++;
					}
			if (divisor) average[m] = sum/divisor;
			//calculating deviation for this environs
			sum=0;
			for (i=lylimit;i<=rylimit;i++)
				for (j=lxlimit;j<=rxlimit;j++)
					if (mask->GetElement(j+xenvsize/2,i+yenvsize/2))
						sum += abs(matrix->GetElement(x+j,y+i) - average[m]);
			if (divisor) deviation[m] = sum/divisor;
			//calculating new direction
			d++;
	}

	min = deviation[0];
	minindex = 0;
	for (m=1;m<9;m++) {
		if ((deviation[m]!=-1) && (deviation[m]<min)) {
			minindex=m;
			min = deviation[m];
		}
	}

	//new value is average from the mask where is least deviation
	return average[minindex];
}

/////////////////////////////////////////////////
// class TDifference
// -------------
// Popis: detektor hran - odecte puvodni obraz od obrazu prumerovaneho
// Description: edge detector - use subtraction between original and averaged image
//
Element TDifference::FilterElement(int x,int y)
{Element average;

	average = TAverage::FilterElement(x,y);

	if (intensity*(abs(matrix->GetElement(x,y) - average)) > 255)
		return 255;
	else
		return intensity*(abs(matrix->GetElement(x,y) - average));
}


/////////////////////////////////////////////////
// class TEdgeDetector
// -------------
// Popis: zakladni detektor hran
// Description: basic edge detector
//
Element TEdgeDetector::FilterElement(int x,int y)
{	signed int sum = 0;
	int i,j;
	int lxlimit,lylimit,rxlimit,rylimit;

	GetLimits(x,y,lxlimit,lylimit,rxlimit,rylimit);

	for (i=lylimit;i<=rylimit;i++)
		for (j=lxlimit;j<=rxlimit;j++) {
			sum += matrix->GetElement(x+j,y+i) * (SignedElement)(mask->GetElement(j+xenvsize/2,i+yenvsize/2));
		}
	return abs(sum)>255 ? 255 : abs(sum);
}


/////////////////////////////////////////////////
// class TLaplace
// -------------
// Popis: detektor hran - tzv. Laplacuv operator
// Description: edge detector - Laplace opterator
//
TLaplace::TLaplace(TMatrix &m,BOOL CreateCopy)
	:TEdgeDetector(m,CreateCopy)
{
	mask = new TMask(3,3,1);
	mask->SetElement(1,1,-8); //8-neighbourhood used
}

/////////////////////////////////////////////////
// class TSobel
// -------------
// Popis: detektor hran - tzv. Sobeluv operator
// Description: edge detector - Sobel's opterator
//
TSobel::TSobel(TMatrix &m,BOOL CreateCopy)
	:TEdgeDetector(m,CreateCopy)
{
	mask = new TMask(3,3,0);
}

Element TSobel::FilterElement(int x,int y)
{	int i;
	Element *EdgeSizes,Max;

	EdgeSizes = new Element[8];

	//calculating edge sizes of all Sobel's environs
	for (i=0;i<8;i++) {
		SetDirection(i+1);
		EdgeSizes[i]=TEdgeDetector::FilterElement(x,y);
	}

	//calculating maximum
	Max = EdgeSizes[0];
	for (i=1;i<8;i++)
		if (EdgeSizes[i]>Max) Max = EdgeSizes[i];

	delete[] EdgeSizes;

	return Max;
}

void TSobel::SetDirection(int dir)
{
	mask->FillRect(0,0,xenvsize,yenvsize,0);
	switch((TDirection) dir) {
		case North:
			mask->SetElement(0,0,1);
			mask->SetElement(1,0,2);
			mask->SetElement(2,0,1);
			mask->SetElement(0,2,-1);
			mask->SetElement(1,2,-2);
			mask->SetElement(2,2,-1);
			break;
		case NorthEast:
			mask->SetElement(1,0,1);
			mask->SetElement(2,0,2);
			mask->SetElement(2,1,1);
			mask->SetElement(0,1,-1);
			mask->SetElement(0,2,-2);
			mask->SetElement(1,2,-1);
			break;
		case East:
			mask->SetElement(2,0,1);
			mask->SetElement(2,1,2);
			mask->SetElement(2,2,1);
			mask->SetElement(0,0,-1);
			mask->SetElement(0,1,-2);
			mask->SetElement(0,2,-1);
			break;
		case SouthEast:
			mask->SetElement(2,1,1);
			mask->SetElement(2,2,2);
			mask->SetElement(1,2,1);
			mask->SetElement(1,0,-1);
			mask->SetElement(0,0,-2);
			mask->SetElement(0,1,-1);
			break;
		case South:
			mask->SetElement(0,2,1);
			mask->SetElement(1,2,2);
			mask->SetElement(2,2,1);
			mask->SetElement(0,0,-1);
			mask->SetElement(1,0,-2);
			mask->SetElement(2,0,-1);
			break;
		case SouthWest:
			mask->SetElement(0,1,1);
			mask->SetElement(0,2,2);
			mask->SetElement(1,2,1);
			mask->SetElement(1,0,-1);
			mask->SetElement(2,0,-2);
			mask->SetElement(2,1,-1);
			break;
		case West:
			mask->SetElement(0,0,1);
			mask->SetElement(0,1,2);
			mask->SetElement(0,2,1);
			mask->SetElement(2,0,-1);
			mask->SetElement(2,1,-2);
			mask->SetElement(2,2,-1);
			break;
		case NorthWest:
			mask->SetElement(1,0,1);
			mask->SetElement(0,0,2);
			mask->SetElement(0,1,1);
			mask->SetElement(2,1,-1);
			mask->SetElement(2,2,-2);
			mask->SetElement(1,2,-1);
			break;
	}
}

/////////////////////////////////////////////////
// class TLaplaceSharp
// ---------------------
// Popis: ostrici operator - pouziva Laplaceuv operator k ostreni
// Description: sharpening operator - use Laplace operator to sharpening the image
//
Element TLaplaceSharp::FilterElement(int x,int y)
{int result;
	result = matrix->GetElement(x,y) - intensity*TLaplace::FilterElement(x,y);
	return (result<0) ? 0 : result;
}

/////////////////////////////////////////////////
// class TSobelSharp
// ---------------------
// Popis: ostrici operator - pouziva Sobeluv operator k ostreni
// Description: sharpening operator - use Sobel operator to sharpening the image
//
Element TSobelSharp::FilterElement(int x,int y)
{int result;
	result = matrix->GetElement(x,y) - intensity*TSobel::FilterElement(x,y);
	return (result<0) ? 0 : result;
}

//////////////////////////////////////////////////
// class TFunctionalAproximation
// -----------------------------
// Popis: hleda homogenni oblasti, ktere nahrazuje prumerem hodnot
// Description: looks for homogenous areas and replaces them by mean
// see [Pavlidis82], pg. 61
void TFunctionalAproximation::FilterImage(int start,int count)
{int i,xasize;
	matrix->GetSize(xsize,ysize);
	if ((xsize>=10) && (ysize>=10))
		for (i=start;i<start+count;i++) {
			// I have spend two hours to set these numbers
			// Have a nice time to think about them
			if ((i+1)%10)
				xasize = (int) ( (((long)xsize*((i+1)%10))/10) - (((long)xsize*(i%10))/10));
			else
				xasize = (int) (xsize - (((long)xsize*(i%10))/10));
			FilterArea((int)(((long)xsize*(i%10))/10),xasize,
								 (int)((long)ysize*(i/10)/10),(int) ((long)ysize*((i/10)+1)/10 - (long)ysize*(i/10)/10));
      }
}

void TFunctionalAproximation::FilterArea(int xleft,int xareasize, int ytop, int yareasize)
{ long sum=0;
	int i,j;

	// calculate average
	for (i=ytop;i<ytop+yareasize;i++)
		for (j=xleft;j<xleft+xareasize;j++)
			sum +=matrix->GetElement(j,i);
	mean = (int) (sum/((long)xareasize*yareasize));

	// calculate variance
	sum=0;
	for (i=ytop;i<ytop+yareasize;i++)
		for (j=xleft;j<xleft+xareasize;j++)
			sum +=  abs(matrix->GetElement(j,i) - mean);
	variance = (int) (sum/((long)xareasize*yareasize));

	if ((variance<maxvariance) || (xareasize<2) || (yareasize<2))
		 //	call FilterElement for each pixel
		for (i=ytop;i<ytop+yareasize;i++)
			for (j=xleft;j<xleft+xareasize;j++)
				matrix->SetElement(j,i,FilterElement(j,i));
	else {
		FilterArea(xleft,xareasize/2,ytop,yareasize/2);
		FilterArea(xleft+xareasize/2, xareasize%2 ? xareasize/2+1 : xareasize/2 ,ytop,yareasize/2);
		FilterArea(xleft,xareasize/2,ytop+yareasize/2, yareasize%2 ? yareasize/2+1 : yareasize/2 );
		FilterArea(xleft+xareasize/2, xareasize%2 ? xareasize/2+1 : xareasize/2 ,ytop+yareasize/2, yareasize%2 ? yareasize/2+1 : yareasize/2 );
	}
}

#pragma argsused
Element TFunctionalAproximation::FilterElement(int x,int y)
{
	return mean;
}

