; This file Copyright 1994 Dan Wesnor
;
; This file may be freely distributed as long as no 
; information is changed, no charge is made for
; the file, and the archive in which the file was
; contained is distributed in tact.  Any violation 
; of these conditions requires written authorization 
; by the copyright holder.

/*
	This is a script for the Magic Camera rendering system.
	Magic Camera (MC) uses ray-tracing for realisting rendering
	of shadows, reflections, and transparency.  It is in the
	final phase of completion and should be released shortly.
	Plans are for release as shareware with a nominal charge
	for a full working version.

	See comments for a description of features.
	Comments are either C-style (like this one), or a semi-colon
	followed by text	; like this.
*/

/*
	Primitive objects include triangles and parallelograms,
	both of which may be phong smoothed, spheres, rings (or disks),
	and planes.

	Compound objects, which allow complex shapes to be built,
	include extrusions, spins, skin-over-frame, filled complex
	2-d polygons (with holes), heightfields, boxes, and
	polygon spheres.

	Available patterns are check, wood, marble, brick,
	several types of bitmap wraps, blotch, and clouds.

	Textures include waves and several type of bumpmaps.

	Lamps include point-source, directional, directional 
	spotlights (w/ beam width and beam falloff sharpness
	controls), or spherical extended.

	Variables, which may be used in complex expressions, include
	real numbers, integers, 3d-vectors, and arrays of reals, 2-d vectors,
	and 3-d vectors.  Variables and arrays may be used to facilitate
	animation.
*/



maxoctdepth 5	; controls for the octree, which
maxobcube 2	; is an adaptive automatic object bounding
		; mechanism

maxaadepth 2	; set the maximum number of pixel splits
		; for adaptive anti-aliasing

; a couple of variables to set the board size
real boardsize = 64
real squaresize = boardsize/8


camera {	; set up the camera
	loc	<60, 30, 140>	; location
	target	<3.5*squaresize, 6, 4*squaresize>	; point at
	res 	640 480		; image resolution
	aspect 	1.0		; pixel aspect
	hfov	45		; horizontal field of view
}


lamp {
	direct	<.25, -1.75, -1>	; a directional (non-
					; point source) lamp
	color	<0.5, 0.5, 0.5>		; the lamp color in RGB
}

lamp {
	direct	<-1, -.75, -1>	; another directional lamp
	color	<0.5, 0.5, 0.5>
}

include "bishop.mco"	; include file which contains
			; bishop.obj

/* this little manipulation assures us that we have a bishop
   which will fit on a square of the chessboard
*/

object mybishop.obj
real bishop_scale = squaresize/(2*bishop_radius)
scale bishop.obj <bishop_scale, bishop_scale, bishop_scale>
instance bishop.obj
endobject

/* facetted */

color red {	; build a 'color' type pattern called 'red'
	diff	<1, 0, 0>	; diffuse color in RGB
	scoef	50		; specular coefficient
				; (hot spot tightness)
	srefl	1		; specular reflection
				; (hot spot brightness)
}

translate mybishop.obj <3.5*squaresize, 0, 3.5*squaresize>	
					; put the bishop in position
instance {				; drop it into the scene
	object	mybishop.obj
	subpatt	__bishopsurf	red	; substitute the pattern 'red'
					; for the pattern '__bishopsurf',
					; effectively changing the object's
					; color
}
reset mybishop.obj	; reset the bishop's transformation matrix

translate mybishop.obj <3.5*squaresize, 0, 5.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	red
}
reset mybishop.obj

/* checked */

color blue {	; another simple 'color' pattern
	diff	<0, 0, 1>
	scoef	50
	srefl	1
}

color white {	; another simple 'color' pattern
	diff	<1, 1, 1>
	scoef	50
	srefl	1
}

check blue_white {	; a checkerboard pattern called 'blue_white'
	patt1	blue	; uses, you guessed it, 'blue'
	patt2	white	; and 'white'
	xsize	.75	; the checks are 0.75 units 
	ysize	.75	; square
	zsize	.75
}

/*
	translate and create a blue-white checked bishop
*/

translate mybishop.obj <6.5*squaresize, 0, 4.5*squaresize>	
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	blue_white
}
reset mybishop.obj

/* brick */

color brick_red {	; new 'color' pattern
	diff	<.8, 0, 0>
}

color mortar_white {	; new 'color' pattern
	diff	<.8, .8, .8>
}

brick brick1 {		; a brick pattern
	mortar	mortar_white	; pattern for 'mortar'
	brick	brick_red	; pattern for brick
	xsize	.9		; dimensions of the bricks
	ysize	.45
	zsize	.45
	msize	.1	; thickness of the mortar
	yoffset .4	; offset each (y dimension) row
			; of bricks 0.4 units in X
	zoffset 0	; don't offset any z-dimension
			; row of bricks
}

; move the object into position and instance it with the brick pattern

translate mybishop.obj <5.5*squaresize, 0, 5.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	brick1
}
reset mybishop.obj

/* glass */

color glass {	; another simple 'color' pattern, but this
		; one will be transparent
	diff	<0, 0, 0>	; to be completely clear, we don't
				; add any diffuse color
	refl	<.1, .1, .1>	; glass IS slightly reflective
	trans	<.9, .9, .9>	; the glass transmits 90% of all
				; RGB color
	filt	<.01, .01, .01>	; this controls how much color
				; the glass doesn't transmit per
				; unit length, i.e., how much
				; light is filtered by the glass
	scoef	50	; specular hot spot numbers
	srefl	1
	index	1.67	; the glass's index of refraction
}

; position it and drop it into the scene with the new pattern

translate mybishop.obj <4.5*squaresize, 0, 6.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	glass
}
reset mybishop.obj

/* mirror */

color mirror {	; create a reflective 'color'
	diff	<0, 0, 0>	; if diff was not <0,0,0>, the mirror
				; would appear tinted
	refl	<.9, .9, .9>	; 90% of light is reflected
	scoef	50		; specular reflection numbers
	srefl	1
}

; movie it, drop it with new pattern

translate mybishop.obj <2.5*squaresize, 0, 4.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	mirror
}
reset mybishop.obj

/* wooden */

color light_brown {	/* new 'color' for wood */
	diff	<0.7, 0.5, 0.0>
}

color dark_brown {	/* new 'color' for wood */
	diff	<0.3, 0.1, 0.0>
}

wood light_wood {	; a wood pattern
	patt	light_brown	; the meat of the wood
	grain	dark_brown	; the pattern to use for the grain
	scale	10		; scale the pattern so it appears
				; the right size
}

; move it, drop it

translate mybishop.obj <2.5*squaresize, 0, 6.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	light_wood
}
reset mybishop.obj

/* blotch */

; new colors

color a_red { diff <1, 0, 0> }
color a_blue { diff <0, 0, 1> }
color a_green { diff <0, 1, 0> }
color a_yellow { diff <1, 1, 0> }
color a_white { diff <1, 1, 1> }

blotch asvpt {			; create a blotch pattern
	patt	a_red		; a blotch is basically
	patt	a_blue		; a mildly interesting blend
	patt	a_white		; of several colors
	patt	a_green
	patt	a_yellow
	scale	3
}

; move it, drop it

translate mybishop.obj <6.5*squaresize, 0, 6.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	asvpt
}
reset mybishop.obj

/* cloudy */

; colors for clouds

color cl_white {
	diff	<1, 1, 1>
}

color cl_blue	{
	diff	<.3, .3, 1>
}

clouds cloudy {
	clouds	cl_white	; the clouds are white on a
	sky	cl_blue		; blue sky
	scale	.5		; set the size
			; the apparent turbulence of the
			; clouds is also controllable
}

; place the object

translate mybishop.obj <1.5*squaresize, 0, 5.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	cloudy
}
reset mybishop.obj

/* marble */

; color for marble

color ma_grain {
	diff	<.3, .2, 0>
}

color ma_white {
	diff	<.9, .8, .5>
}

marble marb {
	patt	ma_white	; once again, this is the 'meat'
	grain	ma_grain	; and the grain of the marble
	scale	3		; size it
	pow	.75		; control the sharpness between the
				; meat and the grain
}

; place the object

translate mybishop.obj <1.5*squaresize, 0, 3.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	marb
}
reset mybishop.obj

/* waves */

color wa_yellow {
	diff	<1, 1, 0>
}

waves wavy {
	scale	1.2	; set the size of the texture,
			; also the apparent height and
			; complexity could have been changed
}

; notice that NULLTEXT is the name used when substituting for
; a texture that does not exist

translate mybishop.obj <4.5*squaresize, 0, 4.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	wa_yellow
	subtext	NULLTEXT	wavy
}
reset mybishop.obj

/* texture wrapped */

/*
iff "mapfiles/tartan.iff" wrap.map	; load the iff file
					; and name it wrap.map

wrapcy wrappatt {
	bitmap	wrap.map	; use 'wrap.map'
	patt	red		; characteristics like
				; reflectivity and trasparency
				; will be pulled from the pattern
				; called 'red'
	height	bishop_height*bishop_scale/6
				; the height of the cylinder used
				; as an intermediate surface by the wrap
	xrep	4		; wrap the bitmap around the cylinder once
	filter	1		; turn off pre-filtering of the image
	dodiff			; the bitmap will be used to modulate
				; the diffuse color of the object.
				; it could also modulate reflection
				; and transmission
}

; place the object

translate mybishop.obj <5.5*squaresize, 0, 3.5*squaresize>
instance {
	object	mybishop.obj
	subpatt	__bishopsurf	wrappatt
}
*/
reset mybishop.obj

/* the checkerboard */

/* the chess pieces will be placed on a checkerboard made
   of squares of alternating mirrors and reflective marble
*/

color mirror_black {
	diff	<0, 0, 0>
	refl	<.55, .55, .55>
	scoef	50
	srefl	1
}


color mirror_white {
	diff	<.75, .75, .75>
	refl	<.25, .25, .25>
	scoef	50
	srefl	1
}

marble black_marble {
	patt	mirror_black
	grain	mirror_white
	scale	.075
	pow	.75
}

marble white_marble {
	patt	mirror_white
	grain	mirror_black
	scale	.6
	pow	.75
}

check checker {
	patt1	white_marble
	patt2	mirror_black
	xsize	8
	zsize	8
	ysize	1
}

/* this box is the checker board itself */

box {
	loc	<0, 0, 0>
	v1	<boardsize, 0, 0>
	v2	<0, -boardsize/32, 0>
	v3	<0, 0, boardsize>
	patt	checker
}

