/****************************************************************************

    Module  : INI_File.c

	Contains all funtions to access / modify .INI file

    3.0 beta 3 : Created this module
    3.0 beta 5 : Added ConvertINI function to import v2.5 INI file
    3.0 beta 7 : Corrected bugs in ReadSlotOptions & ReadDockOptions
*****************************************************************************/


#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>
#include <shellapi.h>
#include "freedock.h"
#include "dock_reg.h"

/**********************************************************
	Global Data for the various system options we need
**********************************************************/
extern GLOBAL_OPTIONS gOptions;

/**********************************************************
	Name of the .INI file, local to this module
	Different name depending on OS
**********************************************************/
#ifdef WIN32
	static char           *IniFile = "FreeDk32.ini";
#else
	static char           *IniFile = "FreeDk16.ini";
#endif


/****************************************************************
	ReadGlobalOptions()
	This function reads in all the options which have a 
	global effect on FreeDock and all the Slots/sub-docks.
	These global options are read into the Global Options
	data structure gOptions. Any read errors are ignored
	and sensible defaults assumed.
****************************************************************/
void ReadGlobalOptions( void ){

    char     TmpBuffer[MAX_FPATH_LEN];
	char 	 SectionName[MAX_FPATH_LEN];

    // Setup configurable non-publicised options (Global to ALL FreeDock Instances
	sprintf( SectionName, "Global All" );
    gOptions.FixedTitle   = GetPrivateProfileInt(SectionName, "FixedTitle", 0, IniFile);
    gOptions.SleepTime    = GetPrivateProfileInt(SectionName, "SleepTime", 1, IniFile);

	sprintf( SectionName, "Global %d",  gOptions.RootDockID  );
	/*********************************
		Initialise Mail box Flag
	*********************************/                           
    gOptions.bMailInBox = FALSE;          
    
	// Read orientation of root dock
    GetPrivateProfileString(SectionName, "Position", "Horizontal", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'H') {
   	    gOptions.RootOrientation = DOCK_HORZ;
    } 
   	else{
        gOptions.RootOrientation = DOCK_VERT;
	}

	// Read Location of RootDock
    gOptions.RootDockLeft = GetPrivateProfileInt(SectionName, "RootDockLeft", 0, IniFile);
   	gOptions.RootDockTop = GetPrivateProfileInt(SectionName, "RootDockTop", 0, IniFile);

	// Read in and setup the slot scale.
    gOptions.SlotScale   = GetPrivateProfileInt(SectionName, "SlotScale", DEF_SLOT_SCALE, IniFile);
 	gOptions.SlotHeight  = gOptions.SlotWidth = gOptions.SlotScale;

    // Setup configurable non-publicised options
    gOptions.MaxDockSize  = GetPrivateProfileInt(SectionName, "MaxDockSize", DEF_MAX_SLOTCOUNT, IniFile);

	// Handle Win Exit Options
    GetPrivateProfileString(SectionName, "WinExitActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.WinExitActive = TRUE;
        GetPrivateProfileString(SectionName, "WinExitConfirm", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            gOptions.WinExitConfirm = FALSE;
        else
            gOptions.WinExitConfirm = TRUE;
    } 
    else
        gOptions.WinExitActive = FALSE;

	// Handle Pop-In-On-Exec Option
    GetPrivateProfileString(SectionName, "PopInOnExec", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.PopInOnExec = TRUE;
    } 
    else
        gOptions.PopInOnExec = FALSE;

	// Handle pop-up Titles Options
    GetPrivateProfileString(SectionName, "TitlesActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.TitlesActive = TRUE;
    } 
    else
        gOptions.TitlesActive = FALSE;

    // Read Title Timer value (if Any)
    gOptions.TitleTimer = GetPrivateProfileInt(SectionName, "TitleTImer", DEF_TITLE_TIMEOUT, IniFile);

	// Handle Clock Options
    GetPrivateProfileString(SectionName, "ClockActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.ClockActive = TRUE;
    } 
    else
        gOptions.ClockActive = FALSE;

	// Handle Mail Options
    GetPrivateProfileString(SectionName, "MailActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.MailActive = TRUE;
        GetPrivateProfileString(SectionName, "MailSound", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            gOptions.MailSound = FALSE;
        else
            gOptions.MailSound = TRUE;

        GetPrivateProfileString(SectionName, "MailPath", "c:\\mail.box", gOptions.MailPath, MAX_FPATH_LEN, IniFile);
        /* GetPrivateProfileString Cannot handle long, so do it manually */
        GetPrivateProfileString(SectionName, "MailBoxSize", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &gOptions.MailBoxSize);

        GetPrivateProfileString(SectionName, "MailBoxDateTime", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &gOptions.MailBoxDateTime);

        gOptions.MailFreq = GetPrivateProfileInt(SectionName, "MailFreq", 10, IniFile);
    } 
    else
        gOptions.MailActive = FALSE;
	
	// Handle Hide Dialog On Grab Option
    GetPrivateProfileString(SectionName, "HideDlgOnGrab", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.HideDlgOnGrab = TRUE;
    else
        gOptions.HideDlgOnGrab = FALSE;

	// Handle Always on top option
    GetPrivateProfileString(SectionName, "AlwaysOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.AlwaysOnTop = TRUE;
    else
        gOptions.AlwaysOnTop = FALSE;

	// Handle Single Click Start Option
    GetPrivateProfileString(SectionName, "SingleClickStart", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.SingleClickStart = TRUE;
    else
        gOptions.SingleClickStart = FALSE;

	// Handle Max View Option
    GetPrivateProfileString(SectionName, "MaxView", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.MaxView = TRUE;
    else
        gOptions.MaxView = FALSE;

	// Handle DockLocked Option
    GetPrivateProfileString(SectionName, "DockLocked", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.DockLocked = TRUE;
    else
        gOptions.DockLocked = FALSE;

	// Read ShutDown Options
    GetPrivateProfileString(SectionName, "ShutdownAction", "LogOff", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if ( !stricmp( TmpBuffer, "LogOff" )) {
   	    gOptions.ShutdownAction = FD_LOGOFF;
    } 
    else if ( !stricmp( TmpBuffer, "ShutDown" )) {
   	    gOptions.ShutdownAction = FD_SHUTDOWN;
    } 
    else if ( !stricmp( TmpBuffer, "Reboot" )) {
   	    gOptions.ShutdownAction = FD_REBOOT;
    } 
    else {
   	    gOptions.ShutdownAction = FD_LOCKUP;
    } 
}


void ReadDockOptions( DOCK_STRUCT *Dock ){

	char 			BlockLabel[MAX_DOCK_ID_LEN];

	sprintf( BlockLabel, "Dock Options %s", Dock->DockID );

    /**********************************************************************
        Setup the dock options
    ***********************************************************************/

    Dock->SlotCount = GetPrivateProfileInt(BlockLabel, "SlotCount", gOptions.MaxDockSize, IniFile);
}

/****************************************************************
   Read in and process all slot data, also carry out any setup
   which is required.
****************************************************************/
 
void ReadSlotOptions( DOCK_STRUCT *Dock )
{
    char            SlotSection[MAX_FPATH_LEN], TmpBuffer[MAX_FPATH_LEN];
    int             StartSlot, SlotNum;
    int             i;
	SLOT_ENTRY	   *Slot;
	char 			SubDockID[MAX_DOCK_ID_LEN];
	DOCK_STRUCT    *SubDock;

	Slot = Dock->Slot;
                          
    /**********************************************************************
       Set the index to the first free slot to the slot after the title
       slot	and any actice special slots.
    ***********************************************************************/
    StartSlot = 0;

	// Only the root Dock can have special built-in slots.
	if(Dock->DockType == ROOT_DOCK){
	    /**********************************************************************
    	       Start by initialising the first slot to be the title icon
	    ***********************************************************************/
    	Slot[StartSlot].SlotType = SLOT_SPECIAL_TITLE;
	    Slot[StartSlot].Index = StartSlot;
	    UtilLoadIcon( Dock, &(Slot[StartSlot]) );
        strcpy( Slot[StartSlot].Title, "FreeDock Title Slot" );
		StartSlot ++;

		if(gOptions.WinExitActive){
		    Slot[StartSlot].Index = StartSlot;
    	    Slot[StartSlot].SlotType = SLOT_SPECIAL_EXIT;
        	UtilLoadIcon(Dock, &(Slot[StartSlot]) );
            strcpy( Slot[StartSlot].Title, "Built-In Slot : Exit Windows" );
			StartSlot ++;
		}
		if(gOptions.MailActive){
		    Slot[StartSlot].Index = StartSlot;
    	    Slot[StartSlot].SlotType = SLOT_SPECIAL_MAIL;
	       	UtilLoadIcon(Dock, &(Slot[StartSlot]) );
            strcpy( Slot[StartSlot].Title, "Built-In Slot : Check Mail" );
			StartSlot ++;
		}
		if(gOptions.ClockActive){
		    Slot[StartSlot].Index = StartSlot;
        	Slot[StartSlot].SlotType = SLOT_SPECIAL_CLOCK;
	        UtilLoadIcon(Dock, &(Slot[StartSlot]) );
            strcpy( Slot[StartSlot].Title, "Built-In Slot : Clock" );
			StartSlot ++;
		}
	}

    /*******************************************************************************
       SlotNum is used to index the sections in the .INI file, each slot is
       uniquely numbered in a section of the type [Slot xx] where xx is the slot
       number.
       The for loop attempts to read the AppName field from each slot, if this
       fails the slot is empty so the slot type is set to SLOT_FREE and the next
       slot is read. If the reading of AppName succeeds, the slot is used and
       the rest of the details are read in.
    *******************************************************************************/
    SlotNum = 0;

    for (i = StartSlot; i < gOptions.MaxDockSize; i++) {

       	UtilEmptySlot( &(Slot[i]) );	// Ensure Default Empty struct is in Slot 


		SlotNum++;

        Slot[i].Index = i;
		Slot[i].ID = SlotNum;

        sprintf(SlotSection, "Slot %s.%d", Dock->DockID, SlotNum);
		
		// First Readin the SlotType to see if we are dealing with a Sub-Dock Slot

        GetPrivateProfileString(SlotSection, "SlotType", "Normal", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'S') {

            Slot[i].SlotType = SLOT_SPECIAL_SUB;

            GetPrivateProfileString(SlotSection, "IconFile", "", Slot[i].IconFile, MAX_FPATH_LEN, IniFile);
   	        Slot[i].IconPos = GetPrivateProfileInt(SlotSection, "IconPos", 0, IniFile);
       	    // Load the Icon into the slot's own bitmap & update the IconTotal Field
            UtilLoadIcon( Dock, &Slot[i] );

            GetPrivateProfileString(SlotSection, "Title", "SubDock", Slot[i].Title, MAX_FPATH_LEN, IniFile);

			sprintf( SubDockID, "%s.%d", Dock->DockID, Slot[i].ID );
			Slot[i].hwndSubDock = CreateDock( SUB_DOCK, Dock->hwndDock, SubDockID );
			SubDock = GetDockStruct( Slot[i].hwndSubDock );
			SubDock->ParentSlot = &Slot[i];
//			SetSubDockWinPos( SubDock, FALSE );		// Hide SubDock;
			// SlotOptions are read in response to WM_CREATE message
		}
        else if (toupper(TmpBuffer[0]) == 'N'){
			// Normal Slot, so read in all other slot details
	        GetPrivateProfileString(SlotSection, "AppName", "", Slot[i].AppName, MAX_FPATH_LEN, IniFile);
    	    if (Slot[i].AppName[0] != '\0') {
        	    GetPrivateProfileString(SlotSection, "CmdLine", "", Slot[i].CmdLine, MAX_CMDLINE_LEN, IniFile);
            	GetPrivateProfileString(SlotSection, "RunTimeDir", "", Slot[i].RunTimeDir, MAX_FPATH_LEN, IniFile);

	            Slot[i].StartState = GetPrivateProfileInt(SlotSection, "StartState", START_NORMAL, IniFile);
		
                GetPrivateProfileString(SlotSection, "Title", "", Slot[i].Title, MAX_FPATH_LEN, IniFile);
                if( Slot[i].Title[0] =='\0' ){
                    strcpy( Slot[i].Title, Slot[i].AppName );
                }

        	    GetPrivateProfileString(SlotSection, "StartOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
            	if (toupper(TmpBuffer[0]) == 'N')
                	Slot[i].StartOnTop = FALSE;
	            else
    	            Slot[i].StartOnTop = TRUE;

        	    GetPrivateProfileString(SlotSection, "StartSeparate", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
            	if (toupper(TmpBuffer[0]) == 'N')
                	Slot[i].StartSeparate = FALSE;
	            else
    	            Slot[i].StartSeparate = TRUE;
			
            	Slot[i].WinX = GetPrivateProfileInt(SlotSection, "WinX", 0, IniFile);
	            Slot[i].WinY = GetPrivateProfileInt(SlotSection, "WinY", 0, IniFile);
    	        Slot[i].WinWidth = GetPrivateProfileInt(SlotSection, "WinWidth", DEF_STORE_W, IniFile);
        	    Slot[i].WinHeight = GetPrivateProfileInt(SlotSection, "WinHeight", DEF_STORE_H, IniFile);
				
	            GetPrivateProfileString(SlotSection, "IconFile", "", Slot[i].IconFile, MAX_FPATH_LEN, IniFile);

    	        Slot[i].IconPos = GetPrivateProfileInt(SlotSection, "IconPos", 0, IniFile);

        	    // Load the Icon into the slot's own bitmap & update the IconTotal Field
	            UtilLoadIcon( Dock, &Slot[i] );

    	        Slot[i].SlotType = SLOT_USED;
			}
	        else
    	        Slot[i].SlotType = SLOT_FREE;
        } 		
    }
}

void WriteGlobalOptions(void){
										 
    char     PosStr[2][11] =  { {"Horizontal"}, {"Vertical"}};
    char     TmpBuffer[MAX_FPATH_LEN];
	char 	 SectionName[MAX_FPATH_LEN];

	sprintf( SectionName, "Global %d",  gOptions.RootDockID  );
	// Global Options are Always stored in the [Global xx] block

    WritePrivateProfileString(SectionName, "Position", PosStr[gOptions.RootOrientation], IniFile);

    sprintf(TmpBuffer, "%d", gOptions.RootDockLeft);
    WritePrivateProfileString(SectionName, "RootDockLeft", TmpBuffer, IniFile);
    sprintf(TmpBuffer, "%d", gOptions.RootDockTop);
    WritePrivateProfileString(SectionName, "RootDockTop", TmpBuffer, IniFile);

    WritePrivateProfileString(SectionName, "MaxView", (gOptions.MaxView) ? "Yes" : "No", IniFile);
    WritePrivateProfileString(SectionName, "AlwaysOnTop", (gOptions.AlwaysOnTop) ? "Yes" : "No", IniFile);
    WritePrivateProfileString(SectionName, "SingleClickStart", (gOptions.SingleClickStart) ? "Yes" : "No", IniFile);
    WritePrivateProfileString(SectionName, "HideDlgOnGrab", (gOptions.HideDlgOnGrab) ? "Yes" : "No", IniFile);

    WritePrivateProfileString(SectionName, "ClockActive", (gOptions.ClockActive) ? "Yes" : "No", IniFile);

    WritePrivateProfileString(SectionName, "TitlesActive", (gOptions.TitlesActive) ? "Yes" : "No", IniFile);

    WritePrivateProfileString(SectionName, "PopInOnExec", (gOptions.PopInOnExec) ? "Yes" : "No", IniFile);

    WritePrivateProfileString(SectionName, "MailActive", (gOptions.MailActive) ? "Yes" : "No", IniFile);
    WritePrivateProfileString(SectionName, "MailPath", gOptions.MailPath, IniFile);
    WritePrivateProfileString(SectionName, "MailSound", (gOptions.MailSound) ? "Yes" : "No", IniFile);
    sprintf(TmpBuffer, "%d", gOptions.MailFreq);
    WritePrivateProfileString(SectionName, "MailFreq", TmpBuffer, IniFile);

    WritePrivateProfileString(SectionName, "WinExitActive", (gOptions.WinExitActive) ? "Yes" : "No", IniFile);
    WritePrivateProfileString(SectionName, "WinExitConfirm", (gOptions.WinExitConfirm) ? "Yes" : "No", IniFile);

    WritePrivateProfileString(SectionName, "DockLocked", (gOptions.DockLocked) ? "Yes" : "No", IniFile);

    sprintf(TmpBuffer, "%lu", gOptions.MailBoxDateTime);
    WritePrivateProfileString(SectionName, "MailBoxDateTime", TmpBuffer, IniFile);

    sprintf(TmpBuffer, "%lu", gOptions.MailBoxSize);
    WritePrivateProfileString(SectionName, "MailBoxSize", TmpBuffer, IniFile);

    sprintf(TmpBuffer, "%d", gOptions.SlotScale);
    WritePrivateProfileString(SectionName, "SlotScale", TmpBuffer, IniFile);

    switch( gOptions.ShutdownAction ){
        case FD_LOGOFF :
             WritePrivateProfileString(SectionName, "ShutdownAction", "LogOff", IniFile);
             break;
        
        case FD_SHUTDOWN :
             WritePrivateProfileString(SectionName, "ShutdownAction", "ShutDown", IniFile);
             break;
        
        case FD_REBOOT :
             WritePrivateProfileString(SectionName, "ShutdownAction", "Reboot", IniFile);
             break;
        
        default : 
             WritePrivateProfileString(SectionName, "ShutdownAction", "LockUp", IniFile);
             break;
    }
}

void WriteDockOptions ( DOCK_STRUCT *Dock ){

    char    TmpBuffer[MAX_FPATH_LEN];
	char 	BlockLabel[MAX_DOCK_ID_LEN];
	int		i;

	sprintf( BlockLabel, "Dock Options %s", Dock->DockID );

    sprintf(TmpBuffer, "%d", Dock->SlotCount);
    WritePrivateProfileString(BlockLabel, "SlotCount", TmpBuffer, IniFile);

	for( i=0; i < Dock->SlotCount; i++){
		if( Dock->Slot[i].SlotType == SLOT_SPECIAL_SUB ){
		    {
				WriteDockOptions( GetDockStruct(Dock->Slot[i].hwndSubDock) );
			}
		}
	}
}


/****************************************************************
	Deletes the INI file entries for a slot and any child
	slots which it may have.
****************************************************************/
void DeleteSlotOptions( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot ){

    char    SlotName[MAX_FPATH_LEN];
	char    DockName[MAX_FPATH_LEN];
	int		i;

	if( (Slot->SlotType == SLOT_USED) || (Slot->SlotType == SLOT_SPECIAL_SUB) ){
	    sprintf(SlotName, "Slot %s.%d", Dock->DockID, Slot->ID);
    	WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
		// Delete Dock Options for Sub-Dock
	    sprintf(DockName, "Dock Options %s.%d", Dock->DockID, Slot->ID);
   		WritePrivateProfileString(DockName, NULL, NULL, IniFile);
	}

	if( Slot->SlotType == SLOT_SPECIAL_SUB ){
	    {
			DOCK_STRUCT *SubDock;
			SubDock = GetDockStruct( Slot->hwndSubDock );
			for( i=0; i < SubDock->SlotCount; i++ ){
				DeleteSlotOptions( SubDock, &(SubDock->Slot[i]) );
			}
		}
	}
}

/* Updates a slot entry within an INI file */
void WriteSlotOptions( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot )
{
    char            TmpBuffer[MAX_FPATH_LEN], SlotName[MAX_FPATH_LEN];

		if( (Slot->SlotType == SLOT_USED) ||
			(Slot->SlotType == SLOT_FREE) ||
			(Slot->SlotType == SLOT_SPECIAL_SUB) ){
		            sprintf(SlotName, "Slot %s.%d", Dock->DockID, Slot->ID);
					// Erase fields which are not required
        		    WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
        }
        if (Slot->SlotType == SLOT_USED) {
            WritePrivateProfileString(SlotName, "SlotType", "Normal", IniFile);
            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
            WritePrivateProfileString(SlotName, "AppName", Slot->AppName, IniFile);
            WritePrivateProfileString(SlotName, "CmdLine", Slot->CmdLine, IniFile);
            WritePrivateProfileString(SlotName, "RunTimeDir", Slot->RunTimeDir, IniFile);
            WritePrivateProfileString(SlotName, "Title", Slot->Title, IniFile);

            sprintf(TmpBuffer, "%d", Slot->StartState);
            WritePrivateProfileString(SlotName, "StartState", TmpBuffer, IniFile);
            WritePrivateProfileString(SlotName, "StartOnTop", (Slot->StartOnTop) ? "Yes" : "No", IniFile);
            WritePrivateProfileString(SlotName, "StartSeparate", (Slot->StartSeparate) ? "Yes" : "No", IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinX);
            WritePrivateProfileString(SlotName, "WinX", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinY);
            WritePrivateProfileString(SlotName, "WinY", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinWidth);
            WritePrivateProfileString(SlotName, "WinWidth", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinHeight);
            WritePrivateProfileString(SlotName, "WinHeight", TmpBuffer, IniFile);

            WritePrivateProfileString(SlotName, "IconFile", Slot->IconFile, IniFile);

            sprintf(TmpBuffer, "%d", Slot->IconPos);
            WritePrivateProfileString(SlotName, "IconPos", TmpBuffer, IniFile);
        }
        if (Slot->SlotType == SLOT_SPECIAL_SUB) {
			DOCK_STRUCT *SubDock;
			int 	     i;

			SubDock = GetDockStruct( Slot->hwndSubDock );
			// Ensure SubDock ID's are upto date before writing
		    UpdateSubDockID ( SubDock, Slot->ID );
			// Write Slot Type
            WritePrivateProfileString(SlotName, "SlotType", "SubDock", IniFile);
			//Write Icon File and Position
            WritePrivateProfileString(SlotName, "IconFile", Slot->IconFile, IniFile);
            sprintf(TmpBuffer, "%d", Slot->IconPos);
            WritePrivateProfileString(SlotName, "IconPos", TmpBuffer, IniFile);

            WritePrivateProfileString(SlotName, "Title", Slot->Title, IniFile);

			// Write out all sub-dock slot options
			for( i=0; i < SubDock->SlotCount; i++ ){
			 	WriteSlotOptions( SubDock, &(SubDock->Slot[i]) );
			}
		}
}


/******************************************************************
    Convert the .INI file into the new format, from v2.5 to v3.0
    If required, otherwise returns immediately
    This function is only supported in WIN32, since the 
    GetPrivateProfileSection & WritePrivateProfileSection calls
    are not available in 16-bit windows.
******************************************************************/
void ConvertINIFile( void ){
    
#ifdef WIN32

    char            TmpBuffer[MAX_FPATH_LEN];
	char 	        SectionName[MAX_FPATH_LEN];
    char            ConvertBuffer[1024];
    int             tmpInt, i, MaxDockSize;

	// Read orientation of root dock from v3.0 block
    GetPrivateProfileString("Global 1", "Position", "Invalid", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if ( stricmp( TmpBuffer, "Invalid" ) ){
         // .INI file is in the new format already, so return
         return;
    }

    // INI file needs to be converted.
    // Convert entire old [Main Options] to [Global 1]
    if( GetPrivateProfileSection ( "Main Options", 
                                   ConvertBuffer,
                                   1024,
                                   IniFile ) > 0 ){
        // Write new section
        WritePrivateProfileSection ( "Global 1", 
                                     ConvertBuffer,
                                     IniFile );
    }
    
    // Modify Changed field names
    // DockTop
    tmpInt = GetPrivateProfileInt( "Main Options", "DockTop", 0, IniFile );
    sprintf(TmpBuffer, "%d", tmpInt);
    // Write new value
    WritePrivateProfileString("Global 1", "RootDockTop", TmpBuffer, IniFile);
    // erase old value
    WritePrivateProfileString("Global 1", "DockTop", NULL, IniFile);

    // Dock Left
    tmpInt = GetPrivateProfileInt( "Main Options", "DockLeft", 0, IniFile );
    sprintf(TmpBuffer, "%d", tmpInt);
    // Write new value
    WritePrivateProfileString("Global 1", "RootDockLeft", TmpBuffer, IniFile);
    // erase old value
    WritePrivateProfileString("Global 1", "DockLeft", NULL, IniFile);

    tmpInt = GetPrivateProfileInt( "Main Options", "SlotCount", 0, IniFile );
    sprintf(TmpBuffer, "%d", tmpInt);
    // Write new value
    WritePrivateProfileString("Dock Options 1", "SlotCount", TmpBuffer, IniFile);
    // erase old value
    WritePrivateProfileString("Global 1", "SlotCount", NULL, IniFile);

    // Converted All old Global Options Options
    // Read MaxDock Size
    MaxDockSize = GetPrivateProfileInt( "Main Options", "MaxDockSize", 32, IniFile );

    // Now Delete the old [Main Options] Section
    WritePrivateProfileString("Main Options", NULL, NULL, IniFile);

    // Now we need to process each slot section, renameing it
    // from [Slot x] to [Slot 1.x]
    for( i = 1; i < MaxDockSize; i++ ){
    	sprintf( SectionName, "Slot %d", i );
        if( GetPrivateProfileSection ( SectionName, 
                                       ConvertBuffer,
                                       1024,
                                       IniFile ) > 0 ){
            // Delete old section
            WritePrivateProfileString(SectionName, NULL, NULL, IniFile);
        	sprintf( SectionName, "Slot 1.%d", i );
            // Write new section
            WritePrivateProfileSection ( SectionName, 
                                         ConvertBuffer,
                                         IniFile );
        }
    }       
#endif
}
