/*  Project filter
					DCS, UP Olomouc
					Copyright  1995. All Rights Reserved.

					SUBSYSTEM:    filter.exe Application
					FILE:         fourier.cpp
					AUTHOR:       Robert Batusek

		rozhrani tridy TFourier - tridy realizujici Fourierovu transformaci

		interface of TFourier class - class realizing Fourier transformation
*/

#include "filter.h"

#ifndef _fourier_h
#define _fourier_h

/////////////////////////////////////////////////
// class TComplexFilter
// --------------------
// Popis: trida obsahujici komplexni matici
// Description: class containing complex matrix
//
class TComplexFilter : public TFilter {
	public:
		TComplexFilter(TMatrix &m, BOOL CreateCopy);
		TComplexFilter(TComplexMatrix &m, BOOL CreateCopy);

		virtual BOOL SetMatrix(TMatrix &m,BOOL CreateCopy);
		virtual BOOL FinishFiltering();

	protected:
		virtual Element FilterElement(int x,int y){return 0;};
};


/////////////////////////////////////////////////
// class TFTFilter
// ---------------
// Popis: bazova trida pro realizaci FFT (rychla Fourierova transformace)
// Description: base class used for realizing FFT (fast Fourier transform)
//
class TFTFilter: public TComplexFilter {
	public:
		TFTFilter(TMatrix &m, BOOL CreateCopy)
			:TComplexFilter(m,CreateCopy) {};
		TFTFilter(TComplexMatrix &m, BOOL CreateCopy)
			:TComplexFilter(m,CreateCopy) {};

		virtual void FilterImage(int start = 0,int count = 100);
		virtual BOOL StartFiltering();
		virtual BOOL FinishFiltering();

	protected:
		Complex *p,*w,*t;
		int outN;

		virtual void LineTransform(Complex p[], int N, int k);
		virtual void RowTransform(long from, long to);
		virtual void ColumnTransform(long from, long to);
};

/////////////////////////////////////////////////
// class TFFT
// -------------
// Popis: trida realizujici FFT (rychlou Fourierovu transformaci)
// Description: class realizing FFT (fast Fourier transform)
//
class TFFT : public TFTFilter {
	public:
		TFFT(TMatrix &m, BOOL CreateCopy)
			:TFTFilter(m,CreateCopy) {};

		virtual BOOL StartFiltering();
};


/////////////////////////////////////////////////
// class TIFFT
// -------------
// Popis: trida realizujici inverzni FFT (rychlou Fourierovu transformaci)
// Description: class realizing inverse FFT (fast Fourier transform)
//
class TIFFT : public TFTFilter {
	public:
		TIFFT(TComplexMatrix &m, BOOL CreateCopy)
			:TFTFilter(m,CreateCopy) {};

		virtual BOOL StartFiltering();
		virtual BOOL FinishFiltering();

	protected:
		virtual void RowTransform(long from, long to);
};


/////////////////////////////////////////////////
// class TFrequencyFilter
// --------------------
// Popis: abstraktni trida predka pro filtry ve frekvencni oblasti
// Description: abstract base class for filters in frequency spectrum
//
class TFrequencyFilter: public TComplexFilter {
	public:
		TFrequencyFilter(TComplexMatrix &m, BOOL CreateCopy,int r);

		virtual void FilterImage(int start = 0,int count = 100);
	protected:
		float radius,xmiddle,ymiddle;

		virtual Complex FilterComplexElement(int x,int y);
		virtual BOOL ChangePixel(int x, int y) = 0;
};

/////////////////////////////////////////////////
// class TLowPass
// --------------------
// Popis: Dolni propust
// Description: lowpass filtering
//
class TLowPass: public TFrequencyFilter {
	public:
		TLowPass(TComplexMatrix &m, BOOL CreateCopy,float r)
			:TFrequencyFilter(m,CreateCopy,r){};

	protected:
		virtual BOOL ChangePixel(int x, int y);
};


/////////////////////////////////////////////////
// class THighPass
// --------------------
// Popis: Horni propust
// Description: highpass filtering
//
class THighPass: public TFrequencyFilter {
	public:
		THighPass(TComplexMatrix &m, BOOL CreateCopy,float r)
			:TFrequencyFilter(m,CreateCopy,r){};

	protected:
		virtual BOOL ChangePixel(int x, int y);
};

/////////////////////////////////////////////////
// class TBandFilter
// --------------------
// Popis: Pasmovy filtr
// Description: Band filter
//
class TBandFilter: public TFrequencyFilter {
	public:
		TBandFilter(TComplexMatrix &m, BOOL CreateCopy,float r,float r2);

	protected:
		float secondradius;
};

/////////////////////////////////////////////////
// class TBandPass
// --------------------
// Popis: Pasmova propust
// Description: bandpass filtering
//
class TBandPass: public TBandFilter {
	public:
		TBandPass(TComplexMatrix &m, BOOL CreateCopy,float r,float r2)
			:TBandFilter(m,CreateCopy,r,r2){};

	protected:
		virtual BOOL ChangePixel(int x, int y);
};

/////////////////////////////////////////////////
// class TBandStop
// --------------------
// Popis: Opak pasmove propusti (nevim, jak to prelozit)
// Description: bandstop filtering
//
class TBandStop: public TBandFilter {
	public:
		TBandStop(TComplexMatrix &m, BOOL CreateCopy,float r,float r2)
			:TBandFilter(m,CreateCopy,r,r2){};

	protected:
		virtual BOOL ChangePixel(int x, int y);
};

/////////////////////////////////////////////////
// class TButterworthFilter
// --------------------
// Popis: filtry pana Butterwortha - meni se kazdy pixel
// Description: Butterworth's filters - every pixel is changing
//
class TButterworthFilter: public TFrequencyFilter {
	public:
		TButterworthFilter(TComplexMatrix &m, BOOL CreateCopy,float r)
			:TFrequencyFilter(m,CreateCopy,r){};

	protected:
		virtual BOOL ChangePixel(int x, int y){return TRUE;}
};

/////////////////////////////////////////////////
// class TButterworthLowpass
// --------------------
// Popis: filtry pana Butterwortha - dolni propust
// Description: Butterworth's filters - lowpass
//
class TButterworthLowpass: public TButterworthFilter {
	public:
		TButterworthLowpass(TComplexMatrix &m, BOOL CreateCopy,float r)
			:TButterworthFilter(m,CreateCopy,r){};

	protected:
		virtual Complex FilterComplexElement(int x,int y);
};

/////////////////////////////////////////////////
// class TButterworthHighpass
// --------------------------
// Popis: filtry pana Butterwortha - horni propust
// Description: Butterworth's filters - highpass
//
class TButterworthHighpass: public TButterworthFilter {
	public:
		TButterworthHighpass(TComplexMatrix &m, BOOL CreateCopy,float r)
			:TButterworthFilter(m,CreateCopy,r){};

	protected:
		virtual Complex FilterComplexElement(int x,int y);
};

#endif
