//	catfish.cpp  -  main application sample code  -  1.1
//
//	This is a part of the MetaKit library.
//	Copyright (c) 1996 Meta Four Software.
//	All rights reserved.
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "catfish.h"
#include "setupdlg.h"

	#include <dos.h>	// _dos_findfirst in GetCatalogDate
	
#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#define IDM_ABOUTBOX 0x0010

/////////////////////////////////////////////////////////////////////////////

	CTheApp ThisApp;

/////////////////////////////////////////////////////////////////////////////
// Use a simple version of localized date, time, and number formating.

	static CString	sShortDate	= "MM/dd/yy";	// "d.M.yyyy", etc
	static bool		iTime		= false;		// true if 24h format
	static bool		iTLZero		= true;        	// true if hour has 2 digits
	static char		sThousand	= ',';      	// thousands separator
	static char		sTime		= ':';      	// time separator

	static void SetInternationalSettings()
	{
		iTime = GetProfileInt("intl", "iTime", 0) != 0;
		iTLZero = GetProfileInt("intl", "iTLZero", 1) != 0;
		
		char buf [30];
		
		if (GetProfileString("intl", "sShortDate", "MM/dd/yy", buf, sizeof buf))
			sShortDate = buf;
		
		if (GetProfileString("intl", "sThousand", ",", buf, sizeof buf))
			sThousand = *buf;
		
		if (GetProfileString("intl", "sTime", ":", buf, sizeof buf))
			sTime = *buf;
	}
		
/////////////////////////////////////////////////////////////////////////////
// Convert a number to comma-separated format, grouped in units of three.
// Optionally prefix with spaces (assuming two spaces is width of one digit).
// Finally, the zero value can be changed to a '-' upon request.
//
// Note:	In many places, the code is simplified by the assumption that
//			every digit has the same width, and that 2 spaces is the same.
//			This works for the selected font (MS Sans Serif, font size 8).
//			It allows us to present a nice columnar interface without having
//			to figure out each of the string position in pixels. There are
//			several more assumptions like this (e.g. "k   " is like "Mb").

	static CString CommaNum(DWORD num, int groups =0, BOOL zero =TRUE)
	{
		CString s;
		s.Format("%lu", num);
		
		int g = 0;
		int n = s.GetLength();
		while (n > 3)
		{
			n -= 3;
			s = s.Left(n) + sThousand + s.Mid(n);
			++g;
		}
		
		if (--groups >= 0)
		{
			int w = ((3 - n) % 3) * 2;
			if (g < groups)
				w += 7 * (groups - g);

			s = CString (' ', w) + s;
		}
		
		if (!zero && (s == "0" || s.Right(2) == " 0"))
			s = s.Left(s.GetLength() - 1) + " -";
			
		return s;
	}
	
/////////////////////////////////////////////////////////////////////////////
// Convert a DOS date and TIME words to short format strings.
// Lets be nice to a lot of people and adopt their local conventions.

	static CString ShortDate(WORD date)
	{
		int w = 0;
		
		char buf [10];
		char* q = buf;
		
			// decode the short date, deal with 1- and 2-digit fields
		const char* p = sShortDate;
		while (*p)
		{
			int i;
			
			switch (*p++)
			{            
				default:	*q++ = *(p-1);
							continue;
				
				case 'd':	i = date & 0x1F;
							break;
							
				case 'M':   i = (date >> 5) & 0x0F;
							break;
							
				case 'y':	i = ((date >> 9) + 80) % 100;
							break; // 4-digit years are treated as 2-digit
							
			}
			
			if (i < 10 && *p != *(p-1))
				++w;
            else
            	*q++ = (char) (i / 10 + '0');
            
           	*q++ = (char) (i % 10 + '0');

			while (*p == *(p-1))
				++p;
		}
		
			// centering is easy, since one digit is as wide as two spaces
		CString t (' ', w);	
        return t + CString (buf, q - buf) + t;
	}
	
	static CString ShortTime(WORD time)
	{
		int h = time >> 11;
		int m = (time >> 5) & 0x3F;
		
		if (!iTime)
			h = (h + 11) % 12 + 1; // dec, then inc, so 0 becomes 12
			
		CString s;
		s.Format("%02d%c%02d", h, sTime, m);
		
		if (!iTime)
			s += h < 12 ? 'a' : 'p';
		
		if (!iTLZero && s[0] == '0')
			s = "  " + s.Mid(1); // replace leading zero with two spaces
			
		return s;
	}
	
/////////////////////////////////////////////////////////////////////////////
// Make a string fit in the specified number of pixels on given device.
// Characters at the end are replaced by an ellipsis to make the string fit.
// There is some trickery in here to optimize this very common calculation.

	static BOOL FitString(CDC* dc, CString& text, int width)
	{
		CSize sz = dc->GetTextExtent(text, text.GetLength());
		if (sz.cx <= width)
			return TRUE;	// make the most common case fast
		
			// Assumption: "...xyz" is just as wide as "xyz..."	
		CString s = "..." + text;
		
		int n = s.GetLength();
		while (--n > 3)
		{            
			sz = dc->GetTextExtent(text, n);
			if (sz.cx <= width)
				break;
		}
			 
		text = text.Left(n - 3) + "...";
		return FALSE;
	}
	
/////////////////////////////////////////////////////////////////////////////
// Disables redraw and clears listbox, will reset normal state in destructor

	class ListBoxFreezer
	{
	public:
		ListBoxFreezer (CListBox& lb)
			: list (lb)
		{
			list.SetRedraw(FALSE);
			list.ResetContent();
		}
		
		~ListBoxFreezer ()
		{
			list.SetRedraw(TRUE);
			list.Invalidate();
		}
	
	private:
		CListBox& list;
	};
	
/////////////////////////////////////////////////////////////////////////////
// Return file date in display format, or an empty string if file not present

CString GetCatalogDate(CString& catName)
{
	CString s = catName;
	s += FILE_TYPE;
	
    _find_t fbuf;
	if (_dos_findfirst(s, _A_NORMAL, &fbuf) != 0)
		return "";
	
		// pick up the name as it is stored on disk (properly capitalized)
	s = fbuf.name;
	ASSERT(s.Right(4).CompareNoCase(FILE_TYPE) == 0);
	catName = s.Left(s.GetLength() - 4);
		
	return ShortDate((WORD) fbuf.wr_date) + "  "
			+ ShortTime((WORD) fbuf.wr_time);
}

/////////////////////////////////////////////////////////////////////////////
// The one and only application object

CTheApp::CTheApp ()
	: CWinApp ("CatFish")
{
}

BOOL CTheApp::InitInstance()
{
	SetDialogBkColor();
	SetInternationalSettings();

		// the following is required to let a dialog box have an icon	
	static WNDCLASS wndclass;
	if (!wndclass.lpfnWndProc)
	{
	    wndclass.lpfnWndProc    = DefDlgProc;
	    wndclass.cbWndExtra     = DLGWINDOWEXTRA ;
	    wndclass.hInstance      = m_hInstance;
	    wndclass.hIcon          = LoadIcon(AFX_IDI_STD_FRAME);
	    wndclass.lpszClassName  = "CATFISHCLASS";
		
	    RegisterClass(&wndclass);
	}
	
		// enter a modal loop right now	
	CMainDlgWindow mainDlg;
	m_pMainWnd = &mainDlg;
	mainDlg.DoModal();
	
		// and then return false to skip the main application run loop
	return FALSE;
}

/////////////////////////////////////////////////////////////////////////////

BEGIN_MESSAGE_MAP(CMainDlgWindow, CDialog)
	//{{AFX_MSG_MAP(CMainDlgWindow)
	ON_WM_CLOSE()
	ON_WM_DRAWITEM()
	ON_LBN_SELCHANGE(IDC_CAT_LIST, OnSelchangeCatList)
	ON_LBN_SELCHANGE(IDC_TREE_LIST, OnSelchangeTreeList)
	ON_LBN_DBLCLK(IDC_TREE_LIST, OnDblclkTreeList)
	ON_LBN_SELCHANGE(IDC_FILE_LIST, OnSelchangeFileList)
	ON_WM_SYSCOMMAND()
	ON_BN_CLICKED(IDC_FIND_BTN, OnFindBtn)
	ON_BN_CLICKED(IDC_SETUP_BTN, OnSetupBtn)
	ON_WM_CHAR()
	ON_LBN_DBLCLK(IDC_FILE_LIST, OnDblclkFileList)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_HELP, OnHelp)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////

CMainDlgWindow::CMainDlgWindow ()
	: CDialog (IDD_MAIN_DIALOG),
	  m_storage (0), m_fileDir (-1), m_treeDir (-1), m_dc (0)
{
	//{{AFX_DATA_INIT(CMainDlgWindow)
	//}}AFX_DATA_INIT
}

CMainDlgWindow::~CMainDlgWindow ()
{
	SetCatalog("");
	delete m_storage;
}

void CMainDlgWindow::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CMainDlgWindow)
	DDX_Control(pDX, IDC_PATH_FRAME, m_pathFrame);
	DDX_Control(pDX, IDC_TREE_LIST, m_treeList);
	DDX_Control(pDX, IDC_FILE_LIST, m_fileList);
	DDX_Control(pDX, IDC_CAT_LIST, m_catList);
	DDX_Control(pDX, IDC_MSG_TEXT, m_msgText);
	DDX_Control(pDX, IDC_INFO_TEXT, m_infoText);
	DDX_Control(pDX, IDC_TREE_PATH, m_treePath);
	//}}AFX_DATA_MAP
}

void CMainDlgWindow::OnCancel()
{
	::MessageBeep(-1);					// don't go away on ESC key
}

void CMainDlgWindow::OnClose()
{
	EndDialog(IDOK);
}

BOOL CMainDlgWindow::OnInitDialog()
{
	CDialog::OnInitDialog();
	
		// Add "About..." menu item to system menu.
    	// IDM_ABOUTBOX must be in the system command range.
	ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
	ASSERT(IDM_ABOUTBOX < 0xF000);

	CMenu* pSysMenu = GetSystemMenu(FALSE);
	pSysMenu->AppendMenu(MF_SEPARATOR);
	pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, "&About CatFish...");
        
        // create a small font for several of the dialog box items
	LOGFONT lf;
	memset(&lf, 0, sizeof(LOGFONT));
	lf.lfHeight = -8;
	strcpy(lf.lfFaceName, "MS Sans Serif");
	m_font.CreateFontIndirect(&lf);
	
    m_msgText.SetFont(&m_font, FALSE);
    m_infoText.SetFont(&m_font, FALSE);
    m_catList.SetFont(&m_font, FALSE);
    m_treeList.SetFont(&m_font, FALSE);
    m_fileList.SetFont(&m_font, FALSE);
    
		// determine the character height and set owner-draw lists accordingly
	{
		CClientDC dc (this);
		CFont* oldFont = dc.SelectObject(&m_font);
			
		TEXTMETRIC tm;
		VERIFY(dc.GetTextMetrics(&tm));
			   
		dc.SelectObject(oldFont);
			
		m_catList.SetItemHeight(0, tm.tmHeight);
		m_treeList.SetItemHeight(0, tm.tmHeight);
		m_fileList.SetItemHeight(0, tm.tmHeight);
    }
    
    	// fill the list of catalogs
    m_catList.Dir(0, "*" FILE_TYPE);
    
    	// show contents now, before potential slow catalog loading starts
    ShowWindow(ThisApp.m_nCmdShow);
	UpdateWindow(); 
    
    m_catList.SetCurSel(0);
    OnSelchangeCatList();

	m_infoText.SetWindowText("http://www.meta4.com/meta4/metakit.htm");
    
    if (m_catList.GetCount() == 0)
    {
    	CDialog dlg (IDD_WELCOME_DLG);
    	dlg.DoModal();
    }
    
	return TRUE;  	// return TRUE  unless you set the focus to a control
}

	// notification handler for owner-draw listboxes
void CMainDlgWindow::OnDrawItem(int nIDCtl, LPDRAWITEMSTRUCT lpDrawItemStruct)
{
    int n = (int) lpDrawItemStruct->itemID;
    if (n == -1)
       return;
    
    m_item = lpDrawItemStruct;
    m_dc = CDC::FromHandle(m_item->hDC);
    
    if (m_item->itemAction == ODA_FOCUS)
    {
        m_dc->DrawFocusRect(&m_item->rcItem);
        return;
    }
    
    if (m_item->itemState & ODS_SELECTED)
    {
        m_dc->SetBkColor(GetSysColor(COLOR_HIGHLIGHT));
        m_dc->SetTextColor(GetSysColor(COLOR_HIGHLIGHTTEXT));
    }
    else
    {
        m_dc->SetBkColor(GetSysColor(COLOR_WINDOW));
		m_dc->SetTextColor(GetSysColor(COLOR_WINDOWTEXT));
    }

	switch (nIDCtl)
	{
		case IDC_CAT_LIST:		OnDrawCatItem(n); break;
		case IDC_TREE_LIST:		OnDrawDirItem(n); break;
		case IDC_FILE_LIST:		OnDrawFileItem(n); break;
	}
	
    if ((m_item->itemState & ODS_FOCUS) && m_item->itemAction != ODA_SELECT)
        m_dc->DrawFocusRect(&m_item->rcItem);
}

	// common code to draw a text string in a listbox item
void CMainDlgWindow::DrawItemText(const CString& text, int off)
{
	RECT& rect = m_item->rcItem;
	
   	m_dc->ExtTextOut(rect.left + off + 2, rect.top,
   						off ? 0 : ETO_OPAQUE, &rect,
   						text, text.GetLength(), 0);
}

	// draw one item in the catalog listbox
void CMainDlgWindow::OnDrawCatItem(int n)
{
    CString s;
    m_catList.GetText(n, s);
    
    ASSERT(s.Right(4).CompareNoCase(FILE_TYPE) == 0);
	s = s.Left(s.GetLength() - 4);
    
    CString date = GetCatalogDate(s);

    FitString(m_dc, s, 72);
    DrawItemText(s);	
    
    DrawItemText(date, 72);
}

	// draw one item in the tree listbox
void CMainDlgWindow::OnDrawDirItem(int n)
{
    int dir = (int) m_treeList.GetItemData(n);
    BOOL up = n == 0 || pParent (m_currCat[dir]) != m_treeDir;
    
    if (up && !(m_item->itemState & ODS_SELECTED))
		m_dc->SetTextColor(GetSysColor(COLOR_GRAYTEXT));

    CString s = pName (m_currCat[dir]);
	if (dir == 0)
		s = "(root)";
	
	if (!up)
		s = (m_dirCounts[dir] ? "+ " : "   ") + s;

    FitString(m_dc, s, 78);
    DrawItemText(s);	
    
    DWORD t = m_kiloTotals[dir];
    if (t <= 999999)
    	s = CommaNum(t, 2) + " k   ";
    else
    	s = CommaNum((t + 999) / 1000, 2) + " Mb";
    	
   	s += CommaNum(m_dirCounts[dir], 2, FALSE).Mid(2) + "  "
       + CommaNum(m_fileCounts[dir], 2, FALSE) + "    "
       + ShortDate(m_lastDates[dir]);
       
    DrawItemText(s, 78);	
}

	// draw one item in the file listbox
void CMainDlgWindow::OnDrawFileItem(int n)
{
    c4_RowRef file = m_fileView[(int) m_fileList.GetItemData(n)];
    
    CString s = pName (file);
    FitString(m_dc, s, 85);
    DrawItemText(s);
    
    s = CommaNum(pSize (file), 3) + "    " + ShortDate((WORD) pDate (file));
    DrawItemText(s, 85);	
}

	// pressing F1 leads to an about box
void CMainDlgWindow::OnHelp()
{
	CDialog dlg (IDD_ABOUTBOX);
	dlg.DoModal();
}

	// find file entries
void CMainDlgWindow::OnFindBtn()
{
	// THE FIND BUTTON DOES NOT YET EXIST, TREAT IT AS AN ABOUT BUTTON FOR NOW!
	// Must swap & change button text when Find is eventually implemented...
	OnHelp();
}

	// special processing for the about box menu item
void CMainDlgWindow::OnSysCommand(UINT nID, LPARAM lParam)
{
	if ((nID & 0xFFF0) == IDM_ABOUTBOX)
		OnHelp();
	else
		CDialog::OnSysCommand(nID, lParam);
}

	// setup catalogs
void CMainDlgWindow::OnSetupBtn()
{
	CSetupDialog dlg;

	int n = m_catList.GetCurSel();
	if (n != LB_ERR)
	{                         
		CString s;
		m_catList.GetText(n, s);      
		
		ASSERT(s.Right(4).CompareNoCase(FILE_TYPE) == 0);
		dlg.m_name = s.Left(s.GetLength() - 4);
    }
		
	SetCatalog(""); // make sure no catalog is in use during setup
    
	dlg.DoModal();
    
    {
		ListBoxFreezer frozen (m_catList);
	
	    m_catList.Dir(0, "*" FILE_TYPE);
	
			// attempt to maintain the current selection
	    if (m_catList.SelectString(-1, dlg.m_name) == LB_ERR)
	    	m_catList.SetCurSel(0);
	}
	
    OnSelchangeCatList();
}

	// adjust the title to show which catalog is selected
void CMainDlgWindow::ConstructTitle(const char* catName)
{
    CString s = catName;
	ASSERT(s.Right(4).CompareNoCase(FILE_TYPE) == 0);
	s = s.Left(s.GetLength() - 4);
	
	GetCatalogDate(s); // for side-effect: proper file name capitalization
		
	CString root = pName (m_currCat[0]);
    if (!root.IsEmpty())
    	s += " - " + root;
    
    s = "CatFish - " + s;
    
	CString title;
	GetWindowText(title);

	if (title != s)
		SetWindowText(s);	
}

	// select a catalog and update the dialog contents
void CMainDlgWindow::SetCatalog(const char* catName)
{
	SetTreeDir(-1);

		// An important side effect is that m_fileView is cleared before the 
		// storage class is destroyed. Otherwise, the entire view would be
		// loaded into memory since the underlying file is about to go away.
	SetFileDir(-1);

	m_currCat = c4_View (); // see comment about m_fileView	
	delete m_storage;
    m_storage = 0;
    
    m_dirCounts.RemoveAll();
	m_fileCounts.RemoveAll();
	m_lastDates.RemoveAll();
	m_kiloTotals.RemoveAll();

	if (!*catName)
		return;                      
	
		// loading and calculations may take some time
	HCURSOR oldCursor = SetCursor(LoadCursor(0, IDC_WAIT));
	
	m_storage = DEBUG_NEW c4_Storage (catName);
	m_currCat = m_storage->Get("dirs");
	
    ConstructTitle(catName);	
    
    int n = m_currCat.GetSize();
    m_dirCounts.InsertAt(0, 0, n);
	m_fileCounts.InsertAt(0, 0, n);
	m_lastDates.InsertAt(0, 0, n);
	m_kiloTotals.InsertAt(0, 0, n);
    
    	// this loop calculates all cumulative totals and dates,
    	// mathematicians call this the "transitive closure" ...
	while (--n >= 0)
	{
		c4_RowRef dir = m_currCat[n];
		
		int date = 0;
		DWORD total = 0;

		c4_View files = pFiles (dir);
		
		for (int i = 0; i < files.GetSize(); ++i)
		{
			c4_RowRef file = files[i];
			
			total += pSize (file);
			if (date < pDate (file))
				date = (int) pDate (file);
		}
		
		ASSERT(i == files.GetSize());
		m_fileCounts[n] += (WORD) i;
		m_kiloTotals[n] += (total + 1023) / 1024;
		
		if (m_lastDates[n] < (WORD) date)
			m_lastDates[n] = (WORD) date;
		
		int parDir = pParent (dir);
		if (parDir != n)
		{
			m_dirCounts[parDir] += m_dirCounts[n] + 1;
			m_fileCounts[parDir] += m_fileCounts[n];
			m_kiloTotals[parDir] += m_kiloTotals[n];	
	
			if (m_lastDates[parDir] < m_lastDates[n])
				m_lastDates[parDir] = m_lastDates[n];
		}
	}
	
	SetCursor(oldCursor);
	
	if (m_currCat.GetSize() > 0)
		SetTreeDir(0);
}

	// select a directory in the tree and update the dialog contents
void CMainDlgWindow::SetTreeDir(int dirNum)
{
	if (dirNum != m_treeDir)
	{
		m_treeDir = dirNum;
	    
		ListBoxFreezer frozen (m_treeList);
	
		if (dirNum >= 0)
		{                  
				// select the appropriate subdirectories and sort them by name
		    c4_View selsort = m_currCat.Select(pParent [dirNum]).SortOn(pName);

		    for (int j = 0; j < selsort.GetSize(); ++j)
		    {
		    		// now map each entry back to the m_currCat view
		    	int ix = m_currCat.GetIndexOf(selsort[j]);
		    	ASSERT(ix >= 0);
		    	
		    		// don't add the root entry, it doesn't sort correctly
		    	if (ix > 0)
		    	{
		    		int k = m_treeList.AddString("");
		    		m_treeList.SetItemData(k, ix);
		    	}
	    	}
            
            	// now insert the parent directories in reverse order in front
		    for (;;)
		    {
		    	m_treeList.InsertString(0, "");
		    	m_treeList.SetItemData(0, dirNum);
		    	
				if (dirNum == m_treeDir)
					m_treeList.SetCurSel(0);
					
		    	if (dirNum <= 0)
		    		break;
		    		
		    	dirNum = (int) pParent (m_currCat[dirNum]);
		    }
		}
	}
	
	SetFileDir(m_treeDir);
}

	// select a list of files and update the dialog contents
void CMainDlgWindow::SetFileDir(int dirNum)
{
	if (dirNum != m_fileDir)
	{
		m_fileDir = dirNum;
	
		ListBoxFreezer frozen (m_fileList);
	
		if (dirNum >= 0)
		{				
			m_fileView = pFiles (m_currCat[dirNum]);
		
			CString root = fFullPath(m_currCat, 0);           
			CString path = fFullPath(m_currCat, dirNum);           
			
				// remove common root prefix
			path = path.Mid(root.GetLength());
			if (path.IsEmpty())
				path = "(root)";

			m_treePath.SetWindowText(path);
			
			for (int i = 0; i < m_fileView.GetSize(); ++i)
			{                  
				int k = m_fileList.AddString("");
				m_fileList.SetItemData(k, i);
			}
		}
		else
		{
			m_fileView = c4_View ();
			m_treePath.SetWindowText("");
		}
    }
    
    	// always reset the file list selection
    m_fileList.SetCurSel(-1);
    
	OnSelchangeFileList();
}

	// the catalog selection changed
void CMainDlgWindow::OnSelchangeCatList()
{
	CString s;

	int n = m_catList.GetCurSel();
	if (n != LB_ERR)
		m_catList.GetText(n, s);
		
	SetCatalog(s);
}

	// the directory selection changed
void CMainDlgWindow::OnSelchangeTreeList()
{
	int n = m_treeList.GetCurSel();
	if (n >= 0)
		n = (int) m_treeList.GetItemData(n);
	
	SetFileDir(n);
}

	// descend into an entry in the directory tree
void CMainDlgWindow::OnDblclkTreeList()
{
	int n = m_treeList.GetCurSel();
	if (n >= 0)
	{
		n = (int) m_treeList.GetItemData(n);
			
			// don't allow descending into a dir with no subdirs
		if (m_dirCounts[n] == 0)
		{
			MessageBeep(-1);
			return;
		}
	}
	
	SetTreeDir(n);
}

	// the file selection changed
void CMainDlgWindow::OnSelchangeFileList()
{
	CString s;
	
	int n = m_fileList.GetCurSel();
	if (n >= 0)
	{
	    c4_RowRef file = m_fileView[(int) m_fileList.GetItemData(n)];
		s = pName (file);
	}
	else if (m_fileDir >= 0)
		s.Format("%d files", m_fileView.GetSize());
	
	m_infoText.SetWindowText(s);
}

void CMainDlgWindow::OnDblclkFileList()
{
	int n = m_fileList.GetCurSel();
	if (n >= 0)
	{
	    c4_RowRef file = m_fileView[(int) m_fileList.GetItemData(n)];
		CString s = pName (file);
		
		CString path = fFullPath(m_currCat, m_fileDir); // also the working dir
		 
		if ((UINT) ShellExecute(m_hWnd, 0, path + s, 0, path, SW_SHOWNORMAL) >= 32)
			return; // selected file succesfully launched
	}

	MessageBeep(-1);
}
