/*----------------------------------------------------------------------*
 | Sample code for drag & drop within a single listbox.                 |
 |                                                                      |
 | Notes:                                                               |
 | -----                                                                |
 | 1. The code is not cluttered with FAR, NEAR or LPSTR casts.          |
 |    That's because I use LARGE model: highly recommended.             |
 |                                                                      |
 | 2. Variable names are deliberately lengthy for clarity.              |
 |                                                                      |
 | 3. The code is fully commented, but if you have questions/comments,  |
 |    feel free to e-mail me.                                           |
 |                                                                      |
 | 4. The code also illustrates subclassing                             |
 |                                                                      |
 | 5. Most of the code is standard stuff, except for the function which |
 |    handles the listbox messages (MyListboxWndProc).                  |
 |                                                                      |
 | 6. Notice that even though you're dragging the item, you can still   |
 |    force the listbox to scroll by moving above or below the listbox. |
 |    Don't you wish File Manager did this!                             |
 |                                                                      |
 | 7. I use my own NOTSELECTED if LB_ERR occurs, despite the fact that  |
 |    LB_ERR=-1.                                                        |
 |                                                                      |
 | 8. For the case of moving an item to the bottom of the list, all you |
 |    do is drop it in front of the last item and then move the bottom  |
 |    item in front of it.  This is a small kludge, but it is intuitive |
 |    and means I don't have to have a special case for appending.  For |
 |    example, I would have to allow the item to be dropped even when   |
 |    below the listbox, which is somewhat non-intuitive, since the     |
 |    'no-drop' icon usually appears when you drag outside an object.   |
 |                                                                      |
 | History & Need                                                       |
 | --------------                                                       |
 | I needed this for a list of files to be drawn as layers on a map.    |
 | This allows the user to change the order of the layers quite easily. |
 | This listbox is also maintained by 'add' & 'delete' buttons to select|
 | or remove files from the list.                                       |
 |                                                                      |
 | My previous method had an 'order' button which started up a second   |
 | dialog box and managed drag & drop between two listboxes ('current'  |
 | and 'desired' order).  I like this method better: it eliminates the  |
 | extra dialog box.                                                    |
 |                                                                      |
 | Although this example shows how to sort the listbox entries with	|
 | drag & drop, I certainly don't mean to imply that this is preferable	|
 | to using the LBS_SORT listbox style, if sorting is what you require.	|
 |									|
 | Release Notice                                                       |
 | --------------                                                       |
 | This code is released into the public domain: you can do with it as  |
 | you wish, provided you have fun!                                     |
 |                                                                      |
 |                                      John A. Grant                   |
 |                                      Geological Survey of Canada     |
 |                                      Mineral Resources Division      |
 |                                      Airborne Geophysics Section     |
 |                                      Ottawa, Ontario, K1A 0E8        |
 |                                      jagrant@gsc.emr.ca              |
 *----------------------------------------------------------------------*/
//standard function prototypes
	#define STRICT
	#include <windows.h>
	#include <stdio.h>      //sprintf
	#include <stdlib.h>     //malloc, free
	#include <string.h>     //memcmp

//my function prototypes
	LRESULT CALLBACK _export WndProc(HWND    hwnd,
					 UINT    wm,
					 WPARAM  wparam,
					 LPARAM  lparam);

	BOOL    CALLBACK _export DlgProc(HWND    hwnd,
					 UINT    wm,
					 WPARAM  wparam,
					 LPARAM  lparam);

	LRESULT CALLBACK _export MyListboxWndProc(
					 HWND    hwnd,
					 UINT    wm,
					 WPARAM  wparam,
					 LPARAM  lparam);
					  
	static void FillListbox(         HWND    hwnd_listbox);

	static void SubclassListbox(	void);

	static BOOL MoveItem(            HWND    hwnd_listbox,
					 int     from,
					 int     to);

	//win16/win32 message cracker for portability
	#ifdef WIN32
	  //GET_WM_COMMAND_ID are in Win32 <windowsx.h>
	#else
	  #define GET_WM_COMMAND_ID(wparam,lparam)   ((int)wparam)
	#endif

//constants
	#define CLASSNAME "listdrag"

	#define IDM_DIALOG      101

	#include "listdrag.h"

//globals
	static HINSTANCE appinst;


/*----------------------------------------------------------------------*
 | nothing new in WinMain - just the usual stuff                        |
 *----------------------------------------------------------------------*/
#pragma argsused
int PASCAL WinMain(HINSTANCE hinstance,HINSTANCE hprevinstance,
				LPSTR cmdline,int showtype)
{
HWND    hwnd;
MSG     msg;

	appinst=hinstance;

	if(!hprevinstance){
	  WNDCLASS wc;
	  wc.style=          CS_HREDRAW | CS_VREDRAW;
	  wc.lpfnWndProc=    WndProc;
	  wc.cbClsExtra=     0;
	  wc.cbWndExtra=     0;
	  wc.hInstance=      hinstance;
	  wc.hIcon=          LoadIcon(  NULL,IDI_APPLICATION);
	  wc.hCursor=        LoadCursor(NULL,IDC_ARROW);
	  wc.hbrBackground=  (HBRUSH)(COLOR_WINDOW+1);
	  wc.lpszMenuName=   NULL;
	  wc.lpszClassName=  CLASSNAME;

	  if(!RegisterClass(&wc)) return(FALSE);
	}


	hwnd=CreateWindow(      CLASSNAME,
				"Test: listbox drag & drop",
				WS_OVERLAPPED  | WS_CAPTION    |
				WS_SYSMENU     | WS_THICKFRAME |
				WS_MINIMIZEBOX | WS_MAXIMIZEBOX,
				CW_USEDEFAULT,0,
				CW_USEDEFAULT,0,
				NULL,
				NULL,
				hinstance,
				NULL);

	ShowWindow(hwnd,SW_SHOWNORMAL);
	UpdateWindow(hwnd);

	while(GetMessage(&msg,NULL,0,0)){
	  TranslateMessage(&msg);
	  DispatchMessage(&msg);
	}

	return(msg.wParam);
}



/*-----------------------------------------------------------------------*
 | nothing new here either: just start up the dialog box                 |
 *-----------------------------------------------------------------------*/
LRESULT CALLBACK _export WndProc(HWND hwnd,UINT wm,WPARAM wparam,LPARAM lparam)
{
BOOL handled;

	handled=TRUE;

	switch(wm){
	  //I'll just use the system menu for simplicity
	  case WM_CREATE:       AppendMenu(GetSystemMenu(hwnd,FALSE),MF_SEPARATOR,0,NULL);
				AppendMenu(GetSystemMenu(hwnd,FALSE),MF_STRING,IDM_DIALOG,"Test");
				SetWindowText(hwnd,"Listbox Drag & Drop: select 'Test' from system menu");
				break;

	  case WM_SYSCOMMAND:   if(wparam==IDM_DIALOG){
				  DLGPROC dlghandler;
				  dlghandler=(DLGPROC)MakeProcInstance((FARPROC)DlgProc,appinst);
				  DialogBox(appinst,MAKEINTRESOURCE(ZR_LISTDRAG),hwnd,dlghandler);
				  FreeProcInstance((FARPROC)dlghandler);
				  DestroyWindow(hwnd);
				}else{
				  handled=FALSE;
				}
				break;

	  case WM_DESTROY:      PostQuitMessage(0);
				break;

	  default:              handled=FALSE;

	}
	return (handled ? 0 : DefWindowProc(hwnd,wm,wparam,lparam));
}



/*-----------------------------------------------------------------------*
 | this is where it gets interesting: the listbox is subclassed here     |
 *-----------------------------------------------------------------------*/
static HCURSOR hcursor_drag,hcursor_outside;
static HWND    hwnd_listbox;

#pragma argsused (lparam)
BOOL CALLBACK _export DlgProc(HWND hdlg,UINT wm,WPARAM wparam,LPARAM lparam)
{
BOOL handled;
int id;

	handled=TRUE;

	switch(wm){

	  case WM_INITDIALOG:   hwnd_listbox=GetDlgItem(hdlg,ID_LISTBOX);
				FillListbox(hwnd_listbox);
				SubclassListbox();
				hcursor_drag=   LoadCursor(appinst,MAKEINTRESOURCE(ZR_DRAGCURSOR));
				hcursor_outside=LoadCursor(appinst,MAKEINTRESOURCE(ZR_OUTCURSOR));
				break;


	  case WM_COMMAND:      id=GET_WM_COMMAND_ID(wparam,lparam);
				switch(id){
				  case IDOK:
				  case IDCANCEL: DestroyCursor(hcursor_drag);
						 DestroyCursor(hcursor_outside);
						 EndDialog(hdlg,0);
						 break;
				  default:       handled=FALSE;
				}
				break;

	  default:              handled=FALSE;

	}
	return (handled);
}


/*-----------------------------------------------------------------------*
 | nothing fancy here - just fill the listbox with some unsorted values  |
 *-----------------------------------------------------------------------*/
void FillListbox(HWND hwnd_listbox)
{
int i;
char list[]="DBFGAJKMELICH";
char buffer[10];
	SendMessage(hwnd_listbox,LB_RESETCONTENT,0,0);
	for(i=0;i<sizeof(list)-1;i++){
	  sprintf(buffer,"item %c",list[i]);
	  SendMessage(hwnd_listbox,LB_ADDSTRING,0,(LPARAM)buffer);
	}
	return;
}


/*-----------------------------------------------------------------------*
 | the usual subclass...                                                 |
 *-----------------------------------------------------------------------*/
static WNDPROC lpdefault_handler;
void SubclassListbox(void)
{

WNDPROC lpmy_handler;
	lpmy_handler=(WNDPROC)MakeProcInstance((FARPROC)MyListboxWndProc,appinst);
	lpdefault_handler=(WNDPROC)SetWindowLong(
		hwnd_listbox,GWL_WNDPROC,(LONG)lpmy_handler);
	return;
}




/*----------------------------------------------------------------------*
 | This function receives *all* of the listbox messages, but only cares |
 | about mouse messages - the rest are ignored and are handled by the   |
 | default listbox handler.  In fact, even the 3 mouse messages are     |
 | passed on to the default handler.  If you don't do that, then the    |
 | listbox won't scroll when you move the dragged item above or below   |
 | the listbox and the destination item won't be highlighted.           |
 *----------------------------------------------------------------------*/
typedef enum {dmYES,dmNO,dmMAYBE} DRAGMODE_T;

#define NOTSELECTED (-1)
static DRAGMODE_T dragmode=dmNO;
static POINT      startpt;
static HCURSOR    saved_hcursor;

LRESULT CALLBACK _export MyListboxWndProc(HWND hwnd,UINT wm,WPARAM wparam,LPARAM lparam)
{
POINT pt;
static int from=NOTSELECTED;
static int to=NOTSELECTED;

	pt.x=LOWORD(lparam);
	pt.y=HIWORD(lparam);

	switch(wm){

	  //start dragging or perhaps just a simple click
	  case WM_LBUTTONDOWN:

		//select the item by 'clicking' it (so I can use LB_GETCURSEL)
		CallWindowProc(lpdefault_handler,hwnd,WM_LBUTTONDOWN,wparam,lparam);
		CallWindowProc(lpdefault_handler,hwnd,WM_LBUTTONUP,  wparam,lparam);

		//now get subscript of item to be dragged (current selection if any)
		from=(int)SendMessage(hwnd_listbox,LB_GETCURSEL,0,0);
		if(from==LB_ERR){
		  from=NOTSELECTED;             //nothing was selected?
		}else{
		  startpt=pt;                   //remember where click occurred
		  SetCapture(hwnd_listbox);     //start dragging...
		  dragmode=dmMAYBE;             //...or maybe it was just a click
		}
		to=NOTSELECTED;
		break;

	  //finish dragging (or perhaps just the end of the click)
	  case WM_LBUTTONUP:
		to=NOTSELECTED;
		if(dragmode!=dmNO){
		  ReleaseCapture();
		  SetCursor(saved_hcursor);

		  if(dragmode==dmYES){
		    ClientToScreen(hwnd_listbox,&pt);
		    if(WindowFromPoint(pt)==hwnd_listbox){
		      to=(int)SendMessage(hwnd_listbox,LB_GETCURSEL,0,0);
		      if(to==LB_ERR) to=NOTSELECTED;
		    }
		  }
		  dragmode=dmNO;

		  //move the item in the listbox
		  if(from!=NOTSELECTED && to!=NOTSELECTED && from!=to && from!=(to-1)){
		    MoveItem(hwnd_listbox,from,to);
		  }
		}
		break;

	  //drag the item
	  case WM_MOUSEMOVE:
		//start dragging?
		if(dragmode==dmMAYBE){
		  if(memcmp(&startpt,&pt,sizeof(POINT))!=0){
		     dragmode=dmYES;            //here we go!
		     saved_hcursor=GetCursor(); //remember cursor
		  }

		//continue dragging?
		}else if(dragmode==dmYES){
		  ClientToScreen(hwnd_listbox,&pt);
		  if(WindowFromPoint(pt)==hwnd_listbox){
		    SetCursor(hcursor_drag);
		  }else{
		    SetCursor(hcursor_outside);
		  }
		}

		break;

	}

	//pass *all* messages on to the default handler, even if handled here
	return(CallWindowProc(lpdefault_handler,hwnd,wm,wparam,lparam));
}


/*----------------------------------------------------------------------*
 | standard stuff here: retrieve the 'from' item from the listbox,      |
 | delete it and then insert it at the 'to' location                    |
 *----------------------------------------------------------------------*/
BOOL MoveItem(HWND hwnd_listbox,int from,int to)
{
BOOL moved;
char *string;
int lstring;

	moved=FALSE;
	string=NULL;

	//find out how long it is, allocate memory & get 'from' string
	if((lstring=(int)SendMessage(hwnd_listbox,LB_GETTEXTLEN,from,0))==LB_ERR) goto done;

	if((string=(char *)malloc(lstring+1))==NULL) goto done;
	if(SendMessage(hwnd_listbox,LB_GETTEXT,from,(LPARAM)string)==LB_ERR) goto done;

	//delete 'from' string
	if(SendMessage(hwnd_listbox,LB_DELETESTRING,from,0)==LB_ERR) goto done;

	//insert 'from' string in front of 'to'
	if(from<to) to--;       //everything moved when the string was deleted
	if(SendMessage(hwnd_listbox,LB_INSERTSTRING,to,(LPARAM)string)==LB_ERR) goto done;

	moved=TRUE;

done:   if(string) free(string);
	return(moved);
}
