#include "copyleft.h"

/*
    GEPASI - a simulator of metabolic pathways and other dynamical systems
    Copyright (C) 1989, 1992, 1993  Pedro Mendes
*/

/*************************************/
/*                                   */
/*         GWSIM - Simulation        */
/*        MS-WINDOWS front end       */
/*                                   */
/*       Initialization and some     */
/*           data structures         */
/*                                   */
/*           QuickC/WIN 1.0          */
/*                                   */
/*   (include here compilers that    */
/*   compiled GWSIM successfully)    */
/*                                   */
/*************************************/


#include <windows.h>
#include <string.h>
#include <float.h>
#include "globals.h"
#include "gaussw.h"
#include "strtbl.h"
#include "simgvar.h"

void get_dfts( void );

struct kint {
		  		unsigned char nconst;
		  		unsigned char nmodf;
		  		LPSTR descr;
				LPSTR constnam;
	   	    };

struct nodet{
             char item;
             unsigned char val;
             unsigned char left;
             unsigned char right;
            };

struct treet{
             struct nodet node[256];
             char id[64][10];
             float constant[32];
             int nnode,
                 nnum,
                 nid,
                 nsub,
                 npro,
                 nmodf,
                 nconst,
                 revers;
             char descr[64];
            };

struct opt {
            int dyn;
            long pfo;
	        double	endtime;
            double	reltol;
            double	abstol;
            double	hrcz;
            int adams;
            int bdf;
            int ss;
            int debug;
            int txt;
            int structan;
            int staban;
            int stdela;
            int nonela;
            int stdcc;
            int noncc;
            int dat;
            int datsep;
            int datwidth;
            int dattit;
            int datmca;
            int datss;
            int quotes;
            int append;
			char timeu[32];
			char concu[32];
            int scan;
            int scanlog;
            unsigned long scandens;
           };

struct plt{
           int type;		/* 0 for 2d, 1 for 3d */
           int file;		/* 0 for dynamics, 1 for steady state */
           int x;			/* column for x */
           int y[10];		/* columns for y */
           int z;			/* column for z */
           int ny;			/* number of y variables */
           int logx;		/* 1 if log x axis */
           int logy;		/* 1 if log y axis */
           int logz;		/* 1 if log z axis */
           int lines;		/* 1 if points connected by lines */
           int colour;		/* 1 if colour to be used */
           int hidden;		/* 1 if hidden line removal */
           int contour;		/* 1 if contour plots on base */
          };

struct ou {
           LPSTR title;
           unsigned int idx;
          };

struct sp{
          int idx;
          LPSTR title;
	      double low;
	      double high;
	      unsigned long dens;
	      int log;
	      int lidx;
	      int linkedto;
	      double factor;
	      int operation;
	     };

GLOBALHANDLE hMetname;								/* handle to memory block w/ metname	*/
GLOBALHANDLE hStepname;								/* handle to memory block w/ stepname	*/
GLOBALHANDLE hStoiu;								/* handle to memory block w/ stoiu		*/
GLOBALHANDLE hSto;									/* handle to memory block w/ stoi		*/
GLOBALHANDLE hLoop;									/* handle to memory block w/ loop		*/
GLOBALHANDLE hParams;								/* handle to memory block w/ params		*/
GLOBALHANDLE hKtype;								/* handle to memory block w/ ktype		*/
GLOBALHANDLE hRstr;									/* handle to memory block w/ rstr		*/
GLOBALHANDLE hOutpEl;								/* handle to memory block w/ outpel		*/
GLOBALHANDLE hStrPool;								/* handle to memory block w/ strings	*/
GLOBALHANDLE hScanPar;								/* handle to memory block w/ strings	*/
GLOBALHANDLE hTree;									/* handle to memory block w/ trees		*/
GLOBALHANDLE hTreeStr;								/* handle to memory block w/ tree str	*/
GLOBALHANDLE hTmpF;									/* handle to memory block w/ temp files	*/

char	(huge *metname)[NAME_L];					/* pointer to work with metname array	*/
char	(huge *stepname)[NAME_L];					/* metabolite names						*/
int		huge *stoiu;								/* pointer to work with stoiu array		*/
int		huge *stoi;									/* pointer to work with stoi array		*/
unsigned char (huge *loop)[MAX_STEP][MAX_MET];		/* def. of modification loops			*/
int		(huge *rstr)[MAX_STEP][MAX_MOL];			/* reaction structure					*/
char	topname[256];								/* title for the topology				*/
char	stepstr[MAX_STEP][256];						/* array to hold description of steps	*/
double 	huge *params[MAX_STEP];						/* ptr to parameters for each rate eq.	*/
DWORD	sizeparam;									/* size of parameters memory block		*/
DWORD	sizeoutp;									/* size of outpel memory block			*/
DWORD	sizestrp;									/* size of StrPool memory block			*/
DWORD	sizestra;									/* size of allocated strings in StrPool	*/
DWORD	sizespar;
DWORD	sizetr;										/* size of allocated strings in TreeStr	*/
double	xu[MAX_MET];								/* metabolite concentrations			*/
double	moiety[MAX_MET];							/* concentration of conserved moieties	*/
int		intmet[MAX_MET];							/* 1 if internal metabolite				*/
unsigned char	revers[MAX_STEP];					/* 1 if reaction is reversible			*/
int		kinetu[MAX_STEP];							/* type of kinetics (user numb.)		*/
int		kfl[MAX_STEP];								/* flags for input of user-def.kinetics */
LPSTR	strpool;									/* points to the string pool mem block	*/
LPSTR	treestr;									/* points to the tree string pool		*/
LPSTR	treeptr;									/* points to the tree string pool		*/
struct	opt options;								/* structure with simulation options	*/
struct	plt plot;									/* structure with plot options			*/
struct	kint huge *ktype;							/* ptr array of kinetic types & proprt	*/
struct	ou huge *outpel;							/* ptr array of output elements			*/
struct	sp huge *spar;								/* ptr array of scanning elements		*/
float	ver_no;										/* .top and .sim version number			*/
double	dft_endtime;								/* time value for last iteration		*/
double	dft_hrcz;									/* highest rate considered zero			*/
int		totmet;										/* number of total metabolites			*/
int		nmetab;										/* number of internal metabolites		*/
int		indmet;										/* number of independent metabolites	*/
int		nextmet;									/* number of external metabolites		*/
int		nsteps;										/* number of steps                		*/
int		nloops;										/* number of modifier loops             */
int		noutpel;                                    /* number of output elements			*/
int		nudf;										/* number of user-defined functions		*/
int		nrateq;										/* number of user-defined functions		*/
int		totsel;                                     /* number of selected output elements	*/
int		totscan;									/* number of dimensions to scan			*/
int		nscanpar;                                   /* number of scanning elements			*/
int		nlinks;										/* number of links between parameters	*/
unsigned char dft_debugval;							/* debug mode							*/
double	dft_conc;									/* default metabolite concentration		*/
double	dft_const;									/* default value for kinetic constants	*/
char	*dft_timeu;									/* default time units					*/
char	*dft_concu;									/* default conctration units			*/
unsigned long dft_pfo;								/* number of points for output			*/
struct	treet huge *tree;							/* function tree for rate equations		*/
struct	treet tr;									/* tree for the input					*/
LPSTR	TmpFiles;									/* string holding names of temp files	*/

int		InitGepasiVar( void );
void	TidyGepasiVar( void );
int 	SetParams( void );
int 	SetOutpEl( void );
void 	step_string( void );
void	conc_dft( void );
void 	numer_dft( void );
void 	def_def( void );
void 	set_dfts( void );
void 	get_dfts( void );

#pragma alloc_text( CODE1, InitGepasiVar, TidyGepasiVar, SetParams, SetOutpEl, step_string, conc_dft, numer_dft, def_def, set_dfts, get_dfts )

/* Initialization of GEPASI's variables */

int InitGepasiVar( void )
{
 int i, j;

 hMetname = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_MET * NAME_L * sizeof( char ) );
 if( hMetname == NULL ) return -1;

 hStepname = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_STEP * NAME_L * sizeof( char ) );
 if( hStepname == NULL )
 {
  GlobalFree( hMetname );
  return -1;
 }

 hStoiu = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_MET * MAX_STEP * sizeof( int ) );
 if( hStoiu == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  return -1;
 }

 hLoop = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_STEP * MAX_MET * sizeof( unsigned char ) );
 if( hLoop == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  return -1;
 }

 hKtype = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_TYP * sizeof( struct kint ) );
 if( hKtype == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  return -1;
 }

 hRstr = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_STEP * MAX_MOL * sizeof( int ) );
 if( hRstr == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  return -1;
 }

 sizeparam = (DWORD) sizeof( double );
 hParams = GlobalAlloc( GMEM_ZEROINIT, sizeparam );
 if( hParams == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  return -1;
 }

 sizeoutp = (DWORD) 4 * sizeof( struct ou );
 hOutpEl = GlobalAlloc( GMEM_ZEROINIT, sizeoutp );
 if( hOutpEl == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  return -1;
 }

 sizestrp = (DWORD) 4 * 17; /* strings will be limited to 17, but this can change */
 hStrPool = GlobalAlloc( GMEM_ZEROINIT, sizestrp );
 if( hStrPool == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  GlobalFree( hOutpEl );
  return -1;
 }

 sizespar = (DWORD) 4 * sizeof( struct sp );
 hScanPar = GlobalAlloc( GMEM_ZEROINIT, sizespar );
 if( hScanPar == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  GlobalFree( hOutpEl );
  GlobalFree( hStrPool );
  return -1;
 }

 hSto = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) MAX_MET * MAX_STEP * sizeof( int ) );
 if( hSto == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  GlobalFree( hOutpEl );
  GlobalFree( hStrPool );
  GlobalFree( hScanPar );
  return -1;
 }

 sizetr = (DWORD) 4 * sizeof( char );
 hTreeStr = GlobalAlloc( GMEM_ZEROINIT, sizetr );
 if( hTreeStr == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  GlobalFree( hOutpEl );
  GlobalFree( hStrPool );
  GlobalFree( hScanPar );
  GlobalFree( hSto );
  return -1;
 }

 nudf = 0;
 hTree = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) sizeof( struct treet ) );
 if( hTree == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  GlobalFree( hOutpEl );
  GlobalFree( hStrPool );
  GlobalFree( hScanPar );
  GlobalFree( hSto );
  GlobalFree( hTreeStr );
  return -1;
 }

 /* allocate memory to hold the names of the temp files	*/
 hTmpF = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (DWORD) 2 );
 if( hTmpF == NULL )
 {
  GlobalFree( hMetname );
  GlobalFree( hStepname );
  GlobalFree( hStoiu );
  GlobalFree( hLoop );
  GlobalFree( hKtype );
  GlobalFree( hRstr );
  GlobalFree( hParams );
  GlobalFree( hOutpEl );
  GlobalFree( hStrPool );
  GlobalFree( hScanPar );
  GlobalFree( hSto );
  GlobalFree( hTreeStr );
  GlobalFree( hTree );
  return -1;
 }

 metname	= ( char (huge *)[NAME_L] ) GlobalLock( hMetname );
 stepname	= ( char (huge *)[NAME_L] ) GlobalLock( hStepname );
 stoiu		= ( int huge * ) GlobalLock( hStoiu );
 loop		= ( unsigned char (huge *)[MAX_STEP][MAX_MET] ) GlobalLock( hLoop );
 ktype		= ( struct kint huge * ) GlobalLock( hKtype );
 rstr		= ( int (huge *)[MAX_STEP][MAX_MOL] ) GlobalLock( hRstr );
 params[0]	= (double huge *) GlobalLock( hParams );
 outpel		= ( struct ou huge * ) GlobalLock( hOutpEl );
 spar		= ( struct sp huge * ) GlobalLock( hScanPar );
 strpool	= GlobalLock( hStrPool );
 stoi		= (int huge *) GlobalLock( hSto );
 treestr	= GlobalLock( hTreeStr );
 tree		= ( struct treet huge * ) GlobalLock( hTree );
 TmpFiles	= (LPSTR) GlobalLock( hTmpF );

 *TmpFiles = '\0';

 /* make the other params[i] point to NULL											*/
 for( i=1; i<MAX_STEP; i++ ) params[i] = NULL;

 get_dfts();										/* read GEPASI default values	*/
 totmet = nmetab = nsteps = 0;						/* clear no. of metb. and steps	*/

 nrateq = MAX_TYP;

 ktype[NOT].nconst = 0;
 ktype[NOT].nmodf = 0;
 ktype[NOT].constnam = (LPSTR) "";
 ktype[NOT].descr = (LPSTR) "<not defined>";
 ktype[I01].nconst = 1;
 ktype[I01].nmodf = 0;
 ktype[I01].constnam = (LPSTR) "k";
 ktype[I01].descr = (LPSTR) "constant rate";
 ktype[I10].nconst = 1;
 ktype[I10].nmodf = 0;
 ktype[I10].constnam = (LPSTR) "k";
 ktype[I10].descr = (LPSTR) ktype[I01].descr;
 ktype[I11].nconst = 1;
 ktype[I11].nmodf = 0;
 ktype[I11].constnam = (LPSTR) "k";
 ktype[I11].descr = (LPSTR) "mass action";
 ktype[R11].nconst = 2;
 ktype[R11].nmodf = 0;
 ktype[R11].constnam = (LPSTR) "k1\0k2";
 ktype[R11].descr = (LPSTR) ktype[I11].descr;
 ktype[I21].nconst = 1;
 ktype[I21].nmodf = 0;
 ktype[I21].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I21].descr = (LPSTR) ktype[I11].descr;
 ktype[R21].nconst = 2;
 ktype[R21].nmodf = 0;
 ktype[R21].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R21].descr = (LPSTR) ktype[I11].descr;
 ktype[I12].nconst = 1;
 ktype[I12].nmodf = 0;
 ktype[I12].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I12].descr = (LPSTR) ktype[I11].descr;
 ktype[R12].nconst = 2;
 ktype[R12].nmodf = 0;
 ktype[R12].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R12].descr = (LPSTR) ktype[I11].descr;
 ktype[I31].nconst = 1;
 ktype[I31].nmodf = 0;
 ktype[I31].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I31].descr = (LPSTR) ktype[I11].descr;
 ktype[R31].nconst = 2;
 ktype[R31].nmodf = 0;
 ktype[R31].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R31].descr = (LPSTR) ktype[I11].descr;
 ktype[I13].nconst = 1;
 ktype[I13].nmodf = 0;
 ktype[I13].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I13].descr = (LPSTR) ktype[I11].descr;
 ktype[R13].nconst = 2;
 ktype[R13].nmodf = 0;
 ktype[R13].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R13].descr = (LPSTR) ktype[I11].descr;
 ktype[I22].nconst = 1;
 ktype[I22].nmodf = 0;
 ktype[I22].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I22].descr = (LPSTR) ktype[I11].descr;
 ktype[R22].nconst = 2;
 ktype[R22].nmodf = 0;
 ktype[R22].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R22].descr = (LPSTR) ktype[I11].descr;
 ktype[I32].nconst = 1;
 ktype[I32].nmodf = 0;
 ktype[I32].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I32].descr = (LPSTR) ktype[I11].descr;
 ktype[R32].nconst = 2;
 ktype[R32].nmodf = 0;
 ktype[R32].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R32].descr = (LPSTR) ktype[I11].descr;
 ktype[I23].nconst = 1;
 ktype[I23].nmodf = 0;
 ktype[I23].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I23].descr = (LPSTR) ktype[I11].descr;
 ktype[R23].nconst = 2;
 ktype[R23].nmodf = 0;
 ktype[R23].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R23].descr = (LPSTR) ktype[I11].descr;
 ktype[I33].nconst = 1;
 ktype[I33].nmodf = 0;
 ktype[I33].constnam = (LPSTR) ktype[I11].constnam;
 ktype[I33].descr = (LPSTR) ktype[I11].descr;
 ktype[R33].nconst = 2;
 ktype[R33].nmodf = 0;
 ktype[R33].constnam = (LPSTR) ktype[R11].constnam;
 ktype[R33].descr = (LPSTR) ktype[I11].descr;
 ktype[IMM].nconst = 2;
 ktype[IMM].nmodf = 0;
 ktype[IMM].constnam = (LPSTR) "Km\0V";
 ktype[IMM].descr = (LPSTR) "Michaelis-Menten";
 ktype[RMM].nconst = 4;
 ktype[RMM].nmodf = 0;
 ktype[RMM].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr";
 ktype[RMM].descr = (LPSTR) "Reversible Michaelis-Menten";
 ktype[PSI].nconst = 5;
 ktype[PSI].nmodf = 1;
 ktype[PSI].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0Ki";
 ktype[PSI].descr = (LPSTR) "specific inhibition";
 ktype[PCI].nconst = 5;
 ktype[PCI].nmodf = 1;
 ktype[PCI].constnam = ktype[PSI].constnam;
 ktype[PCI].descr = (LPSTR) "catalytic inhibition";
 ktype[MXI].nconst = 6;
 ktype[MXI].nmodf = 1;
 ktype[MXI].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0Ki\0Ki'";
 ktype[MXI].descr = (LPSTR) "mixed inhibition";
 ktype[PSA].nconst = 5;
 ktype[PSA].nmodf = 1;
 ktype[PSA].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0Ka";
 ktype[PSA].descr = (LPSTR) "specific activation";
 ktype[PCA].nconst = 5;
 ktype[PCA].nmodf = 1;
 ktype[PCA].constnam = ktype[PSA].constnam;
 ktype[PCA].descr = (LPSTR) "catalytic activation";
 ktype[MXA].nconst = 6;
 ktype[MXA].nmodf = 1;
 ktype[MXA].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0Ka\0Ka'";
 ktype[MXA].descr = (LPSTR) "mixed activation";
 ktype[GOM].nconst = 6;
 ktype[GOM].nmodf = 1;
 ktype[GOM].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0Ki\0Ki'";
 ktype[GOM].descr = (LPSTR) "Botts-Morales";
 ktype[HIL].nconst = 3;
 ktype[HIL].nmodf = 0;
 ktype[HIL].constnam = (LPSTR) "K\0V\0h";
 ktype[HIL].descr = (LPSTR) "Hill";
 ktype[UBS].nconst = 6;
 ktype[UBS].nmodf = 0;
 ktype[UBS].constnam = (LPSTR) "Ka\0Kp\0Kq\0Kpq\0Vf\0Vr";
 ktype[UBS].descr = (LPSTR) "Ordered Uni Bi (A=P+Q)";
 ktype[UBM].nconst = 4;
 ktype[UBM].nmodf = 0;
 ktype[UBM].constnam = (LPSTR) "Ka\0Kp\0Vf\0Vr";
 ktype[UBM].descr = (LPSTR) "Ordered Uni Bi (A=2*P)";
 ktype[RHL].nconst = 6;
 ktype[RHL].nmodf = 0;
 ktype[RHL].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0hs\0hp";
 ktype[RHL].descr = (LPSTR) "Reversible Hill";
 ktype[ALI].nconst = 6;
 ktype[ALI].nmodf = 1;
 ktype[ALI].constnam = (LPSTR) "Kms\0Kmp\0Vf\0Vr\0Ki\0h";
 ktype[ALI].descr = (LPSTR) "Allosteric inhibition";

 nudf = 0;
 return 0;
}

/*
	set up the pointers to params
*/

int SetParams( void )
{
 int i, j;

 /* calculate the size of parameters array				*/
 for( i=0, j=0; i<nsteps; i++ )
  j += ktype[kinetu[i]].nconst;

 /* unlock hParams to be able to ReAlloc it				*/
 GlobalUnlock( hParams );

 sizeparam = (DWORD) j * sizeof( double );
 /* reallocate memory for parameters from global heap	*/
 hParams = GlobalReAlloc( hParams, sizeparam, GMEM_ZEROINIT | GMEM_MOVEABLE );
 if( hParams == NULL ) return -1;

 /* point to the parameters of the first step 			*/
 params[0] = (double huge *) GlobalLock( hParams );
 /* set all constants of step 0 to the default value	*/
 for( j=0; j<(int)ktype[kinetu[0]].nconst; j++ )
  *(params[0]+j) = dft_const;

 /* set the other pointers								*/
 for( i=1; i<nsteps; i++ )
 {
  params[i] = params[i-1] + ktype[kinetu[i-1]].nconst;
  /* set all constants of step j to the default value	*/
  for( j=0; j<(int)ktype[kinetu[i]].nconst; j++ )
   *(params[i]+j) = dft_const;
 }

 return 0;
}


/*
  setup the output elements
*/

int SetOutpEl( void )
{
 int i, j, k, l;
 LPSTR ptr, poolptr;
 char buff[128], c;

 /* calculate the number of output elements				*/
 /* all metabolite concentrations and step fluxes		*/
 noutpel = 2*totmet + nsteps;
 nscanpar = totmet;
 /* all kinetic constants...							*/
 for( i=0; i<nsteps; i++ )
 {
  noutpel += ktype[kinetu[i]].nconst;
  nscanpar += ktype[kinetu[i]].nconst;
 }
 /* all elasticies										*/
 noutpel += nsteps * totmet;
 /* all concentration control coefficients				*/
 noutpel += totmet * nsteps;
 /* all flux control coefficients						*/
 noutpel += nsteps * nsteps;
 /* endtime, rel and abs tol, nfunc, njacob, nistep,
    smstep, flux resolution								*/
 noutpel += 8;
 nscanpar += 4;

 /* unlock memory blocks to be able to ReAlloc them		*/
 GlobalUnlock( hOutpEl );
 GlobalUnlock( hStrPool );
 GlobalUnlock( hScanPar );

 sizeoutp = (DWORD) noutpel * sizeof( struct ou );
 sizespar = (DWORD) nscanpar * sizeof( struct sp );
 sizestrp = (DWORD) noutpel * 44;

 /* reallocate memory for parameters from global heap	*/
 hOutpEl = GlobalReAlloc( hOutpEl, sizeoutp, GMEM_ZEROINIT | GMEM_MOVEABLE );
 if( hOutpEl == NULL )
 {
  return -1;
 }
 hScanPar = GlobalReAlloc( hScanPar, sizespar, GMEM_ZEROINIT | GMEM_MOVEABLE );
 if( hScanPar == NULL )
 {
  return -1;
 }
 hStrPool = GlobalReAlloc( hStrPool, sizestrp, GMEM_ZEROINIT | GMEM_MOVEABLE );
 if( hStrPool == NULL )
 {
  return -1;
 }

 /* lock the memory block and have outpel[0] point to it*/
 outpel  = (struct ou huge *) GlobalLock( hOutpEl );
 spar    = (struct sp huge *) GlobalLock( hScanPar );
 strpool = GlobalLock( hStrPool );

 /* reset the auxiliary pool pointer					*/
 poolptr = strpool;

 /* set all idx to 0 (to signal not selected)			*/
 for( i=0; i<noutpel; i++ )
  outpel[i].idx = 0;
 for( i=0; i<nscanpar; i++ )
 {
  spar[i].idx = 0;
  spar[i].low = 0;
  spar[i].high= 0;
  spar[i].dens = (unsigned long) 1;
  spar[i].log = 0;
  spar[i].factor = 1;
  spar[i].operation = 1;
  spar[i].lidx = -1;
  spar[i].linkedto = -1;
 }

 /* initialize the titles								*/

 /* initial concentrations 								*/
 for( i=0; i<totmet; i++ )
 {
  wsprintf( (LPSTR) buff, "[%s]i", (LPSTR) metname[i] );
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );

  /* store the pointer in outpel[i] and spar[i]			*/
  outpel[i].title = spar[i].title = poolptr;

  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
 }
 l = i;

 /* concentrations at time t							*/
 for( j=0; j<totmet; j++, i++ )
 {
  wsprintf( (LPSTR) buff, "[%s]f", (LPSTR) metname[j] );
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );

  /* store the pointer in outpel[i] and spar[i]			*/
  outpel[i].title = poolptr;

  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
 }

 for( j=0; j<nsteps; j++, i++ )
 {
  wsprintf( (LPSTR) buff, "J(%s)", (LPSTR) stepname[j] );
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );

  /* store the pointer in outpel[i]						*/
  outpel[i].title = poolptr;

  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
 }

 for( j=0; j<nsteps; j++ )
  for( k=0, ptr = ktype[kinetu[j]].constnam;
       k < (int) ktype[kinetu[j]].nconst;
       k++, i++, l++
     )
  {
   wsprintf( (LPSTR) buff, "%s,%s", (LPSTR) ptr, (LPSTR) stepname[j] );
   /* copy buff to the string pool						*/
   _fstrcpy( (char __far *) poolptr,
             (char __far *) buff );

   /* store the pointer in outpel[i]					*/
   outpel[i].title = spar[l].title = poolptr;

   /* ptr points to the following constant name			*/
   do{} while( *(ptr++) );

   /* increase poolptr to point to the first free byte	*/
   poolptr += _fstrlen( buff ) + 1;
  }

  for( j=0; j<nsteps; j++ )
   for(k=0;k<totmet;k++,i++)
   {
    wsprintf( (LPSTR) buff,"e(%s,[%s])", (LPSTR) stepname[j], (LPSTR) metname[k]);
    /* copy buff to the string pool						*/
    _fstrcpy( (char __far *) poolptr,
              (char __far *) buff );

    /* store the pointer in outpel[i]					*/
    outpel[i].title = poolptr;

    /* increase poolptr to point to the first free byte	*/
    poolptr += _fstrlen( buff ) + 1;
   }

  for( j=0; j<totmet; j++ )
   for(k=0;k<nsteps;k++,i++)
   {
    wsprintf( (LPSTR) buff,"C([%s],%s)", (LPSTR) metname[j], (LPSTR) stepname[k]);
    /* copy buff to the string pool						*/
    _fstrcpy( (char __far *) poolptr,
              (char __far *) buff );

    /* store the pointer in outpel[i]					*/
    outpel[i].title = poolptr;

    /* increase poolptr to point to the first free byte	*/
    poolptr += _fstrlen( buff ) + 1;
   }

  for( j=0; j<nsteps; j++ )
   for(k=0;k<nsteps;k++,i++)
   {
    wsprintf( (LPSTR) buff,"C(J(%s),%s)", (LPSTR) stepname[j], (LPSTR) stepname[k]);
    /* copy buff to the string pool						*/
    _fstrcpy( (char __far *) poolptr,
              (char __far *) buff );

    /* store the pointer in outpel[i]					*/
    outpel[i].title = poolptr;

    /* increase poolptr to point to the first free byte	*/
    poolptr += _fstrlen( buff ) + 1;
   }

  /* endtime, tolerance, nfunc, njacob, nistep, smstep	*/
  wsprintf( (LPSTR) buff,"time");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = spar[l].title = poolptr;
  /* increase i											*/
  i++; l++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"integration steps");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = poolptr;
  /* increase i											*/
  i++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"function evaluations");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = poolptr;
  /* increase i											*/
  i++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"jacobian evaluations");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = poolptr;
  /* increase i											*/
  i++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"last step size");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = poolptr;
  /* increase i											*/
  i++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"relative tolerance");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = spar[l].title = poolptr;
  /* increase i											*/
  i++; l++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"absolute tolerance");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = spar[l].title = poolptr;
  /* increase i											*/
  i++; l++;
  /* increase poolptr to point to the first free byte	*/
  poolptr += _fstrlen( buff ) + 1;
  wsprintf( (LPSTR) buff,"flux resolution");
  /* copy buff to the string pool						*/
  _fstrcpy( (char __far *) poolptr,
            (char __far *) buff );
  /* store the pointer in outpel[i]						*/
  outpel[i].title = spar[l].title = poolptr;

 return 0;
}


/*
	make strings for reactions
*/

void step_string( void )
{
 int i, j, st, first, tempstoi[MAX_MET];
 char auxstr[128];

 for( j=0; j<nsteps; j++ )
 {
  stepstr[j][0] = '\0';
  for( i=0; i<totmet; i++ ) tempstoi[i] = 0;
  for( i=0; i<MAX_MOL; i++ )
   if ((*rstr)[j][i]<0) tempstoi[ -(*rstr)[j][i]-1 ]--;
  for( i=0, first=1; i<totmet ; i++ )									/* first the substrates	*/
  {
   st = tempstoi[i];
   if( st < 0 )
   {
    if ( st == -1 )
     wsprintf( auxstr, "%s%s", first==1 ? (LPSTR) "" : (LPSTR) " + ", &metname[i][0] );
    else
     wsprintf( auxstr, "%s%i*%s", first==1 ? (LPSTR) "" : (LPSTR) " + ", -st, &metname[i][0] );
    _fstrcat( stepstr[j], auxstr );
    first = 0;
   }
  }
  if( revers[uc[j]] )											/* the connector		*/
   _fstrcat( stepstr[j], " = " );
  else
   _fstrcat( stepstr[j], " -> " );
  /* setup the stoicheiometries of the products						*/
  for( i=0; i<totmet; i++ ) tempstoi[i] = 0;
  for( i=0; i<MAX_MOL; i++ )
   if ((*rstr)[j][i]>0) tempstoi[ (*rstr)[j][i]-1 ]++;
  for( i=0,  first=1; i<totmet; i++ )						/* then the products	*/
  {
   st = tempstoi[i];
   if( st > 0 )
   {
    if( st == 1 )
     wsprintf( auxstr, "%s%s", first==1 ? (LPSTR) "" : (LPSTR) " + ", &metname[i][0] );
    else
     wsprintf( auxstr, "%s%i*%s", first==1 ? (LPSTR) "" : (LPSTR) " + ", st, &metname[i][0] );
    _fstrcat( stepstr[j], auxstr );
    first = 0;
   }
  }
 }
}

/*
  set default values when there is no .INI file
*/

void def_def( void )
{
 dft_pfo = DFT_PFO;
 dft_endtime = DFT_ENDTIME;
 dft_hrcz = DFT_HRCZ;
 dft_debugval = DFT_DEBUGVAL;
 dft_conc = DFT_CONC;
 dft_const = DFT_CONST;
 dft_timeu = DFT_TIMEU;
 dft_concu = DFT_CONCU;
}

/* sets the concentrations and kin. const. to the defaults */
void conc_dft( void )
{
 int i, j;

 for( i=0; i<totmet; i++ ) xu[i] = dft_conc;
 for( i=0; i<nsteps; i++ )
  for( j=0; j<(int)ktype[kinetu[0]].nconst; j++ )
   *(params[i]+j) = dft_const;
}


void numer_dft( void )
{
 options.reltol = DFT_RELTOL;
 options.abstol = DFT_ABSTOL;
 options.hrcz = dft_hrcz;
 options.adams = DFT_ADAMS;
 options.bdf = DFT_BDF;
}

/*
  set variables to default values
*/

void set_dfts( void )
{
 options.endtime = dft_endtime;
 options.debug = dft_debugval;
 options.dyn = DFT_DYN;
 options.ss = DFT_SS;
 options.txt = DFT_TXT;
 options.dat = DFT_DAT;
 options.pfo = dft_pfo;
 strcpy( options.timeu, dft_timeu );
 strcpy( options.concu, dft_concu );
 options.datwidth = DFT_DATWIDTH;
 options.datsep = DFT_DATSEP;
 options.datmca = DFT_DATMCA;
 options.datss = DFT_DATSS;
 options.scan = 0;
 options.scandens = (unsigned long) 1;
 options.quotes = 0;
 plot.x = -1;
 plot.y[0] = -1;
 plot.z = -1;
 plot.ny = 1;
 plot.type = 0;
 plot.file = 0;
 plot.lines = 1;
 plot.colour = 1;
 plot.hidden = 0;
 plot.contour = 0;
 plot.logx = 0;
 plot.logy = 0;
 plot.logz = 0;
 numer_dft();
}


/*
  read default values
*/

void get_dfts( void )
{
 def_def();
 set_dfts();
}


/*
   create a new function tree object
*/

int new_tree( int idx )
{
 if( !eqefl )
 {
  nudf++;
  GlobalUnlock( hTree );
  hTree = GlobalReAlloc( hTree, (DWORD) nudf * sizeof( struct treet ), GMEM_ZEROINIT | GMEM_MOVEABLE );
  if( hTree == NULL )
   return IDS_ERR_NOEXEC;
  tree = ( struct treet huge * ) GlobalLock( hTree );
 }
 _fmemcpy( (void __far *) &tree[idx], (void __far *) &tr, sizeof( struct treet ) );
 return 0;
}

/*
  add a user-defined rate equation to the database
*/

int new_rateq( int idx )
{
 GLOBALHANDLE hTemp;
 WORD strsize;
 int i;
 DWORD l;
 LPSTR tmpptr;

 /* measure the required length for the constant names				*/
 for( i=0, strsize=0; i<tree[idx].nid; i++ )
  if( tree[idx].id[i][9] == (char) 0 )
   strsize += lstrlen(tree[idx].id[i]) + 1;
 sizetr += (DWORD) strsize * sizeof( char );
 l = (DWORD) (treeptr - treestr);
 /* reallocate the buffer to hold new strings						*/
 GlobalUnlock( hTreeStr );
 hTreeStr = GlobalReAlloc( hTreeStr, sizetr, GMEM_ZEROINIT | GMEM_MOVEABLE );
 if( hTreeStr == NULL ) return IDS_ERR_NOEXEC;
 treestr = GlobalLock( hTreeStr );
 treeptr = treestr + l;
 tmpptr = treeptr;
 /* copy the strings to the buffer									*/
 for( i=0; i<tree[idx].nid; i++ )
  if( tree[idx].id[i][9] == (char) 0 )
  {
   lstrcpy( treeptr, tree[idx].id[i] );
   treeptr += lstrlen( tree[idx].id[i] ) + 1;
  }

 /* unlock the memory handle	*/
 GlobalUnlock( hKtype );
 hTemp = GlobalReAlloc( hKtype, (DWORD) (++nrateq) * sizeof( struct kint ), GMEM_ZEROINIT | GMEM_MOVEABLE );
 if( hTemp != NULL )
 {
  hKtype = hTemp;
  ktype = ( struct kint huge * ) GlobalLock( hKtype );
  ktype[MAX_TYP+idx].nmodf = (unsigned char) tree[idx].nmodf;
  ktype[MAX_TYP+idx].nconst =(unsigned char) tree[idx].nconst;
  ktype[MAX_TYP+idx].descr = (LPSTR) tree[idx].descr;
  ktype[MAX_TYP+idx].constnam = (LPSTR) tmpptr;
  return 0;
 }
 else
 {
  ktype = ( struct kint huge * ) GlobalLock( hKtype );
  return IDS_ERR_NOEXEC;
 }
}


/*
  free global heap space allocated
*/

void TidyGepasiVar( void )
{
 GlobalUnlock( hMetname );							/* unlock all global handles	*/
 GlobalUnlock( hStepname );
 GlobalUnlock( hStoiu );
 GlobalUnlock( hSto );
 GlobalUnlock( hLoop );
 GlobalUnlock( hParams );
 GlobalUnlock( hKtype );
 GlobalUnlock( hRstr );
 GlobalUnlock( hOutpEl );
 GlobalUnlock( hScanPar );
 GlobalUnlock( hStrPool );
 GlobalUnlock( hTreeStr );
 GlobalUnlock( hTree );
 GlobalUnlock( hTmpF );

 GlobalFree( hMetname );							/* free all global mem blocks	*/
 GlobalFree( hStepname );
 GlobalFree( hStoiu );
 GlobalFree( hSto );
 GlobalFree( hLoop );
 GlobalFree( hParams );
 GlobalFree( hKtype );
 GlobalFree( hRstr );
 GlobalFree( hOutpEl );
 GlobalFree( hScanPar );
 GlobalFree( hStrPool );
 GlobalFree( hTreeStr );
 GlobalFree( hTree );
 GlobalFree( hTmpF );
}
