/*************************************************************************/
/*                                                                       */
/*                                CONVERT.CPP                            */
/*                                                                       */
/*       A program to convert CICA index files into                      */
/*       comma seperated value files (CSV) suitable for import           */
/*       into common spreadsheets and relational data base managers.     */
/*                                                                       */
/*       This program and associated source & object files are released  */
/*       to the public domain.  You may do what you want with these      */
/*       files, so long as you do not sell them, or remove the original  */
/*       authors's credits.                                              */
/*                                                                       */
/*                               Allen W. Todd                           */
/*                          2711 N.W. Gibson Hill Road                   */
/*                           Albany, Oregon   97321                      */
/*                               (503) 926-2986                          */
/*                                                                       */
/*                      internet:  todda@jacobs.cs.orst.edu              */
/*                                                                       */
/*                                Version 2.2                            */
/*                               April 10, 1993                          */
/*                                                                       */
/*************************************************************************/

#include <stdio.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>

#define  FALSE    0
#define  TRUE     1


enum DateFormats {MDY, DMY, YMD};

/**********  Module Variables  **********/

FILE  *infile;                 // input file
FILE  *outfile;                // output file
long  error_code;              // module error code
int   done;                    // completion / EOF flag
long  line_count;              // number of lines read
char  directory[80];           // current archive directory name
char  a_seperator[2];          // field seperator
char  a_date_delim[2];         // date delimiter
char  a_string_delim[2];       // string field delimiter
int   a_field_names;           // write field names as first line
DateFormats a_date_format;     // current date format;


/**************************************************************************/
/*                                                                        */
/*                                 read_line                              */
/*                                                                        */
/*                         Reads one line from infile.                    */
/*       Correctly handles lines with various line ending sequences       */
/*       such as CR, LF, and CRLF.  Strips any trailing whitespace.       */
/*                                                                        */
/*       Returns:                 Pointer to line read.                   */
/*       Sets Global Variables:   On EOF, sets done = TRUE.               */
/*                                Updates line_count.                     */
/*                                                                        */
/**************************************************************************/

char  *read_line (void)

{
static   char  line[1024];    /* Line input buffer */
         char  c;
         int   i = 0;

   c = fgetc (infile);
   while ( (c != '\r') && (c != '\n') && (c != EOF) )
      {  /* read all char into buffer until CR, LF, or EOF */
      line[i++] = c;
      c = fgetc (infile);
      }
   if (c == '\r') c = fgetc (infile);
   if (c == '\n') c = fgetc (infile);
   if (c == EOF)
      done = TRUE;
   else
      ungetc (c, infile);

   /* remove any trailing white space */

   while ( (i > 0) && (isspace (line[i-1])) ) i--;
   line[i] = '\0';
   line_count++;

   return (&line[0]);
}


/**************************************************************************/
/*                                                                        */
/*                               format_date                              */
/*                                                                        */
/*        Converts from yymmdd date format into mm/dd/yy format.          */
/*                                                                        */
/**************************************************************************/

char  *format_date (char *s)

{
static   char date[9] = "mm/dd/yy";

   switch (a_date_format)
      {
      case MDY:
         strncpy (&date[0], s+2, 2);
         strncpy (&date[3], s+4, 2);
         strncpy (&date[6], s+0, 2);
      break;
      case DMY:
         strncpy (&date[0], s+4, 2);
         strncpy (&date[3], s+2, 2);
         strncpy (&date[6], s+0, 2);
      break;
      case YMD:
         strncpy (&date[0], s+0, 2);
         strncpy (&date[3], s+2, 2);
         strncpy (&date[6], s+4, 2);
      break;
      }

   return (date);
}


/**************************************************************************/
/*                                                                        */
/*                                parse_line                              */
/*                                                                        */
/*              Processes individual lines from CICA index file.          */
/*                                                                        */
/**************************************************************************/

void parse_line (char *line)

{
   char  *s;
   char  *field0;
   char  *field1;
   char  *field2;

   if (line_count == 1)
      {  /* display archive index date */
//    fprintf (stderr, "%s\n", line);
      }
   else
      {
      if (*line == '*')    /* process directory headers */
         {
         if ((s = strstr (line, "~ftp/pub/pc/win3/")) != NULL)
            {  /* get directory name */
            strcpy (directory, s+17);  /* point past ~ftp/pub/pc/win3/ string */
            s = directory;
            while (isgraph(*s)) s++;   /* prune to end of line */
            *s = '\0';
//          fprintf (stderr, "Processing Directory: %s\n", directory);
            }
         }
      else                 /* process individual file lines*/
         {
         s = line;
         field0 = line;
         while (isgraph (*s)) s++;  /* get first field */
         *s = '\0';
         s++;
         while (isspace (*s)) s++;
         field1 = s;
         while (isgraph (*s)) s++;  /* get second field */
         *s = '\0';
         s++;
         while (isspace (*s)) s++;  /* get third field */
         field2 = s;
         fprintf (outfile,
                  "%s%s%s%s%s%s%s%s%s%s%s\n",
                  directory,
                  a_seperator,
                  field0,
                  a_seperator,
                  a_date_delim,
                  format_date (field1),
                  a_date_delim,
                  a_seperator,
                  a_string_delim,
                  field2,
                  a_string_delim);
         }
      }
}



/**************************************************************************/
/*                                                                        */
/*                              CONVERT_CSV                               */
/*                                                                        */
/*                 Actual CICA index to CSV file converter.               */
/*                                                                        */
/**************************************************************************/

long convert_csv (char *infile_name,
                  char *outfile_name,
                  char *seperator,
                  char *date_delim,
                  char *string_delim,
                  int  field_names,
                  DateFormats DateFormat)

{
   char  *line;

   // make local copy of passed data
   strcpy (a_seperator, seperator);
   strcpy (a_date_delim, date_delim);
   strcpy (a_string_delim, string_delim);
   a_field_names = field_names;
   a_date_format = DateFormat;

   // initialize module vars

   error_code = 0L;
   done = FALSE;
   line_count = 0L;

   if ((infile = fopen (infile_name, "rb")) != NULL)
      {
      if ((outfile = fopen (outfile_name, "w")) != NULL)
         {
         if (a_field_names) fprintf (outfile,
                                   "Directory%sFile_Name%sFile_Date%sDescription\n",
                                    a_seperator,
                                    a_seperator,
                                    a_seperator);

         while (!done) /* read until end of file */
            {
            line = read_line();
            if (*line) parse_line (line); /* only process non-blank lines */
            }
         fclose (outfile);
         }
      else error_code = -2;
      fclose (infile);
      }
   else error_code = -1;

   if (error_code >= 0L)
      return (line_count);
   else
      return (error_code);
}
