
// This module defines all of the "callback" functions.  These modules
// are defined here due to the fact that they must be mult-threaded and
// therefore completely reentrant.  They must be linked with the proper
// C Runtime Multi-Threaded Library (LLIBCDLL.LIB).

#include	<stdlib.h>
#include	<stdio.h>
#include	<string.h>

#define INCL_BASE
#include	<OS2.h>

// For LanMan api calls

#define     INCL_NETUSER
#define     INCL_NETGROUP
#define     INCL_NETERRORS
#define     INCL_NETHANDLE
#include    <lan.h>        // LAN Manager header files

#define BUFFLEN 	   128


#include	<sqlfront.h>
#include	<sqldb.h>

#include	<srv.h>

//  Define some GDK user message codes.
//
#define	SRV_MAXERROR	20000
#define	COMPUTE_ROW	SRV_MAXERROR + 2
#define	REMOTE_MSG	SRV_MAXERROR + 3
#define SEND_FAILED	SRV_MAXERROR + 4

#define BULK_CMD	"insert bulk"

//  This error value must be sent back to the client in the event
//  of a failure to login to the target database.  It is used in the
//  init_remote() function below.
//
#define REMOTE_FAIL 4002


//  Declare the structure to use for our private data area.
//  It will be passed to the event handlers in the SRV_PROC structure.
//  It contains information to make and use a connection to the remote
//  DBMS -- in this case a SQL Server.
//
//  A gateway to a non-SQL Server DBMS would contain a different structure.
//
#define	MAXPARAMS	255
typedef struct remote_dbms
{
    LOGINREC	FAR *login;	    // The SQL Server login structure
    DBPROCESS	FAR *dbproc;	    // The connection to the remote SQL Server
    BOOL	      bulk_mode;	    // Flag indicating "bulk insert" mode
    int 	      retparams[MAXPARAMS];   // tag return parameters
} REMOTE_DBMS;

// The remote server name of this GDK
//
DBCHAR		*remote_server = NULL;
SRV_PROC	*Newsrvproc;		// Latest SRV_PROC allocated.
ULONG		init_remote_SEM = 0L;	// Semaphore used init_remote()


// Flags declared in SECURE.C for turning functions on or off
BOOL fLogEnabled;	 // request logging
BOOL fConnectCheck;	 // LM connection verification

//  Prototype dblib API's
//
void	dbsetuserdata( DBPROCESS FAR * dbproc,
		       void	 FAR * data );

void FAR *
	dbgetuserdata( DBPROCESS FAR * dbproc );

int	dbcolntype   ( DBPROCESS FAR * dbproc,
		       int	       column );

int	attn_handler ( SRV_PROC FAR * srvproc );

int	chk_err      ( SRV_SERVER FAR * server,
		       SRV_PROC   FAR * srvproc,
		       int		srverror,
		       BYTE		severity,
		       BYTE		state,
		       int		oserrnum,
		       DBCHAR	  FAR * errtext,
		       int		errtextlen,
		       DBCHAR	  FAR * oserrtext,
		       int		oserrtextlen );

int	init_remote  ( SRV_PROC FAR * srvproc );

int	init_server  ( SRV_SERVER FAR * server );

int	exit_remote  ( SRV_PROC FAR * srvproc );

int	lang_execute ( SRV_PROC FAR * srvproc );

int	handle_results( DBPROCESS FAR * rmtproc,
			SRV_PROC  FAR * srvproc );

int	remotemsgs   ( DBPROCESS  FAR * dbproc,
		       DBINT		msgno,
		       DBSMALLINT	msgstate,
		       DBSMALLINT	severity,
		       char	  FAR * msgtext );

int	remoteerr    ( DBPROCESS  FAR * dbproc,
		       int		severity,
		       int		dberr,
		       int		oserr,
		       char	  FAR * dberrstr,
		       char	  FAR * oserrstr );

void	set_remote_server_name( char FAR * name );

// New function for setting options
void	set_security_options( BOOL logflag, BOOL checkflag);


//
//  This section defines the "call-back" functions supplied to the gateway
//  application.
//
#pragma check_stack( off )  // turn off stack checking


//
//  SET_REMOTE_SERVER_NAME
//
//	This function sets the name of the remote server.
//
//  Parameters:
//	Pointer to name of server.
//
//  Returns:
//	none
//
void	set_remote_server_name(name)
char FAR *name;
{
    remote_server = name;
}

//  SET_SECURITY_OPTIONS
//
//	This function sets connection checking and request logging on or off.
//
//  Parameters:
//	logflag;	// request logging on or off
//	checkflag;	// LM connection verification on or off
//
//  Returns:
//	none
//
void	set_security_options( logflag, checkflag )
BOOL logflag;
BOOL checkflag;
{
    fLogEnabled = logflag;
    fConnectCheck = checkflag;
}

//
//  INIT_SERVER
//	Initialize the server on a SRV_START event.
//	Event handlers for the server are installed.
//
//  Parameters:
//	Pointer to SRV_SERVER structure
//
//  Returns:
//	SRV_CONTINUE
//
int	init_server(server)
SRV_SERVER  FAR *server;
{
    char    log_buffer[256];

    //	When we get a connection request from a client, we want to
    //	call "init_remote()" to make a connection to the remote
    //	server.
    //
    srv_handle(server, (DBINT)SRV_CONNECT, init_remote);

    // When the client issues a language request, call
    // "lang_execute()" to send the SQL statement to the remote DBMS.
    //
    srv_handle(server, (DBINT)SRV_LANGUAGE, lang_execute);

    // When a disconnect request is issued, call "exit_remote()"
    // to close the connection to the remote DBMS.
    //
    srv_handle(server, (DBINT)SRV_DISCONNECT, exit_remote);

    //	 Install the handler that will be called when the
    //	 GDK receives an attention from one of its
    //	 clients.  An attention gets received anytime a client
    //	 calls dbcancel().
    //
    srv_handle(server, (DBINT)SRV_ATTENTION, attn_handler);

    //	Now we'll install the message and error handlers for any
    //	messages from the remote DBMS.
    //
    dberrhandle(remoteerr);
    dbmsghandle(remotemsgs);

    //	Log Server information to log file
    //
    sprintf( log_buffer,
	     "Server pipe name = %s",
	     srv_sfield(server, SRV_SERVERNAME, (int *)NULL) );

    srv_log( server, FALSE, log_buffer, SRV_NULLTERM );
    printf(  "%s\n", log_buffer );

    sprintf( log_buffer,
	     "Client connections allow = %s",
	     srv_sfield(server, SRV_CONNECTIONS, (int *)NULL) );

    srv_log( server, FALSE, log_buffer, SRV_NULLTERM );
    printf(  "%s\n", log_buffer );

    return (SRV_CONTINUE);
}


//  INIT_REMOTE
//  Event handler for a SRV_CONNECT event.
//  A connection is made to the remote DBMS.
//
//  In this example program, the connection is to a SQL Server.
//  If using a non-SQL Server remote DBMS, the connection code
//  would be different, but would probably still occur here.
//
//  Parameters:
//	srvproc - the handle to the client connection that got the SRV_CONNECT.
//
//  Returns:
//	SRV_CONTINUE
//
//  Side Effects:
//	If the connection to the remote dbms cannot be made, then issue
//	a SRV_DISCONNECT request.
//
//
int	init_remote(srvproc)
SRV_PROC    FAR *srvproc;
{
    REMOTE_DBMS FAR *remote;	    // the private data area to keep
				    // track of the remote DBMS
				    // connection.


// ***** Modifications for Lan Man security verification begin here

    char   achBuffer[BUFFLEN];		  // Data buffer
    unsigned short hPipe;		  // Handle to named pipe
    unsigned short cbAvail;		   // Bytes available from GetInfo
    API_RET_TYPE uReturnCode;		   // LAN Manager API return code
    struct handle_info_2 * pHandleInfo2;
    DBINT uCompareResult;		   // result of string compare
    char    log_buffer[256];
    SRV_SERVER	      FAR * server;	  // pointer to server info for log

    if (fConnectCheck == TRUE) {
	// Query to see who is on the other end of the pipe

	hPipe = (unsigned short) srvproc->srvio.handle;

	uReturnCode = NetHandleGetInfo(
                     hPipe,               // Handle to named pipe
                     2,                   // Level 2
                     achBuffer,           // Data returned here
                     BUFFLEN,             // Size of buffer, in bytes
                     (unsigned short far *) &cbAvail);

	if (uReturnCode == NERR_Success) {
	    pHandleInfo2 = (struct handle_info_2 *) achBuffer;
	    printf("   User of the named pipe is %Fs\n",
				pHandleInfo2->hdli2_username );
	    printf(" login record name is %s\n",
				srv_pfield(srvproc, SRV_USER, (int *) NULL));
	}
	else {
	    printf("  Net HandleGetInfo failed, return code %d", uReturnCode);
	    server = SRV_GETSERVER(srvproc);
	    srv_log( server, TRUE,  "Call to LanMan failed", SRV_NULLTERM );
	}

	// Compare login name with handle user name.  If they are different,
	// reject the login.

	uCompareResult = strcmpi(pHandleInfo2->hdli2_username,
	       srv_pfield(srvproc, SRV_USER, (int *) NULL));
	if (uCompareResult != 0) {
		//  Send a message to the client that
		//  they must connect with their network login.
		//
		srv_sendmsg( srvproc,
			     SRV_MSG_ERROR,
			     (DBINT)REMOTE_FAIL,
			     (DBTINYINT)0,
			     (DBTINYINT)0,
			     NULL,
			     0,
			     0,
			     "Login id must match network user id.",
			     SRV_NULLTERM );

		srv_senddone(srvproc, SRV_DONE_FINAL | SRV_DONE_ERROR , (DBUSMALLINT) 0, (DBINT) 0);

		sprintf( log_buffer,
			 "LOGIN REJECTED: username = %s, login request = %s",
			 pHandleInfo2->hdli2_username,
			 srv_pfield(srvproc, SRV_USER, (int *) NULL));

		server = SRV_GETSERVER(srvproc);
		srv_log( server, TRUE, log_buffer, SRV_NULLTERM );

		return( SRV_DISCONNECT );

	}
    }

// ***** Modifications for Lan Man security verification end here


    //	Set Newsrvproc.  This is used if we get an error on
    //	the open from DBLIB.  Since there isn't a dbproc,
    //	it is clear which srvproc to send the msg back on when the
    //	DBLIB error-handler gets called.
    //
    //	First lock out any other threads trying to connect using
    //	this same function.
    //
    DosSemRequest( &init_remote_SEM, -1L );

    Newsrvproc = srvproc;

    //	Allocate a REMOTE_DBMS information structure.
    //
    remote = (REMOTE_DBMS *) srv_alloc((DBINT)sizeof(*remote));

    //	Try to open a connection to the remote DBMS.
    //
    if( remote == NULL ) {
	//  Send a message to the client that
	//  the remote connection failed.
	//
	srv_sendmsg( srvproc,
		     SRV_MSG_ERROR,
		     (DBINT)REMOTE_FAIL,
		     (DBTINYINT)0,
		     (DBTINYINT)0,
		     NULL,
		     0,
		     0,
		     "Login to remote DBMS failed.",
		     SRV_NULLTERM );

	// Now allow other threads to enter this function.
	//
	DosSemClear( &init_remote_SEM );
	return( SRV_DISCONNECT );
    }

    //	Set "bulk insert" mode flag to false.
    //
    remote->bulk_mode = FALSE;


    //	Allocate the LOGINREC structure used to make connections to the
    //	remote server.	Open the connection in the SRV_CONNECT handler.
    //
    remote->login = dblogin();

    if( remote->login == NULL ) {
	//  Send a message to the client that the
	//  remote connection failed.
	//
	srv_sendmsg( srvproc,
		     SRV_MSG_ERROR,
		     (DBINT)REMOTE_FAIL,
		     (DBTINYINT)0,
		     (DBTINYINT)0,
		     NULL,
		     0,
		     0,
		     "Login to remote DBMS failed.",
		     SRV_NULLTERM );

	//  Deallocate the remote structure and set the user data
	//  pointer in "srvproc" to null so that the disconnect
	//  handler won't try to disconnect from the remote dbms.
	//
	srv_free(remote);

	// Now allow other threads to enter this function.
	//
	DosSemClear( &init_remote_SEM );
	return( SRV_DISCONNECT );
    }

    remote->dbproc = (DBPROCESS *) NULL;

    //	Set the user name, password, and application name for the remote DBMS.
    //
    DBSETLUSER(remote->login, srv_pfield(srvproc, SRV_USER,	(int *) NULL));
    DBSETLPWD (remote->login, srv_pfield(srvproc, SRV_PWD,	(int *) NULL));
    DBSETLAPP (remote->login, srv_pfield(srvproc, SRV_APPLNAME,	(int *) NULL));

    // See if client has set Bulk Copy flag
    //
    if( strcmp( srv_pfield(srvproc, SRV_BCPFLAG, (int *) NULL), "TRUE") == 0 )
	BCP_SETL( remote->login, TRUE );
    else
	BCP_SETL( remote->login, FALSE);

    //	If no server name was specified as a parameter to the main program,
    //	then assume that the name is coming from the client.
    //
    if( remote_server == NULL || remote_server == "" ) {
	remote_server = srv_pfield( srvproc, SRV_HOST, (int *)NULL );
    }

    //	Try to open a connection to the remote DBMS.
    //
    if ((remote->dbproc = dbopen(remote->login, remote_server))
	    == (DBPROCESS *) NULL) {

	//  Send a message to the client that
	//  the remote connection failed.
	//
	srv_sendmsg( srvproc,
		     SRV_MSG_ERROR,
		     (DBINT)REMOTE_FAIL,
		     (DBTINYINT)0,
		     (DBTINYINT)0,
		     NULL,
		     0,
		     0,
		     "Login to remote DBMS failed.",
		     SRV_NULLTERM );

	//  Deallocate the remote structure and set the user data
	//  pointer in "srvproc" to null so that the disconnect
	//  handler won't try to disconnect from the remote DBMS.
	//
	srv_free(remote);
	srv_setuserdata(srvproc, (BYTE *) NULL);

	// Now allow other threads to enter this function.
	//
	DosSemClear( &init_remote_SEM );
	return( SRV_DISCONNECT );
    }
    else {
	//  Connection to the remote DBMS successful.  Save
	//  remote data structure in the "srvproc" so it will be
	//  available to the other handlers. We'll also map the remote
	//  DBMS connection to our "srvproc".
	//
	srv_setuserdata(srvproc, (BYTE *) remote);
	dbsetuserdata ( remote->dbproc, (SRV_PROC *)srvproc );
    }

    // Now allow other threads to enter this function.
    //
    DosSemClear( &init_remote_SEM );

    //	Display connect info on console
    //
    printf(  "\nClient process ID: %s\n",
	     srv_pfield(srvproc, SRV_CPID, (int *) NULL));

    printf(  "User name: %s\n",
	     srv_pfield(srvproc, SRV_USER, (int *) NULL));

    printf(  "Application program name: %s\n",
	     srv_pfield(srvproc, SRV_APPLNAME, (int *) NULL));

    return (SRV_CONTINUE);
}


//
//  LANG_EXECUTE
//	Execute a client language request on the remote dbms.
//	Any results from the remote dbms are passed back to the GDK client.
//
//  Parameters:
//	srvproc - GDK process handle to the current client connection.
//
//  Returns:
//	SRV_CONTINUE
//
#define MAXQUERY	(64 * 1024)

int	lang_execute(srvproc)
SRV_PROC    FAR *srvproc;
{
    REMOTE_DBMS FAR * rmt_dbms;     // the remote database pointer
    DBPROCESS	FAR * rmtproc;	    // our DBPROCESS pointer
    DBCHAR	FAR * query;	    // pointer to language buffer
    long	      query_len;    // length of query
    int 	      status;	    // status of dblib calls

    SRV_SERVER	      FAR * server;	  // pointer to server info for log
    DBCHAR    FAR * user;	  // userid for log file

    //	Get the remote dbms pointer we saved in the srvproc via
    //	srv_setuserdata.
    //
    rmt_dbms = (REMOTE_DBMS *)srv_getuserdata( srvproc );

    //	Get the pointer to the remote DBPROCESS
    //
    rmtproc = rmt_dbms->dbproc;

    //	Get the pointer to the client language request command buffer.
    //
    query = srv_langptr( srvproc );


    // ***** Request Logging modifications start here

    if (fLogEnabled == TRUE) {
	server = SRV_GETSERVER(srvproc);
	user = srv_pfield (srvproc, SRV_USER,(int *) NULL);
	srv_log (server, TRUE, user, SRV_NULLTERM);
	srv_log (server, FALSE, query, SRV_NULLTERM );
    }
    // ***** Request Logging modifications end here


    //	See if the previous command was a "bulk insert" command
    //
    if( rmt_dbms->bulk_mode ) {

	// Get length of the SQL command.
	//
	query_len = srv_langlen( srvproc );

	// If length of data is zero, then send a zero length buffer
	// to the destination SQL Server.  This is required in order to
	// signal the SQL Server that no more data follows.
	//
	if( query_len == -1L )
	    query_len = 0L;

	//	Place buffer into target SQL server's buffer.  Since
	//	the buffer is binary data from the client, we need some
	//	method of sending it to the target SQL Server.	We will use
	//  dbbcmd(), an undocumented DBLIB API that is used for sending 
	// binary data.
	//
	  status = dbfcmd( rmtproc, query, query_len );

	rmt_dbms->bulk_mode = FALSE;
    }
    else {
	//  Let's check for "insert bulk" request
	//
	if( srv_langlen(srvproc) > (sizeof(BULK_CMD) - 1) )
	    if( strnicmp(query, BULK_CMD, (sizeof(BULK_CMD) - 1)) == 0 )
		rmt_dbms->bulk_mode = TRUE;

	//  Place buffer into target SQL server's buffer.
	//
	status = dbcmd(rmtproc, query);
    }

#ifdef _ASYNC_ATTENTIONS
    //	Enable "attention" checking from the client.  Whenever an "attention"
    //	is received, the call back function 'attn_handler()' is invoked.
    //
    srv_enable_attentions( srvproc );
#endif

    // If previous DBLIB call successful, send command buffer to SQL Server.
    //
    if( status == SUCCEED ) {	// if previous DBLIB call successful
	status = dbsqlexec(rmtproc);
    }

    if (!SRV_GOT_ATTENTION(srvproc) && (status == SUCCEED)) {
	//
	//  Get the remote dbms results and pass them back to
	//  GDK client.
	//
	handle_results(rmtproc, srvproc);
    }
    else {
	//
	// If an attention event was issued or the dbsqlexec failed,
	// acknowledge with senddone.
	//
        if ( DBDEAD(rmtproc) )
        {
            printf("thread shutting down");
            srv_sendmsg( srvproc,
                 SRV_MSG_ERROR,
                 (DBINT)REMOTE_FAIL,
                 (DBTINYINT)23,
                 (DBTINYINT)17,
                 NULL,
                 0,
                 0,
                 "Remote Server connection no longer active: thread shutting down",
                 SRV_NULLTERM );
            srv_senddone(srvproc, SRV_DONE_FINAL | SRV_DONE_ERROR , (DBUSMALLINT) 0, (DBINT) 0);
            srv_event(srvproc, SRV_DISCONNECT, NULL );
        }
        else
        {
            srv_senddone(srvproc, SRV_DONE_FINAL, (DBUSMALLINT) 0, (DBINT) 0);
        }
    }

#ifdef _ASYNC_ATTENTIONS
    //	Disable "attention" checking.
    //
    srv_disable_attentions( srvproc );
#endif

return (SRV_CONTINUE);
}


//
//  HANDLE_RESULTS
//	Once a command has been sent to the remote DBMS by the
//	SRV_LANGUAGE  handler, this routine processes the results
//	and passes them back to the client.
//
//  Parameters:
//	rmtproc - The DBPROCESS handle to the remote DBMS.
//	srvproc - The GDK process handle to use to send results to the client.
//
//  Returns:
//	SUCCEED or FAIL.
//
int	handle_results(rmtproc, srvproc)
DBPROCESS   FAR *rmtproc;
SRV_PROC    FAR *srvproc;
{
    int 	    cols;	    // data columns returned
    int 	    i;		    // index variable
    DBINT	    rows;	    // number of rows sent
    BOOL	    results_sent;   // number of result sets sent
    BOOL	    gotcompute;     // COMPUTE row indicator
    int 	    *paramarray;
    RETCODE	    returnvalue;    // value returned from dblib call

    results_sent = FALSE;
    rows = 0L;
    gotcompute = FALSE;
    paramarray = ((REMOTE_DBMS *) srv_getuserdata(srvproc))->retparams;

    //	Process the results from the remote DBMS.
    //	Since a command may consist of multiple commands or a single
    //	command that has multiple sets of results, we'll loop through
    //	each results set.
    //
    while( TRUE ) {

#ifdef _ASYNC_ATTENTIONS
	//
	//  Disable "attention" checking.
	//
	srv_disable_attentions( srvproc );
#endif

	returnvalue = dbresults(rmtproc);
	if( returnvalue == NO_MORE_RESULTS ) {
	    break;
	}

#ifdef _ASYNC_ATTENTIONS
	//  Reenable "attention" checking
	//
	srv_enable_attentions( srvproc );
#endif

	// Check to see if the client has sent an attention event.  If
	// so, simply discard data from remote server
	//
	if (SRV_GOT_ATTENTION(srvproc)) {
	    continue;
	}

	//
	//  If this is the second time through the loop,
	//  send a completion message to the client
	//  for the previous results sent.
	//
	if (results_sent == TRUE) {

	    //	If there are some COMPUTE rows, send a message
	    //	to the client that GDK Library doesn't yet handle them.
	    //
	    if (gotcompute == TRUE) {
		gotcompute = FALSE;
		srv_sendmsg( srvproc,
			     SRV_MSG_ERROR,
			     (DBINT)COMPUTE_ROW,
			     (DBTINYINT)0,
			     (DBTINYINT)0,
			     NULL,
			     0,
			     0,
			     "GDK can't handle COMPUTE rows.",
			     SRV_NULLTERM );
	    }

	    //	If the previous batch was one that may
	    //	have returned rows, set the DONE status
	    //	accordingly.
	    //
	    if( rows > 0 ) {
		srv_senddone( srvproc,
			      SRV_DONE_MORE | SRV_DONE_COUNT,
			      (DBUSMALLINT)0,
			      rows );
	    }
	    else
		srv_senddone( srvproc,
			      SRV_DONE_MORE,
			      (DBUSMALLINT)0,
			      (DBINT)0 );
	}

	//  How many data columns are in the row?
	//  Non-"select" statements will have 0 columns.
	//
	cols = dbnumcols(rmtproc);

	//  Build the row description for the client return.
	//
	for (i = 1; i <= cols; i++) {

	    //	Call "srv_describe()" for each column in the row.
	    //
	    srv_describe(srvproc,
			 i,
			 dbcolname( rmtproc, i ),
			 SRV_NULLTERM,
			 (DBINT)dbcolntype( rmtproc, i ),
			 dbcollen ( rmtproc, i ),
			 (DBINT)dbcolntype( rmtproc, i ),
			 dbcollen (rmtproc, i),
			 (BYTE *)NULL);
	}

	//  Send each row from the remote DBMS to the client.
	//
	rows = 0;
	while( TRUE ) {

#ifdef _ASYNC_ATTENTIONS
	    //
	    //	Disable "attention" checking.
	    //
	    srv_disable_attentions( srvproc );
#endif

	    returnvalue = dbnextrow( rmtproc );
	    if( returnvalue == NO_MORE_ROWS ) {
		break;
	    }

#ifdef _ASYNC_ATTENTIONS
	    //	Reenable "attention" checking
	    //
	    srv_enable_attentions( srvproc );
#endif

	    //	If it's not a regular row, it's a COMPUTE row.
	    //	This SQL extension is particular to Sybase
	    //	TRANSACT-SQL and is not yet supported.
	    //
	    if (DBROWTYPE(rmtproc) != REG_ROW) {
		gotcompute = TRUE;
		continue;
	    }
	    else
		gotcompute = FALSE;

	    //	The row description is built.  Move the
	    //	rows from the remote server to the client.
	    //
	    for (i = 1; i <= cols; i++) {
		srv_setcollen( srvproc, i, (int)dbdatlen(rmtproc, i) );
		srv_setcoldata( srvproc, i, dbdata(rmtproc, i) );
	    }

	    // Check to see if the client has issued an attention event.
	    // If so, discard data from the remote server.
	    //
	    if (SRV_GOT_ATTENTION(srvproc)) {
		continue;
	    }

	    if (srv_sendrow(srvproc) == SUCCEED)
		rows++;
	}


	//  If the command was one where count is meaningful, DBCOUNT()
	//  will be >= 0 and we will set rows accordingly.
	//
	if (dbnumcols(rmtproc) > 0)
	    rows = DBCOUNT(rmtproc) < 0L ? 0L: rows;
	else
	    rows = DBCOUNT(rmtproc);

	//  Set flag so that we will send a completion
	//  message for the current set of results.
	//
	results_sent = TRUE;
    }

    //	If there are some COMPUTE rows, send a message
    //	to the client that GDK Library doesn't handle them yet.
    //
    if (gotcompute == TRUE) {
	gotcompute = FALSE;
	srv_sendmsg( srvproc,
		     SRV_MSG_ERROR,
		     (DBINT)COMPUTE_ROW,
		     (DBTINYINT)0,
		     (DBTINYINT)0,
		     NULL,
		     0,
		     0,
		     "GDK Library can't yet handle COMPUTE rows.",
		     SRV_NULLTERM );
    }

    //	Send the final done packet for the execution of the command batch.
    //
    //	If the previous batch was one that may
    //	have returned rows, set the DONE status
    //	accordingly.
    //
    if (rows > 0) {
	srv_senddone( srvproc,
		      SRV_DONE_COUNT | SRV_DONE_FINAL,
		      (DBUSMALLINT)0,
		      rows );
    }
    else {
	srv_senddone( srvproc,
		      SRV_DONE_FINAL,
		      (DBUSMALLINT)0,
		      (DBINT)0 );
    }

    return (SUCCEED);
}


//  EXIT_REMOTE
//	Handler for SRV_DISCONNECT events.
//	Closes remote DBMS connection if appropriate.
//
//  The code to disconnect from the remote DBMS is SQL Server specific. If
//  using a non-SQL Server database, the disconnect from the  remote database
//  would be different but would probably still occur here.
//
//  Parameters:
//	srvproc - the handle to the client connection
//
//  Returns:
//	SRV_DISCONNECT
//
int	exit_remote(srvproc)
SRV_PROC    FAR *srvproc;
{
    REMOTE_DBMS FAR *remote;	    // pointer to target connect structure

    remote = (REMOTE_DBMS *) srv_getuserdata(srvproc);

    //	Is there a REMOTE_DBMS structure to clean-up?
    //
    if (remote != (REMOTE_DBMS *) NULL) {

	//  Is there a live dbproc?
	//
	if (remote->dbproc != (DBPROCESS *) NULL) {
	    dbclose(remote->dbproc);
	}

	dbfreelogin( remote->login );
	srv_free(remote);
    }

    //	Display info on console
    //
    printf(  "\nClient connection closed, process ID: %s\n",
	     srv_pfield(srvproc, SRV_CPID, (int *) NULL));

    return (SRV_CONTINUE);
}


//  CHK_ERR
//	Print out GDK errors.
//
//  Parameters:
//	server	     - pointer to gateway server structure.
//	srvproc      - pointer to client connection structure
//	errornum     - error number.
//	severity     - error severity.
//	state	     - error state.
//	oserrnum     - operating system error number, if any.
//	errtext      - the text of the error message.
//	errtextlen   - length of the errtext message
//	oserrtext    - the text of the operating system error message.
//	oserrtextlen - length of the errtext message
//
//  Returns:
//	SRV_CONTINUE, SRV_CANCEL, or SRV_EXIT_PROGRAM
//

int chk_err(server, srvproc, errornum, severity, state, oserrnum, errtext,
	    errtextlen, oserrtext, oserrtextlen)
SRV_SERVER  FAR *server;
SRV_PROC    FAR *srvproc;
int		errornum;
BYTE		severity;
BYTE		state;
int		oserrnum;
DBCHAR	    FAR *errtext;
int		errtextlen;
DBCHAR	    FAR *oserrtext;
int		oserrtextlen;
{
    char    log_buffer[256];

    //	Operating system error?
    //
    if (oserrnum != SRV_ENO_OS_ERR)
    {
	sprintf( log_buffer,
		 "SERVER OS ERROR: %d: %s.",
		 oserrnum,
		 oserrtext );

	srv_log(server, TRUE, log_buffer, SRV_NULLTERM );
    }

    //	Is this a fatal error for the GDK server?
    //
    if (severity >= SRV_FATAL_SERVER)
    {
	sprintf( log_buffer,
		 "SERVER: FATAL SERVER ERROR: errornum = %d, "
		 "severity = %d, state = %d: %s.",
		 errornum,
		 severity,
		 state,
		 errtext );

	srv_log(server, TRUE, log_buffer, SRV_NULLTERM );
	return (SRV_EXIT);
    }
    else {
	//
	//  Did the "srvproc" get a fatal error?
	//
	if (severity >= SRV_FATAL_PROCESS)
	{
	    sprintf( log_buffer,
		     "SERVER: FATAL CONNECT ERROR: errornum = %d, "
		     "severity = %d, state = %d: %s.",
		     errornum,
		     severity,
		     state,
		     errtext );

	    srv_log(server, TRUE, log_buffer, SRV_NULLTERM );

	    //
	    //	Make a disconnect request, to be executed if possible.
	    //
	    // if (srvproc != (SRV_PROC *) NULL)
	    //	       srv_event(srvproc, SRV_DISCONNECT, (BYTE *) NULL);

	    return (SRV_CANCEL);
	}
    }

    //	A non-fatal error or an information message received.
    //	We'll pass it through to the client.
    //
    if (srvproc != (SRV_PROC *) NULL && (server != NULL))

	if( severity < 10 ) {		    // if informational message
	    srv_sendmsg( srvproc,
			 SRV_MSG_INFO,
			 (DBINT)errornum,
			 severity,
			 0,
			 NULL,
			 0,
			 0,
			 errtext,
			 SRV_NULLTERM );
	}
	else {				    // must be an error message
	    srv_sendmsg( srvproc,
			 SRV_MSG_ERROR,
			 (DBINT)errornum,
			 severity,
			 0,
			 NULL,
			 0,
			 0,
			 errtext,
			 SRV_NULLTERM );
	}
    else {
	sprintf( log_buffer,
		 "GATEWAY ERROR: errornum = %d, severity = %d: %s",
		 errornum,
		 severity,
		 errtext );

	srv_log( server, TRUE, log_buffer, SRV_NULLTERM );
    }

    return (SRV_CONTINUE);
}


//  REMOTEMSGS
//	DBLIB calls this routine when any messages are received
//	from the remote DBMS. It gets the remote message information and
//	sends it back to the client as a message.
//
//  Parameters:
//	dbproc	 - The handler to the remote DBMS process that sent the message.
//	msgno	 - The message number.
//	msgstate - The message state.
//	severity - The message severity.
//	msgtext  - The text of the message.
//
//	The following three parameters are available in TDS4.0 and above:
//
//	srvname  - The name of the server that sent the message.
//	procname - The procedure name, if any, of the remote DBMS command.
//	line	 - The remote DBMS command buffer line to which the msg applies.
//
//  Returns:
//	0
//

int	remotemsgs(dbproc, msgno, msgstate, severity, msgtext )

DBPROCESS   FAR * dbproc;
DBINT		  msgno;
DBSMALLINT	  msgstate;
DBSMALLINT	  severity;
char	    FAR * msgtext;
{
    SRV_PROC	FAR *srvproc;

    // If a remote DBMS error was received during the remote
    // open, the dbproc is NULL and a message is sent back on the
    // most recent srvproc.
    //
    if (dbproc == (DBPROCESS *) NULL) {
	 srvproc = Newsrvproc;
    }
    else {
	if( (srvproc = (SRV_PROC *) dbgetuserdata( dbproc )) == NULL ) {
	    //
	    // An error was received after the dbproc was assigned, but
	    // before we were able to associate our srvproc.
	    //
	    srvproc = Newsrvproc;
	}
    }

    if( severity < 10 ) {		    // if informational message

	srv_sendmsg( srvproc,
		     SRV_MSG_INFO,
		     msgno,
		     (DBTINYINT)severity,
		     (DBTINYINT)msgstate,
		     NULL,
		     0,
		     0,
		     msgtext,
		     SRV_NULLTERM );
    }
    else {				    // must be an error message

	srv_sendmsg( srvproc,
		     SRV_MSG_ERROR,
		     msgno,
		     (DBTINYINT)severity,
		     (DBTINYINT)msgstate,
		     NULL,
		     0,
		     0,
		     msgtext,
		     SRV_NULLTERM );
    }

    return (0);
}


//  REMOTEERR
//	This is the handler for error messages from the remote DBMS, in this
//	case DBLIB.  It gets called whenever a DBLIB error occurs.  It takes
//	the error message and passes it back to the GDK client.
//
//  Parameters:
//	dbproc - The process handle for the remote DBMS.
//	severity - The severity of the error.
//	dberr - The DBLIB error number.
//	oserr - The operating system error, if any.
//	dberrstr - The text of the DBLIB error.
//	oserrstr - The text of operating system error, if any.
//
//  Returns:
//	INT_EXIT to exit the program.
//	INT_CANCEL to cause a FAIL return from the DBLIB routine that got
//	the error.
//
int	remoteerr(dbproc, severity, dberr, oserr, dberrstr, oserrstr)
DBPROCESS   FAR *dbproc;
int		severity;
int		dberr;
int		oserr;
char	    FAR *dberrstr;
char	    FAR *oserrstr;
{
    SRV_PROC * srvproc = (SRV_PROC *) NULL;

    //	If the DBLIB process is dead or
    //  we get a DBLIB error 10007 ("General SQL Server Error:...")
    //	then simply ignore it.	The error message has already been sent
    //	to the client.
    //
    if( DBDEAD( dbproc ) || dberr == 10007 ) {
	return( INT_CANCEL );
    }

    //
    // A remote DBMS error may have been issued during the remote
    // open. In this case, the dbproc will be NULL and a message
    // will be sent on the most recent srvproc.
    //
    if (dbproc == (DBPROCESS *) NULL) {
	 srvproc = Newsrvproc;
    }
    else {
	if( (srvproc = (SRV_PROC *)dbgetuserdata( dbproc )) == NULL ) {
	    // An error was issued after the dbproc was assigned but before
	    // we were able to associate our srvproc.
	    //
	    srvproc = Newsrvproc;
	}
    }

    //
    //	Send error message to client.
    //
    srv_sendmsg( srvproc,
		 SRV_MSG_ERROR,
		 (DBINT)REMOTE_MSG,
		 (DBTINYINT)severity,
		 (DBTINYINT)0,
		 NULL,
		 0,
		 0,
		 dberrstr,
		 SRV_NULLTERM );

    if (oserr != DBNOERR) {
	srv_sendmsg(srvproc,
		    SRV_MSG_ERROR,
		    (DBINT)REMOTE_MSG,
		    (DBTINYINT)severity,
		    (DBTINYINT)0,
		    NULL,
		    0,
		    0,
		    oserrstr,
		    SRV_NULLTERM );
    }

    return (INT_CANCEL);
}


//  ATTN_HANDLER
//	This is an event handler that will be called when the GDK receives
//	an attention from a client.
//
//  Parameters:
//	srvproc - Pointer to the client connection structure
//		  receiving the attention.
//
//  Returns:
//	SRV_CONTINUE
//
int attn_handler(srvproc)
SRV_PROC    FAR *srvproc;
{
    DBPROCESS	FAR *rmtproc;	// our DBPROCESS pointer

    rmtproc = ((REMOTE_DBMS *) srv_getuserdata(srvproc))->dbproc;
    dbcancel( rmtproc );
    return( SRV_CONTINUE );
}

#pragma check_stack()	// set stack checking to its default setting
