/*****************************************************************************
 * FILE: fs.c								     *
 *									     *
 * DESC:								     *
 *	- file system functions 					     *
 *	- sys_close, sys_dup, sys_sup2, sys_lseek			     *
 *	- msdos file operations 					     *
 *									     *
 * Copyright (C) 1993,1994						     *
 *	Rainer Schnitker, Heeper Str. 283, 33607 Bielefeld		     *
 *	email: rainer@mathematik.uni-bielefeld.de			     *
 *									     *
 *****************************************************************************/

#include "DPMI.H"
#include "PROCESS.H"
#include "DOSERRNO.H"
#include "CDOSX32.H"
#include "SIGNALS.H"
#include "RMLIB.H"
#include "PRINTF.H"
#include "FS.H"

#define NIL_FP (struct file *)0

struct file rsx_filetab[RSX_NFILES];


/*
** make entries for standard dos_files
*/
void init_rsx_filetab()
{
    rsx_filetab[0].f_mode = FMODE_READ;
    rsx_filetab[0].f_count = 1;
    rsx_filetab[0].f_doshandle = 0;
    rsx_filetab[0].f_op = & msdos_fop;

    rsx_filetab[1].f_mode = FMODE_WRITE;
    rsx_filetab[1].f_count = 1;
    rsx_filetab[1].f_doshandle = 1;
    rsx_filetab[1].f_op = & msdos_fop;

    rsx_filetab[2].f_mode = FMODE_WRITE;
    rsx_filetab[2].f_count = 1;
    rsx_filetab[2].f_doshandle = 2;
    rsx_filetab[2].f_op = & msdos_fop;

    rsx_filetab[3].f_mode = FMODE_WRITE;
    rsx_filetab[3].f_count = 1;
    rsx_filetab[3].f_doshandle = 3;
    rsx_filetab[3].f_op = & msdos_fop;

    rsx_filetab[4].f_mode = FMODE_WRITE;
    rsx_filetab[4].f_count = 1;
    rsx_filetab[4].f_doshandle = 4;
    rsx_filetab[4].f_op = & msdos_fop;
}

/*
** search for free file pointer
*/
int get_empty_proc_filp(void)
{
    int fd, i;

    /* search for entry in proc tab */
    for (fd = 0 ; fd < N_FILES ; fd++)
	if (!npz->filp[fd])
	    break;
    if (fd >= N_FILES)
	return -EMX_EMFILE;

    /* search for entry in rsx_filetab */
    for (i = 0; i < RSX_NFILES; i++)
	if (! rsx_filetab[i].f_count)
	    break;
    if (rsx_filetab[i].f_count)
	return -EMX_EMFILE;

    rsx_filetab[i].f_count = 1;
    rsx_filetab[i].f_mode = FMODE_READ | FMODE_WRITE;
    npz->filp[fd] = & rsx_filetab[i];
    return fd;
}

/*
** convert handle to real DOS file handle
*/
int get_dos_handle(int handle)
{
    if (handle >= N_FILES || !npz->filp[handle])
	return -1;
    else if (npz->filp[handle]->f_doshandle == -1)
	return -2;
    else
	return npz->filp[handle]->f_doshandle;
}

/*
** close a file handle
*/
int sys_close(int fd)
{
    struct file * filp;

    if (fd >= N_FILES)
	return -EMX_EBADF;
    if (!(filp = npz->filp[fd]))
	return -EMX_EBADF;
    npz->filp[fd] = NIL_FP;

    if (filp->f_count == 0) {
	printf("FS error: file %d count is 0\n", fd);
	return 0;
    }
    if (filp->f_count > 1) {
	filp->f_count--;
	return 0;
    }
    if (filp->f_op && filp->f_op->release)
	filp->f_op->release(filp);
    filp->f_count--;
    return 0;
}

/*
** dup a file handle
*/
static int dupfd(unsigned int fd, unsigned int arg)
{
    if (fd >= N_FILES || !npz->filp[fd])
	return -EMX_EBADF;
    if (arg >= N_FILES)
	return -EMX_EINVAL;
    while (arg < N_FILES)
	if (npz->filp[arg])
	    arg++;
	else
	    break;
    if (arg >= N_FILES)
	    return -EMX_EMFILE;
    npz->filp[arg] = npz->filp[fd];
    npz->filp[arg]->f_count++;
    return arg;
}

int sys_dup2(unsigned int oldfd, unsigned int newfd)
{
    if (oldfd >= N_FILES || !npz->filp[oldfd])
	return -EMX_EBADF;
    if (newfd == oldfd)
	return newfd;
    if (newfd > N_FILES)
	return -EMX_EBADF;
    if (newfd == N_FILES)
	return -EMX_EBADF;
    sys_close(newfd);
    return dupfd(oldfd, newfd);
}

int sys_dup(unsigned int fildes)
{
    return dupfd(fildes,0);
}

/*
** read from file handle
*/
ARGUSER sys_read(int fd, ARGUSER buf, ARGUSER bytes)
{
    struct file * file;

    if (fd >= N_FILES)
	return -EMX_EBADF;
    if (!(file = npz->filp[fd]))
	return -EMX_EBADF;
    if (! file->f_op || ! file->f_op->read)
	return -EMX_EBADF;
    if (verify_illegal(npz, buf, bytes)) {
	send_signal(npz, SIGSEGV);
	return -EMX_EINTR;
    }
    return file->f_op->read(file, buf, bytes);
}

/*
** write to file handle
*/
ARGUSER sys_write(int fd, ARGUSER buf, ARGUSER bytes)
{
    struct file * file;

    if (fd >= N_FILES)
	return -EMX_EBADF;
    if (!(file = npz->filp[fd]))
	return -EMX_EBADF;
    if (! file->f_op || ! file->f_op->write)
	return -EMX_EBADF;
    if (verify_illegal(npz, buf, bytes)) {
	send_signal(npz, SIGSEGV);
	return -EMX_EINTR;
    }
    return file->f_op->write(file, buf, bytes);
}

/*
** seek file handle
*/
long sys_lseek(int fd, long off, int orgin)
{
    struct file * file;

    if (fd >= N_FILES)
	return -EMX_EBADF;
    if (!(file = npz->filp[fd]))
	return -EMX_EBADF;
    if (! file->f_op || ! file->f_op->lseek)
	return -EMX_EBADF;
    return file->f_op->lseek(file, off, orgin);
}

/* ----------------------------------------------------------------- */

/*
** MS-DOS
** fileoperations
*/

static long msdos_lseek(struct file *filp, long off, int orgin)
{
    return rm_lseek(filp->f_doshandle, off, orgin);
}

static ARGUSER msdos_read(struct file *filp, ARGUSER buf, ARGUSER bytes)
{
    return cdosx_read(filp->f_doshandle, buf, bytes);
}

static ARGUSER msdos_write(struct file *filp, ARGUSER buf, ARGUSER bytes)
{
    return cdosx_write(filp->f_doshandle, buf, bytes);
}

static void msdos_release(struct file *filp)
{
    if (filp->f_doshandle <= 4)
	printf("FS error: closing doshandle %d\n", filp->f_doshandle);
    else
	rm_close(filp->f_doshandle);
}

struct file_operations msdos_fop =  {
    msdos_lseek,
    msdos_read,
    msdos_write,
    NULL,
    NULL,
    NULL,
    msdos_release
};
