/*****************************************************************************
 * FILE: libc.c                                                              *
 *									     *
 * DESC:								     *
 *      - kernel libc functions                                              *
 *									     *
 * Copyright (C) 1993,1994						     *
 *	Rainer Schnitker, Heeper Str. 283, 33607 Bielefeld		     *
 *	email: rainer@mathematik.uni-bielefeld.de			     *
 *									     *
 *****************************************************************************/

#include <string.h>
#include <malloc.h>
#include <sys\types.h>

#if !defined (NULL)
#define NULL ((void *)0)
#endif

#if !defined (_SIZE_T)
#define _SIZE_T
typedef unsigned long size_t;
#endif

unsigned errno;
unsigned _doserrno;
unsigned _psp;

size_t strlen(__const__ char *s)
{
    register int __res __asm__("cx");
    __asm__("cld\n\t"
	"repne\n\t"
	"scasb\n\t"
	"notl %0\n\t"
	"decl %0"
	:"=c" (__res):"D" (s),"a" (0),"0" (0xffffffff):"di");
    return __res;
}

char * strcat(char *dest, __const__ char *src)
{
    __asm__("cld\n\t"
	"repne\n\t"
	"scasb\n\t"
	"decl %1\n"
	"1:\tlodsb\n\t"
	"stosb\n\t"
	"testb %%al,%%al\n\t"
	"jne 1b"
	: /* no output */
	:"S" (src),"D" (dest),"a" (0),"c" (0xffffffff):"si","di","ax","cx");
    return dest;
}

int strcmp(__const__ char *cs, __const__ char *ct)
{
    register int __res __asm__("ax");
    __asm__("cld\n"
	"1:\tlodsb\n\t"
	"scasb\n\t"
	"jne 2f\n\t"
	"testb %%al,%%al\n\t"
	"jne 1b\n\t"
	"xorl %%eax,%%eax\n\t"
	"jmp 3f\n"
	"2:\tmovl $1,%%eax\n\t"
	"jb 3f\n\t"
	"negl %%eax\n"
	"3:"
	:"=a" (__res):"D" (cs),"S" (ct):"si","di");
    return __res;
}

char *strcpy(char *dest, __const__ char *src)
{
    __asm__("cld\n"
	"1:\tlodsb\n\t"
	"stosb\n\t"
	"testb %%al,%%al\n\t"
	"jne 1b"
	: /* no output */
	:"S" (src),"D" (dest):"si","di","ax","memory");
    return dest;
}

int memcmp(__const__ void *cs, __const__ void *ct, size_t count)
{
    register int __res __asm__("ax");
    __asm__("cld\n\t"
	"repe\n\t"
	"cmpsb\n\t"
	"je 1f\n\t"
	"movl $1,%%eax\n\t"
	"jb 1f\n\t"
	"negl %%eax\n"
	"1:"
	:"=a" (__res):"0" (0),"D" (cs),"S" (ct),"c" (count)
	:"si","di","cx");
    return __res;
}

extern char **environ;
char *getenv(char *name)
{
    int len;
    char **p, *s;

    if (name == NULL || environ == NULL)
	return (NULL);
    len = strlen(name);
    for (p = environ; *p != NULL; ++p) {
	s = *p;
	if (strlen(s) > len && s[len] == '=' && (memcmp(name, s, len) == 0))
	    return (s + len + 1);
    }
    return (NULL);
}

void *memset(void *s, int c, size_t count)
{
    __asm__(
	"cld\n\t"
	"rep\n\t"
	"stosb"
	: /* no output */
	:"a" (c),"D" (s),"c" (count)
	:"cx","di","memory");
    return s;
}

void *memcpy(void *to, __const__ void *from, size_t n)
{
    __asm__(
	"cld\n\t"
	"movl %%edx, %%ecx\n\t"
	"shrl $2,%%ecx\n\t"
	"rep ; movsl\n\t"
	"testb $1,%%dl\n\t"
	"je 1f\n\t"
	"movsb\n"
	"1:\ttestb $2,%%dl\n\t"
	"je 2f\n\t"
	"movsw\n"
	"2:\n"
	: /* no output */
	:"d" (n),"D" ((long) to),"S" ((long) from)
	: "cx","di","si","memory");
    return (to);
}

static inline void bzero(void *p, unsigned size)
{
    __asm__ __volatile__(
	"cld ; rep ; stosl"
	:
	:"a"(0), "D"((long) p), "c"(size >> 2)
	:"di", "cx");
}

void *sbrk(int bytes)
{
    static brk_value = 0x1000;
    static brk_max = 0xA000;
    int retv;

    if (bytes > 0xFFFF)
	return (void *) -1;
    if (brk_value + bytes > brk_max)
	return (void *) -1;

    retv = brk_value;
    brk_value += bytes;
    return (void *) retv;
}

#define MAGIC (Header *)1111

typedef long Align;
union header {
    struct {
	union header *ptr;
	unsigned size;
    } s;
    Align x;
};
typedef union header Header;

static Header base;
static Header *freep = NULL;

void *malloc(size_t nbytes)
{
    Header *p, *prevp;
    static Header *morecore(unsigned);
    unsigned nunits;

    nunits = (nbytes + sizeof(Header) - 1) / sizeof(Header) + 1;

    if ((prevp = freep) == NULL) {
	base.s.ptr = freep = prevp = &base;
	base.s.size = 0;
    }
    for (p = prevp->s.ptr;; prevp = p, p = p->s.ptr) {
	if (p->s.size >= nunits) {
	    if (p->s.size == nunits)
		prevp->s.ptr = p->s.ptr;
	    else {
		p->s.size -= nunits;
		p += p->s.size;
		p->s.size = nunits;
	    }
	    freep = prevp;

	    p->s.ptr = MAGIC;


	    return (void *) (p + 1);
	}
	if (p == freep)
	    if ((p = morecore(nunits)) == NULL)
		return NULL;
    }
}

#define NALLOC 128		/* 128 * Header = 1 KB */

static Header *morecore(unsigned nu)
{
    void *cp, *sbrk(int);
    Header *up;

    if (nu < NALLOC)
	nu = NALLOC;
    cp = sbrk(nu * sizeof(Header));
    if (cp == (void *) -1)
	return NULL;
    up = (Header *) cp;
    up->s.size = nu;
    up->s.ptr = MAGIC;
    free((void *) (up + 1));
    return freep;
}

void free(void *ap)
{
    Header *bp, *p;

    bp = (Header *) ap - 1;

    if (bp->s.ptr != MAGIC)
	return;
    for (p = freep; !(bp > p && bp < p->s.ptr); p = p->s.ptr)
	if (p >= p->s.ptr && (bp > p || bp < p->s.ptr))
	    break;

    if (bp + bp->s.size == p->s.ptr) {
	bp->s.size += p->s.ptr->s.size;
	bp->s.ptr = p->s.ptr->s.ptr;
    } else
	bp->s.ptr = p->s.ptr;
    if (p + p->s.size == bp) {
	p->s.size += bp->s.size;
	p->s.ptr = bp->s.ptr;
    } else
	p->s.ptr = bp;
    freep = p;
}

void *realloc(void *oldp, size_t newsize)
{
    void *newp;
    Header *h;
    unsigned oldsize;

    if (oldp == NULL)
	return (malloc(newsize));

    h = (Header *) oldp - 1;
    if (h->s.ptr != MAGIC)
	return (NULL);

    oldsize = h->s.size * sizeof(Header) - sizeof(Header);

    free(oldp);
    newp = malloc(newsize);
    if (newp == NULL)
	return NULL;
    if (newp != oldp)
	memcpy(newp, oldp, (newsize < oldsize) ? newsize : oldsize);
    return (newp);
}

void *calloc(size_t nelem, size_t size)
{
    void *p = malloc(nelem * size);

    if (p)
	bzero(p, nelem * size);
    return p;
}
