/*****************************************************************************
 * FILE: loadprg.c							     *
 *									     *
 * DESC:								     *
 *	- loader for a.out programs					     *
 *	- set args,environment						     *
 *	- load text,data (if not DPMI 1.0)				     *
 *									     *
 * Copyright (C) 1993,1994						     *
 *	Rainer Schnitker, Heeper Str. 283, 33607 Bielefeld		     *
 *	email: rainer@mathematik.uni-bielefeld.de			     *
 *									     *
 *****************************************************************************/

#include <string.h>
#include "DPMI.H"
#include "DPMI10.H"
#include "RMLIB.H"
#include "PROCESS.H"
#include "FS.H"
#include "GNUAOUT.H"
#include "LOADPRG.H"
#include "COPY32.H"
#include "START32.H"
#include "CDOSX32.H"
#include "RSX.H"
#include "DOSERRNO.H"

struct emx_hdr2 {
    char sig[18];
    char next_hdr[4];
};
int skip_exe_hdr(int filehandle, DWORD * headoff)
{
    struct exe_hdr exehdr;
    struct emx_hdr2 emxhdr;

    rm_read(filehandle, &exehdr, sizeof(struct exe_hdr));

    if (exehdr.signatur == 0x5a4d) {	/* falls exe-kopf */
	*headoff = ((DWORD) exehdr.hdr_para) * 16;
	if (rm_lseek(filehandle, *headoff, SEEK_SET) == -1L)
	    goto fail;
	rm_read(filehandle, &emxhdr, sizeof(struct emx_hdr));
	if (memcmp(emxhdr.sig, "emx", 3) == 0)
	    *headoff = *(DWORD *) emxhdr.next_hdr;
	else {
	    *headoff = (DWORD) exehdr.high * 512L;
	    if (exehdr.low)
		*headoff += (DWORD) exehdr.low - 512L;
	}
    }
    if (rm_lseek(filehandle, *headoff, SEEK_SET) == -1L)
	goto fail;
    return 0;

  fail:
    *headoff = 0;
    rm_lseek(filehandle, 0, SEEK_SET);
    return -1;
}

/*
** arguments and environment at process start
**
**	environment and argument strings
**	argv[]
**	envp[]
**	pointer to env[0]
**	pointer to argv[0]
**	int argc
**	<- ESP
*/
int argvenv(int argc, char **argv, int envc, char **env, NEWPROCESS * proc)
{
    int i;
    DWORD len, stkp;
    DWORD *vectors;
    UINT count = 3;		/* 0=argc 1=argv 2=env */

    vectors = (DWORD *) iobuf;

    stkp = proc->stackp32;

    /* store env strings in user stack, built vectors */
    for (i = 0; i < envc; i++) {
	len = (DWORD) (UINT) strlen(env[i]) + 1;
	stkp -= len;
	stkp &= ~3L;		/* align4 */
	cpy16_32(proc->data32sel, stkp, env[i], len);
	vectors[count++] = stkp;
    }
    vectors[count++] = 0;	/* last is a NULL pointer */

    /* store arg strings in user stack, built vectors */

    /* *** (old)
    for (i = 0; i < argc; i++) {
	len = (DWORD) (UINT) strlen(argv[i]) + 1;
	stkp -= len;
	stkp &= ~3L;
	cpy16_32(proc->data32sel, stkp, argv[i], len);
	vectors[count] = stkp;
	count++;
    }  *** */

    for (i = 0; i < argc; i++) {
	len = (DWORD) (UINT) strlen(argv[i]) + 1;
	stkp -= len;
	stkp &= ~3L;
	cpy16_32(proc->data32sel, stkp, argv[i], len);
	vectors[count] = stkp;
	count++;
	stkp -= 1;
	put_user_byte(proc->data32sel, stkp, 0x80);
    }
    vectors[count++] = 0;	/* last is a NULL pointer */

    stkp &= ~3L;
    len = (DWORD) (count * sizeof(long));
    stkp -= len;
    vectors[0] = argc;
    vectors[1] = stkp + (4L + envc) * sizeof(long);	/* & vectors[3+nenvp+1] */
    vectors[2] = stkp + 3 * sizeof(long);	/* & vectors[3] */
    cpy16_32(proc->data32sel, stkp, vectors, len);

    if (proc->p_flags & PF_EMX_FILE)
	stkp += 3 * 4;

    proc->stackp32 = stkp;
    return 0;
}

void cpy_exename_to_stack(NEWPROCESS *proc, char *name)
{
    int len = strlen(name) + 1;
    long name_pos = proc->stackp32 - (long) (len + 2 * sizeof(DWORD));

    proc->stackp32 -= sizeof(DWORD);
    store32(proc->data32sel, proc->stackp32, name_pos);
    proc->stackp32 -= sizeof(DWORD);
    store32(proc->data32sel, proc->stackp32, (long) len);
    proc->stackp32 -= (long) len;
    cpy16_32(proc->data32sel, proc->stackp32, name, (long) len);
}

int readin_file(short handle, short r_ds, long r_edx, long count)
{
    long bytes;
    int rc;

    while (count > 0) {
	bytes = (count <= IOBUF_SIZE) ? count : (DWORD) IOBUF_SIZE;

	rc = rm_read(handle, iobuf, (short) bytes);
	cpy16_32(r_ds, r_edx, iobuf, rc);
	if (bytes != rc)
	    break;
	count -= bytes;
	r_edx += bytes;
    }
    return 0;
}

/*
**
** RSX program layout:
**
**
**  DPMI 0.9 :	 fixed stack
**		 never ending heap
**
**  emx style
**  |--------------------------------------------------------------
**  |stack|    code    |  data/bss  |  stack,if>64KB |	heap -> ...
**  |--------------------------------------------------------------
**  0	  ^64 KB       ^(n x 64KB)
**
**
**  old djgpp style
**  |--------------------------------------------------------------
**  |	|    code	 | stack    |  data/bss  | heap -> ...
**  |--------------------------------------------------------------
**  0	^4K			    ^0x400000
**
**  djgpp style 1.11
**  |--------------------------------------------------------------
**  |  code	|  data/bss  | stack  | heap -> ...
**  |--------------------------------------------------------------
**  0		^ 4 Kb align ^
**
**
**
**  DPMI 1.0 :	 address room = 64 MegaBytes
**
**  |--------------------------------------------...-----------------------|
**  |	  |  code   |  |  data/bss  |  heap ->	      <- stack	| mappings |
**  |	  |	    |  |	    |				| DOS 1MB  |
**  |--------------------------------------------...-----------------------|
**  0								^60	   64
**
*/

#define DEFAULT_STACK (0x10000L-0x1000L)	/* 64 KB - one R/O page */

int load_protected_program(char *filename, NEWPROCESS * proc)
{
    GNUOUT aout_hdr;
    DWORD segment_size, n_txtaddr, _n_hdroff;
    DWORD textsegm, datasegm, stacksegm;
    DWORD headoff;
    UINT stack32sel;
    int fhandle;
    int use_dpmi10;

    if ((fhandle = rm_open(filename, RM_O_RDONLY | RM_O_DENYWR)) == -1)
	return doserror_to_errno(_doserrno);

    /* skip exe-header, return correct offset in file */
    headoff = 0;
    skip_exe_hdr(fhandle, &headoff);

    /* read gnu aout header */
    rm_read(fhandle, &aout_hdr, sizeof(aout_hdr));

    /* test header */
    if ((WORD) aout_hdr.a_info == 0x14C) {
	rm_lseek(fhandle, headoff + 20, 0);
	rm_read(fhandle, &aout_hdr, sizeof(aout_hdr));
	aout_hdr.a_entry = aout_hdr.a_syms;
    }
    if ((WORD) aout_hdr.a_info != 0x10b) {
	rm_close(fhandle);
	return EMX_ENOEXEC;
    }
    /* dynamic a.out loading for emx or dj files */
    if (aout_hdr.a_entry == 0x10000) {
	proc->p_flags |= PF_EMX_FILE;
	segment_size = 0x10000L;
	_n_hdroff = (1024 - sizeof(struct exec));
	n_txtaddr = 0x10000L;
    } else if (aout_hdr.a_entry == 0x1020) {
	proc->p_flags |= PF_DJGPP_FILE;
	segment_size = 0x400000L;
	_n_hdroff = 0;
	n_txtaddr = 0x1020L;
    } else if (aout_hdr.a_entry == 0x10A8) {
	proc->p_flags |= PF_DJGPP_FILE;
	segment_size = 0x1000L;
	_n_hdroff = 16 + 3 * 40;/* add FILEHDR and SCNHDR */
	n_txtaddr = 0x10A8L;
    } else {
	rm_close(fhandle);
	return EMX_ENOEXEC;
    }

    proc->filehandle = (long) fhandle;
    proc->text_off = headoff + N_TXTOFF(aout_hdr);
    proc->data_off = headoff + N_DATOFF(aout_hdr);
    if (proc->p_flags & PF_DJGPP_FILE)
	proc->text_off = headoff;	/* align */

    /* compute size of text, data & stack segment for AllocMem() */
    textsegm = (aout_hdr.a_entry + aout_hdr.a_text + SEGMENT_SIZE - 1L)
	& ~(SEGMENT_SIZE - 1L);
    datasegm = (aout_hdr.a_data + aout_hdr.a_bss + 4095L) & ~4095L;
    if (opt_stackval)
	stacksegm = (DWORD) opt_stackval *1024;
    else
	stacksegm = DEFAULT_STACK;

    if (proc == &RSX_PROCESS) { /* emu don't need much stack */
	stacksegm = DEFAULT_STACK;
	use_dpmi10 = 0;
    } else
	use_dpmi10 = (dpmi10) ? 1 : 0;

    if (use_dpmi10)
	proc->p_flags |= PF_USEDPMI10;

    /* define start-, endaddresses */
    proc->text_start = N_TXTADDR(aout_hdr) & ~0xFFFL;
    proc->text_end = (N_TXTADDR(aout_hdr) + aout_hdr.a_text + 4095L) & ~4095L;
    proc->data_start = N_DATADDR(aout_hdr);
    proc->data_end = N_DATADDR(aout_hdr) + datasegm;
    proc->bss_start = proc->data_start + aout_hdr.a_data;
    proc->bss_end = proc->data_end;
    proc->entry = aout_hdr.a_entry;

    /* look for empty space for stack */
    if (aout_hdr.a_entry == 0x10000L && stacksegm <= DEFAULT_STACK) {
	stacksegm = 0;
	/* place stack in the first 64KB segment */
	proc->stack_top = proc->text_start;
	proc->stack_down = 0x1000L;
    } else if (aout_hdr.a_entry == 0x1020L
	       && stacksegm <= (proc->data_start - proc->text_end)) {
	stacksegm = 0;
	/* place stack between code and data */
	proc->stack_top = proc->data_start;
	proc->stack_down = proc->text_end + 0x1000;
    } else {
	/* put stack after data/bss */
	proc->stack_top = proc->data_end + stacksegm;
	proc->stack_down = proc->data_end;

	/* sorry, work around window bug */
	if ((proc->p_flags & PF_DJGPP_FILE) && proc->stack_down <= 0x11000L
	    && proc != &RSX_PROCESS) {
	    DWORD add_stack = 0x11000L - proc->stack_down;
	    proc->stack_top += add_stack;
	    proc->stack_down += add_stack;
	    stacksegm += add_stack;
	}
    }

    if (use_dpmi10) {
	stacksegm = 0;
	proc->stack_top = DPMI_PRG_DATA;
	proc->stack_down = proc->data_end;
    }
    proc->stacksize = proc->stack_top - proc->stack_down;
    proc->stackp32 = proc->stack_top - 4L;

    proc->init_brk = proc->data_end + stacksegm;
    proc->brk_value = proc->init_brk;
    proc->pagefree = 0;

    proc->membytes = datasegm + textsegm + stacksegm;

    if (use_dpmi10) {
	proc->pagefree = DPMI_PRG_DATA - proc->membytes;
	proc->membytes = DPMI_PRG_ROOM;
    }
    /* enable first two emx sbrk calls, if heap after bss */
    if (proc->init_brk == proc->data_end &&
	(proc->data_start + aout_hdr.a_data + aout_hdr.a_bss + 8)
	<= proc->data_end) {
	proc->init_brk -= 8;
	proc->brk_value -= 8;
	proc->pagefree += 8;
    }
    /* since dosmem isn't used by DPMI we put rsx387 in dos memory */
    if (proc == &RSX_PROCESS && rsx387_in_dosmem) {
	UINT segment, selectors;
	if (AllocDosMem((UINT) (proc->membytes >> 4), &segment, &selectors))
	    return EMX_ENOMEM;
	proc->memaddress = (DWORD) segment << 4;
	proc->memhandle = (DWORD) selectors;
    }
    /* get memory from DPMI-host */
    else if (!use_dpmi10) {
	if (AllocMem(proc->membytes, &(proc->memhandle), &(proc->memaddress)))
	    return EMX_ENOMEM;
    } else if (AllocLinearMemory(proc->membytes, 0L, 0L, &proc->memhandle, &proc->memaddress))
	return EMX_ENOMEM;

    /* alloc 32bit cs,ds ldt selector */
    if (AllocLDT(3, &(proc->code32sel)))
	return EMX_EIO;
    proc->data32sel = proc->code32sel + sel_incr;
    stack32sel = proc->data32sel + sel_incr;

    /* fill descriptors */
    SetBaseAddress(proc->code32sel, proc->memaddress);
    SetBaseAddress(proc->data32sel, proc->memaddress);
    SetBaseAddress(stack32sel, proc->memaddress);
    SetAccess(proc->code32sel, APP_CODE_SEL, DEFAULT_BIT);
    SetAccess(proc->data32sel, APP_DATA_SEL, BIG_BIT);
    SetAccess(stack32sel, APP_DATA_SEL | EXPAND_BIT, BIG_BIT | GRANULAR_BIT);

    /* allow execute data & stack  (DJGPP / GDB? need this) */
    SetLimit(proc->code32sel, proc->membytes - 1L);

    if (!use_dpmi10) {
	if (opt_memaccess)
	    SetLimit(proc->data32sel, 0xFFFFFFFF);
	else
	    SetLimit(proc->data32sel, proc->membytes - 1L);
    } else
	SetLimit(proc->data32sel, DPMI_PRG_ROOM - 1L);

    /* set stack expand down segment to first stack address */
    SetLimit(stack32sel, proc->stack_down - 1L);

    /* map first DOS MB at the end of linear Addressroom (60MB) */
    if (opt_memaccess && use_dpmi10)
	MapDOSMemInMemoryBlock(proc->memhandle, DPMI_PRG_DATA, 256L, 0L);

    if (!use_dpmi10) {
	/* read in code */
	rm_lseek(fhandle, headoff + N_TXTOFF(aout_hdr), SEEK_SET);
	readin_file(fhandle, proc->data32sel, N_TXTADDR(aout_hdr), aout_hdr.a_text);

	/* read in data,bss */
	rm_lseek(fhandle, headoff + N_DATOFF(aout_hdr), SEEK_SET);
	readin_file(fhandle, proc->data32sel, N_DATADDR(aout_hdr), aout_hdr.a_data);
	rm_close(fhandle);

	/* zero bss segment */
	if (aout_hdr.a_bss)
	    bzero32(proc->data32sel, N_BSSADDR(aout_hdr), aout_hdr.a_bss);
    } else {			/* uncommit first stack page! */
	WORD page = 9;
	ModifyPageAttributes(proc->memhandle, proc->stack_top - 4096, 1, &page);
    }
    return 0;
}
