//
// Finger Version 1.0, a Windows Sockets Finger Client
//
// Copyright (C) 1994 by Zoran Dukic.
//
// Permission to use, modify, and distribute this software and its
// documentation for any purpose and without fee is hereby granted, provided
// that the above copyright notice appears in all copies and that both
// that copyright notice and this permission notice appear in supporting
// documentation.  Zoran Dukic makes no claims as to the suitability of this 
// software for any purpose.
//
// finger.cpp      Finger's user interface.  This module processes all user
//      	   input, and displays query results and errors.
// netwrkm.cpp     Finger's network module.  It isolates the network
//	           interface from the rest of the program, and uses 
//	           asynchronous WS calls to query the remote host.
//
// Module NETWRKM is reconstruction of the same module from the Finger 3.1, 
// for C++, so in netwrkm.cpp is the copyright notice for the Finger 3.1.



#include "finger.h"

char     szAppName[] = "Finger";    // application's name

ERRENTRY wsErrs[] =                 // error text for windows sockets errors
{
   WSAVERNOTSUPPORTED,  "This version of Windows Sockets is not supported",
   WSASYSNOTREADY,      "Windows Sockets is not present or is not responding",
};

ERRENTRY finErrs[] =                // finger specific error text
{
   FE_NOPORT,  "Cannot locate port for finger service",
   FE_NOHOST,  "Unrecognized host name",
   FE_NOSOCK,  "Cannot obtain socket for connection",
   FE_NOCONN,  "Cannot connect to remote server",
   FE_NOSEND,  "Cannot send query to remote server",
   FE_NORECV,  "Error occurred during retrieval"
};




/********************************************
 * Function Definition for TFingerApp class *
 ********************************************/

void TFingerApp::InitMainWindow()
{
   MainWindow = new TFingerWnd(NULL);
}

/**************************************************************************/


/******************************************
 * Function Definition for THostDlg class *
 ******************************************/

THostDlg::THostDlg(PTWindowsObject AParent, LPSTR AName, PTModule)
	    :TDialog(AParent, AName)
{  
     pHostCombo = new TComboBox(this, ID_HOSTNAME, 41);
     new TEdit(this, ID_USER, 41);

     EnableKBHandler();
     TransferBuffer = (void far*)&(((PTFingerWnd)AParent)->HostTransfer);
}


void THostDlg::WMControlColor(TMessage& Msg)
{
  switch (Msg.LP.Hi) {
    case CTLCOLOR_BTN:
        Msg.Result = (LRESULT)GetStockObject(NULL_BRUSH);
	break;
    case CTLCOLOR_LISTBOX:
	Msg.Result = (LRESULT)GetStockObject(LTGRAY_BRUSH);
	break;
    case CTLCOLOR_STATIC:
	SetBkMode((HDC)Msg.WParam, TRANSPARENT);
	Msg.Result = (LRESULT)GetStockObject(LTGRAY_BRUSH);
        break;      
    case CTLCOLOR_DLG:
	SetBkMode((HDC)Msg.WParam, TRANSPARENT);
	Msg.Result = (LRESULT)GetStockObject(LTGRAY_BRUSH);;
        break;
    default:
      DefWndProc(Msg);
  }
}


void THostDlg::HandleFingerMsg(RTMessage)
{
     TransferData(TF_GETDATA);

     ((PTFingerWnd)Parent)->FingerStart();
}


void THostDlg::Cancel(RTMessage)
{
      Parent->CloseWindow();
}


void THostDlg::HandleCloseMsg(RTMessage)
{
      Parent->CloseWindow();
}


void THostDlg::HandleAboutMsg(RTMessage)
{
     GetApplication()->ExecDialog(new TDialog(Parent,"ABOUTBOX"));
     SetFocus(GetItemHandle(ID_ABOUT));
}


void THostDlg::HandleDeleteMsg(RTMessage)
{
      int i;
      char temString[50];

      pHostCombo->GetText(temString, 50);
      if((i = pHostCombo->FindExactString(temString, -1)) >= 0)
      {
	       pHostCombo->DeleteString(i);
	       TransferData(TF_GETDATA);
      }
}


void THostDlg::HandleSaveMsg(RTMessage)
{
      char temString[50];

      pHostCombo->GetText(temString, 50);
      if(((pHostCombo->FindExactString(temString, -1)) < 0) && (strlen(temString) > 0))
      {
	       pHostCombo->AddString(temString);
	       TransferData(TF_GETDATA);
      }
}


void THostDlg::WMMouseMove(RTMessage)
{
   SetCursor(((PTFingerWnd)Parent)->hCursor);
}

/************************************************************************/


/********************************************
 * Function Definition for TOutputWnd class *
 ********************************************/

TOutputWnd::TOutputWnd(PTWindowsObject AParent)
	   :TWindow(AParent, "Output")
{
    Attr.Style = WS_BORDER | WS_VSCROLL | WS_HSCROLL | WS_CHILD;

    Scroller = new TScroller(this, 10, 10 , 10, 10);
    Scroller->TrackMode = FALSE;
    Scroller->AutoMode = FALSE;

    pViewList = NULL;
    nLineItems = 0;
    maxLen = 0;
}


void TOutputWnd::Paint(HDC PaintDC, PAINTSTRUCT&)
{
     int i;
     PTLine pline;

     SelectObject(PaintDC, GetStockObject(SYSTEM_FIXED_FONT));

     for (i = 0; i < nLineItems; i++)
     {
        if ( (pline = (PTLine)&((*pViewList)[i])) != NULL)
        {
	    TextOut(PaintDC, 0, i * (Scroller->YUnit), pline->sztext, pline->LLen);
        }
        else
            break;
     } 
}


//
// SetRange -- sets the vertical scroll range to the length of the display
// list.  The Scrollbar disappears when the list fits within the view.
//
void TOutputWnd::SetRange()
{
   if (pViewList != NULL)nLineItems = pViewList->getItemsInContainer();

   if (nLineItems > nClientLines)
   {   
      if (maxLen > nClientLen) Scroller->SetRange(maxLen - nClientLen, nLineItems - nClientLines);
      else Scroller->SetRange(0, nLineItems - nClientLines);
   }
   else
   {
      if (maxLen > nClientLen) Scroller->SetRange(maxLen - nClientLen, 0);
      else Scroller->SetRange(0, 0);
   }
}


void TOutputWnd::WMSize(RTMessage Msg)
{
  TWindow::WMSize(Msg);
  nClientLines = HIWORD(Msg.LParam) / (Scroller->YUnit);
  nClientLen = LOWORD(Msg.LParam) / (Scroller->XUnit) - 1;
  SetRange();
}


void TOutputWnd::WMMouseMove(RTMessage)
{
   SetCursor(((PTFingerWnd)Parent)->hCursor);
}

/************************************************************************/


/********************************************
 * Function Definition for TFingerWnd class *
 ********************************************/

TFingerWnd::TFingerWnd(PTWindowsObject AParent)
	   :TWindow(AParent, "Finger")
{
     Attr.Style = WS_OVERLAPPED | WS_VISIBLE | WS_SYSMENU | WS_MINIMIZEBOX;
     Attr.X = 20;
     Attr.Y = 20;

     pOutput = new TOutputWnd(this);
     pHostDlg = new THostDlg(this, "HOSTDLG");
     pNetWnd = new TNetWnd(this);

     memset(&HostTransfer, 0, sizeof(THostTransfer));
     HostTransfer.pHost = new TComboBoxData();

     ReadHosts();
}


TFingerWnd::~TFingerWnd()
{
    if(HostTransfer.pHost != NULL)delete HostTransfer.pHost;
};


void TFingerWnd::SetupWindow()
{
   HDC hdc;                            // handle of device context
   TEXTMETRIC tm;                      // contains font dimensions
   RECT rect, rectHost;                // outer dimensions of window
   
   TWindow::SetupWindow();

   hCursor = LoadCursor(NULL, IDC_ARROW);

   hdc = GetDC(HWindow);

   SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));

   GetTextMetrics(hdc, &tm);
   CharY = tm.tmHeight + tm.tmExternalLeading;

   ReleaseDC(HWindow, hdc);

   (pOutput->Scroller)->SetUnits(tm.tmMaxCharWidth, CharY);

   GetApplication()->MakeWindow(pHostDlg);
   GetApplication()->MakeWindow(pOutput);
   GetApplication()->MakeWindow(pNetWnd);

   GetWindowRect(HWindow, &rect);
   GetWindowRect(pHostDlg->HWindow, &rectHost);

   MoveWindow( HWindow, rect.left, rect.top,
	       rectHost.right - rectHost.left,
	       rectHost.bottom - rectHost.top + 10 +
	       10 * CharY + GetSystemMetrics(SM_CYCAPTION),
	       FALSE);

   MoveWindow( pOutput->HWindow, 10, rectHost.bottom - rectHost.top,
	       rectHost.right - rectHost.left - 20,
	       10 * CharY, FALSE);

   ShowWindow( pHostDlg->HWindow, SW_SHOWNA);
   ShowWindow( pOutput->HWindow, SW_SHOWNA);

   GetApplication()->SetKBHandler(pHostDlg);
}


LPSTR TFingerWnd::GetClassName()
{
     return "TFingerWnd";
}


void TFingerWnd::GetWindowClass(WNDCLASS& AWndClass)
{
     TWindow::GetWindowClass(AWndClass);
     AWndClass.hIcon = LoadIcon(GetApplication()->hInstance,"ICON_1");
     AWndClass.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
}


BOOL TFingerWnd::CanClose()
{
     WriteHosts();
     return TRUE;
}


void TFingerWnd::ReadHosts()
{
     char szBuf[256], szHost[30], szEntry[10];
     int i=0;

     GetPrivateProfileString("hosts", NULL, "", szBuf, sizeof(szBuf), "finger.ini");

     while (szBuf[i] != '\0')
     {
         int j = 0;
	 while (szBuf[i] != '\0') szEntry[j++] = szBuf[i++];
	 i++;
	 GetPrivateProfileString("hosts", szEntry, "", szHost, sizeof(szHost), "finger.ini");
	 HostTransfer.pHost->AddString(szHost);
     }

     pHostDlg->TransferData(TF_SETDATA);
}


void TFingerWnd::WriteHosts()
{
     int i;
     String strHost;
     char szEntry[10], tem[5];

     WritePrivateProfileString("hosts", NULL, NULL, "finger.ini");

     for (i = 0; i < (HostTransfer.pHost->Strings)->getItemsInContainer(); i++)
     {
	if ( (strHost = (RString)(*(HostTransfer.pHost->Strings))[i]) != NULL)
	{
	    strcpy(szEntry, "host");
            strcat(szEntry, itoa(i, tem, 10));
	    WritePrivateProfileString("hosts", szEntry, strHost, "finger.ini");
        }
        else
            break;
     }
}


void TFingerWnd::Repaint()
{
   InvalidateRect(pOutput->HWindow, NULL, TRUE);
}


//
// SetWinCaption -- set the frame window caption according to last
// host fingered.
//
void TFingerWnd::SetWinCaption()
{
   char szcaption[80];
   
   strcpy(szcaption, szAppName);
   strcat(szcaption, " - ");
   strcat(szcaption, pNetWnd->szHostName);

   SetWindowText(HWindow, szcaption);
}


void TFingerWnd::FingerStart()
{
    if(((HostTransfer.pHost)->Selection) != NULL)
    {
           strcpy(pNetWnd->szHostName,(HostTransfer.pHost)->Selection);
           strcpy(pNetWnd->szUser,HostTransfer.pUser);

     	   SetCursor(hCursor = LoadCursor(NULL, IDC_WAIT));
	   EnableWindow(pHostDlg->GetItemHandle(ID_FINGER), FALSE);
	   pNetWnd->FingerStart();
    }
}


// FingerFinish -- invoked when the finger operation is complete,
// this function updates the display list & repaints the frame window
// client area if the operation was successful.
//
void TFingerWnd::FingerFinish(UINT Err)
{  
   if (!Err)
   {

      if(pOutput->pViewList != NULL)delete pOutput->pViewList;

      pOutput->pViewList = pNetWnd->pReceiveList;
      pOutput->maxLen = pNetWnd->maxLineLen;

      SetWinCaption();                          // set win title to host name
      pOutput->SetRange();                              // rescale (or delete)
      Repaint();                                // scrollbar & force a repaint
   }

   EnableWindow(pHostDlg->GetItemHandle(ID_FINGER), TRUE);
   SetFocus(pHostDlg->HWindow);
   SetCursor(hCursor = LoadCursor(NULL, IDC_ARROW));
}


void TFingerWnd::ReportFingerErr(UINT Err)
{
     
   int i;
   
   for (i = 0; i < dim(finErrs); i++)
   {
      if (Err == finErrs[i].err)
      {
         MessageBox(HWindow, finErrs[i].sztext, szAppName,
            MB_ICONSTOP | MB_OK);
         return;
      }
   }

   MessageBox(HWindow, "Unrecognized finger error", szAppName,
      MB_ICONSTOP | MB_OK);
   
};


//
// ReportWSError -- prompt user with a windows sockets error message.
//
void TFingerWnd::ReportWSError(UINT Err)
{
   int i;
   char szerr[40];

   for (i = 0; i < dim(wsErrs); i++)
   {
      if (Err == wsErrs[i].err)
      {
         MessageBox(HWindow, wsErrs[i].sztext, szAppName,
            MB_ICONSTOP | MB_OK);
         return;
      }
   }

   wsprintf(szerr, "Windows Sockets reports error %04x", Err);
   MessageBox(HWindow, szerr, szAppName, MB_ICONSTOP | MB_OK);
}


void TFingerWnd::ActivationResponse(WORD Activated, BOOL IsIconified)
{
     if(Activated && !IsIconified)
     {
         SetFocus(pHostDlg->HWindow);
         GetApplication()->SetKBHandler(pHostDlg);
     }
     else
	 GetApplication()->SetKBHandler(NULL);
};


void TFingerWnd::WMMouseMove(RTMessage)
{
   SetCursor(hCursor);
}


/**************************************************************************/


/************************
 * Main Program WinMain *
 ************************/

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
     TFingerApp FingerApp("Finger", hInstance, hPrevInstance, lpCmdLine, nCmdShow);

     FingerApp.Run();
     return FingerApp.Status;
}

/**************************************************************************/


