/****************************************************************************
 *  This file is part of PPMD project                                       *
 *  Written and distributed to public domain by Dmitry Shkarin 1997,        *
 *  1999-2000                                                               *
 *  Contents: main routine                                                  *
 *  Comments: system & compiler dependent file                              *
 ****************************************************************************/
#pragma option -O1
#include <io.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "ppmd.h"

const DWORD PPMDSignature=0x84ACAF8F, Variant='F';
static const char* const MTxt[] = { "Can`t open file %s",
    "read/write error for files %s/%s", "Out of memory!" };
static const char* pFName;
static DWORD StartFilePosition;
static struct ARC_INFO { // FileLength & CRC? Hmm, maybe in another times...
    DWORD signature,attrib;
    WORD  info,FNLen,time,date;
} ai;

#if defined(__WIN32_ENVIRONMENT)
#include <conio.h>

static const char* UsageStr="Usage: ppmd <e|d> [switches] <FileName[s] | Wildcard[s]>\n";
inline void EnvSetNormAttr(const char* FName) { SetFileAttributes(FName,FILE_ATTRIBUTE_NORMAL); }
inline int                         EnvGetCh() { return getch(); }
inline void           EnvGetCWD(char* CurDir) { GetCurrentDirectory(256,CurDir); }
inline void EnvSetDateTimeAttr(const char* WrkStr)
{
    FILETIME ft;
    DosDateTimeToFileTime(ai.date,ai.time,&ft);
    HANDLE hndl=CreateFile(WrkStr,GENERIC_WRITE,0,NULL,OPEN_EXISTING,0,0);
    SetFileTime(hndl,&ft,NULL,&ft);         CloseHandle(hndl);
    SetFileAttributes(WrkStr,ai.attrib);
}
struct ENV_FIND_RESULT {
    HANDLE hndl;
    FILETIME StartTime;
    WIN32_FIND_DATA fdata;
    BOOL isNotAllowed() {
        return ((fdata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0 ||
        CompareFileTime(&fdata.ftLastWriteTime,&StartTime) >= 0);
    }
    BOOL findFirst(const char* Pattern) {
        SYSTEMTIME st;
        GetSystemTime(&st);                 SystemTimeToFileTime(&st,&StartTime);
        return ((hndl=FindFirstFile(Pattern,&fdata)) != INVALID_HANDLE_VALUE);
    }
    BOOL findNext() { return FindNextFile(hndl,&fdata); }
    const char* getFName() { return fdata.cFileName; }
    void copyDateTimeAttr() {
        ai.attrib=fdata.dwFileAttributes;
        FileTimeToDosDateTime(&fdata.ftLastWriteTime,&ai.date,&ai.time);
    }
};

#elif defined(__DOS32_ENVIRONMENT)
#include <conio.h>
#include <dos.h>
#include <direct.h>
#if defined(__DJGPP__)
#include <unistd.h>
#include <crt0.h>
char **__crt0_glob_function (char *arg) { return 0; }
void   __crt0_load_environment_file (char *progname) { }
#endif /* defined(__DJGPP__) */

static const char* UsageStr="Usage: ppmd <e|d> [switches] <FileName[s] | Wildcard[s]>\n";
inline void EnvSetNormAttr(const char* FName) { _dos_setfileattr(FName,_A_NORMAL); }
inline int                         EnvGetCh() { return getch(); }
inline void           EnvGetCWD(char* CurDir) { getcwd(CurDir,256); }
inline void EnvSetDateTimeAttr(const char* WrkStr)
{
    FILE* fpOut = fopen(WrkStr,"a+b");
    _dos_setftime(fileno(fpOut),ai.date,ai.time);
    fclose(fpOut);
    _dos_setfileattr(WrkStr,ai.attrib);
}
struct ENV_FIND_RESULT {
    find_t ffblk;
    UINT StartDate, StartTime;
    BOOL isNotAllowed() { return (ffblk.wr_date > StartDate ||
            (ffblk.wr_date == StartDate && ffblk.wr_time >= StartTime)); }
    BOOL findFirst(const char* Pattern) {
        dosdate_t dd;                       _dos_getdate(&dd);
        dostime_t dt;                       _dos_gettime(&dt);
        StartDate=dd.day+(dd.month << 5)+((dd.year-1980) << 9);
        StartTime=(dt.second >> 1)+(dt.minute << 5)+(dt.hour << 11);
        return (_dos_findfirst(Pattern,_A_RDONLY | _A_HIDDEN |
                _A_SYSTEM | _A_ARCH,&ffblk) == 0);
    }
    BOOL findNext() { return (_dos_findnext(&ffblk) == 0); }
    const char* getFName() { return ffblk.name; }
    void copyDateTimeAttr() {
        ai.attrib=ffblk.attrib;
        ai.time=ffblk.wr_time;              ai.date=ffblk.wr_date;
    }
};

#else /* __UNKNOWN_ENVIRONMENT */
#pragma message ("unknown environment:")
#pragma message ("    1. _fastcall and _stdcall keywords are disabled")
#pragma message ("    2. wildcards and file attributes are disabled")

static const char* UsageStr="Usage: ppmd <e|d> [switches] FileName[s]\n";
inline void     EnvSetNormAttr(const char* ) {}
inline int                        EnvGetCh() { return getchar(); }
inline void          EnvGetCWD(char* CurDir) { CurDir[0]=0; }
inline void EnvSetDateTimeAttr(const char* ) {}
struct ENV_FIND_RESULT {
    const char* pPattern;
    BOOL isNotAllowed() { return FALSE; }
    BOOL findFirst(const char* Pattern) {
        pPattern=Pattern;                   return TRUE;
    }
    BOOL         findNext() { return FALSE; }
    const char*  getFName() { return pPattern; }
    void copyDateTimeAttr() {}
};
#endif /* defined(__WIN32_ENVIRONMENT) */

void PrintInfo(FILE* DecodedFile,FILE* EncodedFile,int NCont,int RunTime)
{
    UINT NDec=ftell(DecodedFile);
    NDec += (NDec == 0);
    UINT NEnc=ftell(EncodedFile)-StartFilePosition;
    UINT n1=(8*NEnc)/NDec;
    UINT n2=(800*NEnc-100*NDec*n1+NDec/2)/NDec;
    if (n2 == 100) { n1++;                  n2=0; }
    printf("%13s: %d/%d, %1d.%02d bpb, conts: %d, speed: %d KB/sec \r",
        pFName,NDec,NEnc,n1,n2,NCont,NDec/(RunTime+1));
}
static char* ChangeExt(const char* In,char* Out,const char* Ext)
{
    char* RetVal=Out;
    const char* p=strrchr(In,'.');
    if (!p || strrchr(In,'\\') > p)         p=In+strlen(In);
    while (In != p)                         *Out++ = *In++;
    *Out++='.';
    while(*Ext)                             *Out++=*Ext++;
    *Out=0;
    return RetVal;
}
inline BOOL RemoveFile(const char* FName)
{
    EnvSetNormAttr(FName);                  remove(FName);
    return TRUE;
}
static BOOL TestAccess(const char* FName)
{
static BOOL YesToAll=FALSE;
    FILE* fp=fopen(FName,"rb");
    if ( !fp )                              return TRUE;
    fclose(fp);
    if ( YesToAll )                         return RemoveFile(FName);
    printf("%s already exists, overwrite?: <Y>es, <N>o, <A>ll, <Q>uit?",FName);
    for ( ; ; )
        switch ( toupper(EnvGetCh()) ) {
            case 'A':                       YesToAll=TRUE;
            case '\r': case 'Y':            return RemoveFile(FName);
            case 0x1B: case 'Q':            printf("User break!");
                                            exit(-1);
            case 'N':                       return FALSE;
        }
}
inline void EncodeFile(ENV_FIND_RESULT& efr,int MaxOrder,int SASize,const char* ArcName)
{
    char WrkStr[256];
    strcpy(WrkStr,ArcName);
    if (!WrkStr[0] && !TestAccess(ChangeExt(efr.getFName(),WrkStr,"PMD")))
                return;
    FILE* fpIn = fopen(efr.getFName(),"rb"), * fpOut = fopen(WrkStr,"a+b");
    if (!fpIn || !fpOut) {
        printf(MTxt[0],efr.getFName());     exit(-1);
    }
    setvbuf(fpIn,NULL,_IOFBF,64*1024);      setvbuf(fpOut,NULL,_IOFBF,64*1024);
    pFName=strrchr(efr.getFName(),'\\');
    pFName=( pFName )?(pFName+1):(efr.getFName());
    efr.copyDateTimeAttr();
    ai.signature=PPMDSignature;             ai.FNLen=strlen(pFName);
    ai.info=(MaxOrder-1) | ((SASize-1) << 4) | ((Variant-'A') << 12);
    fwrite(&ai,sizeof(ai),1,fpOut);         fwrite(pFName,ai.FNLen,1,fpOut);
    if ( !StartSubAllocator(SASize) ) {
        printf(MTxt[2]);                    exit(-1);
    }
    StartFilePosition=ftell(fpOut);
    EncodeFile(fpOut,fpIn,MaxOrder);        putchar('\n');
    if (ferror(fpOut) || ferror(fpIn)) {
        printf(MTxt[1],efr.getFName(),WrkStr);
        exit(-1);
    }
    fclose(fpIn);                           fclose(fpOut);
}
inline BOOL DecodeOneFile(FILE* fpIn)
{
    char WrkStr[256];
    int MaxOrder,SASize;
    if ( !fread(&ai,sizeof(ai),1,fpIn) )    return FALSE;
    fread(WrkStr,ai.FNLen,1,fpIn);          WrkStr[ai.FNLen]=0;
    if ( !TestAccess(WrkStr) )              return FALSE;
    FILE* fpOut=fopen(pFName=WrkStr,"wb");  setvbuf(fpOut,NULL,_IOFBF,64*1024);
    MaxOrder=(ai.info & 0x0F)+1;            SASize=((ai.info >> 4) & 0xFF)+1;
    DWORD Variant=(ai.info >> 12)+'A';
    if (!fpOut || ai.signature != PPMDSignature || Variant != ::Variant) {
        printf(MTxt[0],WrkStr);             exit(-1);
    }
    if ( !StartSubAllocator(SASize) ) {
        printf(MTxt[2]);                    exit(-1);
    }
    StartFilePosition=ftell(fpIn);
    DecodeFile(fpOut,fpIn,MaxOrder);        putchar('\n');
    if (ferror(fpOut) || ferror(fpIn) || feof(fpIn)) {
        printf(MTxt[1],WrkStr,WrkStr);      exit(-1);
    }
    fclose(fpOut);                          EnvSetDateTimeAttr(WrkStr);
    return TRUE;
}
_BIG_INLINE void DecodeFile(ENV_FIND_RESULT& efr)
{
    FILE* fpIn=fopen(efr.getFName(),"rb");  setvbuf(fpIn,NULL,_IOFBF,64*1024);
    if ( !fpIn ) {
        printf(MTxt[0],efr.getFName());     exit(-1);
    }
    while ( DecodeOneFile(fpIn) )           ;
    fclose(fpIn);
}
_BIG_INLINE void TestArchive(char* ArcName,const char* Pattern)
{
    if ( !Pattern[0] ) {
        char CurDir[256];
        EnvGetCWD(CurDir);
        char* p=strrchr(CurDir,'\\');
        strcpy(ArcName,( p )?(p+1):("ppmdfile"));
        if ((p=strrchr(ArcName,'.')) != NULL)
                *p=0;
    } else                                  strcpy(ArcName,Pattern);
    if ( !strrchr(ArcName,'.') )            strcat(ArcName,".PMD");
    FILE* fp = fopen(ArcName,"a+b");        fseek(fp,0,SEEK_END);
    if ( !fp ) {
        printf(MTxt[0],ArcName);            exit(-1);
    }
    if ( ftell(fp) ) {
        fseek(fp,0,SEEK_SET);
        if (!fread(&ai,sizeof(ai),1,fp) || ai.signature != PPMDSignature ||
                                            (ai.info >> 12)+'A' != ::Variant) {
            printf(MTxt[0],ArcName);        exit(-1);
        }
        fseek(fp,0,SEEK_END);
    }
    fclose(fp);
}
int main(int argc, char *argv[])
{
    char ArcName[256];
    BOOL Encode,DeleteFile=FALSE;
    int i,MaxOrder=4,SASize=10;
    printf("Fast PPM compressor for textual data, variant %c, %s\n",char(Variant),__DATE__);
    if (argc < 2) {
        printf("written and distributed to public domain by Dmitry Shkarin(shkarin@arstel.ru)\n");
        printf(UsageStr);
        printf("Switches (for encoding only):\n");
        printf("\t-fName - set output file name to Name\n");
        printf("\t-mN    - use N MB memory - [1,256], default: %d\n",SASize);
        printf("\t-oN    - set model order to N - [1,%d], default: %d\n",MAX_O,MaxOrder);
        printf("\t-d     - delete file[s] after processing, default: %s\n",
                ( DeleteFile )?"enabled":"disabled");
        return -1;
    }
    switch ( toupper(argv[1][0]) ) {
        case 'E': Encode=TRUE;                                  break;
        case 'D': Encode=FALSE;                                 break;
        default : printf("unknown command: %s\n",argv[1]);      return -1;
    }
    for (ArcName[0]=0,i=2;i < argc && (argv[i][0] == '-' || argv[i][0] == '/');i++)
        switch ( toupper(argv[i][1]) ) {
            case 'F': TestArchive(ArcName,argv[i]+2);           break;
            case 'M': SASize=CLAMP(atoi(argv[i]+2),1,256);      break;
            case 'O': MaxOrder=CLAMP(atoi(argv[i]+2),1,MAX_O);  break;
            case 'D': DeleteFile=TRUE;                          break;
            default : printf("unknown switch: %s\n",argv[i]);   return -1;
        }
    for (ENV_FIND_RESULT efr;i < argc;i++)
        if ( efr.findFirst(argv[i]) ) {
            do {
                if ( efr.isNotAllowed() )   continue;
                if ( Encode )               EncodeFile(efr,MaxOrder,SASize,ArcName);
                else                        DecodeFile(efr);
                if ( DeleteFile )           remove(efr.getFName());
            } while ( efr.findNext() );
        }
    return 0;
}
