/****************************************************************************
 *  This file is part of PPMD project                                       *
 *  Written and distributed to public domain by Dmitry Shkarin 1997,        *
 *  1999-2000                                                               *
 *  Contents: memory allocation routines                                    *
 ****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include "ppmd.h"
#include "suballoc.h"

const UINT MAX_UNIT=7;
inline int U2S(int NU) { return 2*NU+4*NU; }
inline int S2B(int Size) { return (Size+sizeof(DWORD)-1) >> 2; }
const UINT MIN_BLK_SIZE=(6*MAX_UNIT+sizeof(DWORD)-1) >> 2,MAX_BLK_SIZE=0x8000-1;
static long SubAllocatorSize=0;
static BYTE * LoUnit, * HiUnit;

static struct NODE {
    NODE * prev, * next;
    inline void insert(int indx=0);
    inline void           remove();
} FreeList[MAX_UNIT];
static struct MEM_BLK {
    UINT Size: 16, PrevSize: 15, IsFree: 1;
    MEM_BLK* getMemNext() { return (this+Size+1); }
    MEM_BLK* getMemPrev() { return (this-PrevSize-1); }
    inline void canMergeNext();
    inline void canSplit(int sz,BOOL UseGap);
} * HeapStart=NULL;
struct FREE_BLK: public MEM_BLK, public NODE {
    inline FREE_BLK* canMergePrev();
};

inline void NODE::insert(int indx) {
    prev=&FreeList[indx];                   next=FreeList[indx].next;
    prev->next=next->prev=this;
}
inline void NODE::remove() {
    next->prev=prev;                        prev->next=next;
}
inline void MEM_BLK::canMergeNext()
{
    FREE_BLK* NextBlk=(FREE_BLK*) getMemNext();
    if (!NextBlk->IsFree || Size+NextBlk->Size >= MAX_BLK_SIZE)
            return;
    NextBlk->remove();                      Size += NextBlk->Size+1;
    getMemNext()->PrevSize=Size;
}
inline void MEM_BLK::canSplit(int sz,BOOL UseGap)
{
    if ( UseGap ) {
        if (sz < 16)                        sz += 6;
        else if (sz < 32)                   sz += 9;
        else if (sz < 64)                   sz += 12;
        else if ((sz += 15) > 6*256/4)      sz=6*256/4;
    }
    if (Size < sz+1+MIN_BLK_SIZE)           return;
    FREE_BLK* NewBlock=(FREE_BLK*) (this+1+sz);
    NewBlock->Size=Size-sz-1;               NewBlock->IsFree=TRUE;
    NewBlock->PrevSize=Size=sz;
    NewBlock->getMemNext()->PrevSize=NewBlock->Size;
    NewBlock->canMergeNext();               NewBlock->insert();
}
inline FREE_BLK* FREE_BLK::canMergePrev()
{
    FREE_BLK* PrevBlk=(FREE_BLK*) getMemPrev();
    if (PrevSize+Size >= MAX_BLK_SIZE || !PrevBlk->IsFree)
            return this;
    PrevBlk->remove();                      PrevBlk->Size += Size+1;
    PrevBlk->getMemNext()->PrevSize=PrevBlk->Size;
    return PrevBlk;
}

void StopSubAllocator() {
    if ( SubAllocatorSize ) {
        SubAllocatorSize=0;                 delete HeapStart;
    }
}
BOOL StartSubAllocator(int SASize)
{
    int t=S2B(SASize << 20);
    if (SubAllocatorSize == t)              return TRUE;
    StopSubAllocator();
    if ((HeapStart=new MEM_BLK[t]) == NULL) return FALSE;
    SubAllocatorSize=t;                     return TRUE;
}
void InitSubAllocator()
{
    MEM_BLK* HeapEnd = HeapStart+SubAllocatorSize-1;
    LoUnit=HiUnit=NULL;
    for (int i=0;i < MAX_UNIT;i++)
            FreeList[i].next=FreeList[i].prev=&FreeList[i];
    FREE_BLK* OldPtr, * BlkPtr = (FREE_BLK*) HeapStart;
    do {
        BlkPtr->PrevSize=BlkPtr->Size=MAX_BLK_SIZE-512;
        BlkPtr->IsFree=TRUE;                BlkPtr->insert();
        OldPtr=BlkPtr;                      BlkPtr=(FREE_BLK*) BlkPtr->getMemNext();
    } while (BlkPtr < HeapEnd);
    HeapStart->PrevSize=MAX_BLK_SIZE;
    OldPtr->remove();
    OldPtr->Size=HeapEnd-(((MEM_BLK*) OldPtr)+1);
    if (OldPtr->Size >= MIN_BLK_SIZE)       OldPtr->insert();
    else                                    OldPtr->IsFree=FALSE;
    HeapEnd->PrevSize=OldPtr->Size;         HeapEnd->IsFree=FALSE;
}
void* _FASTCALL AllocUnits(int NU)
{
    if (FreeList[NU-1].next != FreeList[NU-1].prev) {
        NODE* Block=FreeList[NU-1].next;
        Block->remove();                    return Block;
    }
    int Diff=HiUnit-LoUnit, NBytes=U2S(NU);
    if (Diff < NBytes) {
        if ((Diff /= 6) > 1)                ((NODE*) LoUnit)->insert(Diff-1);
        if (FreeList->next==FreeList->prev) return NULL;
        FREE_BLK* Block=(FREE_BLK*) (FreeList->next);
        Block->remove();                    Block->IsFree=FALSE;
        LoUnit=((BYTE*) Block)+4;           HiUnit=LoUnit+6*((4*Block->Size)/6);
    }
    if (NU == 2)                            return (LoUnit += 12)-12;
    else                                    return (HiUnit -= NBytes);
}
void* _FASTCALL AllocBlk(UINT Size,BOOL UseGap)
{
    FREE_BLK* Block;
    int sz=S2B(Size);
    for (Block=(FREE_BLK*) FreeList->next; ;Block=(FREE_BLK*)Block->next)
            if (Block == FreeList)          return NULL;
            else if (Block->Size >= sz)     break;
    Block->remove();                        Block->IsFree=FALSE;
    Block->canSplit(sz,UseGap);             return ((MEM_BLK*) Block)+1;
}
void _FASTCALL FreeBlk(void* DataPtr,int OldNU)
{
    if (OldNU <= MAX_UNIT) {
        ((NODE*)DataPtr)->insert(OldNU-1);  return;
    }
    FREE_BLK* Block=(FREE_BLK*) (((MEM_BLK*) DataPtr)-1);
    Block->IsFree=TRUE;
    Block=Block->canMergePrev();            Block->canMergeNext();
    Block->insert();
}
void* _FASTCALL ExpandBlk(void* DataPtr,int OldNU,BOOL UseGap)
{
    void* ptr;
    if (OldNU <= MAX_UNIT) {
        ptr=(OldNU < MAX_UNIT)?AllocUnits(OldNU+1):AllocBlk(6*(MAX_UNIT+1),UseGap);
        if ( !ptr )                         return NULL;
        memcpy(ptr,DataPtr,U2S(OldNU));     ((NODE*) DataPtr)->insert(OldNU-1);
        return ptr;
    }
    int Size=U2S(OldNU+1);
    int sz=S2B(Size);
    MEM_BLK* Block=((MEM_BLK*) DataPtr) - 1;
    Block->canMergeNext();
    if (Block->Size < sz) {
        if ((ptr=AllocBlk(Size,UseGap)) == NULL)
                return NULL;
        memcpy(ptr,DataPtr,Size-6);         FreeBlk(DataPtr,OldNU);
        return ptr;
    }
    Block->canSplit(sz,UseGap);             return DataPtr;
}
void* _FASTCALL ShrinkBlk(void* DataPtr,int OldNU,int NewNU,BOOL UseGap)
{
    void* ptr;
    if (NewNU <= MAX_UNIT) {
        if (OldNU == NewNU)                 return DataPtr;
        if ((ptr=AllocUnits(NewNU))==NULL)  return NULL;
        memcpy(ptr,DataPtr,U2S(NewNU));     FreeBlk(DataPtr,OldNU);
        return ptr;
    }
    MEM_BLK* Block=((MEM_BLK*) DataPtr) - 1;
    Block->canSplit(S2B(U2S(NewNU)),UseGap);
    return DataPtr;
}
