/*
 *    The term 'program' used in the copyright and the license below means
 *    PReP Version 1.0 
 *
 *    Copyright (C) 1998, 1999 DELECis
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *    This copyright notice must appear in all copies and modifications.
 *
 *    Contact: prep@delec.com
 *
 *    Read the manual: http://www.delec.com/is/products/prep/manual/jprep/
 */

/**
 * PReP Version 1.0
 *
 * Store: Object to store tables from a database received with a
 *        specified SQL statement
 *
 * History: - Creation 12/09/98 nh
 *
 * Date: 12/09/98
 *
 * @version 1.0
 */

package com.delec.prep;

import java.util.Vector;

public class Store
{
	/** stores the field names
	 */
	protected Vector name;

	/** stores the actual values
	 */
	protected Vector value;

	/** current record that is worked on
	 */
	protected Vector current;

	/** current cursor position within the records (used with moveFirst and moveNext)
	**/
	protected int cursor = 0;

	protected int count;

	/**
	 * constructs the Store with no knowledge of the amount of fields that will be used
	 */
	public Store()
	{
		this(4);
	}

	/**
	 * constructs the Store from a given input stream. stream should start as follows:
	 * %%head
	 * {lines of header cols}
	 * %%body
	 * {lines for the recordset}
	 * %%body  (as long as there are more records)
	 *
	 */

	/**
	 * constructs the Store with an intial count of fields (better for performance)
	 *
	 * @param count the actual count of cols used
	 */
	public Store(int count)
	{
		super();
		name = new Vector(count);
		value= new Vector(count);
		current = new Vector(count);
		value.addElement(current);    
		this.count = count;
	}

	/**
	 * adds a title to a column
	 *
	 * @param String the Title to add (added at the end)
	 * @param int the type of the object
	 */
	public void addName(String strLabel, int iType)
	{
		HeaderInfo headerinfo = new HeaderInfo(strLabel, iType);
		name.addElement(headerinfo);
	}

	/**
	 * adds a value to a column
	 *
	 * @param PRePTagObject the value to add (added at the end, creates new column when size equal to header size)
	 */
	public void addValue(PRePTagObject prepTagObject)
	{
		current.addElement(prepTagObject);
		if ( current.size() == name.size() )
		{
			current = new Vector(count);
			value.addElement(current);
		}
	}

	/**
	* removes a row in the store object
	*
	* @param int row
	*/

	public void removeRow(int iRow)
	{
	    moveTo(iRow);

	    for (int i = 0; i < ((Vector)value.elementAt(cursor)).size(); i++)
	    {
		((Vector)value.elementAt(cursor)).remove(i);
	    }
	    value.remove(cursor);
	}

	public void removeCol(int iCol)
	{
	    for (int i = 0; i < values(); i++)
	    {
		moveTo(i);
		((Vector)value.elementAt(cursor)).remove(iCol);
	    }
	    name.remove(iCol);
	}

	/**
	 * returns the value of the first record that has the passed name<p>
	 *
	 * should be modified in the future to throw an exception if no param was found
	 *
	 * @param String Name of the column to look in
	 * @return null if there was no record to read,
	 *                otherwise the first record value in this column is returned as PRePTagObject
	 *
	 */
	public PRePTagObject getValue(String strArgument)
	{
		if ( values() == 0 ) return null;
		int i;
		for ( i = 0; i < name.size(); i++ )
		{
			HeaderInfo header = (HeaderInfo)name.elementAt(i);
			if ( (header.strLabel.equals(strArgument)) )
			{
				return getValue(i);
			}
		}
		return null;      
	}

	/**
	 * returns the value of the first record that has the passed name<p>
	 *
	 * should be modified in the future to throw an exception if no param was found
	 *
	 * @param int number of the column to look in
	 * @return null if there was no record to read, otherwise the first
	 *			record value in this column 
	 *
	 */
	public PRePTagObject getValue(int i)
	{
		if ( values() == 0 )
		{
			return null;
		}
		return(PRePTagObject)((Vector)value.elementAt(cursor)).elementAt(i);
	}

	public void output()
	{
		for ( int i = 0; i < getNbrOfColumns(); i++ )
		{
			System.out.println(getLabel(i));
			System.out.println(getValue(getLabel(i)).getValue());
			System.out.println("");
		}
	}

	/**
	 * returns the count of records in this store
	 * @return int count of records available
	 */
	public int values()
	{
		return value.size()-1;
	}

	/**
	 * move cursor to a specified position
	 * @param int position
	 */
	public void moveTo(int iRow)
	{
		if ( cursor > values() )
		{
			cursor = values();
		}
		else
		{
			cursor = iRow;
		}
	} 

	/**
	 * gets the number of how many columns exist
	 * @return the number of columns
	 */
	public int getNbrOfColumns() 
	{
		return name.size();
	}

	/**
	 * gets the name of the label in the specified position (column)
	 * @param int position (column)
	 * @return the label as String
	 */
	public String getLabel(int iPosition)
	{
		HeaderInfo header = (HeaderInfo)name.elementAt(iPosition);
		return header.strLabel;
	}

	/**
	 * pushes one row of the store object in the taglist
	 * it creates a new PRePTagText which has as name the label
	 * of the column and as value the value of this row in this
	 * column
	 */
	public void push(Session session)
	{
		//PRePTagText pttField;
		PRePTagObject prepTagObjField;
		for ( int i = 0; i < name.size(); i++ )
		{
			//pttField = new PRePTagText(getLabel(i));
			prepTagObjField = new PRePTagObject(getLabel(i));
			//pttField.setValue(getValue(i).getValue().trim());
			prepTagObjField.setValue(getValue(i).getObjectValue());
			session.getTagList().addElement(prepTagObjField);
		}
	}

	public void compileFields(Session session, PRePTag prepTagOutput, int iNbr)
	{
		if ( iNbr == 0 )
		{
			compileFields(session, prepTagOutput);
		}
		else if ( iNbr < values() )
		{
			for ( int iRow = 0; iRow < iNbr; iRow++ )
			{
				moveTo(iRow);
				int iTagListSize = session.getTagList().size();
				push(session);
				prepTagOutput.compile(session);
				session.getTagList().setSize(iTagListSize);
			}
		}
		else
		{
			compileFields(session, prepTagOutput);
			session.getExceptionList().addElement(new PRePException(this, "'Number too big'"));
		}
	}

	public void compileFields(Session session, PRePTag prepTagOutput)
	{
		for ( int iRow = 0; iRow < values(); iRow++ )
		{
			moveTo(iRow);
			int iTagListSize = session.getTagList().size();
			push(session);
			prepTagOutput.compile(session);
			session.getTagList().setSize(iTagListSize);
		}
	}

	/**
	 * inner class HeaderInfo
	 *
	 * provides a label name and the type of objects which are stored
	 * in this column
	 *
	 */
	public class HeaderInfo
	{
		protected String strLabel;
		protected int iType;

		public HeaderInfo(String strLabel, int iType)
		{
			this.strLabel = strLabel;
			this.iType = iType;
		}
	}
}

