/**
*       corbac.c
*       --------
*
*       Base CORBA Module
*
*
*       Copyright Mitch Britton 1998-1999
*       All Rights Reserved
*       See the file licence.txt
*
**/

/**
*       Compilation Switches and Definitions
*       ------------------------------------
**/

#define BUFF_INCREMENT     1024
#define NUM_DEFINED_OBJECTS 500


/**
*       Header Files
*       ------------
**/

#include <stdio.h>
#include <stdarg.h>
#include "corba.h"
#include "corba_i.h"
#include "IOP_i.h"


/**
*       External Data
*       -------------
**/

/**
*       Internal Global Data
*       --------------------
**/

static char ** object_ids[ NUM_DEFINED_OBJECTS ] ;

static int object_ids_index = 0 ;

/**
*       Internal Function Declarations
*       ------------------------------
**/

static void noMemory( CORBA_Dbuff * b ) ;

static void CORBA_BOA_release_slot( CORBA_BOA boa, CORBA_iObject * o ) ;

/**
*       External Function Declarations
*       ------------------------------
**/

/**
*       Public Global Data
*       ------------------
**/


MSC_EXPORT CORBA_TypeCode CORBA__tk_null       = { CORBA_tk_null,       NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_void       = { CORBA_tk_void,       NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_short      = { CORBA_tk_short,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_long       = { CORBA_tk_long,       NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_ushort     = { CORBA_tk_ushort,     NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_ulong      = { CORBA_tk_ulong,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_float      = { CORBA_tk_float,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_double     = { CORBA_tk_double,     NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_boolean    = { CORBA_tk_boolean,    NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_char       = { CORBA_tk_char,       NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_octet      = { CORBA_tk_octet,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_any        = { CORBA_tk_any,        NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_TypeCode   = { CORBA_tk_TypeCode,   NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_Principal  = { CORBA_tk_Principal,  NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_objref     = { CORBA_tk_objref,     NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_struct     = { CORBA_tk_struct,     NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_union      = { CORBA_tk_union,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_enum       = { CORBA_tk_enum,       NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_string     = { CORBA_tk_string,     NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_sequence   = { CORBA_tk_sequence,   NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_array      = { CORBA_tk_array,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_alias      = { CORBA_tk_alias,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_except     = { CORBA_tk_except,     NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_longlong   = { CORBA_tk_longlong,   NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_ulonglong  = { CORBA_tk_ulonglong,  NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_longdouble = { CORBA_tk_longdouble, NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_wchar      = { CORBA_tk_wchar,      NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_wstring    = { CORBA_tk_wstring,    NULL, NULL, NULL } ;
MSC_EXPORT CORBA_TypeCode CORBA__tk_fixed      = { CORBA_tk_fixed,      NULL, NULL, NULL } ;

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_add_object_ids( char ** ids )
{
    if ( object_ids_index >= ( NUM_DEFINED_OBJECTS - 1 ) )
    {
        fprintf( stderr, 
            "CORBA_add_object_ids: NUM_DEFINED_OBJECTS exceeded - recompile with higher value" ) ;
        exit( -1 ) ;
    }

    object_ids[ object_ids_index++ ] = ids ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_boolean BC_EXPORT CORBA_implements( char ** implements, const char * id )
{
    int i ;

    for ( i = 0 ; implements[ i ] ; i++ )
    {
        if ( strcmp( id, implements[ i ] )== 0 )   /* found the object */
            return CORBA_TRUE ;
    }
    
    return CORBA_FALSE ;
}

/**
*
*
*
*
**/
MSC_EXPORT char ** BC_EXPORT CORBA_implements_nothing()
{
    static char * nothing[] = { NULL } ;

    return nothing ;
}

/**
*
*
*
*
**/
static CORBA_boolean CORBA_can_narrow( const char * src, const char * trg )
{
    int i ;
    char ** implements ;
    
    for ( i = 0 ; i < object_ids_index ; i++ )
    {
        implements = object_ids[ i ] ;
        
        if ( strcmp( src, implements[ 0 ] )== 0 )   /* found the object */
            return CORBA_implements( implements, trg ) ;
    }
    
    return CORBA_FALSE ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_boolean BC_EXPORT CORBA_Object_is_a( CORBA_Object obj, const char * logical_type_id )
{ 
    CORBA_char * repId = CORBA_getRepId( obj ) ;

    if ( ( strcmp( logical_type_id, repId ) == 0 ) ||
         ( strcmp( logical_type_id, "Object" ) == 0 ) ) 
        return CORBA_TRUE ;

    return CORBA_can_narrow( repId, logical_type_id ) ;
}


/**
*
*
*
*
**/
CORBA_char * CORBA_getRepId( CORBA_Object obj )
{
    CORBA_BOA_OBJS * bobj = &BOA_OBJ( ( ( CORBA_iObject * )obj ) ) ;
    
    CORBA_IOR * ior = &bobj->ior ;

    return ior->id ;
}

/**
*       MEMORY FUNCTIONS
*       ----------------
*
**/


typedef struct
{
    CORBA_short type ;
    CORBA_unsigned_short flags ;
    CORBA_iFreeFunc freeFnc ;
}
CORBA_iMemory ;

typedef struct
{
    CORBA_unsigned_long size ;      /* size of elements   */
    CORBA_unsigned_long len  ;      /* number of elements */
    CORBA_iMemory h ;
}
CORBA_iSizedMemory ;

#define MA_GET_OS_PTR(A)          (((CORBA_iMemory*)A)-1)

#define MA_GET_SIZED_OS_PTR(A)    (((CORBA_iSizedMemory*)A)-1)


/**
*       freeSequenceMembers
*       -------------------
*
*       Free the sequence members in the sequence starting from a particular
*       point
**/
static void freeSequenceMembers( CORBA_iSizedMemory * mem, int where )
{
    CORBA_unsigned_long len ;

    char * p = ( ( char * )mem ) + sizeof( CORBA_iSizedMemory ) ;

    CORBA_unsigned_long size = mem->size ;

    p += size * where ;

    for ( len = mem->len ; where < len ; where++ )
    {
       ( mem->h.freeFnc )( p ) ;

        p += size ;
    }
}

/**
*       sequenceFreeBuffer
*       ------------------
*
*       Free the buffer memory associated with the sequence.
*
**/
static void sequenceFreeBuffer( void * p )
{
    if ( p != NULL )   /* sequence may have zero length i.e. p == NULL */
    {
        CORBA_iSizedMemory * mem = MA_GET_SIZED_OS_PTR( p ) ;

        if ( mem->h.flags & CORBA_MA_OWNS )   /* variable sequence */
            freeSequenceMembers( mem, 0 ) ;

        free( mem ) ;
    }
}


/**
*       sequenceFree
*       ------------
*
*       Free the memory associated with the sequence. This function is
*       only called if the sequence "owns" its buffer
**/
static void sequenceFree( CORBA_Generic_Sequence * s )
{
    sequenceFreeBuffer( s->_buffer ) ;
}


/**
*
*
*
*
**/
MSC_EXPORT void * BC_EXPORT CORBA_seqBuffAlloc( CORBA_iFreeFunc fnc, CORBA_unsigned_long size, CORBA_unsigned_long len )
{
    CORBA_unsigned_long total = ( size * len ) ;

    CORBA_iSizedMemory * mem = malloc( total + + sizeof( CORBA_iSizedMemory ) ) ;

    if ( mem )
    {
        mem->h.type    = CORBA_MA_SIZED_MEM ;
        mem->h.freeFnc = ( CORBA_iFreeFunc )fnc ;
        mem->size = size ;
        mem->len  = len ;

        if ( fnc )
            mem->h.flags = CORBA_MA_OWNS ;
        else
            mem->h.flags = 0 ;

        mem++ ;

        if ( fnc )
            memset( mem, 0, total ) ;
    }

    return mem ;
}


/**
*
*
*
*
**/
MSC_EXPORT void * BC_EXPORT CORBA_alloc( CORBA_short type, CORBA_long size, CORBA_boolean zero )
{
    CORBA_iMemory * t = malloc( size + sizeof( CORBA_iMemory ) ) ;

    if ( t )
    {
        t->type = type ;
        t->flags = 0 ;

        if ( type == CORBA_MA_SEQ )
            t->freeFnc = ( CORBA_iFreeFunc )sequenceFree ;

        t++ ;

        if ( zero )
            memset( t, 0, size ) ;
    }

    return t ;
}

/**
*
*
*
*
**/
MSC_EXPORT void * BC_EXPORT CORBA_varAlloc( CORBA_iFreeFunc freeFnc, CORBA_long size )
{
    CORBA_iMemory * mem = calloc( 1, size + sizeof( CORBA_iMemory ) ) ;

    if ( mem )
    {
        mem->type     = CORBA_MA_VAR ;
        mem->freeFnc  = freeFnc ;
        mem->flags    = CORBA_MA_OWNS ;
        mem++ ;
    }

    return mem ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_free( void * p )
{
    CORBA_iMemory * mem = MA_GET_OS_PTR( p ) ;

    if ( mem->type == CORBA_MA_SIZED_MEM )
    {
        sequenceFreeBuffer( p ) ;
    }
    else
    {
        if ( mem->flags & CORBA_MA_OWNS )
        {
            if ( mem->type == CORBA_MA_ANY )
            {
                CORBA_free( ( ( CORBA_any * )p )->_value ) ;
            }
            else
                ( mem->freeFnc )( p ) ;
        }

        free( mem ) ;
    }
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_conditionalFree( void * p )
{
    if ( p )
        CORBA_free( p ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_any * BC_EXPORT CORBA_any_alloc( void )
{
    return CORBA_alloc( CORBA_MA_ANY, sizeof( CORBA_any ), CORBA_TRUE ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT void * BC_EXPORT CORBA_Sequence_Alloc( CORBA_unsigned_long max )
{
    CORBA_Generic_Sequence * seq = CORBA_alloc( CORBA_MA_SEQ,
        sizeof( CORBA_Generic_Sequence ), CORBA_FALSE ) ;

    if ( seq )
    {
        seq->_maximum = max ;
        seq->_length  = 0 ;
        seq->_buffer  = NULL ;
    }

    return seq ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_any_set_release( CORBA_any * any, CORBA_boolean value )
{
    CORBA_iMemory * t = MA_GET_OS_PTR( any ) ;

    if ( t->type != CORBA_MA_FIXED )
    {
        if ( value )
            t->flags |= CORBA_MA_OWNS ;
        else
            t->flags &= ~CORBA_MA_OWNS ;
    }
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_boolean BC_EXPORT CORBA_any_get_release( CORBA_any * any )
{
    CORBA_iMemory * t = MA_GET_OS_PTR( any ) ;

    if ( t->flags & CORBA_MA_OWNS )
        return CORBA_TRUE ;

    return CORBA_FALSE ;
}


/**
*
*
*
*
**/
void CORBA_TypeCode_init( CORBA_TypeCode * tc )
{
    tc->tc    = CORBA_tk_null ;
    tc->_id   = NULL ;
    tc->_name = NULL ;
    tc->_ptr  = NULL ;
}

/**
*
*
*
*
**/
void CORBA_TypeCode_release( CORBA_TypeCode * tc )
{
    if ( tc->_id )
        CORBA_free( tc->_id ) ;

    if ( tc->_name )
        CORBA_free( tc->_name ) ;

    tc->_id   = NULL ;
    tc->_name = NULL ;

    switch ( tc->tc )
    {
        case CORBA_tk_fixed :
        break ;
        
        case CORBA_tk_struct :
        case CORBA_tk_except :
        break ;

        case CORBA_tk_enum :
        break ;

        case CORBA_tk_union :
        break ;

        case CORBA_tk_array :
        case CORBA_tk_sequence :
        break ;

        case CORBA_tk_alias :
        {
            CORBA_TypeCode * tcp = ( CORBA_TypeCode * )tc->_ptr ;

            CORBA_TypeCode_release( tcp ) ;
        }
        break ;
    }
}

/**
*
*
*
*
**/
void CORBA_any_replace( CORBA_any * any, CORBA_TypeCode * tc, void * value, CORBA_boolean release )
{
    CORBA_TypeCode_release( &any->_type ) ;    

    CORBA_any_set_release( any, release ) ;

    CORBA_s_copy_TypeCode( &any->_type, tc ) ;
   
    any->_value = value ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_sequence_set_release( void * buff, CORBA_boolean value )
{
    CORBA_iMemory * t = MA_GET_OS_PTR( buff ) ;

    if ( t->type != CORBA_MA_FIXED )
    {
        if ( value )
            t->flags |= CORBA_MA_OWNS ;
        else
            t->flags &= ~CORBA_MA_OWNS ;
    }
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_boolean BC_EXPORT CORBA_sequence_get_release( void * buff )
{
    CORBA_iMemory * t = MA_GET_OS_PTR( buff ) ;

    if ( t->flags & CORBA_MA_OWNS )
        return CORBA_TRUE ;

    return CORBA_FALSE ;
}

/**
*
*
*
*
**/
MSC_EXPORT void * BC_EXPORT CORBA_calloc( CORBA_long num, CORBA_long size )
{
    CORBA_unsigned_long sz = num * size ;

    return CORBA_alloc( CORBA_MA_FIXED, sz, CORBA_TRUE ) ;
}

/**
*
*
*
*
**/
static void CORBA_free_IOR( CORBA_IOR * ior )
{
    if ( ior->id )
        CORBA_free( ior->id ) ;

    IIOP_ProfileBody_1_1__free( &ior->pb ) ;
}


/**
*
*
*
*
**/
void CORBA_free_ORB( CORBA_iORB * orb )
{
    if ( orb->id )
        CORBA_free( orb->id ) ; 

    if ( orb->freeOA && orb->oa )
        CORBA_free( orb->oa ) ; 

    if ( orb->log )
        fclose( orb->log ) ;
}

/**
*
*
*
*
**/
CORBA_iORB * CORBA_alloc_ORB( CORBA_char * orbId )
{
    CORBA_iORB * orb = CORBA_calloc( 1, sizeof( CORBA_iORB ) ) ;

    CORBA_iMemory * t ;
    
    if ( orb )
    {
        orb->id = CORBA_string_copy( orbId ) ;
        
        t = MA_GET_OS_PTR( orb ) ;

        t->freeFnc = ( CORBA_iFreeFunc )CORBA_free_ORB ;

        t->flags |= CORBA_MA_OWNS ;
    }

    return orb ;
}

/**
*
*
*
*
**/
static void CORBA_free_BOA( CORBA_BOA boa )
{
/*
    int i, conns ;

    for ( i = 0, conns = boa->params.connections ; i < conns ; i++ )
    {
        if ( boa->objs[ i ].obj != NULL )
            CORBA_BOA_release_slot( boa, boa->objs[ i ].obj ) ;
    }
*/
    boa->orb->oa = NULL ;
}

/**
*
*
*
*
**/
void CORBA_BOA_onConnect( CORBA_BOA boa, CORBA_Function onConnect )
{
    boa->onConnect = onConnect ;
}

/**
*
*
*
*
**/
CORBA_BOA CORBA_alloc_BOA( int conns )
{
    CORBA_BOA boa = CORBA_calloc( 1, SIZEOF_CORBA_BOA( conns ) ) ;
    CORBA_iMemory * t ;
    
    if ( boa )
    {
        boa->params.connections = conns ;
        
        t = MA_GET_OS_PTR( boa ) ;

        t->freeFnc = ( CORBA_iFreeFunc )CORBA_free_BOA ;

        t->flags |= CORBA_MA_OWNS ;
    }

    return boa ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_free_object( CORBA_iObject * o )
{
    CORBA_Dbuff_free( &o->buff ) ;

    CORBA_Dbuff_free( &o->obuff ) ;

    if ( o->adaptorSlot >= 0 )
        CORBA_BOA_release_slot( BOA_FROM_OBJ( o ), o ) ;
}

/**
*
*
*
*
**/
void CORBA_init_object( CORBA_iObject * obj )
{
    obj->adaptorSlot = -1 ;
    obj->Oadaptor    = NULL ;

    CORBA_Dbuff_Init( &obj->buff ) ;
    CORBA_Dbuff_Init( &obj->obuff ) ;

    CORBA_request_init( &obj->req ) ;

    obj->req.obj = obj ;
}

/**
*
*
*
*
**/
CORBA_iObject * CORBA_alloc_object()
{
    CORBA_iObject * o ;

    if ( ( o = CORBA_alloc( CORBA_MA_OBJECT, 
        sizeof( CORBA_iObject ), CORBA_TRUE ) ) != NULL )
    {
        CORBA_iMemory * t = MA_GET_OS_PTR( o ) ;

        t->freeFnc = ( CORBA_iFreeFunc )CORBA_free_object ;

        t->flags |= CORBA_MA_OWNS ;

        CORBA_init_object( o ) ;
    }

    return o ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_duplicate_object( CORBA_Object trg, CORBA_Object src )
{
    CORBA_iObject * t = ( CORBA_iObject * )trg ;

    CORBA_iObject * s = ( CORBA_iObject * )src ;

    CORBA_BOA boa = ( CORBA_BOA )s->Oadaptor ;
    
    int slot = s->adaptorSlot ;
    
    CORBA_BOA_OBJS * bo = &boa->objs[ slot ] ;

    bo->num++ ;

    t->Oadaptor    = s->Oadaptor ;

    t->adaptorSlot = s->adaptorSlot ;
}

/**
*
*
*
*
**/
MSC_EXPORT char * BC_EXPORT CORBA_string_alloc( CORBA_unsigned_long len )
{
    char * p = CORBA_alloc( CORBA_MA_FIXED, len + 1, CORBA_FALSE ) ;

    if ( p )
        p[ len ] = '\0' ;

    return p ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_wchar * BC_EXPORT CORBA_wstring_alloc( CORBA_unsigned_long len )
{
    CORBA_wchar * p = CORBA_alloc( CORBA_MA_FIXED,
                        ( len + 1 ) * sizeof( CORBA_wchar ), CORBA_FALSE ) ;

    if ( p )
        p[ len ] = 0 ;

    return p ;
}


/**
*
*
*
*
**/
MSC_EXPORT CORBA_char * BC_EXPORT CORBA_string_copy( CORBA_char * src )
{
    CORBA_char * targ ;

    int l = strlen( src ) ;

    targ = CORBA_string_alloc( l ) ;

    if ( targ )
        memcpy( targ, src, ++l ) ;

    return targ ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_wstring_len( CORBA_wchar * src )
{
    int i ;

    for ( i = 0 ; *src ; i++, src++ )
        ;
        
    return i ;    
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_wchar * BC_EXPORT CORBA_wstring_copy( CORBA_wchar * src )
{
    CORBA_wchar * targ ;

    int l = CORBA_wstring_len( src ) ;

    targ = CORBA_wstring_alloc( l ) ;

    if ( targ )
        memcpy( targ, src, ++l ) ;

    return targ ;
}

/**
*
*
*
*
**/
int CORBA_Dbuff_allocate( CORBA_Dbuff * buff, int size )
{
    CORBA_long oldLen = buff->len ;

    if ( size < BUFF_INCREMENT )
        size = BUFF_INCREMENT ;
    else
        size += BUFF_INCREMENT ;

    buff->len += size ;

    if ( oldLen == 0 )
        buff->p = malloc( size ) ;
    else
        buff->p = realloc( buff->p, buff->len ) ;

    if ( buff->p == NULL )
        return -1 ;

    return 0 ;
}

/**
*
*
*
*
**/
void CORBA_Dbuff_reset( CORBA_Dbuff * buff )
{
    buff->offset = GIOP_MESSAGEHEADER_SIZE ; 
    buff->curr   = 0 ; 
    buff->endian = ENDIAN ;
}

/**
*
*
*
*
**/
void CORBA_Dbuff_Init( CORBA_Dbuff * buff )
{
    memset( buff, 0, sizeof( CORBA_Dbuff ) ) ;
}

/**
*
*
*
*
**/
void CORBA_Dbuff_free( CORBA_Dbuff * buff )
{
    if ( buff->p )
    {
        free( buff->p ) ;
        
        CORBA_Dbuff_Init( buff ) ;
    }        
}

/**
*       READ AND WRITE FUNCTIONS
*       ------------------------
*
**/


/**
*
*
*
*
**/
CORBA_short CORBA_swap_short( CORBA_short * v )
{
    union
    {
        CORBA_short v ;
        char c[ 2 ] ;
    } x ;

    char * s = ( char * )v ;
    x.v = *v ;

    s[ 0 ] = x.c[ 1 ] ;
    s[ 1 ] = x.c[ 0 ] ;

    return x.v ;
}

/**
*
*
*
*
**/
CORBA_long CORBA_swap_long( CORBA_long * v )
{
    union
    {
        CORBA_long v ;
        char c[ 4 ] ;
    } x ;

    char * s = ( char * )v ;
    x.v      = *v ;

    s[ 0 ] = x.c[ 3 ] ;
    s[ 1 ] = x.c[ 2 ] ;
    s[ 2 ] = x.c[ 1 ] ;
    s[ 3 ] = x.c[ 0 ] ;

    return x.v ;
}

/**
*
*
*
*
**/
CORBA_long_long CORBA_swap_long_long( CORBA_long_long * v )
{
    union
    {
        CORBA_long_long v ;
        char c[ 8 ] ;
    } x ;

    char * s = ( char * )v ;
    x.v = *v ;


    s[ 0 ] = x.c[ 7 ] ;
    s[ 1 ] = x.c[ 6 ] ;
    s[ 2 ] = x.c[ 5 ] ;
    s[ 3 ] = x.c[ 4 ] ;
    s[ 4 ] = x.c[ 3 ] ;
    s[ 5 ] = x.c[ 2 ] ;
    s[ 6 ] = x.c[ 1 ] ;
    s[ 7 ] = x.c[ 0 ] ;

    return x.v ;
}


/**
*
*
*
*
**/
CORBA_long ALIGN_32( CORBA_long p )
{
    register int i = ( p % 4 ) ;

    if ( i )
        return p + ( 4 - i ) ;

    return p ;
}

/**
*
*
*
*
**/
CORBA_long ALIGN_64( CORBA_long p )
{
    register int i = ( p % 8 ) ;

    if ( i )
        return p + ( 8 - i ) ;

    return p ;
}

/**
*       CORBA_Dbuff_write
*       -----------------
*
*       General function to write data to the buffer
*
*
**/
int CORBA_Dbuff_write( CORBA_Dbuff * buff, char * u, CORBA_unsigned_long size, CORBA_unsigned_long num )
{
    CORBA_long next, now, required ;
    register unsigned char * p ;

    if ( CORBA_exceptionOn( buff ) )
        return -1 ;

    required = num * size ;

    now = buff->curr + buff->offset ;

    switch ( size )
    {
        case sizeof( CORBA_unsigned_short ) :
        next = ALIGN_16( now ) ;
        break ;

        case sizeof( CORBA_unsigned_long ) :
        next = ALIGN_32( now ) ;
        break ;

        case sizeof( CORBA_double ) :
        next = ALIGN_64( now ) ;
        break ;

        default :
        next = now ;
        break ;
    }

    next -= buff->offset ;

    if ( ( buff->len - next ) < required )
    {
        if ( CORBA_Dbuff_allocate( buff, required ) )
        {
            noMemory( buff ) ;
            return -1 ;
        }
    }

    p = &buff->p[ next ] ;

    memcpy( p, u, required ) ;

    buff->curr = next + required ;
    
    return required ;
}

/**
*       CORBA_Dbuff_read
*       ----------------
*
*       General function to read data from the buffer
*
*
**/
int CORBA_Dbuff_read( CORBA_Dbuff * buff, char * tgt, 
    CORBA_unsigned_long size, CORBA_unsigned_long num )
{
    CORBA_long next, now, required ;
    register unsigned char * p ;
    int i ;

    if ( CORBA_exceptionOn( buff ) )
        return -1 ;

    required = num * size ;

    /* calculate alignment for current position and offset */

    now = buff->curr + buff->offset ;

    switch ( size )
    {
        case sizeof( CORBA_unsigned_short ) :
        next = ALIGN_16( now ) ;
        break ;

        case sizeof( CORBA_unsigned_long ) :
        next = ALIGN_32( now ) ;
        break ;

        case sizeof( CORBA_double ) :
        next = ALIGN_64( now ) ;
        break ;

        default :
        next = now ;
        break ;
    }

    next -= buff->offset ;

    if ( ( buff->len - next ) < required )
    {
        if ( CORBA_Dbuff_allocate( buff, required ) )
        {
            noMemory( buff ) ;
            return -1 ;
        }
    }

    p = &buff->p[ next ] ;

    memcpy( tgt, p, required ) ;

    if ( buff->endian != ENDIAN )
    {
        for ( i = 0 ; i < num ; i++ )
        {
            switch ( size )
            {
                case sizeof( CORBA_short ) :
                CORBA_swap_short( ( CORBA_short * )tgt ) ;
                break ;

                case sizeof( CORBA_long ) :
                CORBA_swap_long( ( CORBA_long * )tgt ) ;
                break ;

                case sizeof( CORBA_long_long ) :
                CORBA_swap_long_long( ( CORBA_long_long * )tgt ) ;
                break ;

                default :
                i = num ;
                break ;
            }

            tgt += size ;
        }
    }

    buff->curr = next + required ;
    
    return required ;
}


/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_any_write( CORBA_Dbuff * b, CORBA_any * v )
{
    CORBA_long_write( b, v->_type.tc ) ;  /* check - is this right */

    switch ( v->_type.tc )
    {
        case CORBA_tk_short :
        return CORBA_short_write( b, *( ( CORBA_short * )v->_value ) ) ;

        case CORBA_tk_ushort :
        return CORBA_unsigned_short_write( b, *( ( CORBA_unsigned_short * )v->_value ) ) ;

        case CORBA_tk_long :
        return CORBA_long_write( b, *( ( CORBA_long * )v->_value ) ) ;

        case CORBA_tk_ulong :
        return CORBA_unsigned_long_write( b, *( ( CORBA_unsigned_long * )v->_value ) ) ;

        case CORBA_tk_float :
        return CORBA_float_write( b, *( ( CORBA_float * )v->_value ) ) ;

        case CORBA_tk_double :
        return CORBA_double_write( b, *( ( CORBA_double * )v->_value ) ) ;

        case CORBA_tk_boolean :
        return CORBA_boolean_write( b, *( ( CORBA_boolean * )v->_value ) ) ;

        case CORBA_tk_char :
        return CORBA_char_write( b, *( ( CORBA_char * )v->_value ) ) ;

        case CORBA_tk_octet :
        return CORBA_octet_write( b, *( ( CORBA_octet * )v->_value ) ) ;

        case CORBA_tk_enum :
        return CORBA_long_write( b, *( ( CORBA_long * )v->_value ) ) ;

        case CORBA_tk_longlong :
        return CORBA_long_long_write( b, *( ( CORBA_long_long * )v->_value ) ) ;

        case CORBA_tk_ulonglong :
        return CORBA_unsigned_long_long_write( b, *( ( CORBA_unsigned_long_long * )v->_value ) ) ;

        case CORBA_tk_longdouble :
        return CORBA_long_double_write( b, *( ( CORBA_long_double * )v->_value ) ) ;

        case CORBA_tk_wchar :
        return CORBA_wchar_write( b, *( ( CORBA_wchar * )v->_value ) ) ;

        case CORBA_tk_string :
        CORBA_long_write( b, 0 ) ;  /* ignore length - unbounded */
        return CORBA_string_write( b, ( CORBA_char * )v->_value, 0 ) ;

        case CORBA_tk_wstring :
        CORBA_long_write( b, 0 ) ;  /* ignore length - unbounded */
        return CORBA_wstring_write( b, ( CORBA_wchar * )v->_value, 0 ) ;
    }

    return -1 ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_any_read( CORBA_Dbuff * b, CORBA_any * v )
{
    CORBA_long len ;

    int r = CORBA_long_read( b, ( CORBA_long * )&v->_type.tc ) ;  /* check - is this right */

    if ( CORBA_any_get_release( v ) )
        CORBA_free( v->_value ) ;

    v->_value = NULL ;

    switch ( v->_type.tc )
    {
        case CORBA_tk_short :
        case CORBA_tk_ushort :

        case CORBA_tk_long :
        case CORBA_tk_ulong :
        case CORBA_tk_float :

        case CORBA_tk_double :
        case CORBA_tk_boolean :
        case CORBA_tk_char :
        case CORBA_tk_octet :

        case CORBA_tk_enum :
        case CORBA_tk_longlong :
        case CORBA_tk_ulonglong :
        case CORBA_tk_longdouble :
        case CORBA_tk_wchar :

        v->_value = CORBA_alloc( CORBA_MA_FIXED, 16, CORBA_FALSE ) ;
        CORBA_any_set_release( v, CORBA_TRUE ) ;
        break ;
    }

    switch ( v->_type.tc )
    {
        case CORBA_tk_short :
        r = CORBA_short_read( b, ( CORBA_short * )v->_value ) ;
        break ;

        case CORBA_tk_ushort :
        r = CORBA_unsigned_short_read( b, ( CORBA_unsigned_short * )v->_value ) ;
        break ;

        case CORBA_tk_long :
        r = CORBA_long_read( b, ( CORBA_long * )v->_value ) ;
        break ;

        case CORBA_tk_ulong :
        r = CORBA_unsigned_long_read( b, ( CORBA_unsigned_long * )v->_value ) ;
        break ;

        case CORBA_tk_float :
        r = CORBA_float_read( b, ( CORBA_float * )v->_value ) ;
        break ;

        case CORBA_tk_double :
        r = CORBA_double_read( b, ( CORBA_double * )v->_value ) ;
        break ;

        case CORBA_tk_boolean :
        r = CORBA_boolean_read( b, ( CORBA_boolean * )v->_value ) ;
        break ;

        case CORBA_tk_char :
        r = CORBA_char_read( b, ( CORBA_char * )v->_value ) ;
        break ;

        case CORBA_tk_octet :
        r = CORBA_octet_read( b, ( CORBA_octet * )v->_value ) ;
        break ;

        case CORBA_tk_enum :
        r = CORBA_long_read( b, ( CORBA_long * )v->_value ) ;
        break ;

        case CORBA_tk_longlong :
        r = CORBA_long_long_read( b, ( CORBA_long_long * )v->_value ) ;
        break ;

        case CORBA_tk_ulonglong :
        r = CORBA_unsigned_long_long_read( b, ( CORBA_unsigned_long_long * )v->_value ) ;
        break ;

        case CORBA_tk_longdouble :
        r = CORBA_long_double_read( b, ( CORBA_long_double * )v->_value ) ;
        break ;

        case CORBA_tk_wchar :
        r = CORBA_wchar_read( b, ( CORBA_wchar * )v->_value ) ;
        break ;

        case CORBA_tk_string :
        {
            if ( ( r = CORBA_long_read( b, &len ) ) != -1 ) 
                r = CORBA_string_read( b, ( CORBA_char ** )&v->_value, len ) ;
        }
        break ;

        case CORBA_tk_wstring :
        {
            if ( ( r = CORBA_long_read( b, &len ) ) != -1 ) 
                r = CORBA_wstring_read( b, ( CORBA_wchar ** )v->_value, len ) ;
        }
        break ;
    }

    CORBA_any_set_release( v, CORBA_TRUE ) ;
    
    return r ;
}

/**
*   Shallow copy of typecode
*
**/
MSC_EXPORT void BC_EXPORT CORBA_s_copy_TypeCode( CORBA_TypeCode * target, CORBA_TypeCode * src )
{
    memcpy( target, src, sizeof( CORBA_TypeCode ) ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Fixed_Type_Read( CORBA_Dbuff * b, CORBA_char * v, CORBA_unsigned_short size )
{
    return CORBA_Dbuff_read( b, v, sizeof( CORBA_char ), size ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Fixed_Type_Write( CORBA_Dbuff * b, CORBA_char * v, CORBA_unsigned_short size )
{
    return CORBA_Dbuff_write( b, v, sizeof( CORBA_char ), size ) ;
}


/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_boolean_read( CORBA_Dbuff * b, CORBA_boolean * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_boolean ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_boolean_write( CORBA_Dbuff * b, CORBA_boolean v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_boolean ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_octet_read( CORBA_Dbuff * b, CORBA_octet * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_octet ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_octet_write( CORBA_Dbuff * b, CORBA_octet v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_octet ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_char_read( CORBA_Dbuff * b, CORBA_char * v )
{
    return CORBA_Dbuff_read( b, v, sizeof( CORBA_char ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_char_write( CORBA_Dbuff * b, CORBA_char v )
{
    return CORBA_Dbuff_write( b, &v, sizeof( CORBA_char ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_wchar_read( CORBA_Dbuff * b, CORBA_wchar * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_wchar ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_wchar_write( CORBA_Dbuff * b, CORBA_wchar v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_wchar ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_short_read( CORBA_Dbuff * b, CORBA_short * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_short ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_short_write( CORBA_Dbuff * b, CORBA_short v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_short ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_long_write( CORBA_Dbuff * b, CORBA_long v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_long_read( CORBA_Dbuff * b, CORBA_long * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_unsigned_short_write( CORBA_Dbuff * b, CORBA_unsigned_short v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_unsigned_short ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_unsigned_short_read( CORBA_Dbuff * b, CORBA_unsigned_short * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_unsigned_short ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_unsigned_long_write( CORBA_Dbuff * b, CORBA_unsigned_long v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_unsigned_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_unsigned_long_read( CORBA_Dbuff * b, CORBA_unsigned_long * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_unsigned_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_long_long_write( CORBA_Dbuff * b, CORBA_long_long v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_long_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_long_long_read( CORBA_Dbuff * b, CORBA_long_long * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_long_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_unsigned_long_long_read( CORBA_Dbuff * b, CORBA_unsigned_long_long * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_unsigned_long_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_unsigned_long_long_write( CORBA_Dbuff * b, CORBA_unsigned_long_long v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_unsigned_long_long ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_float_write( CORBA_Dbuff * b, CORBA_float v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_float ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_float_read( CORBA_Dbuff * b, CORBA_float * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_float ), 1  ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_double_write( CORBA_Dbuff * b, CORBA_double v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_double ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_double_read( CORBA_Dbuff * b, CORBA_double * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_double ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_long_double_read( CORBA_Dbuff * b, CORBA_long_double * v )
{
    return CORBA_Dbuff_read( b, ( char * )v, sizeof( CORBA_long_double ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_long_double_write( CORBA_Dbuff * b, CORBA_long_double v )
{
    return CORBA_Dbuff_write( b, ( char * )&v, sizeof( CORBA_long_double ), 1 ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_string_write( CORBA_Dbuff * b, CORBA_char * v, CORBA_long max )
{
    CORBA_long size = strlen( v ) + 1 ;

    if ( max == -1 )
        max = b->bounds ;
        
    if ( ( max != 0 ) && ( size > max ) )
    {
        CORBA_throw_system_exception( THROW_BENV( b ), EX_CORBA_MARSHAL, 0, CORBA_COMPLETED_NO ) ;
        return -1 ;
    }
    
    CORBA_Dbuff_write( b, ( char * )&size, sizeof( CORBA_long ), 1 ) ;

    return CORBA_Dbuff_write( b, v, sizeof( CORBA_char ), size ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_string_read( CORBA_Dbuff * b, CORBA_char ** v, CORBA_long max )
{
    CORBA_long size ;
    CORBA_char * s ;

    int r = CORBA_long_read( b, &size ) ;

    if ( r >= 0 )
    {
        if ( max == -1 )
            max = b->bounds ;
        
        if ( ( max != 0 ) && ( size > max ) )
        {
            CORBA_throw_system_exception( THROW_BENV( b ), EX_CORBA_MARSHAL, 0, CORBA_COMPLETED_NO ) ;
            return -1 ;
        }
        
        if ( *v )
            CORBA_free( *v ) ;

        if ( ( s = CORBA_string_alloc( size ) ) == NULL )
        {
            noMemory( b ) ;
            return -1 ;
        }

        if ( ( r = CORBA_Dbuff_read( b, s, sizeof( CORBA_char ), size ) ) == -1 )
            CORBA_free( s ) ;
        else
            *v = s ;
    }

    return r ;
}


/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_wstring_write( CORBA_Dbuff * b, CORBA_wchar * v, CORBA_long max )
{
    CORBA_long size = CORBA_wstring_len( v ) ;

    if ( max == -1 )
        max = b->bounds ;
        
    if ( ( max != 0 ) && ( size > max ) )
    {
        CORBA_throw_system_exception( THROW_BENV( b ), EX_CORBA_MARSHAL, 0, CORBA_COMPLETED_NO ) ;
        return -1 ;
    }

    CORBA_Dbuff_write( b, ( char * )&size, sizeof( CORBA_long ), 1 ) ;

    return CORBA_Dbuff_write( b, ( CORBA_char * )v, sizeof( CORBA_wchar ), size ) ;
}

MSC_EXPORT int BC_EXPORT CORBA_wstring_read( CORBA_Dbuff * b, CORBA_wchar ** v, CORBA_long max )
{
    CORBA_long size ;
    CORBA_wchar * s ;

    int r = CORBA_long_read( b, &size ) ;

    if ( r < 0 )
        return r ;

    if ( max == -1 )
        max = b->bounds ;
        
    if ( ( max != 0 ) && ( size > max ) )
    {
        CORBA_throw_system_exception( THROW_BENV( b ), EX_CORBA_MARSHAL, 0, CORBA_COMPLETED_NO ) ;
        return -1 ;
    }

    if ( *v )
        CORBA_free( *v ) ;

    s = CORBA_wstring_alloc( size ) ;

    r = CORBA_Dbuff_read( b, ( CORBA_char * )s, sizeof( CORBA_wchar ), size ) ;

    s[ size * 2 ] = '\0' ;

    *v = s ;
    
    return r ;
}


/*
static void readUTF( CORBA_iObject * o )
{
    CORBA_unsigned_short utflen ;
    CORBA_wchar * s ;
    int count = 0 ;
    int strlen = 0 ;
    int c, char2, char3 ;
    char * buffer ;

    CORBA_unsigned_short_read( o, &utflen ) ;

    buffer = CORBA_wstring_alloc( utflen ) ;

    CORBA_read( o, s, sizeof( CORBA_char ), utflen ) ;

    for ( int i = 0 ; count < utflen ; i++ )
    {
        c = buffer[ i ] & 0xFF ;

        switch ( c >> 4 )
        {
            case 0 :
            case 1 :
            case 2 :
            case 3 :
            case 4 :
            case 5 :
            case 6 :
            case 7 :
            {
                count++ ;
                str[ strlen++ ] = ( char )c ;
            }
            break ;

            case 12 :
            case 13 :
            {
                // 110x xxxx   10xx xxxx

                count += 2 ;

                if ( count > utflen )
                    throw new org.omg.CORBA.MARSHAL( "Bad UTF format (1)" ) ;

                char2 = buffer[ ++i ] & 0xFF ;

                if ( ( char2 & 0xC0 ) != 0x80 )
                    throw new org.omg.CORBA.MARSHAL( "Bad UTF format (2)" ) ;

                str[ strlen++ ] = ( char )( ( ( c & 0x1F ) << 6 ) | ( char2 & 0x3F ) ) ;
            }
            break ;

            case 14 :
            {
                // 1110 xxxx  10xx xxxx  10xx xxxx

                count += 3 ;

                if ( count > utflen )
                    throw new org.omg.CORBA.MARSHAL( "Bad UTF format (3)" ) ;

                char2 = buffer[ ++i ] & 0xFF ;
                char3 = buffer[ ++i ] & 0xFF ;

                if ( ( ( char2 & 0xC0 ) != 0x80 ) || ( ( char3 & 0xC0 ) != 0x80 ) )
                    throw new org.omg.CORBA.MARSHAL( "Bad UTF format (4)" ) ;

                str[ strlen++ ] = ( char )( ( ( c & 0x0F ) << 12 ) |
                                            ( ( char2 & 0x3F ) << 6 ) |
                                            ( ( char3 & 0x3F ) << 0 ) ) ;
            }
            break ;

            default :
            // 10xx xxxx,  1111 xxxx
            throw new org.omg.CORBA.MARSHAL( "Bad UTF format (5)" ) ;
        }
    }

    return str ;
}

*/
 
#define CHECK_SEQ_LEN( S, LEN, BUFF ) if ( ( S->_maximum > 0 ) && ( LEN > S->_maximum ) ) { \
        CORBA_throw_system_exception( THROW_BENV( BUFF ), EX_CORBA_MARSHAL, 0, CORBA_COMPLETED_NO ) ; \
        return -1 ; \
    }

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Basic_Sequence_read( CORBA_Dbuff * b, CORBA_Generic_Sequence * s,
    CORBA_unsigned_long size )
{
    CORBA_long l ;
    char * seq ;

    int r = CORBA_long_read( b, &l ) ;

    if ( r < 0 )
        return r ;

    CHECK_SEQ_LEN( s, l, b )

    if ( l > s->_length )
    {
        if ( s->_buffer )
            CORBA_free( s->_buffer ) ;

        s->_buffer = CORBA_seqBuffAlloc( NULL, size, l ) ;
    }

    s->_length = l ;

    seq = s->_buffer ;

    return CORBA_Dbuff_read( b, seq, size, l ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Basic_Sequence_write( CORBA_Dbuff * b,
    CORBA_Generic_Sequence * s, CORBA_unsigned_long size )
{
    CORBA_long l = s->_length ;
    char * seq = s->_buffer ;

    CHECK_SEQ_LEN( s, l, b )

    CORBA_long_write( b, l ) ;

    return CORBA_Dbuff_write( b, seq, size, l ) ;
}


/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Generic_Sequence_read( CORBA_Dbuff * b, CORBA_Generic_Sequence * s,
    CORBA_rwFunc func, CORBA_unsigned_long size, CORBA_seqAllocFunc allocFunc )
{
    CORBA_long i, l ;
    char * seq ;
    int err ; 

    int r = CORBA_long_read( b, &l ) ;

    if ( r < 0 )
        return r ;
        
    CHECK_SEQ_LEN( s, l, b )

    seq = ( allocFunc )( l ) ;

    if ( s->_buffer )
        sequenceFree( s ) ;

    s->_length = l ;
    s->_buffer = seq ;

    for ( i = 0 ; i < l ; i++ )
    {
        if ( ( err = ( func )( b, seq ) ) == -1 )
        {
            r = err ;
            break ;
        }
        
        r += err ;
        
        seq += size ;
    }
    
    return r ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Generic_Sequence_write( CORBA_Dbuff * b,
    CORBA_Generic_Sequence * s, CORBA_rwFunc func, CORBA_unsigned_long size )
{
    CORBA_long i, l = s->_length ;
    char * seq = s->_buffer ;
    int r, err ;
    
    CHECK_SEQ_LEN( s, l, b )

    r = CORBA_long_write( b, l ) ;

    for ( i = 0 ; i < l ; i++ )
    {
        if ( ( err = ( func )( b, seq ) ) == -1  )
        {
            r = -1 ;
            break ;
        }            

        r += err ;
        
        seq += size ;
    }
    
    return r ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Context_write( CORBA_Dbuff * b, CORBA_Context * ctx )
{
    ( void )b ;
    ( void )ctx ;
    return 0 ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Context_read( CORBA_Dbuff * b, CORBA_Context * ctx )
{
    ( void )b ;
    ( void )ctx ;
    return 0 ;
}


/**
*       ORB FUNCTIONS
*       -------------
**/

/**
*
*   CORBA_ORB_init
*   --------------
*
*   Initialize the ORB
*
**/
MSC_EXPORT CORBA_ORB BC_EXPORT CORBA_ORB_init( int * argc, char ** argv,
    CORBA_ORBid orb_identifier, CORBA_Environment * ev )
{
    CORBA_iORB * orb = ( CORBA_iORB * )CORBA_iORB_init( argc, argv, orb_identifier, ev ) ;

    FILE * fp ;
    
    if ( orb->logName )
    {
        fp = fopen( orb->logName, "a" ) ;
        
        if ( fp )
            orb->log = fp ;
    }
    else
        orb->log = stderr ;

    return orb ;
}

/**
*
*
*
*
**/
void initMsgHdr( GIOP_MessageHeader_1_1 * hdr, CORBA_octet minor )
{
    static GIOP_MessageHeader_1_1 h = { "GIOP", { 1, 1 }, ENDIAN, 0, 0 } ;

    memcpy( hdr, &h, sizeof( h ) ) ;

    hdr->GIOP_version.minor = minor ;
}

/**
*
*
*
*
**/
CORBA_BOA_OBJS * CORBA_BOA_find_slot( CORBA_BOA boa, CORBA_iObject * o )
{
    int i, conns ;
    CORBA_BOA_OBJS * bo ;

    for ( i = 0, bo = boa->objs, conns = boa->params.connections ;
          i < conns ; i++, bo++ )
    {
        if ( bo->obj == NULL )
        {
            bo->obj        = o ;
            bo->num        = 1 ;
            o->Oadaptor    = boa ;
            o->adaptorSlot = i ;
            bo->channel    = -1 ;
            bo->socket     = -1 ;
            return bo ;
        }            
    }

    return NULL ;
}

/**
*
*
*
*
**/
static void CORBA_BOA_release_slot( CORBA_BOA boa, CORBA_iObject * o )
{
    int slot = o->adaptorSlot ;

    CORBA_BOA_OBJS * bo = &boa->objs[ slot ] ;

    if ( slot >= boa->params.connections )
    {
        CORBA_ORB_log( o, LOG_SYS_ERR, "bad slot" ) ;
    }
    else
    {
        if ( --bo->num == 0 )
        {
            o->adaptorSlot = -1 ;
        
            if ( bo->channel >= 0 )
                CORBA_Disconnect( bo->channel ) ;

            CORBA_free_IOR( &bo->ior ) ;
        }            
    }            
}

#define writeExceptionId( B, ID ) CORBA_string_write( B, ID, 0 )

#define readExceptionId( B ) CORBA_string_read( B, &B->env.id, 0 )

/**
*
*
*
*
**/
static void writeSysException( CORBA_Dbuff * b, CORBA_Environment * ev )
{
    CORBA_unsigned_long_write( b, ev->sysExcept.minor_code_value ) ;
    CORBA_unsigned_long_write( b, ev->sysExcept.completion_status ) ;
}

/**
*
*
*
*
**/
static void readSysException( CORBA_Dbuff * b )
{
    CORBA_System_Exception sysE ;

    CORBA_unsigned_long_read( b, &sysE.minor_code_value ) ;
    CORBA_unsigned_long_read( b, &sysE.completion_status ) ;

    if ( ! CORBA_exceptionOn( b ) )
        b->env.sysExcept = sysE ;
}

/**
*
*
*
*
**/
static void replyWrite( CORBA_Dbuff * b, CORBA_long reqId, CORBA_unsigned_long reply_status )
{
    CORBA_long_write( b, 0 ) ; /* context stuff */
    CORBA_long_write( b, reqId ) ;
    CORBA_long_write( b, reply_status ) ; /* reply status = OK */
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_Object BC_EXPORT CORBA_create_implementation( void * oa, CORBA_string service, 
    CORBA_iIFACE_TABLES * tables, CORBA_Environment * ev )
{
    CORBA_iObject * o = CORBA_alloc_object() ;

    if ( o )
    {
        initMsgHdr( &o->msgHdr, 1 ) ;

        ev->_major = CORBA_NO_EXCEPTION ;

        o->server = tables ;

        _CORBA_BOA_object_is_ready( oa, o, service, ev ) ;
    }
    else
        noMemory( &o->buff ) ;

    return o ;
}

/**
*
*
*
*
**/
void _CORBA_BOA_object_is_ready( CORBA_BOA boa, CORBA_iObject * obj,
    CORBA_char * service, CORBA_Environment * env )
{
    CORBA_BOA_OBJS * o ;
    CORBA_boolean reconnect = CORBA_FALSE ;
    CORBA_long port, localAddr ;

    if ( boa == NULL )
    {
        CORBA_throw_system_exception( THROW_OENV( obj ), EX_CORBA_INITIALIZE, 0, CORBA_COMPLETED_NO ) ;
        return ;
    }

    if ( obj->adaptorSlot < 0 )
    {
        if ( ( o = CORBA_BOA_find_slot( boa, obj ) ) == NULL )
        {
            CORBA_throw_system_exception( THROW_OENV( obj ), EX_CORBA_INITIALIZE, 0, CORBA_COMPLETED_NO ) ;
            return ;
        }
    }
    else
    {
        reconnect = CORBA_TRUE ;
        o = &boa->objs[ obj->adaptorSlot ] ;
    }

    if ( env )
        env->_major = CORBA_NO_EXCEPTION ;

    if ( o->socket < 0 ) 
    {
        /* no server socket */

        initMsgHdr( &obj->msgHdr, 1 ) ;

        port = 0 ; 

        if ( ( o->socket = CORBA_getSocket( port ) ) < 0 )
            CORBA_throw_system_exception( THROW_OENV( obj ), EX_CORBA_INITIALIZE, 0, CORBA_COMPLETED_NO ) ;
        else
        {
            o->isServer = CORBA_TRUE ;
            
            CORBA_getSockAddr( o->socket, &localAddr, &port ) ;

            localAddr = CORBA_getLocalAddress() ;

            o->ior.id                         = CORBA_string_copy( service ) ;
            o->ior.pb.iiop_version.minor      = 1 ;
            o->ior.pb.iiop_version.major      = 1 ;
            o->ior.pb.port                    = ( unsigned short )port ;
            o->ior.pb.host                    =  CORBA_inetAddr( localAddr, 0 ) ;

            o->ior.pb.object_key._length      = 1 ;
            o->ior.pb.object_key._buffer      = CORBA_calloc( 1, 1 ) ;
            o->ior.pb.object_key._buffer[ 0 ] = ( CORBA_octet )obj->adaptorSlot ;
        }
    }

    if ( o->socket >= 0 ) 
    {
        if ( ! reconnect )
            CORBA_Listen( o->socket ) ;
#if 0
        if ( engroom_CORBA_ORBagent_implementation( orb, obj->id, &o->port, &obj->idNum, env ) )
            CORBA_iSetSysException( ex_CORBA_INITIALIZE, 0, env ) ;
#endif
    }
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_BOA_implementation_is_ready( CORBA_BOA boa, 
    CORBA_Environment * env )
{
    int r ;
    
    env->_major = CORBA_NO_EXCEPTION ;

    if ( boa )
    {
        for ( ; ; )
        {
            if ( ( r = serverLoop( boa, env ) ) != 0 ) 
                return r ;

            if ( env->_major )
                break ;
        }
    }
    else
        CORBA_throw_system_exception( env, EX_CORBA_INITIALIZE, 0, CORBA_COMPLETED_NO ) ;

    return 0 ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_boolean BC_EXPORT CORBA_check_sys_exception( CORBA_iObject * o )
{
    CORBA_Environment * env ;

    if ( CORBA_exceptionOnObj( o ) )
    {
        env = &CORBA_OBJ_ENV( o );

        CORBA_BOA_set_exception( o, env, env->_major, env->id, &env->sysExcept ) ;

        return CORBA_TRUE ;
    }

    return CORBA_FALSE ;
}


/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_BOA_set_exception( CORBA_Object obj, CORBA_Environment * ev,
    CORBA_exception_type major, CORBA_char * exceptname, void * param )
{
    CORBA_iObject * o = ( CORBA_iObject * )obj ;
    CORBA_Dbuff * buff = &o->buff ;
    char * opName = o->req.opName ;

    ev->_major = CORBA_NO_EXCEPTION ;  /* reset otherwise the exception might not get written */

    switch ( major )
    {
        case CORBA_NO_EXCEPTION     :
        return ;

        case CORBA_SYSTEM_EXCEPTION :
        break ;

        case CORBA_USER_EXCEPTION   :
        if ( o->exceptions == NULL )
        {
            CORBA_ORB_log( obj, LOG_SYS_ERR, "Can't raise user exceptions on operation %s", opName ) ;
            return ;
        }
        break ;

        default :
        CORBA_ORB_log( obj, LOG_SYS_ERR, "Unknown major exception type %d", major ) ;
        return ;
    }

    if ( ! o->req.response_expected )
    {
        CORBA_ORB_log( obj, LOG_SYS_ERR, "Can't raise exception - no response expected on operation %s", opName ) ;
        return ;
    }

    CORBA_postImpl( o ) ;

    replyWrite( buff, o->req.request_id, major ) ;

    writeExceptionId( buff, exceptname ) ;

    if ( major == CORBA_SYSTEM_EXCEPTION )
    {
        if ( param )
            ev->sysExcept = *( ( CORBA_System_Exception * )param ) ;
        else
            memset( &ev->sysExcept, 0, sizeof( CORBA_System_Exception ) ) ;

        writeSysException( buff, ev ) ;
    }
    else
    {
        if ( ( o->exceptions )( o, exceptname, param ) )
        {
            CORBA_ORB_log( o, LOG_SYS_ERR, "Error raising exception %s on operation %s",
                exceptname, opName ) ;
            return ;
        }
    }

    ev->_major = major ;
}

/**
*
*
*
*
**/
void CORBA_unknownException( CORBA_iObject * o )
{
    CORBA_System_Exception e ;

    e.completion_status = CORBA_COMPLETED_MAYBE ;
    e.minor_code_value = 0 ;

    CORBA_BOA_set_exception( o, &CORBA_OBJ_ENV( o ), CORBA_SYSTEM_EXCEPTION,
        ex_CORBA_UNKNOWN, &e ) ;
}

/**
*
*
*
*
**/
static void noMemory( CORBA_Dbuff * b )
{
    CORBA_throw_system_exception( THROW_BENV( b ), EX_CORBA_NO_MEMORY, 0, CORBA_COMPLETED_NO ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_char * BC_EXPORT CORBA_exception_id( CORBA_Environment * ev )
{
    return ev->id ;
}

/**
*
*
*
*
**/
MSC_EXPORT void * BC_EXPORT CORBA_exception_value( CORBA_Environment * ev )
{
    if ( ev->_major == CORBA_SYSTEM_EXCEPTION )
        return &ev->sysExcept ;

    return ev->value ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_exception_free( CORBA_Environment * ev )
{
    if ( ev->id )
        CORBA_free( ev->id ) ;

    if ( ev->value )
        CORBA_free( ev->value ) ;
}

/**
*
*       CORBA_request
*       -------------
*
*       Generate a request
*
**/
MSC_EXPORT int BC_EXPORT CORBA_request( CORBA_iObject * o, CORBA_iClientExceptionFunc fnc )
{
    int r, channel = OBJ_CHANNEL( o ) ;

    if ( CORBA_send_message( o, &o->buff, channel, GIOP_MsgType_1_1_Request ) )
        return -1 ;

    if ( o->req.response_expected )
    {
        if ( ( r = CORBA_receive_message( o, channel ) ) != 0 )
        {
            return r ;
        }            
        else
        {
            CORBA_reply_read( o, fnc ) ;
        }
    }

    return 0 ;
}


/**
*
*
*
*
**/
void CORBA_reply_read( CORBA_iObject * o, CORBA_iClientExceptionFunc fnc )
{
    CORBA_long discard ;
    CORBA_unsigned_long reply_status ;
    CORBA_Dbuff * buff = &o->buff ;

    CORBA_long_read( buff, &discard ) ; /* context stuff */
    CORBA_long_read( buff, &discard ) ;
    CORBA_unsigned_long_read( buff, &reply_status ) ; /* reply status */

    if ( reply_status != CORBA_NO_EXCEPTION )
    {
        CORBA_OBJ_ENV( o ).value = NULL ;
        CORBA_OBJ_ENV( o ).id    = NULL ;

        readExceptionId( buff ) ;

        if ( reply_status == CORBA_SYSTEM_EXCEPTION )
        {
            readSysException( buff ) ;
        }
        else if ( reply_status == CORBA_USER_EXCEPTION )
        {
            if ( fnc )
            {
                if ( ( fnc )( o, CORBA_OBJ_ENV( o ).id ) )
                    CORBA_ORB_log( o, LOG_SYS_ERR, "Error Reading exception %s", CORBA_OBJ_ENV( o ).id ) ;
            }
            else
                CORBA_ORB_log( o, LOG_SYS_ERR, "Incoming User exception! - Server object error" ) ;
        }
    }

    CORBA_OBJ_ENV( o )._major = ( CORBA_exception_type )reply_status ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_Reply_Write( CORBA_iObject * o )
{
    CORBA_Dbuff * buff = &o->buff ;
    
    buff->endian = ENDIAN ;    
    CORBA_long_write( buff, 0 ) ;                   /* context stuff     */
    CORBA_long_write( buff, o->req.request_id ) ;   /* request id        */
    CORBA_long_write( buff, 0 ) ;                   /* reply status = OK */
}

/**
*
*
*
*
**/
void CORBA_request_init( CORBA_iRequest * r )
{
    memset( r, 0, sizeof( CORBA_iRequest ) ) ;
}

/**
*
*
*
*
**/
void CORBA_Request_read( CORBA_Dbuff * buff, CORBA_iRequest * req )
{
    char * op = NULL ;
    CORBA_long discard ;

    CORBA_long_read( buff, &discard ) ; /* context stuff */
    CORBA_long_read( buff, &req->request_id ) ;
    CORBA_boolean_read( buff, &req->response_expected ) ;
    CORBA_Sequence_octet_read( buff, &req->object_key ) ; /* object key */
    CORBA_string_read( buff, &op, 0 ) ;
    req->opName = op ;
    CORBA_long_read( buff, &discard ) ; /* principal */
}

/**
*
*
*
*
**/
void CORBA_request_free( CORBA_iRequest * req )
{
    CORBA_free( req->opName ) ;
    CORBA_conditionalFree( req->object_key._buffer ) ; 

    req->opName             = NULL ;
    req->object_key._buffer = NULL ; 
    req->object_key._length = 0 ; 
}

/**
*
*
*
*
**/
void CORBA_request_read( CORBA_iObject * o )
{
    CORBA_iRequest * req = &o->req ;
    CORBA_Dbuff * buff = &o->buff ;
    
    CORBA_Request_read( buff, req ) ;
}

#define CORBA_reset_exception( O ) O->buff.env._major = CORBA_NO_EXCEPTION

/**
*
*
*
*
**/
void CORBA_Request_write( CORBA_Dbuff * buff, CORBA_iRequest * req )
{
    CORBA_resetpBuff( buff ) ;

    buff->offset = GIOP_MESSAGEHEADER_SIZE ;
    buff->endian = ENDIAN ;
    
    CORBA_long_write( buff, 0 ) ;                      /* context stuff */
    CORBA_long_write( buff, req->request_id ) ;
    CORBA_boolean_write( buff, req->response_expected ) ;
    CORBA_Sequence_octet_write( buff, &req->object_key ) ;
    CORBA_string_write( buff, req->opName, 0 ) ;
    CORBA_long_write( buff, 0 ) ;                      /* principal */
}

/**
*
*
*
*
**/
void CORBA_Request_set( CORBA_iRequest * req, CORBA_iObject * o )
{
    static CORBA_unsigned_long request_id = 0 ;  /* - Thread protect !! */
    CORBA_BOA_OBJS * bobj = &BOA_OBJ( o ) ;
    CORBA_IOR * ior = &bobj->ior ;

    req->request_id         = ++request_id ;
    req->object_key         = ior->pb.object_key ;
    req->ctxList._length    = 0 ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_request_write( CORBA_iObject * o, char * operation, CORBA_boolean response )
{
    CORBA_Request_set( &o->req, o ) ;

    o->req.response_expected  = response ;
    o->req.opName             = operation ;

    CORBA_reset_exception( o ) ;

    CORBA_Request_write( &o->buff, &o->req ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_set_exception_handler( CORBA_iObject * o,
    CORBA_iServerExceptionFunc fnc )
{
    o->exceptions = fnc ;
}

/* TypeCode Functions */

/**
*
*
*
*
**/
MSC_EXPORT CORBA_boolean BC_EXPORT CORBA_TypeCode_equal( CORBA_TypeCode * t1, CORBA_TypeCode * t2 )
{
    if ( t1->tc != t2->tc )
        return CORBA_FALSE ;
        
    switch ( t1->tc )
    {
        case CORBA_tk_string     :
        case CORBA_tk_wstring    :
        {
            if ( t1->_ptr != t2->_ptr )
                return CORBA_FALSE ;
        }                
        break ;
        
        case CORBA_tk_struct     :
        case CORBA_tk_union      :
        case CORBA_tk_enum       :
        case CORBA_tk_sequence   :
        case CORBA_tk_array      :
        case CORBA_tk_alias      :
        case CORBA_tk_except     :
        case CORBA_tk_fixed      :
        return CORBA_FALSE ;

        case CORBA_tk_TypeCode   :
        case CORBA_tk_Principal  :
        case CORBA_tk_objref     :
        break ;
        
        case CORBA_tk_null       :
        case CORBA_tk_void       :
        case CORBA_tk_short      :
        case CORBA_tk_long       :
        case CORBA_tk_ushort     :
        case CORBA_tk_ulong      :
        case CORBA_tk_float      :
        case CORBA_tk_double     :
        case CORBA_tk_boolean    :
        case CORBA_tk_char       :
        case CORBA_tk_octet      :
        case CORBA_tk_any        :
        case CORBA_tk_longlong   :
        case CORBA_tk_ulonglong  :
        case CORBA_tk_longdouble :
        case CORBA_tk_wchar      :
        case CORBA_tk_none       :
        break ;
    }

    return CORBA_TRUE ;
}

/**
*
*   CORBA_get_log
*   --------------
*
*
**/
void * CORBA_get_log( CORBA_iObject * o )
{
    CORBA_iORB * orb = ( CORBA_iORB * )ORB_FROM_OBJ( o ) ;
    
    if ( orb->log )
        return orb->log ;

    return NULL ;
}


/* WARNING - ONLY FOR ASCII CHARACTER SETS */

#define HEX2NIBBLE(A) (isdigit(A)?((A)-'0'):(toupper(A)-'A'+10))

#define NIBBLE2HEX(A) ( char )((A)<10?((A)+'0'):((A)-10+'A'))

/**
*
*
*
*
**/
static int hex2buff( char * hex, unsigned char * buff )
{
    int i, len = strlen( hex ) ;
    unsigned char s, t ;
    
    if ( len % 2 )
        return -1 ;  /* should be an even number of bytes */
        
    len /= 2;
    
    for ( i = 0 ; i < len ; i++, hex++, buff++ )
    {
        s = *hex++ ;

        if ( isxdigit( s ) )
            t = ( unsigned char )( ( HEX2NIBBLE( s ) << 4 ) & 0xF0 ) ;
        else
            return -1 ;

        s = *hex ;

        if ( isxdigit( s ) )
            t = ( unsigned char )( t + HEX2NIBBLE( s ) ) ;
        else
            return -1 ;
            
        *buff = t ;            

        /* printf( "i=%d b=%d %x c=%c\n", i, t, t, (isalpha(t)?t:'?')); */
    }
    
    return len ;
}

/**
*
*
*
*
**/
static void buff2hex( unsigned char * buff, char * hex, int len )
{
    int i ;
    unsigned char s ;
    
    for ( i = 0 ; i < len ; i++, hex++, buff++ )
    {
        s = ( unsigned char )( ( *buff & 0xF0 ) >> 4 ) ;

        /* printf( "%02X %c %c", *buff, *buff, NIBBLE2HEX( s ) ) ; */

        *hex++ = NIBBLE2HEX( s ) ;

        s = ( unsigned char )( *buff & 0x0F ) ;

        /* printf( " %c\n", NIBBLE2HEX( s ) ) ; */

        *hex = NIBBLE2HEX( s ) ;
    }

    *hex = '\0' ;
}


/**
*
*
*
*
**/
static int read_hex_string( CORBA_Dbuff * buff, char * str )
{
    int len = strlen( str ) ;
    
    int require = ( len / 2 ) + 50 ; /* add 50 for null terminator and luck */
    
    if ( CORBA_Dbuff_allocate( buff, require ) ||
        ( hex2buff( str, buff->p ) < 0 ) )
    {
        return -1 ;
    }        

    return 0 ;
}


/**
*
*
*
*
**/
static void readProfileBody( IOP_TaggedProfile * tp, struct IIOP_ProfileBody_1_1 * pb )
{
    CORBA_Dbuff buff ; 
    
    memset( &buff, 0, sizeof( CORBA_Dbuff ) ) ;
    memset( pb, 0, sizeof( struct IIOP_ProfileBody_1_1 ) ) ;

    buff.p   = tp->profile_data._buffer ;
    buff.len = tp->profile_data._length ;
                
    CORBA_read_endian( &buff ) ;

    IIOP_ProfileBody_1_1_read( &buff, pb ) ;
}

/**
*
*
*
*
**/
static void writeProfileBody( struct IIOP_ProfileBody_1_1 * pb,
    CORBA_Sequence_octet * seq )
{
    CORBA_Dbuff buff ; 
    
    CORBA_Dbuff_Init( &buff ) ; 

    CORBA_write_endian( &buff ) ;

    IIOP_ProfileBody_1_1_write( &buff, pb ) ;

    seq->_length = buff.curr ;

    seq->_buffer = ( CORBA_octet * )CORBA_calloc( 1, seq->_length ) ;

    memcpy( seq->_buffer, buff.p, seq->_length ) ;
    
    CORBA_Dbuff_free( &buff ) ;
}

static const char IOR_normalPrefix[] = "IOR:" ;

static const int IORpfxLen = 4 ;

/**
*
*
*
*
**/
static CORBA_boolean CORBA_readIOR( CORBA_iObject * o, char * str )
{
    CORBA_Dbuff * buff ;
    CORBA_BOA_OBJS * bobj = &BOA_OBJ( o ) ;
    CORBA_IOR * ior = &bobj->ior ;
    IOP_IOR iop_ior ;
    
    memset( &iop_ior, 0, sizeof( struct IOP_IOR ) ) ;

    ior->tag = IOP_TAG_INTERNET_IOP ;

    if ( strncmp( str, IOR_normalPrefix, IORpfxLen ) == 0 )
    {
        /* it's an encapsulation */

        buff = &o->buff ;
        
        if ( read_hex_string( buff, str + IORpfxLen ) == 0 )
        {
            CORBA_read_endian( buff ) ;

            IOP_IOR_read( buff, &iop_ior ) ;

            if ( iop_ior.profiles._length > 0 )
                readProfileBody( iop_ior.profiles._buffer, &ior->pb ) ;

            ior->id = CORBA_string_copy( iop_ior.type_id ) ;
            
            IOP_IOR__free( &iop_ior ) ;

            return CORBA_TRUE ;
        }
    }

    return CORBA_FALSE ;
}

/**
*
*
*
*
**/
static CORBA_char * CORBA_createIOR( CORBA_iObject * o )
{
    CORBA_char * iorString ;
    IOP_IOR iop_ior ;
    int len ;

    CORBA_Dbuff * buff    = &o->buff ;
    CORBA_IOR * ior       = &BOA_OBJ( o ).ior ;
    IOP_TaggedProfile * tp  = CORBA_calloc( 1, sizeof( IOP_TaggedProfile ) ) ;

    buff->curr   = 0 ;
    buff->offset = 0 ;

    tp->tag = IOP_TAG_INTERNET_IOP ;

    writeProfileBody( &ior->pb, &tp->profile_data ) ;
    
    iop_ior.type_id          = CORBA_string_copy( ior->id ) ;
    iop_ior.profiles._length = 1 ;
    iop_ior.profiles._buffer = tp ;

    CORBA_write_endian( buff ) ;

    IOP_IOR_write( buff, &iop_ior ) ;

    IOP_TaggedProfile__free( tp ) ;
    IOP_IOR__free( &iop_ior ) ; 

    
    len = IORpfxLen + ( buff->curr * 2 ) + 1 ;

    if ( ( iorString = CORBA_string_alloc( len ) ) != NULL )
    {
        memcpy( iorString, IOR_normalPrefix, IORpfxLen ) ;

        buff2hex( buff->p, iorString + IORpfxLen, buff->curr ) ;
    }

    return iorString ;
}

/**
*
*
*
*
**/
static void CORBA_connect_client_object( CORBA_iObject * o )
{
    CORBA_BOA_OBJS * op = &BOA_OBJ( o ) ;

    int socket ;

    if ( ( socket = CORBA_ClientConnect( op->ior.pb.host, 0, 
                                         op->ior.pb.port ) ) < 0 )
    {
        CORBA_throw_system_exception( THROW_OENV( o ), EX_CORBA_COMM_FAILURE, 
            COI_MIN_NO_CLIENT_CONNECT, CORBA_COMPLETED_NO ) ;
    }
    else
        op->channel = socket ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_char * BC_EXPORT CORBA_ORB_object_to_string( CORBA_ORB orb, 
    CORBA_Object obj, CORBA_Environment * ev )
{
    CORBA_iObject * o = ( CORBA_iObject * )obj ;
    CORBA_char * s = NULL ;
    CORBA_Environment * env = &CORBA_OBJ_ENV( ( CORBA_iObject * )obj ) ;
    
    ( void )orb ;
    
    if ( o )
    {
        s = CORBA_createIOR( o ) ;
    
        if ( ev != NULL )
            memcpy( ev, env, sizeof( CORBA_Environment ) ) ;
    }
    
    return s ;
}

/**
*
*
*
*
**/
void _string_to_object( CORBA_ORB orb, CORBA_char * str, CORBA_iObject * o, CORBA_Environment * ev )
{
    CORBA_Environment * env ;
    CORBA_iORB * _orb    = ( CORBA_iORB * )orb ;
    CORBA_BOA boa        = _orb->oa ;

    if ( o == NULL )
    {
        CORBA_throw_system_exception( ev, EX_CORBA_NO_MEMORY, 0, CORBA_COMPLETED_NO ) ;
        return ;
    }

    env = &CORBA_OBJ_ENV( ( CORBA_iObject * )o ) ;

    env->_major = CORBA_NO_EXCEPTION ;

    if ( CORBA_BOA_find_slot( boa, o ) == NULL )
    {
        CORBA_throw_system_exception( env, EX_CORBA_INITIALIZE, 0, CORBA_COMPLETED_NO ) ;
        return ;
    }

    if ( CORBA_readIOR( o, str ) )
        CORBA_connect_client_object( o ) ;

    if ( ev != NULL )
        memcpy( ev, env, sizeof( CORBA_Environment ) ) ;
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_Object * BC_EXPORT CORBA_ORB_string_to_object( CORBA_ORB orb, 
    CORBA_char * str, CORBA_Environment * ev )
{
    CORBA_iObject * o ;
    
    if ( ev == NULL )
    {
        fprintf( stderr, "Env is NULL\n" ) ; 
        return NULL ;
    }

    o = CORBA_alloc_object() ;
    
    _string_to_object( orb, str, o, ev ) ;

    if ( ev->_major != CORBA_NO_EXCEPTION )    
        if ( o ) 
            CORBA_free( o ) ;

    return ( CORBA_Object )o ;
}

/**
*
*   CORBA_ORB_log
*   -------------
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_ORB_log( CORBA_Object obj, int severity, char * fmt, ... )
{
    char str[ 200 ] ;
    CORBA_iObject * o = ( CORBA_iObject * )obj ;
    FILE * fp ;
    unsigned long pid, tid ;
    
    va_list arglist ;

    va_start( arglist, fmt ) ;

    vsprintf( str, fmt, arglist ) ;

    va_end( arglist ) ;

    CORBA_getPID( &pid, &tid ) ;
    
    fp = CORBA_get_log( o ) ;

    fprintf( fp, "S=%d PID=%u TID=%u %s\n", severity, pid, tid, str ) ;
}


#if 0
#endif
