/**
*       tcpip.c
*       -------
*
*       Module for TCP/IP
*
*
*       Copyright Mitch Britton 1998-1999
*       All Rights Reserved
*       See the file licence.txt
*
**/

/**
*       Compilation Switches and Definitions
*       ------------------------------------
**/

/**
*       Header Files
*       ------------
**/

#include <stdio.h>
#include <corba.h>
#include "corba_i.h"
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <malloc.h>
#include <string.h>
#include <memory.h>
#include <sys/types.h>

#ifdef _Win32
#include <io.h>
#include <winsock.h>
#else
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#endif

/**
*       External Data
*       -------------
**/

/**
*       Internal Global Data
*       --------------------
**/

static int sockInitFail = 0 ;

/**
*       Internal Function Declarations
*       ------------------------------
**/


/**
*       External Function Declarations
*       ------------------------------
**/

/**
*       Public Global Data
*       ------------------
**/

#ifdef _Win32

/**
*
*
*
*
**/
void CORBA_getPID( unsigned long * pid, unsigned long * tid )
{
    *pid = GetCurrentProcessId() ;
    *tid = GetCurrentThreadId() ;
}

/**
*
*
*
*
**/
MSC_EXPORT int BC_EXPORT CORBA_sockErr( char * msg )
{
    int err = WSAGetLastError() ;
    char * p ;
    int r = -1 ;
    
    switch ( err )
    {
        case 0 :
        r = CORBA_SOCK_CLOSE ;
        break ;

        case WSAECONNRESET :
        p = "Connection reset" ;
        r = CORBA_SOCK_CLOSE ;
        break ;

        case WSAEADDRINUSE :
        p = "Address in use " ;
        break ;

        case WSAECONNREFUSED :
        p = "Connection refused " ;
        break ;

        default :
        p = "Windows Socket error " ;
        break ;
    }

    if ( msg )
    {
        msg[ 0 ] = '\0' ;

        if ( err )
            sprintf( msg, "%s (%d)", p, err ) ;
    }
    
    return r ;
}

/**
*
*
*
*
**/
int win32_socketStart( void )
{
    WORD wVersionRequested ;
    WSADATA wsaData ;

    wVersionRequested = MAKEWORD( 1, 1 ) ;

    return WSAStartup( wVersionRequested, &wsaData ) ;
}

/**
*
*
*
*
**/
void win32_socketShutdown( void )
{
    WSACleanup() ;
}


/**
*
*
*
*
**/
BOOL WINAPI DllMain( HINSTANCE hinstDll, DWORD reason, LPVOID reserved )
{
    ( void )hinstDll ;
    ( void )reserved ;

    switch ( reason )
    {
        case DLL_PROCESS_ATTACH :
        {
            sockInitFail = win32_socketStart() ;
        }
        break ;

        case DLL_THREAD_ATTACH :
        break ;

        case DLL_THREAD_DETACH :
        break ;

        case DLL_PROCESS_DETACH :
        {
            WSACleanup() ;
        }
        break ;

    }

    return TRUE ;
}

/**
*
*
*
*
**/
BOOL WINAPI DllEntryPoint( HINSTANCE hinstDll, DWORD reason, LPVOID reserved )
{
    return DllMain( hinstDll, reason, reserved ) ;
}

#else

/**
*
*
*
*
**/
void CORBA_getPID( unsigned long * pid, unsigned long * tid )
{
    *pid = getpid() ;
    *tid = 0 ;
}

/**
*
*
*
*
**/
int CORBA_sockErr( char * msg )
{
    char * p ;
    int r = -1 ;
    
    switch ( errno )
    {
        case 0 :
        r = CORBA_SOCK_CLOSE ;
        break ;
        
        case EBADF :
        p = "EBADF" ;
        r = CORBA_SOCK_CLOSE ;
        break ;
        
        case ENOENT :
        p = "ENOENT" ;
        r = CORBA_SOCK_CLOSE ;
        break ;
        
        case ECONNRESET :
        p = "Connection reset" ;
        r = CORBA_SOCK_CLOSE ;
        break ;
        
        case ECONNREFUSED :
        p = "Connection refused " ;
        break ;
        
        case EADDRINUSE :
        p = "Address in use " ;
        break ;

        default :
        p = "Socket error " ;
        break ;
    } 

    if ( msg )
    {
        msg[ 0 ] = '\0' ;

        if ( errno )
            sprintf( msg, "%s (%d)", p, errno ) ;
    }
    
    return r ;
}

#endif

/**
*
*
*
*
**/
static int log_sockErr( void )
{
    char msg[ 60 ] ;
    
    int err = CORBA_sockErr( msg ) ;

    return err ;
}

/**
*
*
*
*
**/
MSC_EXPORT void BC_EXPORT CORBA_ORB_Params( int * argc, char ** argv, CORBA_iORB * orb )
{
    static char prefix[] = "-ORB" ;

    int i, len ;
    char * p, * val, * logFile ;
    
    len = strlen( prefix ) ;

    if ( ( p = getenv( "ER_ORB_DEBUG" ) ) == NULL )
        orb->debug = 0 ;
    else
        orb->debug = 1 ;

    if ( ( p = getenv( "ER_ORB_LOG_FILE" ) ) == NULL )
        logFile = NULL ;
    else
        logFile = p ;

    for ( i = 1 ; i < *argc ; i++ )
    {
        p = argv[ i ] ;

        if ( ( strlen( p ) > len ) && ( memcmp( p, prefix, len ) == 0 ) )
        {
            p += len ;

            if ( ( val = strchr( p, '=' ) ) != NULL )
            {
                *val = '\0' ;
                val++ ;
            }

            if ( ( strcmp( p, "debug" ) == 0 ) && val )
                orb->debug = 1 ;
            else if ( ( strcmp( p, "log" ) == 0 ) && val )
                logFile = val ;
        }
    }
    
    orb->logName = logFile ;
}

/**
*
*   CORBA_iORB_init
*   ---------------
*
*   Initialize the ORB ( internal )
*
**/
CORBA_ORB CORBA_iORB_init( int * argc, char ** argv,
    CORBA_ORBid orb_identifier, CORBA_Environment * ev )
{
    CORBA_iORB * orb = CORBA_alloc_ORB( orb_identifier ) ;

    if ( ev )
        ev->_major = CORBA_NO_EXCEPTION ;

    if ( orb )
    {
        CORBA_ORB_Params( argc, argv, orb ) ;
    }
    else 
    {
        CORBA_throw_system_exception( ev, EX_CORBA_NO_MEMORY, 0, CORBA_COMPLETED_NO ) ;
        return NULL ;
    }

    CORBA_BOA_init( orb, argc, argv, "BOA", ev ) ;

    orb->freeOA = CORBA_TRUE ;

    return orb ;
}


/**
*
*
*
*
**/
static void CORBA_BOA_params( int * argc, char ** argv, CORBA_BOA_PARAMS * ap )
{
    static char prefix[] = "-BOA" ;

    int i, len ;
    char * p, * val ;
    
    len = strlen( prefix ) ;

    if ( ( p = getenv( "ER_BOA_CONNS" ) ) == NULL )
        ap->connections = 3 ;
    else
        ap->connections = atoi( p ) ;

    if ( ( p = getenv( "ER_BOA_TIMEOUT" ) ) == NULL )
        ap->timeout = 30 ;  /* default timeout in seconds */
    else
        ap->timeout = atoi( p ) ;

    if ( ( argc == NULL ) || ( argv == NULL ) )
        return ;
        
    for ( i = 1 ; i < *argc ; i++ )
    {
        p = argv[ i ] ;

        if ( ( strlen( p ) > len ) && ( memcmp( p, prefix, len ) == 0 ) )
        {
            p += len ;

            if ( ( val = strchr( p, '=' ) ) != NULL )
            {
                *val = '\0' ;
                val++ ;
            }

            if ( ( strcmp( p, "conns" ) == 0 ) && val )
                ap->connections = atoi( val ) ;
            else if ( ( strcmp( p, "timeout" ) == 0 ) && val )
                ap->timeout = atoi( val ) ;
        }
    }

    if ( ap->timeout == 0 )
        ap->timeout = -1 ;    /* infinite timeout */
}

/**
*
*
*
*
**/
MSC_EXPORT CORBA_BOA BC_EXPORT CORBA_BOA_init( CORBA_ORB orb, int * argc, char ** argv,
    CORBA_ORB_OAid boa_identifier, CORBA_Environment * env )
{
    CORBA_BOA boa = NULL ;
    CORBA_iORB * Orb ;
    CORBA_BOA_PARAMS p ;

    if ( env )
        env->_major = CORBA_NO_EXCEPTION ;

    if ( orb )
    {
        Orb = ( CORBA_iORB * )orb ;

        if ( ( boa = Orb->oa ) == NULL )             
        {
            CORBA_BOA_params( argc, argv, &p ) ;

            boa = CORBA_alloc_BOA( p.connections ) ;

            if ( boa )
            {
                memcpy( &boa->params, &p, sizeof( CORBA_BOA_PARAMS ) ) ;
            
                boa->orb = orb ;

                Orb->freeOA = CORBA_FALSE ;
            }
        }

        if ( boa == NULL )
            CORBA_throw_system_exception( env, EX_CORBA_NO_MEMORY, 0, CORBA_COMPLETED_NO ) ;
    }
    else
        CORBA_throw_system_exception( env, EX_CORBA_INITIALIZE, 0, CORBA_COMPLETED_NO ) ;
    
    if ( boa ) 
    {
        strcpy( boa->name, boa_identifier ) ;

        Orb->oa = boa ;
    }
    
    return boa ;
}


/**
*
*       CORBA_send_message
*       ------------------
*
*       Send a message
*
**/
int CORBA_send_message( CORBA_iObject * o, CORBA_Dbuff * buff, 
    int socket, CORBA_octet messageType )
{
    CORBA_Environment * env = &CORBA_OBJ_ENV( o ) ;

    CORBA_octet minor = ( CORBA_octet )( o->server ? o->msgHdr.GIOP_version.minor : 1 ) ;  

    initMsgHdr( &o->msgHdr, minor ) ;

    o->msgHdr.message_type = messageType ;
    o->msgHdr.message_size = buff->curr ;

    env->_major = CORBA_NO_EXCEPTION ;

    if ( CORBA_SendMsg( socket, &o->msgHdr, sizeof( GIOP_MessageHeader_1_1 ) ) )
    {
        CORBA_throw_system_exception( env, EX_CORBA_COMM_FAILURE, 0, CORBA_COMPLETED_NO ) ;
        return -1 ;
    }        
    else
    {
        if ( CORBA_SendMsg( socket, buff->p, buff->curr ) )
        {
            CORBA_throw_system_exception( env, EX_CORBA_COMM_FAILURE, 0, CORBA_COMPLETED_NO ) ;
            return -1 ;
        }
        
        CORBA_resetpBuff( buff ) ;
    }

    return 0 ;
}

/**
*
*
*
*
**/
static int waitOnSocket( int sockFd, int timeout )
{
    struct timeval to ;
    fd_set readFds ;

    FD_ZERO( &readFds ) ;
    FD_SET( sockFd, &readFds ) ;

    to.tv_sec  = timeout ;
    to.tv_usec = 0 ;

    if ( select( sockFd + 1, &readFds, NULL, NULL, &to ) < 0 )
        return log_sockErr() ;

    if ( FD_ISSET( sockFd, &readFds ) )
        return 0 ;                         /* socket ready */

    return COI_MIN_SOCK_TO ;
}

/**
*
*
*
*
**/
int CORBA_RecvMsg( int sockFd, void * msg, CORBA_long size )
{
    int err = waitOnSocket( sockFd, DEFAULT_TIMEOUT ) ;

    if ( err == 0 )
    {
        if ( recv( sockFd, ( char * )msg, size, 0 ) != size )
            err = log_sockErr() ;
    }

    return err ;
}

/**
*
*       CORBA_getMsgHdr
*       ---------------
*
*       Receive a message
*
**/
int CORBA_getMsgHdr( int channel, GIOP_MessageHeader_1_1 * msgHdr )
{
    int r = CORBA_RecvMsg( channel, msgHdr, sizeof( GIOP_MessageHeader_1_1 ) ) ;

    if ( msgHdr->flags != ENDIAN )
        CORBA_swap_long( ( CORBA_long * )&msgHdr->message_size ) ;

    return r ;
}

/**
*
*       CORBA_receive_message
*       ---------------------
*
*       Receive a message
*
**/
int CORBA_receive_message( CORBA_iObject * o, int channel )
{
    CORBA_unsigned_long msgSize ;
    int r ;
    CORBA_Environment * env = THROW_OENV( o ) ;
    
    r = CORBA_getMsgHdr( channel, &o->msgHdr ) ;

    if ( r )
    {
        if ( r == COI_MIN_SOCK_TO )
            return r ;

        if ( r == CORBA_SOCK_CLOSE )
            return r ;

        CORBA_throw_system_exception( env, EX_CORBA_COMM_FAILURE, r, CORBA_COMPLETED_NO ) ;

        return -1 ;
    }

    msgSize = o->msgHdr.message_size ;

    if ( msgSize == 0 )
        return 0 ;

    if ( msgSize > o->buff.len )
    {
        if ( CORBA_Dbuff_allocate( &o->buff, msgSize ) )
        {
            CORBA_throw_system_exception( env, EX_CORBA_NO_MEMORY, 0, CORBA_COMPLETED_NO ) ;

            return -2 ;  /* memory failure */
        }
    }

    o->buff.endian = o->msgHdr.flags ;    
    
    CORBA_preImpl( o ) ;

    if ( ( r = CORBA_RecvMsg( channel, o->buff.p, msgSize ) ) != 0 )
    {
        CORBA_throw_system_exception( env, EX_CORBA_COMM_FAILURE, r, CORBA_COMPLETED_NO ) ;
        return -1 ;
    }

    return 0 ;
}



/**
*
*       CORBA_getSocket
*       ---------------
*
*       Get a new socket
*       A value < 0 is an error
*
**/
MSC_EXPORT int BC_EXPORT CORBA_getSocket( CORBA_long port )
{
    struct sockaddr_in srvAddr ;
    int sockFd ;
    char msg[ 60 ] ;
    
    if ( sockInitFail )
        return CORBA_SOCKFAIL ;

    if ( ( sockFd = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP ) ) < 0 )
    {
        log_sockErr() ;

        return CORBA_SOCKFAIL ;
    }

    /*
    *  Bind to a well known address. INADDR_ANY says any network
    *  interface will do. htonX() routines put things in network
    *  byte order
    */

    memset( &srvAddr, 0, sizeof( srvAddr ) ) ;

    srvAddr.sin_family      = AF_INET ;
    srvAddr.sin_port        = ( u_short )htons( ( u_short )port ) ;
    srvAddr.sin_addr.s_addr = htonl( INADDR_ANY ) ;

    if ( bind( sockFd, ( struct sockaddr * )&srvAddr, sizeof( srvAddr ) ) < 0 )
    {
        CORBA_sockErr( msg ) ;
        
        CORBA_ORB_log( NULL, LOG_SYS_ERR, "bind error %s", msg ) ;
    
        CORBA_Disconnect( sockFd ) ;

        return CORBA_BINDFAIL ;
    }

    return sockFd ;
}

/**
*
*
*
*
**/
int CORBA_getSockAddr( int sock, CORBA_long * addr, CORBA_long * port )
{
    struct sockaddr a ;
    struct sockaddr_in * ad ;
    int len = sizeof( struct sockaddr_in ) ;

    if ( getsockname( sock, &a, &len ) < 0 )
        return -1 ;

    ad = ( struct sockaddr_in * )&a ;

    *port = ( unsigned long )ntohs( ad->sin_port ) ;
    
    *addr = ntohl( ad->sin_addr.s_addr ) ;

    return 0 ;
}

/**
*
*
*
*
**/
CORBA_long CORBA_getLocalAddress()
{
    struct hostent * hptr ;
    struct in_addr * a ;

#ifndef _Win32

    char hname[ 200 ] ;

    if ( gethostname( hname, sizeof( hname ) ) == -1 )
        return -1 ;

    if ( ( hptr = gethostbyname( hname ) ) == NULL )

#else

    if ( ( hptr = gethostbyname( NULL ) ) == NULL )

#endif

        return -1 ;

    a = ( struct in_addr * )*hptr->h_addr_list ;

    return ( CORBA_long )a->s_addr ;
}

/**
*
*       CORBA_Listen
*       ------------
*
*       Queue up some listens
*       A value < 0 is an error
*
**/
MSC_EXPORT int BC_EXPORT CORBA_Listen( int sockFd )
{
    /* Queue the usual five requests */

    if ( listen( sockFd, 5 ) < 0 )
    {
        CORBA_Disconnect( sockFd ) ;

        return CORBA_LISTFAIL ;
    }

    return sockFd ;
}

/**
*
*       CORBA_inetAddr
*       -------------
*
*       Create a string representation of an internet address
*
**/
MSC_EXPORT CORBA_char * BC_EXPORT CORBA_inetAddr( unsigned int addr, long port )
{
    char a[ 24 ] ;

    unsigned char * p = ( unsigned char * )&addr ;

    if ( port )
      sprintf( a, "%u.%u.%u.%u:%u", (unsigned)p[0], (unsigned)p[1],
        (unsigned)p[2], (unsigned)p[3], port ) ;
    else
      sprintf( a, "%u.%u.%u.%u", (unsigned)p[0], (unsigned)p[1],
        (unsigned)p[2], (unsigned)p[3] ) ;

    return CORBA_string_copy( a ) ;
}

/**
*
*       CORBA_string_to_inetAddr
*       ------------------------
*
*       Create a host address from a string
*       Takes an address of format XXX.XXX.XXX.XXX:XXX and 
*       returns the port number. The internet address is returned in param "addr".
*       The returned port number is -1 if the format is XXX.XXX.XXX.XXX
*
**/
MSC_EXPORT long BC_EXPORT CORBA_string_to_inetAddr( char * str, CORBA_unsigned_long * addr )
{
    char adrArray[ 24 ], * a  ;
    unsigned char * p ;
    char * s ;
    CORBA_long port ; 
    int i ;
    
    a = adrArray ;
    
    strcpy( a, str ) ;

    if ( ( s = strchr( a, ':' ) ) != NULL )
    {
        *s++ = '\0' ;
        port = atol( s ) ;
    }        
    else
        port = -1 ;
        
    p = ( unsigned char * )addr ;

    for ( i = 0 ; i < 4 ; i++ )
    {
        if ( i <  3 )
        {
            if ( ( s = strchr( a, '.' ) ) == NULL ) 
            {
                *addr = 0 ;
                return -1 ;
            }       
            else
                *s = '\0' ;
        }
        
        for ( s = a ; *s ; s++ )
        {
            if ( ! isdigit( *s ) )
            {
                *addr = 0 ;
                return -1 ;
            }       
        }            
        
        p[ i ] = ( unsigned char )atoi( a ) ;
        
        a += strlen( a ) + 1 ;
    }

    return port ;
}


/**
*
*       CORBA_Connect
*       -------------
*
*       New connection
*
**/
int CORBA_Connect( int sockFd, CORBA_long * addr )
{
    struct sockaddr_in clientAddr ;
    int clientAddrLength, newSockFd ;

    if ( sockFd < 0 )
        return sockFd ;

    /* Service requests */

    clientAddrLength = sizeof( struct sockaddr_in ) ;

    newSockFd = accept( sockFd, ( struct sockaddr * )&clientAddr,
                        &clientAddrLength ) ;

    if ( newSockFd < 0 )
    {
        log_sockErr() ;

        return CORBA_ACCPTFAIL ;
    }

    *addr = clientAddr.sin_addr.s_addr ;

    return newSockFd ;
}


/**
*
*       CORBA_ClientConnect
*       -------------------
*
*       Connect to the server and return the socket id.
*       A value < 0 is an error
*
**/
int CORBA_ClientConnect( char * server, CORBA_long addr, int port )
{
    struct sockaddr_in srvAddr ;
    int sockFd, one = 1 ;
    CORBA_unsigned_long addr1 ;
    
    struct hostent * server_entry ;

    if ( ( sockFd = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP ) ) < 0 )
    {
        log_sockErr() ;

        return CORBA_SOCKFAIL ;
    }

    memset( &srvAddr, 0, sizeof( srvAddr ) ) ;
    
    if ( server )
    {
        CORBA_string_to_inetAddr( server, &addr1 ) ;
        
        if ( addr1 )
            srvAddr.sin_addr.s_addr = addr1 ;
        else if ( ( server_entry = ( struct hostent * )gethostbyname( server ) ) == NULL ) 
            return CORBA_HOSTNOTFND ;
        else
            memcpy( &srvAddr.sin_addr, server_entry->h_addr, server_entry->h_length ) ;
    }
    else
        srvAddr.sin_addr.s_addr = addr ;

    /* printf( "Connecting to %s\n", CORBA_inetAddr( srvAddr.sin_addr.s_addr,
        ( unsigned short )port ) ) ; */

    srvAddr.sin_family      = AF_INET ;
    srvAddr.sin_port        = ( u_short )htons( ( u_short )port ) ;

    if ( connect( sockFd, ( struct sockaddr * )&srvAddr, sizeof( srvAddr ) ) < 0 )
    {
        log_sockErr() ;

        CORBA_Disconnect( sockFd ) ;

        return CORBA_CONNFAIL ;
    }

    setsockopt( sockFd, SOL_SOCKET, SO_KEEPALIVE, ( char * )&one, sizeof( one ) ) ;
    
    return sockFd ;
}

/**
*
*       CORBA_Disconnect
*       ----------------
*
*       Disconnect the socket
*
**/
MSC_EXPORT void BC_EXPORT CORBA_Disconnect( int sockFd )
{
    /* printf( "%d closes socket %d\n", getpid(), sockFd ) ;  */

    closesocket( sockFd ) ;
}

/**
*
*       CORBA_sendMsg
*       -------------
*
*       Disconnect the socket
*
**/
int CORBA_SendMsg( int sockFd, void * msg, CORBA_long size )
{
    if ( ( size != 0 ) && ( send( sockFd, msg, size, 0 ) != size ) )
        return CORBA_FAILTOSEND ;

    return 0 ;
}


#if 0

CORBA_long CORBA_getPort()
{
    int sockFd ;
    CORBA_long port ;
    struct sockaddr_in sAddr ;
    CORBA_long laddr ;

    if ( ( sockFd = socket( AF_INET, SOCK_DGRAM, IPPROTO_IP ) ) < 0 )
    {
        log_sockErr() ;

        return CORBA_SOCKFAIL ;
    }

    memset( &addr, 0, sizeof( addr ) ) ;

    sAddr.sin_family      = AF_INET ;
    sAddr.sin_port        = ( u_short )htons( 0 ) ;
    sAddr.sin_addr.s_addr = htonl( INADDR_ANY ) ;

    if ( bind( sockFd, ( struct sockaddr * )&sAddr, sizeof( sAddr ) ) < 0 )
        return CORBA_BINDFAIL ;

    port = ( unsigned long )ntohs( sAddr.sin_port ) ;

    if ( CORBA_getSockAddr( sockFd, &laddr, &port ) )
        return -1 ;

    if ( addr )
        *addr = laddr ;
        
    CORBA_Disconnect( sockFd ) ;

    return port ;
}

#endif

