// <><><><><><><><><><><><><>  Plot3DZ.h  <><><><><><><><><><><><><><>
//
// ----------------------------------------------------
// (C) Copyright Clark Dailey, Inc. 1999
//
// C++ Class Name:
//    Plot3DZ
//
// Purpose:
//    Plot3DZ is an abstract C++ class used to plot a 3D
//    "ruled surface" view of a function defined as z=func(x,y).
//    The plot is generated as a set of constant x and/or constant y lines.
//    Plotting order tries to minimize pen movement for pen plotters.
//
// Highlights:
//   1) Plots an arbitrary equation z=func(x,y) as a ruled surface.
//   2) Constant x and/or constant y lines are drawn.
//   3) Plot to any type of device;
//      MoveTo(), DrawTo() pure virtual methods must be
//      supplied by the derived class to do the actual plotting.
//   4) User selectable 3D view point.
//   5) Supports hidden line removal.
//   6) Provides for scaling.
//   7) Video resolution is settable.
//   8) Allows for user supplied (optional) call-back functions
//      that can be used to monitor plotting progress.
//   9) Has the ability to produce color gradient plots.
//      Virtual method ColorTo() is overridable.
//
// Author:
//   Clark Dailey
//   9037 Cambridge Road
//   Woodbury, MN 55125
//   ccdailey@imation.com
//   (651)-714-9036
//
// This source code may be modified or used as is,
// as long as credit is given to the author.
// ----------------------------------------------------

#ifndef	_PLOT3DZ_H_	 	// avoid repeated including
#define	_PLOT3DZ_H_	1	// has been included

#include "..\Common\plot3d.h"

class Plot3DZ : public Plot3D // abstract class - must be derived from to use
{
public:
	Plot3DZ();	// constructor
	virtual ~Plot3DZ();	// destructor

	// pure virtual functions must be defined by derived class
	virtual double Zfunc(double x,double y)=0;	// the mathematical plotting function z=func(x,y)

	// settings
	int SetParameters(unsigned nxlines=50,unsigned nylines=50,
					  unsigned show_xlines=1, unsigned show_ylines=1,
					  unsigned remove_hidden_lines=1);

	// plotting
	int Plot();

	// call back routines
	virtual void cbBegXLine(double x) { }	// called at beg draw of each constant x line
	virtual void cbEndXLine(double x) { }	// called at end draw of each constant x line
	virtual void cbBegYLine(double y) { }	// called at beg draw of each constant y line
	virtual void cbEndYLine(double y) { }	// called at end draw of each constant y line

	// debugging routines
	void Dump();

protected:	// routines
	int  InitPlot();
	void CalcDeltas();
	int  AllocateMaskArrays();
	void DeleteMaskArrays();
	void InitMaskArrays();
	int  InitHiddenLine(int xv,int yv);
	int  CalcHiddenLine(int xv,int yv,long color);
	int  PlotXLines(void);
	int  PlotYLines(void);
	void PlotXLine(unsigned nPoints,double x,double ystart,double yend,double dy,int issueCB);
	void PlotYLine(unsigned nPoints,double y,double xstart,double xend,double dx,int issueCB);
	void PlotPoint(unsigned nPoint,double x,double y);
	void PlotFrontEdge(int yDir);
	void DrawAxis(void);
	int  HiYMask(int xv);
	int  LoYMask(int xv);
	void SetHiYMask(int xv,int yv);
	void SetLoYMask(int xv,int yv);

protected:	// user input data
	// number of constant x,y lines to draw
	unsigned m_nxlines, m_nylines;

	// flags
	unsigned m_show_xlines;
	unsigned m_show_ylines;
	unsigned m_remove_hidden_lines;

protected:	// internal data
	// values for plotting
	double   m_dxline, m_dyline;		// spacing between constant x,y lines
	unsigned m_nxincs, m_nyincs;		// number of fine line plotting increments
	unsigned m_nxedge, m_nyedge;		// number of fine line edge tick increments
	double   m_dxedge,   m_dyedge;		// x,y incremental for plotting edge tick
	double   m_dxinc[2], m_dyinc[2];	// x,y incremental for plotting: 0=even, 1=odd
	double   m_xstart[2],m_ystart[2], m_zstart[2];	// starting x,y,z points: 0=even,1=odd

	// hidden line removel variables
	int  m_region_last;			// PLOT_REGION_nnn
	int	 m_xv_last, m_yv_last;	// last video point
	long m_color_last;			// color of last video line drawm
	int  m_front_edge_drawn;		// 0=no, 1=front edge has been drawn

	// hidden line removal masking arrays
	unsigned m_mask_size;
	int* m_pLoYMask;
	int* m_pHiYMask;
};

// plot mask initial values
#define	PLOT_LO_MASK   2147483600	// a large positive number
#define	PLOT_HI_MASK  -2147483600	// a large negative number

// masking regions (Don't change values; used in algebraic form)
#define	PLOT_REGION_ABOVE	(1)
#define	PLOT_REGION_HIDDEN	(0)
#define	PLOT_REGION_BELOW	(2)

// checks if plot mask is set
#define	IfPlotMaskNotSet(v)   ((v)==PLOT_LO_MASK || (v)==PLOT_HI_MASK)


#endif	/* if _PLOT3DZ_H_ included */

/*	<><><><><><><><><><><><><>  Plot3DZ.h  <><><><><><><><><><><><><><> */
