// SuNT service copyright Steffen Krause 1995
// Steffen Krause
// Wichertstr. 3
// D-10439 Berlin
// Germany
// skrause@informatik.hu-berlin.de

// module: susrv.c
// implementation of the su server service worker functions
// based on simple.c from the SDK

// TODO: to make the su service more secure, one could restrict 
// access to the named pipe it creates to certain user accounts

#include <windows.h>
#include <stdio.h>
#include "service.h"

// this event is signalled when the
// service should end
//
HANDLE  hServerStopEvent = NULL;

//
//  FUNCTION: ServiceStart
//
//  PURPOSE: Actual code of the service
//           that does the work.
//
//  PARAMETERS:
//    dwArgc   - number of command line arguments
//    lpszArgv - array of command line arguments
//
//  RETURN VALUE:
//    none
//
//  COMMENTS:
//    The default behavior is to open a
//    named pipe, \\.\pipe\simple, and read
//    from it.  It the modifies the data and
//    writes it back to the pipe.  The service
//    stops when hServerStopEvent is signalled
//


VOID ServiceStart (DWORD dwArgc, LPTSTR *lpszArgv)
{
    HANDLE                  hPipe = INVALID_HANDLE_VALUE;
    HANDLE                  hEvents[2] = {NULL, NULL};
    OVERLAPPED              os;
    PSECURITY_DESCRIPTOR    pSD = NULL;
    SECURITY_ATTRIBUTES     sa;
    TCHAR                   szIn[622];
    TCHAR                   szOut[100];
    LPTSTR                  lpszPipeName = TEXT("\\\\.\\pipe\\susrv");
    BOOL                    bRet;
    DWORD                   cbRead;
    DWORD                   cbWritten;
    DWORD                   dwWait;
    UINT                    ndx;
	
	///////////////////////////////////////////////////
    //
    // Service initialization
    //

    // report the status to the service control manager.
    //
    if (!ReportStatusToSCMgr(
        SERVICE_START_PENDING, // service state
        NO_ERROR,              // exit code
        3000))                 // wait hint
        goto cleanup;
    // create the event object. The control handler function signals
    // this event when it receives the "stop" control code.
    //
    hServerStopEvent = CreateEvent(
        NULL,    // no security attributes
        TRUE,    // manual reset event
        FALSE,   // not-signalled
        NULL);   // no name

    if ( hServerStopEvent == NULL)
        goto cleanup;

    hEvents[0] = hServerStopEvent;

    // report the status to the service control manager.
    //
    if (!ReportStatusToSCMgr(
        SERVICE_START_PENDING, // service state
        NO_ERROR,              // exit code
        3000))                 // wait hint
        goto cleanup;

    // create the event object object use in overlapped i/o
    //
    hEvents[1] = CreateEvent(
        NULL,    // no security attributes
        TRUE,    // manual reset event
        FALSE,   // not-signalled
        NULL);   // no name

    if ( hEvents[1] == NULL)
        goto cleanup;

    // report the status to the service control manager.
    //
    if (!ReportStatusToSCMgr(
        SERVICE_START_PENDING, // service state
        NO_ERROR,              // exit code
        3000))                 // wait hint
        goto cleanup;

    // create a security descriptor that allows anyone to write to
    //  the pipe...
    //
	// StK - this could be used to make the su service inaccessible to certain accounts
	// (e.g. guests)

    pSD = (PSECURITY_DESCRIPTOR) malloc( SECURITY_DESCRIPTOR_MIN_LENGTH );

    if (pSD == NULL)
        goto cleanup;

    if (!InitializeSecurityDescriptor(pSD, SECURITY_DESCRIPTOR_REVISION))
        goto cleanup;

    // add a NULL disc. ACL to the security descriptor.
    //
    if (!SetSecurityDescriptorDacl(pSD, TRUE, (PACL) NULL, FALSE))
        goto cleanup;

    sa.nLength = sizeof(sa);
    sa.lpSecurityDescriptor = pSD;
    sa.bInheritHandle = TRUE;


    // report the status to the service control manager.
    //
    if (!ReportStatusToSCMgr(
        SERVICE_START_PENDING, // service state
        NO_ERROR,              // exit code
        3000))                 // wait hint
        goto cleanup;
    // allow user to define pipe name
    for ( ndx = 1; ndx < dwArgc-1; ndx++ )
    {

        if ( ( (*(lpszArgv[ndx]) == TEXT('-')) ||
               (*(lpszArgv[ndx]) == TEXT('/')) ) &&
             _stricmp( TEXT("pipe"), lpszArgv[ndx]+1 ) == 0 )
        {
            lpszPipeName = lpszArgv[++ndx];
        }

    }
	
    // open our named pipe...
    //
    hPipe = CreateNamedPipe(
                    lpszPipeName         ,  // name of pipe
                    FILE_FLAG_OVERLAPPED |
                    PIPE_ACCESS_DUPLEX,     // pipe open mode
                    PIPE_TYPE_MESSAGE |
                    PIPE_READMODE_MESSAGE |
                    PIPE_WAIT,              // pipe IO type
                    1,                      // number of instances
                    0,                      // size of outbuf (0 == allocate as necessary)
                    0,                      // size of inbuf
                    1000,                   // default time-out value
                    &sa);                   // security attributes
	if (hPipe == INVALID_HANDLE_VALUE) {
        AddToMessageLog(TEXT("Unable to create named pipe"));
        goto cleanup;
    }

	    // report the status to the service control manager.
    //
    if (!ReportStatusToSCMgr(
        SERVICE_RUNNING,       // service state
        NO_ERROR,              // exit code
        0))                    // wait hint
        goto cleanup;

    //
    // End of initialization
    //
    ////////////////////////////////////////////////////////

    ////////////////////////////////////////////////////////
    //
    // Service is now running, perform work until shutdown
    //

    while ( 1 )
    {
        // init the overlapped structure
        //
		memset( &os, 0, sizeof(OVERLAPPED) );
        os.hEvent = hEvents[1];
        ResetEvent( hEvents[1]);

		ConnectNamedPipe(hPipe, &os);

        if ( GetLastError() == ERROR_IO_PENDING )
        {
            dwWait = WaitForMultipleObjects( 2, hEvents, FALSE, INFINITE );
            if ( dwWait != WAIT_OBJECT_0+1 )     // not overlapped i/o event - error occurred,
                break;                           // or server stop signaled
        }
        // init the overlapped structure
        //
        memset( &os, 0, sizeof(OVERLAPPED) );
        os.hEvent = hEvents[1];
        ResetEvent( hEvents[1] );

        // grab whatever's coming through the pipe...
        //
        bRet = ReadFile(
                    hPipe,          // file to read from
                    szIn,           // address of input buffer
                    sizeof(szIn),   // number of bytes to read
                    &cbRead,        // number of bytes read
                    &os);           // overlapped stuff, not needed

        if ( !bRet && ( GetLastError() == ERROR_IO_PENDING ) )
        {
            dwWait = WaitForMultipleObjects( 2, hEvents, FALSE, INFINITE );
            if ( dwWait != WAIT_OBJECT_0+1 )     // not overlapped i/o event - error occurred,
                break;                           // or server stop signaled
        }

				
    	//** OK, here I start with my own code.
		// Analyze the input we get for commandline, username, password and indirect
		// and call a function that creates a process for the user

		AnalyzeAndExecute(szIn, szOut, hPipe);
		
		//** give the result back
		
		// init the overlapped structure
        //
        memset( &os, 0, sizeof(OVERLAPPED) );
        os.hEvent = hEvents[1];
        ResetEvent( hEvents[1] );

        // send it back out...
        //
        bRet = WriteFile(
                    hPipe,          // file to write to
                    szOut,          // address of output buffer
                    sizeof(szOut),  // number of bytes to write
                    &cbWritten,     // number of bytes written
                    &os);           // overlapped stuff, not needed

        if ( !bRet && ( GetLastError() == ERROR_IO_PENDING ) )
        {
            dwWait = WaitForMultipleObjects( 2, hEvents, FALSE, INFINITE );
            if ( dwWait != WAIT_OBJECT_0+1 )     // not overlapped i/o event - error occurred,
                break;                           // or server stop signaled
        }

        // drop the connection...
        //
        DisconnectNamedPipe(hPipe);
    }

  cleanup:

    if (hPipe != INVALID_HANDLE_VALUE )
        CloseHandle(hPipe);

    if (hServerStopEvent)
        CloseHandle(hServerStopEvent);

    if (hEvents[1]) // overlapped i/o event
        CloseHandle(hEvents[1]);

    if ( pSD )
        free( pSD );

}


//
//  FUNCTION: ServiceStop
//
//  PURPOSE: Stops the service
//
//  PARAMETERS:
//    none
//
//  RETURN VALUE:
//    none
//
//  COMMENTS:
//    If a ServiceStop procedure is going to
//    take longer than 3 seconds to execute,
//    it should spawn a thread to execute the
//    stop code, and return.  Otherwise, the
//    ServiceControlManager will believe that
//    the service has stopped responding.
//    
VOID ServiceStop()
{
    if ( hServerStopEvent )
        SetEvent(hServerStopEvent);
}

//** this function does the actual work

BOOL AnalyzeAndExecute(char * szIn, char * szReturn, HANDLE hPipe)
{	
	char 					szApplication[512];
	char					szUsername[50];
	char * 					pszUserPtr;
	char 					szPassword[50];
	char *					pszPos;
	char *					pszOldPos;
	HANDLE hToken;			// token handle for the logged on user
	STARTUPINFO si;			// process startup info
	PROCESS_INFORMATION pi;	// process information
	HDESK hDesktop;
	HWINSTA hWindowStation;
	HWINSTA hOldWindowStation;
	char lpDomain[256];
	DWORD dwDomainSize = 256;
	char lpSid[1024];
	DWORD dwSidSize = 1024;
	SID_NAME_USE snu;


		
	// first get the arguments
	memset(szApplication, 0, 512);		
	memset(szUsername, 0, 50);
	memset(szPassword, 0, 50);
	pszOldPos = szIn;
	pszPos = strchr(pszOldPos, 0x0a);
	if (pszPos == NULL)
	{
		strcpy(szReturn, "SU service: Cannot get commandline to start");
		return(FALSE);
	}
	strncpy(szApplication, pszOldPos, pszPos - pszOldPos);
	pszOldPos = pszPos + 1;
	pszPos = strchr(pszOldPos, 0x0a);
	if (pszPos == NULL)
	{
		strcpy(szReturn, "SU service: Cannot get username");
		return(FALSE);
	}
	strncpy(szUsername, pszOldPos, pszPos - pszOldPos);
	pszOldPos = pszPos + 1;
	pszPos = strchr(pszOldPos, 0x0a);
	if (pszPos == NULL)
	{
		strcpy(szReturn, "SU service: Cannot get password");
		return(FALSE);
	}
	strncpy(szPassword, pszOldPos, pszPos - pszOldPos);
	//check if a domain Name is specified - thanks to Darrel Schneider
	if (strchr(szUsername, '\\') != NULL) 
	{
    	pszUserPtr = strchr(szUsername, '\\');
        *pszUserPtr++ = 0;
		strcpy(lpDomain,szUsername);
    }
	else
	{
		pszUserPtr = szUsername;
		strcpy(lpDomain,".");
	}

	// log on the user	
	if (!LogonUser(pszUserPtr, lpDomain, szPassword, LOGON32_LOGON_INTERACTIVE, LOGON32_PROVIDER_DEFAULT, &hToken))
	{
		// if this fails, try to log the user in to the primary domain if no domain was specified
		 if ((GetLastError() == ERROR_LOGON_FAILURE) && (szUsername == pszUserPtr))
		{
			if (!LookupAccountName(NULL, pszUserPtr, lpSid, &dwSidSize, lpDomain, &dwDomainSize, & snu))
			{
				ErrorHandler("LookupAccountName", szReturn);
				return(FALSE);
			}
 			// OK, let's try it again in this domain
			if (!LogonUser(szUsername, lpDomain, szPassword, LOGON32_LOGON_INTERACTIVE, LOGON32_PROVIDER_DEFAULT, &hToken))
			{
				ErrorHandler("LogonUser", szReturn);
				return(FALSE);
			}
		//later, we will use the sid, too, to set the security on the window station and desktop

 		}
 		else
		{
			ErrorHandler("LogonUser", szReturn);
			return(FALSE);
		}
	}
	
	// initialize the startupinfo

    si.cb = sizeof(STARTUPINFO);
    si.lpReserved = NULL;
    si.lpDesktop = "WinSta0\\Default";
    si.lpTitle = NULL;
    si.dwFlags = 0;
    si.cbReserved2 = 0;
    si.lpReserved2 = NULL;
	

	// impersonate the pipe client to get access to the user's Window station and Desktop
	if (!ImpersonateNamedPipeClient(hPipe))
	{
		ErrorHandler("ImpersonateNamedPipeClient", szReturn);
		return(FALSE);
	}
	// DebugBreak();
	hOldWindowStation = GetProcessWindowStation();
	hWindowStation = OpenWindowStation("WinSta0", FALSE, MAXIMUM_ALLOWED); 
	if (!SetProcessWindowStation(hWindowStation))  //required for OpenDesktop since the service has no associated WindowStation
	{
		ErrorHandler("SetProcessWindowStation", szReturn);
		CloseHandle(hToken);
		CloseHandle(hWindowStation);
		return(FALSE);
	} 
	hDesktop = OpenDesktop("Default", 0, FALSE, MAXIMUM_ALLOWED);
	SetProcessWindowStation(hOldWindowStation);
	RevertToSelf();
	if (!hDesktop)
	{
		ErrorHandler("OpenDesktop", szReturn);
		CloseHandle(hToken);
		CloseHandle(hWindowStation);
		CloseHandle(hDesktop);
		return(FALSE);
	} 
	
	// give the user access to the WindowStation and Desktop
	if (!SetUserObjectAllAccess(hWindowStation, szReturn))
	{
		strcpy(szReturn, "Cannot set WindowStation security"); 
		CloseHandle(hToken);
		CloseHandle(hWindowStation);
		CloseHandle(hDesktop);
		return(FALSE);
	}
	if (!SetUserObjectAllAccess(hDesktop, szReturn))
	{
		strcpy(szReturn, "Cannot set Desktop security"); 
	 	CloseHandle (hToken);
		CloseHandle(hWindowStation);
		CloseHandle(hDesktop);	  
		return(FALSE);
	} 
	// create the process that sets the WindowStation and starts the commandline	
	if (!CreateProcessAsUser(hToken,NULL, szApplication, NULL, NULL, TRUE, CREATE_NEW_CONSOLE , NULL, NULL, &si, &pi))
	{
		ErrorHandler("CreateProcessAsUser", szReturn);
	}
	else
	{
		strcpy(szReturn, "OK");
	}
		
    CloseHandle (hToken);
	CloseHandle(hWindowStation);
	CloseHandle(hDesktop);

	return(TRUE);
}

// ** Error handler for my own functions		
void ErrorHandler(LPSTR szFunctionName, LPSTR szReturn)
{
	LPTSTR lpszTemp = NULL;
	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |FORMAT_MESSAGE_ARGUMENT_ARRAY,
                           NULL,
                           GetLastError(),
                           LANG_NEUTRAL,
                           (LPTSTR)&lpszTemp,
                           0,
                           NULL );

	wsprintf(szReturn, "%s failed. Error Code %d - %s", szFunctionName, GetLastError(), lpszTemp);
	if ( lpszTemp )
        LocalFree((HLOCAL) lpszTemp );

}

//** gives acess to a user object to all users by adding a NULL discretionary ACL
//** TODO: give only access to the user we start the program for
//** TODO: revert access after the program finishes

BOOL SetUserObjectAllAccess(HANDLE hObject, LPSTR szReturn)
{
	PSECURITY_DESCRIPTOR pSD;
	SECURITY_INFORMATION si;

	/* Initialize a security descriptor. */

	pSD = (PSECURITY_DESCRIPTOR) LocalAlloc(LPTR, SECURITY_DESCRIPTOR_MIN_LENGTH);   /* defined in WINNT.H */
	if (pSD == NULL) 
	{
    	ErrorHandler("LocalAlloc", szReturn);
    	return FALSE;
    }

	if (!InitializeSecurityDescriptor(pSD, SECURITY_DESCRIPTOR_REVISION)) 
	{ /* defined in WINNT.H */
    	ErrorHandler("InitializeSecurityDescriptor", szReturn);
    	goto Cleanup;
	}

	/* Add a NULL disc. ACL to the security descriptor. */
	if (!SetSecurityDescriptorDacl(pSD,
        TRUE,     /* specifying a disc. ACL  */
        (PACL) NULL,
        FALSE))  /* not a default disc. ACL */
    {
    	ErrorHandler("SetSecurityDescriptorDacl", szReturn);
    	goto Cleanup;
	}

	/* Add the security descriptor to the file. */
	si = DACL_SECURITY_INFORMATION; 
	if (!SetUserObjectSecurity(hObject,
    	    &si,
        	pSD)) 
    {
    	ErrorHandler("SetUserObjectSecurity", szReturn);
    	goto Cleanup;
	}

	if(pSD != NULL)
        LocalFree((HLOCAL) pSD);
	return TRUE;
	
	Cleanup:
    if(pSD != NULL)
        LocalFree((HLOCAL) pSD);
	return FALSE;
}


